import pytest

from dep_logic.tags import os
from dep_logic.tags.platform import Arch, Platform


def test_platform_tags_manylinux():
    tags = Platform(os.Manylinux(2, 20), Arch.X86_64).compatible_tags
    assert tags == [
        "manylinux_2_20_x86_64",
        "manylinux_2_19_x86_64",
        "manylinux_2_18_x86_64",
        "manylinux_2_17_x86_64",
        "manylinux2014_x86_64",
        "manylinux_2_16_x86_64",
        "manylinux_2_15_x86_64",
        "manylinux_2_14_x86_64",
        "manylinux_2_13_x86_64",
        "manylinux_2_12_x86_64",
        "manylinux2010_x86_64",
        "manylinux_2_11_x86_64",
        "manylinux_2_10_x86_64",
        "manylinux_2_9_x86_64",
        "manylinux_2_8_x86_64",
        "manylinux_2_7_x86_64",
        "manylinux_2_6_x86_64",
        "manylinux_2_5_x86_64",
        "manylinux1_x86_64",
        "linux_x86_64",
    ]


def test_platform_tags_macos():
    tags = Platform(os.Macos(21, 6), Arch.X86_64).compatible_tags
    assert tags == [
        "macosx_21_0_x86_64",
        "macosx_21_0_intel",
        "macosx_21_0_fat64",
        "macosx_21_0_fat32",
        "macosx_21_0_universal2",
        "macosx_21_0_universal",
        "macosx_20_0_x86_64",
        "macosx_20_0_intel",
        "macosx_20_0_fat64",
        "macosx_20_0_fat32",
        "macosx_20_0_universal2",
        "macosx_20_0_universal",
        "macosx_19_0_x86_64",
        "macosx_19_0_intel",
        "macosx_19_0_fat64",
        "macosx_19_0_fat32",
        "macosx_19_0_universal2",
        "macosx_19_0_universal",
        "macosx_18_0_x86_64",
        "macosx_18_0_intel",
        "macosx_18_0_fat64",
        "macosx_18_0_fat32",
        "macosx_18_0_universal2",
        "macosx_18_0_universal",
        "macosx_17_0_x86_64",
        "macosx_17_0_intel",
        "macosx_17_0_fat64",
        "macosx_17_0_fat32",
        "macosx_17_0_universal2",
        "macosx_17_0_universal",
        "macosx_16_0_x86_64",
        "macosx_16_0_intel",
        "macosx_16_0_fat64",
        "macosx_16_0_fat32",
        "macosx_16_0_universal2",
        "macosx_16_0_universal",
        "macosx_15_0_x86_64",
        "macosx_15_0_intel",
        "macosx_15_0_fat64",
        "macosx_15_0_fat32",
        "macosx_15_0_universal2",
        "macosx_15_0_universal",
        "macosx_14_0_x86_64",
        "macosx_14_0_intel",
        "macosx_14_0_fat64",
        "macosx_14_0_fat32",
        "macosx_14_0_universal2",
        "macosx_14_0_universal",
        "macosx_13_0_x86_64",
        "macosx_13_0_intel",
        "macosx_13_0_fat64",
        "macosx_13_0_fat32",
        "macosx_13_0_universal2",
        "macosx_13_0_universal",
        "macosx_12_0_x86_64",
        "macosx_12_0_intel",
        "macosx_12_0_fat64",
        "macosx_12_0_fat32",
        "macosx_12_0_universal2",
        "macosx_12_0_universal",
        "macosx_11_0_x86_64",
        "macosx_11_0_intel",
        "macosx_11_0_fat64",
        "macosx_11_0_fat32",
        "macosx_11_0_universal2",
        "macosx_11_0_universal",
        "macosx_10_16_x86_64",
        "macosx_10_16_intel",
        "macosx_10_16_fat64",
        "macosx_10_16_fat32",
        "macosx_10_16_universal2",
        "macosx_10_16_universal",
        "macosx_10_15_x86_64",
        "macosx_10_15_intel",
        "macosx_10_15_fat64",
        "macosx_10_15_fat32",
        "macosx_10_15_universal2",
        "macosx_10_15_universal",
        "macosx_10_14_x86_64",
        "macosx_10_14_intel",
        "macosx_10_14_fat64",
        "macosx_10_14_fat32",
        "macosx_10_14_universal2",
        "macosx_10_14_universal",
        "macosx_10_13_x86_64",
        "macosx_10_13_intel",
        "macosx_10_13_fat64",
        "macosx_10_13_fat32",
        "macosx_10_13_universal2",
        "macosx_10_13_universal",
        "macosx_10_12_x86_64",
        "macosx_10_12_intel",
        "macosx_10_12_fat64",
        "macosx_10_12_fat32",
        "macosx_10_12_universal2",
        "macosx_10_12_universal",
        "macosx_10_11_x86_64",
        "macosx_10_11_intel",
        "macosx_10_11_fat64",
        "macosx_10_11_fat32",
        "macosx_10_11_universal2",
        "macosx_10_11_universal",
        "macosx_10_10_x86_64",
        "macosx_10_10_intel",
        "macosx_10_10_fat64",
        "macosx_10_10_fat32",
        "macosx_10_10_universal2",
        "macosx_10_10_universal",
        "macosx_10_9_x86_64",
        "macosx_10_9_intel",
        "macosx_10_9_fat64",
        "macosx_10_9_fat32",
        "macosx_10_9_universal2",
        "macosx_10_9_universal",
        "macosx_10_8_x86_64",
        "macosx_10_8_intel",
        "macosx_10_8_fat64",
        "macosx_10_8_fat32",
        "macosx_10_8_universal2",
        "macosx_10_8_universal",
        "macosx_10_7_x86_64",
        "macosx_10_7_intel",
        "macosx_10_7_fat64",
        "macosx_10_7_fat32",
        "macosx_10_7_universal2",
        "macosx_10_7_universal",
        "macosx_10_6_x86_64",
        "macosx_10_6_intel",
        "macosx_10_6_fat64",
        "macosx_10_6_fat32",
        "macosx_10_6_universal2",
        "macosx_10_6_universal",
        "macosx_10_5_x86_64",
        "macosx_10_5_intel",
        "macosx_10_5_fat64",
        "macosx_10_5_fat32",
        "macosx_10_5_universal2",
        "macosx_10_5_universal",
        "macosx_10_4_x86_64",
        "macosx_10_4_intel",
        "macosx_10_4_fat64",
        "macosx_10_4_fat32",
        "macosx_10_4_universal2",
        "macosx_10_4_universal",
    ]

    tags = Platform(os.Macos(14, 0), Arch.X86_64).compatible_tags
    assert tags == [
        "macosx_14_0_x86_64",
        "macosx_14_0_intel",
        "macosx_14_0_fat64",
        "macosx_14_0_fat32",
        "macosx_14_0_universal2",
        "macosx_14_0_universal",
        "macosx_13_0_x86_64",
        "macosx_13_0_intel",
        "macosx_13_0_fat64",
        "macosx_13_0_fat32",
        "macosx_13_0_universal2",
        "macosx_13_0_universal",
        "macosx_12_0_x86_64",
        "macosx_12_0_intel",
        "macosx_12_0_fat64",
        "macosx_12_0_fat32",
        "macosx_12_0_universal2",
        "macosx_12_0_universal",
        "macosx_11_0_x86_64",
        "macosx_11_0_intel",
        "macosx_11_0_fat64",
        "macosx_11_0_fat32",
        "macosx_11_0_universal2",
        "macosx_11_0_universal",
        "macosx_10_16_x86_64",
        "macosx_10_16_intel",
        "macosx_10_16_fat64",
        "macosx_10_16_fat32",
        "macosx_10_16_universal2",
        "macosx_10_16_universal",
        "macosx_10_15_x86_64",
        "macosx_10_15_intel",
        "macosx_10_15_fat64",
        "macosx_10_15_fat32",
        "macosx_10_15_universal2",
        "macosx_10_15_universal",
        "macosx_10_14_x86_64",
        "macosx_10_14_intel",
        "macosx_10_14_fat64",
        "macosx_10_14_fat32",
        "macosx_10_14_universal2",
        "macosx_10_14_universal",
        "macosx_10_13_x86_64",
        "macosx_10_13_intel",
        "macosx_10_13_fat64",
        "macosx_10_13_fat32",
        "macosx_10_13_universal2",
        "macosx_10_13_universal",
        "macosx_10_12_x86_64",
        "macosx_10_12_intel",
        "macosx_10_12_fat64",
        "macosx_10_12_fat32",
        "macosx_10_12_universal2",
        "macosx_10_12_universal",
        "macosx_10_11_x86_64",
        "macosx_10_11_intel",
        "macosx_10_11_fat64",
        "macosx_10_11_fat32",
        "macosx_10_11_universal2",
        "macosx_10_11_universal",
        "macosx_10_10_x86_64",
        "macosx_10_10_intel",
        "macosx_10_10_fat64",
        "macosx_10_10_fat32",
        "macosx_10_10_universal2",
        "macosx_10_10_universal",
        "macosx_10_9_x86_64",
        "macosx_10_9_intel",
        "macosx_10_9_fat64",
        "macosx_10_9_fat32",
        "macosx_10_9_universal2",
        "macosx_10_9_universal",
        "macosx_10_8_x86_64",
        "macosx_10_8_intel",
        "macosx_10_8_fat64",
        "macosx_10_8_fat32",
        "macosx_10_8_universal2",
        "macosx_10_8_universal",
        "macosx_10_7_x86_64",
        "macosx_10_7_intel",
        "macosx_10_7_fat64",
        "macosx_10_7_fat32",
        "macosx_10_7_universal2",
        "macosx_10_7_universal",
        "macosx_10_6_x86_64",
        "macosx_10_6_intel",
        "macosx_10_6_fat64",
        "macosx_10_6_fat32",
        "macosx_10_6_universal2",
        "macosx_10_6_universal",
        "macosx_10_5_x86_64",
        "macosx_10_5_intel",
        "macosx_10_5_fat64",
        "macosx_10_5_fat32",
        "macosx_10_5_universal2",
        "macosx_10_5_universal",
        "macosx_10_4_x86_64",
        "macosx_10_4_intel",
        "macosx_10_4_fat64",
        "macosx_10_4_fat32",
        "macosx_10_4_universal2",
        "macosx_10_4_universal",
    ]

    tags = Platform(os.Macos(10, 6), Arch.X86_64).compatible_tags
    assert tags == [
        "macosx_10_6_x86_64",
        "macosx_10_6_intel",
        "macosx_10_6_fat64",
        "macosx_10_6_fat32",
        "macosx_10_6_universal2",
        "macosx_10_6_universal",
        "macosx_10_5_x86_64",
        "macosx_10_5_intel",
        "macosx_10_5_fat64",
        "macosx_10_5_fat32",
        "macosx_10_5_universal2",
        "macosx_10_5_universal",
        "macosx_10_4_x86_64",
        "macosx_10_4_intel",
        "macosx_10_4_fat64",
        "macosx_10_4_fat32",
        "macosx_10_4_universal2",
        "macosx_10_4_universal",
    ]


def test_platform_tags_windows():
    tags = Platform(os.Windows(), Arch.X86_64).compatible_tags
    assert tags == ["win_amd64"]


def test_platform_tags_musl():
    tags = Platform(os.Musllinux(1, 2), Arch.Aarch64).compatible_tags
    assert tags == ["linux_aarch64", "musllinux_1_1_aarch64", "musllinux_1_2_aarch64"]


@pytest.mark.parametrize(
    "text,expected,normalized",
    [
        ("linux", Platform(os.Manylinux(2, 17), Arch.X86_64), "manylinux_2_17_x86_64"),
        ("macos", Platform(os.Macos(14, 0), Arch.Aarch64), "macos_14_0_arm64"),
        ("windows", Platform(os.Windows(), Arch.X86_64), "windows_amd64"),
        ("alpine", Platform(os.Musllinux(1, 2), Arch.X86_64), "musllinux_1_2_x86_64"),
        (
            "manylinux_2_20_aarch64",
            Platform(os.Manylinux(2, 20), Arch.Aarch64),
            "manylinux_2_20_aarch64",
        ),
        (
            "macos_14_0_arm64",
            Platform(os.Macos(14, 0), Arch.Aarch64),
            "macos_14_0_arm64",
        ),
        ("windows_amd64", Platform(os.Windows(), Arch.X86_64), "windows_amd64"),
        ("windows_arm64", Platform(os.Windows(), Arch.Aarch64), "windows_arm64"),
        (
            "macos_12_0_x86_64",
            Platform(os.Macos(12, 0), Arch.X86_64),
            "macos_12_0_x86_64",
        ),
        (
            "mingw_x86_64",
            Platform(os.Generic("mingw"), Arch.X86_64),
            "mingw_x86_64",
        ),
    ],
)
def test_parse_platform(text, expected, normalized):
    platform = Platform.parse(text)
    assert platform == expected
    assert str(platform) == normalized
