#define USER_LEN 16 /* must be same as value in ipc.h */

#include "bool.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <sys/mman.h>
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#include <config.h>
#include "compat.h"
#include "makeauth.h"

#ifndef _XOPEN_SOURCE
# define _XOPEN_SOURCE
#endif
#ifndef __USE_XOPEN
# define __USE_XOPEN
#endif
#include <unistd.h>

char *progname;

int main(int argc, char *argv[])
{
	char *filename;
	char *user;
	int action;

	progname = argv[0];

	action = args(argc, argv, &filename, &user);
	switch(action) {
	case USER_ERR: return EXIT_FAILURE;
	case USER_END: return EXIT_SUCCESS;
	case USER_ADD: return user_add(user, filename);
    	case USER_REM: return user_rem(user, filename);
	case USER_INIT: return init_file(filename);
	}
	fprintf(stderr, "%s: unimplemented option\n", progname);
	return EXIT_FAILURE;
}

bool init_file(const char *filename)
{
	int fd;
	FILE *fp;
	umask(0);
	fd = open(filename, O_WRONLY | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
	if (fd==-1) {
		fprintf(stderr, "%s: creating '%s' failed: %s\n",
				progname, filename, strerror(errno));
		return EXIT_FAILURE;
	}
	fp = fdopen(fd, "w");
	if (!fp) {
		fprintf(stderr, "%s: fdopen on '%s' failed: %s\n",
				progname, filename, strerror(errno));
		close(fd);
		return EXIT_FAILURE;
	}
	fprintf(fp, "%%unknown:*\n%%con:*\n%%all:*\n");
	if (EOF==fclose(fp)) {
		fprintf(stderr, "%s: closing '%s' failed: %s\n",
				progname, filename, strerror(errno));
		return EXIT_FAILURE;
	}
	fprintf(stderr, "%s: the authentication file '%s' has been created\n",
			progname, filename);
	return EXIT_SUCCESS;
}

bool user_add(const char *user, const char *filename)
{
	int fd;
	int ret;
	FILE *fp;
	char *pass;
	char salt[3];
	
	fd = open(filename, O_RDWR);
	if (fd==-1) {
		fprintf(stderr, "%s: opening '%s' for reading and writing "
				"failed: %s\n", progname, filename,
				strerror(errno));
		return EXIT_FAILURE;
	}
	ret = user_exists_changepass(user, fd);
	if (ret!=EXISTS_NO) {
		close(fd);
		if (ret==EXISTS_ERR)
			return EXIT_FAILURE;
		return EXIT_SUCCESS;
	}
	fprintf(stderr, "Adding user %s\n", user);
	fp = fdopen(fd, "a");
	if (!fp) {
		fprintf(stderr, "%s: appending data to '%s' failed: %s\n",
				progname, filename, strerror(errno));
		close(fd);
		return EXIT_FAILURE;
	}
	srand(time(NULL));
	makesalt(salt);
	if (!(pass = makepass()))
		return EXIT_FAILURE;
	/* On some systems, crypt will use MD5 even though we're giving it
	 * a two-byte salt. This is fine, since auth.c is happy with MD5 and
	 * Blowfish crypted passwords. */
	fprintf(fp, "%s:%s\n", user, (char *)crypt(pass, salt));
	memset(pass, 0, strlen(pass)); free(pass);
	if (EOF==fclose(fp)) {
		fprintf(stderr, "%s: closing '%s' failed: %s\n",
				progname, filename, strerror(errno));
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}

bool user_rem(const char *user, const char *filename)
{
	fprintf(stderr, "%s: remove is not yet implemented\n"
			"lock the account as admin or remove the user with a "
			"text editor\n", progname);
	if(filename){}; /* to prevent warning on -W */
	if(user){};
	return EXIT_FAILURE;
}

void makesalt(char salt[3])
{
	const char SELECT_CHAR[] =	"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
					"abcdefghijklmnopqrstuvwxyz"
					"0123456789"
					"./";
	int i;
	for (i=0; i < 2; ++i)
		salt[i] = SELECT_CHAR[ (int) (64.0*rand()/(RAND_MAX)) ];
	salt[2] = '\0';
	return;
}
	
char *makepass(void)
{
	char *pt1 = getpass("Enter password: ");
	char *pass = xstrdup(pt1);
	char *pt2;
	memset(pt1, 0, strlen(pt1));
	
	if (!pass)
		return NULL;
	if (!*pass) {
		fprintf(stderr, "Aborting on empty password\n");
		return NULL;
	}
	pt2 = getpass("Re-enter password: ");
	if (0!=strcmp(pass, pt2)) {
		memset(pass, 0, strlen(pass));
		free(pass);
		memset(pt2, 0, strlen(pt2));
		fprintf(stderr, "Passwords do not match\n");
		return NULL;
	}
	memset(pt2, 0, strlen(pt2));
	return pass;
}

int args(int argc, char *argv[], char **filename, char **user)
{
	char ret;
	int action;
	mmap_area usertmp;
	
	/* defaults */
	action = USER_ADD;
	if (!(*filename = xstrdup("/etc/dwunauth")))
		return USER_ERR;

	while ((ret = getopt(argc, argv, "+rivhf:"))!=EOF) {
		switch(ret) {
		case 'r':
		       action = USER_REM;
		       break;
		case 'i':
		       action = USER_INIT;
		       break;
		case 'f':
		       free(*filename);
		       if (!(*filename = xstrdup(optarg)))
			       return USER_ERR;
		       break;
		case 'v':
		       verinfo();
	       	       return USER_END;
   		case 'h':
		       helpinfo();
		       return USER_END;
		}
	}
	if (action==USER_INIT)
		return action;
	/* else USER_ADD or USER_REM */

	/* Get username. */
	if (!argv[optind]) {
		helpinfo();
		return USER_ERR;
	}
	usertmp.data = argv[optind];
	usertmp.len = strlen(usertmp.data);
	
	if (invalid_username(usertmp))
		return USER_ERR;
	if (!(*user = xstrdup(usertmp.data)))
		return USER_ERR;
	return action;
}

char *xstrdup(const char *s)
{
	char *new = strdup(s);
	if (!new) {
		fprintf(stderr, "%s: memory allocation failed: %s\n",
				progname, strerror(errno));
	}
	return new;
}

void helpinfo(void)
{
	verinfo();
	fprintf(stderr, "usage: %s [-f FILE] [-i] [-v] [-h] [-r] [username]\n\n"
			, progname);
	fprintf(stderr, "-f\tauthfile\n");
	fprintf(stderr, "-i\tinitialize authfile\n");
	fprintf(stderr, "-v\tversion\n");
	fprintf(stderr, "-h\tthis help\n");
	fprintf(stderr, "-r\tremove a user\n");
	return;
}

void verinfo(void)
{
	fprintf(stderr, "dwunmakeauth version %s\n", VERSION);
	fprintf(stderr, "Copyright (C) 2001 Tim Sutherland\n");
	fprintf(stderr, "See the file COPYING in the distribution\n\n");
	return;
}

bool map_authfile(int fd, mmap_area *authfile)
{
	struct stat sbuf;

	if (fstat(fd, &sbuf)) {
		fprintf(stderr, "%s: getting length of authfile failed: %s\n",
				progname, strerror(errno));
		return TRUE;
	}
	if ((int)sbuf.st_size==0) {
		authfile->data = NULL; /* empty file */
		return FALSE;
	}
	authfile->len = sbuf.st_size;
	authfile->data = (char *)mmap(0, authfile->len, PROT_READ | 
					PROT_WRITE, MAP_SHARED, fd, 0);
	if (authfile->data==(char *)-1) {
		fprintf(stderr, "%s: mapping authfile to memory failed: %s\n",
				progname, strerror(errno));
		return TRUE;
	}
	return FALSE;
}

bool unmap_authfile(mmap_area authfile)
{
	bool ret = FALSE;
	if (authfile.len % getpagesize() != 0)
		authfile.len += getpagesize() - (authfile.len % getpagesize());
	if (msync(authfile.data, authfile.len, MS_SYNC)) {
		fprintf(stderr, "%s: syncing authfile failed: %s\n",
				progname, strerror(errno));
		ret = TRUE;
	}
	if (munmap(authfile.data, authfile.len)) {
		fprintf(stderr, "%s: unmapping authfile failed: %s\n",
				progname, strerror(errno));
		ret = TRUE;
	}
	return ret;
}

int user_exists_changepass(const char *user, int fd)
{
	int ret;
	mmap_area authfile;
	struct account ac;

	if (map_authfile(fd, &authfile))
		return EXISTS_ERR;
	if (!authfile.data)
		return EXISTS_NO;
	if (auth_getaccount(user, authfile, &ac)) {
		/* user doesn't exist */
		if (unmap_authfile(authfile))
			return EXISTS_ERR;
		return EXISTS_NO;
	}
	fprintf(stderr, "User %s exists, changing password (press enter to "
			"abort)\n", user);
	ret = changepass(ac);
	if (unmap_authfile(authfile) || ret)
		return EXISTS_ERR;
	return EXISTS_YES;
}

bool changepass(struct account ac)
{
	char status;
	char *pass;
	char *cryptedpass;
	char salt[3];

	srand(time(NULL));
	makesalt(salt);
	if (!(pass = makepass()))
		return TRUE;
	cryptedpass = (char *)crypt(pass, salt);
	memset(pass, 0, strlen(pass)); free(pass);
	if (strlen(cryptedpass)!=ac.crypass.len) {
		fprintf(stderr, "%s: compatiblity error between old and new "
				"crypted passwords\n", progname);
		return TRUE;
	}
	status = *ac.lock;
	*ac.lock = '!'; /* lock account */
	memcpy(ac.crypass.data, cryptedpass, ac.crypass.len);
	*ac.lock = status;
	return FALSE;
}

/* Ripped from auth.c. */

bool auth_getaccount(const char *user, mmap_area authfile, struct account *ac)
{
	do {
		authfile = auth_getline(ac, authfile);
		if (!authfile.data)
			return TRUE;
	} while (ac->user.len!=strlen(user) ||
			0!=strncmp(user, ac->user.data, ac->user.len));
	return FALSE;
}

mmap_area auth_getline(struct account *ac, mmap_area start)
{
	mmap_area curr;

	curr.data = start.data;
	curr.len = start.len;

	/* user */
	ac->user.data = curr.data;
	if (findchrline(curr.data, ":!", "\n", &ac->user.len, curr.len)) {
		curr.len = 0; /* end of authfile */
		curr.data = NULL;
		return curr;
	}
	curr.data = ac->user.data + ac->user.len;
	curr.len -= ac->user.len;

	/* lock */
	ac->lock = curr.data;
	curr.data = ac->lock + 1;
	curr.len -= 1;

	/* pass */
	ac->crypass.data = curr.data;
	if (findchrline(curr.data, "\n", "", &ac->crypass.len, curr.len)) {
		curr.len = 1; /* error flag */
		curr.data = NULL;
		return curr;
	}
	curr.data = ac->crypass.data + ac->crypass.len;
	curr.len -= ac->crypass.len;

	/* newline */
	curr.data += 1;
	curr.len -= 1;
	
	return curr;
}

bool invalid_authfile(mmap_area authfile)
{
	struct account ac;
	authfile = auth_getline(&ac, authfile);
	while (authfile.data) {
		if (invalid_username(ac.user))
			return TRUE;
		authfile = auth_getline(&ac, authfile);
	}
	if (authfile.len!=0) {
		fprintf(stderr, "%s: end of authfile is missing newline\n",
				progname);
		return TRUE;
	}
	return FALSE;
}

bool invalid_username(mmap_area user)
{
	size_t i;
	const char VALID_FIRST[] =	"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
					"abcdefghijklmnopqrstuvwxyz"
					"_-";

	const char VALID[] =		"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
					"abcdefghijklmnopqrstuvwxyz"
					"0123456789"
					"_-";
	if (user.len==0) {
		fprintf(stderr, "%s: username is empty\n", progname);
		return TRUE;
	}
	if (user.len > USER_LEN) {
		fprintf(stderr, "%s: username is longer than %d characters\n",
				progname, USER_LEN);
		return TRUE;
	}
	if (!strchr(VALID_FIRST, user.data[0])) {
		fprintf(stderr, "%s: usernames may not start with a number\n",
				progname);
		return TRUE;
	}
	for (i=1; i < user.len; ++i) {
		if (!strchr(VALID, user.data[i])) {
			fprintf(stderr, "%s: username contains invalid "
					"character '%c'\n", progname,
					user.data[i]);
			return TRUE;
		}
	}
	return FALSE;
}

/* Set /len/ to the number of characters in /s/ before any of /good/ are
 * found.
 * 
 * The search stops and returns TRUE (leaving len unchanged) if
 * 	1) We hit any of the chars in /bad/.
 * 	2) We have searched through /n/ characters.
 * 	3) We hit \0.
 *
 * The search stops, returns FALSE and sets /len/ to the number of characters
 * we've searched through if
 * 	1) We reach any of the chars in /good/.
 */
bool findchrline(const char *s, char *good, char *bad, size_t *len, size_t n)
{
	const char *pt;
	char *tmp;
	for (pt=s; *pt && n; ++pt, --n) {
		for (tmp=bad; *tmp; ++tmp) {
			if (*pt==*tmp)
				return TRUE;
		}
		for (tmp=good; *tmp; ++tmp) {
			if (*pt==*tmp) {
				*len = pt-s;
				return FALSE;
			}
		}
	}
	return TRUE;
}
