/* With an argument of seconds, counts down when it receives SIGUSR1 until
 * SIGUSR2. If you send SIGUSR1 again it will continue counting down. Returns 0
 * when the count reaches zero or EXIT_FAILURE otherwise.
 *
 * It also handles receiving two SIGUSR1's or two SIGUSR2's in a row. (e.g.
 * one user connects multiple times).
 *
 * Will print seconds remaining to stdout and exit with EXIT_FAILURE on SIGHUP.
 * 
 * Written by Tim Sutherland (timsuth@ihug.co.nz) and released into the public
 * domain on Mon Feb 28 20:59:38 NZDT 2000.
 */

#include <time.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <limits.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

int sigsetup(void);
void sighandler(int sig);
int sighandler_write(int c);
int timeout(int fd, unsigned int secs);
void helpinfo(char *progname);

static volatile sig_atomic_t sighandler_pipe;

int main(int argc, char *argv[])
{
	unsigned int secs;
	int fd;

	if (argc!=2) {
		helpinfo(argv[0]);
		return EXIT_FAILURE;
	}
	secs = strtoul(argv[1], NULL, 10);
	if (secs>=UINT_MAX) {
		/* alarm uses unsigned int, we use (unsigned int)-1 for error */
		fprintf(stderr, "%s: seconds argument is too large\n", argv[0]);
		return EXIT_FAILURE;
	}
	fd = sigsetup();
	if (-1==fd)
		return EXIT_FAILURE;
	if (-1==timeout(fd, secs))
		return EXIT_FAILURE;
	return 0; /* time is up */
}

int sigsetup(void)
{
	struct sigaction cursig;
	int filedes[2];

	memset(&cursig, 0, sizeof(cursig));
	cursig.sa_handler = sighandler;
	sigaddset(&cursig.sa_mask, SIGUSR1);
	sigaddset(&cursig.sa_mask, SIGUSR2);
	sigaddset(&cursig.sa_mask, SIGALRM);
	sigaddset(&cursig.sa_mask, SIGHUP);
	if (-1==sigaction(SIGUSR1, &cursig, NULL) ||
			-1==sigaction(SIGUSR2, &cursig, NULL) ||
			-1==sigaction(SIGALRM, &cursig, NULL) ||
			-1==sigaction(SIGHUP, &cursig, NULL))
		return -1;
	if (-1==pipe(filedes))
		return -1;
	sighandler_pipe = filedes[1];
	return filedes[0];
}

void sighandler(int sig)
{
	switch(sig) {
	case SIGUSR1: sighandler_write('c'); /* CONNECT */
		      break;
	case SIGUSR2: sighandler_write('d'); /* DISCONNECT */
		      break;
	case SIGALRM: sighandler_write('t'); /* time is up */
		      break;
	case SIGHUP:  sighandler_write('h'); /* dump time remaining to stdout */
		      break;
	}
	return;
}

int sighandler_write(int c)
{
	return write(sighandler_pipe, &c, sizeof(c));
}

int timeout(int fd, unsigned int secs)
{
	unsigned int running;
	int ret;
	int c;
	int con = 0;

	while (secs) {
		do {
			ret = read(fd, &c, sizeof(c));
		} while (ret==-1 && errno==EINTR);
		if (ret==-1) {
			perror("read failed");
			return -1;
		}
		switch(c) {
		case 'c': ++con;
			  if (con==1)
				  alarm(secs);
			  break;
		case 'd': --con;
			  if (con==0)
				  secs = alarm(0);
			  break;
		case 't': return 0; /* timeout */
		case 'h': /* halt: print the number of seconds remaining and
			     exit. */
			  if (con) {
				  printf("%u\n", alarm(0));
				  return -1;
			  }
			  printf("%u\n", secs);
			  return -1;
		default:  fprintf(stderr, "Invalid message from sighandler\n");
			  return -1;
		}
	}
	return 0;
}

void helpinfo(char *progname)
{
	fprintf(stderr, "Usage: %s seconds\n", progname);
	return;
}
