// Copyright 2023 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "generator/internal/connection_rest_generator.h"
#include "generator/internal/codegen_utils.h"
#include "generator/internal/descriptor_utils.h"
#include "generator/internal/predicate_utils.h"
#include "generator/internal/printer.h"
#include <google/protobuf/descriptor.h>

namespace google {
namespace cloud {
namespace generator_internal {

ConnectionRestGenerator::ConnectionRestGenerator(
    google::protobuf::ServiceDescriptor const* service_descriptor,
    VarsDictionary service_vars,
    std::map<std::string, VarsDictionary> service_method_vars,
    google::protobuf::compiler::GeneratorContext* context,
    std::vector<MixinMethod> const& mixin_methods)
    : ServiceCodeGenerator(
          "connection_rest_header_path", "connection_rest_cc_path",
          service_descriptor, std::move(service_vars),
          std::move(service_method_vars), context, mixin_methods) {}

Status ConnectionRestGenerator::GenerateHeader() {
  HeaderPrint(CopyrightLicenseFileHeader());
  HeaderPrint(R"""(
// Generated by the Codegen C++ plugin.
// If you make any local changes, they will be lost.
// source: $proto_file_name$

#ifndef $header_include_guard$
#define $header_include_guard$
)""");

  auto endpoint_location_style = EndpointLocationStyle();

  // includes
  HeaderPrint("\n");
  HeaderLocalIncludes({vars("connection_header_path"), "google/cloud/options.h",
                       "google/cloud/version.h"});
  HeaderSystemIncludes({"memory"});
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      HeaderSystemIncludes({"string"});
      break;
    default:
      break;
  }

  auto result = HeaderOpenNamespaces();
  if (!result.ok()) return result;
  HeaderPrint(R"""(
/**
 * A factory function to construct an object of type `$connection_class_name$`
 * that uses REST over HTTP as transport in lieu of gRPC. REST transport should
 * only be used for services that do not support gRPC or if the existing network
 * configuration precludes using gRPC.
 *
 * The returned connection object should not be used directly; instead it
 * should be passed as an argument to the constructor of $client_class_name$.
 *
 * The optional @p options argument may be used to configure aspects of the
 * returned `$connection_class_name$`. Expected options are any of the types in
 * the following option lists:
 *
 * - `google::cloud::CommonOptionList`
 * - `google::cloud::RestOptionList`
 * - `google::cloud::UnifiedCredentialsOptionList`
 * - `google::cloud::$product_namespace$::$service_name$PolicyOptionList`
 *
 * @note Unexpected options will be ignored. To log unexpected options instead,
 *     set `GOOGLE_CLOUD_CPP_ENABLE_CLOG=yes` in the environment.
 *)""");
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      HeaderPrint(R"""(
 * @param location Sets the prefix for the default `EndpointOption` value.)""");
      break;
    default:
      break;
  }
  HeaderPrint(R"""(
 * @param options (optional) Configure the `$connection_class_name$` created by
 * this function.
 */
std::shared_ptr<$connection_class_name$> Make$connection_class_name$Rest(
)""");
  HeaderPrint("    " + ConnectionFactoryFunctionArguments() + " = {});\n");
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      HeaderPrint(R"""(
/**
 * A backwards-compatible version of the previous factory function.  Unless
 * the service also offers a global endpoint, the default value of the
 * `EndpointOption` may be useless, in which case it must be overridden.
 *
 * @deprecated Please use the `location` overload instead.
 */
std::shared_ptr<$connection_class_name$> Make$connection_class_name$Rest(
    Options options = {});
)""");
      break;
    default:
      break;
  }

  HeaderCloseNamespaces();
  // close header guard
  HeaderPrint("\n#endif  // $header_include_guard$\n");
  return {};
}

Status ConnectionRestGenerator::GenerateCc() {
  CcPrint(CopyrightLicenseFileHeader());
  CcPrint(R"""(
// Generated by the Codegen C++ plugin.
// If you make any local changes, they will be lost.
// source: $proto_file_name$
)""");

  // includes
  CcPrint("\n");
  CcLocalIncludes(
      {vars("connection_rest_header_path"), vars("options_header_path"),
       vars("connection_impl_rest_header_path"),
       vars("option_defaults_header_path"),
       vars("stub_factory_rest_header_path"),
       vars("tracing_connection_header_path"), "google/cloud/common_options.h",
       "google/cloud/internal/rest_options.h", "google/cloud/credentials.h",
       "google/cloud/internal/rest_background_threads_impl.h"});
  CcSystemIncludes({"memory", "utility"});

  auto result = CcOpenNamespaces();
  if (!result.ok()) return result;
  auto endpoint_location_style = EndpointLocationStyle();

  CcPrint(R"""(
std::shared_ptr<$connection_class_name$> Make$connection_class_name$Rest(
)""");
  CcPrint("    " + ConnectionFactoryFunctionArguments() + ") {");
  CcPrint(R"""(
  internal::CheckExpectedOptions<CommonOptionList, RestOptionList,
      UnifiedCredentialsOptionList, rest_internal::TargetApiVersionOption,
      $service_name$PolicyOptionList>(options, __func__);
  options = $product_internal_namespace$::$service_name$DefaultOptions(
)""");
  CcPrint("      ");
  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      CcPrint("location, ");
      break;
    default:
      break;
  }
  CcPrint("std::move(options));");
  CcPrint(R"""(
  auto background = std::make_unique<
      rest_internal::AutomaticallyCreatedRestBackgroundThreads>();
  auto stub = $product_internal_namespace$::CreateDefault$stub_rest_class_name$(
      options);
  return $product_internal_namespace$::Make$tracing_connection_class_name$(
      std::make_shared<
          $product_internal_namespace$::$connection_impl_rest_class_name$>(
          std::move(background), std::move(stub), std::move(options)));
}
)""");

  switch (endpoint_location_style) {
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      CcPrint(R"""(
std::shared_ptr<$connection_class_name$> Make$connection_class_name$Rest(
    Options options) {
  return Make$connection_class_name$(std::string{}, std::move(options));
}
)""");
      break;
    default:
      break;
  }

  CcCloseNamespaces();
  return {};
}

std::string ConnectionRestGenerator::ConnectionFactoryFunctionArguments()
    const {
  std::string args;
  if (IsExperimental()) args += "ExperimentalTag, ";
  switch (EndpointLocationStyle()) {
    case ServiceConfiguration::LOCATION_DEPENDENT:
    case ServiceConfiguration::LOCATION_DEPENDENT_COMPAT:
      args += "std::string const& location, ";
      break;
    default:
      break;
  }
  args += "Options options";
  return args;
}

}  // namespace generator_internal
}  // namespace cloud
}  // namespace google
