/*----------------------------------------------------------------------------------------
  Database ht
----------------------------------------------------------------------------------------*/

#include "htdatabase.h"

union htTempType_ htTemp_;
struct htRootType_ htRootData;
uint8 htModuleID;
struct htGraphFields htGraphs;
struct htNodeFields htNodes;
struct htDatadrawRootFields htDatadrawRoots;
struct htSparseParentGraphNodeDataFields htSparseParentGraphNodeDatas;
struct htSparseChildGraphNodeDataFields htSparseChildGraphNodeDatas;
struct htSparseNodeNameDataFields htSparseNodeNameDatas;
struct htSparseNodeXDataFields htSparseNodeXDatas;

/*----------------------------------------------------------------------------------------
  Constructor/Destructor hooks.
----------------------------------------------------------------------------------------*/
void(*htGraphConstructorCallback)(htGraph);
void(*htGraphDestructorCallback)(htGraph);
void(*htNodeConstructorCallback)(htNode);
void(*htNodeDestructorCallback)(htNode);
void(*htDatadrawRootConstructorCallback)(htDatadrawRoot);
void(*htSparseParentGraphNodeDataConstructorCallback)(htSparseParentGraphNodeData);
void(*htSparseParentGraphNodeDataDestructorCallback)(htSparseParentGraphNodeData);
void(*htSparseChildGraphNodeDataConstructorCallback)(htSparseChildGraphNodeData);
void(*htSparseChildGraphNodeDataDestructorCallback)(htSparseChildGraphNodeData);
void(*htSparseNodeNameDataConstructorCallback)(htSparseNodeNameData);
void(*htSparseNodeNameDataDestructorCallback)(htSparseNodeNameData);
void(*htSparseNodeXDataConstructorCallback)(htSparseNodeXData);
void(*htSparseNodeXDataDestructorCallback)(htSparseNodeXData);

/*----------------------------------------------------------------------------------------
  Destroy Graph including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void htGraphDestroy(
    htGraph Graph)
{
    htNode Node_, nextNode_;

    if(htGraphDestructorCallback != NULL) {
        htGraphDestructorCallback(Graph);
    }
    for(Node_ = htGraphGetFirstNode(Graph); Node_ != htNodeNull;
            Node_ = nextNode_) {
        nextNode_ = htNodeGetNextGraphNode(Node_);
        htNodeDestroy(Node_);
    }
    htGraphFree(Graph);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocGraph(void)
{
    htGraph Graph = htGraphAlloc();

    return htGraph2Index(Graph);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroyGraph(
    uint64 objectIndex)
{
    htGraphDestroy(htIndex2Graph((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Graph.
----------------------------------------------------------------------------------------*/
static void allocGraphs(void)
{
    htSetAllocatedGraph(2);
    htSetUsedGraph(0);
    htSetFirstFreeGraph(htGraphNull);
    htSetUsedGraphNodeTable(0);
    htSetAllocatedGraphNodeTable(2);
    htSetFreeGraphNodeTable(0);
    htGraphs.NodeTable = utNewA(htNode, htAllocatedGraphNodeTable());
    htGraphs.FreeList = utNewA(htGraph, (htAllocatedGraph()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Graph.
----------------------------------------------------------------------------------------*/
static void reallocGraphs(
    uint32 newSize)
{
    utResizeArray(htGraphs.FreeList, (newSize));
    htSetAllocatedGraph(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Graphs.
----------------------------------------------------------------------------------------*/
void htGraphAllocMore(void)
{
    reallocGraphs((uint32)(htAllocatedGraph() + (htAllocatedGraph() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Graph.NodeTable heap to free memory.
----------------------------------------------------------------------------------------*/
void htCompactGraphNodeTables(void)
{
    uint32 elementSize = sizeof(htNode);
    uint32 usedHeaderSize = (sizeof(htGraph) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htGraph) + sizeof(uint32) + elementSize - 1)/elementSize;
    htNode *toPtr = htGraphs.NodeTable;
    htNode *fromPtr = toPtr;
    htGraph Graph;
    uint32 size;

    while(fromPtr < htGraphs.NodeTable + htUsedGraphNodeTable()) {
        Graph = *(htGraph *)(void *)fromPtr;
        if(Graph != htGraphNull) {
            /* Need to move it to toPtr */
            size = utMax(htGraphGetNumNodeTable(Graph) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            htGraphSetNodeTableIndex(Graph, toPtr - htGraphs.NodeTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((htGraph *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    htSetUsedGraphNodeTable(toPtr - htGraphs.NodeTable);
    htSetFreeGraphNodeTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Graph.NodeTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreGraphNodeTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = htAllocatedGraphNodeTable() - htUsedGraphNodeTable();

    if((htFreeGraphNodeTable() << 2) > htUsedGraphNodeTable()) {
        htCompactGraphNodeTables();
        freeSpace = htAllocatedGraphNodeTable() - htUsedGraphNodeTable();
    }
    if(freeSpace < spaceNeeded) {
        htSetAllocatedGraphNodeTable(htAllocatedGraphNodeTable() + spaceNeeded - freeSpace +
            (htAllocatedGraphNodeTable() >> 1));
        utResizeArray(htGraphs.NodeTable, htAllocatedGraphNodeTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Graph.NodeTable array.
----------------------------------------------------------------------------------------*/
void htGraphAllocNodeTables(
    htGraph Graph,
    uint32 numNodeTables)
{
    uint32 freeSpace = htAllocatedGraphNodeTable() - htUsedGraphNodeTable();
    uint32 elementSize = sizeof(htNode);
    uint32 usedHeaderSize = (sizeof(htGraph) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htGraph) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numNodeTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(htGraphGetNumNodeTable(Graph) == 0);
#endif
    if(numNodeTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreGraphNodeTables(spaceNeeded);
    }
    htGraphSetNodeTableIndex(Graph, htUsedGraphNodeTable() + usedHeaderSize);
    htGraphSetNumNodeTable(Graph, numNodeTables);
    *(htGraph *)(void *)(htGraphs.NodeTable + htUsedGraphNodeTable()) = Graph;
    {
        uint32 xGraph;
        for(xGraph = (uint32)(htGraphGetNodeTableIndex(Graph)); xGraph < htGraphGetNodeTableIndex(Graph) + numNodeTables; xGraph++) {
            htGraphs.NodeTable[xGraph] = htNodeNull;
        }
    }
    htSetUsedGraphNodeTable(htUsedGraphNodeTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htGraphGetNodeTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getGraphNodeTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    htGraph Graph = htIndex2Graph((uint32)objectNumber);

    *numValues = htGraphGetNumNodeTable(Graph);
    return htGraphGetNodeTables(Graph);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htGraphAllocNodeTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocGraphNodeTables(
    uint64 objectNumber,
    uint32 numValues)
{
    htGraph Graph = htIndex2Graph((uint32)objectNumber);

    htGraphSetNodeTableIndex(Graph, 0);
    htGraphSetNumNodeTable(Graph, 0);
    if(numValues == 0) {
        return NULL;
    }
    htGraphAllocNodeTables(Graph, numValues);
    return htGraphGetNodeTables(Graph);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Graph.NodeTable array.
----------------------------------------------------------------------------------------*/
void htGraphFreeNodeTables(
    htGraph Graph)
{
    uint32 elementSize = sizeof(htNode);
    uint32 usedHeaderSize = (sizeof(htGraph) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htGraph) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(htGraphGetNumNodeTable(Graph) + usedHeaderSize, freeHeaderSize);
    htNode *dataPtr = htGraphGetNodeTables(Graph) - usedHeaderSize;

    if(htGraphGetNumNodeTable(Graph) == 0) {
        return;
    }
    *(htGraph *)(void *)(dataPtr) = htGraphNull;
    *(uint32 *)(void *)(((htGraph *)(void *)dataPtr) + 1) = size;
    htGraphSetNumNodeTable(Graph, 0);
    htSetFreeGraphNodeTable(htFreeGraphNodeTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Graph.NodeTable array.
----------------------------------------------------------------------------------------*/
void htGraphResizeNodeTables(
    htGraph Graph,
    uint32 numNodeTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(htNode);
    uint32 usedHeaderSize = (sizeof(htGraph) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htGraph) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numNodeTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(htGraphGetNumNodeTable(Graph) + usedHeaderSize, freeHeaderSize);
    htNode *dataPtr;

    if(numNodeTables == 0) {
        if(htGraphGetNumNodeTable(Graph) != 0) {
            htGraphFreeNodeTables(Graph);
        }
        return;
    }
    if(htGraphGetNumNodeTable(Graph) == 0) {
        htGraphAllocNodeTables(Graph, numNodeTables);
        return;
    }
    freeSpace = htAllocatedGraphNodeTable() - htUsedGraphNodeTable();
    if(freeSpace < newSize) {
        allocMoreGraphNodeTables(newSize);
    }
    dataPtr = htGraphGetNodeTables(Graph) - usedHeaderSize;
    memcpy((void *)(htGraphs.NodeTable + htUsedGraphNodeTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xGraph;
            for(xGraph = (uint32)(htUsedGraphNodeTable() + oldSize); xGraph < htUsedGraphNodeTable() + oldSize + newSize - oldSize; xGraph++) {
                htGraphs.NodeTable[xGraph] = htNodeNull;
            }
        }
    }
    *(htGraph *)(void *)dataPtr = htGraphNull;
    *(uint32 *)(void *)(((htGraph *)(void *)dataPtr) + 1) = oldSize;
    htSetFreeGraphNodeTable(htFreeGraphNodeTable() + oldSize);
    htGraphSetNodeTableIndex(Graph, htUsedGraphNodeTable() + usedHeaderSize);
    htGraphSetNumNodeTable(Graph, numNodeTables);
    htSetUsedGraphNodeTable(htUsedGraphNodeTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Graph.
----------------------------------------------------------------------------------------*/
void htGraphCopyProps(
    htGraph oldGraph,
    htGraph newGraph)
{
}

/*----------------------------------------------------------------------------------------
  Get the Graph.FirstNode field.
----------------------------------------------------------------------------------------*/
htNode htGraphGetFirstNode(
    htGraph Graph)
{
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(htFirstDatadrawRoot(), Graph);

    if(object == htSparseParentGraphNodeDataNull) {
        return htNodeNull;
    }
    return htSparseParentGraphNodeDataGetFirstNode(object);
}

/*----------------------------------------------------------------------------------------
  Set the Graph.FirstNode field.
----------------------------------------------------------------------------------------*/
void htGraphSetFirstNode(
    htGraph Graph,
    htNode value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(theRoot, Graph);

    if(value != htNodeNull || object != htSparseParentGraphNodeDataNull) {
        if(object == htSparseParentGraphNodeDataNull) {
            object = htSparseParentGraphNodeDataAlloc();
            htSparseParentGraphNodeDataSetGraphKey(object, Graph);
            htDatadrawRootInsertSparseParentGraphNodeData(theRoot, object);
        }
        htSparseParentGraphNodeDataSetFirstNode(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Graph.LastNode field.
----------------------------------------------------------------------------------------*/
htNode htGraphGetLastNode(
    htGraph Graph)
{
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(htFirstDatadrawRoot(), Graph);

    if(object == htSparseParentGraphNodeDataNull) {
        return htNodeNull;
    }
    return htSparseParentGraphNodeDataGetLastNode(object);
}

/*----------------------------------------------------------------------------------------
  Set the Graph.LastNode field.
----------------------------------------------------------------------------------------*/
void htGraphSetLastNode(
    htGraph Graph,
    htNode value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(theRoot, Graph);

    if(value != htNodeNull || object != htSparseParentGraphNodeDataNull) {
        if(object == htSparseParentGraphNodeDataNull) {
            object = htSparseParentGraphNodeDataAlloc();
            htSparseParentGraphNodeDataSetGraphKey(object, Graph);
            htDatadrawRootInsertSparseParentGraphNodeData(theRoot, object);
        }
        htSparseParentGraphNodeDataSetLastNode(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Graph.NodeTableIndex field.
----------------------------------------------------------------------------------------*/
uint32 htGraphGetNodeTableIndex(
    htGraph Graph)
{
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(htFirstDatadrawRoot(), Graph);

    if(object == htSparseParentGraphNodeDataNull) {
        return 0;
    }
    return htSparseParentGraphNodeDataGetNodeTableIndex(object);
}

/*----------------------------------------------------------------------------------------
  Set the Graph.NodeTableIndex field.
----------------------------------------------------------------------------------------*/
void htGraphSetNodeTableIndex(
    htGraph Graph,
    uint32 value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(theRoot, Graph);

    if(value != 0 || object != htSparseParentGraphNodeDataNull) {
        if(object == htSparseParentGraphNodeDataNull) {
            object = htSparseParentGraphNodeDataAlloc();
            htSparseParentGraphNodeDataSetGraphKey(object, Graph);
            htDatadrawRootInsertSparseParentGraphNodeData(theRoot, object);
        }
        htSparseParentGraphNodeDataSetNodeTableIndex(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Graph.NumNodeTable field.
----------------------------------------------------------------------------------------*/
uint32 htGraphGetNumNodeTable(
    htGraph Graph)
{
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(htFirstDatadrawRoot(), Graph);

    if(object == htSparseParentGraphNodeDataNull) {
        return 0;
    }
    return htSparseParentGraphNodeDataGetNumNodeTable(object);
}

/*----------------------------------------------------------------------------------------
  Set the Graph.NumNodeTable field.
----------------------------------------------------------------------------------------*/
void htGraphSetNumNodeTable(
    htGraph Graph,
    uint32 value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(theRoot, Graph);

    if(value != 0 || object != htSparseParentGraphNodeDataNull) {
        if(object == htSparseParentGraphNodeDataNull) {
            object = htSparseParentGraphNodeDataAlloc();
            htSparseParentGraphNodeDataSetGraphKey(object, Graph);
            htDatadrawRootInsertSparseParentGraphNodeData(theRoot, object);
        }
        htSparseParentGraphNodeDataSetNumNodeTable(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Graph.NumNode field.
----------------------------------------------------------------------------------------*/
uint32 htGraphGetNumNode(
    htGraph Graph)
{
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(htFirstDatadrawRoot(), Graph);

    if(object == htSparseParentGraphNodeDataNull) {
        return 0;
    }
    return htSparseParentGraphNodeDataGetNumNode(object);
}

/*----------------------------------------------------------------------------------------
  Set the Graph.NumNode field.
----------------------------------------------------------------------------------------*/
void htGraphSetNumNode(
    htGraph Graph,
    uint32 value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseParentGraphNodeData object = htDatadrawRootFindSparseParentGraphNodeData(theRoot, Graph);

    if(value != 0 || object != htSparseParentGraphNodeDataNull) {
        if(object == htSparseParentGraphNodeDataNull) {
            object = htSparseParentGraphNodeDataAlloc();
            htSparseParentGraphNodeDataSetGraphKey(object, Graph);
            htDatadrawRootInsertSparseParentGraphNodeData(theRoot, object);
        }
        htSparseParentGraphNodeDataSetNumNode(object, value);
    }
}

static void addGraphNodeToHashTable(htGraph Graph, htNode _Node);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeGraphNodeHashTable(
    htGraph Graph)
{
    htNode _Node;
    htNode *Nodes;
    uint32 numNodes = htGraphGetNumNodeTable(Graph) << 1;

    if(numNodes == 0) {
        numNodes = 2;
        htGraphAllocNodeTables(Graph, 2);
    } else {
        htGraphResizeNodeTables(Graph, numNodes);
    }
    Nodes = htGraphGetNodeTables(Graph);
    /* Zero out the table */
    while(numNodes-- != 0) {
        *Nodes++ = htNodeNull;
    }
    htGraphSetNumNode(Graph, 0);
    htForeachGraphNode(Graph, _Node) {
        addGraphNodeToHashTable(Graph, _Node);
    } htEndGraphNode;
}

/*----------------------------------------------------------------------------------------
  Add the Node to the Graph.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addGraphNodeToHashTable(
    htGraph Graph,
    htNode _Node)
{
    htNode nextNode;
    uint32 index;

    if(htGraphGetNumNode(Graph) >= htGraphGetNumNodeTable(Graph)) {
        resizeGraphNodeHashTable(Graph);
        return;
    }
    index = (htGraphGetNumNodeTable(Graph) - 1) & (uint32)htNodeGetX(_Node);
    nextNode = htGraphGetiNodeTable(Graph, index);
    htNodeSetNextTableGraphNode(_Node, nextNode);
    htGraphSetiNodeTable(Graph, index, _Node);
    htGraphSetNumNode(Graph, htGraphGetNumNode(Graph) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the Node from the hash table.
----------------------------------------------------------------------------------------*/
static void removeGraphNodeFromHashTable(
   htGraph Graph,
   htNode _Node)
{
    uint32 index = (htGraphGetNumNodeTable(Graph) - 1) & (uint32)htNodeGetX(_Node);
    htNode prevNode, nextNode;
    
    nextNode = htGraphGetiNodeTable(Graph, index);
    if(nextNode == _Node) {
        htGraphSetiNodeTable(Graph, index, htNodeGetNextTableGraphNode(nextNode));
    } else {
        do {
            prevNode = nextNode;
            nextNode = htNodeGetNextTableGraphNode(nextNode);
        } while(nextNode != _Node);
        htNodeSetNextTableGraphNode(prevNode, htNodeGetNextTableGraphNode(_Node));
    }
    htGraphSetNumNode(Graph, htGraphGetNumNode(Graph) - 1);
    htNodeSetNextTableGraphNode(_Node, htNodeNull);
}

/*----------------------------------------------------------------------------------------
  Find the Node from the Graph and its hash key.
----------------------------------------------------------------------------------------*/
htNode htGraphFindNode(
    htGraph Graph,
    int32 X,
    int32 Y)
{
    uint32 mask = htGraphGetNumNodeTable(Graph) - 1;
    htNode _Node;

    if(mask + 1 != 0) {
        _Node = htGraphGetiNodeTable(Graph, (uint32)X & mask);
        while(_Node != htNodeNull) {
            if(htNodeGetX(_Node) == X && htNodeGetY(_Node) == Y) {
                return _Node;
            }
            _Node = htNodeGetNextTableGraphNode(_Node);
        }
    }
    return htNodeNull;
}

/*----------------------------------------------------------------------------------------
  Add the Node to the head of the list on the Graph.
----------------------------------------------------------------------------------------*/
void htGraphInsertNode(
    htGraph Graph,
    htNode _Node)
{
#if defined(DD_DEBUG)
    if(Graph == htGraphNull) {
        utExit("Non-existent Graph");
    }
    if(_Node == htNodeNull) {
        utExit("Non-existent Node");
    }
#endif
    htNodeSetNextGraphNode(_Node, htGraphGetFirstNode(Graph));
    if(htGraphGetFirstNode(Graph) != htNodeNull) {
        htNodeSetPrevGraphNode(htGraphGetFirstNode(Graph), _Node);
    }
    htGraphSetFirstNode(Graph, _Node);
    htNodeSetPrevGraphNode(_Node, htNodeNull);
    if(htGraphGetLastNode(Graph) == htNodeNull) {
        htGraphSetLastNode(Graph, _Node);
    }
    addGraphNodeToHashTable(Graph, _Node);
}

/*----------------------------------------------------------------------------------------
  Add the Node to the end of the list on the Graph.
----------------------------------------------------------------------------------------*/
void htGraphAppendNode(
    htGraph Graph,
    htNode _Node)
{
#if defined(DD_DEBUG)
    if(Graph == htGraphNull) {
        utExit("Non-existent Graph");
    }
    if(_Node == htNodeNull) {
        utExit("Non-existent Node");
    }
#endif
    htNodeSetPrevGraphNode(_Node, htGraphGetLastNode(Graph));
    if(htGraphGetLastNode(Graph) != htNodeNull) {
        htNodeSetNextGraphNode(htGraphGetLastNode(Graph), _Node);
    }
    htGraphSetLastNode(Graph, _Node);
    htNodeSetNextGraphNode(_Node, htNodeNull);
    if(htGraphGetFirstNode(Graph) == htNodeNull) {
        htGraphSetFirstNode(Graph, _Node);
    }
    addGraphNodeToHashTable(Graph, _Node);
}

/*----------------------------------------------------------------------------------------
  Insert the Node to the Graph after the previous Node.
----------------------------------------------------------------------------------------*/
void htGraphInsertAfterNode(
    htGraph Graph,
    htNode prevNode,
    htNode _Node)
{
    htNode nextNode = htNodeGetNextGraphNode(prevNode);

#if defined(DD_DEBUG)
    if(Graph == htGraphNull) {
        utExit("Non-existent Graph");
    }
    if(_Node == htNodeNull) {
        utExit("Non-existent Node");
    }
#endif
    htNodeSetNextGraphNode(_Node, nextNode);
    htNodeSetNextGraphNode(prevNode, _Node);
    htNodeSetPrevGraphNode(_Node, prevNode);
    if(nextNode != htNodeNull) {
        htNodeSetPrevGraphNode(nextNode, _Node);
    }
    if(htGraphGetLastNode(Graph) == prevNode) {
        htGraphSetLastNode(Graph, _Node);
    }
    addGraphNodeToHashTable(Graph, _Node);
}

/*----------------------------------------------------------------------------------------
 Remove the Node from the Graph.
----------------------------------------------------------------------------------------*/
void htGraphRemoveNode(
    htGraph Graph,
    htNode _Node)
{
    htNode pNode, nNode;

#if defined(DD_DEBUG)
    if(_Node == htNodeNull) {
        utExit("Non-existent Node");
    }
#endif
    nNode = htNodeGetNextGraphNode(_Node);
    pNode = htNodeGetPrevGraphNode(_Node);
    if(pNode != htNodeNull) {
        htNodeSetNextGraphNode(pNode, nNode);
    } else if(htGraphGetFirstNode(Graph) == _Node) {
        htGraphSetFirstNode(Graph, nNode);
    }
    if(nNode != htNodeNull) {
        htNodeSetPrevGraphNode(nNode, pNode);
    } else if(htGraphGetLastNode(Graph) == _Node) {
        htGraphSetLastNode(Graph, pNode);
    }
    htNodeSetNextGraphNode(_Node, htNodeNull);
    htNodeSetPrevGraphNode(_Node, htNodeNull);
    removeGraphNodeFromHashTable(Graph, _Node);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowGraph(
    htGraph Graph)
{
    utDatabaseShowObject("ht", "Graph", htGraph2Index(Graph));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy Node including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void htNodeDestroy(
    htNode Node)
{
    if(htNodeDestructorCallback != NULL) {
        htNodeDestructorCallback(Node);
    }
    htNodeFree(Node);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocNode(void)
{
    htNode Node = htNodeAlloc();

    return htNode2Index(Node);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroyNode(
    uint64 objectIndex)
{
    htNodeDestroy(htIndex2Node((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of Node.
----------------------------------------------------------------------------------------*/
static void allocNodes(void)
{
    htSetAllocatedNode(2);
    htSetUsedNode(0);
    htSetFirstFreeNode(htNodeNull);
    htSetUsedNodeName(0);
    htSetAllocatedNodeName(2);
    htSetFreeNodeName(0);
    htNodes.Name = utNewA(char, htAllocatedNodeName());
    htNodes.Y = utNewA(int32, (htAllocatedNode()));
    htNodes.FreeList = utNewA(htNode, (htAllocatedNode()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class Node.
----------------------------------------------------------------------------------------*/
static void reallocNodes(
    uint32 newSize)
{
    utResizeArray(htNodes.Y, (newSize));
    utResizeArray(htNodes.FreeList, (newSize));
    htSetAllocatedNode(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more Nodes.
----------------------------------------------------------------------------------------*/
void htNodeAllocMore(void)
{
    reallocNodes((uint32)(htAllocatedNode() + (htAllocatedNode() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the Node.Name heap to free memory.
----------------------------------------------------------------------------------------*/
void htCompactNodeNames(void)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(htNode) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htNode) + sizeof(uint32) + elementSize - 1)/elementSize;
    char *toPtr = htNodes.Name;
    char *fromPtr = toPtr;
    htNode Node;
    uint32 size;

    while(fromPtr < htNodes.Name + htUsedNodeName()) {
        Node = *(htNode *)(void *)fromPtr;
        if(Node != htNodeNull) {
            /* Need to move it to toPtr */
            size = utMax(htNodeGetNumName(Node) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            htNodeSetNameIndex(Node, toPtr - htNodes.Name + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((htNode *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    htSetUsedNodeName(toPtr - htNodes.Name);
    htSetFreeNodeName(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the Node.Name heap.
----------------------------------------------------------------------------------------*/
static void allocMoreNodeNames(
    uint32 spaceNeeded)
{
    uint32 freeSpace = htAllocatedNodeName() - htUsedNodeName();

    if((htFreeNodeName() << 2) > htUsedNodeName()) {
        htCompactNodeNames();
        freeSpace = htAllocatedNodeName() - htUsedNodeName();
    }
    if(freeSpace < spaceNeeded) {
        htSetAllocatedNodeName(htAllocatedNodeName() + spaceNeeded - freeSpace +
            (htAllocatedNodeName() >> 1));
        utResizeArray(htNodes.Name, htAllocatedNodeName());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new Node.Name array.
----------------------------------------------------------------------------------------*/
void htNodeAllocNames(
    htNode Node,
    uint32 numNames)
{
    uint32 freeSpace = htAllocatedNodeName() - htUsedNodeName();
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(htNode) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htNode) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numNames + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(htNodeGetNumName(Node) == 0);
#endif
    if(numNames == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreNodeNames(spaceNeeded);
    }
    htNodeSetNameIndex(Node, htUsedNodeName() + usedHeaderSize);
    htNodeSetNumName(Node, numNames);
    *(htNode *)(void *)(htNodes.Name + htUsedNodeName()) = Node;
    memset(htNodes.Name + htNodeGetNameIndex(Node), 0, ((numNames))*sizeof(char));
    htSetUsedNodeName(htUsedNodeName() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htNodeGetNames for the database manager.
----------------------------------------------------------------------------------------*/
static void *getNodeNames(
    uint64 objectNumber,
    uint32 *numValues)
{
    htNode Node = htIndex2Node((uint32)objectNumber);

    *numValues = htNodeGetNumName(Node);
    return htNodeGetNames(Node);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htNodeAllocNames for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocNodeNames(
    uint64 objectNumber,
    uint32 numValues)
{
    htNode Node = htIndex2Node((uint32)objectNumber);

    htNodeSetNameIndex(Node, 0);
    htNodeSetNumName(Node, 0);
    if(numValues == 0) {
        return NULL;
    }
    htNodeAllocNames(Node, numValues);
    return htNodeGetNames(Node);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the Node.Name array.
----------------------------------------------------------------------------------------*/
void htNodeFreeNames(
    htNode Node)
{
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(htNode) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htNode) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(htNodeGetNumName(Node) + usedHeaderSize, freeHeaderSize);
    char *dataPtr = htNodeGetNames(Node) - usedHeaderSize;

    if(htNodeGetNumName(Node) == 0) {
        return;
    }
    *(htNode *)(void *)(dataPtr) = htNodeNull;
    *(uint32 *)(void *)(((htNode *)(void *)dataPtr) + 1) = size;
    htNodeSetNumName(Node, 0);
    htSetFreeNodeName(htFreeNodeName() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the Node.Name array.
----------------------------------------------------------------------------------------*/
void htNodeResizeNames(
    htNode Node,
    uint32 numNames)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(char);
    uint32 usedHeaderSize = (sizeof(htNode) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htNode) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numNames + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(htNodeGetNumName(Node) + usedHeaderSize, freeHeaderSize);
    char *dataPtr;

    if(numNames == 0) {
        if(htNodeGetNumName(Node) != 0) {
            htNodeFreeNames(Node);
        }
        return;
    }
    if(htNodeGetNumName(Node) == 0) {
        htNodeAllocNames(Node, numNames);
        return;
    }
    freeSpace = htAllocatedNodeName() - htUsedNodeName();
    if(freeSpace < newSize) {
        allocMoreNodeNames(newSize);
    }
    dataPtr = htNodeGetNames(Node) - usedHeaderSize;
    memcpy((void *)(htNodes.Name + htUsedNodeName()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        memset(htNodes.Name + htUsedNodeName() + oldSize, 0, ((newSize - oldSize))*sizeof(char));
    }
    *(htNode *)(void *)dataPtr = htNodeNull;
    *(uint32 *)(void *)(((htNode *)(void *)dataPtr) + 1) = oldSize;
    htSetFreeNodeName(htFreeNodeName() + oldSize);
    htNodeSetNameIndex(Node, htUsedNodeName() + usedHeaderSize);
    htNodeSetNumName(Node, numNames);
    htSetUsedNodeName(htUsedNodeName() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of Node.
----------------------------------------------------------------------------------------*/
void htNodeCopyProps(
    htNode oldNode,
    htNode newNode)
{
    htNodeSetX(newNode, htNodeGetX(oldNode));
    htNodeSetY(newNode, htNodeGetY(oldNode));
}

/*----------------------------------------------------------------------------------------
  Get the Node.NameIndex field.
----------------------------------------------------------------------------------------*/
uint32 htNodeGetNameIndex(
    htNode Node)
{
    htSparseNodeNameData object = htDatadrawRootFindSparseNodeNameData(htFirstDatadrawRoot(), Node);

    if(object == htSparseNodeNameDataNull) {
        return 0;
    }
    return htSparseNodeNameDataGetNameIndex(object);
}

/*----------------------------------------------------------------------------------------
  Set the Node.NameIndex field.
----------------------------------------------------------------------------------------*/
void htNodeSetNameIndex(
    htNode Node,
    uint32 value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseNodeNameData object = htDatadrawRootFindSparseNodeNameData(theRoot, Node);

    if(value != 0 || object != htSparseNodeNameDataNull) {
        if(object == htSparseNodeNameDataNull) {
            object = htSparseNodeNameDataAlloc();
            htSparseNodeNameDataSetNodeKey(object, Node);
            htDatadrawRootInsertSparseNodeNameData(theRoot, object);
        }
        htSparseNodeNameDataSetNameIndex(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Node.NumName field.
----------------------------------------------------------------------------------------*/
uint32 htNodeGetNumName(
    htNode Node)
{
    htSparseNodeNameData object = htDatadrawRootFindSparseNodeNameData(htFirstDatadrawRoot(), Node);

    if(object == htSparseNodeNameDataNull) {
        return 0;
    }
    return htSparseNodeNameDataGetNumName(object);
}

/*----------------------------------------------------------------------------------------
  Set the Node.NumName field.
----------------------------------------------------------------------------------------*/
void htNodeSetNumName(
    htNode Node,
    uint32 value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseNodeNameData object = htDatadrawRootFindSparseNodeNameData(theRoot, Node);

    if(value != 0 || object != htSparseNodeNameDataNull) {
        if(object == htSparseNodeNameDataNull) {
            object = htSparseNodeNameDataAlloc();
            htSparseNodeNameDataSetNodeKey(object, Node);
            htDatadrawRootInsertSparseNodeNameData(theRoot, object);
        }
        htSparseNodeNameDataSetNumName(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Node.X field.
----------------------------------------------------------------------------------------*/
int32 htNodeGetX(
    htNode Node)
{
    htSparseNodeXData object = htDatadrawRootFindSparseNodeXData(htFirstDatadrawRoot(), Node);

    if(object == htSparseNodeXDataNull) {
        return 1;
    }
    return htSparseNodeXDataGetX(object);
}

/*----------------------------------------------------------------------------------------
  Set the Node.X field.
----------------------------------------------------------------------------------------*/
void htNodeSetX(
    htNode Node,
    int32 value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseNodeXData object = htDatadrawRootFindSparseNodeXData(theRoot, Node);

    if(value != 1) {
        if(object == htSparseNodeXDataNull) {
            object = htSparseNodeXDataAlloc();
            htSparseNodeXDataSetNodeKey(object, Node);
            htDatadrawRootInsertSparseNodeXData(theRoot, object);
        }
        htSparseNodeXDataSetX(object, value);
    } else if(object != htSparseNodeXDataNull) {
        htSparseNodeXDataDestroy(object);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Node.NextGraphNode field.
----------------------------------------------------------------------------------------*/
htNode htNodeGetNextGraphNode(
    htNode Node)
{
    htSparseChildGraphNodeData object = htDatadrawRootFindSparseChildGraphNodeData(htFirstDatadrawRoot(), Node);

    if(object == htSparseChildGraphNodeDataNull) {
        return htNodeNull;
    }
    return htSparseChildGraphNodeDataGetNextGraphNode(object);
}

/*----------------------------------------------------------------------------------------
  Set the Node.NextGraphNode field.
----------------------------------------------------------------------------------------*/
void htNodeSetNextGraphNode(
    htNode Node,
    htNode value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseChildGraphNodeData object = htDatadrawRootFindSparseChildGraphNodeData(theRoot, Node);

    if(value != htNodeNull || object != htSparseChildGraphNodeDataNull) {
        if(object == htSparseChildGraphNodeDataNull) {
            object = htSparseChildGraphNodeDataAlloc();
            htSparseChildGraphNodeDataSetNodeKey(object, Node);
            htDatadrawRootInsertSparseChildGraphNodeData(theRoot, object);
        }
        htSparseChildGraphNodeDataSetNextGraphNode(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Node.PrevGraphNode field.
----------------------------------------------------------------------------------------*/
htNode htNodeGetPrevGraphNode(
    htNode Node)
{
    htSparseChildGraphNodeData object = htDatadrawRootFindSparseChildGraphNodeData(htFirstDatadrawRoot(), Node);

    if(object == htSparseChildGraphNodeDataNull) {
        return htNodeNull;
    }
    return htSparseChildGraphNodeDataGetPrevGraphNode(object);
}

/*----------------------------------------------------------------------------------------
  Set the Node.PrevGraphNode field.
----------------------------------------------------------------------------------------*/
void htNodeSetPrevGraphNode(
    htNode Node,
    htNode value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseChildGraphNodeData object = htDatadrawRootFindSparseChildGraphNodeData(theRoot, Node);

    if(value != htNodeNull || object != htSparseChildGraphNodeDataNull) {
        if(object == htSparseChildGraphNodeDataNull) {
            object = htSparseChildGraphNodeDataAlloc();
            htSparseChildGraphNodeDataSetNodeKey(object, Node);
            htDatadrawRootInsertSparseChildGraphNodeData(theRoot, object);
        }
        htSparseChildGraphNodeDataSetPrevGraphNode(object, value);
    }
}

/*----------------------------------------------------------------------------------------
  Get the Node.NextTableGraphNode field.
----------------------------------------------------------------------------------------*/
htNode htNodeGetNextTableGraphNode(
    htNode Node)
{
    htSparseChildGraphNodeData object = htDatadrawRootFindSparseChildGraphNodeData(htFirstDatadrawRoot(), Node);

    if(object == htSparseChildGraphNodeDataNull) {
        return htNodeNull;
    }
    return htSparseChildGraphNodeDataGetNextTableGraphNode(object);
}

/*----------------------------------------------------------------------------------------
  Set the Node.NextTableGraphNode field.
----------------------------------------------------------------------------------------*/
void htNodeSetNextTableGraphNode(
    htNode Node,
    htNode value)
{
    htDatadrawRoot theRoot = htFirstDatadrawRoot();
    htSparseChildGraphNodeData object = htDatadrawRootFindSparseChildGraphNodeData(theRoot, Node);

    if(value != htNodeNull || object != htSparseChildGraphNodeDataNull) {
        if(object == htSparseChildGraphNodeDataNull) {
            object = htSparseChildGraphNodeDataAlloc();
            htSparseChildGraphNodeDataSetNodeKey(object, Node);
            htDatadrawRootInsertSparseChildGraphNodeData(theRoot, object);
        }
        htSparseChildGraphNodeDataSetNextTableGraphNode(object, value);
    }
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowNode(
    htNode Node)
{
    utDatabaseShowObject("ht", "Node", htNode2Index(Node));
}
#endif

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocDatadrawRoot(void)
{
    htDatadrawRoot DatadrawRoot = htDatadrawRootAlloc();

    return htDatadrawRoot2Index(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of DatadrawRoot.
----------------------------------------------------------------------------------------*/
static void allocDatadrawRoots(void)
{
    htSetAllocatedDatadrawRoot(2);
    htSetUsedDatadrawRoot(0);
    htDatadrawRoots.FirstSparseParentGraphNodeData = utNewA(htSparseParentGraphNodeData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.LastSparseParentGraphNodeData = utNewA(htSparseParentGraphNodeData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.SparseParentGraphNodeDataTableIndex = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.NumSparseParentGraphNodeDataTable = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htSetUsedDatadrawRootSparseParentGraphNodeDataTable(0);
    htSetAllocatedDatadrawRootSparseParentGraphNodeDataTable(2);
    htSetFreeDatadrawRootSparseParentGraphNodeDataTable(0);
    htDatadrawRoots.SparseParentGraphNodeDataTable = utNewA(htSparseParentGraphNodeData, htAllocatedDatadrawRootSparseParentGraphNodeDataTable());
    htDatadrawRoots.NumSparseParentGraphNodeData = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.FirstSparseChildGraphNodeData = utNewA(htSparseChildGraphNodeData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.LastSparseChildGraphNodeData = utNewA(htSparseChildGraphNodeData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.SparseChildGraphNodeDataTableIndex = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.NumSparseChildGraphNodeDataTable = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htSetUsedDatadrawRootSparseChildGraphNodeDataTable(0);
    htSetAllocatedDatadrawRootSparseChildGraphNodeDataTable(2);
    htSetFreeDatadrawRootSparseChildGraphNodeDataTable(0);
    htDatadrawRoots.SparseChildGraphNodeDataTable = utNewA(htSparseChildGraphNodeData, htAllocatedDatadrawRootSparseChildGraphNodeDataTable());
    htDatadrawRoots.NumSparseChildGraphNodeData = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.FirstSparseNodeNameData = utNewA(htSparseNodeNameData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.LastSparseNodeNameData = utNewA(htSparseNodeNameData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.SparseNodeNameDataTableIndex = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.NumSparseNodeNameDataTable = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htSetUsedDatadrawRootSparseNodeNameDataTable(0);
    htSetAllocatedDatadrawRootSparseNodeNameDataTable(2);
    htSetFreeDatadrawRootSparseNodeNameDataTable(0);
    htDatadrawRoots.SparseNodeNameDataTable = utNewA(htSparseNodeNameData, htAllocatedDatadrawRootSparseNodeNameDataTable());
    htDatadrawRoots.NumSparseNodeNameData = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.FirstSparseNodeXData = utNewA(htSparseNodeXData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.LastSparseNodeXData = utNewA(htSparseNodeXData, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.SparseNodeXDataTableIndex = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htDatadrawRoots.NumSparseNodeXDataTable = utNewA(uint32, (htAllocatedDatadrawRoot()));
    htSetUsedDatadrawRootSparseNodeXDataTable(0);
    htSetAllocatedDatadrawRootSparseNodeXDataTable(2);
    htSetFreeDatadrawRootSparseNodeXDataTable(0);
    htDatadrawRoots.SparseNodeXDataTable = utNewA(htSparseNodeXData, htAllocatedDatadrawRootSparseNodeXDataTable());
    htDatadrawRoots.NumSparseNodeXData = utNewA(uint32, (htAllocatedDatadrawRoot()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class DatadrawRoot.
----------------------------------------------------------------------------------------*/
static void reallocDatadrawRoots(
    uint32 newSize)
{
    utResizeArray(htDatadrawRoots.FirstSparseParentGraphNodeData, (newSize));
    utResizeArray(htDatadrawRoots.LastSparseParentGraphNodeData, (newSize));
    utResizeArray(htDatadrawRoots.SparseParentGraphNodeDataTableIndex, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseParentGraphNodeDataTable, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseParentGraphNodeData, (newSize));
    utResizeArray(htDatadrawRoots.FirstSparseChildGraphNodeData, (newSize));
    utResizeArray(htDatadrawRoots.LastSparseChildGraphNodeData, (newSize));
    utResizeArray(htDatadrawRoots.SparseChildGraphNodeDataTableIndex, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseChildGraphNodeDataTable, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseChildGraphNodeData, (newSize));
    utResizeArray(htDatadrawRoots.FirstSparseNodeNameData, (newSize));
    utResizeArray(htDatadrawRoots.LastSparseNodeNameData, (newSize));
    utResizeArray(htDatadrawRoots.SparseNodeNameDataTableIndex, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseNodeNameDataTable, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseNodeNameData, (newSize));
    utResizeArray(htDatadrawRoots.FirstSparseNodeXData, (newSize));
    utResizeArray(htDatadrawRoots.LastSparseNodeXData, (newSize));
    utResizeArray(htDatadrawRoots.SparseNodeXDataTableIndex, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseNodeXDataTable, (newSize));
    utResizeArray(htDatadrawRoots.NumSparseNodeXData, (newSize));
    htSetAllocatedDatadrawRoot(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more DatadrawRoots.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAllocMore(void)
{
    reallocDatadrawRoots((uint32)(htAllocatedDatadrawRoot() + (htAllocatedDatadrawRoot() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Compact the DatadrawRoot.SparseParentGraphNodeDataTable heap to free memory.
----------------------------------------------------------------------------------------*/
void htCompactDatadrawRootSparseParentGraphNodeDataTables(void)
{
    uint32 elementSize = sizeof(htSparseParentGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    htSparseParentGraphNodeData *toPtr = htDatadrawRoots.SparseParentGraphNodeDataTable;
    htSparseParentGraphNodeData *fromPtr = toPtr;
    htDatadrawRoot DatadrawRoot;
    uint32 size;

    while(fromPtr < htDatadrawRoots.SparseParentGraphNodeDataTable + htUsedDatadrawRootSparseParentGraphNodeDataTable()) {
        DatadrawRoot = *(htDatadrawRoot *)(void *)fromPtr;
        if(DatadrawRoot != htDatadrawRootNull) {
            /* Need to move it to toPtr */
            size = utMax(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            htDatadrawRootSetSparseParentGraphNodeDataTableIndex(DatadrawRoot, toPtr - htDatadrawRoots.SparseParentGraphNodeDataTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((htDatadrawRoot *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    htSetUsedDatadrawRootSparseParentGraphNodeDataTable(toPtr - htDatadrawRoots.SparseParentGraphNodeDataTable);
    htSetFreeDatadrawRootSparseParentGraphNodeDataTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the DatadrawRoot.SparseParentGraphNodeDataTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreDatadrawRootSparseParentGraphNodeDataTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseParentGraphNodeDataTable() - htUsedDatadrawRootSparseParentGraphNodeDataTable();

    if((htFreeDatadrawRootSparseParentGraphNodeDataTable() << 2) > htUsedDatadrawRootSparseParentGraphNodeDataTable()) {
        htCompactDatadrawRootSparseParentGraphNodeDataTables();
        freeSpace = htAllocatedDatadrawRootSparseParentGraphNodeDataTable() - htUsedDatadrawRootSparseParentGraphNodeDataTable();
    }
    if(freeSpace < spaceNeeded) {
        htSetAllocatedDatadrawRootSparseParentGraphNodeDataTable(htAllocatedDatadrawRootSparseParentGraphNodeDataTable() + spaceNeeded - freeSpace +
            (htAllocatedDatadrawRootSparseParentGraphNodeDataTable() >> 1));
        utResizeArray(htDatadrawRoots.SparseParentGraphNodeDataTable, htAllocatedDatadrawRootSparseParentGraphNodeDataTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new DatadrawRoot.SparseParentGraphNodeDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAllocSparseParentGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseParentGraphNodeDataTables)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseParentGraphNodeDataTable() - htUsedDatadrawRootSparseParentGraphNodeDataTable();
    uint32 elementSize = sizeof(htSparseParentGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numSparseParentGraphNodeDataTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) == 0);
#endif
    if(numSparseParentGraphNodeDataTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreDatadrawRootSparseParentGraphNodeDataTables(spaceNeeded);
    }
    htDatadrawRootSetSparseParentGraphNodeDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseParentGraphNodeDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseParentGraphNodeDataTable(DatadrawRoot, numSparseParentGraphNodeDataTables);
    *(htDatadrawRoot *)(void *)(htDatadrawRoots.SparseParentGraphNodeDataTable + htUsedDatadrawRootSparseParentGraphNodeDataTable()) = DatadrawRoot;
    {
        uint32 xDatadrawRoot;
        for(xDatadrawRoot = (uint32)(htDatadrawRootGetSparseParentGraphNodeDataTableIndex(DatadrawRoot)); xDatadrawRoot < htDatadrawRootGetSparseParentGraphNodeDataTableIndex(DatadrawRoot) + numSparseParentGraphNodeDataTables; xDatadrawRoot++) {
            htDatadrawRoots.SparseParentGraphNodeDataTable[xDatadrawRoot] = htSparseParentGraphNodeDataNull;
        }
    }
    htSetUsedDatadrawRootSparseParentGraphNodeDataTable(htUsedDatadrawRootSparseParentGraphNodeDataTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootGetSparseParentGraphNodeDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getDatadrawRootSparseParentGraphNodeDataTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    *numValues = htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot);
    return htDatadrawRootGetSparseParentGraphNodeDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootAllocSparseParentGraphNodeDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocDatadrawRootSparseParentGraphNodeDataTables(
    uint64 objectNumber,
    uint32 numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    htDatadrawRootSetSparseParentGraphNodeDataTableIndex(DatadrawRoot, 0);
    htDatadrawRootSetNumSparseParentGraphNodeDataTable(DatadrawRoot, 0);
    if(numValues == 0) {
        return NULL;
    }
    htDatadrawRootAllocSparseParentGraphNodeDataTables(DatadrawRoot, numValues);
    return htDatadrawRootGetSparseParentGraphNodeDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the DatadrawRoot.SparseParentGraphNodeDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootFreeSparseParentGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot)
{
    uint32 elementSize = sizeof(htSparseParentGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseParentGraphNodeData *dataPtr = htDatadrawRootGetSparseParentGraphNodeDataTables(DatadrawRoot) - usedHeaderSize;

    if(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) == 0) {
        return;
    }
    *(htDatadrawRoot *)(void *)(dataPtr) = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = size;
    htDatadrawRootSetNumSparseParentGraphNodeDataTable(DatadrawRoot, 0);
    htSetFreeDatadrawRootSparseParentGraphNodeDataTable(htFreeDatadrawRootSparseParentGraphNodeDataTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the DatadrawRoot.SparseParentGraphNodeDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootResizeSparseParentGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseParentGraphNodeDataTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(htSparseParentGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numSparseParentGraphNodeDataTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseParentGraphNodeData *dataPtr;

    if(numSparseParentGraphNodeDataTables == 0) {
        if(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) != 0) {
            htDatadrawRootFreeSparseParentGraphNodeDataTables(DatadrawRoot);
        }
        return;
    }
    if(htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) == 0) {
        htDatadrawRootAllocSparseParentGraphNodeDataTables(DatadrawRoot, numSparseParentGraphNodeDataTables);
        return;
    }
    freeSpace = htAllocatedDatadrawRootSparseParentGraphNodeDataTable() - htUsedDatadrawRootSparseParentGraphNodeDataTable();
    if(freeSpace < newSize) {
        allocMoreDatadrawRootSparseParentGraphNodeDataTables(newSize);
    }
    dataPtr = htDatadrawRootGetSparseParentGraphNodeDataTables(DatadrawRoot) - usedHeaderSize;
    memcpy((void *)(htDatadrawRoots.SparseParentGraphNodeDataTable + htUsedDatadrawRootSparseParentGraphNodeDataTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xDatadrawRoot;
            for(xDatadrawRoot = (uint32)(htUsedDatadrawRootSparseParentGraphNodeDataTable() + oldSize); xDatadrawRoot < htUsedDatadrawRootSparseParentGraphNodeDataTable() + oldSize + newSize - oldSize; xDatadrawRoot++) {
                htDatadrawRoots.SparseParentGraphNodeDataTable[xDatadrawRoot] = htSparseParentGraphNodeDataNull;
            }
        }
    }
    *(htDatadrawRoot *)(void *)dataPtr = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = oldSize;
    htSetFreeDatadrawRootSparseParentGraphNodeDataTable(htFreeDatadrawRootSparseParentGraphNodeDataTable() + oldSize);
    htDatadrawRootSetSparseParentGraphNodeDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseParentGraphNodeDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseParentGraphNodeDataTable(DatadrawRoot, numSparseParentGraphNodeDataTables);
    htSetUsedDatadrawRootSparseParentGraphNodeDataTable(htUsedDatadrawRootSparseParentGraphNodeDataTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the DatadrawRoot.SparseChildGraphNodeDataTable heap to free memory.
----------------------------------------------------------------------------------------*/
void htCompactDatadrawRootSparseChildGraphNodeDataTables(void)
{
    uint32 elementSize = sizeof(htSparseChildGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    htSparseChildGraphNodeData *toPtr = htDatadrawRoots.SparseChildGraphNodeDataTable;
    htSparseChildGraphNodeData *fromPtr = toPtr;
    htDatadrawRoot DatadrawRoot;
    uint32 size;

    while(fromPtr < htDatadrawRoots.SparseChildGraphNodeDataTable + htUsedDatadrawRootSparseChildGraphNodeDataTable()) {
        DatadrawRoot = *(htDatadrawRoot *)(void *)fromPtr;
        if(DatadrawRoot != htDatadrawRootNull) {
            /* Need to move it to toPtr */
            size = utMax(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            htDatadrawRootSetSparseChildGraphNodeDataTableIndex(DatadrawRoot, toPtr - htDatadrawRoots.SparseChildGraphNodeDataTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((htDatadrawRoot *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    htSetUsedDatadrawRootSparseChildGraphNodeDataTable(toPtr - htDatadrawRoots.SparseChildGraphNodeDataTable);
    htSetFreeDatadrawRootSparseChildGraphNodeDataTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the DatadrawRoot.SparseChildGraphNodeDataTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreDatadrawRootSparseChildGraphNodeDataTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseChildGraphNodeDataTable() - htUsedDatadrawRootSparseChildGraphNodeDataTable();

    if((htFreeDatadrawRootSparseChildGraphNodeDataTable() << 2) > htUsedDatadrawRootSparseChildGraphNodeDataTable()) {
        htCompactDatadrawRootSparseChildGraphNodeDataTables();
        freeSpace = htAllocatedDatadrawRootSparseChildGraphNodeDataTable() - htUsedDatadrawRootSparseChildGraphNodeDataTable();
    }
    if(freeSpace < spaceNeeded) {
        htSetAllocatedDatadrawRootSparseChildGraphNodeDataTable(htAllocatedDatadrawRootSparseChildGraphNodeDataTable() + spaceNeeded - freeSpace +
            (htAllocatedDatadrawRootSparseChildGraphNodeDataTable() >> 1));
        utResizeArray(htDatadrawRoots.SparseChildGraphNodeDataTable, htAllocatedDatadrawRootSparseChildGraphNodeDataTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new DatadrawRoot.SparseChildGraphNodeDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAllocSparseChildGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseChildGraphNodeDataTables)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseChildGraphNodeDataTable() - htUsedDatadrawRootSparseChildGraphNodeDataTable();
    uint32 elementSize = sizeof(htSparseChildGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numSparseChildGraphNodeDataTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) == 0);
#endif
    if(numSparseChildGraphNodeDataTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreDatadrawRootSparseChildGraphNodeDataTables(spaceNeeded);
    }
    htDatadrawRootSetSparseChildGraphNodeDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseChildGraphNodeDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseChildGraphNodeDataTable(DatadrawRoot, numSparseChildGraphNodeDataTables);
    *(htDatadrawRoot *)(void *)(htDatadrawRoots.SparseChildGraphNodeDataTable + htUsedDatadrawRootSparseChildGraphNodeDataTable()) = DatadrawRoot;
    {
        uint32 xDatadrawRoot;
        for(xDatadrawRoot = (uint32)(htDatadrawRootGetSparseChildGraphNodeDataTableIndex(DatadrawRoot)); xDatadrawRoot < htDatadrawRootGetSparseChildGraphNodeDataTableIndex(DatadrawRoot) + numSparseChildGraphNodeDataTables; xDatadrawRoot++) {
            htDatadrawRoots.SparseChildGraphNodeDataTable[xDatadrawRoot] = htSparseChildGraphNodeDataNull;
        }
    }
    htSetUsedDatadrawRootSparseChildGraphNodeDataTable(htUsedDatadrawRootSparseChildGraphNodeDataTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootGetSparseChildGraphNodeDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getDatadrawRootSparseChildGraphNodeDataTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    *numValues = htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot);
    return htDatadrawRootGetSparseChildGraphNodeDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootAllocSparseChildGraphNodeDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocDatadrawRootSparseChildGraphNodeDataTables(
    uint64 objectNumber,
    uint32 numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    htDatadrawRootSetSparseChildGraphNodeDataTableIndex(DatadrawRoot, 0);
    htDatadrawRootSetNumSparseChildGraphNodeDataTable(DatadrawRoot, 0);
    if(numValues == 0) {
        return NULL;
    }
    htDatadrawRootAllocSparseChildGraphNodeDataTables(DatadrawRoot, numValues);
    return htDatadrawRootGetSparseChildGraphNodeDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the DatadrawRoot.SparseChildGraphNodeDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootFreeSparseChildGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot)
{
    uint32 elementSize = sizeof(htSparseChildGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseChildGraphNodeData *dataPtr = htDatadrawRootGetSparseChildGraphNodeDataTables(DatadrawRoot) - usedHeaderSize;

    if(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) == 0) {
        return;
    }
    *(htDatadrawRoot *)(void *)(dataPtr) = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = size;
    htDatadrawRootSetNumSparseChildGraphNodeDataTable(DatadrawRoot, 0);
    htSetFreeDatadrawRootSparseChildGraphNodeDataTable(htFreeDatadrawRootSparseChildGraphNodeDataTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the DatadrawRoot.SparseChildGraphNodeDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootResizeSparseChildGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseChildGraphNodeDataTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(htSparseChildGraphNodeData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numSparseChildGraphNodeDataTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseChildGraphNodeData *dataPtr;

    if(numSparseChildGraphNodeDataTables == 0) {
        if(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) != 0) {
            htDatadrawRootFreeSparseChildGraphNodeDataTables(DatadrawRoot);
        }
        return;
    }
    if(htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) == 0) {
        htDatadrawRootAllocSparseChildGraphNodeDataTables(DatadrawRoot, numSparseChildGraphNodeDataTables);
        return;
    }
    freeSpace = htAllocatedDatadrawRootSparseChildGraphNodeDataTable() - htUsedDatadrawRootSparseChildGraphNodeDataTable();
    if(freeSpace < newSize) {
        allocMoreDatadrawRootSparseChildGraphNodeDataTables(newSize);
    }
    dataPtr = htDatadrawRootGetSparseChildGraphNodeDataTables(DatadrawRoot) - usedHeaderSize;
    memcpy((void *)(htDatadrawRoots.SparseChildGraphNodeDataTable + htUsedDatadrawRootSparseChildGraphNodeDataTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xDatadrawRoot;
            for(xDatadrawRoot = (uint32)(htUsedDatadrawRootSparseChildGraphNodeDataTable() + oldSize); xDatadrawRoot < htUsedDatadrawRootSparseChildGraphNodeDataTable() + oldSize + newSize - oldSize; xDatadrawRoot++) {
                htDatadrawRoots.SparseChildGraphNodeDataTable[xDatadrawRoot] = htSparseChildGraphNodeDataNull;
            }
        }
    }
    *(htDatadrawRoot *)(void *)dataPtr = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = oldSize;
    htSetFreeDatadrawRootSparseChildGraphNodeDataTable(htFreeDatadrawRootSparseChildGraphNodeDataTable() + oldSize);
    htDatadrawRootSetSparseChildGraphNodeDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseChildGraphNodeDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseChildGraphNodeDataTable(DatadrawRoot, numSparseChildGraphNodeDataTables);
    htSetUsedDatadrawRootSparseChildGraphNodeDataTable(htUsedDatadrawRootSparseChildGraphNodeDataTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the DatadrawRoot.SparseNodeNameDataTable heap to free memory.
----------------------------------------------------------------------------------------*/
void htCompactDatadrawRootSparseNodeNameDataTables(void)
{
    uint32 elementSize = sizeof(htSparseNodeNameData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    htSparseNodeNameData *toPtr = htDatadrawRoots.SparseNodeNameDataTable;
    htSparseNodeNameData *fromPtr = toPtr;
    htDatadrawRoot DatadrawRoot;
    uint32 size;

    while(fromPtr < htDatadrawRoots.SparseNodeNameDataTable + htUsedDatadrawRootSparseNodeNameDataTable()) {
        DatadrawRoot = *(htDatadrawRoot *)(void *)fromPtr;
        if(DatadrawRoot != htDatadrawRootNull) {
            /* Need to move it to toPtr */
            size = utMax(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            htDatadrawRootSetSparseNodeNameDataTableIndex(DatadrawRoot, toPtr - htDatadrawRoots.SparseNodeNameDataTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((htDatadrawRoot *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    htSetUsedDatadrawRootSparseNodeNameDataTable(toPtr - htDatadrawRoots.SparseNodeNameDataTable);
    htSetFreeDatadrawRootSparseNodeNameDataTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the DatadrawRoot.SparseNodeNameDataTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreDatadrawRootSparseNodeNameDataTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseNodeNameDataTable() - htUsedDatadrawRootSparseNodeNameDataTable();

    if((htFreeDatadrawRootSparseNodeNameDataTable() << 2) > htUsedDatadrawRootSparseNodeNameDataTable()) {
        htCompactDatadrawRootSparseNodeNameDataTables();
        freeSpace = htAllocatedDatadrawRootSparseNodeNameDataTable() - htUsedDatadrawRootSparseNodeNameDataTable();
    }
    if(freeSpace < spaceNeeded) {
        htSetAllocatedDatadrawRootSparseNodeNameDataTable(htAllocatedDatadrawRootSparseNodeNameDataTable() + spaceNeeded - freeSpace +
            (htAllocatedDatadrawRootSparseNodeNameDataTable() >> 1));
        utResizeArray(htDatadrawRoots.SparseNodeNameDataTable, htAllocatedDatadrawRootSparseNodeNameDataTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new DatadrawRoot.SparseNodeNameDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAllocSparseNodeNameDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseNodeNameDataTables)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseNodeNameDataTable() - htUsedDatadrawRootSparseNodeNameDataTable();
    uint32 elementSize = sizeof(htSparseNodeNameData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numSparseNodeNameDataTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) == 0);
#endif
    if(numSparseNodeNameDataTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreDatadrawRootSparseNodeNameDataTables(spaceNeeded);
    }
    htDatadrawRootSetSparseNodeNameDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseNodeNameDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseNodeNameDataTable(DatadrawRoot, numSparseNodeNameDataTables);
    *(htDatadrawRoot *)(void *)(htDatadrawRoots.SparseNodeNameDataTable + htUsedDatadrawRootSparseNodeNameDataTable()) = DatadrawRoot;
    {
        uint32 xDatadrawRoot;
        for(xDatadrawRoot = (uint32)(htDatadrawRootGetSparseNodeNameDataTableIndex(DatadrawRoot)); xDatadrawRoot < htDatadrawRootGetSparseNodeNameDataTableIndex(DatadrawRoot) + numSparseNodeNameDataTables; xDatadrawRoot++) {
            htDatadrawRoots.SparseNodeNameDataTable[xDatadrawRoot] = htSparseNodeNameDataNull;
        }
    }
    htSetUsedDatadrawRootSparseNodeNameDataTable(htUsedDatadrawRootSparseNodeNameDataTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootGetSparseNodeNameDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getDatadrawRootSparseNodeNameDataTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    *numValues = htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot);
    return htDatadrawRootGetSparseNodeNameDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootAllocSparseNodeNameDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocDatadrawRootSparseNodeNameDataTables(
    uint64 objectNumber,
    uint32 numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    htDatadrawRootSetSparseNodeNameDataTableIndex(DatadrawRoot, 0);
    htDatadrawRootSetNumSparseNodeNameDataTable(DatadrawRoot, 0);
    if(numValues == 0) {
        return NULL;
    }
    htDatadrawRootAllocSparseNodeNameDataTables(DatadrawRoot, numValues);
    return htDatadrawRootGetSparseNodeNameDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the DatadrawRoot.SparseNodeNameDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootFreeSparseNodeNameDataTables(
    htDatadrawRoot DatadrawRoot)
{
    uint32 elementSize = sizeof(htSparseNodeNameData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseNodeNameData *dataPtr = htDatadrawRootGetSparseNodeNameDataTables(DatadrawRoot) - usedHeaderSize;

    if(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) == 0) {
        return;
    }
    *(htDatadrawRoot *)(void *)(dataPtr) = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = size;
    htDatadrawRootSetNumSparseNodeNameDataTable(DatadrawRoot, 0);
    htSetFreeDatadrawRootSparseNodeNameDataTable(htFreeDatadrawRootSparseNodeNameDataTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the DatadrawRoot.SparseNodeNameDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootResizeSparseNodeNameDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseNodeNameDataTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(htSparseNodeNameData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numSparseNodeNameDataTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseNodeNameData *dataPtr;

    if(numSparseNodeNameDataTables == 0) {
        if(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) != 0) {
            htDatadrawRootFreeSparseNodeNameDataTables(DatadrawRoot);
        }
        return;
    }
    if(htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) == 0) {
        htDatadrawRootAllocSparseNodeNameDataTables(DatadrawRoot, numSparseNodeNameDataTables);
        return;
    }
    freeSpace = htAllocatedDatadrawRootSparseNodeNameDataTable() - htUsedDatadrawRootSparseNodeNameDataTable();
    if(freeSpace < newSize) {
        allocMoreDatadrawRootSparseNodeNameDataTables(newSize);
    }
    dataPtr = htDatadrawRootGetSparseNodeNameDataTables(DatadrawRoot) - usedHeaderSize;
    memcpy((void *)(htDatadrawRoots.SparseNodeNameDataTable + htUsedDatadrawRootSparseNodeNameDataTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xDatadrawRoot;
            for(xDatadrawRoot = (uint32)(htUsedDatadrawRootSparseNodeNameDataTable() + oldSize); xDatadrawRoot < htUsedDatadrawRootSparseNodeNameDataTable() + oldSize + newSize - oldSize; xDatadrawRoot++) {
                htDatadrawRoots.SparseNodeNameDataTable[xDatadrawRoot] = htSparseNodeNameDataNull;
            }
        }
    }
    *(htDatadrawRoot *)(void *)dataPtr = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = oldSize;
    htSetFreeDatadrawRootSparseNodeNameDataTable(htFreeDatadrawRootSparseNodeNameDataTable() + oldSize);
    htDatadrawRootSetSparseNodeNameDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseNodeNameDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseNodeNameDataTable(DatadrawRoot, numSparseNodeNameDataTables);
    htSetUsedDatadrawRootSparseNodeNameDataTable(htUsedDatadrawRootSparseNodeNameDataTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Compact the DatadrawRoot.SparseNodeXDataTable heap to free memory.
----------------------------------------------------------------------------------------*/
void htCompactDatadrawRootSparseNodeXDataTables(void)
{
    uint32 elementSize = sizeof(htSparseNodeXData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    htSparseNodeXData *toPtr = htDatadrawRoots.SparseNodeXDataTable;
    htSparseNodeXData *fromPtr = toPtr;
    htDatadrawRoot DatadrawRoot;
    uint32 size;

    while(fromPtr < htDatadrawRoots.SparseNodeXDataTable + htUsedDatadrawRootSparseNodeXDataTable()) {
        DatadrawRoot = *(htDatadrawRoot *)(void *)fromPtr;
        if(DatadrawRoot != htDatadrawRootNull) {
            /* Need to move it to toPtr */
            size = utMax(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
            memmove((void *)toPtr, (void *)fromPtr, size*elementSize);
            htDatadrawRootSetSparseNodeXDataTableIndex(DatadrawRoot, toPtr - htDatadrawRoots.SparseNodeXDataTable + usedHeaderSize);
            toPtr += size;
        } else {
            /* Just skip it */
            size = *(uint32 *)(void *)(((htDatadrawRoot *)(void *)fromPtr) + 1);
        }
        fromPtr += size;
    }
    htSetUsedDatadrawRootSparseNodeXDataTable(toPtr - htDatadrawRoots.SparseNodeXDataTable);
    htSetFreeDatadrawRootSparseNodeXDataTable(0);
}

/*----------------------------------------------------------------------------------------
  Allocate more memory for the DatadrawRoot.SparseNodeXDataTable heap.
----------------------------------------------------------------------------------------*/
static void allocMoreDatadrawRootSparseNodeXDataTables(
    uint32 spaceNeeded)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseNodeXDataTable() - htUsedDatadrawRootSparseNodeXDataTable();

    if((htFreeDatadrawRootSparseNodeXDataTable() << 2) > htUsedDatadrawRootSparseNodeXDataTable()) {
        htCompactDatadrawRootSparseNodeXDataTables();
        freeSpace = htAllocatedDatadrawRootSparseNodeXDataTable() - htUsedDatadrawRootSparseNodeXDataTable();
    }
    if(freeSpace < spaceNeeded) {
        htSetAllocatedDatadrawRootSparseNodeXDataTable(htAllocatedDatadrawRootSparseNodeXDataTable() + spaceNeeded - freeSpace +
            (htAllocatedDatadrawRootSparseNodeXDataTable() >> 1));
        utResizeArray(htDatadrawRoots.SparseNodeXDataTable, htAllocatedDatadrawRootSparseNodeXDataTable());
    }
}

/*----------------------------------------------------------------------------------------
  Allocate memory for a new DatadrawRoot.SparseNodeXDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAllocSparseNodeXDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseNodeXDataTables)
{
    uint32 freeSpace = htAllocatedDatadrawRootSparseNodeXDataTable() - htUsedDatadrawRootSparseNodeXDataTable();
    uint32 elementSize = sizeof(htSparseNodeXData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 spaceNeeded = utMax(numSparseNodeXDataTables + usedHeaderSize, freeHeaderSize);

#if defined(DD_DEBUG)
    utAssert(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) == 0);
#endif
    if(numSparseNodeXDataTables == 0) {
        return;
    }
    if(freeSpace < spaceNeeded) {
        allocMoreDatadrawRootSparseNodeXDataTables(spaceNeeded);
    }
    htDatadrawRootSetSparseNodeXDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseNodeXDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseNodeXDataTable(DatadrawRoot, numSparseNodeXDataTables);
    *(htDatadrawRoot *)(void *)(htDatadrawRoots.SparseNodeXDataTable + htUsedDatadrawRootSparseNodeXDataTable()) = DatadrawRoot;
    {
        uint32 xDatadrawRoot;
        for(xDatadrawRoot = (uint32)(htDatadrawRootGetSparseNodeXDataTableIndex(DatadrawRoot)); xDatadrawRoot < htDatadrawRootGetSparseNodeXDataTableIndex(DatadrawRoot) + numSparseNodeXDataTables; xDatadrawRoot++) {
            htDatadrawRoots.SparseNodeXDataTable[xDatadrawRoot] = htSparseNodeXDataNull;
        }
    }
    htSetUsedDatadrawRootSparseNodeXDataTable(htUsedDatadrawRootSparseNodeXDataTable() + spaceNeeded);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootGetSparseNodeXDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *getDatadrawRootSparseNodeXDataTables(
    uint64 objectNumber,
    uint32 *numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    *numValues = htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot);
    return htDatadrawRootGetSparseNodeXDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Wrapper around htDatadrawRootAllocSparseNodeXDataTables for the database manager.
----------------------------------------------------------------------------------------*/
static void *allocDatadrawRootSparseNodeXDataTables(
    uint64 objectNumber,
    uint32 numValues)
{
    htDatadrawRoot DatadrawRoot = htIndex2DatadrawRoot((uint32)objectNumber);

    htDatadrawRootSetSparseNodeXDataTableIndex(DatadrawRoot, 0);
    htDatadrawRootSetNumSparseNodeXDataTable(DatadrawRoot, 0);
    if(numValues == 0) {
        return NULL;
    }
    htDatadrawRootAllocSparseNodeXDataTables(DatadrawRoot, numValues);
    return htDatadrawRootGetSparseNodeXDataTables(DatadrawRoot);
}

/*----------------------------------------------------------------------------------------
  Free memory used by the DatadrawRoot.SparseNodeXDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootFreeSparseNodeXDataTables(
    htDatadrawRoot DatadrawRoot)
{
    uint32 elementSize = sizeof(htSparseNodeXData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 size = utMax(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseNodeXData *dataPtr = htDatadrawRootGetSparseNodeXDataTables(DatadrawRoot) - usedHeaderSize;

    if(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) == 0) {
        return;
    }
    *(htDatadrawRoot *)(void *)(dataPtr) = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = size;
    htDatadrawRootSetNumSparseNodeXDataTable(DatadrawRoot, 0);
    htSetFreeDatadrawRootSparseNodeXDataTable(htFreeDatadrawRootSparseNodeXDataTable() + size);
}

/*----------------------------------------------------------------------------------------
  Resize the DatadrawRoot.SparseNodeXDataTable array.
----------------------------------------------------------------------------------------*/
void htDatadrawRootResizeSparseNodeXDataTables(
    htDatadrawRoot DatadrawRoot,
    uint32 numSparseNodeXDataTables)
{
    uint32 freeSpace;
    uint32 elementSize = sizeof(htSparseNodeXData);
    uint32 usedHeaderSize = (sizeof(htDatadrawRoot) + elementSize - 1)/elementSize;
    uint32 freeHeaderSize = (sizeof(htDatadrawRoot) + sizeof(uint32) + elementSize - 1)/elementSize;
    uint32 newSize = utMax(numSparseNodeXDataTables + usedHeaderSize, freeHeaderSize);
    uint32 oldSize = utMax(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) + usedHeaderSize, freeHeaderSize);
    htSparseNodeXData *dataPtr;

    if(numSparseNodeXDataTables == 0) {
        if(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) != 0) {
            htDatadrawRootFreeSparseNodeXDataTables(DatadrawRoot);
        }
        return;
    }
    if(htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) == 0) {
        htDatadrawRootAllocSparseNodeXDataTables(DatadrawRoot, numSparseNodeXDataTables);
        return;
    }
    freeSpace = htAllocatedDatadrawRootSparseNodeXDataTable() - htUsedDatadrawRootSparseNodeXDataTable();
    if(freeSpace < newSize) {
        allocMoreDatadrawRootSparseNodeXDataTables(newSize);
    }
    dataPtr = htDatadrawRootGetSparseNodeXDataTables(DatadrawRoot) - usedHeaderSize;
    memcpy((void *)(htDatadrawRoots.SparseNodeXDataTable + htUsedDatadrawRootSparseNodeXDataTable()), dataPtr,
        elementSize*utMin(oldSize, newSize));
    if(newSize > oldSize) {
        {
            uint32 xDatadrawRoot;
            for(xDatadrawRoot = (uint32)(htUsedDatadrawRootSparseNodeXDataTable() + oldSize); xDatadrawRoot < htUsedDatadrawRootSparseNodeXDataTable() + oldSize + newSize - oldSize; xDatadrawRoot++) {
                htDatadrawRoots.SparseNodeXDataTable[xDatadrawRoot] = htSparseNodeXDataNull;
            }
        }
    }
    *(htDatadrawRoot *)(void *)dataPtr = htDatadrawRootNull;
    *(uint32 *)(void *)(((htDatadrawRoot *)(void *)dataPtr) + 1) = oldSize;
    htSetFreeDatadrawRootSparseNodeXDataTable(htFreeDatadrawRootSparseNodeXDataTable() + oldSize);
    htDatadrawRootSetSparseNodeXDataTableIndex(DatadrawRoot, htUsedDatadrawRootSparseNodeXDataTable() + usedHeaderSize);
    htDatadrawRootSetNumSparseNodeXDataTable(DatadrawRoot, numSparseNodeXDataTables);
    htSetUsedDatadrawRootSparseNodeXDataTable(htUsedDatadrawRootSparseNodeXDataTable() + newSize);
}

/*----------------------------------------------------------------------------------------
  Copy the properties of DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootCopyProps(
    htDatadrawRoot oldDatadrawRoot,
    htDatadrawRoot newDatadrawRoot)
{
}

static void addDatadrawRootSparseParentGraphNodeDataToHashTable(htDatadrawRoot DatadrawRoot, htSparseParentGraphNodeData _SparseParentGraphNodeData);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeDatadrawRootSparseParentGraphNodeDataHashTable(
    htDatadrawRoot DatadrawRoot)
{
    htSparseParentGraphNodeData _SparseParentGraphNodeData;
    htSparseParentGraphNodeData *SparseParentGraphNodeDatas;
    uint32 numSparseParentGraphNodeDatas = htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) << 1;

    if(numSparseParentGraphNodeDatas == 0) {
        numSparseParentGraphNodeDatas = 2;
        htDatadrawRootAllocSparseParentGraphNodeDataTables(DatadrawRoot, 2);
    } else {
        htDatadrawRootResizeSparseParentGraphNodeDataTables(DatadrawRoot, numSparseParentGraphNodeDatas);
    }
    SparseParentGraphNodeDatas = htDatadrawRootGetSparseParentGraphNodeDataTables(DatadrawRoot);
    /* Zero out the table */
    while(numSparseParentGraphNodeDatas-- != 0) {
        *SparseParentGraphNodeDatas++ = htSparseParentGraphNodeDataNull;
    }
    htDatadrawRootSetNumSparseParentGraphNodeData(DatadrawRoot, 0);
    htForeachDatadrawRootSparseParentGraphNodeData(DatadrawRoot, _SparseParentGraphNodeData) {
        addDatadrawRootSparseParentGraphNodeDataToHashTable(DatadrawRoot, _SparseParentGraphNodeData);
    } htEndDatadrawRootSparseParentGraphNodeData;
}

/*----------------------------------------------------------------------------------------
  Add the SparseParentGraphNodeData to the DatadrawRoot.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addDatadrawRootSparseParentGraphNodeDataToHashTable(
    htDatadrawRoot DatadrawRoot,
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    htSparseParentGraphNodeData nextSparseParentGraphNodeData;
    uint32 index;

    if(htDatadrawRootGetNumSparseParentGraphNodeData(DatadrawRoot) >= htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot)) {
        resizeDatadrawRootSparseParentGraphNodeDataHashTable(DatadrawRoot);
        return;
    }
    index = (htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) - 1) & htGraph2Index(htSparseParentGraphNodeDataGetGraphKey(_SparseParentGraphNodeData));
    nextSparseParentGraphNodeData = htDatadrawRootGetiSparseParentGraphNodeDataTable(DatadrawRoot, index);
    htSparseParentGraphNodeDataSetNextTableDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, nextSparseParentGraphNodeData);
    htDatadrawRootSetiSparseParentGraphNodeDataTable(DatadrawRoot, index, _SparseParentGraphNodeData);
    htDatadrawRootSetNumSparseParentGraphNodeData(DatadrawRoot, htDatadrawRootGetNumSparseParentGraphNodeData(DatadrawRoot) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the SparseParentGraphNodeData from the hash table.
----------------------------------------------------------------------------------------*/
static void removeDatadrawRootSparseParentGraphNodeDataFromHashTable(
   htDatadrawRoot DatadrawRoot,
   htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    uint32 index = (htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) - 1) & htGraph2Index(htSparseParentGraphNodeDataGetGraphKey(_SparseParentGraphNodeData));
    htSparseParentGraphNodeData prevSparseParentGraphNodeData, nextSparseParentGraphNodeData;
    
    nextSparseParentGraphNodeData = htDatadrawRootGetiSparseParentGraphNodeDataTable(DatadrawRoot, index);
    if(nextSparseParentGraphNodeData == _SparseParentGraphNodeData) {
        htDatadrawRootSetiSparseParentGraphNodeDataTable(DatadrawRoot, index, htSparseParentGraphNodeDataGetNextTableDatadrawRootSparseParentGraphNodeData(nextSparseParentGraphNodeData));
    } else {
        do {
            prevSparseParentGraphNodeData = nextSparseParentGraphNodeData;
            nextSparseParentGraphNodeData = htSparseParentGraphNodeDataGetNextTableDatadrawRootSparseParentGraphNodeData(nextSparseParentGraphNodeData);
        } while(nextSparseParentGraphNodeData != _SparseParentGraphNodeData);
        htSparseParentGraphNodeDataSetNextTableDatadrawRootSparseParentGraphNodeData(prevSparseParentGraphNodeData, htSparseParentGraphNodeDataGetNextTableDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData));
    }
    htDatadrawRootSetNumSparseParentGraphNodeData(DatadrawRoot, htDatadrawRootGetNumSparseParentGraphNodeData(DatadrawRoot) - 1);
    htSparseParentGraphNodeDataSetNextTableDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htSparseParentGraphNodeDataNull);
}

/*----------------------------------------------------------------------------------------
  Find the SparseParentGraphNodeData from the DatadrawRoot and its hash key.
----------------------------------------------------------------------------------------*/
htSparseParentGraphNodeData htDatadrawRootFindSparseParentGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htGraph GraphKey)
{
    uint32 mask = htDatadrawRootGetNumSparseParentGraphNodeDataTable(DatadrawRoot) - 1;
    htSparseParentGraphNodeData _SparseParentGraphNodeData;

    if(mask + 1 != 0) {
        _SparseParentGraphNodeData = htDatadrawRootGetiSparseParentGraphNodeDataTable(DatadrawRoot, htGraph2Index(GraphKey) & mask);
        while(_SparseParentGraphNodeData != htSparseParentGraphNodeDataNull) {
            if(htSparseParentGraphNodeDataGetGraphKey(_SparseParentGraphNodeData) == GraphKey) {
                return _SparseParentGraphNodeData;
            }
            _SparseParentGraphNodeData = htSparseParentGraphNodeDataGetNextTableDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData);
        }
    }
    return htSparseParentGraphNodeDataNull;
}

/*----------------------------------------------------------------------------------------
  Add the SparseParentGraphNodeData to the head of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertSparseParentGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseParentGraphNodeData == htSparseParentGraphNodeDataNull) {
        utExit("Non-existent SparseParentGraphNodeData");
    }
#endif
    htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htDatadrawRootGetFirstSparseParentGraphNodeData(DatadrawRoot));
    if(htDatadrawRootGetFirstSparseParentGraphNodeData(DatadrawRoot) != htSparseParentGraphNodeDataNull) {
        htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(htDatadrawRootGetFirstSparseParentGraphNodeData(DatadrawRoot), _SparseParentGraphNodeData);
    }
    htDatadrawRootSetFirstSparseParentGraphNodeData(DatadrawRoot, _SparseParentGraphNodeData);
    htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htSparseParentGraphNodeDataNull);
    if(htDatadrawRootGetLastSparseParentGraphNodeData(DatadrawRoot) == htSparseParentGraphNodeDataNull) {
        htDatadrawRootSetLastSparseParentGraphNodeData(DatadrawRoot, _SparseParentGraphNodeData);
    }
    addDatadrawRootSparseParentGraphNodeDataToHashTable(DatadrawRoot, _SparseParentGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Add the SparseParentGraphNodeData to the end of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAppendSparseParentGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseParentGraphNodeData == htSparseParentGraphNodeDataNull) {
        utExit("Non-existent SparseParentGraphNodeData");
    }
#endif
    htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htDatadrawRootGetLastSparseParentGraphNodeData(DatadrawRoot));
    if(htDatadrawRootGetLastSparseParentGraphNodeData(DatadrawRoot) != htSparseParentGraphNodeDataNull) {
        htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(htDatadrawRootGetLastSparseParentGraphNodeData(DatadrawRoot), _SparseParentGraphNodeData);
    }
    htDatadrawRootSetLastSparseParentGraphNodeData(DatadrawRoot, _SparseParentGraphNodeData);
    htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htSparseParentGraphNodeDataNull);
    if(htDatadrawRootGetFirstSparseParentGraphNodeData(DatadrawRoot) == htSparseParentGraphNodeDataNull) {
        htDatadrawRootSetFirstSparseParentGraphNodeData(DatadrawRoot, _SparseParentGraphNodeData);
    }
    addDatadrawRootSparseParentGraphNodeDataToHashTable(DatadrawRoot, _SparseParentGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Insert the SparseParentGraphNodeData to the DatadrawRoot after the previous SparseParentGraphNodeData.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertAfterSparseParentGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseParentGraphNodeData prevSparseParentGraphNodeData,
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    htSparseParentGraphNodeData nextSparseParentGraphNodeData = htSparseParentGraphNodeDataGetNextDatadrawRootSparseParentGraphNodeData(prevSparseParentGraphNodeData);

#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseParentGraphNodeData == htSparseParentGraphNodeDataNull) {
        utExit("Non-existent SparseParentGraphNodeData");
    }
#endif
    htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, nextSparseParentGraphNodeData);
    htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(prevSparseParentGraphNodeData, _SparseParentGraphNodeData);
    htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, prevSparseParentGraphNodeData);
    if(nextSparseParentGraphNodeData != htSparseParentGraphNodeDataNull) {
        htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(nextSparseParentGraphNodeData, _SparseParentGraphNodeData);
    }
    if(htDatadrawRootGetLastSparseParentGraphNodeData(DatadrawRoot) == prevSparseParentGraphNodeData) {
        htDatadrawRootSetLastSparseParentGraphNodeData(DatadrawRoot, _SparseParentGraphNodeData);
    }
    addDatadrawRootSparseParentGraphNodeDataToHashTable(DatadrawRoot, _SparseParentGraphNodeData);
}

/*----------------------------------------------------------------------------------------
 Remove the SparseParentGraphNodeData from the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootRemoveSparseParentGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    htSparseParentGraphNodeData pSparseParentGraphNodeData, nSparseParentGraphNodeData;

#if defined(DD_DEBUG)
    if(_SparseParentGraphNodeData == htSparseParentGraphNodeDataNull) {
        utExit("Non-existent SparseParentGraphNodeData");
    }
#endif
    nSparseParentGraphNodeData = htSparseParentGraphNodeDataGetNextDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData);
    pSparseParentGraphNodeData = htSparseParentGraphNodeDataGetPrevDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData);
    if(pSparseParentGraphNodeData != htSparseParentGraphNodeDataNull) {
        htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(pSparseParentGraphNodeData, nSparseParentGraphNodeData);
    } else if(htDatadrawRootGetFirstSparseParentGraphNodeData(DatadrawRoot) == _SparseParentGraphNodeData) {
        htDatadrawRootSetFirstSparseParentGraphNodeData(DatadrawRoot, nSparseParentGraphNodeData);
    }
    if(nSparseParentGraphNodeData != htSparseParentGraphNodeDataNull) {
        htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(nSparseParentGraphNodeData, pSparseParentGraphNodeData);
    } else if(htDatadrawRootGetLastSparseParentGraphNodeData(DatadrawRoot) == _SparseParentGraphNodeData) {
        htDatadrawRootSetLastSparseParentGraphNodeData(DatadrawRoot, pSparseParentGraphNodeData);
    }
    htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htSparseParentGraphNodeDataNull);
    htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(_SparseParentGraphNodeData, htSparseParentGraphNodeDataNull);
    removeDatadrawRootSparseParentGraphNodeDataFromHashTable(DatadrawRoot, _SparseParentGraphNodeData);
}

static void addDatadrawRootSparseChildGraphNodeDataToHashTable(htDatadrawRoot DatadrawRoot, htSparseChildGraphNodeData _SparseChildGraphNodeData);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeDatadrawRootSparseChildGraphNodeDataHashTable(
    htDatadrawRoot DatadrawRoot)
{
    htSparseChildGraphNodeData _SparseChildGraphNodeData;
    htSparseChildGraphNodeData *SparseChildGraphNodeDatas;
    uint32 numSparseChildGraphNodeDatas = htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) << 1;

    if(numSparseChildGraphNodeDatas == 0) {
        numSparseChildGraphNodeDatas = 2;
        htDatadrawRootAllocSparseChildGraphNodeDataTables(DatadrawRoot, 2);
    } else {
        htDatadrawRootResizeSparseChildGraphNodeDataTables(DatadrawRoot, numSparseChildGraphNodeDatas);
    }
    SparseChildGraphNodeDatas = htDatadrawRootGetSparseChildGraphNodeDataTables(DatadrawRoot);
    /* Zero out the table */
    while(numSparseChildGraphNodeDatas-- != 0) {
        *SparseChildGraphNodeDatas++ = htSparseChildGraphNodeDataNull;
    }
    htDatadrawRootSetNumSparseChildGraphNodeData(DatadrawRoot, 0);
    htForeachDatadrawRootSparseChildGraphNodeData(DatadrawRoot, _SparseChildGraphNodeData) {
        addDatadrawRootSparseChildGraphNodeDataToHashTable(DatadrawRoot, _SparseChildGraphNodeData);
    } htEndDatadrawRootSparseChildGraphNodeData;
}

/*----------------------------------------------------------------------------------------
  Add the SparseChildGraphNodeData to the DatadrawRoot.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addDatadrawRootSparseChildGraphNodeDataToHashTable(
    htDatadrawRoot DatadrawRoot,
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    htSparseChildGraphNodeData nextSparseChildGraphNodeData;
    uint32 index;

    if(htDatadrawRootGetNumSparseChildGraphNodeData(DatadrawRoot) >= htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot)) {
        resizeDatadrawRootSparseChildGraphNodeDataHashTable(DatadrawRoot);
        return;
    }
    index = (htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) - 1) & htNode2Index(htSparseChildGraphNodeDataGetNodeKey(_SparseChildGraphNodeData));
    nextSparseChildGraphNodeData = htDatadrawRootGetiSparseChildGraphNodeDataTable(DatadrawRoot, index);
    htSparseChildGraphNodeDataSetNextTableDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, nextSparseChildGraphNodeData);
    htDatadrawRootSetiSparseChildGraphNodeDataTable(DatadrawRoot, index, _SparseChildGraphNodeData);
    htDatadrawRootSetNumSparseChildGraphNodeData(DatadrawRoot, htDatadrawRootGetNumSparseChildGraphNodeData(DatadrawRoot) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the SparseChildGraphNodeData from the hash table.
----------------------------------------------------------------------------------------*/
static void removeDatadrawRootSparseChildGraphNodeDataFromHashTable(
   htDatadrawRoot DatadrawRoot,
   htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    uint32 index = (htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) - 1) & htNode2Index(htSparseChildGraphNodeDataGetNodeKey(_SparseChildGraphNodeData));
    htSparseChildGraphNodeData prevSparseChildGraphNodeData, nextSparseChildGraphNodeData;
    
    nextSparseChildGraphNodeData = htDatadrawRootGetiSparseChildGraphNodeDataTable(DatadrawRoot, index);
    if(nextSparseChildGraphNodeData == _SparseChildGraphNodeData) {
        htDatadrawRootSetiSparseChildGraphNodeDataTable(DatadrawRoot, index, htSparseChildGraphNodeDataGetNextTableDatadrawRootSparseChildGraphNodeData(nextSparseChildGraphNodeData));
    } else {
        do {
            prevSparseChildGraphNodeData = nextSparseChildGraphNodeData;
            nextSparseChildGraphNodeData = htSparseChildGraphNodeDataGetNextTableDatadrawRootSparseChildGraphNodeData(nextSparseChildGraphNodeData);
        } while(nextSparseChildGraphNodeData != _SparseChildGraphNodeData);
        htSparseChildGraphNodeDataSetNextTableDatadrawRootSparseChildGraphNodeData(prevSparseChildGraphNodeData, htSparseChildGraphNodeDataGetNextTableDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData));
    }
    htDatadrawRootSetNumSparseChildGraphNodeData(DatadrawRoot, htDatadrawRootGetNumSparseChildGraphNodeData(DatadrawRoot) - 1);
    htSparseChildGraphNodeDataSetNextTableDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htSparseChildGraphNodeDataNull);
}

/*----------------------------------------------------------------------------------------
  Find the SparseChildGraphNodeData from the DatadrawRoot and its hash key.
----------------------------------------------------------------------------------------*/
htSparseChildGraphNodeData htDatadrawRootFindSparseChildGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htNode NodeKey)
{
    uint32 mask = htDatadrawRootGetNumSparseChildGraphNodeDataTable(DatadrawRoot) - 1;
    htSparseChildGraphNodeData _SparseChildGraphNodeData;

    if(mask + 1 != 0) {
        _SparseChildGraphNodeData = htDatadrawRootGetiSparseChildGraphNodeDataTable(DatadrawRoot, htNode2Index(NodeKey) & mask);
        while(_SparseChildGraphNodeData != htSparseChildGraphNodeDataNull) {
            if(htSparseChildGraphNodeDataGetNodeKey(_SparseChildGraphNodeData) == NodeKey) {
                return _SparseChildGraphNodeData;
            }
            _SparseChildGraphNodeData = htSparseChildGraphNodeDataGetNextTableDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData);
        }
    }
    return htSparseChildGraphNodeDataNull;
}

/*----------------------------------------------------------------------------------------
  Add the SparseChildGraphNodeData to the head of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertSparseChildGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseChildGraphNodeData == htSparseChildGraphNodeDataNull) {
        utExit("Non-existent SparseChildGraphNodeData");
    }
#endif
    htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htDatadrawRootGetFirstSparseChildGraphNodeData(DatadrawRoot));
    if(htDatadrawRootGetFirstSparseChildGraphNodeData(DatadrawRoot) != htSparseChildGraphNodeDataNull) {
        htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(htDatadrawRootGetFirstSparseChildGraphNodeData(DatadrawRoot), _SparseChildGraphNodeData);
    }
    htDatadrawRootSetFirstSparseChildGraphNodeData(DatadrawRoot, _SparseChildGraphNodeData);
    htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htSparseChildGraphNodeDataNull);
    if(htDatadrawRootGetLastSparseChildGraphNodeData(DatadrawRoot) == htSparseChildGraphNodeDataNull) {
        htDatadrawRootSetLastSparseChildGraphNodeData(DatadrawRoot, _SparseChildGraphNodeData);
    }
    addDatadrawRootSparseChildGraphNodeDataToHashTable(DatadrawRoot, _SparseChildGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Add the SparseChildGraphNodeData to the end of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAppendSparseChildGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseChildGraphNodeData == htSparseChildGraphNodeDataNull) {
        utExit("Non-existent SparseChildGraphNodeData");
    }
#endif
    htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htDatadrawRootGetLastSparseChildGraphNodeData(DatadrawRoot));
    if(htDatadrawRootGetLastSparseChildGraphNodeData(DatadrawRoot) != htSparseChildGraphNodeDataNull) {
        htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(htDatadrawRootGetLastSparseChildGraphNodeData(DatadrawRoot), _SparseChildGraphNodeData);
    }
    htDatadrawRootSetLastSparseChildGraphNodeData(DatadrawRoot, _SparseChildGraphNodeData);
    htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htSparseChildGraphNodeDataNull);
    if(htDatadrawRootGetFirstSparseChildGraphNodeData(DatadrawRoot) == htSparseChildGraphNodeDataNull) {
        htDatadrawRootSetFirstSparseChildGraphNodeData(DatadrawRoot, _SparseChildGraphNodeData);
    }
    addDatadrawRootSparseChildGraphNodeDataToHashTable(DatadrawRoot, _SparseChildGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Insert the SparseChildGraphNodeData to the DatadrawRoot after the previous SparseChildGraphNodeData.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertAfterSparseChildGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseChildGraphNodeData prevSparseChildGraphNodeData,
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    htSparseChildGraphNodeData nextSparseChildGraphNodeData = htSparseChildGraphNodeDataGetNextDatadrawRootSparseChildGraphNodeData(prevSparseChildGraphNodeData);

#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseChildGraphNodeData == htSparseChildGraphNodeDataNull) {
        utExit("Non-existent SparseChildGraphNodeData");
    }
#endif
    htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, nextSparseChildGraphNodeData);
    htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(prevSparseChildGraphNodeData, _SparseChildGraphNodeData);
    htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, prevSparseChildGraphNodeData);
    if(nextSparseChildGraphNodeData != htSparseChildGraphNodeDataNull) {
        htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(nextSparseChildGraphNodeData, _SparseChildGraphNodeData);
    }
    if(htDatadrawRootGetLastSparseChildGraphNodeData(DatadrawRoot) == prevSparseChildGraphNodeData) {
        htDatadrawRootSetLastSparseChildGraphNodeData(DatadrawRoot, _SparseChildGraphNodeData);
    }
    addDatadrawRootSparseChildGraphNodeDataToHashTable(DatadrawRoot, _SparseChildGraphNodeData);
}

/*----------------------------------------------------------------------------------------
 Remove the SparseChildGraphNodeData from the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootRemoveSparseChildGraphNodeData(
    htDatadrawRoot DatadrawRoot,
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    htSparseChildGraphNodeData pSparseChildGraphNodeData, nSparseChildGraphNodeData;

#if defined(DD_DEBUG)
    if(_SparseChildGraphNodeData == htSparseChildGraphNodeDataNull) {
        utExit("Non-existent SparseChildGraphNodeData");
    }
#endif
    nSparseChildGraphNodeData = htSparseChildGraphNodeDataGetNextDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData);
    pSparseChildGraphNodeData = htSparseChildGraphNodeDataGetPrevDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData);
    if(pSparseChildGraphNodeData != htSparseChildGraphNodeDataNull) {
        htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(pSparseChildGraphNodeData, nSparseChildGraphNodeData);
    } else if(htDatadrawRootGetFirstSparseChildGraphNodeData(DatadrawRoot) == _SparseChildGraphNodeData) {
        htDatadrawRootSetFirstSparseChildGraphNodeData(DatadrawRoot, nSparseChildGraphNodeData);
    }
    if(nSparseChildGraphNodeData != htSparseChildGraphNodeDataNull) {
        htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(nSparseChildGraphNodeData, pSparseChildGraphNodeData);
    } else if(htDatadrawRootGetLastSparseChildGraphNodeData(DatadrawRoot) == _SparseChildGraphNodeData) {
        htDatadrawRootSetLastSparseChildGraphNodeData(DatadrawRoot, pSparseChildGraphNodeData);
    }
    htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htSparseChildGraphNodeDataNull);
    htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(_SparseChildGraphNodeData, htSparseChildGraphNodeDataNull);
    removeDatadrawRootSparseChildGraphNodeDataFromHashTable(DatadrawRoot, _SparseChildGraphNodeData);
}

static void addDatadrawRootSparseNodeNameDataToHashTable(htDatadrawRoot DatadrawRoot, htSparseNodeNameData _SparseNodeNameData);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeDatadrawRootSparseNodeNameDataHashTable(
    htDatadrawRoot DatadrawRoot)
{
    htSparseNodeNameData _SparseNodeNameData;
    htSparseNodeNameData *SparseNodeNameDatas;
    uint32 numSparseNodeNameDatas = htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) << 1;

    if(numSparseNodeNameDatas == 0) {
        numSparseNodeNameDatas = 2;
        htDatadrawRootAllocSparseNodeNameDataTables(DatadrawRoot, 2);
    } else {
        htDatadrawRootResizeSparseNodeNameDataTables(DatadrawRoot, numSparseNodeNameDatas);
    }
    SparseNodeNameDatas = htDatadrawRootGetSparseNodeNameDataTables(DatadrawRoot);
    /* Zero out the table */
    while(numSparseNodeNameDatas-- != 0) {
        *SparseNodeNameDatas++ = htSparseNodeNameDataNull;
    }
    htDatadrawRootSetNumSparseNodeNameData(DatadrawRoot, 0);
    htForeachDatadrawRootSparseNodeNameData(DatadrawRoot, _SparseNodeNameData) {
        addDatadrawRootSparseNodeNameDataToHashTable(DatadrawRoot, _SparseNodeNameData);
    } htEndDatadrawRootSparseNodeNameData;
}

/*----------------------------------------------------------------------------------------
  Add the SparseNodeNameData to the DatadrawRoot.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addDatadrawRootSparseNodeNameDataToHashTable(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeNameData _SparseNodeNameData)
{
    htSparseNodeNameData nextSparseNodeNameData;
    uint32 index;

    if(htDatadrawRootGetNumSparseNodeNameData(DatadrawRoot) >= htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot)) {
        resizeDatadrawRootSparseNodeNameDataHashTable(DatadrawRoot);
        return;
    }
    index = (htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) - 1) & htNode2Index(htSparseNodeNameDataGetNodeKey(_SparseNodeNameData));
    nextSparseNodeNameData = htDatadrawRootGetiSparseNodeNameDataTable(DatadrawRoot, index);
    htSparseNodeNameDataSetNextTableDatadrawRootSparseNodeNameData(_SparseNodeNameData, nextSparseNodeNameData);
    htDatadrawRootSetiSparseNodeNameDataTable(DatadrawRoot, index, _SparseNodeNameData);
    htDatadrawRootSetNumSparseNodeNameData(DatadrawRoot, htDatadrawRootGetNumSparseNodeNameData(DatadrawRoot) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the SparseNodeNameData from the hash table.
----------------------------------------------------------------------------------------*/
static void removeDatadrawRootSparseNodeNameDataFromHashTable(
   htDatadrawRoot DatadrawRoot,
   htSparseNodeNameData _SparseNodeNameData)
{
    uint32 index = (htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) - 1) & htNode2Index(htSparseNodeNameDataGetNodeKey(_SparseNodeNameData));
    htSparseNodeNameData prevSparseNodeNameData, nextSparseNodeNameData;
    
    nextSparseNodeNameData = htDatadrawRootGetiSparseNodeNameDataTable(DatadrawRoot, index);
    if(nextSparseNodeNameData == _SparseNodeNameData) {
        htDatadrawRootSetiSparseNodeNameDataTable(DatadrawRoot, index, htSparseNodeNameDataGetNextTableDatadrawRootSparseNodeNameData(nextSparseNodeNameData));
    } else {
        do {
            prevSparseNodeNameData = nextSparseNodeNameData;
            nextSparseNodeNameData = htSparseNodeNameDataGetNextTableDatadrawRootSparseNodeNameData(nextSparseNodeNameData);
        } while(nextSparseNodeNameData != _SparseNodeNameData);
        htSparseNodeNameDataSetNextTableDatadrawRootSparseNodeNameData(prevSparseNodeNameData, htSparseNodeNameDataGetNextTableDatadrawRootSparseNodeNameData(_SparseNodeNameData));
    }
    htDatadrawRootSetNumSparseNodeNameData(DatadrawRoot, htDatadrawRootGetNumSparseNodeNameData(DatadrawRoot) - 1);
    htSparseNodeNameDataSetNextTableDatadrawRootSparseNodeNameData(_SparseNodeNameData, htSparseNodeNameDataNull);
}

/*----------------------------------------------------------------------------------------
  Find the SparseNodeNameData from the DatadrawRoot and its hash key.
----------------------------------------------------------------------------------------*/
htSparseNodeNameData htDatadrawRootFindSparseNodeNameData(
    htDatadrawRoot DatadrawRoot,
    htNode NodeKey)
{
    uint32 mask = htDatadrawRootGetNumSparseNodeNameDataTable(DatadrawRoot) - 1;
    htSparseNodeNameData _SparseNodeNameData;

    if(mask + 1 != 0) {
        _SparseNodeNameData = htDatadrawRootGetiSparseNodeNameDataTable(DatadrawRoot, htNode2Index(NodeKey) & mask);
        while(_SparseNodeNameData != htSparseNodeNameDataNull) {
            if(htSparseNodeNameDataGetNodeKey(_SparseNodeNameData) == NodeKey) {
                return _SparseNodeNameData;
            }
            _SparseNodeNameData = htSparseNodeNameDataGetNextTableDatadrawRootSparseNodeNameData(_SparseNodeNameData);
        }
    }
    return htSparseNodeNameDataNull;
}

/*----------------------------------------------------------------------------------------
  Add the SparseNodeNameData to the head of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertSparseNodeNameData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeNameData _SparseNodeNameData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseNodeNameData == htSparseNodeNameDataNull) {
        utExit("Non-existent SparseNodeNameData");
    }
#endif
    htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(_SparseNodeNameData, htDatadrawRootGetFirstSparseNodeNameData(DatadrawRoot));
    if(htDatadrawRootGetFirstSparseNodeNameData(DatadrawRoot) != htSparseNodeNameDataNull) {
        htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(htDatadrawRootGetFirstSparseNodeNameData(DatadrawRoot), _SparseNodeNameData);
    }
    htDatadrawRootSetFirstSparseNodeNameData(DatadrawRoot, _SparseNodeNameData);
    htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(_SparseNodeNameData, htSparseNodeNameDataNull);
    if(htDatadrawRootGetLastSparseNodeNameData(DatadrawRoot) == htSparseNodeNameDataNull) {
        htDatadrawRootSetLastSparseNodeNameData(DatadrawRoot, _SparseNodeNameData);
    }
    addDatadrawRootSparseNodeNameDataToHashTable(DatadrawRoot, _SparseNodeNameData);
}

/*----------------------------------------------------------------------------------------
  Add the SparseNodeNameData to the end of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAppendSparseNodeNameData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeNameData _SparseNodeNameData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseNodeNameData == htSparseNodeNameDataNull) {
        utExit("Non-existent SparseNodeNameData");
    }
#endif
    htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(_SparseNodeNameData, htDatadrawRootGetLastSparseNodeNameData(DatadrawRoot));
    if(htDatadrawRootGetLastSparseNodeNameData(DatadrawRoot) != htSparseNodeNameDataNull) {
        htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(htDatadrawRootGetLastSparseNodeNameData(DatadrawRoot), _SparseNodeNameData);
    }
    htDatadrawRootSetLastSparseNodeNameData(DatadrawRoot, _SparseNodeNameData);
    htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(_SparseNodeNameData, htSparseNodeNameDataNull);
    if(htDatadrawRootGetFirstSparseNodeNameData(DatadrawRoot) == htSparseNodeNameDataNull) {
        htDatadrawRootSetFirstSparseNodeNameData(DatadrawRoot, _SparseNodeNameData);
    }
    addDatadrawRootSparseNodeNameDataToHashTable(DatadrawRoot, _SparseNodeNameData);
}

/*----------------------------------------------------------------------------------------
  Insert the SparseNodeNameData to the DatadrawRoot after the previous SparseNodeNameData.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertAfterSparseNodeNameData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeNameData prevSparseNodeNameData,
    htSparseNodeNameData _SparseNodeNameData)
{
    htSparseNodeNameData nextSparseNodeNameData = htSparseNodeNameDataGetNextDatadrawRootSparseNodeNameData(prevSparseNodeNameData);

#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseNodeNameData == htSparseNodeNameDataNull) {
        utExit("Non-existent SparseNodeNameData");
    }
#endif
    htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(_SparseNodeNameData, nextSparseNodeNameData);
    htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(prevSparseNodeNameData, _SparseNodeNameData);
    htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(_SparseNodeNameData, prevSparseNodeNameData);
    if(nextSparseNodeNameData != htSparseNodeNameDataNull) {
        htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(nextSparseNodeNameData, _SparseNodeNameData);
    }
    if(htDatadrawRootGetLastSparseNodeNameData(DatadrawRoot) == prevSparseNodeNameData) {
        htDatadrawRootSetLastSparseNodeNameData(DatadrawRoot, _SparseNodeNameData);
    }
    addDatadrawRootSparseNodeNameDataToHashTable(DatadrawRoot, _SparseNodeNameData);
}

/*----------------------------------------------------------------------------------------
 Remove the SparseNodeNameData from the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootRemoveSparseNodeNameData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeNameData _SparseNodeNameData)
{
    htSparseNodeNameData pSparseNodeNameData, nSparseNodeNameData;

#if defined(DD_DEBUG)
    if(_SparseNodeNameData == htSparseNodeNameDataNull) {
        utExit("Non-existent SparseNodeNameData");
    }
#endif
    nSparseNodeNameData = htSparseNodeNameDataGetNextDatadrawRootSparseNodeNameData(_SparseNodeNameData);
    pSparseNodeNameData = htSparseNodeNameDataGetPrevDatadrawRootSparseNodeNameData(_SparseNodeNameData);
    if(pSparseNodeNameData != htSparseNodeNameDataNull) {
        htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(pSparseNodeNameData, nSparseNodeNameData);
    } else if(htDatadrawRootGetFirstSparseNodeNameData(DatadrawRoot) == _SparseNodeNameData) {
        htDatadrawRootSetFirstSparseNodeNameData(DatadrawRoot, nSparseNodeNameData);
    }
    if(nSparseNodeNameData != htSparseNodeNameDataNull) {
        htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(nSparseNodeNameData, pSparseNodeNameData);
    } else if(htDatadrawRootGetLastSparseNodeNameData(DatadrawRoot) == _SparseNodeNameData) {
        htDatadrawRootSetLastSparseNodeNameData(DatadrawRoot, pSparseNodeNameData);
    }
    htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(_SparseNodeNameData, htSparseNodeNameDataNull);
    htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(_SparseNodeNameData, htSparseNodeNameDataNull);
    removeDatadrawRootSparseNodeNameDataFromHashTable(DatadrawRoot, _SparseNodeNameData);
}

static void addDatadrawRootSparseNodeXDataToHashTable(htDatadrawRoot DatadrawRoot, htSparseNodeXData _SparseNodeXData);
/*----------------------------------------------------------------------------------------
  Increase the size of the hash table.
----------------------------------------------------------------------------------------*/
static void resizeDatadrawRootSparseNodeXDataHashTable(
    htDatadrawRoot DatadrawRoot)
{
    htSparseNodeXData _SparseNodeXData;
    htSparseNodeXData *SparseNodeXDatas;
    uint32 numSparseNodeXDatas = htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) << 1;

    if(numSparseNodeXDatas == 0) {
        numSparseNodeXDatas = 2;
        htDatadrawRootAllocSparseNodeXDataTables(DatadrawRoot, 2);
    } else {
        htDatadrawRootResizeSparseNodeXDataTables(DatadrawRoot, numSparseNodeXDatas);
    }
    SparseNodeXDatas = htDatadrawRootGetSparseNodeXDataTables(DatadrawRoot);
    /* Zero out the table */
    while(numSparseNodeXDatas-- != 0) {
        *SparseNodeXDatas++ = htSparseNodeXDataNull;
    }
    htDatadrawRootSetNumSparseNodeXData(DatadrawRoot, 0);
    htForeachDatadrawRootSparseNodeXData(DatadrawRoot, _SparseNodeXData) {
        addDatadrawRootSparseNodeXDataToHashTable(DatadrawRoot, _SparseNodeXData);
    } htEndDatadrawRootSparseNodeXData;
}

/*----------------------------------------------------------------------------------------
  Add the SparseNodeXData to the DatadrawRoot.  If the table is near full, build a new one twice
  as big, delete the old one, and return the new one.
----------------------------------------------------------------------------------------*/
static void addDatadrawRootSparseNodeXDataToHashTable(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeXData _SparseNodeXData)
{
    htSparseNodeXData nextSparseNodeXData;
    uint32 index;

    if(htDatadrawRootGetNumSparseNodeXData(DatadrawRoot) >= htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot)) {
        resizeDatadrawRootSparseNodeXDataHashTable(DatadrawRoot);
        return;
    }
    index = (htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) - 1) & htNode2Index(htSparseNodeXDataGetNodeKey(_SparseNodeXData));
    nextSparseNodeXData = htDatadrawRootGetiSparseNodeXDataTable(DatadrawRoot, index);
    htSparseNodeXDataSetNextTableDatadrawRootSparseNodeXData(_SparseNodeXData, nextSparseNodeXData);
    htDatadrawRootSetiSparseNodeXDataTable(DatadrawRoot, index, _SparseNodeXData);
    htDatadrawRootSetNumSparseNodeXData(DatadrawRoot, htDatadrawRootGetNumSparseNodeXData(DatadrawRoot) + 1);
}

/*----------------------------------------------------------------------------------------
  Remove the SparseNodeXData from the hash table.
----------------------------------------------------------------------------------------*/
static void removeDatadrawRootSparseNodeXDataFromHashTable(
   htDatadrawRoot DatadrawRoot,
   htSparseNodeXData _SparseNodeXData)
{
    uint32 index = (htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) - 1) & htNode2Index(htSparseNodeXDataGetNodeKey(_SparseNodeXData));
    htSparseNodeXData prevSparseNodeXData, nextSparseNodeXData;
    
    nextSparseNodeXData = htDatadrawRootGetiSparseNodeXDataTable(DatadrawRoot, index);
    if(nextSparseNodeXData == _SparseNodeXData) {
        htDatadrawRootSetiSparseNodeXDataTable(DatadrawRoot, index, htSparseNodeXDataGetNextTableDatadrawRootSparseNodeXData(nextSparseNodeXData));
    } else {
        do {
            prevSparseNodeXData = nextSparseNodeXData;
            nextSparseNodeXData = htSparseNodeXDataGetNextTableDatadrawRootSparseNodeXData(nextSparseNodeXData);
        } while(nextSparseNodeXData != _SparseNodeXData);
        htSparseNodeXDataSetNextTableDatadrawRootSparseNodeXData(prevSparseNodeXData, htSparseNodeXDataGetNextTableDatadrawRootSparseNodeXData(_SparseNodeXData));
    }
    htDatadrawRootSetNumSparseNodeXData(DatadrawRoot, htDatadrawRootGetNumSparseNodeXData(DatadrawRoot) - 1);
    htSparseNodeXDataSetNextTableDatadrawRootSparseNodeXData(_SparseNodeXData, htSparseNodeXDataNull);
}

/*----------------------------------------------------------------------------------------
  Find the SparseNodeXData from the DatadrawRoot and its hash key.
----------------------------------------------------------------------------------------*/
htSparseNodeXData htDatadrawRootFindSparseNodeXData(
    htDatadrawRoot DatadrawRoot,
    htNode NodeKey)
{
    uint32 mask = htDatadrawRootGetNumSparseNodeXDataTable(DatadrawRoot) - 1;
    htSparseNodeXData _SparseNodeXData;

    if(mask + 1 != 0) {
        _SparseNodeXData = htDatadrawRootGetiSparseNodeXDataTable(DatadrawRoot, htNode2Index(NodeKey) & mask);
        while(_SparseNodeXData != htSparseNodeXDataNull) {
            if(htSparseNodeXDataGetNodeKey(_SparseNodeXData) == NodeKey) {
                return _SparseNodeXData;
            }
            _SparseNodeXData = htSparseNodeXDataGetNextTableDatadrawRootSparseNodeXData(_SparseNodeXData);
        }
    }
    return htSparseNodeXDataNull;
}

/*----------------------------------------------------------------------------------------
  Add the SparseNodeXData to the head of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertSparseNodeXData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeXData _SparseNodeXData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseNodeXData == htSparseNodeXDataNull) {
        utExit("Non-existent SparseNodeXData");
    }
#endif
    htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(_SparseNodeXData, htDatadrawRootGetFirstSparseNodeXData(DatadrawRoot));
    if(htDatadrawRootGetFirstSparseNodeXData(DatadrawRoot) != htSparseNodeXDataNull) {
        htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(htDatadrawRootGetFirstSparseNodeXData(DatadrawRoot), _SparseNodeXData);
    }
    htDatadrawRootSetFirstSparseNodeXData(DatadrawRoot, _SparseNodeXData);
    htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(_SparseNodeXData, htSparseNodeXDataNull);
    if(htDatadrawRootGetLastSparseNodeXData(DatadrawRoot) == htSparseNodeXDataNull) {
        htDatadrawRootSetLastSparseNodeXData(DatadrawRoot, _SparseNodeXData);
    }
    addDatadrawRootSparseNodeXDataToHashTable(DatadrawRoot, _SparseNodeXData);
}

/*----------------------------------------------------------------------------------------
  Add the SparseNodeXData to the end of the list on the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootAppendSparseNodeXData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeXData _SparseNodeXData)
{
#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseNodeXData == htSparseNodeXDataNull) {
        utExit("Non-existent SparseNodeXData");
    }
#endif
    htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(_SparseNodeXData, htDatadrawRootGetLastSparseNodeXData(DatadrawRoot));
    if(htDatadrawRootGetLastSparseNodeXData(DatadrawRoot) != htSparseNodeXDataNull) {
        htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(htDatadrawRootGetLastSparseNodeXData(DatadrawRoot), _SparseNodeXData);
    }
    htDatadrawRootSetLastSparseNodeXData(DatadrawRoot, _SparseNodeXData);
    htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(_SparseNodeXData, htSparseNodeXDataNull);
    if(htDatadrawRootGetFirstSparseNodeXData(DatadrawRoot) == htSparseNodeXDataNull) {
        htDatadrawRootSetFirstSparseNodeXData(DatadrawRoot, _SparseNodeXData);
    }
    addDatadrawRootSparseNodeXDataToHashTable(DatadrawRoot, _SparseNodeXData);
}

/*----------------------------------------------------------------------------------------
  Insert the SparseNodeXData to the DatadrawRoot after the previous SparseNodeXData.
----------------------------------------------------------------------------------------*/
void htDatadrawRootInsertAfterSparseNodeXData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeXData prevSparseNodeXData,
    htSparseNodeXData _SparseNodeXData)
{
    htSparseNodeXData nextSparseNodeXData = htSparseNodeXDataGetNextDatadrawRootSparseNodeXData(prevSparseNodeXData);

#if defined(DD_DEBUG)
    if(DatadrawRoot == htDatadrawRootNull) {
        utExit("Non-existent DatadrawRoot");
    }
    if(_SparseNodeXData == htSparseNodeXDataNull) {
        utExit("Non-existent SparseNodeXData");
    }
#endif
    htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(_SparseNodeXData, nextSparseNodeXData);
    htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(prevSparseNodeXData, _SparseNodeXData);
    htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(_SparseNodeXData, prevSparseNodeXData);
    if(nextSparseNodeXData != htSparseNodeXDataNull) {
        htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(nextSparseNodeXData, _SparseNodeXData);
    }
    if(htDatadrawRootGetLastSparseNodeXData(DatadrawRoot) == prevSparseNodeXData) {
        htDatadrawRootSetLastSparseNodeXData(DatadrawRoot, _SparseNodeXData);
    }
    addDatadrawRootSparseNodeXDataToHashTable(DatadrawRoot, _SparseNodeXData);
}

/*----------------------------------------------------------------------------------------
 Remove the SparseNodeXData from the DatadrawRoot.
----------------------------------------------------------------------------------------*/
void htDatadrawRootRemoveSparseNodeXData(
    htDatadrawRoot DatadrawRoot,
    htSparseNodeXData _SparseNodeXData)
{
    htSparseNodeXData pSparseNodeXData, nSparseNodeXData;

#if defined(DD_DEBUG)
    if(_SparseNodeXData == htSparseNodeXDataNull) {
        utExit("Non-existent SparseNodeXData");
    }
#endif
    nSparseNodeXData = htSparseNodeXDataGetNextDatadrawRootSparseNodeXData(_SparseNodeXData);
    pSparseNodeXData = htSparseNodeXDataGetPrevDatadrawRootSparseNodeXData(_SparseNodeXData);
    if(pSparseNodeXData != htSparseNodeXDataNull) {
        htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(pSparseNodeXData, nSparseNodeXData);
    } else if(htDatadrawRootGetFirstSparseNodeXData(DatadrawRoot) == _SparseNodeXData) {
        htDatadrawRootSetFirstSparseNodeXData(DatadrawRoot, nSparseNodeXData);
    }
    if(nSparseNodeXData != htSparseNodeXDataNull) {
        htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(nSparseNodeXData, pSparseNodeXData);
    } else if(htDatadrawRootGetLastSparseNodeXData(DatadrawRoot) == _SparseNodeXData) {
        htDatadrawRootSetLastSparseNodeXData(DatadrawRoot, pSparseNodeXData);
    }
    htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(_SparseNodeXData, htSparseNodeXDataNull);
    htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(_SparseNodeXData, htSparseNodeXDataNull);
    removeDatadrawRootSparseNodeXDataFromHashTable(DatadrawRoot, _SparseNodeXData);
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowDatadrawRoot(
    htDatadrawRoot DatadrawRoot)
{
    utDatabaseShowObject("ht", "DatadrawRoot", htDatadrawRoot2Index(DatadrawRoot));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy SparseParentGraphNodeData including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void htSparseParentGraphNodeDataDestroy(
    htSparseParentGraphNodeData SparseParentGraphNodeData)
{
    if(htSparseParentGraphNodeDataDestructorCallback != NULL) {
        htSparseParentGraphNodeDataDestructorCallback(SparseParentGraphNodeData);
    }
    htSparseParentGraphNodeDataFree(SparseParentGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocSparseParentGraphNodeData(void)
{
    htSparseParentGraphNodeData SparseParentGraphNodeData = htSparseParentGraphNodeDataAlloc();

    return htSparseParentGraphNodeData2Index(SparseParentGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroySparseParentGraphNodeData(
    uint64 objectIndex)
{
    htSparseParentGraphNodeDataDestroy(htIndex2SparseParentGraphNodeData((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of SparseParentGraphNodeData.
----------------------------------------------------------------------------------------*/
static void allocSparseParentGraphNodeDatas(void)
{
    htSetAllocatedSparseParentGraphNodeData(2);
    htSetUsedSparseParentGraphNodeData(0);
    htSetFirstFreeSparseParentGraphNodeData(htSparseParentGraphNodeDataNull);
    htSparseParentGraphNodeDatas.GraphKey = utNewA(htGraph, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.FirstNode = utNewA(htNode, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.LastNode = utNewA(htNode, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.NumNode = utNewA(uint32, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.NodeTableIndex = utNewA(uint32, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.NumNodeTable = utNewA(uint32, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.NextDatadrawRootSparseParentGraphNodeData = utNewA(htSparseParentGraphNodeData, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.PrevDatadrawRootSparseParentGraphNodeData = utNewA(htSparseParentGraphNodeData, (htAllocatedSparseParentGraphNodeData()));
    htSparseParentGraphNodeDatas.NextTableDatadrawRootSparseParentGraphNodeData = utNewA(htSparseParentGraphNodeData, (htAllocatedSparseParentGraphNodeData()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class SparseParentGraphNodeData.
----------------------------------------------------------------------------------------*/
static void reallocSparseParentGraphNodeDatas(
    uint32 newSize)
{
    utResizeArray(htSparseParentGraphNodeDatas.GraphKey, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.FirstNode, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.LastNode, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.NumNode, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.NodeTableIndex, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.NumNodeTable, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.NextDatadrawRootSparseParentGraphNodeData, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.PrevDatadrawRootSparseParentGraphNodeData, (newSize));
    utResizeArray(htSparseParentGraphNodeDatas.NextTableDatadrawRootSparseParentGraphNodeData, (newSize));
    htSetAllocatedSparseParentGraphNodeData(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more SparseParentGraphNodeDatas.
----------------------------------------------------------------------------------------*/
void htSparseParentGraphNodeDataAllocMore(void)
{
    reallocSparseParentGraphNodeDatas((uint32)(htAllocatedSparseParentGraphNodeData() + (htAllocatedSparseParentGraphNodeData() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of SparseParentGraphNodeData.
----------------------------------------------------------------------------------------*/
void htSparseParentGraphNodeDataCopyProps(
    htSparseParentGraphNodeData oldSparseParentGraphNodeData,
    htSparseParentGraphNodeData newSparseParentGraphNodeData)
{
    htSparseParentGraphNodeDataSetNumNode(newSparseParentGraphNodeData, htSparseParentGraphNodeDataGetNumNode(oldSparseParentGraphNodeData));
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowSparseParentGraphNodeData(
    htSparseParentGraphNodeData SparseParentGraphNodeData)
{
    utDatabaseShowObject("ht", "SparseParentGraphNodeData", htSparseParentGraphNodeData2Index(SparseParentGraphNodeData));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy SparseChildGraphNodeData including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void htSparseChildGraphNodeDataDestroy(
    htSparseChildGraphNodeData SparseChildGraphNodeData)
{
    if(htSparseChildGraphNodeDataDestructorCallback != NULL) {
        htSparseChildGraphNodeDataDestructorCallback(SparseChildGraphNodeData);
    }
    htSparseChildGraphNodeDataFree(SparseChildGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocSparseChildGraphNodeData(void)
{
    htSparseChildGraphNodeData SparseChildGraphNodeData = htSparseChildGraphNodeDataAlloc();

    return htSparseChildGraphNodeData2Index(SparseChildGraphNodeData);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroySparseChildGraphNodeData(
    uint64 objectIndex)
{
    htSparseChildGraphNodeDataDestroy(htIndex2SparseChildGraphNodeData((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of SparseChildGraphNodeData.
----------------------------------------------------------------------------------------*/
static void allocSparseChildGraphNodeDatas(void)
{
    htSetAllocatedSparseChildGraphNodeData(2);
    htSetUsedSparseChildGraphNodeData(0);
    htSetFirstFreeSparseChildGraphNodeData(htSparseChildGraphNodeDataNull);
    htSparseChildGraphNodeDatas.NodeKey = utNewA(htNode, (htAllocatedSparseChildGraphNodeData()));
    htSparseChildGraphNodeDatas.NextGraphNode = utNewA(htNode, (htAllocatedSparseChildGraphNodeData()));
    htSparseChildGraphNodeDatas.PrevGraphNode = utNewA(htNode, (htAllocatedSparseChildGraphNodeData()));
    htSparseChildGraphNodeDatas.NextTableGraphNode = utNewA(htNode, (htAllocatedSparseChildGraphNodeData()));
    htSparseChildGraphNodeDatas.NextDatadrawRootSparseChildGraphNodeData = utNewA(htSparseChildGraphNodeData, (htAllocatedSparseChildGraphNodeData()));
    htSparseChildGraphNodeDatas.PrevDatadrawRootSparseChildGraphNodeData = utNewA(htSparseChildGraphNodeData, (htAllocatedSparseChildGraphNodeData()));
    htSparseChildGraphNodeDatas.NextTableDatadrawRootSparseChildGraphNodeData = utNewA(htSparseChildGraphNodeData, (htAllocatedSparseChildGraphNodeData()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class SparseChildGraphNodeData.
----------------------------------------------------------------------------------------*/
static void reallocSparseChildGraphNodeDatas(
    uint32 newSize)
{
    utResizeArray(htSparseChildGraphNodeDatas.NodeKey, (newSize));
    utResizeArray(htSparseChildGraphNodeDatas.NextGraphNode, (newSize));
    utResizeArray(htSparseChildGraphNodeDatas.PrevGraphNode, (newSize));
    utResizeArray(htSparseChildGraphNodeDatas.NextTableGraphNode, (newSize));
    utResizeArray(htSparseChildGraphNodeDatas.NextDatadrawRootSparseChildGraphNodeData, (newSize));
    utResizeArray(htSparseChildGraphNodeDatas.PrevDatadrawRootSparseChildGraphNodeData, (newSize));
    utResizeArray(htSparseChildGraphNodeDatas.NextTableDatadrawRootSparseChildGraphNodeData, (newSize));
    htSetAllocatedSparseChildGraphNodeData(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more SparseChildGraphNodeDatas.
----------------------------------------------------------------------------------------*/
void htSparseChildGraphNodeDataAllocMore(void)
{
    reallocSparseChildGraphNodeDatas((uint32)(htAllocatedSparseChildGraphNodeData() + (htAllocatedSparseChildGraphNodeData() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of SparseChildGraphNodeData.
----------------------------------------------------------------------------------------*/
void htSparseChildGraphNodeDataCopyProps(
    htSparseChildGraphNodeData oldSparseChildGraphNodeData,
    htSparseChildGraphNodeData newSparseChildGraphNodeData)
{
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowSparseChildGraphNodeData(
    htSparseChildGraphNodeData SparseChildGraphNodeData)
{
    utDatabaseShowObject("ht", "SparseChildGraphNodeData", htSparseChildGraphNodeData2Index(SparseChildGraphNodeData));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy SparseNodeNameData including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void htSparseNodeNameDataDestroy(
    htSparseNodeNameData SparseNodeNameData)
{
    if(htSparseNodeNameDataDestructorCallback != NULL) {
        htSparseNodeNameDataDestructorCallback(SparseNodeNameData);
    }
    htSparseNodeNameDataFree(SparseNodeNameData);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocSparseNodeNameData(void)
{
    htSparseNodeNameData SparseNodeNameData = htSparseNodeNameDataAlloc();

    return htSparseNodeNameData2Index(SparseNodeNameData);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroySparseNodeNameData(
    uint64 objectIndex)
{
    htSparseNodeNameDataDestroy(htIndex2SparseNodeNameData((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of SparseNodeNameData.
----------------------------------------------------------------------------------------*/
static void allocSparseNodeNameDatas(void)
{
    htSetAllocatedSparseNodeNameData(2);
    htSetUsedSparseNodeNameData(0);
    htSetFirstFreeSparseNodeNameData(htSparseNodeNameDataNull);
    htSparseNodeNameDatas.NodeKey = utNewA(htNode, (htAllocatedSparseNodeNameData()));
    htSparseNodeNameDatas.NameIndex = utNewA(uint32, (htAllocatedSparseNodeNameData()));
    htSparseNodeNameDatas.NumName = utNewA(uint32, (htAllocatedSparseNodeNameData()));
    htSparseNodeNameDatas.NextDatadrawRootSparseNodeNameData = utNewA(htSparseNodeNameData, (htAllocatedSparseNodeNameData()));
    htSparseNodeNameDatas.PrevDatadrawRootSparseNodeNameData = utNewA(htSparseNodeNameData, (htAllocatedSparseNodeNameData()));
    htSparseNodeNameDatas.NextTableDatadrawRootSparseNodeNameData = utNewA(htSparseNodeNameData, (htAllocatedSparseNodeNameData()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class SparseNodeNameData.
----------------------------------------------------------------------------------------*/
static void reallocSparseNodeNameDatas(
    uint32 newSize)
{
    utResizeArray(htSparseNodeNameDatas.NodeKey, (newSize));
    utResizeArray(htSparseNodeNameDatas.NameIndex, (newSize));
    utResizeArray(htSparseNodeNameDatas.NumName, (newSize));
    utResizeArray(htSparseNodeNameDatas.NextDatadrawRootSparseNodeNameData, (newSize));
    utResizeArray(htSparseNodeNameDatas.PrevDatadrawRootSparseNodeNameData, (newSize));
    utResizeArray(htSparseNodeNameDatas.NextTableDatadrawRootSparseNodeNameData, (newSize));
    htSetAllocatedSparseNodeNameData(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more SparseNodeNameDatas.
----------------------------------------------------------------------------------------*/
void htSparseNodeNameDataAllocMore(void)
{
    reallocSparseNodeNameDatas((uint32)(htAllocatedSparseNodeNameData() + (htAllocatedSparseNodeNameData() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of SparseNodeNameData.
----------------------------------------------------------------------------------------*/
void htSparseNodeNameDataCopyProps(
    htSparseNodeNameData oldSparseNodeNameData,
    htSparseNodeNameData newSparseNodeNameData)
{
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowSparseNodeNameData(
    htSparseNodeNameData SparseNodeNameData)
{
    utDatabaseShowObject("ht", "SparseNodeNameData", htSparseNodeNameData2Index(SparseNodeNameData));
}
#endif

/*----------------------------------------------------------------------------------------
  Destroy SparseNodeXData including everything in it. Remove from parents.
----------------------------------------------------------------------------------------*/
void htSparseNodeXDataDestroy(
    htSparseNodeXData SparseNodeXData)
{
    if(htSparseNodeXDataDestructorCallback != NULL) {
        htSparseNodeXDataDestructorCallback(SparseNodeXData);
    }
    htSparseNodeXDataFree(SparseNodeXData);
}

/*----------------------------------------------------------------------------------------
  Default constructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static uint64 allocSparseNodeXData(void)
{
    htSparseNodeXData SparseNodeXData = htSparseNodeXDataAlloc();

    return htSparseNodeXData2Index(SparseNodeXData);
}

/*----------------------------------------------------------------------------------------
  Destructor wrapper for the database manager.
----------------------------------------------------------------------------------------*/
static void destroySparseNodeXData(
    uint64 objectIndex)
{
    htSparseNodeXDataDestroy(htIndex2SparseNodeXData((uint32)objectIndex));
}

/*----------------------------------------------------------------------------------------
  Allocate the field arrays of SparseNodeXData.
----------------------------------------------------------------------------------------*/
static void allocSparseNodeXDatas(void)
{
    htSetAllocatedSparseNodeXData(2);
    htSetUsedSparseNodeXData(0);
    htSetFirstFreeSparseNodeXData(htSparseNodeXDataNull);
    htSparseNodeXDatas.NodeKey = utNewA(htNode, (htAllocatedSparseNodeXData()));
    htSparseNodeXDatas.X = utNewA(int32, (htAllocatedSparseNodeXData()));
    htSparseNodeXDatas.NextDatadrawRootSparseNodeXData = utNewA(htSparseNodeXData, (htAllocatedSparseNodeXData()));
    htSparseNodeXDatas.PrevDatadrawRootSparseNodeXData = utNewA(htSparseNodeXData, (htAllocatedSparseNodeXData()));
    htSparseNodeXDatas.NextTableDatadrawRootSparseNodeXData = utNewA(htSparseNodeXData, (htAllocatedSparseNodeXData()));
}

/*----------------------------------------------------------------------------------------
  Realloc the arrays of properties for class SparseNodeXData.
----------------------------------------------------------------------------------------*/
static void reallocSparseNodeXDatas(
    uint32 newSize)
{
    utResizeArray(htSparseNodeXDatas.NodeKey, (newSize));
    utResizeArray(htSparseNodeXDatas.X, (newSize));
    utResizeArray(htSparseNodeXDatas.NextDatadrawRootSparseNodeXData, (newSize));
    utResizeArray(htSparseNodeXDatas.PrevDatadrawRootSparseNodeXData, (newSize));
    utResizeArray(htSparseNodeXDatas.NextTableDatadrawRootSparseNodeXData, (newSize));
    htSetAllocatedSparseNodeXData(newSize);
}

/*----------------------------------------------------------------------------------------
  Allocate more SparseNodeXDatas.
----------------------------------------------------------------------------------------*/
void htSparseNodeXDataAllocMore(void)
{
    reallocSparseNodeXDatas((uint32)(htAllocatedSparseNodeXData() + (htAllocatedSparseNodeXData() >> 1)));
}

/*----------------------------------------------------------------------------------------
  Copy the properties of SparseNodeXData.
----------------------------------------------------------------------------------------*/
void htSparseNodeXDataCopyProps(
    htSparseNodeXData oldSparseNodeXData,
    htSparseNodeXData newSparseNodeXData)
{
    htSparseNodeXDataSetX(newSparseNodeXData, htSparseNodeXDataGetX(oldSparseNodeXData));
}

#if defined(DD_DEBUG)
/*----------------------------------------------------------------------------------------
  Write out all the fields of an object.
----------------------------------------------------------------------------------------*/
void htShowSparseNodeXData(
    htSparseNodeXData SparseNodeXData)
{
    utDatabaseShowObject("ht", "SparseNodeXData", htSparseNodeXData2Index(SparseNodeXData));
}
#endif

/*----------------------------------------------------------------------------------------
  Free memory used by the ht database.
----------------------------------------------------------------------------------------*/
void htDatabaseStop(void)
{
    utFree(htGraphs.NodeTable);
    utFree(htGraphs.FreeList);
    utFree(htNodes.Name);
    utFree(htNodes.Y);
    utFree(htNodes.FreeList);
    utFree(htDatadrawRoots.FirstSparseParentGraphNodeData);
    utFree(htDatadrawRoots.LastSparseParentGraphNodeData);
    utFree(htDatadrawRoots.SparseParentGraphNodeDataTableIndex);
    utFree(htDatadrawRoots.NumSparseParentGraphNodeDataTable);
    utFree(htDatadrawRoots.SparseParentGraphNodeDataTable);
    utFree(htDatadrawRoots.NumSparseParentGraphNodeData);
    utFree(htDatadrawRoots.FirstSparseChildGraphNodeData);
    utFree(htDatadrawRoots.LastSparseChildGraphNodeData);
    utFree(htDatadrawRoots.SparseChildGraphNodeDataTableIndex);
    utFree(htDatadrawRoots.NumSparseChildGraphNodeDataTable);
    utFree(htDatadrawRoots.SparseChildGraphNodeDataTable);
    utFree(htDatadrawRoots.NumSparseChildGraphNodeData);
    utFree(htDatadrawRoots.FirstSparseNodeNameData);
    utFree(htDatadrawRoots.LastSparseNodeNameData);
    utFree(htDatadrawRoots.SparseNodeNameDataTableIndex);
    utFree(htDatadrawRoots.NumSparseNodeNameDataTable);
    utFree(htDatadrawRoots.SparseNodeNameDataTable);
    utFree(htDatadrawRoots.NumSparseNodeNameData);
    utFree(htDatadrawRoots.FirstSparseNodeXData);
    utFree(htDatadrawRoots.LastSparseNodeXData);
    utFree(htDatadrawRoots.SparseNodeXDataTableIndex);
    utFree(htDatadrawRoots.NumSparseNodeXDataTable);
    utFree(htDatadrawRoots.SparseNodeXDataTable);
    utFree(htDatadrawRoots.NumSparseNodeXData);
    utFree(htSparseParentGraphNodeDatas.GraphKey);
    utFree(htSparseParentGraphNodeDatas.FirstNode);
    utFree(htSparseParentGraphNodeDatas.LastNode);
    utFree(htSparseParentGraphNodeDatas.NumNode);
    utFree(htSparseParentGraphNodeDatas.NodeTableIndex);
    utFree(htSparseParentGraphNodeDatas.NumNodeTable);
    utFree(htSparseParentGraphNodeDatas.NextDatadrawRootSparseParentGraphNodeData);
    utFree(htSparseParentGraphNodeDatas.PrevDatadrawRootSparseParentGraphNodeData);
    utFree(htSparseParentGraphNodeDatas.NextTableDatadrawRootSparseParentGraphNodeData);
    utFree(htSparseChildGraphNodeDatas.NodeKey);
    utFree(htSparseChildGraphNodeDatas.NextGraphNode);
    utFree(htSparseChildGraphNodeDatas.PrevGraphNode);
    utFree(htSparseChildGraphNodeDatas.NextTableGraphNode);
    utFree(htSparseChildGraphNodeDatas.NextDatadrawRootSparseChildGraphNodeData);
    utFree(htSparseChildGraphNodeDatas.PrevDatadrawRootSparseChildGraphNodeData);
    utFree(htSparseChildGraphNodeDatas.NextTableDatadrawRootSparseChildGraphNodeData);
    utFree(htSparseNodeNameDatas.NodeKey);
    utFree(htSparseNodeNameDatas.NameIndex);
    utFree(htSparseNodeNameDatas.NumName);
    utFree(htSparseNodeNameDatas.NextDatadrawRootSparseNodeNameData);
    utFree(htSparseNodeNameDatas.PrevDatadrawRootSparseNodeNameData);
    utFree(htSparseNodeNameDatas.NextTableDatadrawRootSparseNodeNameData);
    utFree(htSparseNodeXDatas.NodeKey);
    utFree(htSparseNodeXDatas.X);
    utFree(htSparseNodeXDatas.NextDatadrawRootSparseNodeXData);
    utFree(htSparseNodeXDatas.PrevDatadrawRootSparseNodeXData);
    utFree(htSparseNodeXDatas.NextTableDatadrawRootSparseNodeXData);
    utUnregisterModule(htModuleID);
}

/*----------------------------------------------------------------------------------------
  Allocate memory used by the ht database.
----------------------------------------------------------------------------------------*/
void htDatabaseStart(void)
{
    if(!utInitialized()) {
        utStart();
    }
    htRootData.hash = 0xeda1bf27;
    htModuleID = utRegisterModule("ht", false, htHash(), 7, 56, 0, sizeof(struct htRootType_),
        &htRootData, htDatabaseStart, htDatabaseStop);
    utRegisterClass("Graph", 2, &htRootData.usedGraph, &htRootData.allocatedGraph,
        &htRootData.firstFreeGraph, 1, 4, allocGraph, destroyGraph);
    utRegisterField("NodeTable", &htGraphs.NodeTable, sizeof(htNode), UT_POINTER, "Node");
    utRegisterArray(&htRootData.usedGraphNodeTable, &htRootData.allocatedGraphNodeTable,
        getGraphNodeTables, allocGraphNodeTables);
    utRegisterField("FreeList", &htGraphs.FreeList, sizeof(htGraph), UT_POINTER, "Graph");
    utSetFieldHidden();
    utRegisterClass("Node", 3, &htRootData.usedNode, &htRootData.allocatedNode,
        &htRootData.firstFreeNode, 4, 4, allocNode, destroyNode);
    utRegisterField("Name", &htNodes.Name, sizeof(char), UT_CHAR, NULL);
    utRegisterArray(&htRootData.usedNodeName, &htRootData.allocatedNodeName,
        getNodeNames, allocNodeNames);
    utRegisterField("Y", &htNodes.Y, sizeof(int32), UT_INT, NULL);
    utRegisterField("FreeList", &htNodes.FreeList, sizeof(htNode), UT_POINTER, "Node");
    utSetFieldHidden();
    utRegisterClass("DatadrawRoot", 24, &htRootData.usedDatadrawRoot, &htRootData.allocatedDatadrawRoot,
        NULL, 65535, 4, allocDatadrawRoot, NULL);
    utRegisterField("FirstSparseParentGraphNodeData", &htDatadrawRoots.FirstSparseParentGraphNodeData, sizeof(htSparseParentGraphNodeData), UT_POINTER, "SparseParentGraphNodeData");
    utRegisterField("LastSparseParentGraphNodeData", &htDatadrawRoots.LastSparseParentGraphNodeData, sizeof(htSparseParentGraphNodeData), UT_POINTER, "SparseParentGraphNodeData");
    utRegisterField("SparseParentGraphNodeDataTableIndex", &htDatadrawRoots.SparseParentGraphNodeDataTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumSparseParentGraphNodeDataTable", &htDatadrawRoots.NumSparseParentGraphNodeDataTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("SparseParentGraphNodeDataTable", &htDatadrawRoots.SparseParentGraphNodeDataTable, sizeof(htSparseParentGraphNodeData), UT_POINTER, "SparseParentGraphNodeData");
    utRegisterArray(&htRootData.usedDatadrawRootSparseParentGraphNodeDataTable, &htRootData.allocatedDatadrawRootSparseParentGraphNodeDataTable,
        getDatadrawRootSparseParentGraphNodeDataTables, allocDatadrawRootSparseParentGraphNodeDataTables);
    utRegisterField("NumSparseParentGraphNodeData", &htDatadrawRoots.NumSparseParentGraphNodeData, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstSparseChildGraphNodeData", &htDatadrawRoots.FirstSparseChildGraphNodeData, sizeof(htSparseChildGraphNodeData), UT_POINTER, "SparseChildGraphNodeData");
    utRegisterField("LastSparseChildGraphNodeData", &htDatadrawRoots.LastSparseChildGraphNodeData, sizeof(htSparseChildGraphNodeData), UT_POINTER, "SparseChildGraphNodeData");
    utRegisterField("SparseChildGraphNodeDataTableIndex", &htDatadrawRoots.SparseChildGraphNodeDataTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumSparseChildGraphNodeDataTable", &htDatadrawRoots.NumSparseChildGraphNodeDataTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("SparseChildGraphNodeDataTable", &htDatadrawRoots.SparseChildGraphNodeDataTable, sizeof(htSparseChildGraphNodeData), UT_POINTER, "SparseChildGraphNodeData");
    utRegisterArray(&htRootData.usedDatadrawRootSparseChildGraphNodeDataTable, &htRootData.allocatedDatadrawRootSparseChildGraphNodeDataTable,
        getDatadrawRootSparseChildGraphNodeDataTables, allocDatadrawRootSparseChildGraphNodeDataTables);
    utRegisterField("NumSparseChildGraphNodeData", &htDatadrawRoots.NumSparseChildGraphNodeData, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstSparseNodeNameData", &htDatadrawRoots.FirstSparseNodeNameData, sizeof(htSparseNodeNameData), UT_POINTER, "SparseNodeNameData");
    utRegisterField("LastSparseNodeNameData", &htDatadrawRoots.LastSparseNodeNameData, sizeof(htSparseNodeNameData), UT_POINTER, "SparseNodeNameData");
    utRegisterField("SparseNodeNameDataTableIndex", &htDatadrawRoots.SparseNodeNameDataTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumSparseNodeNameDataTable", &htDatadrawRoots.NumSparseNodeNameDataTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("SparseNodeNameDataTable", &htDatadrawRoots.SparseNodeNameDataTable, sizeof(htSparseNodeNameData), UT_POINTER, "SparseNodeNameData");
    utRegisterArray(&htRootData.usedDatadrawRootSparseNodeNameDataTable, &htRootData.allocatedDatadrawRootSparseNodeNameDataTable,
        getDatadrawRootSparseNodeNameDataTables, allocDatadrawRootSparseNodeNameDataTables);
    utRegisterField("NumSparseNodeNameData", &htDatadrawRoots.NumSparseNodeNameData, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("FirstSparseNodeXData", &htDatadrawRoots.FirstSparseNodeXData, sizeof(htSparseNodeXData), UT_POINTER, "SparseNodeXData");
    utRegisterField("LastSparseNodeXData", &htDatadrawRoots.LastSparseNodeXData, sizeof(htSparseNodeXData), UT_POINTER, "SparseNodeXData");
    utRegisterField("SparseNodeXDataTableIndex", &htDatadrawRoots.SparseNodeXDataTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumSparseNodeXDataTable", &htDatadrawRoots.NumSparseNodeXDataTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("SparseNodeXDataTable", &htDatadrawRoots.SparseNodeXDataTable, sizeof(htSparseNodeXData), UT_POINTER, "SparseNodeXData");
    utRegisterArray(&htRootData.usedDatadrawRootSparseNodeXDataTable, &htRootData.allocatedDatadrawRootSparseNodeXDataTable,
        getDatadrawRootSparseNodeXDataTables, allocDatadrawRootSparseNodeXDataTables);
    utRegisterField("NumSparseNodeXData", &htDatadrawRoots.NumSparseNodeXData, sizeof(uint32), UT_UINT, NULL);
    utRegisterClass("SparseParentGraphNodeData", 9, &htRootData.usedSparseParentGraphNodeData, &htRootData.allocatedSparseParentGraphNodeData,
        &htRootData.firstFreeSparseParentGraphNodeData, 29, 4, allocSparseParentGraphNodeData, destroySparseParentGraphNodeData);
    utRegisterField("GraphKey", &htSparseParentGraphNodeDatas.GraphKey, sizeof(htGraph), UT_POINTER, "Graph");
    utRegisterField("FirstNode", &htSparseParentGraphNodeDatas.FirstNode, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("LastNode", &htSparseParentGraphNodeDatas.LastNode, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("NumNode", &htSparseParentGraphNodeDatas.NumNode, sizeof(uint32), UT_UINT, NULL);
    utRegisterField("NodeTableIndex", &htSparseParentGraphNodeDatas.NodeTableIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumNodeTable", &htSparseParentGraphNodeDatas.NumNodeTable, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NextDatadrawRootSparseParentGraphNodeData", &htSparseParentGraphNodeDatas.NextDatadrawRootSparseParentGraphNodeData, sizeof(htSparseParentGraphNodeData), UT_POINTER, "SparseParentGraphNodeData");
    utRegisterField("PrevDatadrawRootSparseParentGraphNodeData", &htSparseParentGraphNodeDatas.PrevDatadrawRootSparseParentGraphNodeData, sizeof(htSparseParentGraphNodeData), UT_POINTER, "SparseParentGraphNodeData");
    utRegisterField("NextTableDatadrawRootSparseParentGraphNodeData", &htSparseParentGraphNodeDatas.NextTableDatadrawRootSparseParentGraphNodeData, sizeof(htSparseParentGraphNodeData), UT_POINTER, "SparseParentGraphNodeData");
    utRegisterClass("SparseChildGraphNodeData", 7, &htRootData.usedSparseChildGraphNodeData, &htRootData.allocatedSparseChildGraphNodeData,
        &htRootData.firstFreeSparseChildGraphNodeData, 38, 4, allocSparseChildGraphNodeData, destroySparseChildGraphNodeData);
    utRegisterField("NodeKey", &htSparseChildGraphNodeDatas.NodeKey, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("NextGraphNode", &htSparseChildGraphNodeDatas.NextGraphNode, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("PrevGraphNode", &htSparseChildGraphNodeDatas.PrevGraphNode, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("NextTableGraphNode", &htSparseChildGraphNodeDatas.NextTableGraphNode, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("NextDatadrawRootSparseChildGraphNodeData", &htSparseChildGraphNodeDatas.NextDatadrawRootSparseChildGraphNodeData, sizeof(htSparseChildGraphNodeData), UT_POINTER, "SparseChildGraphNodeData");
    utRegisterField("PrevDatadrawRootSparseChildGraphNodeData", &htSparseChildGraphNodeDatas.PrevDatadrawRootSparseChildGraphNodeData, sizeof(htSparseChildGraphNodeData), UT_POINTER, "SparseChildGraphNodeData");
    utRegisterField("NextTableDatadrawRootSparseChildGraphNodeData", &htSparseChildGraphNodeDatas.NextTableDatadrawRootSparseChildGraphNodeData, sizeof(htSparseChildGraphNodeData), UT_POINTER, "SparseChildGraphNodeData");
    utRegisterClass("SparseNodeNameData", 6, &htRootData.usedSparseNodeNameData, &htRootData.allocatedSparseNodeNameData,
        &htRootData.firstFreeSparseNodeNameData, 45, 4, allocSparseNodeNameData, destroySparseNodeNameData);
    utRegisterField("NodeKey", &htSparseNodeNameDatas.NodeKey, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("NameIndex", &htSparseNodeNameDatas.NameIndex, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NumName", &htSparseNodeNameDatas.NumName, sizeof(uint32), UT_UINT, NULL);
    utSetFieldHidden();
    utRegisterField("NextDatadrawRootSparseNodeNameData", &htSparseNodeNameDatas.NextDatadrawRootSparseNodeNameData, sizeof(htSparseNodeNameData), UT_POINTER, "SparseNodeNameData");
    utRegisterField("PrevDatadrawRootSparseNodeNameData", &htSparseNodeNameDatas.PrevDatadrawRootSparseNodeNameData, sizeof(htSparseNodeNameData), UT_POINTER, "SparseNodeNameData");
    utRegisterField("NextTableDatadrawRootSparseNodeNameData", &htSparseNodeNameDatas.NextTableDatadrawRootSparseNodeNameData, sizeof(htSparseNodeNameData), UT_POINTER, "SparseNodeNameData");
    utRegisterClass("SparseNodeXData", 5, &htRootData.usedSparseNodeXData, &htRootData.allocatedSparseNodeXData,
        &htRootData.firstFreeSparseNodeXData, 51, 4, allocSparseNodeXData, destroySparseNodeXData);
    utRegisterField("NodeKey", &htSparseNodeXDatas.NodeKey, sizeof(htNode), UT_POINTER, "Node");
    utRegisterField("X", &htSparseNodeXDatas.X, sizeof(int32), UT_INT, NULL);
    utRegisterField("NextDatadrawRootSparseNodeXData", &htSparseNodeXDatas.NextDatadrawRootSparseNodeXData, sizeof(htSparseNodeXData), UT_POINTER, "SparseNodeXData");
    utRegisterField("PrevDatadrawRootSparseNodeXData", &htSparseNodeXDatas.PrevDatadrawRootSparseNodeXData, sizeof(htSparseNodeXData), UT_POINTER, "SparseNodeXData");
    utRegisterField("NextTableDatadrawRootSparseNodeXData", &htSparseNodeXDatas.NextTableDatadrawRootSparseNodeXData, sizeof(htSparseNodeXData), UT_POINTER, "SparseNodeXData");
    allocGraphs();
    allocNodes();
    allocDatadrawRoots();
    allocSparseParentGraphNodeDatas();
    allocSparseChildGraphNodeDatas();
    allocSparseNodeNameDatas();
    allocSparseNodeXDatas();
    (void)htDatadrawRootAlloc();
}

#if defined(DD_DEBUG)
#undef htGraphGetiNodeTable
htNode htGraphGetiNodeTable(
    htGraph _Graph,
    uint32 x)
{
    return (htGraphs.NodeTable)[htGraphGetNodeTableIndex(_Graph) + x];
}

#undef htGraphGetNodeTables
htNode *htGraphGetNodeTables(
    htGraph Graph)
{
    return htGraphs.NodeTable + htGraphGetNodeTableIndex(Graph);
}

#undef htGraphSetiNodeTable
void htGraphSetiNodeTable(
    htGraph Graph,
    uint32 x,
    htNode value)
{
    htGraphs.NodeTable[htGraphGetNodeTableIndex(Graph) + x] = value;
}

#undef htGraphGetNodeTable
htNode *htGraphGetNodeTable(
    htGraph Graph)
{
    return htGraphs.NodeTable + htGraphGetNodeTableIndex(Graph);
}

#undef htGraphGetFreeList
htGraph htGraphGetFreeList(
    htGraph _Graph)
{
    return htGraphs.FreeList[htGraph2Index(_Graph)];
}

#undef htGraphSetFreeList
void htGraphSetFreeList(
    htGraph _Graph,
    htGraph value)
{
    htGraphs.FreeList[htGraph2Index(_Graph)] = value;
}

#undef htNodeGetiName
char htNodeGetiName(
    htNode _Node,
    uint32 x)
{
    return (htNodes.Name)[htNodeGetNameIndex(_Node) + x];
}

#undef htNodeGetNames
char *htNodeGetNames(
    htNode Node)
{
    return htNodes.Name + htNodeGetNameIndex(Node);
}

#undef htNodeSetiName
void htNodeSetiName(
    htNode Node,
    uint32 x,
    char value)
{
    htNodes.Name[htNodeGetNameIndex(Node) + x] = value;
}

#undef htNodeGetName
char *htNodeGetName(
    htNode Node)
{
    return htNodes.Name + htNodeGetNameIndex(Node);
}

#undef htNodeGetY
int32 htNodeGetY(
    htNode _Node)
{
    return htNodes.Y[htNode2Index(_Node)];
}

#undef htNodeSetY
void htNodeSetY(
    htNode _Node,
    int32 value)
{
    htNodes.Y[htNode2Index(_Node)] = value;
}

#undef htNodeGetFreeList
htNode htNodeGetFreeList(
    htNode _Node)
{
    return htNodes.FreeList[htNode2Index(_Node)];
}

#undef htNodeSetFreeList
void htNodeSetFreeList(
    htNode _Node,
    htNode value)
{
    htNodes.FreeList[htNode2Index(_Node)] = value;
}

#undef htDatadrawRootGetFirstSparseParentGraphNodeData
htSparseParentGraphNodeData htDatadrawRootGetFirstSparseParentGraphNodeData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.FirstSparseParentGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetFirstSparseParentGraphNodeData
void htDatadrawRootSetFirstSparseParentGraphNodeData(
    htDatadrawRoot _DatadrawRoot,
    htSparseParentGraphNodeData value)
{
    htDatadrawRoots.FirstSparseParentGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetLastSparseParentGraphNodeData
htSparseParentGraphNodeData htDatadrawRootGetLastSparseParentGraphNodeData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.LastSparseParentGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetLastSparseParentGraphNodeData
void htDatadrawRootSetLastSparseParentGraphNodeData(
    htDatadrawRoot _DatadrawRoot,
    htSparseParentGraphNodeData value)
{
    htDatadrawRoots.LastSparseParentGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetSparseParentGraphNodeDataTableIndex
uint32 htDatadrawRootGetSparseParentGraphNodeDataTableIndex(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.SparseParentGraphNodeDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetSparseParentGraphNodeDataTableIndex
void htDatadrawRootSetSparseParentGraphNodeDataTableIndex(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.SparseParentGraphNodeDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetNumSparseParentGraphNodeDataTable
uint32 htDatadrawRootGetNumSparseParentGraphNodeDataTable(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseParentGraphNodeDataTable[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseParentGraphNodeDataTable
void htDatadrawRootSetNumSparseParentGraphNodeDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseParentGraphNodeDataTable[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetiSparseParentGraphNodeDataTable
htSparseParentGraphNodeData htDatadrawRootGetiSparseParentGraphNodeDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 x)
{
    return (htDatadrawRoots.SparseParentGraphNodeDataTable)[htDatadrawRootGetSparseParentGraphNodeDataTableIndex(_DatadrawRoot) + x];
}

#undef htDatadrawRootGetSparseParentGraphNodeDataTables
htSparseParentGraphNodeData *htDatadrawRootGetSparseParentGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseParentGraphNodeDataTable + htDatadrawRootGetSparseParentGraphNodeDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootSetiSparseParentGraphNodeDataTable
void htDatadrawRootSetiSparseParentGraphNodeDataTable(
    htDatadrawRoot DatadrawRoot,
    uint32 x,
    htSparseParentGraphNodeData value)
{
    htDatadrawRoots.SparseParentGraphNodeDataTable[htDatadrawRootGetSparseParentGraphNodeDataTableIndex(DatadrawRoot) + x] = value;
}

#undef htDatadrawRootGetSparseParentGraphNodeDataTable
htSparseParentGraphNodeData *htDatadrawRootGetSparseParentGraphNodeDataTable(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseParentGraphNodeDataTable + htDatadrawRootGetSparseParentGraphNodeDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootGetNumSparseParentGraphNodeData
uint32 htDatadrawRootGetNumSparseParentGraphNodeData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseParentGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseParentGraphNodeData
void htDatadrawRootSetNumSparseParentGraphNodeData(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseParentGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetFirstSparseChildGraphNodeData
htSparseChildGraphNodeData htDatadrawRootGetFirstSparseChildGraphNodeData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.FirstSparseChildGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetFirstSparseChildGraphNodeData
void htDatadrawRootSetFirstSparseChildGraphNodeData(
    htDatadrawRoot _DatadrawRoot,
    htSparseChildGraphNodeData value)
{
    htDatadrawRoots.FirstSparseChildGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetLastSparseChildGraphNodeData
htSparseChildGraphNodeData htDatadrawRootGetLastSparseChildGraphNodeData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.LastSparseChildGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetLastSparseChildGraphNodeData
void htDatadrawRootSetLastSparseChildGraphNodeData(
    htDatadrawRoot _DatadrawRoot,
    htSparseChildGraphNodeData value)
{
    htDatadrawRoots.LastSparseChildGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetSparseChildGraphNodeDataTableIndex
uint32 htDatadrawRootGetSparseChildGraphNodeDataTableIndex(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.SparseChildGraphNodeDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetSparseChildGraphNodeDataTableIndex
void htDatadrawRootSetSparseChildGraphNodeDataTableIndex(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.SparseChildGraphNodeDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetNumSparseChildGraphNodeDataTable
uint32 htDatadrawRootGetNumSparseChildGraphNodeDataTable(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseChildGraphNodeDataTable[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseChildGraphNodeDataTable
void htDatadrawRootSetNumSparseChildGraphNodeDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseChildGraphNodeDataTable[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetiSparseChildGraphNodeDataTable
htSparseChildGraphNodeData htDatadrawRootGetiSparseChildGraphNodeDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 x)
{
    return (htDatadrawRoots.SparseChildGraphNodeDataTable)[htDatadrawRootGetSparseChildGraphNodeDataTableIndex(_DatadrawRoot) + x];
}

#undef htDatadrawRootGetSparseChildGraphNodeDataTables
htSparseChildGraphNodeData *htDatadrawRootGetSparseChildGraphNodeDataTables(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseChildGraphNodeDataTable + htDatadrawRootGetSparseChildGraphNodeDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootSetiSparseChildGraphNodeDataTable
void htDatadrawRootSetiSparseChildGraphNodeDataTable(
    htDatadrawRoot DatadrawRoot,
    uint32 x,
    htSparseChildGraphNodeData value)
{
    htDatadrawRoots.SparseChildGraphNodeDataTable[htDatadrawRootGetSparseChildGraphNodeDataTableIndex(DatadrawRoot) + x] = value;
}

#undef htDatadrawRootGetSparseChildGraphNodeDataTable
htSparseChildGraphNodeData *htDatadrawRootGetSparseChildGraphNodeDataTable(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseChildGraphNodeDataTable + htDatadrawRootGetSparseChildGraphNodeDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootGetNumSparseChildGraphNodeData
uint32 htDatadrawRootGetNumSparseChildGraphNodeData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseChildGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseChildGraphNodeData
void htDatadrawRootSetNumSparseChildGraphNodeData(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseChildGraphNodeData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetFirstSparseNodeNameData
htSparseNodeNameData htDatadrawRootGetFirstSparseNodeNameData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.FirstSparseNodeNameData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetFirstSparseNodeNameData
void htDatadrawRootSetFirstSparseNodeNameData(
    htDatadrawRoot _DatadrawRoot,
    htSparseNodeNameData value)
{
    htDatadrawRoots.FirstSparseNodeNameData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetLastSparseNodeNameData
htSparseNodeNameData htDatadrawRootGetLastSparseNodeNameData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.LastSparseNodeNameData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetLastSparseNodeNameData
void htDatadrawRootSetLastSparseNodeNameData(
    htDatadrawRoot _DatadrawRoot,
    htSparseNodeNameData value)
{
    htDatadrawRoots.LastSparseNodeNameData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetSparseNodeNameDataTableIndex
uint32 htDatadrawRootGetSparseNodeNameDataTableIndex(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.SparseNodeNameDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetSparseNodeNameDataTableIndex
void htDatadrawRootSetSparseNodeNameDataTableIndex(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.SparseNodeNameDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetNumSparseNodeNameDataTable
uint32 htDatadrawRootGetNumSparseNodeNameDataTable(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseNodeNameDataTable[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseNodeNameDataTable
void htDatadrawRootSetNumSparseNodeNameDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseNodeNameDataTable[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetiSparseNodeNameDataTable
htSparseNodeNameData htDatadrawRootGetiSparseNodeNameDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 x)
{
    return (htDatadrawRoots.SparseNodeNameDataTable)[htDatadrawRootGetSparseNodeNameDataTableIndex(_DatadrawRoot) + x];
}

#undef htDatadrawRootGetSparseNodeNameDataTables
htSparseNodeNameData *htDatadrawRootGetSparseNodeNameDataTables(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseNodeNameDataTable + htDatadrawRootGetSparseNodeNameDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootSetiSparseNodeNameDataTable
void htDatadrawRootSetiSparseNodeNameDataTable(
    htDatadrawRoot DatadrawRoot,
    uint32 x,
    htSparseNodeNameData value)
{
    htDatadrawRoots.SparseNodeNameDataTable[htDatadrawRootGetSparseNodeNameDataTableIndex(DatadrawRoot) + x] = value;
}

#undef htDatadrawRootGetSparseNodeNameDataTable
htSparseNodeNameData *htDatadrawRootGetSparseNodeNameDataTable(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseNodeNameDataTable + htDatadrawRootGetSparseNodeNameDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootGetNumSparseNodeNameData
uint32 htDatadrawRootGetNumSparseNodeNameData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseNodeNameData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseNodeNameData
void htDatadrawRootSetNumSparseNodeNameData(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseNodeNameData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetFirstSparseNodeXData
htSparseNodeXData htDatadrawRootGetFirstSparseNodeXData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.FirstSparseNodeXData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetFirstSparseNodeXData
void htDatadrawRootSetFirstSparseNodeXData(
    htDatadrawRoot _DatadrawRoot,
    htSparseNodeXData value)
{
    htDatadrawRoots.FirstSparseNodeXData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetLastSparseNodeXData
htSparseNodeXData htDatadrawRootGetLastSparseNodeXData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.LastSparseNodeXData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetLastSparseNodeXData
void htDatadrawRootSetLastSparseNodeXData(
    htDatadrawRoot _DatadrawRoot,
    htSparseNodeXData value)
{
    htDatadrawRoots.LastSparseNodeXData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetSparseNodeXDataTableIndex
uint32 htDatadrawRootGetSparseNodeXDataTableIndex(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.SparseNodeXDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetSparseNodeXDataTableIndex
void htDatadrawRootSetSparseNodeXDataTableIndex(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.SparseNodeXDataTableIndex[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetNumSparseNodeXDataTable
uint32 htDatadrawRootGetNumSparseNodeXDataTable(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseNodeXDataTable[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseNodeXDataTable
void htDatadrawRootSetNumSparseNodeXDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseNodeXDataTable[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htDatadrawRootGetiSparseNodeXDataTable
htSparseNodeXData htDatadrawRootGetiSparseNodeXDataTable(
    htDatadrawRoot _DatadrawRoot,
    uint32 x)
{
    return (htDatadrawRoots.SparseNodeXDataTable)[htDatadrawRootGetSparseNodeXDataTableIndex(_DatadrawRoot) + x];
}

#undef htDatadrawRootGetSparseNodeXDataTables
htSparseNodeXData *htDatadrawRootGetSparseNodeXDataTables(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseNodeXDataTable + htDatadrawRootGetSparseNodeXDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootSetiSparseNodeXDataTable
void htDatadrawRootSetiSparseNodeXDataTable(
    htDatadrawRoot DatadrawRoot,
    uint32 x,
    htSparseNodeXData value)
{
    htDatadrawRoots.SparseNodeXDataTable[htDatadrawRootGetSparseNodeXDataTableIndex(DatadrawRoot) + x] = value;
}

#undef htDatadrawRootGetSparseNodeXDataTable
htSparseNodeXData *htDatadrawRootGetSparseNodeXDataTable(
    htDatadrawRoot DatadrawRoot)
{
    return htDatadrawRoots.SparseNodeXDataTable + htDatadrawRootGetSparseNodeXDataTableIndex(DatadrawRoot);
}

#undef htDatadrawRootGetNumSparseNodeXData
uint32 htDatadrawRootGetNumSparseNodeXData(
    htDatadrawRoot _DatadrawRoot)
{
    return htDatadrawRoots.NumSparseNodeXData[htDatadrawRoot2Index(_DatadrawRoot)];
}

#undef htDatadrawRootSetNumSparseNodeXData
void htDatadrawRootSetNumSparseNodeXData(
    htDatadrawRoot _DatadrawRoot,
    uint32 value)
{
    htDatadrawRoots.NumSparseNodeXData[htDatadrawRoot2Index(_DatadrawRoot)] = value;
}

#undef htSparseParentGraphNodeDataGetGraphKey
htGraph htSparseParentGraphNodeDataGetGraphKey(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.GraphKey[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetGraphKey
void htSparseParentGraphNodeDataSetGraphKey(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    htGraph value)
{
    htSparseParentGraphNodeDatas.GraphKey[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetFirstNode
htNode htSparseParentGraphNodeDataGetFirstNode(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.FirstNode[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetFirstNode
void htSparseParentGraphNodeDataSetFirstNode(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    htNode value)
{
    htSparseParentGraphNodeDatas.FirstNode[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetLastNode
htNode htSparseParentGraphNodeDataGetLastNode(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.LastNode[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetLastNode
void htSparseParentGraphNodeDataSetLastNode(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    htNode value)
{
    htSparseParentGraphNodeDatas.LastNode[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetNumNode
uint32 htSparseParentGraphNodeDataGetNumNode(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.NumNode[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetNumNode
void htSparseParentGraphNodeDataSetNumNode(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    uint32 value)
{
    htSparseParentGraphNodeDatas.NumNode[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetNodeTableIndex
uint32 htSparseParentGraphNodeDataGetNodeTableIndex(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.NodeTableIndex[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetNodeTableIndex
void htSparseParentGraphNodeDataSetNodeTableIndex(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    uint32 value)
{
    htSparseParentGraphNodeDatas.NodeTableIndex[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetNumNodeTable
uint32 htSparseParentGraphNodeDataGetNumNodeTable(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.NumNodeTable[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetNumNodeTable
void htSparseParentGraphNodeDataSetNumNodeTable(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    uint32 value)
{
    htSparseParentGraphNodeDatas.NumNodeTable[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetNextDatadrawRootSparseParentGraphNodeData
htSparseParentGraphNodeData htSparseParentGraphNodeDataGetNextDatadrawRootSparseParentGraphNodeData(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.NextDatadrawRootSparseParentGraphNodeData[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData
void htSparseParentGraphNodeDataSetNextDatadrawRootSparseParentGraphNodeData(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    htSparseParentGraphNodeData value)
{
    htSparseParentGraphNodeDatas.NextDatadrawRootSparseParentGraphNodeData[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetPrevDatadrawRootSparseParentGraphNodeData
htSparseParentGraphNodeData htSparseParentGraphNodeDataGetPrevDatadrawRootSparseParentGraphNodeData(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.PrevDatadrawRootSparseParentGraphNodeData[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData
void htSparseParentGraphNodeDataSetPrevDatadrawRootSparseParentGraphNodeData(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    htSparseParentGraphNodeData value)
{
    htSparseParentGraphNodeDatas.PrevDatadrawRootSparseParentGraphNodeData[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseParentGraphNodeDataGetNextTableDatadrawRootSparseParentGraphNodeData
htSparseParentGraphNodeData htSparseParentGraphNodeDataGetNextTableDatadrawRootSparseParentGraphNodeData(
    htSparseParentGraphNodeData _SparseParentGraphNodeData)
{
    return htSparseParentGraphNodeDatas.NextTableDatadrawRootSparseParentGraphNodeData[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)];
}

#undef htSparseParentGraphNodeDataSetNextTableDatadrawRootSparseParentGraphNodeData
void htSparseParentGraphNodeDataSetNextTableDatadrawRootSparseParentGraphNodeData(
    htSparseParentGraphNodeData _SparseParentGraphNodeData,
    htSparseParentGraphNodeData value)
{
    htSparseParentGraphNodeDatas.NextTableDatadrawRootSparseParentGraphNodeData[htSparseParentGraphNodeData2Index(_SparseParentGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetNodeKey
htNode htSparseChildGraphNodeDataGetNodeKey(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.NodeKey[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetNodeKey
void htSparseChildGraphNodeDataSetNodeKey(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htNode value)
{
    htSparseChildGraphNodeDatas.NodeKey[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetNextGraphNode
htNode htSparseChildGraphNodeDataGetNextGraphNode(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.NextGraphNode[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetNextGraphNode
void htSparseChildGraphNodeDataSetNextGraphNode(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htNode value)
{
    htSparseChildGraphNodeDatas.NextGraphNode[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetPrevGraphNode
htNode htSparseChildGraphNodeDataGetPrevGraphNode(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.PrevGraphNode[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetPrevGraphNode
void htSparseChildGraphNodeDataSetPrevGraphNode(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htNode value)
{
    htSparseChildGraphNodeDatas.PrevGraphNode[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetNextTableGraphNode
htNode htSparseChildGraphNodeDataGetNextTableGraphNode(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.NextTableGraphNode[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetNextTableGraphNode
void htSparseChildGraphNodeDataSetNextTableGraphNode(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htNode value)
{
    htSparseChildGraphNodeDatas.NextTableGraphNode[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetNextDatadrawRootSparseChildGraphNodeData
htSparseChildGraphNodeData htSparseChildGraphNodeDataGetNextDatadrawRootSparseChildGraphNodeData(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.NextDatadrawRootSparseChildGraphNodeData[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData
void htSparseChildGraphNodeDataSetNextDatadrawRootSparseChildGraphNodeData(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htSparseChildGraphNodeData value)
{
    htSparseChildGraphNodeDatas.NextDatadrawRootSparseChildGraphNodeData[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetPrevDatadrawRootSparseChildGraphNodeData
htSparseChildGraphNodeData htSparseChildGraphNodeDataGetPrevDatadrawRootSparseChildGraphNodeData(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.PrevDatadrawRootSparseChildGraphNodeData[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData
void htSparseChildGraphNodeDataSetPrevDatadrawRootSparseChildGraphNodeData(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htSparseChildGraphNodeData value)
{
    htSparseChildGraphNodeDatas.PrevDatadrawRootSparseChildGraphNodeData[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseChildGraphNodeDataGetNextTableDatadrawRootSparseChildGraphNodeData
htSparseChildGraphNodeData htSparseChildGraphNodeDataGetNextTableDatadrawRootSparseChildGraphNodeData(
    htSparseChildGraphNodeData _SparseChildGraphNodeData)
{
    return htSparseChildGraphNodeDatas.NextTableDatadrawRootSparseChildGraphNodeData[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)];
}

#undef htSparseChildGraphNodeDataSetNextTableDatadrawRootSparseChildGraphNodeData
void htSparseChildGraphNodeDataSetNextTableDatadrawRootSparseChildGraphNodeData(
    htSparseChildGraphNodeData _SparseChildGraphNodeData,
    htSparseChildGraphNodeData value)
{
    htSparseChildGraphNodeDatas.NextTableDatadrawRootSparseChildGraphNodeData[htSparseChildGraphNodeData2Index(_SparseChildGraphNodeData)] = value;
}

#undef htSparseNodeNameDataGetNodeKey
htNode htSparseNodeNameDataGetNodeKey(
    htSparseNodeNameData _SparseNodeNameData)
{
    return htSparseNodeNameDatas.NodeKey[htSparseNodeNameData2Index(_SparseNodeNameData)];
}

#undef htSparseNodeNameDataSetNodeKey
void htSparseNodeNameDataSetNodeKey(
    htSparseNodeNameData _SparseNodeNameData,
    htNode value)
{
    htSparseNodeNameDatas.NodeKey[htSparseNodeNameData2Index(_SparseNodeNameData)] = value;
}

#undef htSparseNodeNameDataGetNameIndex
uint32 htSparseNodeNameDataGetNameIndex(
    htSparseNodeNameData _SparseNodeNameData)
{
    return htSparseNodeNameDatas.NameIndex[htSparseNodeNameData2Index(_SparseNodeNameData)];
}

#undef htSparseNodeNameDataSetNameIndex
void htSparseNodeNameDataSetNameIndex(
    htSparseNodeNameData _SparseNodeNameData,
    uint32 value)
{
    htSparseNodeNameDatas.NameIndex[htSparseNodeNameData2Index(_SparseNodeNameData)] = value;
}

#undef htSparseNodeNameDataGetNumName
uint32 htSparseNodeNameDataGetNumName(
    htSparseNodeNameData _SparseNodeNameData)
{
    return htSparseNodeNameDatas.NumName[htSparseNodeNameData2Index(_SparseNodeNameData)];
}

#undef htSparseNodeNameDataSetNumName
void htSparseNodeNameDataSetNumName(
    htSparseNodeNameData _SparseNodeNameData,
    uint32 value)
{
    htSparseNodeNameDatas.NumName[htSparseNodeNameData2Index(_SparseNodeNameData)] = value;
}

#undef htSparseNodeNameDataGetNextDatadrawRootSparseNodeNameData
htSparseNodeNameData htSparseNodeNameDataGetNextDatadrawRootSparseNodeNameData(
    htSparseNodeNameData _SparseNodeNameData)
{
    return htSparseNodeNameDatas.NextDatadrawRootSparseNodeNameData[htSparseNodeNameData2Index(_SparseNodeNameData)];
}

#undef htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData
void htSparseNodeNameDataSetNextDatadrawRootSparseNodeNameData(
    htSparseNodeNameData _SparseNodeNameData,
    htSparseNodeNameData value)
{
    htSparseNodeNameDatas.NextDatadrawRootSparseNodeNameData[htSparseNodeNameData2Index(_SparseNodeNameData)] = value;
}

#undef htSparseNodeNameDataGetPrevDatadrawRootSparseNodeNameData
htSparseNodeNameData htSparseNodeNameDataGetPrevDatadrawRootSparseNodeNameData(
    htSparseNodeNameData _SparseNodeNameData)
{
    return htSparseNodeNameDatas.PrevDatadrawRootSparseNodeNameData[htSparseNodeNameData2Index(_SparseNodeNameData)];
}

#undef htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData
void htSparseNodeNameDataSetPrevDatadrawRootSparseNodeNameData(
    htSparseNodeNameData _SparseNodeNameData,
    htSparseNodeNameData value)
{
    htSparseNodeNameDatas.PrevDatadrawRootSparseNodeNameData[htSparseNodeNameData2Index(_SparseNodeNameData)] = value;
}

#undef htSparseNodeNameDataGetNextTableDatadrawRootSparseNodeNameData
htSparseNodeNameData htSparseNodeNameDataGetNextTableDatadrawRootSparseNodeNameData(
    htSparseNodeNameData _SparseNodeNameData)
{
    return htSparseNodeNameDatas.NextTableDatadrawRootSparseNodeNameData[htSparseNodeNameData2Index(_SparseNodeNameData)];
}

#undef htSparseNodeNameDataSetNextTableDatadrawRootSparseNodeNameData
void htSparseNodeNameDataSetNextTableDatadrawRootSparseNodeNameData(
    htSparseNodeNameData _SparseNodeNameData,
    htSparseNodeNameData value)
{
    htSparseNodeNameDatas.NextTableDatadrawRootSparseNodeNameData[htSparseNodeNameData2Index(_SparseNodeNameData)] = value;
}

#undef htSparseNodeXDataGetNodeKey
htNode htSparseNodeXDataGetNodeKey(
    htSparseNodeXData _SparseNodeXData)
{
    return htSparseNodeXDatas.NodeKey[htSparseNodeXData2Index(_SparseNodeXData)];
}

#undef htSparseNodeXDataSetNodeKey
void htSparseNodeXDataSetNodeKey(
    htSparseNodeXData _SparseNodeXData,
    htNode value)
{
    htSparseNodeXDatas.NodeKey[htSparseNodeXData2Index(_SparseNodeXData)] = value;
}

#undef htSparseNodeXDataGetX
int32 htSparseNodeXDataGetX(
    htSparseNodeXData _SparseNodeXData)
{
    return htSparseNodeXDatas.X[htSparseNodeXData2Index(_SparseNodeXData)];
}

#undef htSparseNodeXDataSetX
void htSparseNodeXDataSetX(
    htSparseNodeXData _SparseNodeXData,
    int32 value)
{
    htSparseNodeXDatas.X[htSparseNodeXData2Index(_SparseNodeXData)] = value;
}

#undef htSparseNodeXDataGetNextDatadrawRootSparseNodeXData
htSparseNodeXData htSparseNodeXDataGetNextDatadrawRootSparseNodeXData(
    htSparseNodeXData _SparseNodeXData)
{
    return htSparseNodeXDatas.NextDatadrawRootSparseNodeXData[htSparseNodeXData2Index(_SparseNodeXData)];
}

#undef htSparseNodeXDataSetNextDatadrawRootSparseNodeXData
void htSparseNodeXDataSetNextDatadrawRootSparseNodeXData(
    htSparseNodeXData _SparseNodeXData,
    htSparseNodeXData value)
{
    htSparseNodeXDatas.NextDatadrawRootSparseNodeXData[htSparseNodeXData2Index(_SparseNodeXData)] = value;
}

#undef htSparseNodeXDataGetPrevDatadrawRootSparseNodeXData
htSparseNodeXData htSparseNodeXDataGetPrevDatadrawRootSparseNodeXData(
    htSparseNodeXData _SparseNodeXData)
{
    return htSparseNodeXDatas.PrevDatadrawRootSparseNodeXData[htSparseNodeXData2Index(_SparseNodeXData)];
}

#undef htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData
void htSparseNodeXDataSetPrevDatadrawRootSparseNodeXData(
    htSparseNodeXData _SparseNodeXData,
    htSparseNodeXData value)
{
    htSparseNodeXDatas.PrevDatadrawRootSparseNodeXData[htSparseNodeXData2Index(_SparseNodeXData)] = value;
}

#undef htSparseNodeXDataGetNextTableDatadrawRootSparseNodeXData
htSparseNodeXData htSparseNodeXDataGetNextTableDatadrawRootSparseNodeXData(
    htSparseNodeXData _SparseNodeXData)
{
    return htSparseNodeXDatas.NextTableDatadrawRootSparseNodeXData[htSparseNodeXData2Index(_SparseNodeXData)];
}

#undef htSparseNodeXDataSetNextTableDatadrawRootSparseNodeXData
void htSparseNodeXDataSetNextTableDatadrawRootSparseNodeXData(
    htSparseNodeXData _SparseNodeXData,
    htSparseNodeXData value)
{
    htSparseNodeXDatas.NextTableDatadrawRootSparseNodeXData[htSparseNodeXData2Index(_SparseNodeXData)] = value;
}

#endif
