/* Piper - the Swiss Army Knife of SOCKS/WinGate manipulation
 * Written by gcc@i.am (Chris Wilson) 16.1.99
 *
 * RCS ChangeLog:
 * $Log: util.c,v $
 * Revision 1.5  2001/07/20 18:24:19  chris
 * Added bouncer support (finally!)
 * More generic multiplex code (re-used by bouncer)
 *
 * Revision 1.4  2001/07/14 21:16:08  chris
 * Added stolen sigaction code from the Curl project:
 *   http://curl.haxx.se/mail/archive-2000-10/0091.html
 *
 * Revision 1.3  2001/07/14 15:10:25  chris
 * Removed limited distribution messages.
 * Licensed under GPL.
 *
 * Revision 1.2  1999/01/18 23:00:00  chris
 * Added sig_die to kill all children when parent dies
 * Included piper.h (doh!)
 * Added buffer flushes to cure weird problems in forking mode
 * Added more checks for malloc()s failing
 * Added test_host, investigate_host and portscan (from piper.c)
 *
 * Revision 1.1  1999/01/18 14:14:56  chris
 * Initial revision
 *
 */

#include "piper.h"

/*
 * resolve
 * Resolve a hostname to an IP address, return 0 on failure
 */
u_long resolve(char* host) {
  struct hostent * lu;
  u_long addr = inet_addr(host);

  if(addr == -1 ) {
    lu = gethostbyname(host);
    if((lu == NULL) || (lu->h_name == NULL) || (lu->h_addr_list == NULL)) {
      return 0;
    }
    memcpy(&(addr), *(lu->h_addr_list), sizeof(lu->h_addr_list));
  }
  return(addr);
}



/*
 * sig_timeout
 * signal handler which we install for SIGALRM if we want to die on timeouts
 */
void sig_timeout(int signum) {
  signal(SIGALRM, SIG_DFL);
  puts("Connection timed out, exiting.");
  exit(2);
}



void sig_die(int signum) {
  kill(0, signum);
  exit(3);
}



void sig_ignore(int signum) {
  /* this handler does nothing, but if it's called during a blocked system
     call, that call will be cancelled (unlike SIG_IGN) */
}



/*
 * split_host_port
 * separates a "host:port" string into host and port components
 *
 * If host is not supplied, return defhost.
 * If port is not supplied, return defport. 
 */
void split_host_port(char* hostport, char* hostbuf, int* portbuf,
		     char* defhost, int defport) {
  char* portstr;
  char* ptr;

  if (hostport == NULL || strcmp(hostport,"") == 0) {
    // Neither host nor port supplied, use defhost and defport
    printf("Empty hostname supplied, using default '%s:%i'\n",
	   defhost, defport);
    strcpy(hostbuf, defhost);
    *portbuf = defport;
    return;
  }

  portstr = strchr(hostport, ':');

  if (portstr == NULL) {
    // No port supplied, use defport
    printf("No port supplied, using default port %i\n", defport); 
    fflush(stdout);
    strcpy(hostbuf, hostport);
    *portbuf = defport;
    return;
  } else if (portstr == hostport) {
    // No host supplied (first character is ':') so use defhost
    printf("No hostname supplied, using default '%s'\n", defhost);
    fflush(stdout);
    strcpy(hostbuf, defhost);
  } else {
    // Both host and port supplied
    // First, replace the ':' with an end-of-string marker ('\0')
    *portstr = '\0';
    strcpy(hostbuf, hostport);
    // Change '\0' back to ':'
    *portstr = ':';
  }
  *portbuf = (int)strtol(portstr+1, &ptr, 0);
  if (*ptr != '\0') {
    // String contains invalid chars (see 'man strtol'), print error and exit
    printf("Invalid port number '%s', giving up\n", portstr+1);
    fflush(stdout);
    exit(1);
  }
}



/*
 * socks_error
 * wrapper for strerror, which adds helpful messages for our custom errors
 */
char* socks_error(int errnum) {
  if (errnum == ESOCK || errnum == -ESOCK) {
    return "can't get a socket (none left?)";
  } else if (errnum == ERESOLV || errnum == -ERESOLV) {
    return "can't resolve hostname";
  } else if (errnum == EFAIL || errnum == -EFAIL) {
    return "socks server failed to connect";
  } else if (errnum == ECLOSED || errnum == -ECLOSED) {
    return "connection refused by target host";
  } else {
    return strerror(errnum);
  }
}



/*
 * chomp
 * like Perl's chomp(), strips trailing newlines from strings
 */
void chomp(char* str) {
  int i;

  i = strlen(str) - 1;
  if (str[i] == '\r')
    str[i] = '\0';

  i = strlen(str) - 1;
  if (str[i] == '\n')
    str[i] = '\0';

  i = strlen(str) - 1;
  if (str[i] == '\r')
    str[i] = '\0';

}



/*
 * readhostlist
 * opens a file, reads a list of hosts from it, and returns an array of strings
 *
 * hosts MUST be one per line, without any extra formatting or spaces,
 * or resolve() will fail and the host will be skipped
 * the array of strings returned is NULL terminated
 */
char** readhostlist(char* hostfile) {
  char **oldlist = NULL, **newlist = NULL;
  char string[1024];
  char *buf;
  int numhosts = 0;
  int ptrsize = sizeof(&string);
  FILE* fp;

  fp = fopen(hostfile, "r");
  if (fp == NULL) {
    snprintf(string, 1023, "cannot open host file [%s]", hostfile);
    perror(string);
    exit(2);
  }

  while (fgets(string, 1023, fp) != NULL) {
    chomp(string);

    // chop the hostname at the first colon: allows output re-use
    buf = strchr(string, ':');
    if (buf) { *buf = '\0'; }

    // check that the host is valid
    if (resolve(string) == 0) {
      printf("cannot resolve [%s], skipping...\n", string);
      fflush(stdout);
    } else {
      // make a new list, with room for new host and the null terminator
      newlist = malloc((numhosts+2) * ptrsize);
      if (newlist == NULL) {
	perror("malloc()");
	exit(3);
      }

      if (numhosts > 0) memcpy(newlist, oldlist, numhosts*ptrsize);
      newlist[numhosts] = strdup(string);
      newlist[numhosts+1] = NULL;
      if (oldlist != NULL) free(oldlist);
      oldlist = newlist;
      numhosts++;
    }
  }

  fclose(fp);

  if (numhosts == 0) {
    printf("error: did not load any hosts from [%s]. aborting\n", hostfile);
    if (oldlist != NULL) free(oldlist);
    exit(2);
  }

  newlist[numhosts] = NULL;
  return newlist;
}

/*
 * freehostlist
 * frees the memory created by readhostlist
 */
void freehostlist(char** hostlist) {
  int i;

  for (i = 0; hostlist[i] != NULL; i++)
    free(hostlist[i]);

  free(hostlist);
}

/*
 * counthostlist
 * counts the number of hosts in a list
 */
int counthostlist(char** hostlist) {
  int i;

  for (i = 0; hostlist[i] != NULL; i++)
    ;

  return i;
}

/* 
 * makehostlist
 * makes a hostlist of one from a string (used for single-host mode)
 */
char** makehostlist(char* host) {
  char** hostlist;

  hostlist = malloc(sizeof(char*)*2);
  if (hostlist == NULL) {
    perror("malloc()");
    exit(2);
  }
  hostlist[0] = strdup(host);
  hostlist[1] = NULL;
  return hostlist;
}

/* makeintlist(intspec, &intlist)
 * makes a list of integers (e.g. ports) from a list specification
 * e.g. "1,20-30,40"
 *
 * the list is malloc()ed so it must be free()d after use.
 */
int makeintlist(char* intspec, int** intlist) {
  char *portspec = NULL, *endportspec = NULL;
  int   startport, endport;
  int  *ports = NULL, *newports = NULL;
  int   numports = 0, i = 0;
  char *buffer;

  buffer = strdup(intspec);
  if (buffer == NULL) {
    perror("strdup()");
    exit(2);
  }

  portspec = strtok(buffer, ",");
  while(portspec != NULL) {
    endportspec = strchr(portspec, '-');
    if (endportspec != NULL) {
      /* we were given a range of ports, so add them all */
      
      *endportspec = '\0'; /* replace '-' with string terminator */
      endportspec++;       /* skip to next character */
      startport = atoi(portspec);
      if (startport < 1 || startport > 65535) {
	printf("%s is not a valid port (should be 0-65535)\n", 
	       portspec);
	exit(1);
      }
      endport = atoi(endportspec);
      if (endport < 1 || endport > 65535) {
	printf("%s is not a valid port (should be 0-65535)\n",
	       endportspec);
	exit(1);
      }
      if (endport <= startport) {
	printf("dumbass, the second port should be bigger than the first\n");
	exit(1);
      }
      
      /* allocate more space, copy over, append new ports, free and swap */
      newports = malloc((numports + endport - startport + 1)*sizeof(int));
      if (newports == NULL) {
	perror("malloc()");
	exit(3);
      }

      if (ports != NULL) memcpy(newports, ports, numports*sizeof(int));
      for (i = startport; i <= endport; i++) {
	newports[numports++] = i;
      }
      if (ports != NULL) free(ports);
      ports = newports;
    } else {
      /* we were given a single port, so add it to the list */
      
      startport = atoi(portspec);
      if (startport < 1 || startport > 65535) {
	printf("%s is not a valid port (should be 0-65535)\n", 
	       portspec);
	exit(1);
      }
      
      /* allocate more memory, copy over, append new port, free and swap */
      newports = malloc((numports + 1)*sizeof(int));
      if (newports == NULL) {
	perror("malloc()");
	exit(3);
      }

      if (ports != NULL) memcpy(newports, ports, numports*sizeof(int));
      newports[numports++] = startport;
      if (ports != NULL) free(ports);
      ports = newports;
    }
    portspec = strtok(NULL, ",");
  }

  free(buffer);
  if (*intlist != NULL) free(*intlist);
  *intlist = ports;
  return numports;
}

/* 
 * readservices
 * retrieve a list of ports from /etc/services
 */
int readservices(int** portlist) {
  char  buffer[1024], *bufp = NULL;
  int  *ports = NULL, *newports = NULL;
  int   numports = 0, port = 0;
  FILE* fp;

  fp = fopen("/etc/services", "r");
  if (fp == NULL) {
    perror("/etc/services");
    exit(2);
  }
  while (fgets(buffer, 1023, fp) != NULL) {
    bufp = buffer;
    
    /* don't process udp lines or comments */
    if (strstr(buffer, "tcp") == NULL) continue;

    while (isspace(*bufp)) bufp++; /* skip any leading blanks */
    if (*bufp == '#') continue;    /* ignore comment lines */
    if (*bufp == 0) continue;      /* ignore lines which end there */
    while (isgraph(*bufp)) bufp++; /* skip the protocol name */
    if (*bufp == 0) continue;      /* ignore lines which end there */
    while (isspace(*bufp)) bufp++; /* skip the spaces/tabs */
    if (*bufp == 0) continue;      /* ignore lines which end there */
    
    /* *bufp = "portnumber/transport" hopefully e.g. "1/tcp" */
    if (strtok(bufp,"/") == NULL) continue; /* maybe there isn't one :) */
    /* this changes the '/' to a null terminator
       we don't want anything after the / anyway, so it's OK */
    port = atoi(bufp);
    if (port < 1 || port > 65536) {
      printf("invalid port %s in /etc/services, ignoring\n", bufp);
    }
    
    newports = malloc((numports+1)*sizeof(int));
    if (newports == NULL) {
      perror("malloc()");
      exit(3);
    }

    if (ports != NULL) memcpy(newports, ports, numports*sizeof(int));
    newports[numports++] = port;
    if (ports != NULL) free(ports);
    ports = newports;
  }

  fclose(fp);
  if (*portlist != NULL) free(*portlist);
  *portlist = ports;
  return numports;
}

/* socks_connect
 * Connect to a remote host using SOCKS
 *
 * NB Timeout is in seconds, 0 is no timeout. If you use a nonzero timeout,
 * it won't work unless you install a signal handler for SIGALRM *BEFORE*
 * you call socks_connect.
 */
int socks_connect(char* wingate, char* remotehost, int remoteport,
		  int verbose, int timeout) {
  int sock;

  int verr = (verbose >= V_ERR);
  int vtest = (verbose == V_TEST);

  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    if (verr) perror("not enough sockets");
    if (vtest) perror(wingate);
    return -ENOTSOCK;
  }

  return socks_connect_socket(wingate, remotehost, remoteport, verbose, 
                              timeout, sock);
}

int socks_connect_socket(char* wingate, char* remotehost, int remoteport,
                         int verbose, int timeout, int sock) {
  struct sockaddr_in wingate_addr;
  unsigned char buff[1024];
  int pktlen;
  u_long addr;
  struct timeval tv1, tv2;
  double tm = 0;
  struct sigaction sigact;

  int verr = (verbose >= V_ERR);
  int vall = (verbose >= V_ALL);
  int vtest = (verbose == V_TEST);

  wingate_addr.sin_family = AF_INET;
  wingate_addr.sin_port = htons(1080);

  if ((wingate_addr.sin_addr.s_addr = resolve(wingate)) == 0) {
    if (verr) printf("cannot resolve address of SOCKS server\n");
    if (vtest) printf("%s: unknown host)\n", wingate);
    return -ERESOLV;
  }

  if ((addr = resolve(remotehost)) == 0) {
    if (verr) printf("cannot resolve address of target host\n");
    if (vtest) printf("%s: unknown host\n", remotehost);
    return -ERESOLV;
  }

  if (vall) {
    gettimeofday(&tv1, (struct timezone *) NULL);
    printf("Connecting to %s:1080...", wingate);
    fflush(stdout);
  }

  if (timeout > 0) {
    // signal(SIGALRM, SIG_DFL);
    // signal(SIGALRM, sig_ignore);
    // signal(SIGALRM, SIG_IGN);
    // Stolen from Curl: http://curl.haxx.se/mail/archive-2000-10/0091.html
    sigaction(SIGALRM, NULL, &sigact);
    sigact.sa_handler = sig_ignore;
    sigact.sa_flags &= ~SA_RESTART;
    sigaction(SIGALRM, &sigact, NULL);
  } else if (timeout < 0) {
    signal(SIGALRM, sig_timeout);
  }
  alarm(abs(timeout));

  if (connect(sock, (struct sockaddr*)&wingate_addr, sizeof(wingate_addr))<0) {
    if (verr) perror("connect()");
    if (vtest) perror(wingate);
    alarm(0);
    return -ECONNREFUSED;
  }

  if (vall) {
    gettimeofday(&tv2, (struct timezone *) NULL);
    tm =  (tv2.tv_sec - tv1.tv_sec);
    tm += ((double)(tv2.tv_usec - tv1.tv_usec)) / 1000000;
  }

  alarm(abs(timeout));
  
  if (vall) printf("connected.\n"
		   "Connected to %s in %5.3f seconds.\n"
		   "Trying to authenticate...", wingate, tm);

  // Set up the authentication request
  pktlen = 3;
  buff[0] = 0x5; // SOCKS version 5
  buff[1] = 0x1; // We only support one authentication method...
  buff[2] = 0x0; // ... no authentication :)

  // Now send it to the wingate...
  if (write(sock, buff, pktlen) < pktlen) {
    if (verr) perror("write error");
    if (vtest) perror(wingate);
    alarm(0);
    return -EIO;
  }

  // Read back the response...
  if (read(sock, buff, 2) < 2) {
    if (verr) perror("read error");
    if (vtest) perror(wingate);
    alarm(0);
    return -EIO;
  }

  alarm(0);

  if (buff[0] != 0x5) {
    if (verr) printf("invalid SOCKS 5 response");
    if (vtest) printf("%s: invalid response - maybe not SOCKS 5\n", wingate);
    return -EINVAL;
  }

  if (buff[1] == 0xFF) {
    if (verr) printf("SOCKS server requires username/password\n");
    if (vtest) perror(wingate);
    return -EPERM;
  }

  if (buff[1] != 0x0) {
    if (verr) printf("invalid SOCKS 5 response\n");
    if (vtest) printf("%s: invalid response - maybe not SOCKS 5\n", wingate);
    return -EINVAL;
  }

  if (vall) printf("done.\nSending connection request...");

  /* Set up the connection request
     buff[0] is still 0x5 */
  buff[1] = 0x1; // Request to CONNECT
  buff[2] = 0x0; // reserved
  buff[3] = 0x1; // Address (to follow) is in IPv4 format
  buff[7] = (addr & 0xFF000000) >> 24;
  buff[6] = (addr & 0x00FF0000) >> 16;
  buff[5] = (addr & 0x0000FF00) >> 8;
  buff[4] = addr & 0xFF;
  *((u_int *)(buff+8)) = htons(remoteport);
  pktlen = 10;
  
  alarm(abs(timeout));

  // Send connection request to wingate
  if (write(sock, buff, pktlen) < pktlen) {
    if (verr) perror("write error");
    if (vtest) perror(wingate);
    alarm(0);
    return -EIO;
  }

  // Get the wingate's response...
  if (read(sock, buff, 4) < 4) {
    if (verr) perror("read error");
    if (vtest) perror(wingate);
    alarm(0);
    return -EIO;
  }

  alarm(0);

  if (buff[1] != 0) {
    if (vtest || verr) {
      fflush(stdout);
      if (vtest) printf("%s: ", wingate);
      switch(buff[1]) {
      case 1: printf("connection failed [%s]\n", remotehost); break;
      case 2: printf("access denied [%s]\n", remotehost); break;
      case 3: printf("network unreachable [%s]\n", remotehost); break;
      case 4: printf("host unreachable [%s]\n", remotehost); break;
      case 5: printf("connection refused [%s]\n", remotehost); break;
      case 6: printf("routing failure [%s]\n", remotehost); break;
      case 7: printf("command not supported\n"); break;
      case 8: printf("IPv4 not supported\n"); break;
      default: printf("unknown error code\n");
      }
      fflush(stdout);
    }

    close(sock); /* oops, forgot this one :) */
    
    switch(buff[1]) {
    case 1: return -EFAIL;
    case 2: return -EACCES;
    case 3: return -ENETUNREACH;
    case 4: return -EHOSTUNREACH;
    case 5: return -ECLOSED;
    case 6: return -EFAULT;
    case 7: return -EFAULT;
    case 8: return -EFAULT;
    case 9: return -EFAULT;
    default: return -EFAULT;
    }
  }

  /* server responded OK, read the host/port part of the reply and dump it */
  
  alarm(abs(timeout));

  switch(buff[3]) {
  case 1: // hostname is in IPv4 format...
    if (read(sock, buff+4, 6) != 6) {
      if (verr) perror("Read error");
      if (vtest) perror(wingate);
      alarm(0);
      return -EIO;
    }
    break;
  case 3: // hostname as text
    if (read(sock, buff+4, 1)!=1) {
      if (verr) perror("Read error");
      if (vtest) perror(wingate);
      alarm(0);
      return -EIO;
    }

    pktlen = buff[4];
    // 5th byte of reply contains length of returned hostname
    
    if (read(sock, buff+5, pktlen+2) != pktlen+2) {
      if (verr) perror("Read error");
      if (vtest) perror(wingate);
      alarm(0);
      return -EIO;
    }
    break;
  default:
    if (verr) printf("proxy returned neither IPv4 address nor hostname\n");
    if (vtest) printf("%s: bad address reply\n", wingate);

    // This wouldn't be critical, except that we need to be able to read
    // the rest of the reply, so the buffer is clear when we talk to the
    // remote host. And we can't do that unless we know how big the address is

    alarm(0);
    return -EFAULT;
  }

  alarm(0);
  return sock;
}



void multiplex_con(int sock) {
  multiplex_con_2(0, sock);
}



void multiplex_con_2(int insock, int outsock) {
  // struct termios tios, oldtios;
  int ret, open = 1;
  char buff[2048];
  fd_set read_fds;
  int maxfd;

  // Do some magic incantations on the stdin. This makes your terminal fully
  // interactive, like telnet, so ^C is sent to the remote host rather than
  // killing piper. Also you can use arrow keys in bash and even interactive
  // editors like vi and pico! :)

  /*
  tcgetattr(0, &tios);    // "gimme your money or your terminal settings!"
  oldtios = tios;         // Save a backup to restore settings later
  tios.c_lflag &= !(ICANON);  // set to interactive (non-buffered) mode
  tios.c_lflag &= !(ISIG);    // Signals (like ^Z) lose special meaning
  tios.c_iflag |= IGNBRK;     // Ignore break... at long last I found it!
  tios.c_iflag |= BRKINT;     // Make sure break is sent as '\0'
  tcsetattr(0, TCSANOW, &tios); // take THAT <bam!>
  */

  maxfd = 0;
  if (insock  > maxfd) maxfd = insock;
  if (outsock > maxfd) maxfd = outsock;

  while (open) {
    FD_ZERO(&read_fds);
    FD_SET(insock,  &read_fds);
    FD_SET(outsock, &read_fds);
    select(maxfd+1, &read_fds, NULL, NULL, NULL);

    if (FD_ISSET(insock, &read_fds)) {
      // Data is waiting on the internet socket
      ret = read(insock, buff, 2048);
      if (ret <= 0) {
	printf("Connection closed by client host.\n");
	open = 0;
      } else if (write(outsock, buff, ret) != ret) {
	perror("Writing to remote host");
	open = 0;
      }
    }
    if (FD_ISSET(outsock, &read_fds)) {
      // Data has been typed and is waiting to be sent
      ret = read(outsock, buff, 2048);
      if (ret <= 0) {
	printf("Connection closed by remote host.\n");
	open = 0;
      } else if (write(insock, buff, ret) != ret) {
	// perror("multiplex() write error");
	perror("Writing to client host");
	open = 0;
      }
    }
  }

  // Will the terminal restore its own settings? wait and see...
}



/* 
 * testhost
 * the lowlevel routine for socks-server testing
 * connects to a SOCKS server and uses it to connect to a known host:port
 * prints an appropriate status message (including time taken)
 */
int testhost(char* server, char* host, int port, int verbose, int timeout) {
  struct timeval tv1, tv2;
  double tm;
  int sock;

  if (verbose >= V_ALL) {
    printf("Connecting to %s:%i (via %s:1080)...\n",
	   host, port, server);
  }
    
  if (gettimeofday(&tv1, (struct timezone *) NULL) < 0) {
    perror("gettimeofday()");
    return errno;
  }
    
  /* call socks_connect with a special verbosity setting, V_TEST
     this gives us nice-looking and more appropriate output */
  sock = socks_connect(server, host, port, V_TEST, timeout);

  if (gettimeofday(&tv2, (struct timezone *) NULL) < 0) {
    perror("gettimeofday()");
    return errno;
  }
  if (sock < 0) {
    return sock;
  } else {
    tm =  (tv2.tv_sec - tv1.tv_sec);
    tm += ((double)(tv2.tv_usec - tv1.tv_usec)) / 1000000;
    printf("%s: test passed (%1.3f seconds)\n", server, tm);
    close(sock);
  }

  return 0;
}



/*
 * investigate_host
 * the low-level routine for host investigation
 *
 * connects to a port on a host, reads the first line of text, prints it
 * and disconnects.
 */
int investigate_host(char* wingate, char* host, int port, int verbose) {
  int sock, i;
  char* buffer;
  char buf2[1024];

  if (verbose >= V_ALL) {
    printf("Connecting to %s:%i (via %s:1080)...\n",
	   host, port, wingate);
  }
  
  sock = socks_connect(wingate, host, port, verbose, 20);

  if (sock < 0) {
    if (verbose == V_TEST) printf("%s: %s\n", host, socks_error(sock));
    if (verbose >= V_ERR) puts(socks_error(sock));
    if (sock == -EFAIL || sock == -ECLOSED) {
      return ST_FAIL; /* we know the problem is with the remote host */
    } else {
      return ST_UNKNOWN; /* the problem is probably with the wingate */
    }
  } else {
    buffer = buf2;
    buffer[0] = 0;

    while (strlen(buffer) == 0) {
      i = read(sock, buffer, 1024);
      buffer[i] = 0;
      chomp(buffer);

      // get rid of any control characters at the start of the line
      // makes the output from telnet ports look nicer :)

      while (buffer[1] != 0 && !isalnum(buffer[0]))
	buffer++;
    }
    puts(buffer);
    close(sock);
    return ST_OK;
  }
  puts("investigate_host(): oops! we shouldn't be here");
  return ST_UNKNOWN;
}



/*
 * portscan
 * the lowlevel routine for portscanning
 * connects to a port on a host, using a SOCKS server, and prints
 * an appropriate status message
 */
  
int portscan(char* server, char* host, int port, int verbose) {
  int sock;

  /* we don't want any warnings or errors from socks_connect */
  sock = socks_connect(server, host, port, V_NONE, 20);
  
  printf("%s:%i ", host, port);

  if (sock > 0) {
    puts("is open");
    close(sock);
  } else if (sock == -EFAIL || sock == -ECLOSED) {
    puts("is probably closed");
  } else {
    printf("is unknown (server failed: %s)\n", server);
  }
  fflush(stdout);
  if (sock > 0) {
    return ST_OK;
  } else if (sock == -EFAIL || sock == -ECLOSED) {
    return ST_FAIL;
  } else {
    return ST_UNKNOWN;
  }
  puts("portscan(): oops, we shouldn't be here!");
  return ST_UNKNOWN;
}



char* nexthost(char*** ptr, char** list) {
  if (**ptr == NULL)
    *ptr = list;
  return *((*ptr)++);
}

