/************************************************************/
/*                                                          */
/* Module ID  - cpint.c                                     */
/*                                                          */
/* Function   - Provide interface between Linux/390 (VM) &  */
/*              various CP interfaces.                      */
/*                                                          */
/* Called By  -                                             */
/*                                                          */
/* Calling To -                                             */
/*                                                          */
/* Parameters - (1)                                         */
/*              DESCRIPTION:                                */
/*                                                          */
/* Notes      - (1) ....................................... */
/*                                                          */
/*              (2) ....................................... */
/*                                                          */
/*                                                          */
/* Name       - Neale Ferguson.                             */
/*                                                          */
/* Date       - January, 2000.                              */
/*                                                          */
/*                                                          */
/* Associated    - (1) Refer To ........................... */
/* Documentation                                            */
/*                 (2) Refer To ........................... */
/*                                                          */
/************************************************************/

/************************************************************/
/*                                                          */
/*                     DEFINES                              */
/*                     -------                              */
/*                                                          */
/************************************************************/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

/*=============== End of Defines ===========================*/

/************************************************************/
/*                                                          */
/*              INCLUDE STATEMENTS                          */
/*              ------------------                          */
/*                                                          */
/************************************************************/

#include <linux/config.h>
#include <linux/version.h>
#include <linux/kmod.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/init.h>
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/device.h>
#include <linux/sysdev.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/ioctl.h>
#include <linux/ioctl32.h>

#include <asm/system.h>		/* cli(), *_flags */
#include <asm/segment.h>	/* memcpy and such */
#include <asm/ebcdic.h>		/* ebcdic stuff */
#include <asm/uaccess.h>	/* copy to/from user space */

#include "cpint.h"		/* local definitions */
#include "cpcmd.h"
#include "applmon.h"
#include "idcmd.h"

/*================== End of Include Statements =============*/

/************************************************************/
/*                                                          */
/*              TYPE DEFINITIONS                            */
/*              ----------------                            */
/*                                                          */
/************************************************************/

#ifdef __s390x__
typedef struct {
	int code;
	int (*handler) (unsigned int,
			unsigned int, unsigned long, struct file *);
} Converters;
#endif

/*================== End of Type Definitions ===============*/

/************************************************************/
/*                                                          */
/*             FUNCTION PROTOTYPES                          */
/*             -------------------                          */
/*                                                          */
/************************************************************/

MODULE_AUTHOR("Neale Ferguson <Neale.Ferguson@SoftwareAG-USA.com>");
MODULE_DESCRIPTION("Linux on S/390 interface to VM CP command."
		   " Copyright 2000-2003 Neale Ferguson");
MODULE_LICENSE("GPL");

static int cpint_open(struct inode *, struct file *);

static int __init cpint_init(void);
static void __exit cpint_cleanup(void);

/*================== End of Prototypes =====================*/

/************************************************************/
/*                                                          */
/*           GLOBAL VARIABLE DECLARATIONS                   */
/*           ----------------------------                   */
/*                                                          */
/************************************************************/

static int cpint_major = CPINT_MAJOR;
static int cpint_devs_map = 0;
int cpint_nr_devs = CPINT_NR_DEVS;

CPInt_Dev *cpint_devices;

static struct class_simple *cpint_class;

/*----------------------------------------------------------*/
/* The different file operations                            */
/*----------------------------------------------------------*/

static struct file_operations cpint_fops = {
      open:cpint_open,
};

static struct file_operations idcmd_fops = {
      open:idcmd_open,
      ioctl:idcmd_ioctl,
      release:idcmd_release,
};

static struct file_operations cpcmd_fops = {
      read:cpcmd_read,
      write:cpcmd_write,
      open:cpcmd_open,
      ioctl:cpcmd_ioctl,
      release:cpcmd_release,
};

static struct file_operations applmon_fops = {
      write:applmon_write,
      open:applmon_open,
      ioctl:applmon_ioctl,
      release:applmon_release,
};

static struct file_operations account_fops = {
      write:account_write,
      open:account_open,
      release:account_release,
};

/*----------------------------------------------------------*/
/* Map the minor number to an index into a file-ops table   */
/* - value represents an index into the cpFops structure    */
/*----------------------------------------------------------*/
static int fopMap[256] =
    { 3, -1, -1, -1, -1, -1, -1, -1, 0, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 2, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};

/*----------------------------------------------------------*/
/* Array of file-operations structures for each device type */
/*----------------------------------------------------------*/
struct file_operations *cpFops[] = {
	&cpcmd_fops,
	&applmon_fops,
	&account_fops,
	&idcmd_fops
};

char *cpNames[] = {
	"cpcmd",
	"cpmod",
	"cpact",
	"cpid",
};

#ifdef __s390x__
static Converters converters[] = {
	{CPIOCRC, NULL},
	{CPIOUPC, NULL},
	{IDGETBF, NULL},
	{MONSETPROD, NULL},
	{-1, NULL}
};
#endif

/*============== End of Variable Declarations ==============*/

/************************************************************/
/*                                                          */
/* Name       - cpint_open.                                 */
/*                                                          */
/* Function   - Determine what CP interface "device" the    */
/*              caller requires. Check that it exists and   */
/*              if so, install the appropriate file_ops     */
/*              pointers and call the required open routine.*/
/*                                                          */
/* Parameters - inode -                                     */
/*              filp  -                                     */
/*                                                          */
/************************************************************/

static int
cpint_open(struct inode *inode, struct file *filp)
{
	int num = MINOR(inode->i_rdev), opIndex;
	CPInt_Dev *dev;

	if (num >= cpint_nr_devs) {
		printk(KERN_ERR
		       "cpint: device number %d exceeds max of %d\n",
		       num, cpint_nr_devs);
		return -ENODEV;
	}

	dev = &cpint_devices[num];

	opIndex = fopMap[num];
	if (opIndex > -1) {
		filp->f_op = cpFops[opIndex];
		return (filp->f_op->open(inode, filp));
	}

	return -ENODEV;
}

/*===================== End of Function ====================*/

/************************************************************/
/*                                                          */
/* Name       - cpint_init.                                 */
/*                                                          */
/* Function   - Initialize this device driver.              */
/*                                                          */
/************************************************************/

static int __init
cpint_init(void)
{
	int result, i_dev, i_conv;
	CPInt_Dev *dev;
	union {
		long long cpuid;
		char cpu[8];
	} id;

      asm("STIDP\t%0\n":"=m"(id.cpuid)
	    );

	if (id.cpu[0] != 0xff)
		return -EACCES;

    /*------------------------------------------------------*/
	/* Register your major, and accept a dynamic number     */
    /*------------------------------------------------------*/
	result = register_chrdev(cpint_major, "cpint", &cpint_fops);
	if (result < 0) {
		printk(KERN_WARNING "cpint: can't get major %d\n", cpint_major);
		return result;
	}
	if (cpint_major == 0)
		cpint_major = result;

    /*------------------------------------------------------*/
	/* allocate the devices -- we can't have them static,   */
	/* as the number can be specified at load time          */
    /*------------------------------------------------------*/
	cpint_devices = kmalloc(cpint_nr_devs * sizeof (CPInt_Dev), GFP_KERNEL);
	if (!cpint_devices) {
		result = -ENOMEM;
		unregister_chrdev(cpint_major, "cpint");
		return result;
	}

    /*------------------------------------------------------*/
	/* Register this device class                           */
    /*------------------------------------------------------*/
	cpint_class = class_simple_create(THIS_MODULE, "cpint");
	if (IS_ERR(cpint_class)) {
		printk(KERN_ERR "Error creating cpint class.\n");
		kfree(cpint_devices);
		unregister_chrdev(cpint_major, "cpint");
		return -ENOMEM;
	}

    /*------------------------------------------------------*/
	/* Register the individual devices                      */
    /*------------------------------------------------------*/
	for (i_dev = 0; i_dev < CPINT_NR_DEVS; i_dev++) {
		int fopIdx;

		fopIdx = fopMap[i_dev];
		if (fopIdx > -1 && !(cpint_devs_map & (1 << fopIdx))) {
			class_simple_device_add(cpint_class,
						MKDEV(cpint_major, i_dev), NULL,
						cpNames[fopIdx]);
			cpint_devs_map |= 1 << fopIdx;
		}
	}

	memset(cpint_devices, 0, cpint_nr_devs * sizeof (CPInt_Dev));
	for (dev = cpint_devices, i_dev = 0;
	     i_dev < cpint_nr_devs; dev++, i_dev++)
		init_waitqueue_head(&dev->devWait);

#ifdef __s390x__
	for (i_conv = 0; converters[i_conv].code != -1; i_conv++) {
		int rc;

		rc = register_ioctl32_conversion(converters[i_conv].code,
						 converters[i_conv].handler);

		if (rc < 0)
			printk("Unable to register ioctl32 handlers");
	}
#endif

	return 0;

}

module_init(cpint_init);

/*===================== End of Function ====================*/

/************************************************************/
/*                                                          */
/* Name       - cpint_cleanup.                              */
/*                                                          */
/* Function   - Unload this device driver.                  */
/*                                                          */
/************************************************************/

static void __exit
cpint_cleanup(void)
{
	int i_dev, i_conv;

#ifdef __s390x__
	for (i_conv = 0; converters[i_conv].code != -1; i_conv++) {
		int rc;

		rc = unregister_ioctl32_conversion(converters[i_conv].code);

		if (rc < 0)
			printk("Unable to unregister ioctl32 handlers");
	}
#endif

	for (i_dev = 0; i_dev < CPINT_NR_DEVS; i_dev++) {
		int fopIdx;

		fopIdx = fopMap[i_dev];
		if (fopIdx > -1 && (cpint_devs_map & (1 << fopIdx))) {
			class_simple_device_remove(MKDEV(cpint_major, i_dev));
		}
	}
	cpint_devs_map = 0;
	class_simple_destroy(cpint_class);

	unregister_chrdev(cpint_major, "cpint");
	for (i_dev = 0; i_dev < CPINT_NR_DEVS; i_dev++) {
		if (cpint_devices[i_dev].devExt)
			kfree(cpint_devices[i_dev].devExt);
	}
	kfree(cpint_devices);

}

module_exit(cpint_cleanup);

/*===================== End of Function ====================*/
