/**
 * @jsxRuntime classic
 * @jsx jsx
 */
import { type ComponentPropsWithoutRef, type ComponentPropsWithRef, type ReactElement, type ReactNode } from 'react';
import { type StrictXCSSProp, type XCSSAllProperties, type XCSSAllPseudos } from '@atlaskit/css';
import type { BackgroundColorToken, SVGElements } from '../../utils/types';
import type { BasePrimitiveProps, PaddingToken, StyleProp, SurfaceColorToken } from './types';
type AllowedElements = Exclude<keyof JSX.IntrinsicElements, SVGElements | 'button' | 'a'>;
type CustomElementType<P = any> = {
    [K in AllowedElements]: P extends JSX.IntrinsicElements[K] ? K : never;
}[AllowedElements];
export type BoxProps<T extends CustomElementType> = Omit<ComponentPropsWithoutRef<T>, 'as' | 'className'> & Omit<BasePrimitiveProps, 'xcss' | 'style'> & BaseBoxProps<T>;
type BaseBoxProps<T extends CustomElementType> = {
    /**
     * The DOM element to render as the Box.
     * - This cannot be any SVG-related element such as `'svg'`, `'animate', `'circle'`, and many more
     * - This cannot be a `'a'` (use the `Anchor` primitive instead)
     * - This cannot be a `'button'` (use the `Anchor` primitive instead)
     * @default 'div'
     */
    as?: T;
    /**
     * Elements to be rendered inside the Box.
     */
    children?: ReactNode;
    /**
     * Token representing background color with a built-in fallback value.
     */
    backgroundColor?: SurfaceColorToken | BackgroundColorToken;
    /**
     * Tokens representing CSS shorthand for `paddingBlock` and `paddingInline` together.
     *
     * @see paddingBlock
     * @see paddingInline
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     */
    padding?: PaddingToken;
    /**
     * Tokens representing CSS shorthand `paddingBlock`.
     *
     * @see paddingBlockStart
     * @see paddingBlockEnd
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     */
    paddingBlock?: PaddingToken;
    /**
     * Tokens representing CSS `paddingBlockStart`.
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     */
    paddingBlockStart?: PaddingToken;
    /**
     * Tokens representing CSS `paddingBlockEnd`.
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     */
    paddingBlockEnd?: PaddingToken;
    /**
     * Tokens representing CSS shorthand `paddingInline`.
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     *
     * @see paddingInlineStart
     * @see paddingInlineEnd
     */
    paddingInline?: PaddingToken;
    /**
     * Tokens representing CSS `paddingInlineStart`.
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     */
    paddingInlineStart?: PaddingToken;
    /**
     * Tokens representing CSS `paddingInlineEnd`.
     * @private
     * @deprecated – Do not use shorthand props, use `props.xcss` instead as these will be deprecated in the future.
     */
    paddingInlineEnd?: PaddingToken;
    /**
     * Forwarded ref.
     */
    ref?: ComponentPropsWithRef<T>['ref'];
    /**
     * Inline styles to be applied to the Box. Only apply as a last resort, or where
     * styles cannot otherwise be calculated outside of the runtime of the component they're applied.
     */
    style?: Omit<StyleProp['style'], 'background' | 'backgroundColor'>;
    /**
     * Apply a subset of permitted styles powered by Atlassian Design System design tokens.
     * It's preferred you do not use `background` in `xcss` or `cssMap()` and instead use `props.backgroundColor` for surface awareness.
     */
    xcss?: StrictXCSSProp<Exclude<XCSSAllProperties, 'background'>, XCSSAllPseudos>;
};
type BoxComponent = <T extends CustomElementType>(props: BoxProps<T>) => ReactElement | null;
/**
 * __Box__
 *
 * A Box is a primitive component that has the design decisions of the Atlassian Design System baked in.
 * Renders a `div` by default.
 *
 * - [Examples](https://atlassian.design/components/primitives/box/examples)
 * - [Code](https://atlassian.design/components/primitives/box/code)
 * - [Usage](https://atlassian.design/components/primitives/box/usage)
 */
declare const Box: BoxComponent;
export default Box;
