use crate::extractors;
use crate::signatures;

/// Returns a list of all supported signatures, including their "magic" byte patterns, parser functions, and any associated extractor.
pub fn patterns() -> Vec<signatures::common::Signature> {
    let binary_signatures: Vec<signatures::common::Signature> = vec![
        // gzip
        signatures::common::Signature {
            name: "gzip".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::gzip::gzip_magic(),
            parser: signatures::gzip::gzip_parser,
            description: signatures::gzip::DESCRIPTION.to_string(),
            extractor: Some(extractors::gzip::gzip_extractor()),
        },
        // .deb
        signatures::common::Signature {
            name: "deb".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::deb::deb_magic(),
            parser: signatures::deb::deb_parser,
            description: signatures::deb::DESCRIPTION.to_string(),
            extractor: None,
        },
        // 7-zip
        signatures::common::Signature {
            name: "7zip".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::sevenzip::sevenzip_magic(),
            parser: signatures::sevenzip::sevenzip_parser,
            description: signatures::sevenzip::DESCRIPTION.to_string(),
            extractor: Some(extractors::sevenzip::sevenzip_extractor()),
        },
        // xz
        signatures::common::Signature {
            name: "xz".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::xz::xz_magic(),
            parser: signatures::xz::xz_parser,
            description: signatures::xz::DESCRIPTION.to_string(),
            extractor: Some(extractors::lzma::lzma_extractor()),
        },
        // tarball
        signatures::common::Signature {
            name: "tarball".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::tarball::tarball_magic(),
            parser: signatures::tarball::tarball_parser,
            description: signatures::tarball::DESCRIPTION.to_string(),
            extractor: Some(extractors::tarball::tarball_extractor()),
        },
        // squashfs
        signatures::common::Signature {
            name: "squashfs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::squashfs::squashfs_magic(),
            parser: signatures::squashfs::squashfs_parser,
            description: signatures::squashfs::DESCRIPTION.to_string(),
            extractor: Some(extractors::squashfs::squashfs_extractor()),
        },
        // dlob
        signatures::common::Signature {
            name: "dlob".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::dlob::dlob_magic(),
            parser: signatures::dlob::dlob_parser,
            description: signatures::dlob::DESCRIPTION.to_string(),
            extractor: None,
        },
        // lzma
        signatures::common::Signature {
            name: "lzma".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::lzma::lzma_magic(),
            parser: signatures::lzma::lzma_parser,
            description: signatures::lzma::DESCRIPTION.to_string(),
            //extractor: Some(extractors::sevenzip::sevenzip_extractor()),
            extractor: Some(extractors::lzma::lzma_extractor()),
        },
        // bzip2
        signatures::common::Signature {
            name: "bzip2".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::bzip2::bzip2_magic(),
            parser: signatures::bzip2::bzip2_parser,
            description: signatures::bzip2::DESCRIPTION.to_string(),
            extractor: Some(extractors::bzip2::bzip2_extractor()),
        },
        // uimage
        signatures::common::Signature {
            name: "uimage".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::uimage::uimage_magic(),
            parser: signatures::uimage::uimage_parser,
            description: signatures::uimage::DESCRIPTION.to_string(),
            extractor: Some(extractors::uimage::uimage_extractor()),
        },
        // packimg header
        signatures::common::Signature {
            name: "packimg".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::packimg::packimg_magic(),
            parser: signatures::packimg::packimg_parser,
            description: signatures::packimg::DESCRIPTION.to_string(),
            extractor: None,
        },
        // crc32 constants
        signatures::common::Signature {
            name: "crc32".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::hashes::crc32_magic(),
            parser: signatures::hashes::crc32_parser,
            description: signatures::hashes::CRC32_DESCRIPTION.to_string(),
            extractor: None,
        },
        // sha256 constants
        signatures::common::Signature {
            name: "sha256".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::hashes::sha256_magic(),
            parser: signatures::hashes::sha256_parser,
            description: signatures::hashes::SHA256_DESCRIPTION.to_string(),
            extractor: None,
        },
        // cpio
        signatures::common::Signature {
            name: "cpio".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::cpio::cpio_magic(),
            parser: signatures::cpio::cpio_parser,
            description: signatures::cpio::DESCRIPTION.to_string(),
            extractor: Some(extractors::sevenzip::sevenzip_extractor()),
        },
        // iso9660 primary volume
        signatures::common::Signature {
            name: "iso9660".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::iso9660::iso_magic(),
            parser: signatures::iso9660::iso_parser,
            description: signatures::iso9660::DESCRIPTION.to_string(),
            extractor: Some(extractors::tsk::tsk_extractor()),
        },
        // linux kernel
        signatures::common::Signature {
            name: "linux_kernel".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::linux::linux_kernel_version_magic(),
            parser: signatures::linux::linux_kernel_version_parser,
            description: signatures::linux::LINUX_KERNEL_VERSION_DESCRIPTION.to_string(),
            extractor: Some(extractors::linux::linux_kernel_extractor()),
        },
        // linux boot image
        signatures::common::Signature {
            name: "linux_boot_image".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::linux::linux_boot_image_magic(),
            parser: signatures::linux::linux_boot_image_parser,
            description: signatures::linux::LINUX_BOOT_IMAGE_DESCRIPTION.to_string(),
            extractor: None,
        },
        // zstd
        signatures::common::Signature {
            name: "zstd".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::zstd::zstd_magic(),
            parser: signatures::zstd::zstd_parser,
            description: signatures::zstd::DESCRIPTION.to_string(),
            extractor: Some(extractors::zstd::zstd_extractor()),
        },
        // zip
        signatures::common::Signature {
            name: "zip".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::zip::zip_magic(),
            parser: signatures::zip::zip_parser,
            description: signatures::zip::DESCRIPTION.to_string(),
            extractor: Some(extractors::sevenzip::sevenzip_extractor()),
        },
        // Intel PCH ROM
        signatures::common::Signature {
            name: "pchrom".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::pchrom::pch_rom_magic(),
            parser: signatures::pchrom::pch_rom_parser,
            description: signatures::pchrom::DESCRIPTION.to_string(),
            extractor: Some(extractors::uefi::uefi_extractor()),
        },
        // UEFI PI volume
        signatures::common::Signature {
            name: "uefi_pi_volume".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::uefi::uefi_volume_magic(),
            parser: signatures::uefi::uefi_volume_parser,
            description: signatures::uefi::VOLUME_DESCRIPTION.to_string(),
            extractor: Some(extractors::uefi::uefi_extractor()),
        },
        // UEFI capsule image
        signatures::common::Signature {
            name: "uefi_capsule".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::uefi::uefi_capsule_magic(),
            parser: signatures::uefi::uefi_capsule_parser,
            description: signatures::uefi::CAPSULE_DESCRIPTION.to_string(),
            extractor: Some(extractors::uefi::uefi_extractor()),
        },
        // PDF document
        signatures::common::Signature {
            name: "pdf".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::pdf::pdf_magic(),
            parser: signatures::pdf::pdf_parser,
            description: signatures::pdf::DESCRIPTION.to_string(),
            extractor: None,
        },
        // ELF
        signatures::common::Signature {
            name: "elf".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::elf::elf_magic(),
            parser: signatures::elf::elf_parser,
            description: signatures::elf::DESCRIPTION.to_string(),
            extractor: None,
        },
        // CramFS
        signatures::common::Signature {
            name: "cramfs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::cramfs::cramfs_magic(),
            parser: signatures::cramfs::cramfs_parser,
            description: signatures::cramfs::DESCRIPTION.to_string(),
            extractor: Some(extractors::sevenzip::sevenzip_extractor()),
        },
        // QNX IFS
        // TODO: The signature and extractor are untested. Need a sample IFS image.
        signatures::common::Signature {
            name: "qnx_ifs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::qnx::qnx_ifs_magic(),
            parser: signatures::qnx::qnx_ifs_parser,
            description: signatures::qnx::IFS_DESCRIPTION.to_string(),
            extractor: Some(extractors::dumpifs::dumpifs_extractor()),
        },
        // RomFS
        signatures::common::Signature {
            name: "romfs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::romfs::romfs_magic(),
            parser: signatures::romfs::romfs_parser,
            description: signatures::romfs::DESCRIPTION.to_string(),
            extractor: Some(extractors::romfs::romfs_extractor()),
        },
        // EXT
        signatures::common::Signature {
            name: "ext".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::ext::ext_magic(),
            parser: signatures::ext::ext_parser,
            description: signatures::ext::DESCRIPTION.to_string(),
            extractor: Some(extractors::tsk::tsk_extractor()),
        },
        // CAB archive
        signatures::common::Signature {
            name: "cab".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::cab::cab_magic(),
            parser: signatures::cab::cab_parser,
            description: signatures::cab::DESCRIPTION.to_string(),
            extractor: Some(extractors::cab::cab_extractor()),
        },
        // JFFS2
        signatures::common::Signature {
            name: "jffs2".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::jffs2::jffs2_magic(),
            parser: signatures::jffs2::jffs2_parser,
            description: signatures::jffs2::DESCRIPTION.to_string(),
            extractor: Some(extractors::jffs2::jffs2_extractor()),
        },
        // YAFFS
        signatures::common::Signature {
            name: "yaffs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::yaffs::yaffs_magic(),
            parser: signatures::yaffs::yaffs_parser,
            description: signatures::yaffs::DESCRIPTION.to_string(),
            extractor: Some(extractors::yaffs2::yaffs2_extractor()),
        },
        // lz4
        signatures::common::Signature {
            name: "lz4".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::lz4::lz4_magic(),
            parser: signatures::lz4::lz4_parser,
            description: signatures::lz4::DESCRIPTION.to_string(),
            extractor: Some(extractors::lz4::lz4_extractor()),
        },
        // lzop
        signatures::common::Signature {
            name: "lzop".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::lzop::lzop_magic(),
            parser: signatures::lzop::lzop_parser,
            description: signatures::lzop::DESCRIPTION.to_string(),
            extractor: Some(extractors::lzop::lzop_extractor()),
        },
        // lzop
        signatures::common::Signature {
            name: "pe".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::pe::pe_magic(),
            parser: signatures::pe::pe_parser,
            description: signatures::pe::DESCRIPTION.to_string(),
            extractor: None,
        },
        // zlib
        signatures::common::Signature {
            name: "zlib".to_string(),
            // The magic bytes for this signature are only 2 bytes, only match on the beginning of a file
            short: true,
            magic_offset: 0,
            always_display: false,
            magic: signatures::zlib::zlib_magic(),
            parser: signatures::zlib::zlib_parser,
            description: signatures::zlib::DESCRIPTION.to_string(),
            extractor: Some(extractors::zlib::zlib_extractor()),
        },
        // gpg signed data
        signatures::common::Signature {
            name: "gpg_signed".to_string(),
            // The magic bytes for this signature are only 2 bytes, only match on the beginning of a file
            short: true,
            magic_offset: 0,
            always_display: false,
            magic: signatures::gpg::gpg_signed_magic(),
            parser: signatures::gpg::gpg_signed_parser,
            description: signatures::gpg::GPG_SIGNED_DESCRIPTION.to_string(),
            extractor: Some(extractors::zlib::zlib_extractor()),
        },
        // pem certificates
        signatures::common::Signature {
            name: "pem_certificate".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::pem::pem_certificate_magic(),
            parser: signatures::pem::pem_parser,
            description: signatures::pem::PEM_CERTIFICATE_DESCRIPTION.to_string(),
            extractor: Some(extractors::pem::pem_certificate_extractor()),
        },
        // pem public keys
        signatures::common::Signature {
            name: "pem_public_key".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::pem::pem_public_key_magic(),
            parser: signatures::pem::pem_parser,
            description: signatures::pem::PEM_PUBLIC_KEY_DESCRIPTION.to_string(),
            extractor: Some(extractors::pem::pem_key_extractor()),
        },
        // pem private keys
        signatures::common::Signature {
            name: "pem_private_key".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::pem::pem_private_key_magic(),
            parser: signatures::pem::pem_parser,
            description: signatures::pem::PEM_PRIVATE_KEY_DESCRIPTION.to_string(),
            extractor: Some(extractors::pem::pem_key_extractor()),
        },
        // netgear chk
        signatures::common::Signature {
            name: "chk".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::chk::chk_magic(),
            parser: signatures::chk::chk_parser,
            description: signatures::chk::DESCRIPTION.to_string(),
            extractor: None,
        },
        // trx
        signatures::common::Signature {
            name: "trx".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::trx::trx_magic(),
            parser: signatures::trx::trx_parser,
            description: signatures::trx::DESCRIPTION.to_string(),
            extractor: Some(extractors::trx::trx_extractor()),
        },
        // Motorola S-record
        signatures::common::Signature {
            name: "srecord".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::srec::srec_magic(),
            parser: signatures::srec::srec_parser,
            description: signatures::srec::SREC_DESCRIPTION.to_string(),
            extractor: Some(extractors::srec::srec_extractor()),
        },
        // Motorola S-record (generic)
        signatures::common::Signature {
            name: "srecord_generic".to_string(),
            short: true,
            magic_offset: 0,
            always_display: false,
            magic: signatures::srec::srec_short_magic(),
            parser: signatures::srec::srec_parser,
            description: signatures::srec::SREC_SHORT_DESCRIPTION.to_string(),
            extractor: Some(extractors::srec::srec_extractor()),
        },
        // Android sparse
        signatures::common::Signature {
            name: "android_sparse".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::androidsparse::android_sparse_magic(),
            parser: signatures::androidsparse::android_sparse_parser,
            description: signatures::androidsparse::DESCRIPTION.to_string(),
            extractor: Some(extractors::androidsparse::android_sparse_extractor()),
        },
        // device tree blob
        signatures::common::Signature {
            name: "dtb".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::dtb::dtb_magic(),
            parser: signatures::dtb::dtb_parser,
            description: signatures::dtb::DESCRIPTION.to_string(),
            extractor: Some(extractors::dtb::dtb_extractor()),
        },
        // ubi
        signatures::common::Signature {
            name: "ubi".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::ubi::ubi_magic(),
            parser: signatures::ubi::ubi_parser,
            description: signatures::ubi::UBI_IMAGE_DESCRIPTION.to_string(),
            extractor: Some(extractors::ubi::ubi_extractor()),
        },
        // ubifs
        signatures::common::Signature {
            name: "ubifs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::ubi::ubifs_magic(),
            parser: signatures::ubi::ubifs_parser,
            description: signatures::ubi::UBI_FS_DESCRIPTION.to_string(),
            extractor: Some(extractors::ubi::ubifs_extractor()),
        },
        // cfe bootloader
        signatures::common::Signature {
            name: "cfe".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::cfe::cfe_magic(),
            parser: signatures::cfe::cfe_parser,
            description: signatures::cfe::DESCRIPTION.to_string(),
            extractor: None,
        },
        // SEAMA firmware header
        signatures::common::Signature {
            name: "seama".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::seama::seama_magic(),
            parser: signatures::seama::seama_parser,
            description: signatures::seama::DESCRIPTION.to_string(),
            extractor: None,
        },
        // compress'd
        signatures::common::Signature {
            name: "compressd".to_string(),
            short: true,
            magic_offset: 0,
            always_display: false,
            magic: signatures::compressd::compressd_magic(),
            parser: signatures::compressd::compressd_parser,
            description: signatures::compressd::DESCRIPTION.to_string(),
            extractor: Some(extractors::sevenzip::sevenzip_extractor()),
        },
        // rar archive
        signatures::common::Signature {
            name: "rar".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::rar::rar_magic(),
            parser: signatures::rar::rar_parser,
            description: signatures::rar::DESCRIPTION.to_string(),
            extractor: Some(extractors::rar::rar_extractor()),
        },
        // PNG image
        signatures::common::Signature {
            name: "png".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::png::png_magic(),
            parser: signatures::png::png_parser,
            description: signatures::png::DESCRIPTION.to_string(),
            extractor: Some(extractors::png::png_extractor()),
        },
        // JPEG image
        signatures::common::Signature {
            name: "jpeg".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::jpeg::jpeg_magic(),
            parser: signatures::jpeg::jpeg_parser,
            description: signatures::jpeg::DESCRIPTION.to_string(),
            extractor: Some(extractors::jpeg::jpeg_extractor()),
        },
        // arcadyan obfuscated lzma
        signatures::common::Signature {
            name: "arcadyan".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::arcadyan::obfuscated_lzma_magic(),
            parser: signatures::arcadyan::obfuscated_lzma_parser,
            description: signatures::arcadyan::DESCRIPTION.to_string(),
            extractor: Some(extractors::arcadyan::obfuscated_lzma_extractor()),
        },
        // copyright text
        signatures::common::Signature {
            name: "copyright".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::copyright::copyright_magic(),
            parser: signatures::copyright::copyright_parser,
            description: signatures::copyright::DESCRIPTION.to_string(),
            extractor: None,
        },
        // WIND kernel version
        signatures::common::Signature {
            name: "wind_kernel".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::vxworks::wind_kernel_magic(),
            parser: signatures::vxworks::wind_kernel_parser,
            description: signatures::vxworks::WIND_KERNEL_DESCRIPTION.to_string(),
            extractor: None,
        },
        // vxworks symbol table
        signatures::common::Signature {
            name: "vxworks_symtab".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::vxworks::symbol_table_magic(),
            parser: signatures::vxworks::symbol_table_parser,
            description: signatures::vxworks::SYMTAB_DESCRIPTION.to_string(),
            extractor: Some(extractors::vxworks::vxworks_symtab_extractor()),
        },
        // ecos mips exception handler
        signatures::common::Signature {
            name: "ecos".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::ecos::exception_handler_magic(),
            parser: signatures::ecos::exception_handler_parser,
            description: signatures::ecos::EXCEPTION_HANDLER_DESCRIPTION.to_string(),
            extractor: None,
        },
        // dmg
        signatures::common::Signature {
            name: "dmg".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::dmg::dmg_magic(),
            parser: signatures::dmg::dmg_parser,
            description: signatures::dmg::DESCRIPTION.to_string(),
            extractor: Some(extractors::dmg::dmg_extractor()),
        },
        // riff
        signatures::common::Signature {
            name: "riff".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::riff::riff_magic(),
            parser: signatures::riff::riff_parser,
            description: signatures::riff::DESCRIPTION.to_string(),
            extractor: Some(extractors::riff::riff_extractor()),
        },
        // openssl
        signatures::common::Signature {
            name: "openssl".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::openssl::openssl_crypt_magic(),
            parser: signatures::openssl::openssl_crypt_parser,
            description: signatures::openssl::DESCRIPTION.to_string(),
            extractor: None,
        },
        // lzfse
        signatures::common::Signature {
            name: "lzfse".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::lzfse::lzfse_magic(),
            parser: signatures::lzfse::lzfse_parser,
            description: signatures::lzfse::DESCRIPTION.to_string(),
            extractor: Some(extractors::lzfse::lzfse_extractor()),
        },
        // MBR
        signatures::common::Signature {
            name: "mbr".to_string(),
            short: true,
            magic_offset: signatures::mbr::MAGIC_OFFSET,
            always_display: true,
            magic: signatures::mbr::mbr_magic(),
            parser: signatures::mbr::mbr_parser,
            description: signatures::mbr::DESCRIPTION.to_string(),
            extractor: Some(extractors::mbr::mbr_extractor()),
        },
        // tp-link
        signatures::common::Signature {
            name: "tplink".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::tplink::tplink_magic(),
            parser: signatures::tplink::tplink_parser,
            description: signatures::tplink::DESCRIPTION.to_string(),
            extractor: None,
        },
        // HP PJL
        signatures::common::Signature {
            name: "pjl".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::pjl::pjl_magic(),
            parser: signatures::pjl::pjl_parser,
            description: signatures::pjl::DESCRIPTION.to_string(),
            extractor: None,
        },
        // JBOOT ARM firmware image
        signatures::common::Signature {
            name: "jboot_arm".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::jboot::jboot_arm_magic(),
            parser: signatures::jboot::jboot_arm_parser,
            description: signatures::jboot::JBOOT_ARM_DESCRIPTION.to_string(),
            extractor: None,
        },
        // JBOOT STAG header
        signatures::common::Signature {
            name: "jboot_stag".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::jboot::jboot_stag_magic(),
            parser: signatures::jboot::jboot_stag_parser,
            description: signatures::jboot::JBOOT_STAG_DESCRIPTION.to_string(),
            extractor: None,
        },
        // JBOOT SCH2 header
        signatures::common::Signature {
            name: "jboot_sch2".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::jboot::jboot_sch2_magic(),
            parser: signatures::jboot::jboot_sch2_parser,
            description: signatures::jboot::JBOOT_SCH2_DESCRIPTION.to_string(),
            extractor: Some(extractors::jboot::sch2_extractor()),
        },
        // pcap-ng
        signatures::common::Signature {
            name: "pcapng".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::pcap::pcapng_magic(),
            parser: signatures::pcap::pcapng_parser,
            description: signatures::pcap::PCAPNG_DESCRIPTION.to_string(),
            extractor: Some(extractors::pcap::pcapng_extractor()),
        },
        // RSA encrypted data
        signatures::common::Signature {
            name: "rsa".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::rsa::rsa_magic(),
            parser: signatures::rsa::rsa_parser,
            description: signatures::rsa::DESCRIPTION.to_string(),
            extractor: None,
        },
        // GIF image
        signatures::common::Signature {
            name: "gif".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::gif::gif_magic(),
            parser: signatures::gif::gif_parser,
            description: signatures::gif::DESCRIPTION.to_string(),
            extractor: Some(extractors::gif::gif_extractor()),
        },
        // SVG image
        signatures::common::Signature {
            name: "svg".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::svg::svg_magic(),
            parser: signatures::svg::svg_parser,
            description: signatures::svg::DESCRIPTION.to_string(),
            extractor: Some(extractors::svg::svg_extractor()),
        },
        // Linux ARM64 boot image
        signatures::common::Signature {
            name: "linux_arm64_boot_image".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::linux::linux_arm64_boot_image_magic(),
            parser: signatures::linux::linux_arm64_boot_image_parser,
            description: signatures::linux::LINUX_ARM64_BOOT_IMAGE_DESCRIPTION.to_string(),
            extractor: None,
        },
        // FAT
        signatures::common::Signature {
            name: "fat".to_string(),
            short: true,
            magic_offset: signatures::fat::MAGIC_OFFSET,
            always_display: false,
            magic: signatures::fat::fat_magic(),
            parser: signatures::fat::fat_parser,
            description: signatures::fat::DESCRIPTION.to_string(),
            extractor: Some(extractors::tsk::tsk_extractor()),
        },
        // EFI GPT
        signatures::common::Signature {
            name: "efigpt".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::efigpt::efigpt_magic(),
            parser: signatures::efigpt::efigpt_parser,
            description: signatures::efigpt::DESCRIPTION.to_string(),
            extractor: Some(extractors::sevenzip::sevenzip_extractor()),
        },
        // RTK firmware header
        signatures::common::Signature {
            name: "rtk".to_string(),
            short: true,
            magic_offset: 0,
            always_display: false,
            magic: signatures::rtk::rtk_magic(),
            parser: signatures::rtk::rtk_parser,
            description: signatures::rtk::DESCRIPTION.to_string(),
            extractor: None,
        },
        // AES S-Box
        signatures::common::Signature {
            name: "aes_sbox".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::aes::aes_sbox_magic(),
            parser: signatures::aes::aes_sbox_parser,
            description: signatures::aes::DESCRIPTION.to_string(),
            extractor: None,
        },
        // LUKS
        signatures::common::Signature {
            name: "luks".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::luks::luks_magic(),
            parser: signatures::luks::luks_parser,
            description: signatures::luks::DESCRIPTION.to_string(),
            extractor: None,
        },
        // TP-Link RTOS
        signatures::common::Signature {
            name: "tplink_rtos".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::tplink::tplink_rtos_magic(),
            parser: signatures::tplink::tplink_rtos_parser,
            description: signatures::tplink::RTOS_DESCRIPTION.to_string(),
            extractor: None,
        },
        // BIN firmware header
        signatures::common::Signature {
            name: "binhdr".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::binhdr::bin_hdr_magic(),
            parser: signatures::binhdr::bin_hdr_parser,
            description: signatures::binhdr::DESCRIPTION.to_string(),
            extractor: None,
        },
        // Autel obfuscated firmware
        signatures::common::Signature {
            name: "autel".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::autel::autel_magic(),
            parser: signatures::autel::autel_parser,
            description: signatures::autel::DESCRIPTION.to_string(),
            extractor: Some(extractors::autel::autel_extractor()),
        },
        // NTFS
        signatures::common::Signature {
            name: "ntfs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::ntfs::ntfs_magic(),
            parser: signatures::ntfs::ntfs_parser,
            description: signatures::ntfs::DESCRIPTION.to_string(),
            extractor: Some(extractors::tsk::tsk_extractor()),
        },
        // APFS
        signatures::common::Signature {
            name: "apfs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::apfs::apfs_magic(),
            parser: signatures::apfs::apfs_parser,
            description: signatures::apfs::DESCRIPTION.to_string(),
            extractor: Some(extractors::apfs::apfs_extractor()),
        },
        // BTRFS
        signatures::common::Signature {
            name: "btrfs".to_string(),
            short: false,
            magic_offset: 0,
            always_display: true,
            magic: signatures::btrfs::btrfs_magic(),
            parser: signatures::btrfs::btrfs_parser,
            description: signatures::btrfs::DESCRIPTION.to_string(),
            extractor: None,
        },
        // WinCE
        signatures::common::Signature {
            name: "wince".to_string(),
            short: false,
            magic_offset: 0,
            always_display: false,
            magic: signatures::wince::wince_magic(),
            parser: signatures::wince::wince_parser,
            description: signatures::wince::DESCRIPTION.to_string(),
            extractor: Some(extractors::wince::wince_extractor()),
        },
    ];

    binary_signatures
}
