# Usage: cmake [[-D <var>=<value>] ...] -P run_tests.cmake -- [test1 ...]
# Run the given test cases and report a summary result.  Meant as an
# alternative to src/tests/run-tests for use with CMake builds.
# Variables with special meanings to this script:
# QUIET
#     If set to ON, only display a summary result and override the setting
#     for VERBOSE.  The default value is OFF.
# VERBOSE
#     If set to ON, causes test cases to be run with the -v flag to produce
#     verbose output, and all output from the test cases will be echoed by
#     this script.  If set to OFF, test cases will not be run with the -v flag,
#     and output from the test cases will not be echoed.  The default value
#     is ON if VERBOSE is set and non-empty in the environment.
# FORCE_PATH
#     If set to ON, causes test cases to be run with the -f flag (so that they
#     will use the same paths as the game's main.c would use even if alternate
#     paths have been configured for the test cases).  The default value is
#     ON if FORCE_PATH is set and non-empty in the environment.
# USECOLOR
#     If set to ON, use ANSI terminal color sequences to highlight some parts
#     of the output generated by this script.  The default value is OFF
#     when if(WIN32) evaulates to true; otherwise the default value is ON.
# TOOL
#     If set, the test case will be run as ${TOOL} test_case_path
#     [arg_for_test_case ...].  To pass arguments to the tool itself, have
#     TOOL be a CMake-style list:  i.e. tool_path;tool_arg_1[;tool_arg_2...].

cmake_policy(VERSION 3.5...3.31)

# Set up defaults for the variables influenced by the environment or by
# what the user has set.
set(_QUIET OFF)
set(_VERBOSE OFF)
if(DEFINED ENV{VERBOSE})
    string(LENGTH "$ENV{VERBOSE}" _RESLENGTH)
    if(_RESLENGTH GREATER 0)
        set(_VERBOSE ON)
    endif()
endif()
set(_FORCE_PATH OFF)
if(DEFINED ENV{FORCE_PATH})
    string(LENGTH "$ENV{FORCE_PATH}" _RESLENGTH)
    if(_RESLENGTH GREATER 0)
        set(_FORCE_PATH ON)
    endif()
endif()
if(WIN32)
    set(_USECOLOR OFF)
else()
    set(_USECOLOR ON)
endif()

# Modify those variables by what the user has set.
if(DEFINED QUIET)
    if(QUIET)
        set(_QUIET ON)
        set(_VERBOSE OFF)
    elseif(DEFINED VERBOSE)
        if(VERBOSE)
            set(_QUIET OFF)
            set(_VERBOSE ON)
        else()
            set(_VERBOSE OFF)
        endif()
    endif()
elseif(DEFINED VERBOSE)
    if(VERBOSE)
        set(_QUIET OFF)
        set(_VERBOSE ON)
    else()
        set(_VERBOSE OFF)
    endif()
endif()
if(DEFINED FORCE_PATH)
    if(FORCE_PATH)
        set(_FORCE_PATH ON)
    else()
        set(_FORCE_PATH OFF)
    endif()
endif()
if(DEFINED USECOLOR)
    if(USECOLOR)
        set(_USECOLOR ON)
    else()
        set(_USECOLOR OFF)
    endif()
endif()

# Skip over the arguments to CMake.
set(_X 0)
while(_X LESS CMAKE_ARGC)
    if(CMAKE_ARGV${_X} STREQUAL "--")
        break()
    endif()
    math(EXPR _X "${_X} + 1")
endwhile()
if(_X LESS CMAKE_ARGC)
    math(EXPR _STARTX "${_X} + 1")
else()
    set(_STARTX CMAKE_ARGC)
endif()

# Remember the length of the longest name for a test suite excluding the
# initial path components that are common to all the test suites.
set(_MAXLENGTH 0)
unset(_COMMON)
set(_X ${_STARTX})
while(_X LESS CMAKE_ARGC)
    string(LENGTH "${CMAKE_ARGV${_X}}" _THISLENGTH)
    if(_MAXLENGTH LESS _THISLENGTH)
        set(_MAXLENGTH ${_THISLENGTH})
    endif()
    if(DEFINED _COMMON)
        if((NOT (_COMMON STREQUAL "")) AND (NOT ("${CMAKE_ARGV${_X}}" MATCHES "^${_COMMON}/")))
            set(_LOOKING ON)
            while(_LOOKING)
                string(FIND "${_COMMON}" "/" _SLASH REVERSE)
                if(_SLASH EQUAL -1)
                    set(_COMMON "")
                    set(_LOOKING OFF)
                else()
                    string(SUBSTRING "${_COMMON}" 0 ${_SLASH} _COMMON)
                    if("${CMAKE_ARGV${_X}}" MATCHES "^${_COMMON}/")
                        SET(_LOOKING OFF)
                    endif()
                endif()
            endwhile()
        endif()
    else()
        set(_COMMON "${CMAKE_ARGV${_X}}")
    endif()
    math(EXPR _X "${_X} + 1")
endwhile()
if(DEFINED _COMMON)
    string(LENGTH "${_COMMON}" _THISLENGTH)
    if(_THISLENGTH LESS _MAXLENGTH)
        math(EXPR _MAXLENGTH "${_MAXLENGTH} - ${_THISLENGTH} - 1")
    endif()
endif()

# Store some strings for ANSI color.
string(ASCII 27 _ESC)
set(_RED "${_ESC}[01;31m")
set(_GREEN "${_ESC}[01;32m")
set(_YELLOW "${_ESC}[01;33m")
set(_RESET "${_ESC}[0m")

# Remember how long a newline is.
string(LENGTH "\n" _NLLENGTH)

# Run each test suite accumulating the number of test cases run and run with
# passing results.
if(NOT _QUIET)
    math(EXPR _COUNT "${CMAKE_ARGC} - ${_STARTX}")
    message("Running ${_COUNT} suites:")
endif()
math(EXPR _DISLENGTH "${_MAXLENGTH} + 8")
set(_PASS 0)
set(_TOTAL 0)
set(_EXITCODE 0)
if(_VERBOSE)
    if(_FORCE_PATH)
        set(_TESTOPTS "-vf")
    else()
        set(_TESTOPTS "-v")
    endif()
elseif(_FORCE_PATH)
    set(_TESTOPTS "-f")
else()
    unset(_TESTOPTS)
endif()
set(_X ${_STARTX})
while(_X LESS CMAKE_ARGC)
    # Use file(TO_NATIVE_PATH ...) rather than cmake_path(CONVERT ...) since
    # the latter requires at least CMake 3.20.
    file(TO_NATIVE_PATH "${CMAKE_ARGV${_X}}" _CONVERTED_PATH)
    if(DEFINED TOOL)
        if(DEFINED _TESTOPTS)
            execute_process(COMMAND ${TOOL} "${_CONVERTED_PATH}" ${_TESTOPTS}
                WORKING_DIRECTORY . RESULT_VARIABLE _RUN_RESULT
                OUTPUT_VARIABLE _RUN_OUTPUT ERROR_VARIABLE _RUN_OUTPUT)
        else()
            execute_process(COMMAND ${TOOL} "${_CONVERTED_PATH}"
                WORKING_DIRECTORY . RESULT_VARIABLE _RUN_RESULT
                OUTPUT_VARIABLE _RUN_OUTPUT ERROR_VARIABLE _RUN_OUTPUT)
        endif()
    elseif(DEFINED _TESTOPTS)
        execute_process(COMMAND "${_CONVERTED_PATH}" ${_TESTOPTS}
            WORKING_DIRECTORY . RESULT_VARIABLE _RUN_RESULT
            OUTPUT_VARIABLE _RUN_OUTPUT ERROR_VARIABLE _RUN_OUTPUT)
    else()
        execute_process(COMMAND "${_CONVERTED_PATH}" WORKING_DIRECTORY .
            RESULT_VARIABLE _RUN_RESULT OUTPUT_VARIABLE _RUN_OUTPUT
            ERROR_VARIABLE _RUN_OUTPUT)
    endif()
    if(NOT (_RUN_RESULT MATCHES "^[+-]?[0-9]+$"))
        if(NOT (_RUN_RESULT STREQUAL ""))
            # Print out string result codes; record them as an exit code of
            # one in _EXITCODE.
            if(_USECOLOR)
                message("${_RED}${_CONVERTED_PATH}: Suite died: ${_RUN_RESULT}${_RESET}")
            else()
                message("${_CONVERTED_PATH}: Suite died: ${_RUN_RESULT}")
            endif()
            set(_EXITCODE 1)
            set(_RUN_RESULT 1)
            # We do not know how many tests the suite had, but at least one
            # failed.
            math(EXPR _TOTAL "${_TOTAL} + 1")
        else()
            set(_RUN_RESULT 0)
        endif()
    elseif(_RUN_RESULT)
        # Non-zero integer exit codes get a generic message and are directly
        # recorded in _EXITCODE.
        if(_USECOLOR)
            message("${_RED}${_CONVERTED_PATH}: Suite died${_RESET}")
        else()
            message("${_CONVERTED_PATH}: Suite died")
        endif()
        set(_EXITCODE ${_RUN_RESULT})
        # We do not know how many tests the suite had, but at least one failed.
        math(EXPR _TOTAL "${_TOTAL} + 1")
    endif()
    if(NOT _RUN_RESULT)
        string(REGEX MATCHALL "[^:\n]+ finished: [0-9]+/[0-9]+ passed\r?\n" _SUMMARY_LINES ${_RUN_OUTPUT})
        list(LENGTH _SUMMARY_LINES _COUNT)
        if(_COUNT)
            list(GET _SUMMARY_LINES -1 _LAST_LINE)
            string(REGEX REPLACE "[^:\n]+ finished: ([0-9]+)/[0-9]+ passed\r?\n" "\\1" _THISPASS ${_LAST_LINE})
            string(REGEX REPLACE "[^:\n]+ finished: [0-9]+/([0-9]+) passed\r?\n" "\\1" _THISTOTAL ${_LAST_LINE})
            # Tally the results
            math(EXPR _PASS "${_PASS} + ${_THISPASS}")
            math(EXPR _TOTAL "${_TOTAL} + ${_THISTOTAL}")
            if(NOT _QUIET)
                if(_VERBOSE)
                    # Echo the output from test case.
                    set(_MATCHCOUNT 0)
                    string(LENGTH "${_RUN_OUTPUT}" _RESLENGTH)
                    while(_RESLENGTH GREATER 0)
                        string(FIND "${_RUN_OUTPUT}" "\n" _ENDLINE)
                        if(_ENDLINE EQUAL -1)
                            set(_LINE "{_RUN_OUTPUT}")
                            set(_RESLENGTH 0)
                        else()
                            string(SUBSTRING "${_RUN_OUTPUT}" 0 ${_ENDLINE} _LINE)
                            math(EXPR _ENDLINE "${_ENDLINE} + ${_NLLENGTH}")
                            string(SUBSTRING "${_RUN_OUTPUT}" ${_ENDLINE} -1
                                _RUN_OUTPUT)
                            string(LENGTH "${_RUN_OUTPUT}" _RESLENGTH)
                        endif()
                        if(_LINE MATCHES "[^:\n]+ finished: [0-9]+/[0-9]+ passed\r?")
                            # But strip out the line corresponding to
                            # _LAST_LINE since it will printed out separately
                            # below.
                            math(EXPR _MATCHCOUNT "${_MATCHCOUNT} + 1")
                            if(NOT (_MATCHCOUNT EQUAL _COUNT))
                                message("${_LINE}")
                            endif()
                        else()
                            message("${_LINE}")
                        endif()
                    endwhile()
                endif()
                string(REGEX REPLACE "([^:\n]+) finished: [0-9]+/[0-9]+ passed\r?\n" "\\1" _NAME ${_LAST_LINE})
                if(_THISTOTAL)
                    math(EXPR _PERCENT "(100 * ${_THISPASS}) / ${_THISTOTAL}")
                else()
                    set(_PERCENT 100)
                endif()
                set(_RESULT "${_THISPASS}/${_THISTOTAL}")
                string(LENGTH "${_NAME}" _NAMLENGTH)
                string(LENGTH "${_RESULT}" _RESLENGTH)
                math(EXPR _PADLENGTH "${_DISLENGTH} - ${_NAMLENGTH} - ${_RESLENGTH}")
                if(_PADLENGTH GREATER 1)
                    # string(REPEAT ...) requires CMake 3.15 so use this
                    # instead.
                    set(_Y 2)
                    set(_PAD "  ")
                    while(_Y LESS _PADLENGTH)
                        string(APPEND _PAD " ")
                        math(EXPR _Y "${_Y} + 1")
                    endwhile()
                else()
                    set(_PAD " ")
                endif()
                if(_USECOLOR)
                    if(_THISPASS EQUAL _THISTOTAL)
                        set(_COLOR "${_GREEN}")
                    elseif(_PERCENT GREATER 89)
                        set(_COLOR "${_YELLOW}")
                    else()
                        set(_COLOR "${_RED}")
                    endif()
                    if(_VERBOSE)
                        message("    ${_NAME} finished: ${_COLOR}${_RESULT}${_RESET} passed")
                    else()
                        message("    ${_NAME}${_PAD}${_COLOR}${_RESULT}${_RESET} passed")
                    endif()
                else()
                    if(_VERBOSE)
                        message("    ${_NAME} finished: ${_RESULT} passed")
                    else()
                        message("    ${_NAME}${_PAD}${_RESULT} passed")
                    endif()
                endif()
            endif()
        else()
            if(_USECOLOR)
                messsge("${_RED}${_CONVERTED_PATH}: Malformed output${_RESET}")
            else()
                message("${_CONVERTED_PATH}: Malformed output")
            endif()
            set(_EXITCODE 1)
        endif()
    endif()
    math(EXPR _X "${_X} + 1")
endwhile()

# Print a summary of all the test results.
if(_TOTAL)
    math(EXPR _PERCENT "(100 * ${_PASS}) / ${_TOTAL}")
    math(EXPR _DEC1 "((1000 * ${_PASS}) / ${_TOTAL}) % 10")
else()
    set(_PERCENT 100)
    set(_DEC1 0)
endif()
if(_USECOLOR)
    if(_PASS EQUAL _TOTAL)
        set(_COLOR "${_GREEN}")
    elseif(_PERCENT GREATER 89)
        set(_COLOR "${_YELLOW}")
    else()
        set(_COLOR "${_RED}")
    endif()
    message("Total: ${_COLOR}${_PASS}/${_TOTAL}${_RESET} passed (${_COLOR}${_PERCENT}.${_DEC1}%${_RESET})")
else()
    message("Total: ${_PASS}/${_TOTAL} passed (${_PERCENT}.${_DEC1}%)")
endif()

if(_EXITCODE)
    # Cause CMake to exit with an exit code of 1.  Apparently, CMake does not
    # provide a way to exit with other non-zero exit codes.
    message(FATAL_ERROR "Non-zero result: ${_EXITCODE}")
endif()
