/*
 * DVIlib - a library for handling DVI files
 * 
 * Copyright (c) 2000 Evgeny Stambulchik
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>

#include <string.h>

#include "dvicodes.h"
#include "dvilibP.h"

void dvi_errmsg(const char *msg)
{
    fprintf(stderr, msg);
    fprintf(stderr, "\n");
}

void dvi_free(void *p)
{
    if (p) {
        free(p);
    }
}

void *dvi_malloc(size_t size)
{
    void *p;
    
    if (size <= 0) {
        p = NULL;
    } else {
        p = malloc(size);
        if (!p) {
            dvi_errmsg("Fatal error - no memory available");
            exit(1);
        }
    }
    return p;
}

void *dvi_realloc(void *p, size_t size)
{
    if (size <= 0) {
        dvi_free(p);
        p = NULL;
    } else if (!p) {
        p = dvi_malloc(size);
    } else {
        p = realloc(p, size);
        if (!p) {
            dvi_errmsg("Fatal error - no memory available");
            exit(1);
        }
    }
    return p;
}

DVI *DVI_new(FILE *fp)
{
    DVI *dvip;
    
    dvip = dvi_malloc(sizeof(DVI));
    dvip->state = dvi_malloc(sizeof(dvi_state));
    dvip->state->h = 0;
    dvip->state->v = 0;
    dvip->state->w = 0;
    dvip->state->x = 0;
    dvip->state->y = 0;
    dvip->state->z = 0;
    dvip->font_id = -1;

    dvip->stack = dvi_malloc(sizeof(dvi_stack));
    dvip->stack->states = NULL;
    dvip->stack->slength = 0;
    dvip->stack->sp = 0;
    
    dvip->fonts = NULL;
    dvip->nfonts = 0;

    dvip->fp = fp;
    
    dvip->nbread = 0;
    dvip->nops = 0;

    dvip->npages = 0;
    dvip->maxh = 0;
    dvip->maxw = 0;
    
    dvip->dvi_id = 0;
    dvip->resolution = 0.0;
    dvip->magnification = 0;
    dvip->comments = NULL;

    dvip->pre_parsed  = 0;
    dvip->post_parsed = 0;

    dvip->pbop = -1;
    dvip->post = -1;

    dvip->hooks = dvi_malloc(sizeof(dvi_hooks));
    ((dvi_hooks *)(dvip->hooks))->write_char = NULL;
    ((dvi_hooks *)(dvip->hooks))->draw_rule = NULL;

    return dvip;
}

void DVI_free(DVI *dvip)
{
    int i;
    
    if (!dvip) {
        return;
    }
    
    dvi_free(dvip->hooks);
    
    dvi_free(dvip->comments);
    
    for (i = 0; i < dvip->nfonts; i++) {
        dvi_free(dvip->fonts[i].name);
        dvi_free(dvip->fonts[i].area);
    }
    dvi_free(dvip->fonts);
    
    dvi_free(dvip->stack->states);
    dvi_free(dvip->stack);
    
    dvi_free(dvip->state);
    
    dvi_free(dvip);
}

void DVI_add_hooks(DVI *dvip,
    DVI_write_char_hook write_char,
    DVI_draw_rule_hook draw_rule)
{
    ((dvi_hooks *)(dvip->hooks))->write_char = write_char;
    ((dvi_hooks *)(dvip->hooks))->draw_rule = draw_rule;
}

static int FGETC(DVI *dvip)
{
    int c;
    
    c = fgetc(dvip->fp);
    if (c != EOF) {
        dvip->nbread++;
    }
    
    return c;
}

void DVI_process(DVI *dvip)
{
    int c = FGETC(dvip);
    if (c != PRE) {
        dvi_errmsg("Fatal error: not a valid DVI file");
        exit(1);
    }
    
    pre(dvip);
    
    while (dvi_cmd(dvip) != EOF) {
        dvip->nops++;
    }
}

int dvi_cmd(DVI *dvip)
{
    int c;

    c = FGETC(dvip);
    if (c == EOF) {
        return(EOF);
    }

    if (c >= SET_CHAR_0 && c <= SET_CHAR_127) {
        setput_char(dvip, c, 1);
    } else if (c >= FNT_NUM_0 && c <= FNT_NUM_63) {
        fnt_num(dvip, c - FNT_NUM_0);
    } else {
        switch (c) {
        case SET1:
        case SET2:
        case SET3:
        case SET4:
            setput_charn(dvip, c - SET1 + 1, 1);
            break;
        case SET_RULE:
            setput_rule(dvip, 1);
            break;
        case PUT1:
        case PUT2:
        case PUT3:
        case PUT4:
            setput_charn(dvip, c - PUT1 + 1, 0);
            break;
        case PUT_RULE:
            setput_rule(dvip, 0);
            break;
        case NOP:
            break;
        case BOP:
            bop(dvip);
            break;
        case EOP:
            eop(dvip);
            break;
        case PUSH:
            push(dvip);
            break;
        case POP:
            pop(dvip);
            break;
        case RIGHT1:
        case RIGHT2:
        case RIGHT3:
        case RIGHT4:
            rightn(dvip, c - RIGHT1 + 1);
            break;
        case W0:
        case W1:
        case W2:
        case W3:
        case W4:
            wn(dvip, c - W0);
            break;
        case X0:
        case X1:
        case X2:
        case X3:
        case X4:
            xn(dvip, c - X0);
            break;
        case DOWN1:
        case DOWN2:
        case DOWN3:
        case DOWN4:
            downn(dvip, c - DOWN1 + 1);
            break;
        case Y0:
        case Y1:
        case Y2:
        case Y3:
        case Y4:
            yn(dvip, c - Y0);
            break;
        case Z0:
        case Z1:
        case Z2:
        case Z3:
        case Z4:
            zn(dvip, c - Z0);
            break;
        case FNT1:
        case FNT2:
        case FNT3:
        case FNT4:
            fntn(dvip, c - FNT1 + 1);
            break;
        case XXX1:
        case XXX2:
        case XXX3:
        case XXX4:
            xxxn(dvip, c - XXX1 + 1);
            break;
        case FNT_DEF1:
        case FNT_DEF2:
        case FNT_DEF3:
        case FNT_DEF4:
            fnt_defn(dvip, c - FNT_DEF1 + 1);
            break;
        case PRE:
            dvi_errmsg("Fatal error: unexpected PRE in the middle");
            exit(1);
            break;
        case POST:
            post(dvip);
            break;
        case POST_POST:
            post_post(dvip);
            break;
        default:
            printf("unknown dvi_cmd: c = %d?\n", c);
            break;
        }
    }
    
    return c;
}

DVI_U32 ugetn(DVI *dvip, int n)
{
    int i, c;
    DVI_U32 word;
    
    if (n < 1 || n > 4) {
        dvi_errmsg("Fatal error");
        exit(1);
    } else {
        word = 0;
        for (i = 0; i < n; i++) {
            c = FGETC(dvip);
            if (c == EOF) {
                dvi_errmsg("Fatal error: unexpected EOF");
                exit(1);
            }
            word <<= 0x8;
            word |= c;
        }
        return word;
    }
}


DVI_S32 sgetn(DVI *dvip, int n)
{
    int i, c;
    DVI_S32 word;
    
    if (n < 1 || n > 4) {
        dvi_errmsg("Fatal error");
        exit(1);
    } else {
        word = 0;
        for (i = 0; i < n; i++) {
            c = FGETC(dvip);
            if (c == EOF) {
                dvi_errmsg("Fatal error: unexpected EOF");
                exit(1);
            }
            if (i == 0 && c & 0x80) {
                c -= 0x100;
            }
            word <<= 0x8;
            word |= c;
        }
        return word;
    }
}


void setput_char(DVI *dvip, int c, int advance)
{
    DVI_write_char_hook write_char;
    write_char = ((dvi_hooks *)(dvip->hooks))->write_char;
    
    if (write_char != NULL) {
        dvip->state->h += write_char(dvip, c, advance);
    }
}

/* 132, 137 */
void setput_rule(DVI *dvip, int advance)
{
    DVI_S32 a, b;

    DVI_draw_rule_hook draw_rule;
    draw_rule = ((dvi_hooks *)(dvip->hooks))->draw_rule;
    
    a = sget4(dvip);
    b = sget4(dvip);

    if (a > 0 && b > 0 && draw_rule != NULL) {
        draw_rule(dvip, a, b);
    }

    /* even for negative a or b ! */
    if (advance) {
        dvip->state->h += b;
    }
}


/* 139 */
void bop(DVI *dvip)
{
    /* c0 = */ sget4(dvip);
    /* c1 = */ sget4(dvip);
    /* c2 = */ sget4(dvip);
    /* c3 = */ sget4(dvip);
    /* c4 = */ sget4(dvip);
    /* c5 = */ sget4(dvip);
    /* c6 = */ sget4(dvip);
    /* c7 = */ sget4(dvip);
    /* c8 = */ sget4(dvip);
    /* c9 = */ sget4(dvip);
    dvip->pbop = sget4(dvip);

    dvip->stack->sp = 0;

    dvip->font_id = -1;

    dvip->state->h = 0;
    dvip->state->v = 0;

    dvip->state->w = 0;
    dvip->state->x = 0;

    dvip->state->y = 0;
    dvip->state->z = 0;
}


/* 140 */
void eop(DVI *dvip)
{
    if (dvip->stack->sp != 0) {
        dvi_errmsg("Warning: non-empty stack at EOP");
    }
    /* we may free the stack here */
}


/* 141 */
void push(DVI *dvip)
{
    if (dvip->stack->sp == dvip->stack->slength) {
        dvip->stack->states = dvi_realloc(dvip->stack->states,
            (dvip->stack->sp + 1)*sizeof(dvi_state));
        dvip->stack->slength++;
    }
    
    memcpy(&(dvip->stack->states[dvip->stack->sp]), dvip->state,
        sizeof(dvi_state));

    dvip->stack->sp++;
}


/* 142 */
void pop(DVI *dvip)
{
    if (dvip->stack->sp == 0) {
        dvi_errmsg("Fatal error: dvi stack underflow");
        exit(1);
    }

    dvip->stack->sp--;

    memcpy(dvip->state, &(dvip->stack->states[dvip->stack->sp]),
        sizeof(dvi_state));
}

/* 143 - 146 */
void rightn(DVI *dvip, int n)
{
    dvip->state->h += sgetn(dvip, n);
}

/* 147 - 151 */
void wn(DVI *dvip, int n)
{
    if (n) {
        dvip->state->w = sgetn(dvip, n);
    }
    dvip->state->h += dvip->state->w;
}

/* 152 - 156 */
void xn(DVI *dvip, int n)
{
    if (n) {
        dvip->state->x = sgetn(dvip, n);
    }
    dvip->state->h += dvip->state->x;
}

/* 157 - 160 */
void downn(DVI *dvip, int n)
{
    dvip->state->v += sgetn(dvip, n);
}

/* 161 - 165 */
void yn(DVI *dvip, int n)
{
    if (n) {
        dvip->state->y = sgetn(dvip, n);
    }
    dvip->state->v += dvip->state->y;
}

/* 166 - 170 */
void zn(DVI *dvip, int n)
{
    if (n) {
        dvip->state->z = sgetn(dvip, n);
    }
    dvip->state->v += dvip->state->z;
}

int dvi_valid_font_id(DVI *dvip, int font_id)
{
    int i;
    
    for (i = 0; i < dvip->nfonts; i++) {
        if (dvip->fonts[i].id == font_id) {
            return 1;
        }
    }
    
    return 0;
}

/* 171 through 234 */
void fnt_num(DVI *dvip, int fn)
{
    if (dvi_valid_font_id(dvip, fn)) {
        dvip->font_id = fn;
    } else {
        dvi_errmsg("Fatal error: invalid font ID");
        exit(1);
    }
}

/* 235 through 238 */
void fntn(DVI *dvip, int n)
{
    int fn;
    
    if (n == 4) {
        fn = sgetn(dvip, n);
    } else {
        fn = ugetn(dvip, n);
    }
    
    fnt_num(dvip, fn);
}

void fnt_defn(DVI *dvip, int n)
{
    int i, al, nl, font_id;
    DVI_finfo finfo;

    if (n == 4) {
        font_id = sgetn(dvip, n);
    } else {
        font_id = ugetn(dvip, n);
    }
    
    /* already defined */
    if (dvi_valid_font_id(dvip, font_id)) {
        return;
    }
    
    finfo.id      = font_id;
    finfo.chksum  = uget4(dvip);
    finfo.magsize = uget4(dvip);
    finfo.dessize = uget4(dvip);

    al = uget1(dvip);
    nl = uget1(dvip);

    /* area name */
    finfo.area = dvi_malloc(al + 1);
    for (i = 0; i < al; i++) {
        finfo.area[i] = uget1(dvip);
    }
    finfo.area[i] = 0;

    finfo.name = dvi_malloc(nl + 1);
    for (i = 0; i < nl; i++) {
        finfo.name[i] = uget1(dvip);
    }
    finfo.name[i] = 0;
    
    load_font(dvip, &finfo);
    
    dvi_free(finfo.name);
    dvi_free(finfo.area);
}

/* 239 through 242 */
void xxxn(DVI *dvip, int n)
{
    int i, k;

    k = ugetn(dvip, n);

    for (i = 0; i < k; i++) {
        uget1(dvip);
    }
}


void pre(DVI *dvip)
{
    int i, cl;
    DVI_U32 num, den;

    dvip->dvi_id = uget1(dvip);
    if (dvip->dvi_id != DVI_ID) {
        dvi_errmsg("Warning: experimental version of DVI");
    }
    
    num = uget4(dvip);
    den = uget4(dvip);
    dvip->resolution = (float) num/den;
    
    dvip->magnification = uget4(dvip);

    cl = uget1(dvip);
    dvip->comments = dvi_malloc(cl + 1);
    for (i = 0; i < cl; i++) {
        dvip->comments[i] = uget1(dvip);
    }
    dvip->comments[i] = '\0';
    
    dvip->pre_parsed = 1;
}


void post(DVI *dvip)
{
    DVI_U32 num, den;

    dvip->pbop = uget4(dvip);

    num = uget4(dvip);
    den = uget4(dvip);
    dvip->resolution = (float) num/den;

    dvip->magnification = uget4(dvip);

    dvip->maxh = uget4(dvip);
    dvip->maxw = uget4(dvip);

    /* max stack depth = */ uget2(dvip);
    dvip->npages = uget2(dvip);
}

void post_post(DVI *dvip)
{   
    dvip->post = uget4(dvip);
    
    dvip->dvi_id = uget1(dvip);
    if (dvip->dvi_id != DVI_ID) {
        dvi_errmsg("Warning: experimental version of DVI");
    }
/*
 *     while ((c = uget1(dvip)) == PADDING) {
 *         ;
 *     }
 */
    dvip->post_parsed = 1;
    
    return;
}

void load_font(DVI *dvip, DVI_finfo *finfo)
{
    dvip->fonts = dvi_realloc(dvip->fonts, (dvip->nfonts + 1)*sizeof(DVI_finfo));
    
    memcpy(&(dvip->fonts[dvip->nfonts]), finfo, sizeof(DVI_finfo));
    
    if (finfo->name != NULL) {
        dvip->fonts[dvip->nfonts].name = dvi_malloc(strlen(finfo->name) + 1);
        strcpy(dvip->fonts[dvip->nfonts].name, finfo->name);
    }

    if (finfo->area != NULL) {
        dvip->fonts[dvip->nfonts].area = dvi_malloc(strlen(finfo->area) + 1);
        strcpy(dvip->fonts[dvip->nfonts].area, finfo->area);
    }

    dvip->nfonts++;
}


void setput_charn(DVI *dvip, int n, int advance)
{
    int c;
    
    if (n == 4) {
        c = sgetn(dvip, n);
    } else {
        c = ugetn(dvip, n);
    }
    setput_char(dvip, c, advance);
}

DVI_finfo *DVI_get_finfo(DVI *dvip, int font_id)
{
    int i;
    
    for (i = 0; i < dvip->nfonts; i++) {
        if (dvip->fonts[i].id == font_id) {
            return &(dvip->fonts[i]);
        }
    }
    
    return NULL;
}
