// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Play.h"




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::Play::Play( GameTimer* _timer_ptr ) :
        timer_ptr(_timer_ptr),
        have_drawn_reminder(false),
        my_pause(PAUSE_NONE)
{
    // ...
    
} // FS::Play::Play()

// ******************
// *** DESTRUCTOR ***
// ******************
FS::Play::~Play()
{
    // ...
    
} // FS::Play::~Play()




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void FS::Play::load_level(int lev)
{
    // FIXME: test_level messes up logic of mode!!!
    
    // construct filename from 'lev'
    char my_filename[256];
    if ( test_level )
    {
        // FIXME: don't allow test level to be played in any mode other
        // than MODE_GAME!
        strcpy(my_filename, test_file);
    }
    else if (current_mode == MODE_GAME)
    {
        sprintf(my_filename, PKG_DATA_DIR "/levels/level_%02d.fsl", lev);
    }
    else if (current_mode == MODE_TUTORIAL)
    {
        sprintf(my_filename, PKG_DATA_DIR "/tutorials/level_%02d.txt", lev);
    } // if ... else
    
    // open file - if can't open it that's a v. serious error so exit
    // program with message
    std::ifstream fin(my_filename);
    if (!fin.is_open())
    {
        std::cerr << "Can't open " << my_filename << ".\n";
        exit(EXIT_FAILURE);
    } // if
    
    // Used to hold lines as read from file...
    std::string my_line;
    
    // max crystals
    getline(fin, my_line);
    // FIXME: no exception-checking.
    my_player_data.set_max_crystals( boost::lexical_cast<int>(my_line) );
    
    // number of starting crystals
    getline(fin, my_line);
    // FIXME: no exception-checking.
    num_starting_crystals = boost::lexical_cast<int>(my_line);
    
    // the starting crystals themselves - stored in this class 
    // in case we need to restart the level
    int start = 0;
    int comma_pos;
    getline(fin, my_line);
    for (int i = 0; i < num_starting_crystals; ++i)
    {
        comma_pos = my_line.find(',', start);
        int cc_as_int = boost::lexical_cast<int>( my_line.substr(start, comma_pos - start) );
        starting_crystals[i] = static_cast<Crystal::Color>(cc_as_int);
        start = comma_pos + 1;
    } // for
    my_player_data.set_crystals_held(starting_crystals, num_starting_crystals);
                
    // Hand stream over to MapEngine to let it initialize itself.
    my_scroller.load_data(fin);
        
    fin.close();
    
} // FS::Play::load_level()

// **************************************************

FS::PlayReport FS::Play::update(int gt)
{
    //std::cout << "Play::update\n";

    previous_update_time = current_update_time;
    current_update_time = gt;
    my_scroller.update( current_update_time - previous_update_time );
    
    // If game is paused, call PauseMenu::update first.
    // This gobbles up all events, so you can't move cat while
    // game paused!
    if ( my_pause == PAUSE_USER )
    {
        switch ( my_pause_menu.update() )
        {
            case Pause_menu::PMS_NONE:
                break;
            case Pause_menu::PMS_CONTINUE:
                timer_ptr->pause();
                my_pause = PAUSE_NONE;
                my_sprite_mngr.deactivate_pause_menu();    
                break;
            case Pause_menu::PMS_QUIT:
                // leave function immediately after resuming timer
                timer_ptr->pause();
                snd_engine_ptr->stop_all_sounds();
                my_pause = PAUSE_NONE;
                return PR_GAME_OVER;
                break;  // ?!
        } // switch
    } // if ... else
    
    PlayReport my_report = PR_OK;   // initial value to keep compiler happy
    switch ( my_sprite_mngr.update( gt - start_time, my_scroller.get_data(), my_pause ) )
    {
        case Sprite_manager::PLAYING_GAME:
            my_report = PR_OK;
            break;
        case Sprite_manager::REQUEST_LEVEL_RESTART:
            my_player_data.set_crystals_held(starting_crystals, num_starting_crystals);
            my_sprite_mngr.restart_level();
            my_scroller.restore_original_directions_and_velocities();
            if (current_mode == MODE_TUTORIAL)
            {
                my_tutorial.restart();
                if (have_drawn_reminder)
                {
                    //erase_task_reminder();
                    //erase_task_message();
                    have_drawn_reminder = false;
                } // if
            } // if
            my_report = PR_OK;
            break;
        case Sprite_manager::LEVEL_COMPLETED:
            my_player_data.inc_level();
            
            // FIXME: write a proper sequence for when game is completed.
            if ( my_player_data.get_level() > NUM_LEVELS )
            {
                my_report = PR_GAME_OVER;
            }
            else
            {
                my_report = PR_LEVEL_COMPLETED;
            } // if ... else
            
            break;
        case Sprite_manager::GAME_OVER:
            my_report = PR_GAME_OVER;
            break;
        case Sprite_manager::RUNNING_LEVEL_COMPLETED_ROUTINE:
        case Sprite_manager::RUNNING_GAME_OVER_ROUTINE:
            my_report = PR_OK;
            break;
    } // switch
    
    // Tutorial task handling goes here.
    if ( current_mode == MODE_TUTORIAL )
    {
        check_task();
    } // if
    
    my_sprite_mngr.redraw_screen();
    
    if ( escape_pressed )
    {
        escape_pressed = false;
        handle_escape_pressed(my_report);
    } // if 
        
    return my_report;
    
} // FS::Play::update()

// **************************************************

void FS::Play::init_sprite_manager( Sound_engine* se )
{
    my_sprite_mngr.set_player_data( &my_player_data );
    my_sprite_mngr.set_sound_engine( se );
    my_sprite_mngr.set_scroller( &my_scroller );
    my_sprite_mngr.set_pause_menu( &my_pause_menu );

    snd_engine_ptr = se;
    
} // FS::Play::init_sprite_manager()
      
// **************************************************

void FS::Play::new_game(int start_level)
{
    my_player_data.reset(start_level);
    
} // FS::Play::new_game()
     
// **************************************************

void FS::Play::new_level(int st)
{
    load_level( my_player_data.get_level() );
    
    my_sprite_mngr.create_scrolling_sprites();
    my_sprite_mngr.reset();
    
    start_time = previous_update_time = current_update_time = st;
    
    // If we're in TUTORIAL mode, prepare task list.
    if (current_mode == MODE_TUTORIAL)
    {
        my_tutorial.load_task_list(my_player_data.get_level());
        have_drawn_reminder = false;
    } // if
    
} // FS::Play::new_level()

// **************************************************

void FS::Play::handle_escape_pressed( PlayReport report )
{
    // If level or game are finished, no point pausing because we're
    // about to move away from the game screen.
    if ( report == PR_LEVEL_COMPLETED || report == PR_GAME_OVER )
    {
        return;
    } // if
    
    // N.B. if my_pause == PAUSE_AUTO, user's request will be ignored,
    // which is what we want.    
    if ( my_pause == PAUSE_NONE )
    {
        timer_ptr->pause();
        my_pause = PAUSE_USER;
        my_pause_menu.reset();
        my_sprite_mngr.activate_pause_menu();
    }
    else if ( my_pause == PAUSE_USER )
    {
        timer_ptr->pause();
        my_pause = PAUSE_NONE;
        my_sprite_mngr.deactivate_pause_menu();
    } // if ... else
    
} // FS::Play::handle_escape

// **************************************************

void FS::Play::check_task()
{
    const Tutorial::Task* current_task = my_tutorial.get_current_task();
    bool looping = true; // i.e. stay in loop
    
    do
    {
        // null pointer tells us that all tasks are completed
        if (!current_task) { return; }
    
        if ( current_task->do_once_only && current_task->done_once_already )
        {
            current_task = my_tutorial.next_task();
            continue; // skip switch statement below
        } // if
        
        switch (current_task->type)
        {
            case Tutorial::READ_TASK:
                if (my_pause == PAUSE_NONE)
                {
                    my_pause = PAUSE_AUTO;
                    timer_ptr->pause();
                    snd_engine_ptr->pause_all_sounds();
                    draw_task_message(current_task->msg);
                    looping = false;
                } 
                else if (my_pause_menu.has_pressed_jump())
                {
                    my_pause = PAUSE_NONE;
                    timer_ptr->pause();
                    //erase_task_message();
                    current_task = my_tutorial.next_task();
                    if ( (current_task && current_task->type == Tutorial::PLAY_TASK) 
                            || !current_task )
                    {
                        snd_engine_ptr->resume_all_sounds();
                    } // if
                } 
                else
                {
                    //erase_task_message();
                    draw_task_message(current_task->msg);
                    looping = false;
                } // if ... else
                break;
                
            case Tutorial::PLAY_TASK:
                RowDescription row = my_sprite_mngr.get_alberts_row();
                bool row_check_ok = false;
                bool crystal_check_ok = false;
                
                // correct row?
                if ( row == current_task->row_desc )
                {
                    // on a log?
                    if ( row >= ROW_0 && row <= ROW_5 )
                    {
                        // correct log?
                        if ( my_sprite_mngr.get_host_log_color() == current_task->lc )
                        {
                            row_check_ok = true;
                        }
                    }
                    else
                    {
                        row_check_ok = true;
                    } // if ... else
                } // if
                
                // row ok?
                if ( row_check_ok )
                {
                    // num crystals ok?
                    if ( my_player_data.check_crystals(current_task->required_crystals,
                                                       current_task->num_required_crystals) )
                    {
                        crystal_check_ok = true;
                    } // if
                } // if               
                            
                if ( row_check_ok && crystal_check_ok )
                {
                    //erase_task_message();
                    current_task = my_tutorial.next_task(); 
                }
                else
                {
                    //erase_task_message();
                    draw_task_message(current_task->msg);
                    looping = false;
                } // if ... else
                break;
        } // switch      
    } while (looping);  
    
} // FS::Play::check_task 

// **************************************************

