/*  Remote Windows CE device filesystem.

    Based on lufs "Userland filesystem" ( http://lufs.sourceforge.net/ )
    and synce ( http://synce.sourceforge.net/ )
    projects.

    Author: Fedor Bezrukov <fedor@ms2.inr.ac.ru>
 */

#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <utime.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <vector>
#include <string>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "cefs.h"

#include <rapi.h>
#include <iconv.h>
#include <sys/mman.h>
#include <errno.h>

extern "C" {

void*
cefs_init(struct list_head *cfg, struct dir_cache *cache, struct credentials *cred, void **global_ctx){
    return (void*)new CEFS(cfg, cache);
}

void
cefs_free(void *ctx){
    CEFS *p = (CEFS*)ctx;

    delete p;
}

int 	
cefs_mount(void *ctx){
    return ((CEFS*)ctx)->do_mount();
}

void 	
cefs_umount(void *ctx){
//    return ((CEFS*)ctx)->do_umount();
}

int 	
cefs_readdir(void *ctx, char *dir_name, struct directory *dir){
    return ((CEFS*)ctx)->do_readdir(dir_name, dir);
}

int 	
cefs_stat(void *ctx, char *name, struct lufs_fattr *fattr){
    return ((CEFS*)ctx)->do_stat(name, fattr);
}

int 	
cefs_mkdir(void *ctx, char *dir, int mode){
    return ((CEFS*)ctx)->do_mkdir(dir, mode);
}

int 	
cefs_rmdir(void *ctx, char *dir){
    return ((CEFS*)ctx)->do_rmdir(dir);
}

int 	
cefs_create(void *ctx, char *file, int mode){
    return ((CEFS*)ctx)->do_create(file, mode);
}

int 	
cefs_unlink(void *ctx, char *file){
    return ((CEFS*)ctx)->do_unlink(file);
}

int 	
cefs_rename(void *ctx, char *old_name, char *new_name){
    return ((CEFS*)ctx)->do_rename(old_name, new_name);
}

int 	
cefs_open(void *ctx, char *file, unsigned mode){
    return ((CEFS*)ctx)->do_open(file, mode);
}

int 	
cefs_release(void *ctx, char *file){
    return ((CEFS*)ctx)->do_release(file);
}

int 	
cefs_read(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((CEFS*)ctx)->do_read(file, offset, count, buf);
}

int	
cefs_write(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((CEFS*)ctx)->do_write(file, offset, count, buf);
}

int 	
cefs_readlink(void *ctx, char *link, char *buf, int buflen){
    return ((CEFS*)ctx)->do_readlink(link, buf, buflen);
}

int 	
cefs_link(void *ctx, char *target, char *link){
    return ((CEFS*)ctx)->do_link(target, link);
}

int 	
cefs_symlink(void *ctx, char *target, char *link){
    return ((CEFS*)ctx)->do_symlink(target, link);
}

int 	
cefs_setattr(void *ctx, char *file, struct lufs_fattr *fattr){
    return ((CEFS*)ctx)->do_setattr(file, fattr);
}

} /* extern "C" */


/*
 * It controls whether a new Rapi connection is initialized for every
 * function or only once per process.  I am not sure at the moment,
 * what is better.
 */
#define CERAPIINIT_PER_PROCESS

// WinCE Send buffer size
//#define MYBUF_SIZE (16*1024)
#define MYBUF_SIZE (64*1024)

//
// Return values from main()
//
#define TEST_SUCCEEDED 0
#define TEST_FAILED -1
//
// HRESULT tests
// 
#define VERIFY_HRESULT(call) \
TRACE("CeRapiInit");\
if (FAILED((call))) { WARN("FAIL."); return TEST_FAILED; }


//
// Convert text
//
char* from_unicode(const WCHAR* inbuf)
{
	size_t length = wcslen(inbuf);
	size_t inbytesleft = length * 2, outbytesleft = length;
	char* outbuf = new char[outbytesleft+1];
	char* outbuf_iterator = outbuf;
	char* inbuf_iterator = (char*)inbuf;
	
	iconv_t cd = iconv_open("KOI8-R", "UNICODELITTLE");
	size_t result = iconv(cd, &inbuf_iterator, &inbytesleft, &outbuf_iterator, &outbytesleft);
	iconv_close(cd);

	if ((size_t)-1 == result)
		strcpy(outbuf, "(failed)");
	else
		outbuf[length] = 0;
	
	return outbuf;
}

WCHAR* to_unicode(const char* inbuf)
{
	size_t length = strlen(inbuf);
	size_t inbytesleft = length, outbytesleft = (length+1)* 2;
	char * inbuf_iterator = const_cast<char*>(inbuf);
	WCHAR* outbuf = new WCHAR[inbytesleft+1];
	WCHAR* outbuf_iterator = outbuf;
	
	iconv_t cd = iconv_open("UNICODELITTLE", "KOI8-R");
	size_t result = iconv(cd, &inbuf_iterator, &inbytesleft, (char**)&outbuf_iterator, &outbytesleft);
	iconv_close(cd);

	if ((size_t)-1 == result)
		outbuf[0] = 0;
	else
		outbuf[length] = 0;

	return outbuf;
}

char* ce_to_unix_name(const WCHAR* cefile)
{
	char* file = from_unicode(cefile);
	for (char* pp=file; *pp!=0; pp++)
		if (*pp=='/') *pp = '%';
	return file;
}

WCHAR* unix_to_ce_name(const char* file)
{
	WCHAR* path = to_unicode(file);
	for (WCHAR* pp=path; *pp!=0; pp++)
		if (*pp=='/') *pp='\\';
		else if (*pp=='\\') *pp='%';
	return path;
}

//static void noalrmHandler(int sig){
//    TRACE("False timeout...");
//    
//    signal(SIGALRM, noalrmHandler);
//    siginterrupt(SIGALRM, 1);
//}

CEFS::CEFS(struct list_head* cf, struct dir_cache *cache, struct credentials *cred){
    cfg = cf;
    this->cache = cache;
    this->cred = cred;


	stored_data = NULL;
	stored_data_len = 0;
	stored_file = NULL;
	writing = FALSE;

	rhnd = new RAPIHandle;
	rhnd->sock=0;
	rhnd->_lasterror=0;
	rhnd->buffer=NULL;
	rhnd->hostname=NULL;
	rhnd->lockbuffer=NULL;
	rhnd->lockbuffersize=0;

	TRACE("in CEFS constructor");
}

CEFS::~CEFS(){
	delete rhnd;
	TRACE("in CEFS destructor");
}

int
CEFS::do_mount(){
	TRACE("mounting a cefs.");
#ifdef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
	return 1;
}

void
CEFS::do_umount(){
	TRACE("umounting a cefs.");
#ifdef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
}

int
CEFS::do_readdir(char* d, struct directory *dir){
	int result = -1;
	
	TRACE("reading '"<<d<<"'");
	
	WCHAR* path_tmp = unix_to_ce_name(d);
	int dlen=wcslen(path_tmp);
	WCHAR path[dlen+5];
	wcscpy(path,path_tmp);
	delete[] path_tmp;
	WCHAR* pd = path+dlen;
	if ( *(pd-1) != '\\' ) *(pd++) = '\\';
	*(pd++) = '*'; *(pd++) = '.'; *(pd++) = '*';
	*(pd++) = 0;

#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif

	CE_FIND_DATA* find_data = NULL;
	DWORD file_count = 0;

	/* lufsd is now using pthreads, sorry no more alarms... */
//	signal(SIGALRM, noalrmHandler);
//	siginterrupt(SIGALRM, 1);

	if( !CeFindAllFiles(rhnd,path,
			    ( FAF_NAME|FAF_SIZE_LOW
			      |FAF_LASTWRITE_TIME|FAF_LASTACCESS_TIME
			      |FAF_CREATION_TIME
			      |FAF_ATTRIBUTES ),
			    &file_count, &find_data) ) {
		TRACE("could not open directory!");
		goto out;
	}

	struct lufs_fattr fattr;
	fattr.f_mode = S_IFDIR;
//	fattr.f_mode &= ~options.o_umask;
	fattr.f_nlink = 1;
	fattr.f_uid = 1;
//	fattr.f_uid = options.o_uid;
	fattr.f_gid = 1;
//	fattr.f_gid = options.o_gid;
	/* Are these lines correct? */
	/* I mean, they are surely wrong, but does this change anything? */

	/* F.M.: yup, they're ok, not really used. they're just placeholders */
	/*       in cache, so that each dir contains al least 2 entries... */
//	lu_cache_add2dir(dir, ".", NULL, &fattr);
//	lu_cache_add2dir(dir, "..", NULL, &fattr);
	
	for (unsigned i = 0; i < file_count; i++) {
		char* d_name = ce_to_unix_name(find_data[i].cFileName);
		TRACE("adding direntry " << d_name);


		fattr.f_mode =  S_IFREG;
		if ( FILE_ATTRIBUTE_DIRECTORY & find_data[i].dwFileAttributes ) {
			fattr.f_mode = S_IFDIR;
//			fattr.f_mode |= 0111;
		}
//		fattr.f_mode |= 0666;
//		fattr.f_mode &= ~options.o_umask;
		fattr.f_size = find_data[i].nFileSizeLow;
		fattr.f_atime = DOSFS_FileTimeToUnixTime(
			&(find_data[i].ftLastAccessTime),NULL );
		fattr.f_mtime = DOSFS_FileTimeToUnixTime(
			&(find_data[i].ftLastWriteTime),NULL );
		fattr.f_ctime =  DOSFS_FileTimeToUnixTime(
			&(find_data[i].ftCreationTime),NULL );
		lu_cache_add2dir(dir, d_name, NULL, &fattr);
		delete[] d_name;
	}

	CeRapiFreeBuffer(rhnd,find_data);
	result = 0;
out:
#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return result;
}

int
CEFS::do_stat(char *nm, struct lufs_fattr *fattr){
	int result = -1;

	TRACE("stat: " << nm);

	string s(nm);
	unsigned i;
	if((i = s.find_last_of('/')) == string::npos){
		WARN("couldn't isolate dir in "<<nm<<"!");
		return -1;
	}
	string dir = (i == 0) ? string("/") : s.substr(0, i+1);
	if (dir=="//") dir="/";
	string f = dir+s.substr(i + 1, s.length() - i - 1);
	TRACE("stat: using " << f.c_str());
	WCHAR* path = unix_to_ce_name(f.c_str());
//	WCHAR* path = unix_to_ce_name(nm);

	fattr->f_mode =  S_IFREG;
//	fattr->f_mode |= 0664;
//	fattr->f_mode &= ~options.o_umask;
	fattr->f_uid = 1;
//	fattr->f_uid = options.o_uid;
	fattr->f_gid = 1;
//	fattr->f_gid = options.o_gid;

	if ( f=="/." ) { // Artificial root directory
		fattr->f_mode =  S_IFDIR;
		return 0;
	}

#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif

	CE_FIND_DATA find_data;
	HANDLE hnd=CeFindFirstFile(rhnd,path, &find_data);
	delete[] path;
	if (INVALID_HANDLE_VALUE==hnd) {
		TRACE("CeFindFirstFile of '"<<nm<<"'failed in do_stat!");
		goto out;
	}
	if ( FAILED(CeFindClose( rhnd,hnd )) )
		WARN("CeFindClose failed in do_stat!");

	if ( FILE_ATTRIBUTE_DIRECTORY & find_data.dwFileAttributes ) {
		fattr->f_mode = S_IFDIR;
//		fattr->f_mode |= 0111;
	}
//	fattr->f_mode |= 0666;
//	fattr->f_mode &= ~options.o_umask;
	fattr->f_nlink = 1;
	fattr->f_size = find_data.nFileSizeLow;
	fattr->f_atime = DOSFS_FileTimeToUnixTime(
		&(find_data.ftLastAccessTime),NULL );
	fattr->f_mtime = DOSFS_FileTimeToUnixTime(
		&(find_data.ftLastWriteTime),NULL );
	fattr->f_ctime =  DOSFS_FileTimeToUnixTime(
		&(find_data.ftCreationTime),NULL );

	result = 0;
out:
#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return result;
}

int
CEFS::do_mkdir(char *dir, int mode){
#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
	WCHAR* path = unix_to_ce_name(dir);
	int res = CeCreateDirectory(rhnd,path, NULL);
	delete[] path;
#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return res ? 0:-1;
}

int
CEFS::do_rmdir(char *dir){
#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
	WCHAR* path = unix_to_ce_name(dir);
	int res =  CeRemoveDirectory(rhnd,path);
	delete[] path;
#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return res ? 0:-1;
}

int
CEFS::do_create(char *file, int mode){
	int result = -1;
#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
	TRACE(file<< " mode " << mode);
	WCHAR* path = unix_to_ce_name(file);
	HANDLE handle = CeCreateFile(rhnd,path,
				     GENERIC_WRITE,
				     0,
				     NULL,
				     CREATE_ALWAYS,
				     FILE_ATTRIBUTE_NORMAL,
				     NULL);
	delete[] path;
	if (INVALID_HANDLE_VALUE==handle) goto out;
	CeCloseHandle(rhnd,handle);

	result = 0;
out:
#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return result;
}

int
CEFS::do_unlink(char *file){
#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
	WCHAR* path =unix_to_ce_name(file);
	int res =  CeDeleteFile(rhnd,path);
	delete[] path;
#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return res ? 0:-1;
}

int
CEFS::do_rename(char *old, char *nnew){
#ifndef CERAPIINIT_PER_PROCESS
	VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
	TRACE(old << " to " << nnew);

	WCHAR* oldpath = unix_to_ce_name(old);
	WCHAR* newpath = unix_to_ce_name(nnew);

	int res=CeMoveFile(rhnd, oldpath,newpath );

	delete[] oldpath;
	delete[] newpath;

#ifndef CERAPIINIT_PER_PROCESS
	CeRapiUninit(rhnd);
#endif
	return res ? 0:-1;
}

int
CEFS::do_open(char *file, unsigned mode){

	TRACE(file <<"mode" << mode);
  	if (stored_data_len) {
  		do_release(stored_file);
  	}
  	if ( mode&O_WRONLY ) {
  		writing = TRUE;
		stored_file = strdup(file);
	}
	return 0;
}

int
CEFS::do_release(char *file){
	TRACE(file);
	int result = 0;
	if (writing && stored_data_len) {
		result = -1;
#ifndef CERAPIINIT_PER_PROCESS
		VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
		WCHAR* path = unix_to_ce_name(file);
		HANDLE handle = CeCreateFile(rhnd,path,
					     GENERIC_WRITE,
					     0,
					     NULL,
					     CREATE_ALWAYS,
					     FILE_ATTRIBUTE_NORMAL,
					     NULL);
		delete[] path;
		if (handle == INVALID_HANDLE_VALUE) goto out;
		for ( char* buffer=stored_data;
		      buffer<stored_data+stored_data_len;
		      buffer+=MYBUF_SIZE ) {
			DWORD did_write = (DWORD)-1;
			size_t bytes_read = MYBUF_SIZE;
			if ( buffer+MYBUF_SIZE >= stored_data+stored_data_len )
				bytes_read = stored_data+stored_data_len-buffer;

			if ( 0==CeWriteFile(rhnd,handle, buffer, bytes_read, &did_write, NULL) )
				goto out;
			if (bytes_read != did_write)
				WARN("Only wrote %lu bytes to file of %u possible.\n"<< did_write<< bytes_read);
			usleep(100);
		}
		CeCloseHandle(rhnd,handle);

		result = 0;
out:
#ifndef CERAPIINIT_PER_PROCESS
		CeRapiUninit(rhnd);
#endif
		;
	}
	if (stored_data_len) {
		free(stored_data);
		free(stored_file);
		stored_data_len = 0;
		stored_data=NULL;
		stored_file=NULL;
	}
	writing = FALSE;
	return result;
}

int
CEFS::do_read(char *file, long long offset, unsigned long count, char *buf){
	TRACE("read "<<file<<", "<<offset<<", "<<count);

	int result = -1;
	if ( stored_data_len==0 || strcmp(file,stored_file)!=0 ) {
//		signal(SIGALRM, noalrmHandler);
//		siginterrupt(SIGALRM, 1);

		struct lufs_fattr fattr;
		if ( do_stat(file, &fattr)!=0 )
			return -1;
		if (stored_data_len) {
			free(stored_data);
			free(stored_file);
		}
		stored_file=strdup(file);
		stored_data_len = fattr.f_size;
		stored_data = (char*)malloc(stored_data_len*sizeof(char));
#ifndef CERAPIINIT_PER_PROCESS
		VERIFY_HRESULT(CeRapiInit(rhnd));
#endif
		WCHAR* path = unix_to_ce_name(file);

		HANDLE handle = CeCreateFile(rhnd,path,
					     GENERIC_READ,
					     0,
					     NULL,
					     OPEN_EXISTING,
					     FILE_ATTRIBUTE_NORMAL,
					     NULL);
		delete[] path;
		if (INVALID_HANDLE_VALUE == handle ) goto out;
		DWORD bytes_read;
		for ( char* point=stored_data;
		      CeReadFile(rhnd,handle,point,MYBUF_SIZE,&bytes_read,NULL);
		      point += bytes_read )
			if (bytes_read==0) break;
		CeCloseHandle(rhnd,handle);

		result = 0;
	out:
#ifndef CERAPIINIT_PER_PROCESS
		CeRapiUninit(rhnd);
#endif
		if (result < 0) return result;
	}
	if (offset+count>stored_data_len)
		count = stored_data_len-offset;
	memcpy(buf,stored_data+offset,count);
	result = count;
	return result;
}

int
CEFS::do_write(char *file, long long offset, unsigned long count, char *buf){

	TRACE("write "<<file<<", "<<offset<<", "<<count);
	if ( stored_data_len!=0 && strcmp(file,stored_file)!=0 ) {
		do_release(file);
		stored_file=strdup(file);
	}
	if (!writing)
		return -1;
	if (offset+count>stored_data_len) {
		stored_data_len = offset+count;
		stored_data = (char*)realloc(stored_data,stored_data_len);
	}
	memcpy(stored_data+offset,buf,count);
	return count;
}


int
CEFS::do_setattr(char* file, struct lufs_fattr*)
{
	TRACE(file);
	return 0;
}
