// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package fingerprint

import (
	"crypto/md5"
	"crypto/sha1"
	"crypto/sha256"
	"crypto/sha512"
	"hash"
	"strings"

	"github.com/cespare/xxhash/v2"
)

type hashMethod func() hash.Hash

var hashes = map[string]hashMethod{
	"md5":    md5.New,
	"sha1":   sha1.New,
	"sha256": sha256.New,
	"sha384": sha512.New384,
	"sha512": sha512.New,
	"xxhash": newXxHash,
}

// Unpack creates the hashMethod from the given string
func (f *hashMethod) Unpack(str string) error {
	str = strings.ToLower(str)

	m, found := hashes[str]
	if !found {
		return makeErrUnknownMethod(str)
	}

	*f = m
	return nil
}

// newXxHash returns a hash.Hash instead of the *Digest which implements the same
func newXxHash() hash.Hash {
	return xxhash.New()
}
