/*
** Copyright (C) 2004 Andrew R. Baker <andrewb@snort.org>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "CommandLineArgs.h"
#include "util.h"

extern char *optarg;              /* for getopt */
extern int optind;                /* for getopt */

int CommandLineArgs_Parse(int argc, char **argv, CommandLineArgs **clargs)
{
    int ch;
    CommandLineArgs *tmp;
    
    if(!clargs)
        return -1;

    if(!(tmp = (CommandLineArgs *)calloc(1, sizeof(CommandLineArgs))))
    {
        FatalError("Out of memory (wanted %u bytes)", sizeof(CommandLineArgs));
        return -1;
    }

    /* Process options */
    while((ch = getopt(argc, argv, "a:c:d:f:g:hnop:s:vw:DL:RVX:?")) != -1)
    {
        switch(ch)
        {
            case 'a':
                if(tmp->archive_dir)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->archive_dir = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
                
            case 'c':
                if(tmp->config_file)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->config_file = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
            
            case 'd':
                if(tmp->spool_dir)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->spool_dir = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
                
            case 'f':
                if(tmp->file_base)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->file_base = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
            
            case 'g':  /* set the generator names file */
                if(tmp->gen_msg_file)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->gen_msg_file = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
            
            case 'h':
            case '?':
                tmp->usage_flag = 1;
                break;

            case 'n':
                tmp->new_records_only_flag = 1;
                break;
    
            case 'o':
                tmp->batch_mode_flag = 1;
                break;
    
            case 'p':
                if(tmp->class_file)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->class_file = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
                
            case 's': /* load the sid map from this file */
                if(tmp->sid_msg_file)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->sid_msg_file = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
        
            case 'v': /* increment verbosity */
                if(tmp->verbosity == 255)
                    LogMessage("WARNING: Trying to increase verbosity above 255\n");
                else
                    tmp->verbosity++;
                break;

            case 'w': /* set the waldo (aka bookmark) filename */
                if(tmp->waldo_file)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->waldo_file = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
            
            case 'D': /* run in daemon mode */
                tmp->daemon_flag = 1;
                break;

            case 'L': /* specify output log dir and file */
                if(tmp->log_dir)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->log_dir = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;

            case 'R':
                tmp->dry_run_flag = 1;
                break;

            case 'V':
                tmp->version_flag = 1;
                break;
                
            case 'X':   /* specify the pid file */
                if(tmp->pid_file)
                    FatalError("Multiple '-%c' command line options", ch);
                if(!(tmp->pid_file = strdup((const char *)optarg)))
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(optarg) + 1);
                break;
        }
    }

    /* Process the rest of the command line */
    if(optind < argc)
    {
        int idx = 0;
        tmp->extra_args_count = argc - optind;
        if(!(tmp->extra_args = (char **)calloc(tmp->extra_args_count + 1, 
                        sizeof(char *))))
        {
            FatalError("Out of memory (wanted %u bytes)\n", 
                    (tmp->extra_args_count + 1) * sizeof(char *));
        }
        while(optind < argc)
        {
            if(!(tmp->extra_args[idx++] = strdup(argv[optind])))
            {
                FatalError("Out of memory (wanted %u bytes)\n",
                        strlen(argv[optind]) + 1);
            }
            optind++;
        }
    }

    *clargs = tmp;

    return 0;
}

int CommandLineArgs_Fprintf(CommandLineArgs *clargs, FILE *stream)
{
    if(!clargs)
        return -1;

    if(!stream)
        stream = stdout;

    fprintf(stream, "Command line arguments:\n");
    fprintf(stream, "  Config file:           %s\n", 
            clargs->config_file ? clargs->config_file : "Not specified");
    fprintf(stream, "  Spool dir:             %s\n", 
            clargs->spool_dir ? clargs->spool_dir : "Not specified");
    fprintf(stream, "  Gen-msg file:          %s\n", 
            clargs->gen_msg_file ? clargs->gen_msg_file : "Not specified");
    fprintf(stream, "  Sid-msg file:          %s\n", 
            clargs->sid_msg_file ? clargs->sid_msg_file : "Not specified");
    fprintf(stream, "  Class file:            %s\n", 
            clargs->class_file ? clargs->class_file : "Not specified");
    fprintf(stream, "  Log dir:               %s\n", 
            clargs->log_dir ? clargs->log_dir : "Not specified");
    fprintf(stream, "  Archive dir:           %s\n", 
            clargs->archive_dir ? clargs->archive_dir : "Not specified");
    fprintf(stream, "  File base:             %s\n", 
            clargs->file_base ? clargs->file_base : "Not specified");
    fprintf(stream, "  Waldo file:            %s\n", 
            clargs->waldo_file ? clargs->waldo_file : "Not specified");
    fprintf(stream, "  Pid file:              %s\n", 
            clargs->pid_file ? clargs->pid_file : "Not specified");
    fprintf(stream, "  Verbosity level:       %u\n", clargs->verbosity);
    fprintf(stream, "  Dry run flag:          %s\n", 
            clargs->dry_run_flag ? "Set" : "Not Set");
    fprintf(stream, "  Batch mode flag:       %s\n", 
            clargs->batch_mode_flag ? "Set" : "Not Set");
    fprintf(stream, "  Daemon flag:           %s\n", 
            clargs->daemon_flag ? "Set" : "Not Set");
    fprintf(stream, "  New records only flag: %s\n", 
            clargs->new_records_only_flag ? "Set" : "Not Set");
    fprintf(stream, "  Usage flag:            %s\n", 
            clargs->usage_flag ? "Set" : "Not Set");
    fprintf(stream, "  Version flag:          %s\n", 
            clargs->version_flag ? "Set" : "Not Set");
    
    /* Print extra arguments */
    if(clargs->extra_args && clargs->extra_args[0])
    {
        int idx = 0;
        fprintf(stream, "  Additional arguments:\n");
        while(clargs->extra_args[idx])
        {
            fprintf(stream, "    %s\n", clargs->extra_args[idx]);
            idx++;
        }
    }

    return 0;
}

int CommandLineArgs_Validate(CommandLineArgs *clargs)
{
    if(!clargs)
        return -1;

    if(clargs->batch_mode_flag)
    {
        /* Make sure no continual mode flags are set */
        if(clargs->archive_dir || clargs->new_records_only_flag ||
                clargs->waldo_file || clargs->daemon_flag ||
                clargs->pid_file)
        {
            fprintf(stderr, "ERROR: Continual processing mode options "
                    "specified with batch mode flag\n");
            return -1;
        }
        if(!clargs->file_base && (!clargs->extra_args || 
                !(clargs->extra_args[0])))
        {
            fprintf(stderr, "ERROR: No files specified for batch mode\n");
            return -1;
        }
    }
    else
    {
        /* Continual processing mode */
        if(!clargs->file_base && !clargs->waldo_file)
        {
            fprintf(stderr, "ERROR: No files specified for continual "
                    "processing mode\n");
            return -1;
        }
    }
    return 0;
}

int CommandLineArgs_PrintUsage(FILE *stream)
{
    if(!stream)
        stream = stderr;

    fprintf(stream, "Usage: barnyard [OPTIONS]...             (continual mode)\n");
    fprintf(stream, "   or: barnyard -o [OPTIONS]... FILES... (batch mode)\n");
    fprintf(stream, "Information Options:\n");
    fprintf(stream, "  -h         Show this help information\n");
    fprintf(stream, "  -?         Show this help information\n");
    fprintf(stream, "  -V         Show version and exit\n");
    fprintf(stream, "  -R         Display processed configuration and exit\n");
    fprintf(stream, "General Configuration Options:\n");
    fprintf(stream, "  -c <file>  Use configuration file <file>\n");
    fprintf(stream, "  -d <dir>   Read spool files from <dir>\n");
    fprintf(stream, "  -L <dir>   Write output files in <dir>\n");
    fprintf(stream, "  -v         Increase the verbosity level by 1\n");
    fprintf(stream, "  -s <file>  Read the sid-msg map from <file>\n");
    fprintf(stream, "  -g <file>  Read the gen-msg map from <file>\n");
    fprintf(stream, "  -p <file>  Read the classification map from <file>\n");
    fprintf(stream, "Continual Processing Mode Options:\n");
    fprintf(stream, "  -a <dir>   Archive processed files to <dir>\n");
    fprintf(stream, "  -f <base>  Use <base> as the base unified filename\n");
    fprintf(stream, "  -n         Only process new events\n");
    fprintf(stream, "  -w <file>  Enable bookmarking using <file>\n");
    fprintf(stream, "  -D         Run in daemon mode\n");
    fprintf(stream, "  -X <file>  Use <file> as the pid file\n");
    fprintf(stream, "Batch Processing Mode Options:\n");
    fprintf(stream, "  -o         Enable batch processing mode\n");

    return 0;
}
