// CD_GUI.hh for BBCD - a CD player for X11 / BlackBox
// Copyright (c) 2002 Bertrand Duret <bertrand.duret at libertysurf.fr>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef __CD_GUI_HH
#define __CD_GUI_HH

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/time.h>

#include "bbcd.hh"
#include "Button.hh"
#include "Texture.hh"

// Forward declaration
class CD_Controler;


// Specialized ButtonDrawers: Play, Pause, Forward, Backward
class PlayBDrawer: public ButtonDrawer{
public:
  // PlayDrawer has an additional GC used while player is playing
  PlayBDrawer(Display*, GC*, GC*, BTexture*, BTexture*);
  void draw(Button*, bool pressed = false, bool = false) const;
  inline bool isPlaying() const { return playing; }
  inline void setPlaying(bool p) { playing = p; }
protected:
  bool playing;
}; // class PlayBDrawer

class PauseBDrawer: public ButtonDrawer{
public:
  PauseBDrawer(Display*, GC*, GC*, BTexture*, BTexture*);
  void draw(Button*, bool pressed = false, bool = false) const;
  void blink() { blinkState = !blinkState; }
protected:
  bool blinkState;
}; // class PauseBDrawer

class StopBDrawer: public ButtonDrawer{
public:
  StopBDrawer(Display*, GC*, GC*, BTexture*, BTexture*);
  void draw(Button*, bool pressed = false, bool = false) const;
}; // class StopBDrawer

class ForwardBDrawer: public ButtonDrawer{
public:
  ForwardBDrawer(Display*, GC*, GC*, BTexture*, BTexture*);
  void draw(Button*, bool pressed = false, bool = false) const;
}; // class ForwardBDrawer

class BackwardBDrawer: public ButtonDrawer{
public:
  BackwardBDrawer(Display*, GC*, GC*, BTexture*, BTexture*);
  void draw(Button*, bool pressed = false, bool = false) const;
}; // class BackwardBDrawer




// Main GUI of bbcd
class CD_GUI
{
public:
  // Exceptions
  class UnableToOpenDisplay{};

  CD_GUI(Option*) throw (UnableToOpenDisplay);
  ~CD_GUI();

  void run(); //  Process X Events and Timeouts

private:
  Display* display;
  int screen;
  Window playerW;
  Pixmap background;
  bool running; // while true, bbcd is running
  unsigned int width, height;
  int x, y;
  Option* option;
    // In compact mode, only playB and stopB are used !
  Button* playB, * backwardB, * forwardB, * stopB;
  PlayBDrawer* playBD;
  PauseBDrawer* pauseBD;
  ButtonDrawer* backwardBD, * forwardBD, * stopBD, *restoreBD;
  BTexture* backgroundT, * releasedT, * pressedT;
  BImageControl* img_ctrl;
  GC gc, gcActive;
  unsigned long timeoutDelay;
  timeval startTime;
  bool timerStarted;

    // Create main window and buttons
  void makeGUI();
  void computeSizeAndPosition(XSizeHints *);
  void cleanupGUI(); // Destroy main window and buttons
  inline Pixmap getPixmap() const { return background; }
  inline void setPixmap(Pixmap pm)
    { XSetWindowBackgroundPixmap(display, playerW, pm);
      background = pm; }
  void setBackground(); // Set background (pixmap, color, transparent)

    // Style/Look Management
  void readStyles(); // Read bbcd config + bb style, over the default style
  void readBBStyle(); // Read the BlackBox file style
  void readBBCDStyle(); // Read the BBCD file style (i.e. configuration file)
  void readStyle(const XrmDatabase &); // Read a style from a database
  void setDefaultStyle(); // Generate a default style

    // CD drive controler
  CD_Controler * cd_ctrl;

  void processExpose(const XExposeEvent &);
  void processButtonPress(const XButtonPressedEvent &);
  void processButtonRelease(const XButtonReleasedEvent &);
  void processConfigure(const XConfigureEvent &);
  void processClientMessage(const XClientMessageEvent &);
  void processTimeout();
  void redraw(bool forced = false);

  void startTimer(unsigned long delay = 0l);
  inline void stopTimer() { timerStarted = false; }
  inline void setDelay(unsigned long delay) { timeoutDelay = delay; }

  Atom wm_delete_window; // Protocol stuff...
}; // class CD_GUI

#endif // __CD_GUI_HH
