/* Copyright 2019 The Chromium OS Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

/**
 * @file
 * @brief Register map for the STM32L4 family of chips
 *
 * This header file should not be included directly.
 * Please include registers.h instead.
 *
 * Known Chip Variants
 * - STM32L442
 * - STM32L476
 */

#ifndef __CROS_EC_REGISTERS_H
#error "This header file should not be included directly."
#endif

/******  STM32 specific Interrupt Numbers ********/
#define   STM32_IRQ_WWDG                    0
#define   STM32_IRQ_PVD_PVM                 1
#define   STM32_IRQ_TAMP_STAMP              2
#define   STM32_IRQ_RTC_WKUP                3
#define   STM32_IRQ_FLASH                   4
#define   STM32_IRQ_RCC                     5
#define   STM32_IRQ_EXTI0                   6
#define   STM32_IRQ_EXTI1                   7
#define   STM32_IRQ_EXTI2                   8
#define   STM32_IRQ_EXTI3                   9
#define   STM32_IRQ_EXTI4                   10
#define   STM32_IRQ_DMA_CHANNEL_1           11
#define   STM32_IRQ_DMA_CHANNEL_2           12
#define   STM32_IRQ_DMA_CHANNEL_3           13
#define   STM32_IRQ_DMA_CHANNEL_4           14
#define   STM32_IRQ_DMA_CHANNEL_5           15
#define   STM32_IRQ_DMA_CHANNEL_6           16
#define   STM32_IRQ_DMA_CHANNEL_7           17
#define   STM32_IRQ_ADC1                    18
#define   STM32_IRQ_CAN1_TX                 19
#define   STM32_IRQ_CAN1_RX0                20
#define   STM32_IRQ_CAN1_RX1                21
#define   STM32_IRQ_CAN1_SCE                22
#define   STM32_IRQ_EXTI9_5                 23
#define   STM32_IRQ_TIM1_BRK_TIM15          24
#define   STM32_IRQ_TIM1_UP_TIM16           25
#define   STM32_IRQ_TIM1_TRG_COM            26
#define   STM32_IRQ_TIM1_CC                 27
#define   STM32_IRQ_TIM2                    28
#define   STM32_IRQ_I2C1_EV                 31
#define   STM32_IRQ_I2C1_ER                 32
#define   STM32_IRQ_I2C2_EV                 33
#define   STM32_IRQ_I2C2_ER                 34
#define   STM32_IRQ_SPI1                    35
#define   STM32_IRQ_SPI2                    36
#define   STM32_IRQ_USART1                  37
#define   STM32_IRQ_USART2                  38
#define   STM32_IRQ_USART3                  39
#define   STM32_IRQ_EXTI15_10               40
#define   STM32_IRQ_RTC_ALARM               41
#define   STM32_IRQ_SDMMC1                  49
#define   STM32_IRQ_TIM5                    50
#define   STM32_IRQ_SPI3                    51
#define   STM32_IRQ_TIM6_DAC                54
#define   STM32_IRQ_TIM7                    55
#define   STM32_IRQ_DMA2_CHANNEL1           56
#define   STM32_IRQ_DMA2_CHANNEL2           57
#define   STM32_IRQ_DMA2_CHANNEL3           58
#define   STM32_IRQ_DMA2_CHANNEL4           59
#define   STM32_IRQ_DMA2_CHANNEL5           60
#define   STM32_IRQ_COMP                    64
#define   LSTM32_IRQ_PTIM1                  65
#define   STM32_IRQ_LPTIM2                  66
#define   STM32_IRQ_DMA2_CHANNEL6           68
#define   STM32_IRQ_DMA2_CHANNEL7           69
#define   STM32_IRQ_LPUART1                 70
#define   STM32_IRQ_QUADSPI                 71
#define   STM32_IRQ_I2C3_EV                 72
#define   STM32_IRQ_I2C3_ER                 73
#define   STM32_IRQ_SAI1                    74
#define   STM32_IRQ_SWPMI1                  76
#define   STM32_IRQ_TSC                     77
#define   STM32_IRQ_RNG                     80
#define   STM32_IRQ_FPU                     81
#define   STM32_IRQ_CRS                     82

/* To simplify code generation, define DMA channel 9..10 */
#define STM32_IRQ_DMA_CHANNEL_9    STM32_IRQ_DMA2_CHANNEL1
#define STM32_IRQ_DMA_CHANNEL_10   STM32_IRQ_DMA2_CHANNEL2
#define STM32_IRQ_DMA_CHANNEL_11   STM32_IRQ_DMA2_CHANNEL3
#define STM32_IRQ_DMA_CHANNEL_12   STM32_IRQ_DMA2_CHANNEL4
#define STM32_IRQ_DMA_CHANNEL_13   STM32_IRQ_DMA2_CHANNEL5
#define STM32_IRQ_DMA_CHANNEL_14   STM32_IRQ_DMA2_CHANNEL6
#define STM32_IRQ_DMA_CHANNEL_15   STM32_IRQ_DMA2_CHANNEL7
#define STM32_IRQ_USART9           STM32_IRQ_LPUART1


/* Peripheral base addresses */
#define FLASH_BASE            0x08000000UL
#define FLASH_END             0x0803FFFFUL
#define FLASH_BANK1_END       0x0803FFFFUL
#define SRAM1_BASE            0x20000000UL
#define SRAM2_BASE            0x10000000UL
#define PERIPH_BASE           0x40000000UL
#define QSPI_BASE             0x90000000UL
#define QSPI_R_BASE           0xA0001000UL
#define SRAM1_BB_BASE         0x22000000UL
#define PERIPH_BB_BASE        0x42000000UL

/* Legacy defines */
#define SRAM_BASE             SRAM1_BASE
#define SRAM_BB_BASE          SRAM1_BB_BASE
#define SRAM1_SIZE_MAX        0x0000C000UL
#define SRAM2_SIZE            0x00004000UL
#define FLASH_SIZE_DATA_REGISTER ((uint32_t)0x1FFF75E0)
#define FLASH_SIZE               (((((*((uint32_t *)FLASH_SIZE_DATA_REGISTER)) \
			& (0x0000FFFFU)) == 0x0000FFFFU)) ? (0x100U << 10U) : \
			(((*((uint32_t *)FLASH_SIZE_DATA_REGISTER)) & \
			(0x0000FFFFU)) << 10U))

/*!< Peripheral memory map */
#define APB1PERIPH_BASE        PERIPH_BASE
#define APB2PERIPH_BASE       (PERIPH_BASE + 0x00010000UL)
#define AHB1PERIPH_BASE       (PERIPH_BASE + 0x00020000UL)
#define AHB2PERIPH_BASE       (PERIPH_BASE + 0x08000000UL)

/*!< APB1 peripherals */
#define STM32_TIM2_BASE             (APB1PERIPH_BASE + 0x0000UL)
#define STM32_TIM5_BASE             (APB1PERIPH_BASE + 0x0C00UL)
#define STM32_TIM6_BASE             (APB1PERIPH_BASE + 0x1000UL)
#define STM32_TIM7_BASE             (APB1PERIPH_BASE + 0x1400UL)
#define STM32_RTC_BASE              (APB1PERIPH_BASE + 0x2800UL)
#define STM32_WWDG_BASE             (APB1PERIPH_BASE + 0x2C00UL)
#define STM32_IWDG_BASE             (APB1PERIPH_BASE + 0x3000UL)
#define STM32_SPI2_BASE             (APB1PERIPH_BASE + 0x3800UL)
#define STM32_SPI3_BASE             (APB1PERIPH_BASE + 0x3C00UL)
#define STM32_USART2_BASE           (APB1PERIPH_BASE + 0x4400UL)
#define STM32_USART3_BASE           (APB1PERIPH_BASE + 0x4800UL)
#define STM32_I2C1_BASE             (APB1PERIPH_BASE + 0x5400UL)
#define STM32_I2C2_BASE             (APB1PERIPH_BASE + 0x5800UL)
#define STM32_I2C3_BASE             (APB1PERIPH_BASE + 0x5C00UL)
#define STM32_CRS_BASE              (APB1PERIPH_BASE + 0x6000UL)
#define STM32_CAN1_BASE             (APB1PERIPH_BASE + 0x6400UL)
#define STM32_PWR_BASE              (APB1PERIPH_BASE + 0x7000UL)
#define STM32_DAC_BASE              (APB1PERIPH_BASE + 0x7400UL)
#define STM32_DAC1_BASE             (APB1PERIPH_BASE + 0x7400UL)
#define STM32_OPAMP_BASE            (APB1PERIPH_BASE + 0x7800UL)
#define STM32_OPAMP1_BASE           (APB1PERIPH_BASE + 0x7800UL)
#define STM32_LPTIM1_BASE           (APB1PERIPH_BASE + 0x7C00UL)
#define STM32_LPUART1_BASE          (APB1PERIPH_BASE + 0x8000UL)
#define STM32_SWPMI1_BASE           (APB1PERIPH_BASE + 0x8800UL)
#define STM32_LPTIM2_BASE           (APB1PERIPH_BASE + 0x9400UL)

#define STM32_USART9_BASE           STM32_LPUART1_BASE

/*!< APB2 peripherals */
#define STM32_SYSCFG_BASE           (APB2PERIPH_BASE + 0x0000UL)
#define STM32_VREFBUF_BASE          (APB2PERIPH_BASE + 0x0030UL)
#define STM32_OMP1_BASE             (APB2PERIPH_BASE + 0x0200UL)
#define STM32_COMP2_BASE            (APB2PERIPH_BASE + 0x0204UL)
#define STM32_EXTI_BASE             (APB2PERIPH_BASE + 0x0400UL)
#define STM32_FIREWALL_BASE         (APB2PERIPH_BASE + 0x1C00UL)
#define STM32_SDMMC1_BASE           (APB2PERIPH_BASE + 0x2800UL)
#define STM32_TIM1_BASE             (APB2PERIPH_BASE + 0x2C00UL)
#define STM32_SPI1_BASE             (APB2PERIPH_BASE + 0x3000UL)
#define STM32_USART1_BASE           (APB2PERIPH_BASE + 0x3800UL)
#define STM32_TIM15_BASE            (APB2PERIPH_BASE + 0x4000UL)
#define STM32_TIM16_BASE            (APB2PERIPH_BASE + 0x4400UL)
#define STM32_SAI1_BASE             (APB2PERIPH_BASE + 0x5400UL)
#define STM32_SAI1_Block_A_BASE     (SAI1_BASE + 0x0004UL)
#define STM32_SAI1_Block_B_BASE     (SAI1_BASE + 0x0024UL)

/*!< AHB1 peripherals */
#define STM32_DMA1_BASE             (AHB1PERIPH_BASE)
#define STM32_DMA2_BASE             (AHB1PERIPH_BASE + 0x0400UL)
#define STM32_RCC_BASE              (AHB1PERIPH_BASE + 0x1000UL)
#define STM32_FLASH_R_BASE          (AHB1PERIPH_BASE + 0x2000UL)
#define STM32_CRC_BASE              (AHB1PERIPH_BASE + 0x3000UL)
#define STM32_TSC_BASE              (AHB1PERIPH_BASE + 0x4000UL)
#define STM32_DMA1_Channel1_BASE    (DMA1_BASE + 0x0008UL)
#define STM32_DMA1_Channel2_BASE    (DMA1_BASE + 0x001CUL)
#define STM32_DMA1_Channel3_BASE    (DMA1_BASE + 0x0030UL)
#define STM32_DMA1_Channel4_BASE    (DMA1_BASE + 0x0044UL)
#define STM32_DMA1_Channel5_BASE    (DMA1_BASE + 0x0058UL)
#define STM32_DMA1_Channel6_BASE    (DMA1_BASE + 0x006CUL)
#define STM32_DMA1_Channel7_BASE    (DMA1_BASE + 0x0080UL)
#define STM32_DMA1_CSELR_BASE       (DMA1_BASE + 0x00A8UL)
#define STM32_DMA2_Channel1_BASE    (DMA2_BASE + 0x0008UL)
#define STM32_DMA2_Channel2_BASE    (DMA2_BASE + 0x001CUL)
#define STM32_DMA2_Channel3_BASE    (DMA2_BASE + 0x0030UL)
#define STM32_DMA2_Channel4_BASE    (DMA2_BASE + 0x0044UL)
#define STM32_DMA2_Channel5_BASE    (DMA2_BASE + 0x0058UL)
#define STM32_DMA2_Channel6_BASE    (DMA2_BASE + 0x006CUL)
#define STM32_DMA2_Channel7_BASE    (DMA2_BASE + 0x0080UL)
#define STM32_DMA2_CSELR_BASE       (DMA2_BASE + 0x00A8UL)

/*!< AHB2 peripherals */
#define STM32_GPIOA_BASE            (AHB2PERIPH_BASE + 0x0000UL)
#define STM32_GPIOB_BASE            (AHB2PERIPH_BASE + 0x0400UL)
#define STM32_GPIOC_BASE            (AHB2PERIPH_BASE + 0x0800UL)
#define STM32_GPIOD_BASE            (AHB2PERIPH_BASE + 0x0C00UL)
#define STM32_GPIOE_BASE            (AHB2PERIPH_BASE + 0x1000UL)
#define STM32_GPIOF_BASE            (AHB2PERIPH_BASE + 0x1400UL)
#define STM32_GPIOG_BASE            (AHB2PERIPH_BASE + 0x1800UL) /* stm32l4x6 */
#define STM32_GPIOH_BASE            (AHB2PERIPH_BASE + 0x1C00UL)
#define STM32_ADC1_BASE             (AHB2PERIPH_BASE + 0x08040000UL)
#define STM32_ADC1_COMMON_BASE      (AHB2PERIPH_BASE + 0x08040300UL)
#define STM32_RNG_BASE              (AHB2PERIPH_BASE + 0x08060800UL)

/* Debug MCU registers base address */
#define STM32_DBGMCU_BASE           0xE0042000UL
#define STM32_PACKAGE_BASE          0x1FFF7500UL
#define STM32_UID_BASE              0x1FFF7590UL
#define STM32_FLASHSIZE_BASE        0x1FFF75E0UL

#define STM32_FLASH_REGS_BASE       STM32_FLASH_R_BASE
#define STM32_UNIQUE_ID_BASE        STM32_UID_BASE
#define STM32_OPTB_BASE             0x1FFF7800

#ifndef __ASSEMBLER__

/* Register definitions */

/* --- USART --- */
#define STM32_USART_CR1(base)       STM32_USART_REG(base, 0x00)
#define STM32_USART_CR1_UE          BIT(0)
#define STM32_USART_CR1_UESM        BIT(1)
#define STM32_USART_CR1_RE          BIT(2)
#define STM32_USART_CR1_TE          BIT(3)
#define STM32_USART_CR1_RXNEIE      BIT(5)
#define STM32_USART_CR1_TCIE        BIT(6)
#define STM32_USART_CR1_TXEIE       BIT(7)
#define STM32_USART_CR1_PS          BIT(9)
#define STM32_USART_CR1_PCE         BIT(10)
#define STM32_USART_CR1_M           BIT(12)
#define STM32_USART_CR1_OVER8       BIT(15)

#define STM32_USART_CR2(base)       STM32_USART_REG(base, 0x04)
#define STM32_USART_CR2_SWAP        BIT(15)

#define STM32_USART_CR3(base)       STM32_USART_REG(base, 0x08)
#define STM32_USART_CR3_EIE         BIT(0)
#define STM32_USART_CR3_DMAR        BIT(6)
#define STM32_USART_CR3_DMAT        BIT(7)
#define STM32_USART_CR3_ONEBIT      BIT(11)
#define STM32_USART_CR3_OVRDIS      BIT(12)
#define STM32_USART_CR3_WUS_START_BIT (2 << 20)
#define STM32_USART_CR3_WUFIE       BIT(22)

#define STM32_USART_BRR(base)       STM32_USART_REG(base, 0x0C)
#define STM32_USART_GTPR(base)      STM32_USART_REG(base, 0x10)
#define STM32_USART_RTOR(base)      STM32_USART_REG(base, 0x14)
#define STM32_USART_RQR(base)       STM32_USART_REG(base, 0x18)
#define STM32_USART_ISR(base)       STM32_USART_REG(base, 0x1C)
#define STM32_USART_ICR(base)       STM32_USART_REG(base, 0x20)
#define STM32_USART_ICR_ORECF       BIT(3)
#define STM32_USART_ICR_TCCF        BIT(6)
#define STM32_USART_RDR(base)       STM32_USART_REG(base, 0x24)
#define STM32_USART_TDR(base)       STM32_USART_REG(base, 0x28)
#define STM32_USART_PRESC(base)     STM32_USART_REG(base, 0x2C)
/* register alias */
#define STM32_USART_SR(base)        STM32_USART_ISR(base)
#define STM32_USART_SR_ORE          BIT(3)
#define STM32_USART_SR_RXNE         BIT(5)
#define STM32_USART_SR_TC           BIT(6)
#define STM32_USART_SR_TXE          BIT(7)

/* --- GPIO --- */
#define STM32_GPIO_MODER(b)     REG32((b) + 0x00)
#define STM32_GPIO_OTYPER(b)    REG16((b) + 0x04)
#define STM32_GPIO_OSPEEDR(b)   REG32((b) + 0x08)
#define STM32_GPIO_PUPDR(b)     REG32((b) + 0x0C)
#define STM32_GPIO_IDR(b)       REG16((b) + 0x10)
#define STM32_GPIO_ODR(b)       REG16((b) + 0x14)
#define STM32_GPIO_BSRR(b)      REG32((b) + 0x18)
#define STM32_GPIO_LCKR(b)      REG32((b) + 0x1C)
#define STM32_GPIO_AFRL(b)      REG32((b) + 0x20)
#define STM32_GPIO_AFRH(b)      REG32((b) + 0x24)
#define STM32_GPIO_BRR(b)       REG32((b) + 0x28)
#define STM32_GPIO_ASCR(b)      REG32((b) + 0x2C) /* only for stm32l4x6 */

#define GPIO_ALT_F0		0x0
#define GPIO_ALT_F1		0x1
#define GPIO_ALT_F2		0x2
#define GPIO_ALT_F3		0x3
#define GPIO_ALT_F4		0x4
#define GPIO_ALT_F5		0x5
#define GPIO_ALT_F6		0x6
#define GPIO_ALT_F7		0x7
#define GPIO_ALT_F8		0x8
#define GPIO_ALT_F9		0x9
#define GPIO_ALT_FA		0xA
#define GPIO_ALT_FB		0xB
#define GPIO_ALT_FC		0xC
#define GPIO_ALT_FD		0xD
#define GPIO_ALT_FE		0xE
#define GPIO_ALT_FF		0xF

/* --- I2C --- */
#define STM32_I2C_CR1(n)            REG32(stm32_i2c_reg(n, 0x00))
#define STM32_I2C_CR1_PE            BIT(0)
#define STM32_I2C_CR1_TXIE          BIT(1)
#define STM32_I2C_CR1_RXIE          BIT(2)
#define STM32_I2C_CR1_ADDRIE        BIT(3)
#define STM32_I2C_CR1_NACKIE        BIT(4)
#define STM32_I2C_CR1_STOPIE        BIT(5)
#define STM32_I2C_CR1_ERRIE         BIT(7)
#define STM32_I2C_CR1_WUPEN         BIT(18)
#define STM32_I2C_CR2(n)            REG32(stm32_i2c_reg(n, 0x04))
#define STM32_I2C_CR2_RD_WRN        BIT(10)
#define STM32_I2C_CR2_START         BIT(13)
#define STM32_I2C_CR2_STOP          BIT(14)
#define STM32_I2C_CR2_NACK          BIT(15)
#define STM32_I2C_CR2_RELOAD        BIT(24)
#define STM32_I2C_CR2_AUTOEND       BIT(25)
#define STM32_I2C_OAR1(n)           REG32(stm32_i2c_reg(n, 0x08))
#define STM32_I2C_OAR2(n)           REG32(stm32_i2c_reg(n, 0x0C))
#define STM32_I2C_TIMINGR(n)        REG32(stm32_i2c_reg(n, 0x10))
#define STM32_I2C_TIMEOUTR(n)       REG32(stm32_i2c_reg(n, 0x14))
#define STM32_I2C_ISR(n)            REG32(stm32_i2c_reg(n, 0x18))
#define STM32_I2C_ISR_TXE           BIT(0)
#define STM32_I2C_ISR_TXIS          BIT(1)
#define STM32_I2C_ISR_RXNE          BIT(2)
#define STM32_I2C_ISR_ADDR          BIT(3)
#define STM32_I2C_ISR_NACK          BIT(4)
#define STM32_I2C_ISR_STOP          BIT(5)
#define STM32_I2C_ISR_TC            BIT(6)
#define STM32_I2C_ISR_TCR           BIT(7)
#define STM32_I2C_ISR_BERR          BIT(8)
#define STM32_I2C_ISR_ARLO          BIT(9)
#define STM32_I2C_ISR_OVR           BIT(10)
#define STM32_I2C_ISR_PECERR        BIT(11)
#define STM32_I2C_ISR_TIMEOUT       BIT(12)
#define STM32_I2C_ISR_ALERT         BIT(13)
#define STM32_I2C_ISR_BUSY          BIT(15)
#define STM32_I2C_ISR_DIR           BIT(16)
#define STM32_I2C_ISR_ADDCODE(isr)  (((isr) >> 16) & 0xfe)
#define STM32_I2C_ICR(n)            REG32(stm32_i2c_reg(n, 0x1C))
#define STM32_I2C_ICR_ADDRCF        BIT(3)
#define STM32_I2C_ICR_NACKCF        BIT(4)
#define STM32_I2C_ICR_STOPCF        BIT(5)
#define STM32_I2C_ICR_BERRCF        BIT(8)
#define STM32_I2C_ICR_ARLOCF        BIT(9)
#define STM32_I2C_ICR_OVRCF         BIT(10)
#define STM32_I2C_ICR_TIMEOUTCF     BIT(12)
#define STM32_I2C_ICR_ALL           0x3F38
#define STM32_I2C_PECR(n)           REG32(stm32_i2c_reg(n, 0x20))
#define STM32_I2C_RXDR(n)           REG32(stm32_i2c_reg(n, 0x24))
#define STM32_I2C_TXDR(n)           REG32(stm32_i2c_reg(n, 0x28))

/* --- Power / Reset / Clocks --- */
#define STM32_PWR_CR1               REG32(STM32_PWR_BASE + 0x00)
#define STM32_PWR_CR2               REG32(STM32_PWR_BASE + 0x04)
#define STM32_PWR_CSR               REG32(STM32_PWR_BASE + 0x10)
#define STM32_PWR_SCR               REG32(STM32_PWR_BASE + 0x18)

#define PWR_CR1_LPMS_POS             0U
#define PWR_CR1_LPMS_MSK             (0x7UL << PWR_CR1_LPMS_POS)
#define PWR_CR1_LPMS                 PWR_CR1_LPMS_MSK
#define PWR_CR1_LPMS_STOP0           (0x00000000UL)
#define PWR_CR1_LPMS_STOP1_POS       0U
#define PWR_CR1_LPMS_STOP1_MSK       (0x1UL << PWR_CR1_LPMS_STOP1_POS)
#define PWR_CR1_LPMS_STOP1           PWR_CR1_LPMS_STOP1_MSK
#define PWR_CR1_LPMS_STOP2_POS       1U
#define PWR_CR1_LPMS_STOP2_MSK       (0x1UL << PWR_CR1_LPMS_STOP2_POS)
#define PWR_CR1_LPMS_STOP2           PWR_CR1_LPMS_STOP2_MSK
#define PWR_CR1_LPMS_STANDBY_POS     0U
#define PWR_CR1_LPMS_STANDBY_MSK     (0x3UL << PWR_CR1_LPMS_STANDBY_POS)
#define PWR_CR1_LPMS_STANDBY         PWR_CR1_LPMS_STANDBY_MSK
#define PWR_CR1_LPMS_SHUTDOWN_POS    2U
#define PWR_CR1_LPMS_SHUTDOWN_MSK    (0x1UL << PWR_CR1_LPMS_SHUTDOWN_POS)
#define PWR_CR1_LPMS_SHUTDOWN        PWR_CR1_LPMS_SHUTDOWN_MSK
#define PWR_CR1_VOS_POS              9U
#define PWR_CR1_VOS_MSK              (0x3UL << PWR_CR1_VOS_POS)
#define PWR_CR1_VOS                  PWR_CR1_VOS_MSK
#define PWR_CR1_VOS_0                (0x1UL << PWR_CR1_VOS_POS)
#define PWR_CR1_VOS_1                (0x2UL << PWR_CR1_VOS_POS)


/* --- Macro usage in ec code --- */
#define STM32_RCC_AHB2ENR_GPIOMASK                               \
	(STM32_RCC_AHB2ENR_GPIOAEN | STM32_RCC_AHB2ENR_GPIOBEN | \
	 STM32_RCC_AHB2ENR_GPIOCEN | STM32_RCC_AHB2ENR_GPIODEN | \
	 STM32_RCC_AHB2ENR_GPIOEEN | STM32_RCC_AHB2ENR_GPIOHEN)
#define STM32_RCC_ICSCR_MSIRANGE(n) ((n) << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_ICSCR_MSIRANGE_1MHZ STM32_RCC_ICSCR_MSIRANGE(4)
#define STM32_RCC_ICSCR_MSIRANGE_2MHZ STM32_RCC_ICSCR_MSIRANGE(5)
#define STM32_RCC_ICSCR_MSIRANGE_MASK STM32_RCC_CR_MSIRANGE_MSK
#define STM32_RCC_SYSCFGEN            STM32_RCC_APB2ENR_SYSCFGEN

#define STM32_RCC_PB2_TIM1            STM32_RCC_APB2ENR_TIM1EN
#define STM32_RCC_PB2_TIM15           STM32_RCC_APB2ENR_TIM15EN
#define STM32_RCC_PB2_TIM16           STM32_RCC_APB2ENR_TIM16EN
#ifndef CHIP_VARIANT_STM32L431X
#define STM32_RCC_PB2_TIM8            BIT(13)
#endif
#define STM32_RCC_PWREN STM32_RCC_APB1ENR1_PWREN

#define STM32_RCC_AHB2ENR_GPIO_PORTA	BIT(0)
#define STM32_RCC_AHB2ENR_GPIO_PORTB	BIT(1)
#define STM32_RCC_AHB2ENR_GPIO_PORTC	BIT(2)
#define STM32_RCC_AHB2ENR_GPIO_PORTD	BIT(3)
#define STM32_RCC_AHB2ENR_GPIO_PORTE	BIT(4)
#define STM32_RCC_AHB2ENR_GPIO_PORTH	BIT(7)
#define STM32_RCC_CCIPR_USART1SEL_SHIFT (0)
#define STM32_RCC_CCIPR_USART1SEL_MASK  (3 << STM32_RCC_CCIPR_USART1SEL_SHIFT)
#define STM32_RCC_CCIPR_USART2SEL_SHIFT (2)
#define STM32_RCC_CCIPR_USART2SEL_MASK  (3 << STM32_RCC_CCIPR_USART2SEL_SHIFT)
#define STM32_RCC_CCIPR_USART3SEL_SHIFT (4)
#define STM32_RCC_CCIPR_USART3SEL_MASK  (3 << STM32_RCC_CCIPR_USART3SEL_SHIFT)
#define STM32_RCC_CCIPR_UART4SEL_SHIFT (6)
#define STM32_RCC_CCIPR_UART4SEL_MASK  (3 << STM32_RCC_CCIPR_UART4SEL_SHIFT)
#define STM32_RCC_CCIPR_UART5SEL_SHIFT (8)
#define STM32_RCC_CCIPR_UART5SEL_MASK  (3 << STM32_RCC_CCIPR_UART5SEL_SHIFT)
#define STM32_RCC_CCIPR_LPUART1SEL_SHIFT (10)
#define STM32_RCC_CCIPR_LPUART1SEL_MASK  (3 << STM32_RCC_CCIPR_LPUART1SEL_SHIFT)
#define STM32_RCC_CCIPR_I2C1SEL_SHIFT (12)
#define STM32_RCC_CCIPR_I2C1SEL_MASK  (3 << STM32_RCC_CCIPR_I2C1SEL_SHIFT)
#define STM32_RCC_CCIPR_I2C2SEL_SHIFT (14)
#define STM32_RCC_CCIPR_I2C2SEL_MASK  (3 << STM32_RCC_CCIPR_I2C2SEL_SHIFT)
#define STM32_RCC_CCIPR_I2C3SEL_SHIFT (16)
#define STM32_RCC_CCIPR_I2C3SEL_MASK  (3 << STM32_RCC_CCIPR_I2C3SEL_SHIFT)
#define STM32_RCC_CCIPR_LPTIM1SEL_SHIFT (18)
#define STM32_RCC_CCIPR_LPTIM1SEL_MASK  (3 << STM32_RCC_CCIPR_LPTIM1SEL_SHIFT)
#define STM32_RCC_CCIPR_LPTIM2SEL_SHIFT (20)
#define STM32_RCC_CCIPR_LPTIM2SEL_MASK  (3 << STM32_RCC_CCIPR_LPTIM2SEL_SHIFT)
#define STM32_RCC_CCIPR_SAI1SEL_SHIFT (22)
#define STM32_RCC_CCIPR_SAI1SEL_MASK  (3 << STM32_RCC_CCIPR_SAI1SEL_SHIFT)
#define STM32_RCC_CCIPR_SAI2SEL_SHIFT (24)
#define STM32_RCC_CCIPR_SAI2SEL_MASK  (3 << STM32_RCC_CCIPR_SAI2SEL_SHIFT)
#define STM32_RCC_CCIPR_CLK48SEL_SHIFT (26)
#define STM32_RCC_CCIPR_CLK48SEL_MASK  (3 << STM32_RCC_CCIPR_CLK48SEL_SHIFT)
#define STM32_RCC_CCIPR_ADCSEL_SHIFT (28)
#define STM32_RCC_CCIPR_ADCSEL_MASK  (3 << STM32_RCC_CCIPR_ADCSEL_SHIFT)
#define STM32_RCC_CCIPR_SWPMI1SEL_SHIFT (30)
#define STM32_RCC_CCIPR_SWPMI1SEL_MASK  BIT(STM32_RCC_CCIPR_SWPMI1SEL_SHIFT)
#define STM32_RCC_CCIPR_DFSDM1SEL_SHIFT (31)
#define STM32_RCC_CCIPR_DFSDM1SEL_MASK  BIT(STM32_RCC_CCIPR_DFSDM1SEL_SHIFT)
/* Possible clock sources for each peripheral */
#define STM32_RCC_CCIPR_UART_PCLK	0
#define STM32_RCC_CCIPR_UART_SYSCLK	1
#define STM32_RCC_CCIPR_UART_HSI16	2
#define STM32_RCC_CCIPR_UART_LSE	3

#define STM32_RCC_CCIPR_I2C_PCLK	0
#define STM32_RCC_CCIPR_I2C_SYSCLK	1
#define STM32_RCC_CCIPR_I2C_HSI16	2

#define STM32_RCC_CCIPR_LPTIM_PCLK	0
#define STM32_RCC_CCIPR_LPTIM_LSI	1
#define STM32_RCC_CCIPR_LPTIM_HSI16	2
#define STM32_RCC_CCIPR_LPTIM_LSE	3

#define STM32_RCC_CCIPR_SAI_PLLSAI1CLK	0
#define STM32_RCC_CCIPR_SAI_PLLSAI2CLK	1
#define STM32_RCC_CCIPR_SAI_PLLSAI3CLK	2
#define STM32_RCC_CCIPR_SAI_EXTCLK		3

#define STM32_RCC_CCIPR_CLK48_NONE			0
#define STM32_RCC_CCIPR_CLK48_PLL48M2CLK	1
#define STM32_RCC_CCIPR_CLK48_PLL48M1CLK	2
#define STM32_RCC_CCIPR_CLK48_MSI			3

#define STM32_RCC_CCIPR_ADC_NONE		0
#define STM32_RCC_CCIPR_ADC_PLLADC1CLK	1
#define STM32_RCC_CCIPR_ADC_PLLADC2CLK	2
#define STM32_RCC_CCIPR_ADC_SYSCLK	3

#define STM32_RCC_CCIPR_SWPMI_PCLK	0
#define STM32_RCC_CCIPR_SWPMI_HSI16	1

#define STM32_RCC_CCIPR_DFSDM_PCLK		0
#define STM32_RCC_CCIPR_DFSDM_SYSCLK	1



#define STM32_RCC_CR          REG32(STM32_RCC_BASE + 0x00)
#define STM32_RCC_ICSCR       REG32(STM32_RCC_BASE + 0x04)
#define STM32_RCC_CFGR        REG32(STM32_RCC_BASE + 0x08)
#define STM32_RCC_PLLCFGR     REG32(STM32_RCC_BASE + 0x0C)
#define STM32_RCC_PLLSAI1CFGR REG32(STM32_RCC_BASE + 0x10)
#define STM32_RCC_RESERVED    REG32(STM32_RCC_BASE + 0x14)
#define STM32_RCC_CIER        REG32(STM32_RCC_BASE + 0x18)
#define STM32_RCC_CIFR        REG32(STM32_RCC_BASE + 0x1C)
#define STM32_RCC_CICR        REG32(STM32_RCC_BASE + 0x20)
#define STM32_RCC_RESERVED0   REG32(STM32_RCC_BASE + 0x24)
#define STM32_RCC_AHB1RSTR    REG32(STM32_RCC_BASE + 0x28)
#define STM32_RCC_AHB2RSTR    REG32(STM32_RCC_BASE + 0x2C)
#define STM32_RCC_AHB3RSTR    REG32(STM32_RCC_BASE + 0x30)
#define STM32_RCC_RESERVED1   REG32(STM32_RCC_BASE + 0x34)
#define STM32_RCC_APB1RSTR1   REG32(STM32_RCC_BASE + 0x38)
#define STM32_RCC_APB1RSTR2   REG32(STM32_RCC_BASE + 0x3C)
#define STM32_RCC_APB2RSTR    REG32(STM32_RCC_BASE + 0x40)
#define STM32_RCC_RESERVED2   REG32(STM32_RCC_BASE + 0x44)
#define STM32_RCC_AHB1ENR     REG32(STM32_RCC_BASE + 0x48)
#define STM32_RCC_AHB2ENR     REG32(STM32_RCC_BASE + 0x4C)
#define STM32_RCC_AHB3ENR     REG32(STM32_RCC_BASE + 0x50)
#define STM32_RCC_RESERVED3   REG32(STM32_RCC_BASE + 0x54)
#define STM32_RCC_APB1ENR1    REG32(STM32_RCC_BASE + 0x58)
#define STM32_RCC_APB1ENR2    REG32(STM32_RCC_BASE + 0x5C)
#define STM32_RCC_APB2ENR     REG32(STM32_RCC_BASE + 0x60)
#define STM32_RCC_RESERVED4   REG32(STM32_RCC_BASE + 0x64)
#define STM32_RCC_AHB1SMENR   REG32(STM32_RCC_BASE + 0x68)
#define STM32_RCC_AHB2SMENR   REG32(STM32_RCC_BASE + 0x6C)
#define STM32_RCC_AHB3SMENR   REG32(STM32_RCC_BASE + 0x70)
#define STM32_RCC_RESERVED5   REG32(STM32_RCC_BASE + 0x74)
#define STM32_RCC_APB1SMENR1  REG32(STM32_RCC_BASE + 0x78)
#define STM32_RCC_APB1SMENR2  REG32(STM32_RCC_BASE + 0x7C)
#define STM32_RCC_APB2SMENR   REG32(STM32_RCC_BASE + 0x80)
#define STM32_RCC_RESERVED6   REG32(STM32_RCC_BASE + 0x84)
#define STM32_RCC_CCIPR       REG32(STM32_RCC_BASE + 0x88)
#define STM32_RCC_RESERVED7   REG32(STM32_RCC_BASE + 0x8C)
#define STM32_RCC_BDCR        REG32(STM32_RCC_BASE + 0x90)
#define STM32_RCC_CSR         REG32(STM32_RCC_BASE + 0x94)
#define STM32_RCC_CRRCR       REG32(STM32_RCC_BASE + 0x98)

#define STM32_RCC_PLLSAI1_SUPPORT
#define STM32_RCC_PLLP_SUPPORT
#define STM32_RCC_HSI48_SUPPORT
#define STM32_RCC_PLLP_DIV_2_31_SUPPORT
#define STM32_RCC_PLLSAI1P_DIV_2_31_SUPPORT

#define STM32_RCC_APB1ENR                STM32_RCC_APB1ENR1

/********************  BIT DEFINITION FOR STM32_RCC_CR REGISTER  **************/
#define STM32_RCC_CR_MSION_POS           0U
#define STM32_RCC_CR_MSION_MSK           (0x1UL << STM32_RCC_CR_MSION_POS)
#define STM32_RCC_CR_MSION               STM32_RCC_CR_MSION_MSK
#define STM32_RCC_CR_MSIRDY_POS          1U
#define STM32_RCC_CR_MSIRDY_MSK          (0x1UL << STM32_RCC_CR_MSIRDY_POS)
#define STM32_RCC_CR_MSIRDY              STM32_RCC_CR_MSIRDY_MSK
#define STM32_RCC_CR_MSIPLLEN_POS        2U
#define STM32_RCC_CR_MSIPLLEN_MSK        (0x1UL << STM32_RCC_CR_MSIPLLEN_POS)
#define STM32_RCC_CR_MSIPLLEN            STM32_RCC_CR_MSIPLLEN_MSK
#define STM32_RCC_CR_MSIRGSEL_POS        3U
#define STM32_RCC_CR_MSIRGSEL_MSK        (0x1UL << STM32_RCC_CR_MSIRGSEL_POS)
#define STM32_RCC_CR_MSIRGSEL            STM32_RCC_CR_MSIRGSEL_MSK

/*!< MSIRANGE CONFIGURATION : 12 FREQUENCY RANGES AVAILABLE */
#define STM32_RCC_CR_MSIRANGE_POS        4U
#define STM32_RCC_CR_MSIRANGE_MSK        (0xFUL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE            STM32_RCC_CR_MSIRANGE_MSK
#define STM32_RCC_CR_MSIRANGE_0          (0x0UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_1          (0x1UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_2          (0x2UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_3          (0x3UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_4          (0x4UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_5          (0x5UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_6          (0x6UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_7          (0x7UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_8          (0x8UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_9          (0x9UL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_10         (0xAUL << STM32_RCC_CR_MSIRANGE_POS)
#define STM32_RCC_CR_MSIRANGE_11         (0xBUL << STM32_RCC_CR_MSIRANGE_POS)

#define STM32_RCC_CR_HSION_POS           8U
#define STM32_RCC_CR_HSION_MSK           (0x1UL << STM32_RCC_CR_HSION_POS)
#define STM32_RCC_CR_HSION               STM32_RCC_CR_HSION_MSK
#define STM32_RCC_CR_HSIKERON_POS        9U
#define STM32_RCC_CR_HSIKERON_MSK        (0x1UL << STM32_RCC_CR_HSIKERON_POS)
#define STM32_RCC_CR_HSIKERON            STM32_RCC_CR_HSIKERON_MSK
#define STM32_RCC_CR_HSIRDY_POS          10U
#define STM32_RCC_CR_HSIRDY_MSK          (0x1UL << STM32_RCC_CR_HSIRDY_POS)
#define STM32_RCC_CR_HSIRDY              STM32_RCC_CR_HSIRDY_MSK
#define STM32_RCC_CR_HSIASFS_POS         11U
#define STM32_RCC_CR_HSIASFS_MSK         (0x1UL << STM32_RCC_CR_HSIASFS_POS)
#define STM32_RCC_CR_HSIASFS             STM32_RCC_CR_HSIASFS_MSK

#define STM32_RCC_CR_HSEON_POS           16U
#define STM32_RCC_CR_HSEON_MSK           (0x1UL << STM32_RCC_CR_HSEON_POS)
#define STM32_RCC_CR_HSEON               STM32_RCC_CR_HSEON_MSK
#define STM32_RCC_CR_HSERDY_POS          17U
#define STM32_RCC_CR_HSERDY_MSK          (0x1UL << STM32_RCC_CR_HSERDY_POS)
#define STM32_RCC_CR_HSERDY              STM32_RCC_CR_HSERDY_MSK
#define STM32_RCC_CR_HSEBYP_POS          18U
#define STM32_RCC_CR_HSEBYP_MSK          (0x1UL << STM32_RCC_CR_HSEBYP_POS)
#define STM32_RCC_CR_HSEBYP              STM32_RCC_CR_HSEBYP_MSK
#define STM32_RCC_CR_CSSON_POS           19U
#define STM32_RCC_CR_CSSON_MSK           (0x1UL << STM32_RCC_CR_CSSON_POS)
#define STM32_RCC_CR_CSSON               STM32_RCC_CR_CSSON_MSK

#define STM32_RCC_CR_PLLON_POS           24U
#define STM32_RCC_CR_PLLON_MSK           (0x1UL << STM32_RCC_CR_PLLON_POS)
#define STM32_RCC_CR_PLLON               STM32_RCC_CR_PLLON_MSK
#define STM32_RCC_CR_PLLRDY_POS          25U
#define STM32_RCC_CR_PLLRDY_MSK          (0x1UL << STM32_RCC_CR_PLLRDY_POS)
#define STM32_RCC_CR_PLLRDY              STM32_RCC_CR_PLLRDY_MSK
#define STM32_RCC_CR_PLLSAI1ON_POS       26U
#define STM32_RCC_CR_PLLSAI1ON_MSK       (0x1UL << STM32_RCC_CR_PLLSAI1ON_POS)
#define STM32_RCC_CR_PLLSAI1ON           STM32_RCC_CR_PLLSAI1ON_MSK
#define STM32_RCC_CR_PLLSAI1RDY_POS      27U
#define STM32_RCC_CR_PLLSAI1RDY_MSK      (0x1UL << STM32_RCC_CR_PLLSAI1RDY_POS)
#define STM32_RCC_CR_PLLSAI1RDY          STM32_RCC_CR_PLLSAI1RDY_MSK

/********************  BIT DEFINITION FOR STM32_RCC_ICSCR REGISTER  ***********/
/*!< MSICAL CONFIGURATION */
#define STM32_RCC_ICSCR_MSICAL_POS       0U
#define STM32_RCC_ICSCR_MSICAL_MSK       (0xFFUL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL           STM32_RCC_ICSCR_MSICAL_MSK
#define STM32_RCC_ICSCR_MSICAL_0         (0x01UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_1         (0x02UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_2         (0x04UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_3         (0x08UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_4         (0x10UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_5         (0x20UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_6         (0x40UL << STM32_RCC_ICSCR_MSICAL_POS)
#define STM32_RCC_ICSCR_MSICAL_7         (0x80UL << STM32_RCC_ICSCR_MSICAL_POS)

/*!< MSITRIM CONFIGURATION */
#define STM32_RCC_ICSCR_MSITRIM_POS      8U
#define STM32_RCC_ICSCR_MSITRIM_MSK      (0xFFUL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM          STM32_RCC_ICSCR_MSITRIM_MSK
#define STM32_RCC_ICSCR_MSITRIM_0        (0x01UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_1        (0x02UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_2        (0x04UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_3        (0x08UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_4        (0x10UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_5        (0x20UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_6        (0x40UL << STM32_RCC_ICSCR_MSITRIM_POS)
#define STM32_RCC_ICSCR_MSITRIM_7        (0x80UL << STM32_RCC_ICSCR_MSITRIM_POS)

/*!< HSICAL CONFIGURATION */
#define STM32_RCC_ICSCR_HSICAL_POS       16U
#define STM32_RCC_ICSCR_HSICAL_MSK       (0xFFUL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL           STM32_RCC_ICSCR_HSICAL_MSK
#define STM32_RCC_ICSCR_HSICAL_0         (0x01UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_1         (0x02UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_2         (0x04UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_3         (0x08UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_4         (0x10UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_5         (0x20UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_6         (0x40UL << STM32_RCC_ICSCR_HSICAL_POS)
#define STM32_RCC_ICSCR_HSICAL_7         (0x80UL << STM32_RCC_ICSCR_HSICAL_POS)

/*!< HSITRIM CONFIGURATION */
#define STM32_RCC_ICSCR_HSITRIM_POS      24U
#define STM32_RCC_ICSCR_HSITRIM_MSK      (0x1FUL << STM32_RCC_ICSCR_HSITRIM_POS)
#define STM32_RCC_ICSCR_HSITRIM          STM32_RCC_ICSCR_HSITRIM_MSK
#define STM32_RCC_ICSCR_HSITRIM_0        (0x01UL << STM32_RCC_ICSCR_HSITRIM_POS)
#define STM32_RCC_ICSCR_HSITRIM_1        (0x02UL << STM32_RCC_ICSCR_HSITRIM_POS)
#define STM32_RCC_ICSCR_HSITRIM_2        (0x04UL << STM32_RCC_ICSCR_HSITRIM_POS)
#define STM32_RCC_ICSCR_HSITRIM_3        (0x08UL << STM32_RCC_ICSCR_HSITRIM_POS)
#define STM32_RCC_ICSCR_HSITRIM_4        (0x10UL << STM32_RCC_ICSCR_HSITRIM_POS)

/****************  BIT DEFINITION FOR STM32_RCC_CFGR REGISTER  **************/
/*!< SW CONFIGURATION */
#define STM32_RCC_CFGR_SW_POS            0U
#define STM32_RCC_CFGR_SW_MSK            (0x3UL << STM32_RCC_CFGR_SW_POS)
#define STM32_RCC_CFGR_SW                STM32_RCC_CFGR_SW_MSK
#define STM32_RCC_CFGR_SW_0              (0x1UL << STM32_RCC_CFGR_SW_POS)
#define STM32_RCC_CFGR_SW_1              (0x2UL << STM32_RCC_CFGR_SW_POS)

#define STM32_RCC_CFGR_SW_MSI            (0x00000000UL)
#define STM32_RCC_CFGR_SW_HSI            (0x00000001UL)
#define STM32_RCC_CFGR_SW_HSE            (0x00000002UL)
#define STM32_RCC_CFGR_SW_PLL            (0x00000003UL)

/*!< SWS CONFIGURATION */
#define STM32_RCC_CFGR_SWS_POS           2U
#define STM32_RCC_CFGR_SWS_MSK           (0x3UL << STM32_RCC_CFGR_SWS_POS)
#define STM32_RCC_CFGR_SWS               STM32_RCC_CFGR_SWS_MSK
#define STM32_RCC_CFGR_SWS_0             (0x1UL << STM32_RCC_CFGR_SWS_POS)
#define STM32_RCC_CFGR_SWS_1             (0x2UL << STM32_RCC_CFGR_SWS_POS)

#define STM32_RCC_CFGR_SWS_MSI           (0x00000000UL)
#define STM32_RCC_CFGR_SWS_HSI           (0x00000004UL)
#define STM32_RCC_CFGR_SWS_HSE           (0x00000008UL)
#define STM32_RCC_CFGR_SWS_PLL           (0x0000000CUL)

/*!< HPRE CONFIGURATION */
#define STM32_RCC_CFGR_HPRE_POS          4U
#define STM32_RCC_CFGR_HPRE_MSK          (0xFUL << STM32_RCC_CFGR_HPRE_POS)
#define STM32_RCC_CFGR_HPRE              STM32_RCC_CFGR_HPRE_MSK
#define STM32_RCC_CFGR_HPRE_0            (0x1UL << STM32_RCC_CFGR_HPRE_POS)
#define STM32_RCC_CFGR_HPRE_1            (0x2UL << STM32_RCC_CFGR_HPRE_POS)
#define STM32_RCC_CFGR_HPRE_2            (0x4UL << STM32_RCC_CFGR_HPRE_POS)
#define STM32_RCC_CFGR_HPRE_3            (0x8UL << STM32_RCC_CFGR_HPRE_POS)

#define STM32_RCC_CFGR_HPRE_DIV1         (0x00000000UL)
#define STM32_RCC_CFGR_HPRE_DIV2         (0x00000080UL)
#define STM32_RCC_CFGR_HPRE_DIV4         (0x00000090UL)
#define STM32_RCC_CFGR_HPRE_DIV8         (0x000000A0UL)
#define STM32_RCC_CFGR_HPRE_DIV16        (0x000000B0UL)
#define STM32_RCC_CFGR_HPRE_DIV64        (0x000000C0UL)
#define STM32_RCC_CFGR_HPRE_DIV128       (0x000000D0UL)
#define STM32_RCC_CFGR_HPRE_DIV256       (0x000000E0UL)
#define STM32_RCC_CFGR_HPRE_DIV512       (0x000000F0UL)

/*!< PPRE1 CONFIGURATION */
#define STM32_RCC_CFGR_PPRE1_POS         8U
#define STM32_RCC_CFGR_PPRE1_MSK         (0x7UL << STM32_RCC_CFGR_PPRE1_POS)
#define STM32_RCC_CFGR_PPRE1             STM32_RCC_CFGR_PPRE1_MSK
#define STM32_RCC_CFGR_PPRE1_0           (0x1UL << STM32_RCC_CFGR_PPRE1_POS)
#define STM32_RCC_CFGR_PPRE1_1           (0x2UL << STM32_RCC_CFGR_PPRE1_POS)
#define STM32_RCC_CFGR_PPRE1_2           (0x4UL << STM32_RCC_CFGR_PPRE1_POS)

#define STM32_RCC_CFGR_PPRE1_DIV1        (0x00000000UL)
#define STM32_RCC_CFGR_PPRE1_DIV2        (0x00000400UL)
#define STM32_RCC_CFGR_PPRE1_DIV4        (0x00000500UL)
#define STM32_RCC_CFGR_PPRE1_DIV8        (0x00000600UL)
#define STM32_RCC_CFGR_PPRE1_DIV16       (0x00000700UL)

/*!< PPRE2 CONFIGURATION */
#define STM32_RCC_CFGR_PPRE2_POS         11U
#define STM32_RCC_CFGR_PPRE2_MSK         (0x7UL << STM32_RCC_CFGR_PPRE2_POS)
#define STM32_RCC_CFGR_PPRE2             STM32_RCC_CFGR_PPRE2_MSK
#define STM32_RCC_CFGR_PPRE2_0           (0x1UL << STM32_RCC_CFGR_PPRE2_POS)
#define STM32_RCC_CFGR_PPRE2_1           (0x2UL << STM32_RCC_CFGR_PPRE2_POS)
#define STM32_RCC_CFGR_PPRE2_2           (0x4UL << STM32_RCC_CFGR_PPRE2_POS)

#define STM32_RCC_CFGR_PPRE2_DIV1        (0x00000000UL)
#define STM32_RCC_CFGR_PPRE2_DIV2        (0x00002000UL)
#define STM32_RCC_CFGR_PPRE2_DIV4        (0x00002800UL)
#define STM32_RCC_CFGR_PPRE2_DIV8        (0x00003000UL)
#define STM32_RCC_CFGR_PPRE2_DIV16       (0x00003800UL)

#define STM32_RCC_CFGR_STOPWUCK_POS      15U
#define STM32_RCC_CFGR_STOPWUCK_MSK      (0x1UL << STM32_RCC_CFGR_STOPWUCK_POS)
#define STM32_RCC_CFGR_STOPWUCK          STM32_RCC_CFGR_STOPWUCK_MSK

/*!< MCOSEL CONFIGURATION */
#define STM32_RCC_CFGR_MCOSEL_POS        24U
#define STM32_RCC_CFGR_MCOSEL_MSK        (0xFUL << STM32_RCC_CFGR_MCOSEL_POS)
#define STM32_RCC_CFGR_MCOSEL            STM32_RCC_CFGR_MCOSEL_MSK
#define STM32_RCC_CFGR_MCOSEL_0          (0x1UL << STM32_RCC_CFGR_MCOSEL_POS)
#define STM32_RCC_CFGR_MCOSEL_1          (0x2UL << STM32_RCC_CFGR_MCOSEL_POS)
#define STM32_RCC_CFGR_MCOSEL_2          (0x4UL << STM32_RCC_CFGR_MCOSEL_POS)
#define STM32_RCC_CFGR_MCOSEL_3          (0x8UL << STM32_RCC_CFGR_MCOSEL_POS)

#define STM32_RCC_CFGR_MCOPRE_POS        28U
#define STM32_RCC_CFGR_MCOPRE_MSK        (0x7UL << STM32_RCC_CFGR_MCOPRE_POS)
#define STM32_RCC_CFGR_MCOPRE            STM32_RCC_CFGR_MCOPRE_MSK
#define STM32_RCC_CFGR_MCOPRE_0          (0x1UL << STM32_RCC_CFGR_MCOPRE_POS)
#define STM32_RCC_CFGR_MCOPRE_1          (0x2UL << STM32_RCC_CFGR_MCOPRE_POS)
#define STM32_RCC_CFGR_MCOPRE_2          (0x4UL << STM32_RCC_CFGR_MCOPRE_POS)

#define STM32_RCC_CFGR_MCOPRE_DIV1       (0x00000000UL)
#define STM32_RCC_CFGR_MCOPRE_DIV2       (0x10000000UL)
#define STM32_RCC_CFGR_MCOPRE_DIV4       (0x20000000UL)
#define STM32_RCC_CFGR_MCOPRE_DIV8       (0x30000000UL)
#define STM32_RCC_CFGR_MCOPRE_DIV16      (0x40000000UL)

/* LEGACY ALIASES */
#define STM32_RCC_CFGR_MCO_PRE           STM32_RCC_CFGR_MCOPRE
#define STM32_RCC_CFGR_MCO_PRE_1         STM32_RCC_CFGR_MCOPRE_DIV1
#define STM32_RCC_CFGR_MCO_PRE_2         STM32_RCC_CFGR_MCOPRE_DIV2
#define STM32_RCC_CFGR_MCO_PRE_4         STM32_RCC_CFGR_MCOPRE_DIV4
#define STM32_RCC_CFGR_MCO_PRE_8         STM32_RCC_CFGR_MCOPRE_DIV8
#define STM32_RCC_CFGR_MCO_PRE_16        STM32_RCC_CFGR_MCOPRE_DIV16

/****************  BIT DEFINITION FOR STM32_RCC_PLLCFGR REGISTER  ***********/
#define STM32_RCC_PLLCFGR_PLLSRC_POS     0U
#define STM32_RCC_PLLCFGR_PLLSRC_MSK (0x3UL << STM32_RCC_PLLCFGR_PLLSRC_POS)
#define STM32_RCC_PLLCFGR_PLLSRC STM32_RCC_PLLCFGR_PLLSRC_MSK

#define STM32_RCC_PLLCFGR_PLLSRC_MSI_POS 0U
#define STM32_RCC_PLLCFGR_PLLSRC_MSI_MSK \
	(0x1UL << STM32_RCC_PLLCFGR_PLLSRC_MSI_POS)
#define STM32_RCC_PLLCFGR_PLLSRC_MSI STM32_RCC_PLLCFGR_PLLSRC_MSI_MSK
#define STM32_RCC_PLLCFGR_PLLSRC_HSI_POS 1U
#define STM32_RCC_PLLCFGR_PLLSRC_HSI_MSK \
	(0x1UL << STM32_RCC_PLLCFGR_PLLSRC_HSI_POS)
#define STM32_RCC_PLLCFGR_PLLSRC_HSI STM32_RCC_PLLCFGR_PLLSRC_HSI_MSK
#define STM32_RCC_PLLCFGR_PLLSRC_HSE_POS 0U
#define STM32_RCC_PLLCFGR_PLLSRC_HSE_MSK \
	(0x3UL << STM32_RCC_PLLCFGR_PLLSRC_HSE_POS)
#define STM32_RCC_PLLCFGR_PLLSRC_HSE     STM32_RCC_PLLCFGR_PLLSRC_HSE_MSK

#define STM32_RCC_PLLCFGR_PLLM_POS       4U
#define STM32_RCC_PLLCFGR_PLLM_MSK       (0x7UL << STM32_RCC_PLLCFGR_PLLM_POS)
#define STM32_RCC_PLLCFGR_PLLM           STM32_RCC_PLLCFGR_PLLM_MSK
#define STM32_RCC_PLLCFGR_PLLM_0         (0x1UL << STM32_RCC_PLLCFGR_PLLM_POS)
#define STM32_RCC_PLLCFGR_PLLM_1         (0x2UL << STM32_RCC_PLLCFGR_PLLM_POS)
#define STM32_RCC_PLLCFGR_PLLM_2         (0x4UL << STM32_RCC_PLLCFGR_PLLM_POS)

#define STM32_RCC_PLLCFGR_PLLN_POS       8U
#define STM32_RCC_PLLCFGR_PLLN_MSK       (0x7FUL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN           STM32_RCC_PLLCFGR_PLLN_MSK
#define STM32_RCC_PLLCFGR_PLLN_0         (0x01UL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN_1         (0x02UL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN_2         (0x04UL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN_3         (0x08UL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN_4         (0x10UL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN_5         (0x20UL << STM32_RCC_PLLCFGR_PLLN_POS)
#define STM32_RCC_PLLCFGR_PLLN_6         (0x40UL << STM32_RCC_PLLCFGR_PLLN_POS)

#define STM32_RCC_PLLCFGR_PLLPEN_POS     16U
#define STM32_RCC_PLLCFGR_PLLPEN_MSK     (0x1UL << STM32_RCC_PLLCFGR_PLLPEN_POS)
#define STM32_RCC_PLLCFGR_PLLPEN         STM32_RCC_PLLCFGR_PLLPEN_MSK
#define STM32_RCC_PLLCFGR_PLLP_POS       17U
#define STM32_RCC_PLLCFGR_PLLP_MSK       (0x1UL << STM32_RCC_PLLCFGR_PLLP_POS)
#define STM32_RCC_PLLCFGR_PLLP           STM32_RCC_PLLCFGR_PLLP_MSK
#define STM32_RCC_PLLCFGR_PLLQEN_POS     20U
#define STM32_RCC_PLLCFGR_PLLQEN_MSK     (0x1UL << STM32_RCC_PLLCFGR_PLLQEN_POS)
#define STM32_RCC_PLLCFGR_PLLQEN         STM32_RCC_PLLCFGR_PLLQEN_MSK

#define STM32_RCC_PLLCFGR_PLLQ_POS       21U
#define STM32_RCC_PLLCFGR_PLLQ_MSK       (0x3UL << STM32_RCC_PLLCFGR_PLLQ_POS)
#define STM32_RCC_PLLCFGR_PLLQ           STM32_RCC_PLLCFGR_PLLQ_MSK
#define STM32_RCC_PLLCFGR_PLLQ_0         (0x1UL << STM32_RCC_PLLCFGR_PLLQ_POS)
#define STM32_RCC_PLLCFGR_PLLQ_1         (0x2UL << STM32_RCC_PLLCFGR_PLLQ_POS)

#define STM32_RCC_PLLCFGR_PLLREN_POS     24U
#define STM32_RCC_PLLCFGR_PLLREN_MSK     (0x1UL << STM32_RCC_PLLCFGR_PLLREN_POS)
#define STM32_RCC_PLLCFGR_PLLREN         STM32_RCC_PLLCFGR_PLLREN_MSK
#define STM32_RCC_PLLCFGR_PLLR_POS       25U
#define STM32_RCC_PLLCFGR_PLLR_MSK       (0x3UL << STM32_RCC_PLLCFGR_PLLR_POS)
#define STM32_RCC_PLLCFGR_PLLR           STM32_RCC_PLLCFGR_PLLR_MSK
#define STM32_RCC_PLLCFGR_PLLR_0         (0x1UL << STM32_RCC_PLLCFGR_PLLR_POS)
#define STM32_RCC_PLLCFGR_PLLR_1         (0x2UL << STM32_RCC_PLLCFGR_PLLR_POS)

#define STM32_RCC_PLLCFGR_PLLPDIV_POS  27U
#define STM32_RCC_PLLCFGR_PLLPDIV_MSK  (0x1FUL << STM32_RCC_PLLCFGR_PLLPDIV_POS)
#define STM32_RCC_PLLCFGR_PLLPDIV      STM32_RCC_PLLCFGR_PLLPDIV_MSK
#define STM32_RCC_PLLCFGR_PLLPDIV_0    (0x01UL << STM32_RCC_PLLCFGR_PLLPDIV_POS)
#define STM32_RCC_PLLCFGR_PLLPDIV_1    (0x02UL << STM32_RCC_PLLCFGR_PLLPDIV_POS)
#define STM32_RCC_PLLCFGR_PLLPDIV_2    (0x04UL << STM32_RCC_PLLCFGR_PLLPDIV_POS)
#define STM32_RCC_PLLCFGR_PLLPDIV_3    (0x08UL << STM32_RCC_PLLCFGR_PLLPDIV_POS)
#define STM32_RCC_PLLCFGR_PLLPDIV_4    (0x10UL << STM32_RCC_PLLCFGR_PLLPDIV_POS)

/****************  BIT DEFINITION FOR STM32_RCC_PLLSAI1CFGR REGISTER  ********/
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS 8U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_MSK \
	(0x7FUL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N STM32_RCC_PLLSAI1CFGR_PLLSAI1N_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_0 \
	(0x01UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_1 \
	(0x02UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_2 \
	(0x04UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_3 \
	(0x08UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_4 \
	(0x10UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_5 \
	(0x20UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1N_6 \
	(0x40UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1N_POS)

#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PEN_POS  16U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PEN_MSK \
	(0x1UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PEN_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PEN STM32_RCC_PLLSAI1CFGR_PLLSAI1PEN_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1P_POS 17U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1P_MSK \
	(0x1UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1P_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1P STM32_RCC_PLLSAI1CFGR_PLLSAI1P_MSK

#define STM32_RCC_PLLSAI1CFGR_PLLSAI1QEN_POS 20U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1QEN_MSK \
	(0x1UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1QEN_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1QEN STM32_RCC_PLLSAI1CFGR_PLLSAI1QEN_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_POS 21U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_MSK \
	(0x3UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1Q STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_0 \
	(0x1UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_1 \
	(0x2UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1Q_POS)

#define STM32_RCC_PLLSAI1CFGR_PLLSAI1REN_POS 24U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1REN_MSK \
	(0x1UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1REN_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1REN STM32_RCC_PLLSAI1CFGR_PLLSAI1REN_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1R_POS 25U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1R_MSK \
	(0x3UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1R_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1R STM32_RCC_PLLSAI1CFGR_PLLSAI1R_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1R_0 \
	(0x1UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1R_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1R_1 \
	(0x2UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1R_POS)

#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS 27U
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_MSK \
	(0x1FUL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_MSK
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_0 \
	(0x01UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_1 \
	(0x02UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_2 \
	(0x04UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_3 \
	(0x08UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS)
#define STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_4 \
	(0x10UL << STM32_RCC_PLLSAI1CFGR_PLLSAI1PDIV_POS)

/**************  BIT DEFINITION FOR STM32_RCC_CIER REGISTER  ******************/
#define STM32_RCC_CIER_LSIRDYIE_POS 0U
#define STM32_RCC_CIER_LSIRDYIE_MSK (0x1UL << STM32_RCC_CIER_LSIRDYIE_POS)
#define STM32_RCC_CIER_LSIRDYIE STM32_RCC_CIER_LSIRDYIE_MSK
#define STM32_RCC_CIER_LSERDYIE_POS 1U
#define STM32_RCC_CIER_LSERDYIE_MSK (0x1UL << STM32_RCC_CIER_LSERDYIE_POS)
#define STM32_RCC_CIER_LSERDYIE STM32_RCC_CIER_LSERDYIE_MSK
#define STM32_RCC_CIER_MSIRDYIE_POS 2U
#define STM32_RCC_CIER_MSIRDYIE_MSK (0x1UL << STM32_RCC_CIER_MSIRDYIE_POS)
#define STM32_RCC_CIER_MSIRDYIE STM32_RCC_CIER_MSIRDYIE_MSK
#define STM32_RCC_CIER_HSIRDYIE_POS 3U
#define STM32_RCC_CIER_HSIRDYIE_MSK (0x1UL << STM32_RCC_CIER_HSIRDYIE_POS)
#define STM32_RCC_CIER_HSIRDYIE STM32_RCC_CIER_HSIRDYIE_MSK
#define STM32_RCC_CIER_HSERDYIE_POS 4U
#define STM32_RCC_CIER_HSERDYIE_MSK (0x1UL << STM32_RCC_CIER_HSERDYIE_POS)
#define STM32_RCC_CIER_HSERDYIE STM32_RCC_CIER_HSERDYIE_MSK
#define STM32_RCC_CIER_PLLRDYIE_POS 5U
#define STM32_RCC_CIER_PLLRDYIE_MSK (0x1UL << STM32_RCC_CIER_PLLRDYIE_POS)
#define STM32_RCC_CIER_PLLRDYIE STM32_RCC_CIER_PLLRDYIE_MSK
#define STM32_RCC_CIER_PLLSAI1RDYIE_POS 6U
#define STM32_RCC_CIER_PLLSAI1RDYIE_MSK \
	(0x1UL << STM32_RCC_CIER_PLLSAI1RDYIE_POS)
#define STM32_RCC_CIER_PLLSAI1RDYIE STM32_RCC_CIER_PLLSAI1RDYIE_MSK
#define STM32_RCC_CIER_LSECSSIE_POS 9U
#define STM32_RCC_CIER_LSECSSIE_MSK (0x1UL << STM32_RCC_CIER_LSECSSIE_POS)
#define STM32_RCC_CIER_LSECSSIE STM32_RCC_CIER_LSECSSIE_MSK
#define STM32_RCC_CIER_HSI48RDYIE_POS 10U
#define STM32_RCC_CIER_HSI48RDYIE_MSK (0x1UL << STM32_RCC_CIER_HSI48RDYIE_POS)
#define STM32_RCC_CIER_HSI48RDYIE STM32_RCC_CIER_HSI48RDYIE_MSK

/**************  BIT DEFINITION FOR STM32_RCC_CIFR REGISTER  ******************/
#define STM32_RCC_CIFR_LSIRDYF_POS 0U
#define STM32_RCC_CIFR_LSIRDYF_MSK (0x1UL << STM32_RCC_CIFR_LSIRDYF_POS)
#define STM32_RCC_CIFR_LSIRDYF STM32_RCC_CIFR_LSIRDYF_MSK
#define STM32_RCC_CIFR_LSERDYF_POS 1U
#define STM32_RCC_CIFR_LSERDYF_MSK (0x1UL << STM32_RCC_CIFR_LSERDYF_POS)
#define STM32_RCC_CIFR_LSERDYF STM32_RCC_CIFR_LSERDYF_MSK
#define STM32_RCC_CIFR_MSIRDYF_POS 2U
#define STM32_RCC_CIFR_MSIRDYF_MSK (0x1UL << STM32_RCC_CIFR_MSIRDYF_POS)
#define STM32_RCC_CIFR_MSIRDYF STM32_RCC_CIFR_MSIRDYF_MSK
#define STM32_RCC_CIFR_HSIRDYF_POS 3U
#define STM32_RCC_CIFR_HSIRDYF_MSK (0x1UL << STM32_RCC_CIFR_HSIRDYF_POS)
#define STM32_RCC_CIFR_HSIRDYF STM32_RCC_CIFR_HSIRDYF_MSK
#define STM32_RCC_CIFR_HSERDYF_POS 4U
#define STM32_RCC_CIFR_HSERDYF_MSK (0x1UL << STM32_RCC_CIFR_HSERDYF_POS)
#define STM32_RCC_CIFR_HSERDYF STM32_RCC_CIFR_HSERDYF_MSK
#define STM32_RCC_CIFR_PLLRDYF_POS 5U
#define STM32_RCC_CIFR_PLLRDYF_MSK (0x1UL << STM32_RCC_CIFR_PLLRDYF_POS)
#define STM32_RCC_CIFR_PLLRDYF STM32_RCC_CIFR_PLLRDYF_MSK
#define STM32_RCC_CIFR_PLLSAI1RDYF_POS 6U
#define STM32_RCC_CIFR_PLLSAI1RDYF_MSK (0x1UL << STM32_RCC_CIFR_PLLSAI1RDYF_POS)
#define STM32_RCC_CIFR_PLLSAI1RDYF STM32_RCC_CIFR_PLLSAI1RDYF_MSK
#define STM32_RCC_CIFR_CSSF_POS 8U
#define STM32_RCC_CIFR_CSSF_MSK (0x1UL << STM32_RCC_CIFR_CSSF_POS)
#define STM32_RCC_CIFR_CSSF STM32_RCC_CIFR_CSSF_MSK
#define STM32_RCC_CIFR_LSECSSF_POS 9U
#define STM32_RCC_CIFR_LSECSSF_MSK (0x1UL << STM32_RCC_CIFR_LSECSSF_POS)
#define STM32_RCC_CIFR_LSECSSF STM32_RCC_CIFR_LSECSSF_MSK
#define STM32_RCC_CIFR_HSI48RDYF_POS 10U
#define STM32_RCC_CIFR_HSI48RDYF_MSK (0x1UL << STM32_RCC_CIFR_HSI48RDYF_POS)
#define STM32_RCC_CIFR_HSI48RDYF STM32_RCC_CIFR_HSI48RDYF_MSK

/**************  BIT DEFINITION FOR STM32_RCC_CICR REGISTER  ******************/
#define STM32_RCC_CICR_LSIRDYC_POS 0U
#define STM32_RCC_CICR_LSIRDYC_MSK (0x1UL << STM32_RCC_CICR_LSIRDYC_POS)
#define STM32_RCC_CICR_LSIRDYC STM32_RCC_CICR_LSIRDYC_MSK
#define STM32_RCC_CICR_LSERDYC_POS 1U
#define STM32_RCC_CICR_LSERDYC_MSK (0x1UL << STM32_RCC_CICR_LSERDYC_POS)
#define STM32_RCC_CICR_LSERDYC STM32_RCC_CICR_LSERDYC_MSK
#define STM32_RCC_CICR_MSIRDYC_POS 2U
#define STM32_RCC_CICR_MSIRDYC_MSK (0x1UL << STM32_RCC_CICR_MSIRDYC_POS)
#define STM32_RCC_CICR_MSIRDYC STM32_RCC_CICR_MSIRDYC_MSK
#define STM32_RCC_CICR_HSIRDYC_POS 3U
#define STM32_RCC_CICR_HSIRDYC_MSK (0x1UL << STM32_RCC_CICR_HSIRDYC_POS)
#define STM32_RCC_CICR_HSIRDYC STM32_RCC_CICR_HSIRDYC_MSK
#define STM32_RCC_CICR_HSERDYC_POS 4U
#define STM32_RCC_CICR_HSERDYC_MSK (0x1UL << STM32_RCC_CICR_HSERDYC_POS)
#define STM32_RCC_CICR_HSERDYC STM32_RCC_CICR_HSERDYC_MSK
#define STM32_RCC_CICR_PLLRDYC_POS 5U
#define STM32_RCC_CICR_PLLRDYC_MSK (0x1UL << STM32_RCC_CICR_PLLRDYC_POS)
#define STM32_RCC_CICR_PLLRDYC STM32_RCC_CICR_PLLRDYC_MSK
#define STM32_RCC_CICR_PLLSAI1RDYC_POS 6U
#define STM32_RCC_CICR_PLLSAI1RDYC_MSK (0x1UL << STM32_RCC_CICR_PLLSAI1RDYC_POS)
#define STM32_RCC_CICR_PLLSAI1RDYC STM32_RCC_CICR_PLLSAI1RDYC_MSK
#define STM32_RCC_CICR_CSSC_POS 8U
#define STM32_RCC_CICR_CSSC_MSK (0x1UL << STM32_RCC_CICR_CSSC_POS)
#define STM32_RCC_CICR_CSSC STM32_RCC_CICR_CSSC_MSK
#define STM32_RCC_CICR_LSECSSC_POS 9U
#define STM32_RCC_CICR_LSECSSC_MSK (0x1UL << STM32_RCC_CICR_LSECSSC_POS)
#define STM32_RCC_CICR_LSECSSC STM32_RCC_CICR_LSECSSC_MSK
#define STM32_RCC_CICR_HSI48RDYC_POS 10U
#define STM32_RCC_CICR_HSI48RDYC_MSK (0x1UL << STM32_RCC_CICR_HSI48RDYC_POS)
#define STM32_RCC_CICR_HSI48RDYC STM32_RCC_CICR_HSI48RDYC_MSK

/**************  BIT DEFINITION FOR STM32_RCC_AHB1RSTR REGISTER  **************/
#define STM32_RCC_AHB1RSTR_DMA1RST_POS 0U
#define STM32_RCC_AHB1RSTR_DMA1RST_MSK (0x1UL << STM32_RCC_AHB1RSTR_DMA1RST_POS)
#define STM32_RCC_AHB1RSTR_DMA1RST STM32_RCC_AHB1RSTR_DMA1RST_MSK
#define STM32_RCC_AHB1RSTR_DMA2RST_POS 1U
#define STM32_RCC_AHB1RSTR_DMA2RST_MSK (0x1UL << STM32_RCC_AHB1RSTR_DMA2RST_POS)
#define STM32_RCC_AHB1RSTR_DMA2RST STM32_RCC_AHB1RSTR_DMA2RST_MSK
#define STM32_RCC_AHB1RSTR_FLASHRST_POS 8U
#define STM32_RCC_AHB1RSTR_FLASHRST_MSK \
	(0x1UL << STM32_RCC_AHB1RSTR_FLASHRST_POS)
#define STM32_RCC_AHB1RSTR_FLASHRST STM32_RCC_AHB1RSTR_FLASHRST_MSK
#define STM32_RCC_AHB1RSTR_CRCRST_POS 12U
#define STM32_RCC_AHB1RSTR_CRCRST_MSK (0x1UL << STM32_RCC_AHB1RSTR_CRCRST_POS)
#define STM32_RCC_AHB1RSTR_CRCRST STM32_RCC_AHB1RSTR_CRCRST_MSK
#define STM32_RCC_AHB1RSTR_TSCRST_POS 16U
#define STM32_RCC_AHB1RSTR_TSCRST_MSK (0x1UL << STM32_RCC_AHB1RSTR_TSCRST_POS)
#define STM32_RCC_AHB1RSTR_TSCRST STM32_RCC_AHB1RSTR_TSCRST_MSK

/**************  BIT DEFINITION FOR STM32_RCC_AHB2RSTR REGISTER  **************/
#define STM32_RCC_AHB2RSTR_GPIOARST_POS 0U
#define STM32_RCC_AHB2RSTR_GPIOARST_MSK \
	(0x1UL << STM32_RCC_AHB2RSTR_GPIOARST_POS)
#define STM32_RCC_AHB2RSTR_GPIOARST STM32_RCC_AHB2RSTR_GPIOARST_MSK
#define STM32_RCC_AHB2RSTR_GPIOBRST_POS 1U
#define STM32_RCC_AHB2RSTR_GPIOBRST_MSK \
	(0x1UL << STM32_RCC_AHB2RSTR_GPIOBRST_POS)
#define STM32_RCC_AHB2RSTR_GPIOBRST STM32_RCC_AHB2RSTR_GPIOBRST_MSK
#define STM32_RCC_AHB2RSTR_GPIOCRST_POS 2U
#define STM32_RCC_AHB2RSTR_GPIOCRST_MSK \
	(0x1UL << STM32_RCC_AHB2RSTR_GPIOCRST_POS)
#define STM32_RCC_AHB2RSTR_GPIOCRST STM32_RCC_AHB2RSTR_GPIOCRST_MSK
#define STM32_RCC_AHB2RSTR_GPIODRST_POS 3U
#define STM32_RCC_AHB2RSTR_GPIODRST_MSK \
	(0x1UL << STM32_RCC_AHB2RSTR_GPIODRST_POS)
#define STM32_RCC_AHB2RSTR_GPIODRST STM32_RCC_AHB2RSTR_GPIODRST_MSK
#define STM32_RCC_AHB2RSTR_GPIOERST_POS 4U
#define STM32_RCC_AHB2RSTR_GPIOERST_MSK \
	(0x1UL << STM32_RCC_AHB2RSTR_GPIOERST_POS)
#define STM32_RCC_AHB2RSTR_GPIOERST STM32_RCC_AHB2RSTR_GPIOERST_MSK
#define STM32_RCC_AHB2RSTR_GPIOHRST_POS 7U
#define STM32_RCC_AHB2RSTR_GPIOHRST_MSK \
	(0x1UL << STM32_RCC_AHB2RSTR_GPIOHRST_POS)
#define STM32_RCC_AHB2RSTR_GPIOHRST STM32_RCC_AHB2RSTR_GPIOHRST_MSK
#define STM32_RCC_AHB2RSTR_ADCRST_POS 13U
#define STM32_RCC_AHB2RSTR_ADCRST_MSK (0x1UL << STM32_RCC_AHB2RSTR_ADCRST_POS)
#define STM32_RCC_AHB2RSTR_ADCRST STM32_RCC_AHB2RSTR_ADCRST_MSK
#define STM32_RCC_AHB2RSTR_AESRST_POS 16U
#define STM32_RCC_AHB2RSTR_AESRST_MSK (0x1UL << STM32_RCC_AHB2RSTR_AESRST_POS)
#define STM32_RCC_AHB2RSTR_AESRST STM32_RCC_AHB2RSTR_AESRST_MSK
#define STM32_RCC_AHB2RSTR_RNGRST_POS 18U
#define STM32_RCC_AHB2RSTR_RNGRST_MSK (0x1UL << STM32_RCC_AHB2RSTR_RNGRST_POS)
#define STM32_RCC_AHB2RSTR_RNGRST STM32_RCC_AHB2RSTR_RNGRST_MSK

/*************  BIT DEFINITION FOR STM32_RCC_AHB3RSTR REGISTER  **************/
#define STM32_RCC_AHB3RSTR_QSPIRST_POS 8U
#define STM32_RCC_AHB3RSTR_QSPIRST_MSK (0x1UL << STM32_RCC_AHB3RSTR_QSPIRST_POS)
#define STM32_RCC_AHB3RSTR_QSPIRST STM32_RCC_AHB3RSTR_QSPIRST_MSK

/*************  BIT DEFINITION FOR STM32_RCC_APB1RSTR1 REGISTER  **************/
#define STM32_RCC_APB1RSTR1_TIM2RST_POS 0U
#define STM32_RCC_APB1RSTR1_TIM2RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_TIM2RST_POS)
#define STM32_RCC_APB1RSTR1_TIM2RST STM32_RCC_APB1RSTR1_TIM2RST_MSK
#define STM32_RCC_APB1RSTR1_TIM6RST_POS 4U
#define STM32_RCC_APB1RSTR1_TIM6RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_TIM6RST_POS)
#define STM32_RCC_APB1RSTR1_TIM6RST STM32_RCC_APB1RSTR1_TIM6RST_MSK
#define STM32_RCC_APB1RSTR1_TIM7RST_POS 5U
#define STM32_RCC_APB1RSTR1_TIM7RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_TIM7RST_POS)
#define STM32_RCC_APB1RSTR1_TIM7RST STM32_RCC_APB1RSTR1_TIM7RST_MSK
#define STM32_RCC_APB1RSTR1_LCDRST_POS 9U
#define STM32_RCC_APB1RSTR1_LCDRST_MSK (0x1UL << STM32_RCC_APB1RSTR1_LCDRST_POS)
#define STM32_RCC_APB1RSTR1_LCDRST STM32_RCC_APB1RSTR1_LCDRST_MSK
#define STM32_RCC_APB1RSTR1_SPI2RST_POS 14U
#define STM32_RCC_APB1RSTR1_SPI2RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_SPI2RST_POS)
#define STM32_RCC_APB1RSTR1_SPI2RST STM32_RCC_APB1RSTR1_SPI2RST_MSK
#define STM32_RCC_APB1RSTR1_SPI3RST_POS 15U
#define STM32_RCC_APB1RSTR1_SPI3RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_SPI3RST_POS)
#define STM32_RCC_APB1RSTR1_SPI3RST STM32_RCC_APB1RSTR1_SPI3RST_MSK
#define STM32_RCC_APB1RSTR1_USART2RST_POS 17U
#define STM32_RCC_APB1RSTR1_USART2RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_USART2RST_POS)
#define STM32_RCC_APB1RSTR1_USART2RST STM32_RCC_APB1RSTR1_USART2RST_MSK
#define STM32_RCC_APB1RSTR1_USART3RST_POS 18U
#define STM32_RCC_APB1RSTR1_USART3RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_USART3RST_POS)
#define STM32_RCC_APB1RSTR1_USART3RST STM32_RCC_APB1RSTR1_USART3RST_MSK
#define STM32_RCC_APB1RSTR1_I2C1RST_POS 21U
#define STM32_RCC_APB1RSTR1_I2C1RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_I2C1RST_POS)
#define STM32_RCC_APB1RSTR1_I2C1RST STM32_RCC_APB1RSTR1_I2C1RST_MSK
#define STM32_RCC_APB1RSTR1_I2C2RST_POS 22U
#define STM32_RCC_APB1RSTR1_I2C2RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_I2C2RST_POS)
#define STM32_RCC_APB1RSTR1_I2C2RST STM32_RCC_APB1RSTR1_I2C2RST_MSK
#define STM32_RCC_APB1RSTR1_I2C3RST_POS 23U
#define STM32_RCC_APB1RSTR1_I2C3RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_I2C3RST_POS)
#define STM32_RCC_APB1RSTR1_I2C3RST STM32_RCC_APB1RSTR1_I2C3RST_MSK
#define STM32_RCC_APB1RSTR1_CRSRST_POS 24U
#define STM32_RCC_APB1RSTR1_CRSRST_MSK (0x1UL << STM32_RCC_APB1RSTR1_CRSRST_POS)
#define STM32_RCC_APB1RSTR1_CRSRST STM32_RCC_APB1RSTR1_CRSRST_MSK
#define STM32_RCC_APB1RSTR1_CAN1RST_POS 25U
#define STM32_RCC_APB1RSTR1_CAN1RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_CAN1RST_POS)
#define STM32_RCC_APB1RSTR1_CAN1RST STM32_RCC_APB1RSTR1_CAN1RST_MSK
#define STM32_RCC_APB1RSTR1_USBFSRST_POS 26U
#define STM32_RCC_APB1RSTR1_USBFSRST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_USBFSRST_POS)
#define STM32_RCC_APB1RSTR1_USBFSRST STM32_RCC_APB1RSTR1_USBFSRST_MSK
#define STM32_RCC_APB1RSTR1_PWRRST_POS 28U
#define STM32_RCC_APB1RSTR1_PWRRST_MSK (0x1UL << STM32_RCC_APB1RSTR1_PWRRST_POS)
#define STM32_RCC_APB1RSTR1_PWRRST STM32_RCC_APB1RSTR1_PWRRST_MSK
#define STM32_RCC_APB1RSTR1_DAC1RST_POS 29U
#define STM32_RCC_APB1RSTR1_DAC1RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_DAC1RST_POS)
#define STM32_RCC_APB1RSTR1_DAC1RST STM32_RCC_APB1RSTR1_DAC1RST_MSK
#define STM32_RCC_APB1RSTR1_OPAMPRST_POS 30U
#define STM32_RCC_APB1RSTR1_OPAMPRST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_OPAMPRST_POS)
#define STM32_RCC_APB1RSTR1_OPAMPRST STM32_RCC_APB1RSTR1_OPAMPRST_MSK
#define STM32_RCC_APB1RSTR1_LPTIM1RST_POS 31U
#define STM32_RCC_APB1RSTR1_LPTIM1RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR1_LPTIM1RST_POS)
#define STM32_RCC_APB1RSTR1_LPTIM1RST STM32_RCC_APB1RSTR1_LPTIM1RST_MSK

/*************  BIT DEFINITION FOR STM32_RCC_APB1RSTR2 REGISTER  **************/
#define STM32_RCC_APB1RSTR2_LPUART1RST_POS 0U
#define STM32_RCC_APB1RSTR2_LPUART1RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR2_LPUART1RST_POS)
#define STM32_RCC_APB1RSTR2_LPUART1RST STM32_RCC_APB1RSTR2_LPUART1RST_MSK
#define STM32_RCC_APB1RSTR2_SWPMI1RST_POS 2U
#define STM32_RCC_APB1RSTR2_SWPMI1RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR2_SWPMI1RST_POS)
#define STM32_RCC_APB1RSTR2_SWPMI1RST STM32_RCC_APB1RSTR2_SWPMI1RST_MSK
#define STM32_RCC_APB1RSTR2_LPTIM2RST_POS 5U
#define STM32_RCC_APB1RSTR2_LPTIM2RST_MSK \
	(0x1UL << STM32_RCC_APB1RSTR2_LPTIM2RST_POS)
#define STM32_RCC_APB1RSTR2_LPTIM2RST STM32_RCC_APB1RSTR2_LPTIM2RST_MSK

/**************  BIT DEFINITION FOR STM32_RCC_APB2RSTR REGISTER  **************/
#define STM32_RCC_APB2RSTR_SYSCFGRST_POS 0U
#define STM32_RCC_APB2RSTR_SYSCFGRST_MSK \
	(0x1UL << STM32_RCC_APB2RSTR_SYSCFGRST_POS)
#define STM32_RCC_APB2RSTR_SYSCFGRST STM32_RCC_APB2RSTR_SYSCFGRST_MSK
#define STM32_RCC_APB2RSTR_SDMMC1RST_POS 10U
#define STM32_RCC_APB2RSTR_SDMMC1RST_MSK \
	(0x1UL << STM32_RCC_APB2RSTR_SDMMC1RST_POS)
#define STM32_RCC_APB2RSTR_SDMMC1RST STM32_RCC_APB2RSTR_SDMMC1RST_MSK
#define STM32_RCC_APB2RSTR_TIM1RST_POS 11U
#define STM32_RCC_APB2RSTR_TIM1RST_MSK (0x1UL << STM32_RCC_APB2RSTR_TIM1RST_POS)
#define STM32_RCC_APB2RSTR_TIM1RST STM32_RCC_APB2RSTR_TIM1RST_MSK
#define STM32_RCC_APB2RSTR_SPI1RST_POS 12U
#define STM32_RCC_APB2RSTR_SPI1RST_MSK (0x1UL << STM32_RCC_APB2RSTR_SPI1RST_POS)
#define STM32_RCC_APB2RSTR_SPI1RST STM32_RCC_APB2RSTR_SPI1RST_MSK
#define STM32_RCC_APB2RSTR_USART1RST_POS 14U
#define STM32_RCC_APB2RSTR_USART1RST_MSK \
	(0x1UL << STM32_RCC_APB2RSTR_USART1RST_POS)
#define STM32_RCC_APB2RSTR_USART1RST STM32_RCC_APB2RSTR_USART1RST_MSK
#define STM32_RCC_APB2RSTR_TIM15RST_POS 16U
#define STM32_RCC_APB2RSTR_TIM15RST_MSK \
	(0x1UL << STM32_RCC_APB2RSTR_TIM15RST_POS)
#define STM32_RCC_APB2RSTR_TIM15RST STM32_RCC_APB2RSTR_TIM15RST_MSK
#define STM32_RCC_APB2RSTR_TIM16RST_POS 17U
#define STM32_RCC_APB2RSTR_TIM16RST_MSK \
	(0x1UL << STM32_RCC_APB2RSTR_TIM16RST_POS)
#define STM32_RCC_APB2RSTR_TIM16RST STM32_RCC_APB2RSTR_TIM16RST_MSK
#define STM32_RCC_APB2RSTR_SAI1RST_POS 21U
#define STM32_RCC_APB2RSTR_SAI1RST_MSK (0x1UL << STM32_RCC_APB2RSTR_SAI1RST_POS)
#define STM32_RCC_APB2RSTR_SAI1RST STM32_RCC_APB2RSTR_SAI1RST_MSK

/**************  BIT DEFINITION FOR STM32_RCC_AHB1ENR REGISTER  ***************/
#define STM32_RCC_AHB1ENR_DMA1EN_POS 0U
#define STM32_RCC_AHB1ENR_DMA1EN_MSK (0x1UL << STM32_RCC_AHB1ENR_DMA1EN_POS)
#define STM32_RCC_AHB1ENR_DMA1EN STM32_RCC_AHB1ENR_DMA1EN_MSK
#define STM32_RCC_AHB1ENR_DMA2EN_POS 1U
#define STM32_RCC_AHB1ENR_DMA2EN_MSK (0x1UL << STM32_RCC_AHB1ENR_DMA2EN_POS)
#define STM32_RCC_AHB1ENR_DMA2EN STM32_RCC_AHB1ENR_DMA2EN_MSK
#define STM32_RCC_AHB1ENR_FLASHEN_POS 8U
#define STM32_RCC_AHB1ENR_FLASHEN_MSK (0x1UL << STM32_RCC_AHB1ENR_FLASHEN_POS)
#define STM32_RCC_AHB1ENR_FLASHEN STM32_RCC_AHB1ENR_FLASHEN_MSK
#define STM32_RCC_AHB1ENR_CRCEN_POS 12U
#define STM32_RCC_AHB1ENR_CRCEN_MSK (0x1UL << STM32_RCC_AHB1ENR_CRCEN_POS)
#define STM32_RCC_AHB1ENR_CRCEN STM32_RCC_AHB1ENR_CRCEN_MSK
#define STM32_RCC_AHB1ENR_TSCEN_POS 16U
#define STM32_RCC_AHB1ENR_TSCEN_MSK (0x1UL << STM32_RCC_AHB1ENR_TSCEN_POS)
#define STM32_RCC_AHB1ENR_TSCEN STM32_RCC_AHB1ENR_TSCEN_MSK

/***************  BIT DEFINITION FOR STM32_RCC_AHB2ENR REGISTER  *********/
#define STM32_RCC_AHB2ENR_GPIOAEN_POS 0U
#define STM32_RCC_AHB2ENR_GPIOAEN_MSK (0x1UL << STM32_RCC_AHB2ENR_GPIOAEN_POS)
#define STM32_RCC_AHB2ENR_GPIOAEN STM32_RCC_AHB2ENR_GPIOAEN_MSK
#define STM32_RCC_AHB2ENR_GPIOBEN_POS 1U
#define STM32_RCC_AHB2ENR_GPIOBEN_MSK (0x1UL << STM32_RCC_AHB2ENR_GPIOBEN_POS)
#define STM32_RCC_AHB2ENR_GPIOBEN STM32_RCC_AHB2ENR_GPIOBEN_MSK
#define STM32_RCC_AHB2ENR_GPIOCEN_POS 2U
#define STM32_RCC_AHB2ENR_GPIOCEN_MSK (0x1UL << STM32_RCC_AHB2ENR_GPIOCEN_POS)
#define STM32_RCC_AHB2ENR_GPIOCEN STM32_RCC_AHB2ENR_GPIOCEN_MSK
#define STM32_RCC_AHB2ENR_GPIODEN_POS 3U
#define STM32_RCC_AHB2ENR_GPIODEN_MSK (0x1UL << STM32_RCC_AHB2ENR_GPIODEN_POS)
#define STM32_RCC_AHB2ENR_GPIODEN STM32_RCC_AHB2ENR_GPIODEN_MSK
#define STM32_RCC_AHB2ENR_GPIOEEN_POS 4U
#define STM32_RCC_AHB2ENR_GPIOEEN_MSK (0x1UL << STM32_RCC_AHB2ENR_GPIOEEN_POS)
#define STM32_RCC_AHB2ENR_GPIOEEN STM32_RCC_AHB2ENR_GPIOEEN_MSK
#define STM32_RCC_AHB2ENR_GPIOHEN_POS 7U
#define STM32_RCC_AHB2ENR_GPIOHEN_MSK (0x1UL << STM32_RCC_AHB2ENR_GPIOHEN_POS)
#define STM32_RCC_AHB2ENR_GPIOHEN STM32_RCC_AHB2ENR_GPIOHEN_MSK
#define STM32_RCC_AHB2ENR_ADCEN_POS 13U
#define STM32_RCC_AHB2ENR_ADCEN_MSK (0x1UL << STM32_RCC_AHB2ENR_ADCEN_POS)
#define STM32_RCC_AHB2ENR_ADCEN STM32_RCC_AHB2ENR_ADCEN_MSK
#define STM32_RCC_AHB2ENR_AESEN_POS 16U
#define STM32_RCC_AHB2ENR_AESEN_MSK (0x1UL << STM32_RCC_AHB2ENR_AESEN_POS)
#define STM32_RCC_AHB2ENR_AESEN STM32_RCC_AHB2ENR_AESEN_MSK
#define STM32_RCC_AHB2ENR_RNGEN_POS 18U
#define STM32_RCC_AHB2ENR_RNGEN_MSK (0x1UL << STM32_RCC_AHB2ENR_RNGEN_POS)
#define STM32_RCC_AHB2ENR_RNGEN STM32_RCC_AHB2ENR_RNGEN_MSK

/**************  BIT DEFINITION FOR STM32_RCC_AHB3ENR REGISTER  ***************/
#define STM32_RCC_AHB3ENR_QSPIEN_POS 8U
#define STM32_RCC_AHB3ENR_QSPIEN_MSK (0x1UL << STM32_RCC_AHB3ENR_QSPIEN_POS)
#define STM32_RCC_AHB3ENR_QSPIEN STM32_RCC_AHB3ENR_QSPIEN_MSK

/**************  BIT DEFINITION FOR STM32_RCC_APB1ENR1 REGISTER  **************/
#define STM32_RCC_APB1ENR1_TIM2EN_POS 0U
#define STM32_RCC_APB1ENR1_TIM2EN_MSK (0x1UL << STM32_RCC_APB1ENR1_TIM2EN_POS)
#define STM32_RCC_APB1ENR1_TIM2EN STM32_RCC_APB1ENR1_TIM2EN_MSK
#define STM32_RCC_APB1ENR1_TIM6EN_POS 4U
#define STM32_RCC_APB1ENR1_TIM6EN_MSK (0x1UL << STM32_RCC_APB1ENR1_TIM6EN_POS)
#define STM32_RCC_APB1ENR1_TIM6EN STM32_RCC_APB1ENR1_TIM6EN_MSK
#define STM32_RCC_APB1ENR1_TIM7EN_POS 5U
#define STM32_RCC_APB1ENR1_TIM7EN_MSK (0x1UL << STM32_RCC_APB1ENR1_TIM7EN_POS)
#define STM32_RCC_APB1ENR1_TIM7EN STM32_RCC_APB1ENR1_TIM7EN_MSK
#define STM32_RCC_APB1ENR1_LCDEN_POS 9U
#define STM32_RCC_APB1ENR1_LCDEN_MSK (0x1UL << STM32_RCC_APB1ENR1_LCDEN_POS)
#define STM32_RCC_APB1ENR1_LCDEN STM32_RCC_APB1ENR1_LCDEN_MSK
#define STM32_RCC_APB1ENR1_RTCAPBEN_POS 10U
#define STM32_RCC_APB1ENR1_RTCAPBEN_MSK \
	(0x1UL << STM32_RCC_APB1ENR1_RTCAPBEN_POS)
#define STM32_RCC_APB1ENR1_RTCAPBEN STM32_RCC_APB1ENR1_RTCAPBEN_MSK
#define STM32_RCC_APB1ENR1_WWDGEN_POS 11U
#define STM32_RCC_APB1ENR1_WWDGEN_MSK (0x1UL << STM32_RCC_APB1ENR1_WWDGEN_POS)
#define STM32_RCC_APB1ENR1_WWDGEN STM32_RCC_APB1ENR1_WWDGEN_MSK
#define STM32_RCC_APB1ENR1_SPI2EN_POS 14U
#define STM32_RCC_APB1ENR1_SPI2EN_MSK (0x1UL << STM32_RCC_APB1ENR1_SPI2EN_POS)
#define STM32_RCC_APB1ENR1_SPI2EN STM32_RCC_APB1ENR1_SPI2EN_MSK
#define STM32_RCC_APB1ENR1_SPI3EN_POS 15U
#define STM32_RCC_APB1ENR1_SPI3EN_MSK (0x1UL << STM32_RCC_APB1ENR1_SPI3EN_POS)
#define STM32_RCC_APB1ENR1_SPI3EN STM32_RCC_APB1ENR1_SPI3EN_MSK
#define STM32_RCC_APB1ENR1_USART2EN_POS 17U
#define STM32_RCC_APB1ENR1_USART2EN_MSK \
	(0x1UL << STM32_RCC_APB1ENR1_USART2EN_POS)
#define STM32_RCC_APB1ENR1_USART2EN STM32_RCC_APB1ENR1_USART2EN_MSK
#define STM32_RCC_APB1ENR1_USART3EN_POS 18U
#define STM32_RCC_APB1ENR1_USART3EN_MSK \
	(0x1UL << STM32_RCC_APB1ENR1_USART3EN_POS)
#define STM32_RCC_APB1ENR1_USART3EN STM32_RCC_APB1ENR1_USART3EN_MSK
#define STM32_RCC_APB1ENR1_I2C1EN_POS 21U
#define STM32_RCC_APB1ENR1_I2C1EN_MSK (0x1UL << STM32_RCC_APB1ENR1_I2C1EN_POS)
#define STM32_RCC_APB1ENR1_I2C1EN STM32_RCC_APB1ENR1_I2C1EN_MSK
#define STM32_RCC_APB1ENR1_I2C2EN_POS 22U
#define STM32_RCC_APB1ENR1_I2C2EN_MSK (0x1UL << STM32_RCC_APB1ENR1_I2C2EN_POS)
#define STM32_RCC_APB1ENR1_I2C2EN STM32_RCC_APB1ENR1_I2C2EN_MSK
#define STM32_RCC_APB1ENR1_I2C3EN_POS 23U
#define STM32_RCC_APB1ENR1_I2C3EN_MSK (0x1UL << STM32_RCC_APB1ENR1_I2C3EN_POS)
#define STM32_RCC_APB1ENR1_I2C3EN STM32_RCC_APB1ENR1_I2C3EN_MSK
#define STM32_RCC_APB1ENR1_CRSEN_POS 24U
#define STM32_RCC_APB1ENR1_CRSEN_MSK (0x1UL << STM32_RCC_APB1ENR1_CRSEN_POS)
#define STM32_RCC_APB1ENR1_CRSEN STM32_RCC_APB1ENR1_CRSEN_MSK
#define STM32_RCC_APB1ENR1_CAN1EN_POS 25U
#define STM32_RCC_APB1ENR1_CAN1EN_MSK (0x1UL << STM32_RCC_APB1ENR1_CAN1EN_POS)
#define STM32_RCC_APB1ENR1_CAN1EN STM32_RCC_APB1ENR1_CAN1EN_MSK
#define STM32_RCC_APB1ENR1_USBFSEN_POS 26U
#define STM32_RCC_APB1ENR1_USBFSEN_MSK (0x1UL << STM32_RCC_APB1ENR1_USBFSEN_POS)
#define STM32_RCC_APB1ENR1_USBFSEN STM32_RCC_APB1ENR1_USBFSEN_MSK
#define STM32_RCC_APB1ENR1_PWREN_POS 28U
#define STM32_RCC_APB1ENR1_PWREN_MSK (0x1UL << STM32_RCC_APB1ENR1_PWREN_POS)
#define STM32_RCC_APB1ENR1_PWREN STM32_RCC_APB1ENR1_PWREN_MSK
#define STM32_RCC_APB1ENR1_DAC1EN_POS 29U
#define STM32_RCC_APB1ENR1_DAC1EN_MSK (0x1UL << STM32_RCC_APB1ENR1_DAC1EN_POS)
#define STM32_RCC_APB1ENR1_DAC1EN STM32_RCC_APB1ENR1_DAC1EN_MSK
#define STM32_RCC_APB1ENR1_OPAMPEN_POS 30U
#define STM32_RCC_APB1ENR1_OPAMPEN_MSK (0x1UL << STM32_RCC_APB1ENR1_OPAMPEN_POS)
#define STM32_RCC_APB1ENR1_OPAMPEN STM32_RCC_APB1ENR1_OPAMPEN_MSK
#define STM32_RCC_APB1ENR1_LPTIM1EN_POS 31U
#define STM32_RCC_APB1ENR1_LPTIM1EN_MSK \
	(0x1UL << STM32_RCC_APB1ENR1_LPTIM1EN_POS)
#define STM32_RCC_APB1ENR1_LPTIM1EN STM32_RCC_APB1ENR1_LPTIM1EN_MSK

/************  BIT DEFINITION FOR STM32_RCC_APB1RSTR2 REGISTER  **************/
#define STM32_RCC_APB1ENR2_LPUART1EN_POS 0U
#define STM32_RCC_APB1ENR2_LPUART1EN_MSK \
	(0x1UL << STM32_RCC_APB1ENR2_LPUART1EN_POS)
#define STM32_RCC_APB1ENR2_LPUART1EN STM32_RCC_APB1ENR2_LPUART1EN_MSK
#define STM32_RCC_APB1ENR2_SWPMI1EN_POS 2U
#define STM32_RCC_APB1ENR2_SWPMI1EN_MSK \
	(0x1UL << STM32_RCC_APB1ENR2_SWPMI1EN_POS)
#define STM32_RCC_APB1ENR2_SWPMI1EN STM32_RCC_APB1ENR2_SWPMI1EN_MSK
#define STM32_RCC_APB1ENR2_LPTIM2EN_POS 5U
#define STM32_RCC_APB1ENR2_LPTIM2EN_MSK \
	(0x1UL << STM32_RCC_APB1ENR2_LPTIM2EN_POS)
#define STM32_RCC_APB1ENR2_LPTIM2EN STM32_RCC_APB1ENR2_LPTIM2EN_MSK

/**************  BIT DEFINITION FOR STM32_RCC_APB2ENR REGISTER  ***************/
#define STM32_RCC_APB2ENR_SYSCFGEN_POS 0U
#define STM32_RCC_APB2ENR_SYSCFGEN_MSK (0x1UL << STM32_RCC_APB2ENR_SYSCFGEN_POS)
#define STM32_RCC_APB2ENR_SYSCFGEN STM32_RCC_APB2ENR_SYSCFGEN_MSK
#define STM32_RCC_APB2ENR_FWEN_POS 7U
#define STM32_RCC_APB2ENR_FWEN_MSK (0x1UL << STM32_RCC_APB2ENR_FWEN_POS)
#define STM32_RCC_APB2ENR_FWEN STM32_RCC_APB2ENR_FWEN_MSK
#define STM32_RCC_APB2ENR_SDMMC1EN_POS 10U
#define STM32_RCC_APB2ENR_SDMMC1EN_MSK (0x1UL << STM32_RCC_APB2ENR_SDMMC1EN_POS)
#define STM32_RCC_APB2ENR_SDMMC1EN STM32_RCC_APB2ENR_SDMMC1EN_MSK
#define STM32_RCC_APB2ENR_TIM1EN_POS 11U
#define STM32_RCC_APB2ENR_TIM1EN_MSK (0x1UL << STM32_RCC_APB2ENR_TIM1EN_POS)
#define STM32_RCC_APB2ENR_TIM1EN STM32_RCC_APB2ENR_TIM1EN_MSK
#define STM32_RCC_APB2ENR_SPI1EN_POS 12U
#define STM32_RCC_APB2ENR_SPI1EN_MSK (0x1UL << STM32_RCC_APB2ENR_SPI1EN_POS)
#define STM32_RCC_APB2ENR_SPI1EN STM32_RCC_APB2ENR_SPI1EN_MSK
#define STM32_RCC_APB2ENR_USART1EN_POS 14U
#define STM32_RCC_APB2ENR_USART1EN_MSK (0x1UL << STM32_RCC_APB2ENR_USART1EN_POS)
#define STM32_RCC_APB2ENR_USART1EN STM32_RCC_APB2ENR_USART1EN_MSK
#define STM32_RCC_APB2ENR_TIM15EN_POS 16U
#define STM32_RCC_APB2ENR_TIM15EN_MSK (0x1UL << STM32_RCC_APB2ENR_TIM15EN_POS)
#define STM32_RCC_APB2ENR_TIM15EN STM32_RCC_APB2ENR_TIM15EN_MSK
#define STM32_RCC_APB2ENR_TIM16EN_POS 17U
#define STM32_RCC_APB2ENR_TIM16EN_MSK (0x1UL << STM32_RCC_APB2ENR_TIM16EN_POS)
#define STM32_RCC_APB2ENR_TIM16EN STM32_RCC_APB2ENR_TIM16EN_MSK
#define STM32_RCC_APB2ENR_SAI1EN_POS 21U
#define STM32_RCC_APB2ENR_SAI1EN_MSK (0x1UL << STM32_RCC_APB2ENR_SAI1EN_POS)
#define STM32_RCC_APB2ENR_SAI1EN STM32_RCC_APB2ENR_SAI1EN_MSK

/************  BIT DEFINITION FOR STM32_RCC_AHB1SMENR REGISTER  ***************/
#define STM32_RCC_AHB1SMENR_DMA1SMEN_POS 0U
#define STM32_RCC_AHB1SMENR_DMA1SMEN_MSK \
	(0x1UL << STM32_RCC_AHB1SMENR_DMA1SMEN_POS)
#define STM32_RCC_AHB1SMENR_DMA1SMEN STM32_RCC_AHB1SMENR_DMA1SMEN_MSK
#define STM32_RCC_AHB1SMENR_DMA2SMEN_POS 1U
#define STM32_RCC_AHB1SMENR_DMA2SMEN_MSK \
	(0x1UL << STM32_RCC_AHB1SMENR_DMA2SMEN_POS)
#define STM32_RCC_AHB1SMENR_DMA2SMEN STM32_RCC_AHB1SMENR_DMA2SMEN_MSK
#define STM32_RCC_AHB1SMENR_FLASHSMEN_POS 8U
#define STM32_RCC_AHB1SMENR_FLASHSMEN_MSK \
	(0x1UL << STM32_RCC_AHB1SMENR_FLASHSMEN_POS)
#define STM32_RCC_AHB1SMENR_FLASHSMEN STM32_RCC_AHB1SMENR_FLASHSMEN_MSK
#define STM32_RCC_AHB1SMENR_SRAM1SMEN_POS 9U
#define STM32_RCC_AHB1SMENR_SRAM1SMEN_MSK \
	(0x1UL << STM32_RCC_AHB1SMENR_SRAM1SMEN_POS)
#define STM32_RCC_AHB1SMENR_SRAM1SMEN STM32_RCC_AHB1SMENR_SRAM1SMEN_MSK
#define STM32_RCC_AHB1SMENR_CRCSMEN_POS 12U
#define STM32_RCC_AHB1SMENR_CRCSMEN_MSK \
	(0x1UL << STM32_RCC_AHB1SMENR_CRCSMEN_POS)
#define STM32_RCC_AHB1SMENR_CRCSMEN STM32_RCC_AHB1SMENR_CRCSMEN_MSK
#define STM32_RCC_AHB1SMENR_TSCSMEN_POS 16U
#define STM32_RCC_AHB1SMENR_TSCSMEN_MSK \
	(0x1UL << STM32_RCC_AHB1SMENR_TSCSMEN_POS)
#define STM32_RCC_AHB1SMENR_TSCSMEN STM32_RCC_AHB1SMENR_TSCSMEN_MSK

/**************  BIT DEFINITION FOR STM32_RCC_AHB2SMENR REGISTER  *************/
#define STM32_RCC_AHB2SMENR_GPIOASMEN_POS 0U
#define STM32_RCC_AHB2SMENR_GPIOASMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_GPIOASMEN_POS)
#define STM32_RCC_AHB2SMENR_GPIOASMEN STM32_RCC_AHB2SMENR_GPIOASMEN_MSK
#define STM32_RCC_AHB2SMENR_GPIOBSMEN_POS 1U
#define STM32_RCC_AHB2SMENR_GPIOBSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_GPIOBSMEN_POS)
#define STM32_RCC_AHB2SMENR_GPIOBSMEN STM32_RCC_AHB2SMENR_GPIOBSMEN_MSK
#define STM32_RCC_AHB2SMENR_GPIOCSMEN_POS 2U
#define STM32_RCC_AHB2SMENR_GPIOCSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_GPIOCSMEN_POS)
#define STM32_RCC_AHB2SMENR_GPIOCSMEN STM32_RCC_AHB2SMENR_GPIOCSMEN_MSK
#define STM32_RCC_AHB2SMENR_GPIODSMEN_POS 3U
#define STM32_RCC_AHB2SMENR_GPIODSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_GPIODSMEN_POS)
#define STM32_RCC_AHB2SMENR_GPIODSMEN STM32_RCC_AHB2SMENR_GPIODSMEN_MSK
#define STM32_RCC_AHB2SMENR_GPIOESMEN_POS 4U
#define STM32_RCC_AHB2SMENR_GPIOESMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_GPIOESMEN_POS)
#define STM32_RCC_AHB2SMENR_GPIOESMEN STM32_RCC_AHB2SMENR_GPIOESMEN_MSK
#define STM32_RCC_AHB2SMENR_GPIOHSMEN_POS 7U
#define STM32_RCC_AHB2SMENR_GPIOHSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_GPIOHSMEN_POS)
#define STM32_RCC_AHB2SMENR_GPIOHSMEN STM32_RCC_AHB2SMENR_GPIOHSMEN_MSK
#define STM32_RCC_AHB2SMENR_SRAM2SMEN_POS 9U
#define STM32_RCC_AHB2SMENR_SRAM2SMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_SRAM2SMEN_POS)
#define STM32_RCC_AHB2SMENR_SRAM2SMEN STM32_RCC_AHB2SMENR_SRAM2SMEN_MSK
#define STM32_RCC_AHB2SMENR_ADCSMEN_POS 13U
#define STM32_RCC_AHB2SMENR_ADCSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_ADCSMEN_POS)
#define STM32_RCC_AHB2SMENR_ADCSMEN STM32_RCC_AHB2SMENR_ADCSMEN_MSK
#define STM32_RCC_AHB2SMENR_AESSMEN_POS 16U
#define STM32_RCC_AHB2SMENR_AESSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_AESSMEN_POS)
#define STM32_RCC_AHB2SMENR_AESSMEN STM32_RCC_AHB2SMENR_AESSMEN_MSK
#define STM32_RCC_AHB2SMENR_RNGSMEN_POS 18U
#define STM32_RCC_AHB2SMENR_RNGSMEN_MSK \
	(0x1UL << STM32_RCC_AHB2SMENR_RNGSMEN_POS)
#define STM32_RCC_AHB2SMENR_RNGSMEN STM32_RCC_AHB2SMENR_RNGSMEN_MSK

/**************  BIT DEFINITION FOR STM32_RCC_AHB3SMENR REGISTER  *************/
#define STM32_RCC_AHB3SMENR_QSPISMEN_POS 8U
#define STM32_RCC_AHB3SMENR_QSPISMEN_MSK \
	(0x1UL << STM32_RCC_AHB3SMENR_QSPISMEN_POS)
#define STM32_RCC_AHB3SMENR_QSPISMEN STM32_RCC_AHB3SMENR_QSPISMEN_MSK

/*************  BIT DEFINITION FOR STM32_RCC_APB1SMENR1 REGISTER  *************/
#define STM32_RCC_APB1SMENR1_TIM2SMEN_POS 0U
#define STM32_RCC_APB1SMENR1_TIM2SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_TIM2SMEN_POS)
#define STM32_RCC_APB1SMENR1_TIM2SMEN STM32_RCC_APB1SMENR1_TIM2SMEN_MSK
#define STM32_RCC_APB1SMENR1_TIM6SMEN_POS 4U
#define STM32_RCC_APB1SMENR1_TIM6SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_TIM6SMEN_POS)
#define STM32_RCC_APB1SMENR1_TIM6SMEN STM32_RCC_APB1SMENR1_TIM6SMEN_MSK
#define STM32_RCC_APB1SMENR1_TIM7SMEN_POS 5U
#define STM32_RCC_APB1SMENR1_TIM7SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_TIM7SMEN_POS)
#define STM32_RCC_APB1SMENR1_TIM7SMEN STM32_RCC_APB1SMENR1_TIM7SMEN_MSK
#define STM32_RCC_APB1SMENR1_LCDSMEN_POS 9U
#define STM32_RCC_APB1SMENR1_LCDSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_LCDSMEN_POS)
#define STM32_RCC_APB1SMENR1_LCDSMEN STM32_RCC_APB1SMENR1_LCDSMEN_MSK
#define STM32_RCC_APB1SMENR1_RTCAPBSMEN_POS 10U
#define STM32_RCC_APB1SMENR1_RTCAPBSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_RTCAPBSMEN_POS)
#define STM32_RCC_APB1SMENR1_RTCAPBSMEN STM32_RCC_APB1SMENR1_RTCAPBSMEN_MSK
#define STM32_RCC_APB1SMENR1_WWDGSMEN_POS 11U
#define STM32_RCC_APB1SMENR1_WWDGSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_WWDGSMEN_POS)
#define STM32_RCC_APB1SMENR1_WWDGSMEN STM32_RCC_APB1SMENR1_WWDGSMEN_MSK
#define STM32_RCC_APB1SMENR1_SPI2SMEN_POS 14U
#define STM32_RCC_APB1SMENR1_SPI2SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_SPI2SMEN_POS)
#define STM32_RCC_APB1SMENR1_SPI2SMEN STM32_RCC_APB1SMENR1_SPI2SMEN_MSK
#define STM32_RCC_APB1SMENR1_SPI3SMEN_POS 15U
#define STM32_RCC_APB1SMENR1_SPI3SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_SPI3SMEN_POS)
#define STM32_RCC_APB1SMENR1_SPI3SMEN STM32_RCC_APB1SMENR1_SPI3SMEN_MSK
#define STM32_RCC_APB1SMENR1_USART2SMEN_POS 17U
#define STM32_RCC_APB1SMENR1_USART2SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_USART2SMEN_POS)
#define STM32_RCC_APB1SMENR1_USART2SMEN STM32_RCC_APB1SMENR1_USART2SMEN_MSK
#define STM32_RCC_APB1SMENR1_USART3SMEN_POS 18U
#define STM32_RCC_APB1SMENR1_USART3SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_USART3SMEN_POS)
#define STM32_RCC_APB1SMENR1_USART3SMEN STM32_RCC_APB1SMENR1_USART3SMEN_MSK
#define STM32_RCC_APB1SMENR1_I2C1SMEN_POS 21U
#define STM32_RCC_APB1SMENR1_I2C1SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_I2C1SMEN_POS)
#define STM32_RCC_APB1SMENR1_I2C1SMEN STM32_RCC_APB1SMENR1_I2C1SMEN_MSK
#define STM32_RCC_APB1SMENR1_I2C2SMEN_POS 22U
#define STM32_RCC_APB1SMENR1_I2C2SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_I2C2SMEN_POS)
#define STM32_RCC_APB1SMENR1_I2C2SMEN STM32_RCC_APB1SMENR1_I2C2SMEN_MSK
#define STM32_RCC_APB1SMENR1_I2C3SMEN_POS 23U
#define STM32_RCC_APB1SMENR1_I2C3SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_I2C3SMEN_POS)
#define STM32_RCC_APB1SMENR1_I2C3SMEN STM32_RCC_APB1SMENR1_I2C3SMEN_MSK
#define STM32_RCC_APB1SMENR1_CRSSMEN_POS 24U
#define STM32_RCC_APB1SMENR1_CRSSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_CRSSMEN_POS)
#define STM32_RCC_APB1SMENR1_CRSSMEN STM32_RCC_APB1SMENR1_CRSSMEN_MSK
#define STM32_RCC_APB1SMENR1_CAN1SMEN_POS 25U
#define STM32_RCC_APB1SMENR1_CAN1SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_CAN1SMEN_POS)
#define STM32_RCC_APB1SMENR1_CAN1SMEN STM32_RCC_APB1SMENR1_CAN1SMEN_MSK
#define STM32_RCC_APB1SMENR1_USBFSSMEN_POS 26U
#define STM32_RCC_APB1SMENR1_USBFSSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_USBFSSMEN_POS)
#define STM32_RCC_APB1SMENR1_USBFSSMEN STM32_RCC_APB1SMENR1_USBFSSMEN_MSK
#define STM32_RCC_APB1SMENR1_PWRSMEN_POS 28U
#define STM32_RCC_APB1SMENR1_PWRSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_PWRSMEN_POS)
#define STM32_RCC_APB1SMENR1_PWRSMEN STM32_RCC_APB1SMENR1_PWRSMEN_MSK
#define STM32_RCC_APB1SMENR1_DAC1SMEN_POS 29U
#define STM32_RCC_APB1SMENR1_DAC1SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_DAC1SMEN_POS)
#define STM32_RCC_APB1SMENR1_DAC1SMEN STM32_RCC_APB1SMENR1_DAC1SMEN_MSK
#define STM32_RCC_APB1SMENR1_OPAMPSMEN_POS 30U
#define STM32_RCC_APB1SMENR1_OPAMPSMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_OPAMPSMEN_POS)
#define STM32_RCC_APB1SMENR1_OPAMPSMEN STM32_RCC_APB1SMENR1_OPAMPSMEN_MSK
#define STM32_RCC_APB1SMENR1_LPTIM1SMEN_POS 31U
#define STM32_RCC_APB1SMENR1_LPTIM1SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR1_LPTIM1SMEN_POS)
#define STM32_RCC_APB1SMENR1_LPTIM1SMEN STM32_RCC_APB1SMENR1_LPTIM1SMEN_MSK

/*************  BIT DEFINITION FOR STM32_RCC_APB1SMENR2 REGISTER  *************/
#define STM32_RCC_APB1SMENR2_LPUART1SMEN_POS 0U
#define STM32_RCC_APB1SMENR2_LPUART1SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR2_LPUART1SMEN_POS)
#define STM32_RCC_APB1SMENR2_LPUART1SMEN STM32_RCC_APB1SMENR2_LPUART1SMEN_MSK
#define STM32_RCC_APB1SMENR2_SWPMI1SMEN_POS 2U
#define STM32_RCC_APB1SMENR2_SWPMI1SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR2_SWPMI1SMEN_POS)
#define STM32_RCC_APB1SMENR2_SWPMI1SMEN STM32_RCC_APB1SMENR2_SWPMI1SMEN_MSK
#define STM32_RCC_APB1SMENR2_LPTIM2SMEN_POS 5U
#define STM32_RCC_APB1SMENR2_LPTIM2SMEN_MSK \
	(0x1UL << STM32_RCC_APB1SMENR2_LPTIM2SMEN_POS)
#define STM32_RCC_APB1SMENR2_LPTIM2SMEN STM32_RCC_APB1SMENR2_LPTIM2SMEN_MSK

/************  BIT DEFINITION FOR STM32_RCC_APB2SMENR REGISTER  *************/
#define STM32_RCC_APB2SMENR_SYSCFGSMEN_POS 0U
#define STM32_RCC_APB2SMENR_SYSCFGSMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_SYSCFGSMEN_POS)
#define STM32_RCC_APB2SMENR_SYSCFGSMEN STM32_RCC_APB2SMENR_SYSCFGSMEN_MSK
#define STM32_RCC_APB2SMENR_SDMMC1SMEN_POS 10U
#define STM32_RCC_APB2SMENR_SDMMC1SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_SDMMC1SMEN_POS)
#define STM32_RCC_APB2SMENR_SDMMC1SMEN STM32_RCC_APB2SMENR_SDMMC1SMEN_MSK
#define STM32_RCC_APB2SMENR_TIM1SMEN_POS 11U
#define STM32_RCC_APB2SMENR_TIM1SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_TIM1SMEN_POS)
#define STM32_RCC_APB2SMENR_TIM1SMEN STM32_RCC_APB2SMENR_TIM1SMEN_MSK
#define STM32_RCC_APB2SMENR_SPI1SMEN_POS 12U
#define STM32_RCC_APB2SMENR_SPI1SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_SPI1SMEN_POS)
#define STM32_RCC_APB2SMENR_SPI1SMEN STM32_RCC_APB2SMENR_SPI1SMEN_MSK
#define STM32_RCC_APB2SMENR_USART1SMEN_POS 14U
#define STM32_RCC_APB2SMENR_USART1SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_USART1SMEN_POS)
#define STM32_RCC_APB2SMENR_USART1SMEN STM32_RCC_APB2SMENR_USART1SMEN_MSK
#define STM32_RCC_APB2SMENR_TIM15SMEN_POS 16U
#define STM32_RCC_APB2SMENR_TIM15SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_TIM15SMEN_POS)
#define STM32_RCC_APB2SMENR_TIM15SMEN STM32_RCC_APB2SMENR_TIM15SMEN_MSK
#define STM32_RCC_APB2SMENR_TIM16SMEN_POS 17U
#define STM32_RCC_APB2SMENR_TIM16SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_TIM16SMEN_POS)
#define STM32_RCC_APB2SMENR_TIM16SMEN STM32_RCC_APB2SMENR_TIM16SMEN_MSK
#define STM32_RCC_APB2SMENR_SAI1SMEN_POS 21U
#define STM32_RCC_APB2SMENR_SAI1SMEN_MSK \
	(0x1UL << STM32_RCC_APB2SMENR_SAI1SMEN_POS)
#define STM32_RCC_APB2SMENR_SAI1SMEN STM32_RCC_APB2SMENR_SAI1SMEN_MSK

/*************  BIT DEFINITION FOR STM32_RCC_CCIPR REGISTER  ******************/
#define STM32_RCC_CCIPR_USART1SEL_POS 0U
#define STM32_RCC_CCIPR_USART1SEL_MSK (0x3UL << STM32_RCC_CCIPR_USART1SEL_POS)
#define STM32_RCC_CCIPR_USART1SEL STM32_RCC_CCIPR_USART1SEL_MSK
#define STM32_RCC_CCIPR_USART1SEL_0 (0x1UL << STM32_RCC_CCIPR_USART1SEL_POS)
#define STM32_RCC_CCIPR_USART1SEL_1 (0x2UL << STM32_RCC_CCIPR_USART1SEL_POS)

#define STM32_RCC_CCIPR_USART2SEL_POS 2U
#define STM32_RCC_CCIPR_USART2SEL_MSK (0x3UL << STM32_RCC_CCIPR_USART2SEL_POS)
#define STM32_RCC_CCIPR_USART2SEL STM32_RCC_CCIPR_USART2SEL_MSK
#define STM32_RCC_CCIPR_USART2SEL_0 (0x1UL << STM32_RCC_CCIPR_USART2SEL_POS)
#define STM32_RCC_CCIPR_USART2SEL_1 (0x2UL << STM32_RCC_CCIPR_USART2SEL_POS)

#define STM32_RCC_CCIPR_USART3SEL_POS 4U
#define STM32_RCC_CCIPR_USART3SEL_MSK (0x3UL << STM32_RCC_CCIPR_USART3SEL_POS)
#define STM32_RCC_CCIPR_USART3SEL STM32_RCC_CCIPR_USART3SEL_MSK
#define STM32_RCC_CCIPR_USART3SEL_0 (0x1UL << STM32_RCC_CCIPR_USART3SEL_POS)
#define STM32_RCC_CCIPR_USART3SEL_1 (0x2UL << STM32_RCC_CCIPR_USART3SEL_POS)

#define STM32_RCC_CCIPR_LPUART1SEL_POS 10U
#define STM32_RCC_CCIPR_LPUART1SEL_MSK (0x3UL << STM32_RCC_CCIPR_LPUART1SEL_POS)
#define STM32_RCC_CCIPR_LPUART1SEL STM32_RCC_CCIPR_LPUART1SEL_MSK
#define STM32_RCC_CCIPR_LPUART1SEL_0 (0x1UL << STM32_RCC_CCIPR_LPUART1SEL_POS)
#define STM32_RCC_CCIPR_LPUART1SEL_1 (0x2UL << STM32_RCC_CCIPR_LPUART1SEL_POS)

#define STM32_RCC_CCIPR_I2C1SEL_POS 12U
#define STM32_RCC_CCIPR_I2C1SEL_MSK (0x3UL << STM32_RCC_CCIPR_I2C1SEL_POS)
#define STM32_RCC_CCIPR_I2C1SEL STM32_RCC_CCIPR_I2C1SEL_MSK
#define STM32_RCC_CCIPR_I2C1SEL_0 (0x1UL << STM32_RCC_CCIPR_I2C1SEL_POS)
#define STM32_RCC_CCIPR_I2C1SEL_1 (0x2UL << STM32_RCC_CCIPR_I2C1SEL_POS)

#define STM32_RCC_CCIPR_I2C2SEL_POS 14U
#define STM32_RCC_CCIPR_I2C2SEL_MSK (0x3UL << STM32_RCC_CCIPR_I2C2SEL_POS)
#define STM32_RCC_CCIPR_I2C2SEL STM32_RCC_CCIPR_I2C2SEL_MSK
#define STM32_RCC_CCIPR_I2C2SEL_0 (0x1UL << STM32_RCC_CCIPR_I2C2SEL_POS)
#define STM32_RCC_CCIPR_I2C2SEL_1 (0x2UL << STM32_RCC_CCIPR_I2C2SEL_POS)

#define STM32_RCC_CCIPR_I2C3SEL_POS 16U
#define STM32_RCC_CCIPR_I2C3SEL_MSK (0x3UL << STM32_RCC_CCIPR_I2C3SEL_POS)
#define STM32_RCC_CCIPR_I2C3SEL STM32_RCC_CCIPR_I2C3SEL_MSK
#define STM32_RCC_CCIPR_I2C3SEL_0 (0x1UL << STM32_RCC_CCIPR_I2C3SEL_POS)
#define STM32_RCC_CCIPR_I2C3SEL_1 (0x2UL << STM32_RCC_CCIPR_I2C3SEL_POS)

#define STM32_RCC_CCIPR_LPTIM1SEL_POS 18U
#define STM32_RCC_CCIPR_LPTIM1SEL_MSK (0x3UL << STM32_RCC_CCIPR_LPTIM1SEL_POS)
#define STM32_RCC_CCIPR_LPTIM1SEL STM32_RCC_CCIPR_LPTIM1SEL_MSK
#define STM32_RCC_CCIPR_LPTIM1SEL_0 (0x1UL << STM32_RCC_CCIPR_LPTIM1SEL_POS)
#define STM32_RCC_CCIPR_LPTIM1SEL_1 (0x2UL << STM32_RCC_CCIPR_LPTIM1SEL_POS)

#define STM32_RCC_CCIPR_LPTIM2SEL_POS 20U
#define STM32_RCC_CCIPR_LPTIM2SEL_MSK (0x3UL << STM32_RCC_CCIPR_LPTIM2SEL_POS)
#define STM32_RCC_CCIPR_LPTIM2SEL STM32_RCC_CCIPR_LPTIM2SEL_MSK
#define STM32_RCC_CCIPR_LPTIM2SEL_0 (0x1UL << STM32_RCC_CCIPR_LPTIM2SEL_POS)
#define STM32_RCC_CCIPR_LPTIM2SEL_1 (0x2UL << STM32_RCC_CCIPR_LPTIM2SEL_POS)

#define STM32_RCC_CCIPR_SAI1SEL_POS 22U
#define STM32_RCC_CCIPR_SAI1SEL_MSK (0x3UL << STM32_RCC_CCIPR_SAI1SEL_POS)
#define STM32_RCC_CCIPR_SAI1SEL STM32_RCC_CCIPR_SAI1SEL_MSK
#define STM32_RCC_CCIPR_SAI1SEL_0 (0x1UL << STM32_RCC_CCIPR_SAI1SEL_POS)
#define STM32_RCC_CCIPR_SAI1SEL_1 (0x2UL << STM32_RCC_CCIPR_SAI1SEL_POS)

#define STM32_RCC_CCIPR_CLK48SEL_POS 26U
#define STM32_RCC_CCIPR_CLK48SEL_MSK (0x3UL << STM32_RCC_CCIPR_CLK48SEL_POS)
#define STM32_RCC_CCIPR_CLK48SEL STM32_RCC_CCIPR_CLK48SEL_MSK
#define STM32_RCC_CCIPR_CLK48SEL_0 (0x1UL << STM32_RCC_CCIPR_CLK48SEL_POS)
#define STM32_RCC_CCIPR_CLK48SEL_1 (0x2UL << STM32_RCC_CCIPR_CLK48SEL_POS)

#define STM32_RCC_CCIPR_ADCSEL_POS 28U
#define STM32_RCC_CCIPR_ADCSEL_MSK (0x3UL << STM32_RCC_CCIPR_ADCSEL_POS)
#define STM32_RCC_CCIPR_ADCSEL STM32_RCC_CCIPR_ADCSEL_MSK
#define STM32_RCC_CCIPR_ADCSEL_0 (0x1UL << STM32_RCC_CCIPR_ADCSEL_POS)
#define STM32_RCC_CCIPR_ADCSEL_1 (0x2UL << STM32_RCC_CCIPR_ADCSEL_POS)

#define STM32_RCC_CCIPR_SWPMI1SEL_POS 30U
#define STM32_RCC_CCIPR_SWPMI1SEL_MSK (0x1UL << STM32_RCC_CCIPR_SWPMI1SEL_POS)
#define STM32_RCC_CCIPR_SWPMI1SEL STM32_RCC_CCIPR_SWPMI1SEL_MSK

/**************  BIT DEFINITION FOR STM32_RCC_BDCR REGISTER  ******************/
#define STM32_RCC_BDCR_LSEBYP_POS 2U
#define STM32_RCC_BDCR_LSEBYP_MSK (0x1UL << STM32_RCC_BDCR_LSEBYP_POS)
#define STM32_RCC_BDCR_LSEBYP STM32_RCC_BDCR_LSEBYP_MSK

#define STM32_RCC_BDCR_LSEDRV_POS 3U
#define STM32_RCC_BDCR_LSEDRV_MSK (0x3UL << STM32_RCC_BDCR_LSEDRV_POS)
#define STM32_RCC_BDCR_LSEDRV STM32_RCC_BDCR_LSEDRV_MSK
#define STM32_RCC_BDCR_LSEDRV_0 (0x1UL << STM32_RCC_BDCR_LSEDRV_POS)
#define STM32_RCC_BDCR_LSEDRV_1 (0x2UL << STM32_RCC_BDCR_LSEDRV_POS)

#define STM32_RCC_BDCR_LSECSSON_POS 5U
#define STM32_RCC_BDCR_LSECSSON_MSK (0x1UL << STM32_RCC_BDCR_LSECSSON_POS)
#define STM32_RCC_BDCR_LSECSSON STM32_RCC_BDCR_LSECSSON_MSK
#define STM32_RCC_BDCR_LSECSSD_POS 6U
#define STM32_RCC_BDCR_LSECSSD_MSK (0x1UL << STM32_RCC_BDCR_LSECSSD_POS)
#define STM32_RCC_BDCR_LSECSSD STM32_RCC_BDCR_LSECSSD_MSK

#define STM32_RCC_BDCR_RTCSEL_POS 8U
#define STM32_RCC_BDCR_RTCSEL_MSK (0x3UL << STM32_RCC_BDCR_RTCSEL_POS)
#define STM32_RCC_BDCR_RTCSEL STM32_RCC_BDCR_RTCSEL_MSK
#define STM32_RCC_BDCR_RTCSEL_0 (0x1UL << STM32_RCC_BDCR_RTCSEL_POS)
#define STM32_RCC_BDCR_RTCSEL_1 (0x2UL << STM32_RCC_BDCR_RTCSEL_POS)

#define STM32_RCC_BDCR_LSCOEN_POS 24U
#define STM32_RCC_BDCR_LSCOEN_MSK (0x1UL << STM32_RCC_BDCR_LSCOEN_POS)
#define STM32_RCC_BDCR_LSCOEN STM32_RCC_BDCR_LSCOEN_MSK
#define STM32_RCC_BDCR_LSCOSEL_POS 25U
#define STM32_RCC_BDCR_LSCOSEL_MSK (0x1UL << STM32_RCC_BDCR_LSCOSEL_POS)
#define STM32_RCC_BDCR_LSCOSEL STM32_RCC_BDCR_LSCOSEL_MSK

/**************  BIT DEFINITION FOR STM32_RCC_CSR REGISTER  *******************/
#define STM32_RCC_CSR_LSION_POS 0U
#define STM32_RCC_CSR_LSION_MSK (0x1UL << STM32_RCC_CSR_LSION_POS)
#define STM32_RCC_CSR_LSION STM32_RCC_CSR_LSION_MSK
#define STM32_RCC_CSR_LSIRDY_POS 1U
#define STM32_RCC_CSR_LSIRDY_MSK (0x1UL << STM32_RCC_CSR_LSIRDY_POS)
#define STM32_RCC_CSR_LSIRDY STM32_RCC_CSR_LSIRDY_MSK

#define STM32_RCC_CSR_MSISRANGE_POS 8U
#define STM32_RCC_CSR_MSISRANGE_MSK (0xFUL << STM32_RCC_CSR_MSISRANGE_POS)
#define STM32_RCC_CSR_MSISRANGE STM32_RCC_CSR_MSISRANGE_MSK
#define STM32_RCC_CSR_MSISRANGE_1 (0x4UL << STM32_RCC_CSR_MSISRANGE_POS)
#define STM32_RCC_CSR_MSISRANGE_2 (0x5UL << STM32_RCC_CSR_MSISRANGE_POS)
#define STM32_RCC_CSR_MSISRANGE_4 (0x6UL << STM32_RCC_CSR_MSISRANGE_POS)
#define STM32_RCC_CSR_MSISRANGE_8 (0x7UL << STM32_RCC_CSR_MSISRANGE_POS)

#define STM32_RCC_CSR_RMVF_POS 23U
#define STM32_RCC_CSR_RMVF_MSK (0x1UL << STM32_RCC_CSR_RMVF_POS)
#define STM32_RCC_CSR_RMVF STM32_RCC_CSR_RMVF_MSK
#define STM32_RCC_CSR_FWRSTF_POS 24U
#define STM32_RCC_CSR_FWRSTF_MSK (0x1UL << STM32_RCC_CSR_FWRSTF_POS)
#define STM32_RCC_CSR_FWRSTF STM32_RCC_CSR_FWRSTF_MSK
#define STM32_RCC_CSR_OBLRSTF_POS 25U
#define STM32_RCC_CSR_OBLRSTF_MSK (0x1UL << STM32_RCC_CSR_OBLRSTF_POS)
#define STM32_RCC_CSR_OBLRSTF STM32_RCC_CSR_OBLRSTF_MSK
#define STM32_RCC_CSR_PINRSTF_POS 26U
#define STM32_RCC_CSR_PINRSTF_MSK (0x1UL << STM32_RCC_CSR_PINRSTF_POS)
#define STM32_RCC_CSR_PINRSTF STM32_RCC_CSR_PINRSTF_MSK
#define STM32_RCC_CSR_BORRSTF_POS 27U
#define STM32_RCC_CSR_BORRSTF_MSK (0x1UL << STM32_RCC_CSR_BORRSTF_POS)
#define STM32_RCC_CSR_BORRSTF STM32_RCC_CSR_BORRSTF_MSK
#define STM32_RCC_CSR_SFTRSTF_POS 28U
#define STM32_RCC_CSR_SFTRSTF_MSK (0x1UL << STM32_RCC_CSR_SFTRSTF_POS)
#define STM32_RCC_CSR_SFTRSTF STM32_RCC_CSR_SFTRSTF_MSK
#define STM32_RCC_CSR_IWDGRSTF_POS 29U
#define STM32_RCC_CSR_IWDGRSTF_MSK (0x1UL << STM32_RCC_CSR_IWDGRSTF_POS)
#define STM32_RCC_CSR_IWDGRSTF STM32_RCC_CSR_IWDGRSTF_MSK
#define STM32_RCC_CSR_WWDGRSTF_POS 30U
#define STM32_RCC_CSR_WWDGRSTF_MSK (0x1UL << STM32_RCC_CSR_WWDGRSTF_POS)
#define STM32_RCC_CSR_WWDGRSTF STM32_RCC_CSR_WWDGRSTF_MSK
#define STM32_RCC_CSR_LPWRRSTF_POS 31U
#define STM32_RCC_CSR_LPWRRSTF_MSK (0x1UL << STM32_RCC_CSR_LPWRRSTF_POS)
#define STM32_RCC_CSR_LPWRRSTF STM32_RCC_CSR_LPWRRSTF_MSK

/**************  Bit definition for STM32_RCC_CRRCR register  *****************/
#define STM32_RCC_CRRCR_HSI48ON_POS 0U
#define STM32_RCC_CRRCR_HSI48ON_MSK (0x1UL << STM32_RCC_CRRCR_HSI48ON_POS)
#define STM32_RCC_CRRCR_HSI48ON STM32_RCC_CRRCR_HSI48ON_MSK
#define STM32_RCC_CRRCR_HSI48RDY_POS 1U
#define STM32_RCC_CRRCR_HSI48RDY_MSK (0x1UL << STM32_RCC_CRRCR_HSI48RDY_POS)
#define STM32_RCC_CRRCR_HSI48RDY STM32_RCC_CRRCR_HSI48RDY_MSK

/*!< HSI48CAL configuration */
#define STM32_RCC_CRRCR_HSI48CAL_POS 7U
#define STM32_RCC_CRRCR_HSI48CAL_MSK (0x1FFUL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL STM32_RCC_CRRCR_HSI48CAL_MSK
#define STM32_RCC_CRRCR_HSI48CAL_0 (0x001UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_1 (0x002UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_2 (0x004UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_3 (0x008UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_4 (0x010UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_5 (0x020UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_6 (0x040UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_7 (0x080UL << STM32_RCC_CRRCR_HSI48CAL_POS)
#define STM32_RCC_CRRCR_HSI48CAL_8 (0x100UL << STM32_RCC_CRRCR_HSI48CAL_POS)

#define STM32_SYSCFG_CFGR1 REG32(STM32_SYSCFG_BASE + 0x04)
#define STM32_SYSCFG_EXTICR(n) REG32(STM32_SYSCFG_BASE + 8 + 4 * (n))
#define STM32_SYSCFG_I2CFMP(n) BIT(n + 21)

/* Peripheral bits for STM32_RCC_APB/AHB and DBGMCU regs */
#define STM32_RCC_PB1_PWREN			BIT(28)

#define STM32_RCC_PB2_SYSCFGEN		BIT(0)
#define STM32_RCC_PB2_USART1		BIT(14)

#define STM32_RCC_HB1_DMA1		BIT(0)
#define STM32_RCC_HB1_DMA2		BIT(1)

#define STM32_RCC_HB2_GPIOA		BIT(0)
#define STM32_RCC_HB2_GPIOB		BIT(1)
#define STM32_RCC_HB2_GPIOC		BIT(2)
#define STM32_RCC_HB2_GPIOD		BIT(3)
#define STM32_RCC_HB2_GPIOE		BIT(4)
#define STM32_RCC_HB2_GPIOH		BIT(7)
#define STM32_RCC_HB2_ADC1		BIT(13)

/* Reset causes definitions */
/* Reset causes in RCC CSR register */
#define  STM32_RCC_RESET_CAUSE	STM32_RCC_CSR
#define  RESET_CAUSE_WDG                0x60000000
#define  RESET_CAUSE_SFT                0x10000000
#define  RESET_CAUSE_POR                0x08000000
#define  RESET_CAUSE_PIN                0x04000000
#define  RESET_CAUSE_OTHER              0xff000000
#define  RESET_CAUSE_RMVF               BIT(23)
/* Power cause in PWR CSR register */
#define STM32_PWR_RESET_CAUSE			STM32_PWR_CSR
#define STM32_PWR_RESET_CAUSE_CLR		STM32_PWR_SCR
#define  RESET_CAUSE_SBF                BIT(8)
#define  RESET_CAUSE_SBF_CLR            BIT(8)

/* --- Watchdogs --- */

/* --- Real-Time Clock --- */
#define STM32_RTC_TR                REG32(STM32_RTC_BASE + 0x00)
#define STM32_RTC_DR                REG32(STM32_RTC_BASE + 0x04)
#define STM32_RTC_CR                REG32(STM32_RTC_BASE + 0x08)
#define STM32_RTC_CR_BYPSHAD        BIT(5)
#define STM32_RTC_CR_ALRAE          BIT(8)
#define STM32_RTC_CR_WUTE           BIT(10)
#define STM32_RTC_CR_ALRAIE         BIT(12)
#define STM32_RTC_CR_WUTIE          BIT(14)
#define STM32_RTC_ISR               REG32(STM32_RTC_BASE + 0x0C)
#define STM32_RTC_ISR_ALRAWF        BIT(0)
#define STM32_RTC_ISR_WUTWF         BIT(2)
#define STM32_RTC_ISR_INITS         BIT(4)
#define STM32_RTC_ISR_RSF           BIT(5)
#define STM32_RTC_ISR_INITF         BIT(6)
#define STM32_RTC_ISR_INIT          BIT(7)
#define STM32_RTC_ISR_ALRAF         BIT(8)
#define STM32_RTC_ISR_WUTF          BIT(9)
#define STM32_RTC_PRER              REG32(STM32_RTC_BASE + 0x10)
#define STM32_RTC_PRER_A_MASK       (0x7f << 16)
#define STM32_RTC_PRER_S_MASK       (0x7fff << 0)
#define STM32_RTC_WUTR              REG32(STM32_RTC_BASE + 0x14)
#define STM32_RTC_CALIBR            REG32(STM32_RTC_BASE + 0x18)
#define STM32_RTC_ALRMAR            REG32(STM32_RTC_BASE + 0x1C)
#define STM32_RTC_ALRMBR            REG32(STM32_RTC_BASE + 0x20)
#define STM32_RTC_WPR               REG32(STM32_RTC_BASE + 0x24)
#define STM32_RTC_SSR               REG32(STM32_RTC_BASE + 0x28)
#define STM32_RTC_TSTR              REG32(STM32_RTC_BASE + 0x30)
#define STM32_RTC_TSDR              REG32(STM32_RTC_BASE + 0x34)
#define STM32_RTC_TAFCR             REG32(STM32_RTC_BASE + 0x40)
#define STM32_RTC_ALRMASSR          REG32(STM32_RTC_BASE + 0x44)
#define STM32_RTC_BACKUP(n)         REG32(STM32_RTC_BASE + 0x50 + 4 * (n))

#define STM32_RTC_CLEAR_FLAG(x)                         \
			(STM32_RTC_ISR = (~((x) | STM32_RTC_ISR_INIT) | \
					(STM32_RTC_ISR & STM32_RTC_ISR_INIT)))

#define STM32_BKP_DATA(n)           STM32_RTC_BACKUP(n)
#define STM32_BKP_BYTES             128

#define RTC_TR_PM_POS                  22U
#define RTC_TR_PM_MSK                  (0x1UL << RTC_TR_PM_POS)
#define RTC_TR_PM                      RTC_TR_PM_MSK
#define RTC_TR_HT_POS                  20U
#define RTC_TR_HT_MSK                  (0x3UL << RTC_TR_HT_POS)
#define RTC_TR_HT                      RTC_TR_HT_MSK
#define RTC_TR_HU_POS                  16U
#define RTC_TR_HU_MSK                  (0xFUL << RTC_TR_HU_POS)
#define RTC_TR_HU                      RTC_TR_HU_MSK
#define RTC_TR_MNT_POS                 12U
#define RTC_TR_MNT_MSK                 (0x7UL << RTC_TR_MNT_POS)
#define RTC_TR_MNT                     RTC_TR_MNT_MSK
#define RTC_TR_MNU_POS                 8U
#define RTC_TR_MNU_MSK                 (0xFUL << RTC_TR_MNU_POS)
#define RTC_TR_MNU                     RTC_TR_MNU_MSK
#define RTC_TR_ST_POS                  4U
#define RTC_TR_ST_MSK                  (0x7UL << RTC_TR_ST_POS)
#define RTC_TR_ST                      RTC_TR_ST_MSK
#define RTC_TR_SU_POS                  0U
#define RTC_TR_SU_MSK                  (0xFUL << RTC_TR_SU_POS)
#define RTC_TR_SU                      RTC_TR_SU_MSK



/* --- SPI --- */

/* The SPI controller registers */
struct stm32_spi_regs {
	uint16_t cr1;
	uint16_t _pad0;
	uint16_t cr2;
	uint16_t _pad1;
	unsigned int sr;
	uint8_t dr;
	uint8_t _pad2;
	uint16_t _pad3;
	unsigned int crcpr;
	unsigned int rxcrcr;
	unsigned int txcrcr;
	unsigned int i2scfgr;	/* STM32L only */
	unsigned int i2spr;	/* STM32L only */
};
/* Must be volatile, or compiler optimizes out repeated accesses */
typedef volatile struct stm32_spi_regs stm32_spi_regs_t;

#define STM32_SPI1_REGS ((stm32_spi_regs_t *)STM32_SPI1_BASE)
#define STM32_SPI2_REGS ((stm32_spi_regs_t *)STM32_SPI2_BASE)
#define STM32_SPI3_REGS ((stm32_spi_regs_t *)STM32_SPI3_BASE)
#define STM32_SPI4_REGS ((stm32_spi_regs_t *)STM32_SPI4_BASE)

#define STM32_SPI_CR1_BIDIMODE		BIT(15)
#define STM32_SPI_CR1_BIDIOE		BIT(14)
#define STM32_SPI_CR1_CRCEN		BIT(13)
#define STM32_SPI_CR1_SSM		BIT(9)
#define STM32_SPI_CR1_SSI		BIT(8)
#define STM32_SPI_CR1_LSBFIRST		BIT(7)
#define STM32_SPI_CR1_SPE		BIT(6)
#define STM32_SPI_CR1_BR_DIV64R		(5 << 3)
#define STM32_SPI_CR1_BR_DIV4R		BIT(3)
#define STM32_SPI_CR1_MSTR		BIT(2)
#define STM32_SPI_CR1_CPOL		BIT(1)
#define STM32_SPI_CR1_CPHA		BIT(0)
#define STM32_SPI_CR2_FRXTH		BIT(12)
#define STM32_SPI_CR2_DATASIZE(n)	(((n) - 1) << 8)
#define STM32_SPI_CR2_TXEIE		BIT(7)
#define STM32_SPI_CR2_RXNEIE		BIT(6)
#define STM32_SPI_CR2_NSSP		BIT(3)
#define STM32_SPI_CR2_SSOE		BIT(2)
#define STM32_SPI_CR2_TXDMAEN		BIT(1)
#define STM32_SPI_CR2_RXDMAEN		BIT(0)

#define STM32_SPI_SR_RXNE		BIT(0)
#define STM32_SPI_SR_TXE		BIT(1)
#define STM32_SPI_SR_CRCERR		BIT(4)
#define STM32_SPI_SR_BSY		BIT(7)
#define STM32_SPI_SR_FRLVL		(3 << 9)
#define STM32_SPI_SR_FTLVL		(3 << 11)
/* --- Debug --- */
#define STM32_DBGMCU_APB1FZ         REG32(STM32_DBGMCU_BASE + 0x08)
#define STM32_DBGMCU_APB2FZ         REG32(STM32_DBGMCU_BASE + 0x0C)

/* --- Flash --- */
#define STM32_FLASH_ACR             REG32(STM32_FLASH_REGS_BASE + 0x00)
#define STM32_FLASH_ACR_LATENCY_SHIFT (0)
#define STM32_FLASH_ACR_LATENCY_MASK  (7 << STM32_FLASH_ACR_LATENCY_SHIFT)
#define STM32_FLASH_ACR_PRFTEN      BIT(8)
#define STM32_FLASH_ACR_ICEN        BIT(9)
#define STM32_FLASH_ACR_DCEN        BIT(10)
#define STM32_FLASH_ACR_ICRST       BIT(11)
#define STM32_FLASH_ACR_DCRST       BIT(12)
#define STM32_FLASH_PDKEYR          REG32(STM32_FLASH_REGS_BASE + 0x04)
#define STM32_FLASH_KEYR            REG32(STM32_FLASH_REGS_BASE + 0x08)
#define  FLASH_KEYR_KEY1            0x45670123
#define  FLASH_KEYR_KEY2            0xCDEF89AB
#define STM32_FLASH_OPTKEYR         REG32(STM32_FLASH_REGS_BASE + 0x0c)
#define  FLASH_OPTKEYR_KEY1         0x08192A3B
#define  FLASH_OPTKEYR_KEY2         0x4C5D6E7F
#define STM32_FLASH_SR              REG32(STM32_FLASH_REGS_BASE + 0x10)
#define  FLASH_SR_BUSY              BIT(16)
#define  FLASH_SR_ERR_MASK          (0xc3fa)
#define STM32_FLASH_CR              REG32(STM32_FLASH_REGS_BASE + 0x14)
#define  FLASH_CR_PG                BIT(0)
#define  FLASH_CR_PER               BIT(1)
#define  FLASH_CR_STRT              BIT(16)
#define  FLASH_CR_OPTSTRT           BIT(17)
#define  FLASH_CR_OBL_LAUNCH        BIT(27)
#define  FLASH_CR_OPTLOCK           BIT(30)
#define  FLASH_CR_LOCK              BIT(31)
#define  FLASH_CR_PNB(sec)          (((sec) & 0xff) << 3)
#define  FLASH_CR_PNB_MASK          FLASH_CR_PNB(0xff)
#define STM32_FLASH_ECCR            REG32(STM32_FLASH_REGS_BASE + 0x18)
#define STM32_FLASH_OPTR            REG32(STM32_FLASH_REGS_BASE + 0x20)
#define STM32_FLASH_PCROP1SR        REG32(STM32_FLASH_REGS_BASE + 0x24)
#define STM32_FLASH_PCROP1ER        REG32(STM32_FLASH_REGS_BASE + 0x28)
#define STM32_FLASH_WRP1AR          REG32(STM32_FLASH_REGS_BASE + 0x2C)
#define STM32_FLASH_WRP1BR          REG32(STM32_FLASH_REGS_BASE + 0x30)
/* Minimum number of bytes that can be written to flash */
#define STM32_FLASH_MIN_WRITE_SIZE  CONFIG_FLASH_WRITE_SIZE

#define STM32_OPTB_USER_RDP         REG32(STM32_OPTB_BASE + 0x00)
#define STM32_OPTB_WRP1AR           REG32(STM32_OPTB_BASE + 0x18)
#define STM32_OPTB_WRP1BR           REG32(STM32_OPTB_BASE + 0x20)

/* --- External Interrupts --- */
#define STM32_EXTI_IMR              REG32(STM32_EXTI_BASE + 0x00)
#define STM32_EXTI_EMR              REG32(STM32_EXTI_BASE + 0x04)
#define STM32_EXTI_RTSR             REG32(STM32_EXTI_BASE + 0x08)
#define STM32_EXTI_FTSR             REG32(STM32_EXTI_BASE + 0x0c)
#define STM32_EXTI_SWIER            REG32(STM32_EXTI_BASE + 0x10)
#define STM32_EXTI_PR               REG32(STM32_EXTI_BASE + 0x14)

#define EXTI_RTC_ALR_EVENT          BIT(18)

/* --- ADC --- */
#define STM32_ADC1_ISR              REG32(STM32_ADC1_BASE + 0x00)
#define STM32_ADC1_ISR_ADRDY        BIT(0)
#define STM32_ADC1_IER              REG32(STM32_ADC1_BASE + 0x04)
#define STM32_ADC1_IER_AWDIE        BIT(7)
#define STM32_ADC1_IER_OVRIE        BIT(4)
#define STM32_ADC1_IER_EOSEQIE      BIT(3)
#define STM32_ADC1_IER_EOCIE        BIT(2)
#define STM32_ADC1_IER_EOSMPIE      BIT(1)
#define STM32_ADC1_IER_ADRDYIE      BIT(0)

#define STM32_ADC1_CR               REG32(STM32_ADC1_BASE + 0x08)
#define STM32_ADC1_CR_ADEN          BIT(0)
#define STM32_ADC1_CR_ADDIS         BIT(1)
#define STM32_ADC1_CR_ADSTP         BIT(4)
#define STM32_ADC1_CR_ADVREGEN      BIT(28)
#define STM32_ADC1_CR_DEEPPWD	    BIT(29)
#define STM32_ADC1_CR_ADCAL         BIT(31)
#define STM32_ADC1_CFGR             REG32(STM32_ADC1_BASE + 0x0C)
/* Analog watchdog channel selection */
#define STM32_ADC1_CFGR_AWDCH_MASK  (0x1f << 26)
#define STM32_ADC1_CFGR_AWDEN       BIT(23)
#define STM32_ADC1_CFGR_AWDSGL      BIT(22)
#define STM32_ADC1_CFGR_AUTDLY	    BIT(14)
/* Selects single vs continuous */
#define STM32_ADC1_CFGR_CONT        BIT(13)
/* Selects ADC_DR overwrite vs preserve */
#define STM32_ADC1_CFGR_OVRMOD      BIT(12)
/* External trigger polarity selection */
#define STM32_ADC1_CFGR_EXTEN_DIS   (0 << 10)
#define STM32_ADC1_CFGR_EXTEN_RISE  (1 << 10)
#define STM32_ADC1_CFGR_EXTEN_FALL  (2 << 10)
#define STM32_ADC1_CFGR_EXTEN_BOTH  (3 << 10)
#define STM32_ADC1_CFGR_EXTEN_MASK  (3 << 10)
#define STM32_ADC1_CFGR_ALIGN	    BIT(5)
/* External trigger selection */
#define STM32_ADC1_CFGR_TRG0        (0 << 6)
#define STM32_ADC1_CFGR_TRG1        (1 << 6)
#define STM32_ADC1_CFGR_TRG2        (2 << 6)
#define STM32_ADC1_CFGR_TRG3        (3 << 6)
#define STM32_ADC1_CFGR_TRG4        (4 << 6)
#define STM32_ADC1_CFGR_TRG5        (5 << 6)
#define STM32_ADC1_CFGR_TRG6        (6 << 6)
#define STM32_ADC1_CFGR_TRG7        (7 << 6)
#define STM32_ADC1_CFGR_TRG_MASK    (7 << 6)
/* Selects circular vs one-shot */
#define STM32_ADC1_CFGR_DMACFG      BIT(1)
#define STM32_ADC1_CFGR_DMAEN       BIT(0)
#define STM32_ADC1_CFGR2            REG32(STM32_ADC1_BASE + 0x10)
/* Sampling time selection - 1.5 ADC cycles min, 239.5 cycles max */
#define STM32_ADC1_SMPR1            REG32(STM32_ADC1_BASE + 0x14)
#define STM32_ADC1_SMPR2            REG32(STM32_ADC1_BASE + 0x18)
/* Macro to convert enum stm32_adc_smpr to SMP bits of the ADC_SMPR register */
#define STM32_ADC1_SMPR_SMP(s)      ((s) - 1)
#define STM32_ADC1_TR               REG32(STM32_ADC1_BASE + 0x20)
#define STM32_ADC1_CHSELR           REG32(STM32_ADC1_BASE + 0x28)
#define STM32_ADC1_DR               REG32(STM32_ADC1_BASE + 0x40)
#define STM32_ADC1_JSQR             REG32(STM32_ADC1_BASE + 0x4C)
#define STM32_ADC1_JDR1             REG32(STM32_ADC1_BASE + 0x80)
#define STM32_ADC1_JDR2             REG32(STM32_ADC1_BASE + 0x84)
#define STM32_ADC1_JDR3             REG32(STM32_ADC1_BASE + 0x88)
#define STM32_ADC1_JDR4             REG32(STM32_ADC1_BASE + 0x8C)
#define STM32_ADC1_CCR              REG32(STM32_ADC1_BASE + 0x308)

/* --- DMA --- */

/*
 * Available DMA channels, numbered from 0.
 *
 * Note: The STM datasheet tends to number things from 1. We should ask
 * the European elevator engineers to talk to MCU engineer counterparts
 * about this.  This means that if the datasheet refers to channel n,
 * you need to use STM32_DMAC_CHn (=n-1) in the code.
 *
 * Also note that channels are overloaded; obviously you can only use one
 * function on each channel at a time.
 */
enum dma_channel {
	/* Channel numbers */
	STM32_DMAC_CH1 = 0,
	STM32_DMAC_CH2 = 1,
	STM32_DMAC_CH3 = 2,
	STM32_DMAC_CH4 = 3,
	STM32_DMAC_CH5 = 4,
	STM32_DMAC_CH6 = 5,
	STM32_DMAC_CH7 = 6,
	/*
	 * Skip CH8, it should belong to DMA engine 1.
	 * Sharing code with STM32s that have 16 engines will be easier.
	 */
	STM32_DMAC_CH9 = 8,
	STM32_DMAC_CH10 = 9,
	STM32_DMAC_CH11 = 10,
	STM32_DMAC_CH12 = 11,
	STM32_DMAC_CH13 = 12,
	STM32_DMAC_CH14 = 13,
	STM32_DMAC_CH15 = 14,

	/* Channel functions */
	STM32_DMAC_SPI1_RX = STM32_DMAC_CH2,
	STM32_DMAC_SPI1_TX = STM32_DMAC_CH3,
	STM32_DMAC_USART1_TX = STM32_DMAC_CH14,
	STM32_DMAC_USART1_RX = STM32_DMAC_CH15,
	STM32_DMAC_SPI2_RX = STM32_DMAC_CH4,
	STM32_DMAC_SPI2_TX = STM32_DMAC_CH5,
	STM32_DMAC_SPI3_RX = STM32_DMAC_CH9,
	STM32_DMAC_SPI3_TX = STM32_DMAC_CH10,
	STM32_DMAC_COUNT = 15,
};

#define STM32_DMAC_PER_CTLR 8

/* Registers for a single channel of the DMA controller */
struct stm32_dma_chan {
	uint32_t	ccr;		/* Control */
	uint32_t	cndtr;		/* Number of data to transfer */
	uint32_t	cpar;		/* Peripheral address */
	uint32_t	cmar;		/* Memory address */
	uint32_t	reserved;
};

/* Always use stm32_dma_chan_t so volatile keyword is included! */
typedef volatile struct stm32_dma_chan stm32_dma_chan_t;

/* Common code and header file must use this */
typedef stm32_dma_chan_t dma_chan_t;

/* Registers for the DMA controller */
struct stm32_dma_regs {
	uint32_t	isr;
	uint32_t	ifcr;
	stm32_dma_chan_t chan[STM32_DMAC_COUNT];
};

/* Always use stm32_dma_regs_t so volatile keyword is included! */
typedef volatile struct stm32_dma_regs stm32_dma_regs_t;

#define STM32_DMA1_REGS ((stm32_dma_regs_t *)STM32_DMA1_BASE)


#define STM32_DMA_CCR_CHANNEL(channel)		 (0)
#define STM32_DMA2_REGS ((stm32_dma_regs_t *)STM32_DMA2_BASE)
#define STM32_DMA_REGS(channel) \
	((channel) < STM32_DMAC_PER_CTLR ? STM32_DMA1_REGS : STM32_DMA2_REGS)
#define STM32_DMA_CSELR(channel) \
	REG32(((channel) < STM32_DMAC_PER_CTLR ? \
			STM32_DMA1_BASE : STM32_DMA2_BASE)  + 0xA8)

/* Bits for DMA controller regs (isr and ifcr) */
#define STM32_DMA_CH_OFFSET(channel)   (4 * ((channel) % STM32_DMAC_PER_CTLR))
#define STM32_DMA_ISR_MASK(channel, mask) \
	((mask) << STM32_DMA_CH_OFFSET(channel))
#define STM32_DMA_ISR_GIF(channel)	STM32_DMA_ISR_MASK(channel, BIT(0))
#define STM32_DMA_ISR_TCIF(channel)	STM32_DMA_ISR_MASK(channel, BIT(1))
#define STM32_DMA_ISR_HTIF(channel)	STM32_DMA_ISR_MASK(channel, BIT(2))
#define STM32_DMA_ISR_TEIF(channel)	STM32_DMA_ISR_MASK(channel, BIT(3))
#define STM32_DMA_ISR_ALL(channel)	STM32_DMA_ISR_MASK(channel, 0x0f)

#define STM32_DMA_GIF                   BIT(0)
#define STM32_DMA_TCIF                  BIT(1)
#define STM32_DMA_HTIF                  BIT(2)
#define STM32_DMA_TEIF                  BIT(3)
#define STM32_DMA_ALL                   0xf

#define STM32_DMA_GET_ISR(channel)      \
	((STM32_DMA_REGS(channel)->isr >> STM32_DMA_CH_OFFSET(channel)) \
	& STM32_DMA_ALL)
#define STM32_DMA_SET_ISR(channel, val) \
	(STM32_DMA_REGS(channel)->isr = \
	((STM32_DMA_REGS(channel)->isr & \
	~(STM32_DMA_ALL << STM32_DMA_CH_OFFSET(channel))) | \
	(((val) & STM32_DMA_ALL) << STM32_DMA_CH_OFFSET(channel))))
#define STM32_DMA_GET_IFCR(channel)      \
	((STM32_DMA_REGS(channel)->ifcr >> STM32_DMA_CH_OFFSET(channel)) \
	& STM32_DMA_ALL)
#define STM32_DMA_SET_IFCR(channel, val) \
	(STM32_DMA_REGS(channel)->ifcr = \
	((STM32_DMA_REGS(channel)->ifcr & \
	~(STM32_DMA_ALL << STM32_DMA_CH_OFFSET(channel))) | \
	(((val) & STM32_DMA_ALL) << STM32_DMA_CH_OFFSET(channel))))


/* Bits for DMA channel regs */
#define STM32_DMA_CCR_EN		BIT(0)
#define STM32_DMA_CCR_TCIE		BIT(1)
#define STM32_DMA_CCR_HTIE		BIT(2)
#define STM32_DMA_CCR_TEIE		BIT(3)
#define STM32_DMA_CCR_DIR		BIT(4)
#define STM32_DMA_CCR_CIRC		BIT(5)
#define STM32_DMA_CCR_PINC		BIT(6)
#define STM32_DMA_CCR_MINC		BIT(7)
#define STM32_DMA_CCR_PSIZE_8_BIT	(0 << 8)
#define STM32_DMA_CCR_PSIZE_16_BIT	(1 << 8)
#define STM32_DMA_CCR_PSIZE_32_BIT	(2 << 8)
#define STM32_DMA_CCR_MSIZE_8_BIT	(0 << 10)
#define STM32_DMA_CCR_MSIZE_16_BIT	(1 << 10)
#define STM32_DMA_CCR_MSIZE_32_BIT	(2 << 10)
#define STM32_DMA_CCR_PL_LOW		(0 << 12)
#define STM32_DMA_CCR_PL_MEDIUM		(1 << 12)
#define STM32_DMA_CCR_PL_HIGH		(2 << 12)
#define STM32_DMA_CCR_PL_VERY_HIGH	(3 << 12)
#define STM32_DMA_CCR_MEM2MEM		BIT(14)

#endif /* !__ASSEMBLER__ */
