/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 15 2005)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "urtreepar.h"

UrTreePar::UrTreePar() {
	doConstructor();
#ifdef PARALLEL
	pack_data_type = 0;
	send_request = MPI_REQUEST_NULL;
	recv_request = MPI_REQUEST_NULL;
#endif // PARALLEL
}

/**
	copy from the existed tree
*/
UrTreePar::UrTreePar(UrTree &tree) {
	doConstructor();
#ifdef PARALLEL
	pack_data_type = 0;
	send_request = MPI_REQUEST_NULL;
	recv_request = MPI_REQUEST_NULL;
#endif // PARALLEL
	clone(tree);
}


#ifdef PARALLEL
/**
	pack the information of the tree, preprare to send to the master
*/
void UrTreePar::packData() {
	int num_branches = maxNBr_;
	int num_innodes = maxNInNd_;
	int data_size = num_branches * 4 + num_innodes * 2 + 1;
	int *block_lengths = new int[data_size];
	MPI_Aint *displacements = new MPI_Aint[data_size];
	MPI_Datatype *typelist = new MPI_Datatype[data_size];

	int cnt;
	for (cnt = 0; cnt < data_size; cnt++) {
		block_lengths[cnt] = 1;
	}

	MPI_Aint address;

	// pack information of branches
	for (int cnt = 0; cnt < num_branches; cnt ++) {
		int pos = cnt * 4;

		typelist[pos] = typelist[pos + 1] = typelist[pos+2] = MPI_INT;
		typelist[pos+3] = MPI_DOUBLE;

		// pack smaNdNo_
		MPI_Address(&brArr_[cnt].smaNdNo_, &address);
		displacements[pos] = address;
		// pack greNdNo_
		MPI_Address(&brArr_[cnt].greNdNo_, &address);
		displacements[pos+1] = address;
		// pack isStG_
		MPI_Address(&brArr_[cnt].isStG_, &address);
		displacements[pos+2] = address;

		// pack len_
		MPI_Address(&brArr_[cnt].len_, &address);
		displacements[pos+3] = address;
	}

	// pack 2 first adjacent branches of every internal node, to keep order in tree
	for (cnt = maxNExNd_; cnt < maxNNd_; cnt++) {
		int pos = (cnt - maxNExNd_) * 2 + num_branches * 4;
		typelist[pos] = typelist[pos+1] = MPI_INT;
		MPI_Address(&inNdArr_[cnt].getBr()[0], &address);
		displacements[pos] = address;
		MPI_Address(&inNdArr_[cnt].getBr()[1], &address);
		displacements[pos+1] = address;
	}

	// pack the log likelihood value
	typelist[ data_size - 1] = MPI_DOUBLE;
	MPI_Address(&logLi_, &address);
	displacements[data_size - 1] = address;

	// build the struct
	MPI_Type_struct(data_size, block_lengths, displacements, typelist, &pack_data_type);

	// commit the data type to MPI
	MPI_Type_commit(&pack_data_type);
	
	delete typelist;
	delete displacements;
	delete block_lengths;
}

/**
	return TRUE if the data for sending is already packed
*/
bool UrTreePar::isDataPacked() {
	return pack_data_type != 0;
}

/**
	send the packed data to a specific process
*/
void UrTreePar::sendToProcess(int process, bool nonblock_comm) {
	if (!isDataPacked())
		packData();
	//cout << mpi_myrank<< ": sendToProcess" << endl;
	// start a non blocking synchronous send
	if (nonblock_comm)
		MPI_Isend(MPI_BOTTOM, 1, pack_data_type, process, TAG_TREE, MPI_COMM_WORLD, &send_request);
	else 
		MPI_Send(MPI_BOTTOM, 1, pack_data_type, process, TAG_TREE, MPI_COMM_WORLD);
	//cout << mpi_myrank<< ": sendToProcess" << process << endl;
}

/**
	send the packed data to the master
*/
void UrTreePar::sendToMaster(bool nonblock_comm) {
	/*
		for (int cnt = 0; cnt < maxNBr_; cnt++) 
			cout << "BR " << cnt << ":"<< brArr_[cnt].smaNdNo_ <<" - " << brArr_[cnt].greNdNo_ << endl;
	*/
	//cout << mpi_myrank<< "  send to master" << endl;
	sendToProcess(mpi_master_rank, nonblock_comm);
}

/**
	receive the packed data from a specific process
*/
void UrTreePar::receiveFromProcess(int process, bool nonblock_comm) {
	if (!isDataPacked())
		packData();
	if (nonblock_comm)
		MPI_Irecv(MPI_BOTTOM, 1, pack_data_type, process, TAG_TREE, MPI_COMM_WORLD, &recv_request);
	else {
		MPI_Status status;
		int err = MPI_Recv(MPI_BOTTOM, 1, pack_data_type, process, TAG_TREE, MPI_COMM_WORLD, &status);
		if (err != 0) {
			Utl::announceError("receiveFromProcess(): Something wrong with MPI_Recv");
		}
	}
}

/**
	receive the packed data from master
*/
void UrTreePar::receiveFromMaster(bool nonblock_comm) {
	receiveFromProcess(mpi_master_rank, nonblock_comm);
}

/**
	return TRUE if master already send something, ready to unpack data
*/
bool UrTreePar::isDataArrived() {
	int flag;
	MPI_Status status;
	MPI_Test(&recv_request, &flag, &status);
	return flag != 0;
}

/**
	return TRUE if master already sent something, but process may not received completely
*/
bool UrTreePar::isReadyToReceive() {
	int flag;
	MPI_Status status;
	MPI_Iprobe(mpi_master_rank, TAG_TREE, MPI_COMM_WORLD, &flag, &status);
	return flag;
}

/**
	unpack information from the packet to rebuild the tree
*/
void UrTreePar::unpackData() {
	int cnt;

	/*
		index of branches which attach to the node
		for external node: store the only branch to this node
		for internal node: store the 3rd branch to this node
	*/
	int *node_branches = new int[maxNNd_];
	for (cnt = 0; cnt < maxNNd_; cnt++)
		node_branches[cnt] = -1;

	// now build the node_level array
	for (cnt = 0; cnt < maxNBr_; cnt ++ ) {
		brArr_[cnt].setId(cnt);
		if (brArr_[cnt].smaNdNo_ < maxNExNd_ || brArr_[cnt].greNdNo_ < maxNExNd_)
			brArr_[cnt].turnOnIsEx();
		else
			brArr_[cnt].turnOffIsEx();
		//cout << "branch "<< cnt << " : " << brArr_[cnt].smaNdNo_ << " - " << brArr_[cnt].greNdNo_ << endl;

		// process the 2 nodes of this branch
		for (int i = 0; i < 2; i++) {
			int node = (i==0) ? brArr_[cnt].smaNdNo_ : brArr_[cnt].greNdNo_;
			if (node < 0 || node >= maxNNd_)
				Utl::announceError("unpackData(): node < 0");
			if ( node < maxNExNd_) // external node
				node_branches[node] = cnt;
			else { // internal node
				// ignore the 2 first branches
				if (cnt != inNdArr_[node].getBr(0) && cnt != inNdArr_[node].getBr(1))
					node_branches[node] = cnt;
			}
		}

	}

	bool negative_nb = false;
	for (cnt = 0; cnt < maxNNd_; cnt++)
		if (node_branches[cnt] == -1) {
			cout << mpi_myrank << ": node_branches[" << cnt << "] == -1" << endl;
			negative_nb = true;
		}
	if (negative_nb)
		Utl::announceError("unpackData()");

	// modify suitable internal node array and external node array
	//if (isMasterProc())
	//cout << "FROM PROCESS " << mpi_myrank << endl;
	for (cnt = 0; cnt < maxNNd_; cnt ++) {
		//if (isMasterProc())
		//cout << "Node " << cnt << " with br " << node_branches[cnt] << endl;
		if (cnt < maxNExNd_) { // in case of external node
			//cout << mpi_myrank<< ": ex unpack " << cnt << endl;
			exNdArr_[cnt].setId(cnt);
			if (cnt == brArr_[node_branches[cnt]].smaNdNo_)
				exNdArr_[cnt].setInNd(brArr_[node_branches[cnt]].greNdNo_);
			else
				exNdArr_[cnt].setInNd(brArr_[node_branches[cnt]].smaNdNo_);

			exNdArr_[cnt].setBr(node_branches[cnt]);

		} else { // in case of internal node
			//cout << mpi_myrank<< ": in unpack " << cnt << " level " << node_level[cnt] << endl;
			inNdArr_[cnt].setId(cnt);
			// process branches of this node
			for (int branch = 0; branch < 3; branch++) {
				//if (isMasterProc())
				//cout << "Branch = " << branch << endl;
				// take the 2 first branches from innode, take the 3rd branch from
				// node_branches
				int brNo = (branch < 2) ? inNdArr_[cnt].getBr(branch)
				           : node_branches[cnt];
				// set the 3rd branch of this internal node
				if (branch == 2)
					inNdArr_[cnt].getBr()[branch] = brNo;

				if (cnt == brArr_[brNo].smaNdNo_)
					inNdArr_[cnt].getNeiNd()[branch] = brArr_[brNo].greNdNo_;
				else
					inNdArr_[cnt].getNeiNd()[branch] = brArr_[brNo].smaNdNo_;

			}
		}
	}
	//cout << mpi_myrank<< ": unpack" << endl;
	//if (isMasterProc())
	//cout << "Unpack Finish" << endl;

	delete node_branches;
	
}

/**
	return the send request
*/
MPI_Request *UrTreePar::getSendRequest() {
	return &send_request;
}

/**
	return the receive request
*/
MPI_Request *UrTreePar::getReceiveRequest() {
	return &recv_request;
}


/**
	cancel all the attached receive or send
*/
void UrTreePar::cancelCommunications() {
	cancelSend();
	cancelReceive();
}

/**
	cancel the send operation
*/
void UrTreePar::cancelSend() {
	if (send_request != MPI_REQUEST_NULL)
		MPI_Cancel(&send_request);
}

/**
	cancel the receive operation
*/
void UrTreePar::cancelReceive() {
	if(recv_request != MPI_REQUEST_NULL)
		MPI_Cancel(&recv_request);
}

/**
	wait for the completion of receive request
*/
void UrTreePar::waitReceive() {
	MPI_Status status;
	int flag;
	MPI_Test(&recv_request, &flag, &status);
	const int MAX_WAIT_COUNT = 6000; // max 10 minutes for waiting
	int count;
	for (count = 0; !flag && count < MAX_WAIT_COUNT; count++) {
		// sleep just for 0.1 seconds before testing again
		usleep(100000);
		if (stopSignalCome()) break;
		MPI_Test(&recv_request, &flag, &status);
	}
	if (!flag && !stopSignalCome()) {
		Utl::announceError("Waited too long for master to communicate");
	}
}


#endif // PARALLEL


UrTreePar::~UrTreePar() {}

