/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/***************************************************************************
                          model.h  -  description
                             -------------------
    begin                : Wed Mar 5 2003
    copyright            : (C) 2003 by 
    email                : vinh@cs.uni-duesseldorf.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MODEL_H
#define MODEL_H

#include "constant.h"
#include "mat.h"
#include "basem.h"

#include "gtrm.h"

#include "hky85m.h"
#include "tn93m.h"
#include "codonmodel.h"
#include "codonny98.h"
#include "codonyn98.h"
#include "codonpedersen98.h"
#include "codongtr.h"
#include "codonposrate.h"

#include "aminom.h"
#include "dvec20.h"
#include "dmat20.h"

/**
	this class manages all interface for all models
*/
class Model {
public:
	/**
		the constructor function
	*/
	Model();

	/**
		everything is inited here
	*/
	void init ();

	/**
		get the number of states, 4 for DNA, 20 for protein, 61 for codon
		@return the number of states
	*/
	int getNState ();

	/**
		set the number of states, 4 for DNA, 20 for protein, 61 for codon
		@param nState the number of states
	*/
	void setNState (int nState);


	/**
		set the type of evolutionary change model
		@param modelType type of model (HKY85, TN93, WAG,...)
	*/
	void setModelType (const MODEL modelType);

	/**
		get the type of evolutionary change model
		@return type of model 
	*/
	MODEL getModelType ();

	/**
		get the type of evolutionary change model
		@return type of model 
	*/
	MODEL getType ();

	/**
		set the type of sequences: NUCLEOTIDE, AMINO_ACID, or CODON
		@param dataType type of sequences
	*/
	void setDataType (DATA_TYPE dataType);

	/**
		@return the type of sequences: NUCLEOTIDE, AMINO_ACID, or CODON
	*/
	DATA_TYPE getDataType ();

	/**
		set the Transition/Transversion ratio type for HKY85 model: ESTIMATE or USER_DEFINED
		@param tsTvRatioType the Transition/Transversion ratio type 
	*/
	void setTsTvRatioType (PAM_TYPE tsTvRatioType);

	/**
		get the Transition/Transversion ratio type for HKY85 model: ESTIMATE or USER_DEFINED
		@return the Transition/Transversion ratio type 
	*/
	PAM_TYPE getTsTvRatioType ();

	/**
		set the transition transvertion ratio for HKY85 model
		@param tsTvRatio the Transition/Transversion ratio 
	*/
	void setTsTvRatio (const double tsTvRatio);

	/**
		get the transition transvertion ratio for HKY85 model
		@return the Transition/Transversion ratio 
	*/
	double getTsTvRatio ();

	/**
		set the purine pyrimidine/purine ratio type for TN93 model: ESTIMATE or USER_DEFINED
		@param pyPuRatioType the purine pyrimidine/purine ratio type
	*/
	void setPyPuRatioType (PAM_TYPE pyPuRatioType);

	/**
		get the purine pyrimidine/purine ratio type for TN93 model: ESTIMATE or USER_DEFINED
		@return the purine pyrimidine/purine ratio type
	*/
	PAM_TYPE getPyPuRatioType ();


	/**
		set the purine pyrimidine/purine ratio for TN93 model
		@param pyPuRatio the purine pyrimidine/purine ratio
	*/
	void setPyPuRatio (const double pyPuRatio);


	/**
		get the purine pyrimidine/purine ratio for TN93 model
		@return the purine pyrimidine/purine ratio
	*/
	double getPyPuRatio ();

	/**
		set the type of substitution parameters for GTR model: ESTIMATE or USER_DEFINED
		@param genPamType type of substitution parameters
		
	*/
	void setGenPamType (PAM_TYPE genPamType);

	/**
		get the type of substitution parameters for GTR model: ESTIMATE or USER_DEFINED
		@return type of substitution parameters
		
	*/
	PAM_TYPE getGenPamType ();

	/**
		set the substitution parameters for GTR model
		@param tsAG Transition   rate from A to G
		@param tsCT Transition   rate from C to T
		@param tvAT Transversion rate from A to T
		@param tvAC Transversion rate from A to C
		@param tvCG Transversion rate from C to G
		@param tvGT Transversion rate from G to T
	*/
	void setGenPam (const double tsAG, const double tsCT,
	                const double tvAC, const double tvAT, const double tvCG, const double tvGT);

	/**
		get the substitution parameters for GTR model
		@param tsAG (OUT) Transition   rate from A to G
		@param tsCT (OUT) Transition   rate from C to T
		@param tvAT (OUT) Transversion rate from A to T
		@param tvAC (OUT) Transversion rate from A to C
		@param tvCG (OUT) Transversion rate from C to G
		@param tvGT (OUT) Transversion rate from G to T
	*/
	void getGenPam (double &tsAG, double & tsCT, double & tvAC, 
		double & tvAT, double & tvCG, double & tvGT);

	/**
		set the type of state frequencies: ESTIMATE, EQUAL, or USER_DEFINED
		@param baseFrqType type of state frequencies
	*/
	void setBaseFrqType (PAM_TYPE baseFrqType);
	
	/**
		get the type of state frequencies: ESTIMATE, EQUAL, or USER_DEFINED
		@return type of state frequencies
	*/
	PAM_TYPE getBaseFrqType ();

	/**
		@return true if the model has parameters to estimate
	*/
	int isEstedPam ();

	/**
		@return true if codon model is specified
	*/
	bool isCodonAnalysisNext();

	/**
		set the state frequencies for A, C, G, T in nucleotide model
		@param stateFrqA frequency of A
		@param stateFrqC frequency of C
		@param stateFrqG frequency of G
		@param stateFrqT frequency of T
	*/	
	void setStateFrq (double stateFrqA, double stateFrqC, double stateFrqG, double stateFrqT);

	/**
		get the state frequencies 
		@param stateFrqArr (OUT) the state frequencies
	*/
	void getStateFrq (DVec20 &stateFrqArr);

	/**
		get the frequency of stateNo
		@param stateNo the state Nr.
	*/
	double getStateFrq (const int stateNo);

	/**
		get the propotion of a class
		@param classNo class Nr.
		@return the propotion of classNo
	*/
	inline double getClassProb(int classNo) {
		return ((CodonNY98*)model_) -> nsSyProbVec[classNo];
	}

	/**
		get the ratio parm of a class
		@param classNo class Nr.
		@return ratio paramater of classNo		
	*/
	double getClassRatio(int classNo);

	/**
		optimize all parameter evolving to sequence evolution
		1. tsTvRatio
		
		2. TsRatio
		3. find state frequencies
		@return TODO
	*/
	int optPam ();

	/******************************************************
	* compute the probability changes 
	*******************************************************/

	/**
		compute the probability change matrix P(t) = exp{Qt}
		@param time the evolution time
		@param probMat (OUT) the change matrix
	*/
	void cmpProbChange (const double time, DMat20 &probMat);

	/**
		compute the probability change matrix, for codon-based Nielsen Yang 98 Model
		@param time the evolution time
		@param classNo the class Nr.
		@param probMat (OUT) the change matrix
	*/
	void cmpProbChange (const double time, const int classNo, DMat20 &probMat);
	

	/**
		compute the probability change from stateNo1 into stateNo2 after a period time of time
		@param time the evolution time
		@param stateNo1 the state nr. 1
		@param stateNo2 the state nr. 1
		@return the probability change from stateNo1 into stateNo2
	*/
	double cmpProbChange (const int stateNo1, const int stateNo2, const double time);

	/**
		compute the 1st and 2nd derivatives of probability change matrix
		@param time the evolution time
		@param probMat (OUT) the change matrix
		@param derv1 (OUT) the 1st derivative
		@param derv2 (OUT) the 2nd derivative
	*/
	void cmpProbChangeDerivatives (const double time, DMat20 &probMat, DMat20 &derv1, DMat20 &derv2);


	/**
		compute the 1st and 2nd derivatives of probability change matrix for codon model
		@param time the evolution time
		@param classNo the class Nr.
		@param probMat (OUT) the change matrix
		@param derv1 (OUT) the 1st derivative
		@param derv2 (OUT) the 2nd derivative
	*/
	void cmpProbChangeDerivatives (const double time, int classNo, DMat20 &probMat,
        DMat20 &derv1, DMat20 &derv2);
		

	/**
		set the iteration step
		@param step  iteration step
	*/
	void setPamStep(int step);
	
	/**
		get the iteration step
		@return  iteration step
	*/
	int getPamStep();
	
		
	/**
		release all memmory for this class
	*/
	void release ();

	/**
		the destructor
	*/
	virtual ~Model ();

	/************************************************************************************
	************************************************************************************
	************************************************************************************
	************************************************************************************
	************************************************************************************/

	/**
		model type (HKY85, WAG,...)
	*/
	MODEL modelType_;

	/**
		state frequencies type
	*/
	PAM_TYPE baseFrqType_;

	/**
		the Transition/Transversion ratio type for HKY85 model
	*/
	PAM_TYPE tsTvRatioType_;

	/**
		 the purine pyrimidine/purine ratio type for TN93 model
	*/
	PAM_TYPE pyPuRatioType_;

	/**
		type of substitution parameters for GTR model
	*/
	PAM_TYPE genPamType_;


	/**
		the Transition/Transversion ratio for HKY85 model
	*/
	double tsTvRatio_;


	/**
		 the purine pyrimidine/purine ratio for TN93 model
	*/
	double pyPuRatio_;
	
	/**
		Transition   rate from A to G
	*/
	double tsAG_;

	/**
		Transition   rate from C to T
	*/
	double tsCT_;

	/**
		Transversion rate from A to T
	*/
	double tvAT_;

	/**
		Transversion rate from A to C
	*/
	double tvAC_;

	/**
		Transversion rate from C to G
	*/
	double tvCG_;

	/**
		Transversion rate from G to T
	*/
	double tvGT_;

	/**
		codon model (YN98, NY98, ...)
	*/
	MODEL 		codon_model;

	/**
		frequencies type for codon model
	*/
    PAM_TYPE    codonFrqType;

	/**
		the Transition/Transversion ratio for Codon model
	*/
    double      codon_tsTvRatio;

	/**
		the Transition/Transversion ratio type for Codon model
	*/
    PAM_TYPE    codon_tsTvRatioType;

	/**
		the state frequency list
	*/
	DVec20 stateFrqArr_;

	/**
		number of states
	*/
	int nState_;

	/**
		type of sequences
	*/
	DATA_TYPE dataType_;
	
	/**
		true of want to use POWELL method
	*/
	int pam_brent;


	/**
		the model here!!!!
	*/
	BaseM     *model_;

	/**
		parameter estimation step
	*/
	int nPamStep;

	/**
		TODO
	*/
	bool initialized_tsTv;
	
	/**
		rate file name if load model from file
	*/
	string rate_file;


	/**
		true if all pair (AP) method for site-specific rate
	*/
	bool ap_sitespec;

}
; //end of class model


extern Model mymodel;

#endif
