/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Fri Apr 11 10:08:35 CEST 2003
    copyright            : (C) 2003 by
    email                : vinh@cs.uni-duesseldorf.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include "constant.h"
#include "utl.h"
#include "vec.h"
#include "mat.h"
#include "cube.h"
#include "seq.h"
#include "ali.h"
#include "ptn.h"
#include "ptnls.h"
#include "brent.h"
#include "exnd.h"
#include "exndarr.h"
#include "innd.h"
#include "inndarr.h"
#include "br.h"
#include "brarr.h"
#include "basem.h"
#include "model.h"
#include "optpairseq.h"
#include "rate.h"
#include "opturtree.h"
#include "urtree.h"
#include "outstream.h"
#include "testmethod.h"
#include "bionj.h"
#include "interface.h"
#include "tn93m.h"
#include "usertree.h"

#ifdef PARALLEL
#	include <mpi.h>
#endif // PARALLEL

int isContinuous = 0;


extern ofstream nni_file;
extern ofstream nni_tree;
extern int nni_lh_iter;

void init () {
	opt_urtree.doConstructor ();
	opt_pairseq.doConstructor ();
	alignment.doConstructor ();
	ptnlist.doConstructor ();
	createPosGenChar ();
}


//overload the operator <<
std::ostream &operator << (std::ostream &out, Vec<char> &vec) {
	return OutStream::write (vec, out);
}

//overload the operator <<
std::ofstream &operator << (std::ofstream &out, Vec<char> &vec) {

	return OutStream::write (vec, out);
}

#ifdef PARALLEL
int mpi_myrank;
int mpi_size;
int mpi_master_rank = 0;

long p_randn;
long p_rand;

#endif

void Finalize(int exit_code) {
	if (nni_lh_iter) {
		nni_file.close();
		nni_tree.close();
	}

#ifdef PARALLEL
	MPI_Finalize();
#endif
	if (isMasterProc())
		cout << "Finished!!!" << endl;
	exit(exit_code);
}

void buildBootstrapConsensus(CommonParameters &params) {
	ClusterArr allCluArr_;
	allCluArr_.setupIndex ();
	Vec<char> conTree_;
	string contree_file_name;
	string boottree_file_name;
	getOutFileName(SUF_CONTREE, contree_file_name);
	getOutFileName(SUF_BOOTTREE, boottree_file_name);

	cout << "Building bootstrap consensus tree..." << endl;

	ifstream in;
	in.open (boottree_file_name.c_str());
	if (in == 0)
		Utl::announceError ("Cannot open the user tree file ...");

	int num_tree = 0;

	while (!in.eof () ) {
		while (!in.eof() && in.peek() < 32) {
			in.get();
		}
		if (in.eof()) break;
		UserTree tree_;
		tree_.readFile (in);
		tree_.createUrTree ();

		ClusterArr newClusterArr_;
		tree_.createCluster(newClusterArr_);
		for (int count_ = 0; count_ < newClusterArr_.totalNCluster_; count_ ++)
			allCluArr_.addConTree(newClusterArr_[count_]);
		num_tree++;
	}

	cout << "Loaded " << num_tree << " bootstrap trees" << endl;

	allCluArr_.createConTree(num_tree, conTree_);

	ofstream conFile_(contree_file_name.c_str());
	OutStream::write (conTree_, conFile_) << endl;
	conFile_.close();


	cout << "The results were written to following files:" << endl;
	cout << "   1. " << boottree_file_name << " (all bootstrap trees)" << endl;
	cout << "   2. " << contree_file_name << " (consensus tree)" << endl;
	if (params.print_splits) {
		string splits_file;
		getOutFileName(SUF_SPLITS, splits_file);
		allCluArr_.printSplits(num_tree, params.outGrpSeqNo, splits_file.c_str());
		cout << "   3. " << splits_file << " (all splits)" << endl;
	}

}

//int random_seed = -1;

int main(int argc, char *argv[]) {

#ifdef PARALLEL
	MPI_Init(&argc, &argv);
	MPI_Comm_rank(MPI_COMM_WORLD, &mpi_myrank);
	MPI_Comm_size(MPI_COMM_WORLD, &mpi_size);
	//cout << " Process " << mpi_myrank << " of " << mpi_size << endl;
#endif

	CommonParameters params;
	InputParameters in_pam;

	initializeParams(params);
	initInputParams(in_pam);
	
	Interface inter_;


	//char aliFileName_[FILENAME_LEN] = "";

	init ();

	//char userTreeFileName_[FILENAME_LEN]   = "";

	if (isMasterProc()) {
#ifdef USE_LONG_DOUBLE
		cout << "Use long double (" << sizeof(long double)
			<< ") for numerical operations." << endl;
#else
		//cout << "Use double as numerical operations." << endl;
#endif
	}


	int status_ = inter_.getUserPam (argc, argv, params, in_pam);
	if (status_ == 0)
		Finalize(0);

	if (params.gen_bootstrap && params.codon_model != UNDEF_MODEL) {
		cout << "Bootstrap with codon model is currently not supported" << endl;
		Finalize(0);
	}

	if (in_pam.nni_lh_record && isMasterProc()) {
		nni_lh_iter = 1;
		string nni_file_name;
		string nni_tree_name;
		getOutFileName(SUF_NNI_LH, nni_file_name);
		getOutFileName(SUF_NNI_TREE, nni_tree_name);
		nni_file.open(nni_file_name.c_str());
		nni_tree.open(nni_tree_name.c_str());
		//nni_file.open("/dev/stdout");
		nni_file.precision(10);
		nni_tree.precision(10);
	}

	time_t beginTime_;
	time (&beginTime_);

	char *startedDate_;
	startedDate_ = ctime(&beginTime_);
	if (isMasterProc())
		std::cout << "Date and time: " << startedDate_;


	int num_loop = (params.gen_bootstrap)? params.gen_bootstrap : 1;


	while (params.cur_bootstrap < num_loop) {

		if (isMasterProc() && params.gen_bootstrap) {
			if (isContinuous == 0)
				cout << endl << "===> START BOOTSTRAP REPLICATE NUMBER " <<  params.cur_bootstrap+1 << endl;
			else
				cout << endl << "===> RESUME FROM INTERRUPTED BOOTSTRAP REPLICATE NUMBER " <<  params.cur_bootstrap+1 << endl;
		}


		if (isContinuous == 0 && params.gen_bootstrap) {
			bool iserror = false;
			if (isMasterProc()) {
				cout << "Creating bootstrap sample alignment...." << endl;
				string bootfname;
				getOutFileName(SUF_BOOTSAMPLE, bootfname);
				alignment.generateBootstrap();
				ofstream out(bootfname.c_str());
				if (out.is_open()) {
					alignment.write(out);
					out.close();
					cout << "Bootstrap sample was written to " << bootfname.c_str() << endl;
				} else {
					cout << "ERROR: Cannot write bootstrap sample to " << bootfname.c_str() << endl;
					iserror = true;
				}
			}
#ifdef PARALLEL
			// all slaves wait until master finish creating bootstrap alignment
			MPI_Barrier(MPI_COMM_WORLD);
#endif
			if (iserror)
				Finalize(0);
			if (isSlaveProc()) {
				// now each slave read in the bootstrap alignment
				alignment.loadBootstrap();
			}
		}

		if (isContinuous == 1 && params.gen_bootstrap) {
			// continue from interuption, load the bootstrap sample alignment
			alignment.loadBootstrap();
		}
	
	
	
		
		alignment.convertDataType(params.dataType);
		opt_urtree.allocMemory();
	
		TestMethod testMethod_;
	
		testMethod_.create (params, in_pam, beginTime_);
	
		opt_urtree.release();

		params.cur_bootstrap++;
		isContinuous = 0;

		string checkPointFileName_;
		getOutFileName(SUF_CHECKPOINT, checkPointFileName_);
		int isFinished_ = 1;
		if (params.gen_bootstrap && params.cur_bootstrap < params.gen_bootstrap-1)
			isFinished_ = 0;
		if (isMasterProc())
			writePamCheckPoint(params, checkPointFileName_.c_str(), isFinished_);
	

	
	}

	if (isMasterProc() && params.gen_bootstrap) {

		cout << endl << "===> ALL BOOTSTRAP REPLCATES DONE" << endl;

		buildBootstrapConsensus(params);
	}

	if (isMasterProc()) {
		time_t finishedTime_;
		time (&finishedTime_);
		char *finishedDate_;
		finishedDate_ = ctime(&finishedTime_);

		params.progTime = difftime (finishedTime_, beginTime_);

		int hour_ = static_cast<int> (params.progTime /3600.0);
		int min_ =  static_cast<int> ( (params.progTime - hour_ * 3600)/60.0);
		int sec_ =  static_cast<int> ( params.progTime - hour_ * 3600 - min_ * 60);
		cout << "Total Runtime: " << hour_ << "h:" << min_ << "m:" << sec_ << "s" << endl;
		if (isMasterProc())
			std::cout << "Date and time: " << finishedDate_;
	}

	Finalize(EXIT_SUCCESS);
	return 0;
}

