#include "../common/Spectrum.hpp"
#include <cmath>
#include <limits>
#include <iostream>

constexpr auto resampleWavelengthTolerance=1e-14;
constexpr auto integrationTolerance=1e-10;
#define FAIL(details) { std::cerr << __FILE__ << ":" << __LINE__  << ": test failed: " << details << "\n"; return 1; }

int test1(Spectrum const& spectrum, Spectrum const& reference, const double minWL, const double maxWL, const int pointCount)
{
    const auto resampled=spectrum.resample(minWL,maxWL,pointCount);
    if(resampled.size()!=reference.size())
        FAIL("resampled size doesn't match reference size");
    for(unsigned i=0; i<resampled.size(); ++i)
    {
        if(std::abs(resampled.wavelengths[i]-reference.wavelengths[i])/reference.wavelengths[i] > resampleWavelengthTolerance)
        {
            FAIL("resampled wavelengths #" << i << " don't match reference: expected " << reference.wavelengths[i] << ", actual " << resampled.wavelengths[i]);
            break;
        }
    }
    for(unsigned i=0; i<resampled.size(); ++i)
    {
        if(std::abs(resampled.values[i]-reference.values[i])/reference.values[i] > integrationTolerance)
        {
            FAIL("resampled values #" << i << " don't match reference: expected " << reference.values[i] << ", actual " << resampled.values[i]);
            break;
        }
    }
    return 0;
}

int main(int argc, char** argv)
try
{
    if(argc!=2)
    {
        std::cerr << "Which test to run?\n";
        return 1;
    }
    std::size_t pos=0;
    const std::string arg=argv[1];
    const auto whichTest=std::stoul(arg, &pos);
    if(pos!=arg.size())
    {
        std::cerr << "Bad test number " << arg << "\n";
        return 1;
    }

    std::cerr.precision(std::numeric_limits<double>::max_digits10);

    const Spectrum spect1{{302, 304, 306, 308, 310, 312, 314, 316, 318, 320, 322, 324, 326, 328, 330, 332, 334, 336,
        338, 340, 342, 344, 346, 348, 350, 352, 354, 356, 358, 360, 362, 364, 366, 368, 370, 372, 374, 376, 378, 380,
        382, 384, 386, 388, 390, 392, 394, 396, 398, 400, 402, 404, 406, 408, 410, 412, 414, 416, 418, 420, 422, 424,
        426, 428, 430, 432, 434, 436, 438, 440, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468,
        470, 472, 474, 476, 478, 480, 482, 484, 486, 488, 490, 492, 494, 496, 498, 500, 502, 504, 506, 508, 510, 512,
        514, 516, 518, 520, 522, 524, 526, 528, 530, 532, 534, 536, 538, 540, 542, 544, 546, 548, 550, 552, 554, 556,
        558, 560, 562, 564, 566, 568, 570, 572, 574, 576, 578, 580, 582, 584, 586, 588, 590, 592, 594, 596, 598, 600,
        602, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 644,
        646, 648, 650, 652, 654, 656, 658, 660, 662, 664, 666, 668, 670, 672, 674, 676, 678, 680, 682, 684, 686, 688,
        690, 692, 694, 696, 698, 700, 702, 704, 706, 708, 710, 712, 714, 716, 718, 720, 722, 724, 726, 728, 730, 732,
        734, 736, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776,
        778, 780, 782, 784, 786, 788, 790, 792, 794, 796, 798, 800, 820, 840, 860, 880, 900, 920, 940, 960, 980, 1000,
        1020, 1040, 1060, 1080, 1100, 1120, 1140, 1160, 1180, 1200, 1220, 1240, 1260, 1280, 1300, 1320, 1340, 1360,
        1380, 1400, 1420, 1440, 1460, 1480, 1500, 1520, 1540, 1560, 1580, 1600, 1620, 1640, 1660, 1680, 1700, 1720,
        1740, 1760, 1780, 1800, 1820, 1840, 1860, 1880, 1900, 1920, 1940, 1960, 1980, 2000, 2020, 2040, 2060, 2080,
        2100, 2120, 2140, 2160, 2180, 2200, 2220, 2240, 2260, 2280, 2300, 2320, 2340, 2360, 2380, 2400, 2420, 2440,
        2460, 2480, 2500}, {0.029582, 0.029759, 0.029935, 0.030112, 0.030288, 0.030465, 0.030641, 0.030818, 0.030994,
        0.031171, 0.031347, 0.031523, 0.0317, 0.031877, 0.032053, 0.032229, 0.032406, 0.032582, 0.032759, 0.032935,
        0.033112, 0.033288, 0.033465, 0.033641, 0.033818, 0.033994, 0.034171, 0.034347, 0.034524, 0.0347, 0.034876,
        0.035053, 0.035229, 0.035406, 0.035582, 0.035759, 0.035935, 0.036112, 0.036288, 0.036465, 0.036641,
        0.036818, 0.036994, 0.037171, 0.037347, 0.037524, 0.0377, 0.037876, 0.038053, 0.038229, 0.038406, 0.038533,
        0.03862, 0.038773, 0.038887, 0.038945, 0.039064, 0.039233, 0.039298, 0.039397, 0.03953, 0.039714, 0.039794,
        0.039864, 0.04014, 0.040143, 0.04029, 0.040321, 0.040495, 0.040597, 0.040698, 0.040786, 0.040926, 0.041072,
        0.041194, 0.041294, 0.041362, 0.041487, 0.041629, 0.04179, 0.04189, 0.041996, 0.042063, 0.042185, 0.042327,
        0.042435, 0.042597, 0.042674, 0.042768, 0.042894, 0.043055, 0.0431, 0.043268, 0.043348, 0.043485, 0.043605,
        0.043706, 0.04399, 0.044595, 0.045412, 0.046444, 0.047634, 0.048893, 0.050719, 0.053314, 0.056175, 0.059142,
        0.062804, 0.0671, 0.071183, 0.075386, 0.080256, 0.084949, 0.089023, 0.092914, 0.096696, 0.099535, 0.10169,
        0.10353, 0.10528, 0.1068, 0.10811, 0.10912, 0.1098, 0.11022, 0.11064, 0.1111, 0.11123, 0.11076, 0.10975,
        0.10843, 0.10682, 0.10507, 0.1029, 0.10024, 0.097302, 0.094614, 0.091871, 0.08903, 0.086963, 0.085306,
        0.083636, 0.081922, 0.080257, 0.078833, 0.07765, 0.076732, 0.075929, 0.075385, 0.074901, 0.074273, 0.073543,
        0.072815, 0.072092, 0.071235, 0.070351, 0.06938, 0.068276, 0.067068, 0.065917, 0.06495, 0.064216, 0.063698,
        0.063276, 0.062904, 0.062538, 0.062075, 0.06146, 0.060592, 0.0595, 0.058202, 0.056705, 0.055117, 0.05356,
        0.05212, 0.051095, 0.05046, 0.049945, 0.049367, 0.04872, 0.047927, 0.04712, 0.046474, 0.04571, 0.04508,
        0.04474, 0.044548, 0.04437, 0.044425, 0.04489, 0.046093, 0.04799, 0.050242, 0.052782, 0.056244, 0.061872,
        0.069404, 0.077736, 0.087253, 0.098682, 0.11099, 0.1249, 0.14113, 0.15674, 0.17197, 0.18675, 0.20243,
        0.21793, 0.23376, 0.25036, 0.26632, 0.28417, 0.30119, 0.31709, 0.33283, 0.34914, 0.36304, 0.3755, 0.38774,
        0.39914, 0.41046, 0.42119, 0.42954, 0.43695, 0.44385, 0.44958, 0.45419, 0.45802, 0.46111, 0.46323, 0.4653,
        0.46748, 0.46966, 0.47151, 0.47291, 0.47387, 0.47493, 0.47573, 0.4761, 0.47652, 0.47723, 0.47798, 0.4788,
        0.4796, 0.48016, 0.48067, 0.48147, 0.48245, 0.48347, 0.48432, 0.48909, 0.49366, 0.49974, 0.5042, 0.5079,
        0.51061, 0.50744, 0.49742, 0.49516, 0.50115, 0.50945, 0.51813, 0.52487, 0.52411, 0.52243, 0.51859, 0.50372,
        0.47896, 0.46524, 0.46688, 0.47635, 0.48812, 0.48922, 0.48581, 0.4764, 0.45359, 0.419, 0.38955, 0.32801,
        0.23055, 0.16814, 0.14975, 0.14648, 0.15832, 0.1798, 0.20617, 0.2315, 0.25381, 0.27655, 0.29285, 0.30674,
        0.31808, 0.32257, 0.32579, 0.32332, 0.31501, 0.3048, 0.28774, 0.27768, 0.2765, 0.27127, 0.25853, 0.21605,
        0.13258, 0.072879, 0.051658, 0.050554, 0.055539, 0.065224, 0.073972, 0.087469, 0.097152, 0.10898, 0.12315,
        0.13624, 0.14881, 0.1576, 0.16431, 0.16902, 0.1709, 0.17387, 0.17091, 0.16356, 0.15572, 0.14609, 0.13389,
        0.12382, 0.11704, 0.10618, 0.097125, 0.08753, 0.079555, 0.067907, 0.06236, 0.062297}};
    switch(whichTest)
    {
    case 1:
        return test1(spect1, Spectrum{{360., 391.333333333333, 422.666666666667, 454., 485.333333333333, 516.666666666667, 548.,
                579.333333333333, 610.666666666667, 642., 673.333333333333, 704.666666666667, 736., 767.333333333333,
                798.666666666667, 830.}, {0.0347000000000001, 0.0374525319148937, 0.0395982101063831, 0.0413985248226951,
                0.0433581081560285, 0.0670410913120567, 0.107474800531915, 0.0895826374113473, 0.0703972881205673,
                0.057270329787234, 0.0468806241134751, 0.138158718085106, 0.367105611702127, 0.468352606382978,
                0.483183566489361, 0.491408562943262}}
              ,
              360,830,16);
        break;
    case 2:
        return test1(spect1, Spectrum{{360., 375.666666666667, 391.333333333333, 407., 422.666666666667, 438.333333333333, 454., 469.666666666667,
                485.333333333333, 501., 516.666666666667, 532.333333333333, 548., 563.666666666667, 579.333333333333, 595.,
                610.666666666667, 626.333333333333, 642., 657.666666666667, 673.333333333333, 689., 704.666666666667,
                720.333333333333, 736., 751.666666666667, 767.333333333333, 783., 798.666666666667, 814.333333333333, 830.},
                {0.0347, 0.036082329787234, 0.0374647655141844, 0.0386828054078014, 0.0395969051418439, 0.0404985846631205,
                0.0414044051418439, 0.0423073306737588, 0.0432100846631205, 0.0464349884751773, 0.0650354015957448,
                0.095702930407802, 0.109276298758866, 0.106351583333334, 0.0884621954787239, 0.0767442974290785,
                0.0707501892730501, 0.0638949255319153, 0.0577933257978727, 0.0493583803191493, 0.045010748670213,
                0.0577998661347523, 0.132493962322696, 0.253708253546101, 0.373461609042556, 0.446140935283691,
                0.470135726950358, 0.477757956560287, 0.483414666223407, 0.487737001773053, 0.491375000000003}}
              ,
              360,830,31);
        break;
    case 3:
        return test1(spect1, Spectrum{{360., 361.571906354515, 363.14381270903, 364.715719063545, 366.28762541806,
              367.859531772575, 369.43143812709, 371.003344481605, 372.57525083612, 374.147157190635, 375.719063545151,
              377.290969899666, 378.862876254181, 380.434782608696, 382.006688963211, 383.578595317726, 385.150501672241,
              386.722408026756, 388.294314381271, 389.866220735786, 391.438127090301, 393.010033444816, 394.581939799331,
              396.153846153846, 397.725752508361, 399.297658862876, 400.869565217391, 402.441471571906, 404.013377926421,
              405.585284280936, 407.157190635452, 408.729096989967, 410.301003344482, 411.872909698997, 413.444816053512,
              415.016722408027, 416.588628762542, 418.160535117057, 419.732441471572, 421.304347826087, 422.876254180602,
              424.448160535117, 426.020066889632, 427.591973244147, 429.163879598662, 430.735785953177, 432.307692307692,
              433.879598662207, 435.451505016722, 437.023411371237, 438.595317725753, 440.167224080268, 441.739130434783,
              443.311036789298, 444.882943143813, 446.454849498328, 448.026755852843, 449.598662207358, 451.170568561873,
              452.742474916388, 454.314381270903, 455.886287625418, 457.458193979933, 459.030100334448, 460.602006688963,
              462.173913043478, 463.745819397993, 465.317725752508, 466.889632107023, 468.461538461538, 470.033444816054,
              471.605351170569, 473.177257525084, 474.749163879599, 476.321070234114, 477.892976588629, 479.464882943144,
              481.036789297659, 482.608695652174, 484.180602006689, 485.752508361204, 487.324414715719, 488.896321070234,
              490.468227424749, 492.040133779264, 493.612040133779, 495.183946488294, 496.755852842809, 498.327759197324,
              499.899665551839, 501.471571906355, 503.04347826087, 504.615384615385, 506.1872909699, 507.759197324415,
              509.33110367893, 510.903010033445, 512.47491638796, 514.046822742475, 515.61872909699, 517.190635451505,
              518.76254180602, 520.334448160535, 521.90635451505, 523.478260869565, 525.05016722408, 526.622073578595,
              528.19397993311, 529.765886287625, 531.337792642141, 532.909698996656, 534.481605351171, 536.053511705686,
              537.625418060201, 539.197324414716, 540.769230769231, 542.341137123746, 543.913043478261, 545.484949832776,
              547.056856187291, 548.628762541806, 550.200668896321, 551.772575250836, 553.344481605351, 554.916387959866,
              556.488294314381, 558.060200668896, 559.632107023411, 561.204013377927, 562.775919732442, 564.347826086957,
              565.919732441472, 567.491638795987, 569.063545150502, 570.635451505017, 572.207357859532, 573.779264214047,
              575.351170568562, 576.923076923077, 578.494983277592, 580.066889632107, 581.638795986622, 583.210702341137,
              584.782608695652, 586.354515050167, 587.926421404682, 589.498327759197, 591.070234113712, 592.642140468228,
              594.214046822743, 595.785953177258, 597.357859531773, 598.929765886288, 600.501672240803, 602.073578595318,
              603.645484949833, 605.217391304348, 606.789297658863, 608.361204013378, 609.933110367893, 611.505016722408,
              613.076923076923, 614.648829431438, 616.220735785953, 617.792642140468, 619.364548494983, 620.936454849498,
              622.508361204014, 624.080267558529, 625.652173913044, 627.224080267559, 628.795986622074, 630.367892976589,
              631.939799331104, 633.511705685619, 635.083612040134, 636.655518394649, 638.227424749164, 639.799331103679,
              641.371237458194, 642.943143812709, 644.515050167224, 646.086956521739, 647.658862876254, 649.230769230769,
              650.802675585284, 652.374581939799, 653.946488294314, 655.518394648829, 657.090301003345, 658.66220735786,
              660.234113712375, 661.80602006689, 663.377926421405, 664.94983277592, 666.521739130435, 668.09364548495,
              669.665551839465, 671.23745819398, 672.809364548495, 674.38127090301, 675.953177257525, 677.52508361204,
              679.096989966555, 680.66889632107, 682.240802675585, 683.8127090301, 685.384615384616, 686.956521739131,
              688.528428093646, 690.100334448161, 691.672240802676, 693.244147157191, 694.816053511706, 696.387959866221,
              697.959866220736, 699.531772575251, 701.103678929766, 702.675585284281, 704.247491638796, 705.819397993311,
              707.391304347826, 708.963210702341, 710.535117056856, 712.107023411371, 713.678929765886, 715.250836120401,
              716.822742474916, 718.394648829431, 719.966555183947, 721.538461538462, 723.110367892977, 724.682274247492,
              726.254180602007, 727.826086956522, 729.397993311037, 730.969899665552, 732.541806020067, 734.113712374582,
              735.685618729097, 737.257525083612, 738.829431438127, 740.401337792642, 741.973244147157, 743.545150501672,
              745.117056856187, 746.688963210702, 748.260869565218, 749.832775919733, 751.404682274248, 752.976588628763,
              754.548494983278, 756.120401337793, 757.692307692308, 759.264214046823, 760.836120401338, 762.408026755853,
              763.979933110368, 765.551839464883, 767.123745819398, 768.695652173913, 770.267558528428, 771.839464882943,
              773.411371237458, 774.983277591973, 776.555183946488, 778.127090301003, 779.698996655518, 781.270903010033,
              782.842809364548, 784.414715719064, 785.986622073579, 787.558528428094, 789.130434782609, 790.702341137124,
              792.274247491639, 793.846153846154, 795.418060200669, 796.989966555184, 798.561872909699, 800.133779264214,
              801.705685618729, 803.277591973244, 804.849498327759, 806.421404682274, 807.993311036789, 809.565217391304,
              811.137123745819, 812.709030100335, 814.28093645485, 815.852842809365, 817.42474916388, 818.996655518395,
              820.56856187291, 822.140468227425, 823.71237458194, 825.284280936455, 826.85618729097, 828.428093645485,
              830.}, {0.0346999999999994, 0.0348383481267339, 0.0349772274247486, 0.0351159824930614, 0.0352544943446233,
              0.035393502296662, 0.0355319740713721, 0.0356707959866215, 0.0358096150128081, 0.0359480883103246,
              0.0360870965612319, 0.0362256062922502, 0.0363643645484944, 0.0365032412563148, 0.0366416885522658,
              0.0367806845495618, 0.0369192441471566, 0.0370579337525789, 0.0371968612235815, 0.037335295070447,
              0.0374742662616517, 0.0376128829431432, 0.0377512107023405, 0.0378896789315442, 0.0380286874528564,
              0.0381671950917947, 0.0383059565217385, 0.0384330897850986, 0.0385297848146297, 0.0386034065075066,
              0.0387085250836114, 0.0388145384775486, 0.038893634526435, 0.0389455258467936, 0.0390313900412717,
              0.0391499130434776, 0.0392514864014795, 0.0393080615989462, 0.0393852090123099, 0.0394838052711158,
              0.0396106153846147, 0.0397300390948545, 0.0397937694264563, 0.0398543985163304, 0.0400246153846147,
              0.0401409944051086, 0.0401708538675009, 0.0402729783747236, 0.0403137807283136, 0.040410036789297,
              0.0405249450508781, 0.0406053839304768, 0.040684256036077, 0.0407557634170633, 0.0408478060200662,
              0.0409593086280503, 0.0410714320500954, 0.0411690008005401, 0.041252528428093, 0.041319234526435,
              0.0413836647993304, 0.0414811148100043, 0.0415907118960357, 0.0417119230769224, 0.041819772068241,
              0.041899574848786, 0.0419807744271679, 0.0420402378406739, 0.0421172675585277, 0.0422181039991454,
              0.0423257439514687, 0.0424150039955874, 0.0425303578595311, 0.0426258245125589, 0.0426896746192977,
              0.0427653158186857, 0.0428606378620216, 0.0429774615384608, 0.0430681159823518, 0.0431223391500028,
              0.0432431527645336, 0.0433210870152274, 0.0434093979933103, 0.0435128207055426, 0.0436053458852195,
              0.0436910181865075, 0.0438741204013371, 0.0442186917401971, 0.0447359682416559, 0.045387087125524,
              0.0461729976197246, 0.0470648695652166, 0.0480217038870696, 0.0490963158169066, 0.0505354944869415,
              0.0524466867039057, 0.0546057558528419, 0.0568811694193401, 0.0592881190048378, 0.0621224061659423,
              0.0653614849498317, 0.0686567105297789, 0.0718897334376989, 0.0752400477531476, 0.0789836000551471,
              0.0827202173913029, 0.0862135199263488, 0.0893901887016993, 0.0924532526062036, 0.095441469282358,
              0.0979873177257509, 0.100043853262647, 0.101712354452784, 0.103182962339001, 0.104577658862874,
              0.105864605155481, 0.107016836458406, 0.108029731196184, 0.108856047943498, 0.109479331103677,
              0.109931018394647, 0.110262140468226, 0.110594225361131, 0.11094833784601, 0.111159565217389,
              0.111106796057779, 0.110684362662775, 0.109927169020848, 0.108955351170567, 0.107805379972246,
              0.106511378104318, 0.105106969294811, 0.103445566765102, 0.101485484949831, 0.0993055202625757,
              0.09703662180673, 0.0949078743773554, 0.0927605764996781, 0.0905597692307676, 0.0885288567672368,
              0.086941297605492, 0.0856048845139813, 0.084295063545149, 0.0829653042695496, 0.0816283168166925,
              0.0803377054169914, 0.0791932882391647, 0.0781999565217378, 0.0773561291983905, 0.0766520424019769,
              0.0760284127570613, 0.0755598595673509, 0.0751599966555171, 0.0747416240660345, 0.0742379113392146,
              0.0736724572013081, 0.073099869574111, 0.0725296688963198, 0.0719333791966116, 0.0712614419074207,
              0.0705686111453058, 0.0698281538461526, 0.069021448422044, 0.0681373914039696, 0.0671962790027029,
              0.0662833651889265, 0.0654642240802665, 0.0647662869440678, 0.0642123183234886, 0.0637910177470992,
              0.0634397198640849, 0.0631279464882933, 0.0628368423646186, 0.0625408909965833, 0.062185898256599,
              0.0617417892976578, 0.0611748204422533, 0.0604567125595948, 0.0595983420799819, 0.0586092848662198,
              0.0574960568561863, 0.0562949880256875, 0.0550517132836396, 0.0538292570500952, 0.0526738646107584,
              0.0517086287625409, 0.0509865667651027, 0.0504872285632953, 0.0500680852789431, 0.0496299030100326,
              0.0491526078755418, 0.0486201027360698, 0.048003132772361, 0.0473716943730868, 0.0468132040133772,
              0.0462733855440112, 0.0456907161566918, 0.0451947535223788, 0.0448696450081826, 0.0446623010033437,
              0.0445144315341912, 0.0443944119280573, 0.0444182482210197, 0.0446800501672233, 0.0452939494236099,
              0.0463542037394143, 0.0478325896160954, 0.0495504095353296, 0.0514567826086949, 0.0537064338895602,
              0.0566882747989745, 0.0610132595460035, 0.0665575734718559, 0.072803678929765, 0.0796119596972167,
              0.0872311720771353, 0.0960204269817824, 0.105474040133778, 0.115691799220806, 0.127015376645554,
              0.13962828040276, 0.1519872315164, 0.164074849498325, 0.175920012008109, 0.18765504251761,
              0.199906622464951, 0.212124050967762, 0.22444200668896, 0.237054336671882, 0.250024769088447,
              0.262668558688532, 0.276230033444812, 0.28997473487867, 0.303160364477331, 0.315697859104812,
              0.32809527476339, 0.34073953177257, 0.352882704600436, 0.363644931616019, 0.373533623745813,
              0.383195800967759, 0.392467759197318, 0.401409689745955, 0.410254477140106, 0.418708235287832,
              0.425853712374575, 0.43209120234113, 0.437827636465516, 0.44320184079911, 0.447867941222507,
              0.45183103678929, 0.455233372980851, 0.458153887532904, 0.460599328381833, 0.462450046876104,
              0.464095384615377, 0.465747247900796, 0.467458127090293, 0.469165516384394, 0.470699464882935,
              0.471996372927481, 0.473019622500526, 0.473799164057489, 0.474616416672589, 0.475323311036781,
              0.475829067049732, 0.476130140984124, 0.476467636180879, 0.476971191133558, 0.477546053511698,
              0.478151567761324, 0.478792616487575, 0.479418881804589, 0.479916521739122, 0.480339041396847,
              0.480791775795196, 0.48141989998576, 0.48216511428164, 0.482954882943136, 0.483707438393929,
              0.484326674508994, 0.484726806020059, 0.485101705685611, 0.485476605351162, 0.485851505016714,
              0.486226404682266, 0.486601304347818, 0.48697620401337, 0.487351103678922, 0.487726003344473,
              0.488100903010025, 0.488475802675577, 0.488850702341129, 0.489219766064177, 0.489579096989958,
              0.489938277591965, 0.490297458193972, 0.490656638795979, 0.491015819397985, 0.491374999999992}}
            ,
            360,830,300);
        break;
    case 4:
    {
        const Spectrum spect2{{360, 362, 364, 366, 368, 370, 372, 374, 376, 378, 380, 382, 384, 386, 388, 390, 392, 394,
            396, 398, 400, 402, 404, 406, 408, 410, 412, 414, 416, 418, 420, 422, 424, 426, 428, 430, 432, 434, 436, 438,
            440, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 482,
            484, 486, 488, 490, 492, 494, 496, 498, 500, 502, 504, 506, 508, 510, 512, 514, 516, 518, 520, 522, 524, 526,
            528, 530, 532, 534, 536, 538, 540, 542, 544, 546, 548, 550, 552, 554, 556, 558, 560, 562, 564, 566, 568, 570,
            572, 574, 576, 578, 580, 582, 584, 586, 588, 590, 592, 594, 596, 598, 600, 602, 604, 606, 608, 610, 612, 614,
            616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 644, 646, 648, 650, 652, 654, 656, 658,
            660, 662, 664, 666, 668, 670, 672, 674, 676, 678, 680, 682, 684, 686, 688, 690, 692, 694, 696, 698, 700, 702,
            704, 706, 708, 710, 712, 714, 716, 718, 720, 722, 724, 726, 728, 730, 732, 734, 736, 738, 740, 742, 744, 746,
            748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 778, 780, 782, 784, 786, 788, 790,
            792, 794, 796, 798, 800, 820, 840}, {0.0347, 0.034876, 0.035053, 0.035229, 0.035406, 0.035582, 0.035759,
            0.035935, 0.036112, 0.036288, 0.036465, 0.036641, 0.036818, 0.036994, 0.037171, 0.037347, 0.037524, 0.0377,
            0.037876, 0.038053, 0.038229, 0.038406, 0.038533, 0.03862, 0.038773, 0.038887, 0.038945, 0.039064, 0.039233,
            0.039298, 0.039397, 0.03953, 0.039714, 0.039794, 0.039864, 0.04014, 0.040143, 0.04029, 0.040321, 0.040495,
            0.040597, 0.040698, 0.040786, 0.040926, 0.041072, 0.041194, 0.041294, 0.041362, 0.041487, 0.041629, 0.04179,
            0.04189, 0.041996, 0.042063, 0.042185, 0.042327, 0.042435, 0.042597, 0.042674, 0.042768, 0.042894, 0.043055,
            0.0431, 0.043268, 0.043348, 0.043485, 0.043605, 0.043706, 0.04399, 0.044595, 0.045412, 0.046444, 0.047634,
            0.048893, 0.050719, 0.053314, 0.056175, 0.059142, 0.062804, 0.0671, 0.071183, 0.075386, 0.080256, 0.084949,
            0.089023, 0.092914, 0.096696, 0.099535, 0.10169, 0.10353, 0.10528, 0.1068, 0.10811, 0.10912, 0.1098,
            0.11022, 0.11064, 0.1111, 0.11123, 0.11076, 0.10975, 0.10843, 0.10682, 0.10507, 0.1029, 0.10024, 0.097302,
            0.094614, 0.091871, 0.08903, 0.086963, 0.085306, 0.083636, 0.081922, 0.080257, 0.078833, 0.07765, 0.076732,
            0.075929, 0.075385, 0.074901, 0.074273, 0.073543, 0.072815, 0.072092, 0.071235, 0.070351, 0.06938, 0.068276,
            0.067068, 0.065917, 0.06495, 0.064216, 0.063698, 0.063276, 0.062904, 0.062538, 0.062075, 0.06146, 0.060592,
            0.0595, 0.058202, 0.056705, 0.055117, 0.05356, 0.05212, 0.051095, 0.05046, 0.049945, 0.049367, 0.04872,
            0.047927, 0.04712, 0.046474, 0.04571, 0.04508, 0.04474, 0.044548, 0.04437, 0.044425, 0.04489, 0.046093,
            0.04799, 0.050242, 0.052782, 0.056244, 0.061872, 0.069404, 0.077736, 0.087253, 0.098682, 0.11099, 0.1249,
            0.14113, 0.15674, 0.17197, 0.18675, 0.20243, 0.21793, 0.23376, 0.25036, 0.26632, 0.28417, 0.30119, 0.31709,
            0.33283, 0.34914, 0.36304, 0.3755, 0.38774, 0.39914, 0.41046, 0.42119, 0.42954, 0.43695, 0.44385, 0.44958,
            0.45419, 0.45802, 0.46111, 0.46323, 0.4653, 0.46748, 0.46966, 0.47151, 0.47291, 0.47387, 0.47493, 0.47573,
            0.4761, 0.47652, 0.47723, 0.47798, 0.4788, 0.4796, 0.48016, 0.48067, 0.48147, 0.48245, 0.48347, 0.48432,
            0.48909, 0.49366}};
        return test1(spect2, Spectrum{{360., 391.333333333333, 422.666666666667, 454., 485.333333333333, 516.666666666667, 548.,
                579.333333333333, 610.666666666667, 642., 673.333333333333, 704.666666666667, 736., 767.333333333333,
                798.666666666667, 830.}, {0.0353910372340425, 0.0374525319148937, 0.0395982101063831, 0.0413985248226951,
                0.0433581081560285, 0.0670410913120567, 0.107474800531915, 0.0895826374113473, 0.0703972881205673,
                0.057270329787234, 0.0468806241134751, 0.138158718085106, 0.367105611702127, 0.468352606382978,
                0.483183566489361, 0.490721327922078}}
              ,
              360,830,16);
        break;
    }
    default:
        std::cerr << "Bad test number " << whichTest << "\n";
        return 1;
    }
}
catch(std::exception& ex)
{
    std::cerr << "Error: " << ex.what() << "\n";
    return 1;
}
