//
// jja: swiss army knife for chess file formats
// src/quote.rs: Chess Quotes
//
// The author claims no copyright in the following compilation.
//   Based in part upon LiChess' lila/modules/quote/src/main/Quote.scala
//   which is licensed with
//   GNU Affero General Public License 3 or any later version.
//
// SPDX-License-Identifier: AGPL-3.0-or-later

use std::borrow::Cow;

use console::style;
use rand::{seq::SliceRandom, thread_rng, Rng};
use regex::RegexBuilder;
use textwrap::wrap;

/// The `quote` module provides a function to print a random quote related to chess.
///
/// # Functions
///
/// * `print_quote()` - Prints a random quote from a predefined list of chess-related quotes.
/// * `search_quote()` - Searchs a quote within a predefined list of chess-related quotes.
/// * `wrap_quote()` - Wraps the given text to fit within the specified line width.
///
/// # Constants
///
/// * `QUOTE_MAX: usize` - The number of quotes in the predefined list.
/// * `QUOTE: [(&str, &str); QUOTE_MAX]` - An array containing tuples of quotes and their respective authors.

/// Prints a random quote from a predefined list of chess-related quotes.
/// The quote is formatted with a maximum line width of 72 characters.
/// If `porcelain` is true, no formatting is done.
pub fn print_quote(idx: Option<usize>, porcelain: bool) {
    let idx = if let Some(idx) = idx {
        idx
    } else {
        thread_rng().gen_range(0..QUOTE_MAX)
    };

    let (quote, author) = QUOTE[idx];

    if porcelain {
        println!("{} -- {}", quote, author);
    } else {
        print!("{}: ", style(idx).bold());
        for line in wrap_text(quote, 72) {
            println!("{}", style(line).bold().magenta());
        }
        println!("\t-- {}", style(author).bold().green());
    }
}

/// Searchs a quote within a predefined list of chess-related quotes.
///
/// # Arguments
///
/// - `pattern: &str`: A case-insensitive regular expression
///
/// # Returns
///
/// Returns a `Result<Option<usize>, Box<dyn regex::Error>>` indicating the success or failure
/// of the search operation.
pub fn search_quote(pattern: &str) -> Result<Option<usize>, regex::Error> {
    let mut pattern = String::from(pattern);
    if !pattern
        .chars()
        .any(|ch| ch == '.' || ch == '?' || ch == '*')
    {
        // Match at word boundaries unless wildcard is specified.
        pattern.insert_str(0, r"\b");
        pattern.push_str(r"\b");
    }
    let regex = RegexBuilder::new(&pattern).case_insensitive(true).build()?;
    let mut matches = Vec::new();

    for (i, quote) in QUOTE.iter().enumerate().take(QUOTE_MAX) {
        // Match both author & quote. Concatenate the strings before matching so
        // user can search for both content and author at the same time.
        if regex.is_match(&format!("{} {}", quote.0, quote.1)) {
            matches.push(i)
        }
    }

    Ok(matches.choose(&mut thread_rng()).copied())
}

/// Wraps the given text to fit within the specified line width.
///
/// This function uses the English US hyphenation dictionary to split
/// words across lines when necessary. The returned value is a Vec of
/// Cow<'a, str> where each entry represents a single line of wrapped
/// text.
///
/// # Arguments
///
/// * text - The input text to wrap
/// * width - The maximum line width for the wrapped text
///
/// # Returns
///
/// A Vec<Cow<'a, str>> containing the wrapped lines.
pub fn wrap_text(text: &str, width: usize) -> Vec<Cow<'_, str>> {
    wrap(text, width)
}

/// The maximum number of quotes in the predefined list.
pub const QUOTE_MAX: usize = 1912;

// An array containing tuples of quotes and their respective authors.
const QUOTE: [(&str, &str); QUOTE_MAX] = [
    (
        r#"When you see a good move, look for a better one."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Nothing excites jaded Grandmasters more than a theoretical novelty."#,
        r#"Dominic Lawson"#,
    ),
    (r#"The Pin is mightier than the sword."#, r#"Fred Reinfeld"#),
    (
        r#"We cannot resist the fascination of sacrifice, since a passion for sacrifices is part of a chess player’s nature."#,
        r#"Rudolf Spielman"#,
    ),
    (
        r#"All I want to do, ever, is just play chess."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"A win by an unsound combination, however showy, fills me with artistic horror."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"The chessboard is the world, the pieces are the phenomena of the Universe, the rules of the game are what we call the laws of Nature and the player on the other side is hidden from us."#,
        r#"Thomas Huxley"#,
    ),
    (
        r#"Adequate compensation for a sacrifice is having a sound combination leading to a winning position; adequate compensation for a blunder is having your opponent snatch defeat from the jaws of victory."#,
        r#"Bruce A. Moon"#,
    ),
    (
        r#"Strategy requires thought, tactics require observation."#,
        r#"Max Euwe"#,
    ),
    (
        r#"I don’t believe in psychology. I believe in good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Modern chess is too much concerned with things like pawn structure. Forget it, checkmate ends the game."#,
        r#"Nigel Short"#,
    ),
    (
        r#"Life is a kind of chess, with struggle, competition, good and ill events."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"Even the laziest king flees wildly in the face of a double check!"#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"Combinations have always been the most intriguing aspect of chess.  The masters look for them, the public applauds them, the critics praise them. It is because combinations are possible that chess is more than a lifeless mathematical exercise. They are the poetry of the game; they are to chess what melody is to music. They represent the triumph of mind over matter."#,
        r#"Reuben Fine"#,
    ),
    (
        r#"Chess is a fairy tale of 1001 blunders."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Chess is no whit inferior to the violin, and we have a large number of professional violinists."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Only the player with the initiative has the right to attack."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"The winner of the game is the player who makes the next-to-last mistake."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Your body has to be in top condition. Your chess deteriorates as your body does. You can’t separate body from mind."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Of chess it has been said that life is not long enough for it, but that is the fault of life, not chess."#,
        r#"William Ewart Napier"#,
    ),
    (
        r#"I have added these principles to the law: get the Knights into action before both Bishops are developed."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Life is like a game of chess, changing with each move."#,
        r#"Chinese proverb"#,
    ),
    (
        r#"You cannot play at chess if you are kind-hearted."#,
        r#"French proverb"#,
    ),
    (
        r#"It’s just you and your opponent at the board and you’re trying to prove something."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"It is the aim of the modern school, not to treat every position according to one general law, but according to the principle inherent in the position."#,
        r#"Richard Reti"#,
    ),
    (
        r#"The pawns are the soul of the game."#,
        r#"François-André Danican Philidor"#,
    ),
    (
        r#"In order to improve your game, you must study the endgame before everything else, for whereas the endings can be studied and mastered by themselves, the middlegame and the opening must be studied in relation to the endgame."#,
        r#"Jose Raul Capablanca"#,
    ),
    (
        r#"Without error there can be no brilliancy."#,
        r#"Emanuel Lasker"#,
    ),
    (r#"Chess is like war on a board."#, r#"Bobby Fischer"#),
    (
        r#"Chess is played with the mind and not with the hands!"#,
        r#"Renaud and Kahn"#,
    ),
    (r#"Chess is mental torture."#, r#"Garry Kasparov"#),
    (
        r#"Many have become chess masters, no one has become the master of chess."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"The most important feature of the chess position is the activity of the pieces. This is absolutely fundamental in all phases of the game: Opening, Middlegame and especially Endgame. The primary constraint on a piece’s activity is the pawn structure."#,
        r#"Michael Stean"#,
    ),
    (
        r#"You have to have the fighting spirit. You have to force moves and take chances."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Could we look into the head of a chess player, we should see there a whole world of feelings, images, ideas, emotion and passion."#,
        r#"Alfred Binet"#,
    ),
    (
        r#"Openings teach you openings. Endgames teach you chess!"#,
        r#"Stephan Gerzadowicz"#,
    ),
    (
        r#"My style is somewhere between that of Tal and Petrosian."#,
        r#"Reshevsky"#,
    ),
    (
        r#"Play the opening like a book, the middlegame like a magician, and the endgame like a machine."#,
        r#"Spielmann"#,
    ),
    (
        r#"That’s what chess is all about. One day you give your opponent a lesson, the next day he gives you one."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Some part of a mistake is always correct."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Methodical thinking is of more use in chess than inspiration."#,
        r#"C. J. S. Purdy"#,
    ),
    (r#"When in doubt... play chess!"#, r#"Tevis"#),
    (
        r#"Who is your opponent tonight, tonight I am playing against the black pieces."#,
        r#"Akiba Rubinstein"#,
    ),
    (
        r#"I like the moment when I break a man’s ego."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Excellence at chess is one mark of a scheming mind."#,
        r#"Sir Arthur Conan Doyle"#,
    ),
    (
        r#"A bad day of chess is better than any good day at work."#,
        r#"Anonymous"#,
    ),
    (r#"Chess is the art of analysis."#, r#"Mikhail Botvinnik"#),
    (
        r#"The mistakes are there, waiting to be made."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"After black’s reply to 1.e4 with 1..e5, leaves him always trying to get into the game."#,
        r#"Howard Staunton"#,
    ),
    (r#"A player surprised is half beaten."#, r#"Proverb"#),
    (
        r#"A passed pawn increases in strength as the number of pieces on the board diminishes."#,
        r#"Capablanca"#,
    ),
    (
        r#"The essence of chess is thinking about what chess is."#,
        r#"David Bronstein"#,
    ),
    (
        r#"I am the best player in the world and I am here to prove it."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Chess is a forcing house where the fruits of character can ripen more fully than in life."#,
        r#"Edward Morgan Foster"#,
    ),
    (
        r#"Half the variations which are calculated in a tournament game turn out to be completely superfluous. Unfortunately, no one knows in advance which half."#,
        r#"Jan Timman"#,
    ),
    (
        r#"Good positions don’t win games, good moves do."#,
        r#"Gerald Abrahams"#,
    ),
    (
        r#"If I win a tournament, I win it by myself.  I do the playing. Nobody helps me."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"What would chess be without silly mistakes?"#,
        r#"Kurt Richter"#,
    ),
    (
        r#"Before the endgame, the Gods have placed the middle game."#,
        r#"Siegbert Tarrasch"#,
    ),
    (r#"Chess was Capablanca’s mother tongue."#, r#"Reti"#),
    (
        r#"Alekhine is a poet who creates a work of art out of something that would hardly inspire another man to send home a picture post card."#,
        r#"Max Euwe"#,
    ),
    (
        r#"During a chess competition a chess master should be a combination of a beast of prey and a monk."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"No one ever won a game by resigning."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"The defensive power of a pinned piece is only imaginary."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"When the chess game is over, the pawn and the king go back to the same box."#,
        r#"Irish saying"#,
    ),
    (
        r#"A strong memory, concentration, imagination, and a strong will is required to become a great chess player."#,
        r#"Bobby Fischer"#,
    ),
    (r#"Every chess master was once a beginner."#, r#"Chernev"#),
    (
        r#"One doesn’t have to play well, it’s enough to play better than your opponent."#,
        r#"Siegbert Tarrasch"#,
    ),
    (r#"Chess is above all, a fight!"#, r#"Emanuel Lasker"#),
    (
        r#"Discovered check is the dive bomber of the chessboard."#,
        r#"Reuben Fine"#,
    ),
    (
        r#"I know people who have all the will in the world, but still can’t play good chess."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"A chess game is a dialogue, a conversation between a player and his opponent. Each move by the opponent may contain threats or be a blunder, but a player cannot defend against threats or take advantage of blunders if he does not first ask himself: What is my opponent planning after each move?"#,
        r#"Bruce A. Moon"#,
    ),
    (
        r#"The hardest game to win is a won game."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The most powerful weapon in chess is to have the next move."#,
        r#"David Bronstein"#,
    ),
    (
        r#"He who fears an isolated queen’s pawn should give up chess."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Different people feel differently about resigning."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Chess is not like life... it has rules!"#,
        r#"Mark Pasternak"#,
    ),
    (
        r#"It’s always better to sacrifice your opponent’s men."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"To avoid losing a piece, many a person has lost the game."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"All that matters on the chessboard is good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Help your pieces so they can help you."#,
        r#"Paul Morphy"#,
    ),
    (
        r#"In a gambit you give up a pawn for the sake of getting a lost game."#,
        r#"Samuel Standige Boden"#,
    ),
    (
        r#"It is not enough to be a good player... you must also play well."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"A sacrifice is best refuted by accepting it."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"Tactics flow from a superior position."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Later, I began to succeed in decisive games. Perhaps because I realized a very simple truth: not only was I worried, but also my opponent."#,
        r#"Mikhail Tal"#,
    ),
    (r#"Chess is life."#, r#"Bobby Fischer"#),
    (r#"Chess is a beautiful mistress."#, r#"Bent Larsen"#),
    (
        r#"Some sacrifices are sound; the rest are mine."#,
        r#"Mikhail Tal"#,
    ),
    (r#"Best by test: 1. e4."#, r#"Bobby Fischer"#),
    (
        r#"A bad plan is better than none at all."#,
        r#"Frank Marshall"#,
    ),
    (
        r#"Chess books should be used as we use glasses: to assist the sight, although some players make use of them as if they thought they conferred sight."#,
        r#"Jose Raul Capablanca"#,
    ),
    (
        r#"There are two types of sacrifices: correct ones and mine."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Morphy was probably the greatest genius of them all."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"My opponents make good moves too. Sometimes I don’t take these things into consideration."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"The combination player thinks forward; he starts from the given position, and tries the forceful moves in his mind."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"A chess game is divided into three stages: the first, when you hope you have the advantage, the second when you believe you have an advantage, and the third... when you know you’re going to lose!"#,
        r#"Savielly Tartakower"#,
    ),
    (r#"Chess demands total concentration."#, r#"Bobby Fischer"#),
    (
        r#"Chess, like love, like music, has the power to make people happy."#,
        r#"Siegbert Tarrasch"#,
    ),
    (r#"All my games are real."#, r#"Bobby Fischer"#),
    (
        r#"Chess is everything: art, science and sport."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"Chess is the art which expresses the science of logic."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Not all artists are chess players, but all chess players are artists."#,
        r#"Marcel Duchamp"#,
    ),
    (r#"Chess is imagination."#, r#"David Bronstein"#),
    (
        r#"Chess is thirty to forty percent psychology. You don’t have this when you play a computer. I can’t confuse it."#,
        r#"Judith Polgar"#,
    ),
    (
        r#"On the chessboard, lies and hypocrisy do not survive long."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Chess is war over the board. The object is to crush the opponents mind."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"The passed pawn is a criminal, who should be kept under lock and key.  Mild measures, such as police surveillance, are not sufficient."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"Chess holds its master in its own bonds, shackling the mind and brain so that the inner freedom of the very strongest must suffer."#,
        r#"Albert Einstein"#,
    ),
    (
        r#"Human affairs are like a chess game: only those who do not take it seriously can be called good players."#,
        r#"Hung Tzu Ch’eng"#,
    ),
    (
        r#"The blunders are all there on the board, waiting to be made."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Via the squares on the chessboard, the Indians explain the movement of time and the age, the higher influences which control the world and the ties which link chess with the human soul."#,
        r#"Al-Masudi"#,
    ),
    (
        r#"It is no time to be playing chess when the house is on fire."#,
        r#"Italian proverb"#,
    ),
    (
        r#"You sit at the board and suddenly your heart leaps. Your hand trembles to pick up the piece and move it.  But what chess teaches you is that you must sit there calmly and think about whether it’s really a good idea and whether there are other better ideas."#,
        r#"Stanley Kubrick"#,
    ),
    (
        r#"Daring ideas are like chess men moved forward. They may be beaten, but they may start a winning game."#,
        r#"Johann Wolfgang von Goethe"#,
    ),
    (
        r#"Of all my Russian books, The Defense contains and diffuses the greatest ’warmth’ which may seem odd seeing how supremely abstract chess is supposed to be."#,
        r#"Vladimir Nabokov"#,
    ),
    (
        r#"For surely of all the drugs in the world, chess must be the most permanently pleasurable."#,
        r#"Assiac"#,
    ),
    (
        r#"A thorough understanding of the typical mating continuations makes the most complicated sacrificial combinations leading up to them not only not difficult, but almost a matter of course."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Chess problems demand from the composer the same virtues that characterize all worthwhile art: originality, invention, conciseness, harmony, complexity, and splendid insincerity."#,
        r#"Vladimir Nabokov"#,
    ),
    (
        r#"Personally, I rather look forward to a computer program winning the world chess Championship. Humanity needs a lesson in humility."#,
        r#"Richard Dawkins"#,
    ),
    (
        r#"The boy (then a 12 year old boy named Anatoly Karpov) doesn’t have a clue about chess, and there’s no future at all for him in this profession."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"As one by one I mowed them down, my superiority soon became apparent."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Though most people love to look at the games of the great attacking masters, some of the most successful players in history have been the quiet positional players. They slowly grind you down by taking away your space, tying up your pieces, and leaving you with virtually nothing to do!"#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"Chess is ruthless; you’ve got to be prepared to kill people."#,
        r#"Nigel Short"#,
    ),
    (
        r#"There must have been a time when men were demigods, or they could not have invented chess."#,
        r#"Gustav Schenk"#,
    ),
    (
        r#"Chess is really ninety nine percent calculation."#,
        r#"Soltis"#,
    ),
    (r#"Chess is the gymnasium of the mind."#, r#"Blaise Pascal"#),
    (
        r#"The game of chess is not merely an idle amusement; several very valuable qualities of the mind are to be acquired and strengthened by it, so as to become habits ready on all occasions; for life is a kind of chess."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"Winning isn’t everything... but losing is nothing."#,
        r#"Mednis"#,
    ),
    (
        r#"Look at Garry Kasparov. After he loses, invariably he wins the next game. He just kills the next guy. That’s something that we have to learn to be able to do."#,
        r#"Maurice Ashley"#,
    ),
    (
        r#"There just isn’t enough televised chess."#,
        r#"David Letterman"#,
    ),
    (
        r#"Avoid the crowd. Do your own thinking independently. Be the chess player, not the chess piece."#,
        r#"Ralph Charell"#,
    ),
    (
        r#"Chess is a terrible game. If you have no center, your opponent has a freer position. If you do have a center, then you really have something to worry about!"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Any material change in a position must come about by mate, a capture, or a pawn promotion."#,
        r#"Purdy"#,
    ),
    (
        r#"We don’t really know how the game was invented, though there are suspicions. As soon as we discover the culprits, we’ll let you know."#,
        r#"Bruce Pandolfini"#,
    ),
    (
        r#"The battle for the ultimate truth will never be won.  And that’s why chess is so fascinating."#,
        r#"Hans Kmoch"#,
    ),
    (
        r#"I am still a victim of chess. It has all the beauty of art and much more. It cannot be commercialized. Chess is much purer than art in its social position."#,
        r#"Marcel Duchamp"#,
    ),
    (
        r#"Blessed be the memory of him who gave the world this immortal game."#,
        r#"A. G. Gardiner"#,
    ),
    (
        r#"In the perfect chess combination as in a first-rate short story, the whole plot and counter-plot should lead up to a striking finale, the interest not being allayed until the very last moment."#,
        r#"Yates and Winter"#,
    ),
    (r#"Castle early and often."#, r#"Rob Sillars"#),
    (
        r#"I believe that chess possesses a magic that is also a help in advanced age. A rheumatic knee is forgotten during a game of chess and other events can seem quite unimportant in comparison with a catastrophe on the chessboard."#,
        r#"Vlastimil Hort"#,
    ),
    (
        r#"Chess is a more highly symbolic game, but the aggressions are therefore even more frankly represented in the play. It probably began as a war game; that is, the representation of a miniature battle between the forces of two kingdoms."#,
        r#"Karl Meninger"#,
    ),
    (
        r#"No chess Grandmaster is normal; they only differ in the extent of their madness."#,
        r#"Viktor Korchnoi"#,
    ),
    (r#"Chess is 99 percent tactics."#, r#"Teichmann"#),
    (r#"I’d rather have a pawn than a finger."#, r#"Reuben Fine"#),
    (
        r#"Chess mastery essentially consists of analyzing."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"If your opponent cannot do anything active, then don’t rush the position; instead you should let him sit there, suffer, and beg you for a draw."#,
        r#"Jeremy Silman"#,
    ),
    (
        r#"The chess pieces are the block alphabet which shapes thoughts; and these thoughts, although making a visual design on the chessboard, express their beauty abstractly, like a poem."#,
        r#"Marcel Duchamp"#,
    ),
    (
        r#"Examine moves that smite! A good eye for smites is far more important than a knowledge of strategical principles."#,
        r#"Purdy"#,
    ),
    (r#"Chess is like life."#, r#"Boris Spassky"#),
    (
        r#"If your opponent offers you a draw, try to work out why he thinks he’s worse off."#,
        r#"Nigel Short"#,
    ),
    (
        r#"Chess teaches you to control the initial excitement you feel when you see something that looks good and it trains you to think objectively when you’re in trouble."#,
        r#"Stanley Kubrick"#,
    ),
    (
        r#"Let the perfectionist play postal."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"If chess is a science, it’s a most inexact one. If chess is an art, it is too exacting to be seen as one. If chess is a sport, it’s too esoteric. If chess is a game, it’s too demanding to be just a game. If chess is a mistress, she’s a demanding one. If chess is a passion, it’s a rewarding one. If chess is life, it’s a sad one."#,
        r#"Anonymous"#,
    ),
    (
        r#"Chess is a foolish expedient for making idle people believe they are doing something very clever when they are only wasting their time."#,
        r#"George Bernard Shaw"#,
    ),
    (
        r#"You must take your opponent into a deep dark forest where 2+2=5, and the path leading out is only wide enough for one."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"I feel as if I were a piece in a game of chess, when my opponent says of it: That piece cannot be moved."#,
        r#"Soren Kierkegaard"#,
    ),
    (
        r#"When your house is on fire, you can't be bothered with the neighbors.  Or, as we say in chess, if your king is under attack you don’t worry about losing a pawn on the queen’s side."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Man is a frivolous, a specious creature, and like a chess player, cares more for the process of attaining his goal than for the goal itself."#,
        r#"Dostoyevsky"#,
    ),
    (
        r#"When asked, -How is that you pick better moves than your opponents?, I responded: I’m very glad you asked me that, because, as it happens, there is a very simple answer. I think up my own moves, and I make my opponent think up his."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Mistrust is the most necessary characteristic of the chess player."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"What is the object of playing a gambit opening?... To acquire a reputation of being a dashing player at the cost of losing a game."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Pawns; they are the soul of this game, they alone form the attack and defense."#,
        r#"François-André Danican Philidor"#,
    ),
    (
        r#"In chess, at least, the brave inherit the earth."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"There are two classes of men; those who are content to yield to circumstances and who play whist; those who aim to control circumstances, and who play chess."#,
        r#"Mortimer Collins"#,
    ),
    (
        r#"The tactician must know what to do whenever something needs doing; the strategist must know what to do when nothing needs doing."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"All chess players should have a hobby."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"I played chess with him and would have beaten him sometimes only he always took back his last move, and ran the game out differently."#,
        r#"Mark Twain"#,
    ),
    (
        r#"The tactician knows what to do when there is something to do; whereas the strategian knows what to do when there is nothing to do."#,
        r#"Gerald Abrahams"#,
    ),
    (
        r#"In chess, just as in life, today’s bliss may be tomorrow’s poison."#,
        r#"Assaic"#,
    ),
    (
        r#"You may learn much more from a game you lose than from a game you win. You will have to lose hundreds of games before becoming a good player."#,
        r#"Jose Raul Capablanca"#,
    ),
    (
        r#"The way he plays chess demonstrates a man’s whole nature."#,
        r#"Stanley Ellin"#,
    ),
    (
        r#"You can only get good at chess if you love the game."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"A man that will take back a move at chess will pick a pocket."#,
        r#"Richard Fenton"#,
    ),
    (
        r#"Whoever sees no other aim in the game than that of giving checkmate to one’s opponent will never become a good chess player."#,
        r#"Euwe"#,
    ),
    (
        r#"In blitz, the Knight is stronger than the Bishop."#,
        r#"Vlastimil Hort"#,
    ),
    (
        r#"Chess is a fighting game which is purely intellectual and includes chance."#,
        r#"Richard Reti"#,
    ),
    (
        r#"Chess is a sea in which a gnat may drink and an elephant may bathe."#,
        r#"Hindu proverb"#,
    ),
    (
        r#"Pawn endings are to chess what putting is to golf."#,
        r#"Cecil Purdy"#,
    ),
    (
        r#"Chess opens and enriches your mind."#,
        r#"Saudin Robovic"#,
    ),
    (
        r#"The isolated pawn casts gloom over the entire chessboard."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"For me, chess is life and every game is like a new life. Every chess player gets to live many lives in one lifetime."#,
        r#"Eduard Gufeld"#,
    ),
    (
        r#"Chess is a terrific way for kids to build self image and self esteem."#,
        r#"Saudin Robovic"#,
    ),
    (
        r#"If a ruler does not understand chess, how can he rule over a kingdom?"#,
        r#"King Khusros II"#,
    ),
    (r#"Chess is a cold bath for the mind."#, r#"Sir John Simon"#),
    (
        r#"Becoming successful at chess allows you to discover your own personality. That’s what I want for the kids I teach."#,
        r#"Saudin Robovic"#,
    ),
    (
        r#"Chess is so inspiring that I do not believe a good player is capable of having an evil thought during the game."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"You are for me the queen on d8 and I am the pawn on d7!!"#,
        r#"GM Eduard Gufeld"#,
    ),
    (
        r#"By playing at chess then, we may learn: First: Foresight. Second: Circumspection. Third: Caution. And lastly, we learn by chess the habit of not being discouraged by present bad appearances in the state of our affairs, the habit of hoping for a favorable chance, and that of persevering in the secrets of resources."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"I prefer to lose a really good game than to win a bad one."#,
        r#"David Levy"#,
    ),
    (
        r#"Capture of the adverse king is the ultimate but not the first object of the game."#,
        r#"William Steinitz"#,
    ),
    (
        r#"When I have white, I win because I am white; When I have black, I win because I am Bogolyubov."#,
        r#"Bogolyubov"#,
    ),
    (r#"Every pawn is a potential queen."#, r#"James Mason"#),
    (
        r#"Chess is in its essence a game, in its form an art, and in its execution a science."#,
        r#"Baron Tassilo"#,
    ),
    (
        r#"No price is too great for the scalp of the enemy king."#,
        r#"Koblentz"#,
    ),
    (
        r#"In life, as in chess, ones own pawns block ones way.  A mans very wealth, ease, leisure, children, books, which should help him to win, more often checkmate him."#,
        r#"Charles Buxton"#,
    ),
    (
        r#"Chess is a part of culture and if a culture is declining then chess too will decline."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"A good sacrifice is one that is not necessarily sound but leaves your opponent dazed and confused."#,
        r#"Rudolph Spielmann"#,
    ),
    (
        r#"Chess, like any creative activity, can exist only through the combined efforts of those who have creative talent, and those who have the ability to organize their creative work."#,
        r#"Mikhail Botvinnik"#,
    ),
    (r#"One bad move nullifies forty good ones."#, r#"Horowitz"#),
    (
        r#"Place the contents of the chess box in a hat, shake them up vigorously, pour them on the board from a height of two feet, and you get the style of Steinitz."#,
        r#"H. E. Bird"#,
    ),
    (
        r#"I have never in my life played the French Defence, which is the dullest of all openings."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"Pawns are born free, yet they are everywhere in chains."#,
        r#"Rick Kennedy"#,
    ),
    (
        r#"It is not a move, even the best move that you must seek, but a realizable plan."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"Those who say they understand chess, understand nothing."#,
        r#"Robert Hübner"#,
    ),
    (
        r#"Good offense and good defense both begin with good development."#,
        r#"Bruce A. Moon"#,
    ),
    (
        r#"Botvinnik tried to take the mystery out of chess, always relating it to situations in ordinary life. He used to call chess a typical inexact problem similar to those which people are always having to solve in everyday life."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"A good player is always lucky."#,
        r#"Jose Raul Capablanca"#,
    ),
    (
        r#"The sign of a great master is his ability to win a won game quickly and painlessly."#,
        r#"Irving Chernev"#,
    ),
    (
        r#"One of these modest little moves may be more embarrassing to your opponent than the biggest threat."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Live, lose, and learn, by observing your opponent how to win."#,
        r#"Amber Steenbock"#,
    ),
    (r#"The older I grow, the more I value pawns."#, r#"Keres"#),
    (
        r#"Everything is in a state of flux, and this includes the world of chess."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"The beauty of a move lies not in its appearance but in the thought behind it."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"My God, Bobby Fischer plays so simply."#,
        r#"Alexei Suetin"#,
    ),
    (
        r#"You need not play well - just help your opponent to play badly."#,
        r#"Genrikh Chepukaitis"#,
    ),
    (
        r#"It is difficult to play against Einstein’s theory --on his first loss to Fischer."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"The only thing chess players have in common is chess."#,
        r#"Lodewijk Prins"#,
    ),
    (
        r#"Bobby just drops the pieces and they fall on the right squares."#,
        r#"Miguel Najdorf"#,
    ),
    (
        r#"We must make sure that chess will not be like a dead language, very interesting, but for a very small group."#,
        r#"Sytze Faber"#,
    ),
    (
        r#"The passion for playing chess is one of the most unaccountable in the world."#,
        r#"H.G. Wells"#,
    ),
    (
        r#"Chess is so interesting in itself, as not to need the view of gain to induce engaging in it; and thence it is never played for money."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"The enormous mental resilience, without which no chess player can exist, was so much taken up by chess that he could never free his mind of this game."#,
        r#"Albert Einstein"#,
    ),
    (
        r#"Nowadays, when you’re not a Grandmaster at 14, you can forget about it."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"Do you realize Fischer almost never has any bad pieces? He exchanges them, and the bad pieces remain with his opponents."#,
        r#"Yuri Balashov"#,
    ),
    (
        r#"It is always better to sacrifice your opponent’s men."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"In chess, as it is played by masters, chance is practically eliminated."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"You know you’re going to lose. Even when I was ahead I knew I was going to lose  --on playing against Fischer."#,
        r#"Andrew Soltis"#,
    ),
    (
        r#"I won’t play with you anymore. You have insulted my friend! --when an opponent cursed himself for a blunder."#,
        r#"Miguel Najdorf"#,
    ),
    (
        r#"You know, comrade Pachman, I don’t enjoy being a Minister, I would rather play chess like you."#,
        r#"Che Guevara"#,
    ),
    (
        r#"It began to feel as though you were playing against chess itself --on playing against Robert Fischer."#,
        r#"Walter Shipman"#,
    ),
    (
        r#"When you play Bobby, it is not a question if you win or lose.  It is a question if you survive."#,
        r#"Boris Spassky"#,
    ),
    (
        r#"When you absolutely don’t know what to do anymore, it is time to panic."#,
        r#"John van der Wiel"#,
    ),
    (r#"We like to think."#, r#"Garry Kasparov"#),
    (
        r#"Dazzling combinations are for the many, shifting wood is for the few."#,
        r#"Georg Kieninger"#,
    ),
    (
        r#"In complicated positions, Bobby Fischer hardly had to be afraid of anybody."#,
        r#"Paul Keres"#,
    ),
    (
        r#"It was clear to me that the vulnerable point of the American Grandmaster (Bobby Fischer) was in double-edged, hanging, irrational positions, where he often failed to find a win even in a won position."#,
        r#"Efim Geller"#,
    ),
    (
        r#"I love all positions. Give me a difficult positional game, I will play it.  But totally won positions, I cannot stand them."#,
        r#"Hein Donner"#,
    ),
    (
        r#"In Fischer’s hands, a slight theoretical advantage is as good a being a queen ahead."#,
        r#"Isaac Kashdan"#,
    ),
    (
        r#"Bobby Fischer’s current state of mind is indeed a tragedy. One of the worlds greatest chess players - the pride and sorrow of American chess."#,
        r#"Frank Brady"#,
    ),
    (
        r#"Fischer is an American chess tragedy on par with Morphy and Pillsbury."#,
        r#"Mig Greengard"#,
    ),
    (
        r#"Nonsense was the last thing Fischer was interested in, as far as chess was concerned."#,
        r#"Elie Agur"#,
    ),
    (
        r#"Fischer is the strongest player in the world. In fact, the strongest player who ever lived."#,
        r#"Larry Evans"#,
    ),
    (
        r#"If you aren’t afraid of Spassky, then I have removed the element of money."#,
        r#"Jim Slater"#,
    ),
    (
        r#"I guess a certain amount of temperament is expected of chess geniuses."#,
        r#"Ron Gross"#,
    ),
    (
        r#"Fischer sacrificed virtually everything most of us weakies (to use his term) value, respect, and cherish, for the sake of an artful, often beautiful board game, for the ambivalent privilege of being its greatest master."#,
        r#"Paul Kollar"#,
    ),
    (
        r#"Fischer chess play was always razor-sharp, rational and brilliant. One of the best ever."#,
        r#"Dave Regis"#,
    ),
    (
        r#"Fischer wanted to give the Russians a taste of their own medicine."#,
        r#"Larry Evans"#,
    ),
    (
        r#"With or without the title, Bobby Fischer was unquestionably the greatest player of his time."#,
        r#"Burt Hochberg"#,
    ),
    (
        r#"Fischer is completely natural. He plays no roles.  He’s like a child. Very, very simple."#,
        r#"Zita Rajcsanyi"#,
    ),
    (
        r#"Spassky will not be psyched out by Fischer."#,
        r#"Mike Goodall"#,
    ),
    (
        r#"Already at 15 years of age he was a Grandmaster, a record at that time, and his battle to reach the top was the background for all the major chess events of the 1960."#,
        r#"Tim Harding"#,
    ),
    (
        r#"Fischer, who may or may not be mad as a hatter, has every right to be horrified."#,
        r#"Jeremy Silman"#,
    ),
    (
        r#"When I asked Fischer why he had not played a certain move in our game, he replied: ‘Well, you laughed when I wrote it down!’"#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"I look one move ahead... the best!"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Fischer prefers to enter chess history alone."#,
        r#"Miguel Najdorf"#,
    ),
    (
        r#"Bobby is the most misunderstood, misquoted celebrity walking the face of this earth."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"When you don’t know what to play, wait for an idea to come into your opponent’s mind. You may be sure that idea will be wrong."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"There is no remorse like the remorse of chess."#,
        r#"H. G. Wells"#,
    ),
    (
        r#"By this measure (on the gap between Fischer & his contemporaries), I consider him the greatest world champion."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"By the beauty of his games, the clarity of his play, and the brilliance of his ideas, Fischer made himself an artist of the same stature as Brahms, Rembrandt, and Shakespeare."#,
        r#"David Levy"#,
    ),
    (
        r#"Chess is a terrible game. If you have no center, your opponent has a freer position. If you do have a center, then you really have something to worry about!"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Many chess players were surprised when after the game, Fischer quietly explained: ’I had already analyzed this possibility’ in a position which I thought was not possible to foresee from the opening."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Suddenly it was obvious to me in my analysis I had missed what Fischer had found with the greatest of ease at the board."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"The king is a fighting piece. Use it!"#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"A thorough understanding of the typical mating continuations makes the most complicated sacrificial combinations leading up to them not only not difficult, but almost a matter of course."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Bobby Fischer is the greatest chess genius of all time!"#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"The laws of chess do not permit a free choice: you have to move whether you like it or not."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"First-class players lose to second-class players because second-class players sometimes play a first-class game."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Bobby is the finest chess player this country ever produced. His memory for the moves, his brilliance in dreaming up combinations, and his fierce determination to win are uncanny."#,
        r#"John Collins"#,
    ),
    (
        r#"After a bad opening, there is hope for the middle game. After a bad middle game, there is hope for the endgame. But once you are in the endgame, the moment of truth has arrived."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Weak points or holes in the opponent’s position must be occupied by pieces not pawns."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"There is only one thing Fischer does in chess without pleasure: to lose!"#,
        r#"Boris Spassky"#,
    ),
    (
        r#"Bobby Fischer is the greatest chess player who has ever lived."#,
        r#"Ken Smith"#,
    ),
    (
        r#"Up to this point white has been following well-known analysis. But now he makes a fatal error: he begins to use his own head."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Fischer was a master of clarity and a king of artful positioning. His opponents would see where he was going but were powerless to stop him."#,
        r#"Bruce Pandolfini"#,
    ),
    (
        r#"No other master has such a terrific will to win. At the board he radiates danger, and even the strongest opponents tend to freeze, like rabbits when they smell a panther. Even his weaknesses are dangerous."#,
        r#"Anonymous German Expert"#,
    ),
    (
        r#"White lost because he failed to remember the right continuation and had to think up the moves himself."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Not only will I predict his triumph over Botvinnik, but I’ll go further and say that he’ll probably be the greatest chess player that ever lived."#,
        r#"John Collins"#,
    ),
    (
        r#"I consider Fischer to be one of the greatest opening experts ever."#,
        r#"Keith Hayward"#,
    ),
    (
        r#"I like to say that Bobby Fischer was the greatest player ever. But what made Fischer a genius was his ability to blend an American freshness and pragmatism with Russian ideas about strategy."#,
        r#"Bruce Pandolfini"#,
    ),
    (
        r#"At this time Fischer is simply a level above all the best chess players in the world."#,
        r#"John Jacobs"#,
    ),
    (
        r#"I have always a slight feeling of pity for the man who has no knowledge of chess."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"There’s never before been a chess player with such a thorough knowledge of the intricacies of the game and such an absolutely indomitable will to win. I think Bobby is the greatest player that ever lived."#,
        r#"Lisa Lane"#,
    ),
    (
        r#"He who takes the queen’s Knight’s pawn will sleep in the streets."#,
        r#"Anonymous"#,
    ),
    (
        r#"I had a toothache during the first game. In the second game I had a headache. In the third game it was an attack of rheumatism. In the fourth game, I wasn’t feeling well. And in the fifth game? Well, must one have to win every game?"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"The stomach is an essential part of the chess master."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"I’m not a materialistic person, in that, I don’t suffer the lack or loss of money.  The absence of worldly goods I don’t look back on. For chess is a way I can be as materialistic as I want without having to sell my soul"#,
        r#"Jamie Walter Adams"#,
    ),
    (
        r#"These are not pieces, they are men! For any man to walk into the line of fire will be one less man in your army to fight for you. Value every troop and use him wisely, throw him not to the dogs as he is there to serve his king."#,
        r#"Jamie Walter Adams"#,
    ),
    (
        r#"Chess isn’t a game of speed, it is a game of speech through actions."#,
        r#"Matthew Selman"#,
    ),
    (
        r#"Life like chess is about knowing to do the right move at the right time."#,
        r#"Kaleb Rivera"#,
    ),
    (r#"Come on Harry!"#, r#"Simon Williams"#),
    (
        r#"Some people think that if their opponent plays a beautiful game, it’s okay to lose. I don’t. You have to be merciless."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"It's one of those types of positions where he has pieces on squares."#,
        r#"John ~ZugAddict~ Chernoff"#,
    ),
    (
        r#"On the bright side, I no longer have any more pieces to lose."#,
        r#"John ~ZugAddict~ Chernoff"#,
    ),
    (
        r#"Tactics... Tactics are your friends. But they are weird friends who do strange things."#,
        r#"John ~ZugAddict~ Chernoff"#,
    ),
    (
        r#"You can't take the pawn because then the other will queen. Like wonder twin powers"#,
        r#"John ~ZugAddict~ Chernoff"#,
    ),
    (
        r#"Most of the gods throw dice but Fate plays chess, and you don't find out until too late that he's been using two queens all along."#,
        r#"Terry Pratchett"#,
    ),
    (
        r#"Atomic is just like regular chess, except you're exploding, everything's exploding, and you're in bullet hell."#,
        r#"Unihedron 0"#,
    ),
    (
        r#"lichess is better, but it's free."#,
        r#"Thibault Duplessis"#,
    ),
    (
        r#"When you trade, the key concern is not always the value of the pieces being exchanged, but what’s left on the board."#,
        r#"Dan Heisman"#,
    ),
    (
        r#"I detest the endgame. A well-played game should be practically decided in the middlegame."#,
        r#"David Janowski"#,
    ),
    (
        r#"Many men, many styles; what is chess style but the intangible expression of the will to win."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"Never play for the win, never play for the draw, just play chess!"#,
        r#"Alexander Khalifman"#,
    ),
    (
        r#"In chess, knowledge is a very transient thing. It changes so fast that even a single mouse-slip sometimes changes the evaluation."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"Having good strategies in playing chess is often a good indication of being focused in life."#,
        r#"Martin Dansky"#,
    ),
    (
        r#"Chess is an infinitely complex game, which one can play in infinitely numerous and varied ways."#,
        r#"Vladimir Kramnik"#,
    ),
    (
        r#"Chess: It’s like alcohol. It’s a drug. I have to control it, or it could overwhelm me."#,
        r#"Charles Krauthammer"#,
    ),
    (
        r#"Drawing general conclusions about your main weaknesses can provide a great stimulus to further growth."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"The good thing in chess is that very often the best moves are the most beautiful ones. The beauty of logic."#,
        r#"Boris Gelfand"#,
    ),
    (
        r#"Any experienced player knows how a change in the character of the play influences your psychological mood."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Be a harsh critic of your own wins."#,
        r#"Vasilios Kotronias"#,
    ),
    (
        r#"Good players develop a tactical instinct, a sense of what is possible or likely and what is not worth calculating."#,
        r#"Sam Reshevsky"#,
    ),
    (
        r#"Lack of patience is probably the most common reason for losing a game, or drawing games that should have been won."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"The scheme of a game is played on positional lines; the decision of it, as a rule, is effected by combinations."#,
        r#"Richard Reti"#,
    ),
    (
        r#"On the chessboard lies and hypocrisy do not last long."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The single most important thing in life is to believe in yourself regardless of what everyone else says."#,
        r#"Hikaru Nakamura"#,
    ),
    (
        r#"Attackers may sometimes regret bad moves, but it is much worse to forever regret an opportunity you allowed to pass you by."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"My favorite victory is when it is not even clear where my opponent made a mistake."#,
        r#"Peter Leko"#,
    ),
    (r#"Win with grace, lose with dignity."#, r#"Susan Polgar"#),
    (
        r#"Pawns are such fascinating pieces, too...So small, almost insignificant, and yet--they can depose kings."#,
        r#"Lavie Tidhar"#,
    ),
    (
        r#"The move is there, but you must see it."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"The kings are an apt metaphor for human beings: utterly constrained by the rules of the game, defenseless against bombardment from all sides, able only to temporarily dodge disaster by moving one step in any direction."#,
        r#"Jennifer duBois"#,
    ),
    (
        r#"If chess is an art, Alekhine. If chess is a science, Capablanca. If chess is a struggle, Lasker. --on who he thought was the best player."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Chess is a good mistress, but a bad master."#,
        r#"Gerald Abrahams"#,
    ),
    (
        r#"I often play a move I know how to refute."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"First restrain, next blockade, lastly destroy."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"If you don't know what to do, find your worst piece and look for a better square."#,
        r#"Gerard Schwarz"#,
    ),
    (
        r#"Players who balk at playing one-minute chess are failing to see the whole picture. They shouldn’t be worrying that they will make more mistakes – they should be rubbing their hands in glee at the thought of all the mistakes their opponents will make."#,
        r#"Hikaru Nakamura"#,
    ),
    (
        r#"A Chess game is divided into three stages: the first, when you hope you have the advantage, the second when you believe that you have an advantage, and the third ... when you know you're going to lose !"#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"A Queen's sacrifice, even when fairly obvious, always rejoices the heart of the chess-lover."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"A chess game, after all, is a fight in which all possible factors must be made use of, and in which a knowledge of the opponent's good and bad qualities is of the greatest importance."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"A chess player never has a heart attack in a good position."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"A computer beat me in chess, but it was no match when it came to kickboxing."#,
        r#"Emo Phillips"#,
    ),
    (
        r#"A considerable role in the forming of my style was played by an early attraction to study composition."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"A defeatist spirit must inevitably lead to disaster."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"A draw can be obtained not only by repeating moves, but also by one weak move."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"A draw may be the beautiful and logical result of fine attacks and parries; and the public ought to appreciate such games, in contrast, of course, to the fear-and-laziness draws."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"A gambit never becomes sheer routine as long as you fear you may lose the king and pawn ending!"#,
        r#"Bent Larsen"#,
    ),
    (
        r#"A great chess player always has a very good memory."#,
        r#"Leonid Shamkovich"#,
    ),
    (
        r#"A knight ending is really a pawn ending."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"A male scorpion is stabbed to death after mating. In chess, the powerful queen often does the same to the king without giving him the satisfaction of a lover."#,
        r#"Gregor Piatigorsky"#,
    ),
    (
        r#"A pawn, when separated from his fellows, will seldom or never make a fortune."#,
        r#"Francois-Andre Danican Philidor"#,
    ),
    (
        r#"A plan is made for a few moves only, not for the whole game."#,
        r#"Rueben Fine"#,
    ),
    (
        r#"A player can sometimes afford the luxury of an inaccurate move, or even a definite error, in the opening or middlegame without necessarily obtaining a lost position. In the endgame ... an error can be decisive, and we are rarely presented with a second chance."#,
        r#"Paul Keres"#,
    ),
    (
        r#"A real sacrifice involves a radical change in the character of a game which cannot be effected without foresight, fantasy, and the willingness to risk."#,
        r#"Leonid Shamkovich"#,
    ),
    (
        r#"A sport, a struggle for results and a fight for prizes. I think that the discussion about "chess is science or chess is art" is already inappropriate. The purpose of modern chess is to reach a result."#,
        r#"Alexander Morozevich"#,
    ),
    (
        r#"A strong player requires only a few minutes of thought to get to the heart of the conflict. You see a solution immediately, and half an hour later merely convince yourself that your intuition has not deceived you."#,
        r#"David Bronstein"#,
    ),
    (
        r#"A win gives one a feeling of self-affirmation, and success - a feeling of self-expression, but only a sensible harmonization between these urges can bring really great achievements in chess."#,
        r#"Oleg Romanishin"#,
    ),
    (
        r#"Above all else, before playing in competitions a player must have regard to his health, for if he is suffering from ill-health he cannot hope for success. In this connection the best of all tonics is 15 to 20 days in the fresh air, in the country."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"According to such great attacking players as Bronstein and Tal, most combinations are inspired by the player's memories of earlier games."#,
        r#"Pal Benko"#,
    ),
    (
        r#"After I won the title, I was confronted with the real world. People do not behave naturally anymore – hypocrisy is everywhere."#,
        r#"Boris Spassky"#,
    ),
    (
        r#"After a great deal of discussion in Soviet literature about the correct definition of a combination, it was decided that from the point of view of a methodical approach it was best to settle on this definition - A combination is a forced variation with a sacrifice."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Agreeing to draws in the middlegame, equal or otherwise, deprives you of the opportunity to practice playing endgames, and the endgame is probably where you need the most practice."#,
        r#"Pal Benko"#,
    ),
    (
        r#"All chess masters have on occasion played a magnificent game and then lost it by a stupid mistake, perhaps in time pressure and it may perhaps seem unjust that all their beautiful ideas get no other recognition than a zero on the tournament table."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"All chess players know what a combination is. Whether one makes it oneself, or is its victim, or reads of it, it stands out from the rest of the game and stirs one's admiration."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"All conceptions in the game of chess have a geometrical basis."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"All lines of play which lead to the imprisonment of the bishop are on principle to be condemned. (on the closed Ruy Lopez)"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"All that matters on the chessboard is good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"All that now seems to stand between Nigel and the prospect of the world crown is the unfortunate fact that fate brought him into this world only two years after Kasparov."#,
        r#"(prophetic comment in 1987) - Garry Kasparov"#,
    ),
    (
        r#"Along with my retirement from chess analytical work seems to have gone too."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Although the Knight is generally considered to be on a par with the Bishop in strength, the latter piece is somewhat stronger in the majority of cases in which they are opposed to each other."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Amberley excelled at chess - a mark, Watson, of a scheming mind."#,
        r#"Sir Arthur Conan Doyle"#,
    ),
    (
        r#"Americans really don't know much about chess. But I think when I beat Spassky, that Americans will take a greater interest in chess. Americans like winners."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Among top grandmasters the Dutch is a rare defense, which is good reason to play it! It has not been studied very deeply by many opponents, and theory, based on a small number of 'reliable' games, must be rather unreliable."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"An amusing fact: as far as I can recall, when playing the Ruy Lopez I have not yet once in my life had to face the Marshall Attack!"#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"An innovation need not be especially ingenious, but it must be well worked out."#,
        r#"Paul Keres"#,
    ),
    (
        r#"An isolated pawn spreads gloom all over the chessboard."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Analysis is a glittering opportunity for training: it is just here that capacity for work, perseverance and stamina are cultivated, and these qualities are, in truth, as necessary to a chess player as a marathon runner."#,
        r#"Lev Polugaevsky"#,
    ),
    (
        r#"Analysis, if it is really carried out with a complete concentration of his powers, forms and completes a chess player."#,
        r#"Lev Polugaevsky"#,
    ),
    (
        r#"Anyone who wishes to learn how to play chess well must make himself or herself thoroughly conversant with the play in positions where the players have castled on opposite sides."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Apart from direct mistakes, there is nothing more ruinous than routine play, the aim of which is mechanical development."#,
        r#"Alexei Suetin"#,
    ),
    (
        r#"As Rousseau could not compose without his cat beside him, so I cannot play chess without my king's bishop. In its absence the game to me is lifeless and void. The vitalizing factor is missing, and I can devise no plan of attack."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"As a chess player one has to be able to control one’s feelings, one has to be as cold as a machine."#,
        r#"Levon Aronian"#,
    ),
    (
        r#"As a rule, pawn endings have a forced character, and they can be worked out conclusively."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"As a rule, so-called "positional" sacrifices are considered more difficult, and therefore more praise-worthy than those which are based exclusively on an exact calculation of tactical possibilities."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"As a rule, the more mistakes there are in a game, the more memorable it remains, because you have suffered and worried over each mistake at the board."#,
        r#"Victor Korchnoi"#,
    ),
    (
        r#"As long as my opponent has not yet castled, on each move I seek a pretext for an offensive. Even when I realize that the king is not in danger."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"As often as not, his strategy consists of stifling Black's activity and then winning in an endgame thanks to his superior pawn structure."#,
        r#"Neil McDonald (1998)"#,
    ),
    (r#"Attack! Always Attack!"#, r#"Adolf Anderssen"#),
    (
        r#"Attackers may sometimes regret bad moves, but it is much worse to forever regret an opportunity you allowed to pass you by."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Avoidance of mistakes is the beginning, as it is the end, of mastery in chess."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"Barcza is the most versatile player in the opening. He sometimes plays P-KKt3 on the first, sometimes on the second, sometimes on the third, and sometimes only on the fourth move."#,
        r#"reputedly stated by Harry Golombek"#,
    ),
    (
        r#"Before Geller we did not understand the King's Indian Defence."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Begone! Ignorant and impudent knight, not even in chess can a King be taken."#,
        r#"King Louis VI (reputedly stated to one of his knights in 1110 after he was nearly captured by enemy forces)"#,
    ),
    (
        r#"Black's d5-square is too weak."#,
        r#"Ulf Andersson  (on the Dragon variation)"#,
    ),
    (r#"Blitz chess kills your ideas."#, r#"Bobby Fischer"#),
    (
        r#"Bobby Fischer started off each game with a great advantage: after the opening he had used less time than his opponent and thus had more time available later on. The major reason why he never had serious time pressure was that his rapid opening play simply left sufficient time for the middlegame."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Books on the openings abound; nor are works on the end game wanting; but those on the middle game can be counted on the fingers of one hand."#,
        r#"Harry Golombek"#,
    ),
    (
        r#"Boris Vasilievich was the only top-class player of his generation who played gambits regularly and without fear ... Over a period of 30 years he did not lose a single game with the King's Gambit, and among those defeated were numerous strong players of all generations, from Averbakh, Bronstein and Fischer, to Seirawan."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Botvinnik tried to take the mystery out of Chess, always relating it to situations in ordinary life. He used to call chess a typical inexact problem similar to those which people are always having to solve in everyday life."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"But alas! Like many another consummation devoutly to be wished, the actual performance was a disappointing one. (on the long awaited Lasker-Capablanca match in 1921)"#,
        r#"Fred Reinfeld"#,
    ),
    (
        r#"But how difficult it can be to gain the desired full point against an opponent of inferior strength, when this is demanded by the tournament position!"#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"But whatever you might say and whatever I might say, a machine which can play chess with people is one of the most marvellous wonders of our 20th century!"#,
        r#"David Bronstein"#,
    ),
    (
        r#"But you see when I play a game of Bobby, there is no style. Bobby played perfectly. And perfection has no style."#,
        r#"Miguel Najdorf"#,
    ),
    (
        r#"By all means examine the games of the great chess players, but don't swallow them whole. Their games are valuable not for their separate moves, but for their vision of chess, their way of thinking."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"By positional play a master tries to prove and exploit true values, whereas by combinations he seeks to refute false values ... A combination produces an unexpected re-assessment of values."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"By some ardent enthusiasts Chess has been elevated into a science or an art. It is neither; but its principal characteristic seems to be what human nature mostly delights in a fight."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"By strictly observing Botvinnik's rule regarding the thorough analysis of one's own games, with the years I have come to realize that this provides the foundation for the continuous development of chess mastery."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"By the mid-1990s the number of people with some experience of using computers was many orders of magnitude greater than in the 1960s. In the Kasparov defeat they recognized that here was a great triumph for programmers, but not one that may compete with the human intelligence that helps us to lead our lives."#,
        r#"Igor Aleksander"#,
    ),
    (
        r#"By the time a player becomes a Grandmaster, almost all of his training time is dedicated to work on this first phase. The opening is the only phase that holds out the potential for true creativity and doing something entirely new."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"By what right does White, in an absolutely even position, such as after move one, when both sides have advanced 1. e4, sacrifice a pawn, whose recapture is quite uncertain, and open up his kingside to attack? And then follow up this policy by leaving the check of the black queen open? None whatever !"#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Can you imagine the relief it gives a mother when her child amuses herself quietly for hours on end?"#,
        r#"Klara Polgar"#,
    ),
    (
        r#"Capablanca did not apply himself to opening theory (in which he never therefore achieved much), but delved deeply into the study of end-games and other simple positions which respond to technique rather than to imagination."#,
        r#"Max Euwe"#,
    ),
    (
        r#"Chess can help a child develop logical thinking, decision making, reasoning, and pattern recognition skills, which in turn can help math and verbal skills."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"Chess can learn a lot from poker. First, chess media and sponsors should emphasize its glamorous aspects: worldwide traveling, parties and escape from real world responsibilities."#,
        r#"Jennifer Shahade"#,
    ),
    (
        r#"Chess can never reach its height by following in the path of science ... Let us, therefore, make a new effort and with the help of our imagination turn the struggle of technique into a battle of ideas."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Chess continues to advance over time, so the players of the future will inevitably surpass me in the quality of their play, assuming the rules and regulations allow them to play serious chess. But it will likely be a long time before anyone spends 20 consecutive years as number one, as I did."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Chess is a bond of brotherhood amongst all lovers of the noble game, as perfect as free masonry. It is a leveller of rank - title, wealth, nationality, politics, religion - all are forgotten across the board."#,
        r#"Frederick Milne Edge"#,
    ),
    (
        r#"Chess is a contest between two men which lends itself particularly to the conflicts surrounding aggression."#,
        r#"Rueben Fine"#,
    ),
    (
        r#"Chess is a contributor to net human unhappiness, since the pleasure of victory is greatly exceeded by the pain of defeat."#,
        r#"Bill Hartston"#,
    ),
    (
        r#"Chess is a cure for headaches."#,
        r#"John Maynard Keynes"#,
    ),
    (
        r#"Chess is a game sufficiently rich in meaning that it is easily capable of containing elements of both tragedy and comedy."#,
        r#"Luke McShane"#,
    ),
    (
        r#"Chess is a game which reflects most honor on human wit."#,
        r#"Voltaire"#,
    ),
    (
        r#"Chess is a great game. No matter how good one is, there is always somebody better. No matter how bad one is, there is always somebody worse."#,
        r#"I.A. Horowitz"#,
    ),
    (
        r#"Chess is a matter of delicate judgement, knowing when to punch and how to duck."#,
        r#"Bobby Fischer"#,
    ),
    (r#"Chess is a matter of vanity."#, r#"Alexander Alekhine"#),
    (r#"Chess is a meritocracy."#, r#"Lawrence Day"#),
    (
        r#"Chess is a miniature version of life. To be successful, you need to be disciplined, assess resources, consider responsible choices and adjust when circumstances change."#,
        r#"Susan Polgar"#,
    ),
    (r#"Chess is a natural cerebral high."#, r#"Walter Browne"#),
    (
        r#"Chess is a sea in which a gnat may drink and an elephant may bathe."#,
        r#"Hindu proverb"#,
    ),
    (r#"Chess is a sport. A violent sport."#, r#"Marcel Duchamp"#),
    (r#"Chess is a test of wills."#, r#"Paul Keres"#),
    (
        r#"Chess is a unique cognitive nexus, a place where art and science come together in the human mind and are refined and improved by experience."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Chess is beautiful enough to waste your life for."#,
        r#"Hans Ree"#,
    ),
    (
        r#"Chess is eminently and emphatically the philosopher's game."#,
        r#"Paul Morphy"#,
    ),
    (
        r#"Chess is far too complex to be definitively solved with any technology we can conceive of today. However, our looked-down-upon cousin, checkers, or draughts, suffered this fate quite recently thanks to the work of Jonathan Schaeffer at the University of Alberta and his unbeatable program Chinook."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Chess is infinite, and one has to make only one ill-considered move, and one's opponent's wildest dreams will become reality."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Chess is like a language, the top players are very fluent at it. Talent can be developed scientifically but you have to find first what you are good at."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"Chess is like body-building. If you train every day, you stay in top shape. It is the same with your brain – chess is a matter of daily training."#,
        r#"Vladimir Kramnik"#,
    ),
    (r#"Chess is my life."#, r#"Victor Korchnoi"#),
    (
        r#"Chess is my profession. I am my own boss; I am free. I like literature and music, classical especially. I am in fact quite normal; I have a Bohemian profession without being myself a Bohemian. I am neither a conformist nor a great revolutionary."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"Chess is not for the faint-hearted; it absorbs a person entirely. To get to the bottom of this game, he has to give himself up into slavery. Chess is difficult, it demands work, serious reflection and zealous research."#,
        r#"Wilhelm Steinitz"#,
    ),
    (r#"Chess is not for the timid."#, r#"Irving Chernev"#),
    (
        r#"Chess is not relaxing ; it's stressful even if you win."#,
        r#"Jennifer Shahade"#,
    ),
    (r#"Chess is one long regret."#, r#"Stephen Leacock"#),
    (
        r#"Chess is only a recreation and not an occupation."#,
        r#"Vladimir Lenin"#,
    ),
    (
        r#"Chess is something more than a game. It is an intellectual diversion which has certain artistic qualities and many scientific elements."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Chess is the touchstone of intellect."#,
        r#"Johann Wolfgang von Goethe"#,
    ),
    (
        r#"Chess is thirty to forty percent psychology. You don't have this when you play a computer. I can't confuse it."#,
        r#"Judit Polgar"#,
    ),
    (
        r#"Chess is thriving. There are ever less round robin tournaments and ever more World Champions."#,
        r#"Robert Hübner (1990, Schach)"#,
    ),
    (r#"Chess is, above all, a fight."#, r#"Emanuel Lasker"#),
    (
        r#"Chess is, in essence, a game for children. Computers have exacerbated the trends towards youth because they now have an immensely powerful tool at their disposal and can absorb vast amounts of information extremely quickly."#,
        r#"Nigel Short"#,
    ),
    (
        r#"Chess masters as well as chess computers deserve less reverence than the public accords them."#,
        r#"Eliot Hearst"#,
    ),
    (
        r#"Chess programs are our enemies, they destroy the romance of chess. They take away the beauty of the game. Everything can be calculated."#,
        r#"Levon Aronian"#,
    ),
    (
        r#"Chess strategy as such today is still in its diapers, despite Tarrasch's statement 'We live today in a beautiful time of progress in all fields'. Not even the slightest attempt has been made to explore and formulate the laws of chess strategy."#,
        r#"Aron Nimzowitsch (1925)"#,
    ),
    (
        r#"Chess strength in general and chess strength in a specific match are by no means one and the same thing."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Chess will always be in the doldrums as a spectator sport while a draw is given equal mathematical value as a decisive result."#,
        r#"Michael Basman"#,
    ),
    (
        r#"Chess, like love, is infectious at any age."#,
        r#"Salo Flohr"#,
    ),
    (
        r#"Chess-play is a good and witty exercise of the mind for some kind of men, but if it proceed from overmuch study, in such a case it may do more harm than good; it is a game too troublesome for some men's brains."#,
        r#"Robt. Burton (1621) (clergyman and Librarian at Oxford University)"#,
    ),
    (
        r#"Combinations with a queen sacrifice are among the most striking and memorable ..."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"Concentrate on material gains. Whatever your opponent gives you take, unless you see a good reason not to."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Condemned by theory, the Allgaier, certainly one of the most romantic of gambits, is generally successful in practice (and yet so rarely played). Why does the defender often seem hypnotized, quite demoralized?"#,
        r#"Tony Santasiere"#,
    ),
    (
        r#"Confidence is very important – even pretending to be confident. If you make a mistake but do not let your opponent see what you are thinking then he may overlook the mistake."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"Contrary to many young colleagues I do believe that it makes sense to study the classics."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"Deschapelles became a first-rate player in three days, at the age of something like thirty. Nobody ever believed the statement, not even Deschapelles himself, although his biographer declares he had told the lie so often that he at last forgot the facts of the case."#,
        r#"Frederick Milne Edge"#,
    ),
    (
        r#"Despite the development of chess theory, there is much that remains secret and unexplored in chess."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"Do not bring your Queen out too early."#,
        r#"from Francisco Bernardina Calogno's poem 'On the Game of Chess' circa 1500"#,
    ),
    (
        r#"Do not permit yourself to fall in love with the end-game play to the exclusion of entire games. It is well to have the whole story of how it happened; the complete play, not the denouement only. Do not embrace the rag-time and vaudeville of chess."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Do not pick a move from a list of computer lines - use your own brains. This is important, especially for young players. It's better to study a worse line well than to reproduce a better computer line."#,
        r#"Laszlo Hazai"#,
    ),
    (
        r#"Don't be afraid of losing, be afraid of playing a game and not learning something."#,
        r#"Dan Heisman"#,
    ),
    (
        r#"Don't worry about your rating, work on your playing strength and your rating will follow."#,
        r#"Dan Heisman"#,
    ),
    (
        r#"Don't worry kids, you'll find work. After all, my machine will need strong chess player-programmers. You will be the first."#,
        r#"Mikhail Botvinnik (to Karpov & students, 1965)"#,
    ),
    (
        r#"Drawn games are sometimes more scintillating than any conclusive contest."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"During a chess tournament a master must envisage himself as a cross between an ascetic monk and a beast of prey."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"During the late Victorian period the majority of chess magazines printed increasing numbers of humorous stories, poems and anecdotes about the agonies and idiocies of women chess players, presumably as an antidote to the alarmed reaction of men to the fact that women were encroaching on their 'territory'."#,
        r#"British Chess Magazine"#,
    ),
    (
        r#"Emotional instability can be one of the factors giving rise to a failure by chess players in important duels. Under the influence of surging emotions (and not necessarily negative ones) we sometimes lose concentration and stop objectively evaluating the events that are taking place on the board."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"Endings of one rook and pawns are about the most common sort of endings arising on the chess board. Yet though they do occur so often, few have mastered them thoroughly. They are often of a very difficult nature, and sometimes while apparently very simple they are in reality extremely intricate."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Enormous self-belief, intuition, the ability to take a risk at a critical moment and go in for a very dangerous play with counter-chances for the opponent - it is precisely these qualities that distinguish great players."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Errors have nothing to do with luck; they are caused by time pressure, discomfort or unfamiliarity with a position, distractions, feelings of intimidation, nervous tension, over-ambition, excessive caution, and dozens of other psychological factors."#,
        r#"Pal Benko"#,
    ),
    (
        r#"Even in the King's Gambit ... White is no longer trying to attack at all costs. He has had to adapt his approach and look for moves with a solid positional foundation"#,
        r#"Neil McDonald  (1998)"#,
    ),
    (
        r#"Even in the heat of a middlegame battle the master still has to bear in mind the outlines of a possible future ending."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Even the best grandmasters in the world have had to work hard to acquire the technique of rook endings."#,
        r#"Paul Keres"#,
    ),
    (
        r#"Even the most distinguished players have in their careers experienced severe disappointments due to ignorance of the best lines or suspension of their own common sense."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"Even when the time control has been reached, there is one situation where you want to act as if it has not: when your position is absolutely lost."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Every Chess master was once a beginner."#,
        r#"Irving Chernev"#,
    ),
    (
        r#"Every great master will find it useful to have his own theory on the openings, which only he himself knows, a theory which is closely linked with plans for the middle game."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Every month I look through some ten thousand games, so not as to miss any new ideas and trends."#,
        r#"Vladimir Kramnik"#,
    ),
    (r#"Every move creates a weakness."#, r#"Siegbert Tarrasch"#),
    (
        r#"Excellent ! I will still be in time for the ballet !"#,
        r#"Jose Capablanca (upon defeating Ossip Bernstein in the famous 29 move exhibition game played in Moscow in 1914, before setting off to the Bolshoi Theatre in horse-drawn carriage)"#,
    ),
    (
        r#"Excelling at chess has long been considered a symbol of more general intelligence. That is an incorrect assumption in my view, as pleasant as it might be."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Experience and the constant analysis of the most varied positions builds up a store of knowledge in a player's mind enabling him often at a glance to assess this or that position."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Failing to open the center at the right moment - a common error by White in the Exchange Lopez - can allow Black an excellent game."#,
        r#"Andy Soltis"#,
    ),
    (
        r#"Far from all of the obvious moves that go without saying are correct."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Few things are as psychologically brutal as chess."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"First and foremost it is essential to understand the essence, the overall idea of any fashionable variation, and only then include it in one's repertoire. Otherwise the tactical trees will conceal from the player the strategic picture of the wood, in which his orientation will most likely be lost."#,
        r#"Lev Polugaevsky"#,
    ),
    (
        r#"First restrain, next blockade, lastly destroy."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"First-class players lose to second-class players because second-class players sometimes play a first-class game."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Fischer is Fischer, but a knight is a knight!"#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"For a game it is too serious, for seriousness too much of a game."#,
        r#"Moses Mendelssohn 1729-86"#,
    ),
    (
        r#"For every door the computers have closed they have opened a new one."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"For me right now I think being the world number one is a bigger deal than being the world champion because I think it shows better who plays the best chess. That sounds self-serving but I think it’s also right. (2012)"#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"For me, chess is a language, and if it's not my native tongue, it is one I learned via the immersion method at a young age."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"For me, chess is at the same time a game, a sport, a science and an art. And perhaps even more than that,. There is something hard to explain to those who do not know the game well. One must first learn to play it correctly in order to savor its richness."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"For me, chess is not a profession, it is a way of life, a passion. People may feel that I have conquered the peak and will not have to struggle. Financially, perhaps that is true; but as far as chess goes, I’m still learning a lot!"#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"For my victory over Capablanca I am indebted primarily to my superiority in the field of psychology. Capablanca played, relying almost exclusively on his rich intuitive talent. But for the chess struggle nowadays one needs a subtle knowledge of human nature, an understanding of the opponent's psychology."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"For pleasure you can read the games collections of Andersson and Chigorin, but for benefit you should study Tarrasch, Keres and Bronstein."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Fortunately I’ve got a weak character, so I never did decide to dedicate myself to only one of my professions. And I’m very glad. After all, if I’d rejected chess or music then my life wouldn’t have been two times, but a hundred times less interesting."#,
        r#"Mark Taimanov"#,
    ),
    (
        r#"From time to time, like many other players, I glance through my own games of earlier years, and return to positions and variations which have gone out of practice. I attempt to restore them, to find new ideas and plans."#,
        r#"Yefim Geller"#,
    ),
    (
        r#"Furman astounded me with his chess depth, a depth which he revealed easily and naturally, as if all he were doing was establishing well-known truths."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"GM Naiditsch reckoned that me playing the King's Indian against Anand was something akin to a samurai running at a machine gun with a sword."#,
        r#"Hikaru Nakamura"#,
    ),
    (
        r#"Genius. It's a word. What does it really mean? If I win I'm a genius. If I don't, I'm not."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Go through detailed variations in your own time, think in a general way about the position in the opponent's time and you will soon find that you get into time trouble less often, that your games have more content to them, and that their general standard rises."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Had I not played the Sicilian with Black I could have saved myself the trouble of studying for more than 20 years all the more popular lines of this opening, which comprise probably more than 25 percent of all published opening theory!"#,
        r#"Bent Larsen"#,
    ),
    (
        r#"Has he some psychological antipathy to realism? I am no psychologist, and cannot say. The fact remains that Euwe commits the most inexplicable mistakes in thoroughly favorable positions, and that this weakness has consistently tarnished his record."#,
        r#"Hans Kmoch"#,
    ),
    (
        r#"Haste is never more dangerous than when you feel that victory is in your grasp."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (r#"Haste, the great enemy."#, r#"Eugene Znosko-Borowsky"#),
    (
        r#"Having spent alarmingly large chunks of my life studying the white side of the Open Sicilian, I find myself asking, why did I bother?"#,
        r#"Daniel King"#,
    ),
    (
        r#"He played with enormous energy and great fighting spirit. Offering him a draw was a waste of time. He would decline it politely, but firmly. "No, thank you," he would say and the fight would go on and on and on."#,
        r#"Lubomir Kavalek on Bent Larsen"#,
    ),
    (
        r#"He who has a slight disadvantage plays more attentively, inventively and more boldly than his antagonist who either takes it easy or aspires after too much. Thus a slight disadvantage is very frequently seen to convert into a good, solid advantage."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Here is a definition which correctly reflects the course of thought and action of a grandmaster: The plan in a game of chess is the sum total of successive strategical operations which are each carried out according to separate ideas arising from the demands of the position."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"How come the little things bother you when you are in a bad position? They don't bother you in good positions."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"However hopeless the situation appears to be there yet always exists the possibility of putting up a stubborn resistance."#,
        r#"Paul Keres"#,
    ),
    (
        r#"I ... have two vocations: chess and engineering. If I played chess only, I believe that my success would not have been significantly greater. I can play chess well only when I have fully convalesced from chess and when the 'hunger for chess' once more awakens within me."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"I always urge players to study composed problems and endgames."#,
        r#"Pal Benko"#,
    ),
    (
        r#"I am acutely conscious, from vast experience in opens, that guys around, say 2100 or more can definitely play chess and that one often has to work very hard to beat them."#,
        r#"Nigel Short"#,
    ),
    (
        r#"I am both sad and pleased that in his last tournament, Rashid Gibiatovich came to my home in Latvia. He did not take first place, but the prize for beauty, as always, he took with him. Players die, tournaments are forgotten, but the works of great artists are left behind them to live on forever. (on Nezhmetdinov)"#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"I am pleased that in a match for the World Championship I was able to conduct a game in the style of Akiba Rubinstein, where the entire strategic course was maintained from the first to the last move. (on Game 7 of his 2012 match with Anand)"#,
        r#"Boris Gelfand"#,
    ),
    (
        r#"I am trying to beat the guy sitting across from me and trying to choose the moves that are most unpleasant for him and his style."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"I believe in magic ... There is magic in the creative faculty such as great poets and philosophers conspicuously possess, and equally in the creative chessmaster."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"I believe most definitely that one must not only grapple with the problems on the board, one must also make every effort to combat the thoughts and will of the opponent."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"I believe that the best style is a universal one, tactical and positional at the same time ..."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"I cannot claim to thoroughly enjoy coaching, because it is very hard work if you are even moderately conscientious. Nevertheless it does provide a degree of satisfaction, not to mention a steady income, which is why I do it occasionally."#,
        r#"Nigel Short"#,
    ),
    (
        r#"I cannot think that a player genuinely loving the game can get pleasure just from the number of points scored no matter how impressive the total. I will not speak of myself, but for the masters of the older generation, from whose games we learned, the aesthetic side was the most important."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"I can’t count the times I have lagged seemingly hopelessly far behind, and nobody except myself thinks I can win. But I have pulled myself in from desperate [situations]. When you are behind there are two strategies – counter-attack or all men to the defences. I’m good at finding the right balance between those."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"I claim that nothing else is so effective in encouraging the growth of chess strength as such independent analysis, both of the games of the great players and your own."#,
        r#"Mikhail Botvinnik"#,
    ),
    (r#"Watch out for the tricky knights."#, r#"Chess Network"#),
    (
        r#"I think crazyhouse improves your standard chess."#,
        r#"Chess Network"#,
    ),
    (
        r#"The biggest tool for chess improvement would be playing against stronger opposition"#,
        r#"Peter Svidler"#,
    ),
    (
        r#"Chess is a battle between your aversion to thinking and your aversion to losing."#,
        r#"Anonymous"#,
    ),
    (
        r#"It was once said that Tal sacrificed 9 pawns for an attack"#,
        r#"Mato"#,
    ),
    (
        r#"Be well enough prepared that preparation won't play a role."#,
        r#"Magnus Carlsen"#,
    ),
    (r#"I don't study; I create."#, r#"Viktor Korchnoi"#),
    (
        r#"During the analysis, I discovered something very remarkable: the board is simply too small for two Queens of the same color. They only get in each other's way. I realize that this might sound stupid, but I fully mean it. The advantage is much less than one would expect by counting material."#,
        r#"Viktor Korchnoi"#,
    ),
    (
        r#"You'll be amazed at the people I've lost to while playing online."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"[...], even extremely intoxicated my chess strength and knowledge is still in my bones."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"I don't play unorthodox openings. I prefer to give mainstream openings my own spin."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"Playing long games online just takes too much time. It's fun to play blitz once in a while, where you can rely more on your intuition, your instincts rather than pure calculation and analysis."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"Fortune favors the lucky!"#,
        r#"Robert Houdart (Houdini author)"#,
    ),
    (
        r#"I don't berserk, I am not a caveman"#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"1.e4 is the move you play when you're young, naive, and believe the world owes you something. Open positions, infinite horizons - what's not to love? Well, I've got news for you, buddy: it's a cruel chess board out there."#,
        r#"John Bartholomew"#,
    ),
    (
        r#"Chess as a game is too serious; as a serious pursuit too frivolous."#,
        r#"Moses Mendelssohn"#,
    ),
    (r#"Chess makes me a better person"#, r#"Albert Badosa"#),
    (
        r#"All features for free; for everyone; forever."#,
        r#"lichess.org"#,
    ),
    (r#"We will never display ads."#, r#"lichess.org"#),
    (
        r#"We do not track you. It's a rare feature, nowadays."#,
        r#"lichess.org"#,
    ),
    (r#"Every chess player is a premium user."#, r#"lichess.org"#),
    (
        r#"I never lose. I either win or learn."#,
        r#"Nelson Mandela"#,
    ),
    (
        r#"For me art and chess are closely related, both are forms in which the self finds beauty and expression."#,
        r#"Vladimir Kramnik"#,
    ),
    (
        r#"No fantasy, however rich, no technique, however masterly, no penetration into the psychology of the opponent, however deep, can make a chess game a work of art, if these qualities do not lead to the main goal - the search for truth."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"By strictly observing Botvinnik's rule regarding the thorough analysis of one's own games, with the years I have come to realize that this provides the foundation for the continuous development of chess mastery."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"On the chessboard, lies and hypocrisy do not survive long. The creative combination lays bare the presumption of a lie; the merciless fact, culminating in the checkmate, contradicts the hypocrite."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"'There is nothing new under the sun. It has all been done before.'"#,
        r#"Sherlock Holmes quoted in A Study in Scarlet"#,
    ),
    (
        r#"Everything has been thought of before"#,
        r#"the trick is to think of it again... – paraphrasing Goethe"#,
    ),
    (
        r#"Let no one say that I have said nothing new... the arrangement of the subject is new."#,
        r#"Blaise Pascal, Pensées (1670)"#,
    ),
    (
        r#"Words differently arranged have a different meaning and meanings differently arranged have a different effect."#,
        r#"Blaise Pascal"#,
    ),
    (
        r#"… though combinations are without number, the number of ideas are limited."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"Daring ideas are like Chess men moved forward. They may be beaten, but they may start a winning game."#,
        r#"Johann Wolfgang von Goethe"#,
    ),
    (
        r#"Of Chess it has been said that life is not long enough for it, but that is the fault of life, not chess."#,
        r#"William Ewart Napier"#,
    ),
    (
        r#"For me, Chess is life and every game is like a new life. Every chess player gets to live many lives in one lifetime."#,
        r#"Eduard Gufeld"#,
    ),
    (
        r#"That's what chess is all about. One day you give your opponent a lesson, the next day he gives you one."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Chess is a forcing house where the fruits of character can ripen more fully than in life."#,
        r#"Edward Morgan Foster"#,
    ),
    (
        r#"Chess is a miniature version of life. To be successful, you need to be disciplined, assess resources, consider responsible choices and adjust when circumstances change."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"The game of Chess is not merely an idle amusement; several very valuable qualities of the mind are to be acquired and strengthened by it, so as to become habits ready on all occasions, for life is a kind of chess."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"Viewed in terms of psychoanalytic theory, the invention of chess expressed the triumph of secondary process thinking over the primary process. Actual warfare (is replaced by) a struggle which is organized, controlled, circumscribed and regulated."#,
        r#"Norman Reider"#,
    ),
    (
        r#"The slightest acquaintance with chess shows that it is a play-substitute for the art of war, and indeed it has been a favorite recreation of some of the greatest military leaders, from William the Conqueror to Napoleon. In the contest between the opposing armies the same tactics are displayed as in actual war, the same foresight and powers of calculation are necessary, the same capacity for divining the plans of the opponent, and the rigour with which decisions are followed by their consequences is, if anything, even more ruthless."#,
        r#"Ernest Jones"#,
    ),
    (
        r#"Chess may be but a game, a pastime, a relaxation; but Chess has at times absorbed the faculties of the intellectual in every clime; it numbers amongst its amateurs the greatest names of battle-fields and thrones; it tells of warriors, poets, painters, sculptors, statesmen and divines; it possesses a literature and language of its own; it makes enemies friends, and finds a temple on the ocean, in the fortress, and by the peaceful fireside."#,
        r#"Frederick Milne Edge"#,
    ),
    (
        r#"When a chess player looks at the board, he does not see a static mosaic, a 'still life', but a magnetic field of forces, charged with energy - as Faraday saw the stresses surrounding magnets and currents as curves in space; or as Van Gogh saw vortices in the skies of Provence."#,
        r#"Arthur Koestler"#,
    ),
    (
        r#"The Chess pieces are the block alphabet which shapes thoughts; and these thoughts, although making a visual design on the chessboard, express their beauty abstractly, like a poem."#,
        r#"Marcel Duchamp"#,
    ),
    (
        r#"The chess-board is the world, the pieces are the phenomena of the universe, the rules of the game are what we call the laws of Nature. The player on the other side is hidden from us. We know that his play is always fair, just, and patient. But we also know, to our cost, that he never overlooks a mistake, or makes the smallest allowance for ignorance."#,
        r#"Thomas Henry Huxley"#,
    ),
    (
        r#"Combinations have always been the most intriguing aspect of Chess. The masters look for them, the public applauds them, the critics praise them. It is because combinations are possible that chess is more than a lifeless mathematical exercise. They are the poetry of the game; they are to chess what melody is to music. They represent the triumph of mind over matter."#,
        r#"Rueben Fine"#,
    ),
    (
        r#"The pleasure to be derived from a chess combination lie in the feeling that a human mind is behind the game, dominating the inanimate pieces ... and giving them breath of life."#,
        r#"Richard Reti"#,
    ),
    (r#"Chess is a cold bath for the mind."#, r#"Sir John Simon"#),
    (
        r#"The Game of Chess is not merely an idle amusement; several very valuable qualities of the mind, useful in the course of human life, are to be acquired and strengthened by it, so as to become habits ready on all occasions; for life is a kind of Chess, in which we have points to gain, and competition or adversaries to contend with, and in which there is a vast variety of good and ill events, that are, in some degree, the effect of prudence, or want of it. By playing at Chess then, we may learn: First, Foresight; Second, Circumspection; Third, Caution; And lastly, We learn by Chess the habit of not being discouraged by present bad appearances in the state of our affairs; the habit of hoping for a favorable chance, and that of persevering in the secrets of resources."#,
        r#"Benjamin Franklin, 1779"#,
    ),
    (
        r#"When you are lonely, when you feel yourself an alien in the world, play Chess. This will raise your spirits and be your counselor in war."#,
        r#"Aristotle"#,
    ),
    (
        r#"No man is fit to command another that cannot command himself."#,
        r#"William Penn"#,
    ),
    (
        r#"Be an example to your men, in your duty and in private life. Never spare yourself, and let the troops see that you don't in your endurance of fatigue and privation. Always be tactful and well-mannered and teach your subordinates to do the same. Avoid excessive sharpness or harshness of voice, which usually indicates the man who has shortcomings of his own to hide."#,
        r#"Field Marshall Erwin Rommel"#,
    ),
    (
        r#"Mastering others is strength. Mastering yourself is true power."#,
        r#"Lao Tzu"#,
    ),
    (
        r#"Most of the chess masters of the first rank are men of culture, men of good social as well as intellectual training, as such qualities become more and more necessary every day."#,
        r#"Capablanca"#,
    ),
    (
        r#"Chess is so interesting in itself, as not to need the view of gain to induce engaging in it; and thence it is never played for money."#,
        r#"Benjamin Franklin, (Chess Made Easy, 1802)"#,
    ),
    (
        r#"The stock market and the gridiron and the battlefield aren't as tidy as the chessboard, but in all of them, a single, simple rule holds true: make good decisions and you'll succeed; make bad ones and you'll fail."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"[Chess] is the finest mental exercise. It develops concentration and logical reasoning; and it is one of the few games in which you cannot rectify a mistake. If you make a mistake, you lose, unless your opponent makes a worse mistake."#,
        r#"Capablanca (1919)"#,
    ),
    (
        r#"In all forms of strategy, it is necessary to maintain the combat stance in everyday life and to make your everyday stance your combat stance. You must research this well."#,
        r#"Miyamoto Musashi"#,
    ),
    (
        r#"A prince should therefore have no other aim or thought, nor take up any other thing for his study but war and it's organization and discipline, for that is the only art that is necessary to one who commands."#,
        r#"Niccolo Machiavelli, The Prince"#,
    ),
    (
        r#"Only a warrior chooses pacifism; others are condemned to it."#,
        r#"unknown"#,
    ),
    (
        r#"A cowardly act! What do I care about that? You may be sure that I should never fear to commit one if it were to my advantage."#,
        r#"Napoleon In war the heroes always outnumber the soldiers ten to one. – John Gray"#,
    ),
    (
        r#"Avoid the crowd. Do your own thinking independently. Be the Chess player, not the Chess piece"#,
        r#"Ralph Charell"#,
    ),
    (
        r#"In chess, at least, the brave inherit the earth."#,
        r#"Edmar Mednis, on the play of Tal"#,
    ),
    (r#"One mind, any weapon."#, r#"Hunter B. Armstrong"#),
    (
        r#"The pupil wants not so much to learn, as to learn how to learn."#,
        r#"Samuel Boden"#,
    ),
    (
        r#"The key to ultimate success is the determination to progress day by day."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"I’m not a materialistic person, in that, I don’t suffer the lack or loss of money. The absence of worldly goods I don’t look back on. For Chess is a way I can be as materialistic as I want without having to sell my soul."#,
        r#"Jamie Walter Adams"#,
    ),
    (
        r#"Great results, can be achieved with small forces."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Today is victory over yourself of yesterday; tomorrow is your victory over lesser men."#,
        r#"Musashi"#,
    ),
    (
        r#"No chess grandmaster is normal; they only differ in the extent of their madness."#,
        r#"Viktor Korchnoi"#,
    ),
    (
        r#"Just as the pianist practices the most complicated pieces to improve the technique of his fingers, so too a grandmaster must keep his vision in trim by daily analysis of positions with sharp possibilities, and this applies whether he prefers such positions in his play or not."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Play the move that forces the win in the simplest way. Leave the brilliancies to Alekhine, Keres and Tal."#,
        r#"Irving Chernev"#,
    ),
    (
        r#"Alekhine is a poet who creates a work of art out of something that would hardly inspire another man to send home a picture post card."#,
        r#"Max Euwe"#,
    ),
    (
        r#"It would be idle, and presumptuous, to wish to imitate the achievements of a Morphy or an Alekhine; but their methods and their manner of expressing themselves are within the reach of all."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"Chess problems demand from the composer the same virtues that characterize all worthwhile art: originality, invention, conciseness, harmony, complexity, and splendid insincerity."#,
        r#"Vladimir Nabokov, Poems and Problems, 1969"#,
    ),
    (
        r#"Chess, like any creative activity, can exist only through the combined efforts of those who have creative talent, and those who have the ability to organize their creative work."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"I have always had a very vivid imagination, which I have, after a long struggle, partly succeeded in controlling in order to use it to better purpose, according to the requirements of the occasion."#,
        r#"Capablanca"#,
    ),
    (
        r#"No fantasy, however rich, no technique, however masterly, no penetration into the psychology of the opponent, however deep, can make a chess game a work of art, if these qualities do not lead to the main goal - the search for truth."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"The process of making pieces in Chess do something useful (whatever it may be) has received a special name: it is called the attack. The attack is that process by means of which you remove obstructions."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"On the chessboard lies and hypocrisy do not survive long. The creative combination lays bare the presumption of a lie; the merciless fact, culminating in a checkmate, contradicts the hypocrite."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Truth derives its strength not so much from itself as from the brilliant contrast it makes with what is only apparently true. This applies especially to Chess, where it is often found that the profoundest moves do not much startle the imagination."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Everything in war is very simple, but the simplest thing is difficult."#,
        r#"Clausewitz"#,
    ),
    (
        r#"Botvinnik tried to take the mystery out of Chess, always relating it to situations in ordinary life. He used to call Chess a typical inexact problem similar to those which people are always having to solve in everyday life."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Chess is a very logical game and it is the man who can reason most logically and profoundly in it that ought to win."#,
        r#"Capablanca"#,
    ),
    (
        r#"The Chess pieces are the block alphabet which shapes thoughts; and these thoughts, although making a visual design on the chessboard, express their beauty abstractly, like a poem."#,
        r#"Marcel Duchamp"#,
    ),
    (
        r#"Via the squares on the chessboard, the Indians explain the movement of time and the age, the higher influences which control the world and the ties which link Chess with the human soul."#,
        r#"Al-Masudi"#,
    ),
    (
        r#"The battle for the ultimate truth will never be won. And that's why chess is so fascinating."#,
        r#"Hans Kmoch"#,
    ),
    (
        r#"Ultimately, you must forget about technique. The further you progress, the fewer teachings there are. The Great Path is really No Path."#,
        r#"Ueshiba Morihei"#,
    ),
    (
        r#"We are usually convinced more easily by reasons we have found ourselves than by those which have occurred to others."#,
        r#"Blaise Pascal, Pensées (1670)"#,
    ),
    (
        r#"All that matters on the chessboard is good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"When everything on the board is clear it can be so difficult to conceal your thoughts from your opponent."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Its just you and your opponent at the board and you’re trying to prove something."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Turning chess into poker and hoping for a "bluff" is not one of my convictions."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"On the chessboard lies and hypocrisy do not last long."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"In chess, as it is played by masters, chance is practically eliminated."#,
        r#"Emanuel Lasker"#,
    ),
    (r#"The loser is always at fault."#, r#"Vasily Panov"#),
    (
        r#"The concept of 'talent' is formed under completely abstract criteria, having nothing in common with reality. But the reality is such that I don’t understand chess as a whole. But then again no one understands chess in its entirety. Perhaps talent is something else, in chess it is conditionality."#,
        r#"Alexander Morozevich"#,
    ),
    (
        r#"Our knowledge of circumstances has increased, but our uncertainty, instead of having diminished, has only increased. The reason of this is, that we do not gain all our experience at once, but by degrees; so our determinations continue to be assailed incessantly by fresh experience; and the mind, if we may use the expression, must always be under arms."#,
        r#"Clausewitz"#,
    ),
    (
        r#"The laws of circumstance are abolished by new circumstances."#,
        r#"Napoleon"#,
    ),
    (
        r#"Nothing is so healthy as a trashing at the proper time, and from few won games have I learned as much as I have from most of my defeats."#,
        r#"Capablanca"#,
    ),
    (
        r#"Do not mind losing, for it is only by learning that you will improve, and by losing, if you use the knowledge you gained, you will improve rapidly. If you play with a much better player, so much more likely that you will learn. Any ordinary man can learn a great deal of chess just as of music, art or science, if he cares to devote his time and attention to study of the game."#,
        r#"Capablanca"#,
    ),
    (
        r#"In order to make progress in chess, it is necessary to pay special attention to all the general principles, spending a little less time on the openings. Play the openings on the basis of your general knowledge of how to mobilize pieces and do not become involved in technicalities about whether the books recommend this or that move; to learn the openings by heart it is necessary to study a great number of books which, moreover, are sometimes wrong. However, if you study from the point of view of the general principles you are taking a more certain path for although a player’s intellect can fail at a given moment, principles well used never fail."#,
        r#"Capablanca"#,
    ),
    (
        r#"If the point of playing chess is as a battle of the intellect then most people would say that the memorization of other peoples ideas is something that is anathema to the spirit of chess."#,
        r#"Nigel Davies"#,
    ),
    (
        r#"Lead the ideas of your time and they will accompany and support you; fall behind them and they drag you along with them; oppose them and they will overwhelm you."#,
        r#"Napoleon"#,
    ),
    (
        r#"The Nation that makes a great distinction between its scholars and its warriors will have its thinking done by cowards and its fighting done by fools."#,
        r#"Thucydides"#,
    ),
    (
        r#"It is said the warrior's is the twofold Way of pen and sword, and he should have a taste for both Ways. Even if a man has no natural ability he can be a warrior by sticking assiduously to both divisions of the Way."#,
        r#"Miyamoto Musashi"#,
    ),
    (
        r#"You work for a long period of time and the results don't really show, but at some point everything just comes together and you start to play better, or get more confidence."#,
        r#"Fabiano Caruana"#,
    ),
    (
        r#"In all forms of strategy, it is necessary to maintain the combat stance in everyday life and to make your everyday stance your combat stance. You must research this well."#,
        r#"Miyamoto Musashi"#,
    ),
    (
        r#"The more you sweat in peace, the less you bleed in war."#,
        r#"George Hyman Rickover"#,
    ),
    (
        r#"It has been said that man is distinguished from animal in that he buys more books than he can read. I should like to suggest that the inclusion of a few chess books would help to make the distinction unmistakable."#,
        r#"Edward Lasker"#,
    ),
    (
        r#"Chess books should be used as we use glasses: to assist the sight, although some players make use of them as if they conferred sight."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Fools say that they learn by experience. I prefer to profit by others' experience."#,
        r#"Otto von Bismark"#,
    ),
    (
        r#"It is hardly useful if you trustingly play through variation after variation from a book. It is a great deal more useful and more interesting if you take part actively in the analysis, find something yourself, and try to refute some of the author's conclusions."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"Ninety percent of the book variations have no great value, because either they contain mistakes or they are based on fallacious assumptions; just forget about the openings and spend all that time on the endings."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"The most intelligent inspection of any number of fine paintings will not make the observer a painter, nor will listening to a number of operas make the hearer a musician, but good judges of music and painting may so be formed. Chess differs from these. The intelligent perusal of fine games cannot fail to make the reader a better player and a better judge of the play of others."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The young people have read my book. Now I have no chance."#,
        r#"Efim Bogolubow"#,
    ),
    (
        r#"Act like a man of thought. Think like a man of action."#,
        r#"Thomas Mann"#,
    ),
    (
        r#"Just as one man can beat ten, so a hundred men can beat a thousand, and a thousand men can beat ten thousand. In my strategy, one man is the same as ten thousand, so this strategy is the complete warrior's craft."#,
        r#"Miyamoto Musashi"#,
    ),
    (
        r#"There are two classes of men; those who are content to yield to circumstances and who play whist; those who aim to control circumstances, and who play chess."#,
        r#"Mortimer Collins"#,
    ),
    (
        r#"Whether in an advantageous position or a disadvantageous one, the opposite state should be always present to your mind."#,
        r#"Ts'ao Kung"#,
    ),
    (
        r#"Question to Rubinstein: "Who is your opponent tonight?" Answer: "Tonight I am playing against the black pieces.""#,
        r#"Akiba Rubinstein"#,
    ),
    (
        r#"Knowing the enemy enables you to take the offensive, knowing yourself enables you to stand on the defensive."#,
        r#"Sun Tzu"#,
    ),
    (r#"Every move creates a weakness."#, r#"Siegbert Tarrasch"#),
    (
        r#"Invincibility lies in the defense; the possibility of victory in the attack."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Chess is eminently and emphatically the philosopher's game."#,
        r#"Paul Morphy"#,
    ),
    (
        r#"During a chess tournament a master must envisage himself as a cross between an ascetic monk and a beast of prey."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Chess is a miniature version of life. To be successful, you need to be disciplined, assess resources, consider responsible choices and adjust when circumstances change."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"For success I consider three factors are necessary: firstly, an awareness of my own strengths and weaknesses; secondly, an accurate understanding of my opponent's strengths and weaknesses; thirdly, a higher aim than momentary satisfaction. I see this aim as being scientific and artistic achievements, which place the game of chess on a par with other arts."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Whoever sees no other aim in the game than that of giving checkmate to one's opponent will never become a good chess player."#,
        r#"Max Euwe"#,
    ),
    (
        r#"... The main thing that develops positional judgment, that perfects it and makes it many-sided, is detailed analytical work, sensible tournament practice, a self-critical attitude to your games and a rooting out of all the defects in your play."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"From triumph to downfall there is but one step. I have noted that, in the most momentous occasions, mere nothings have always decided the outcome of the greatest events."#,
        r#"Napoleon"#,
    ),
    (
        r#"To lose one's objective attitude to a position, nearly always means ruining your game."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Chess teaches you to control the initial excitement you feel when you see something that looks good and it trains you to think objectively when in you're trouble."#,
        r#"Stanley Kubrick"#,
    ),
    (
        r#"When you see a good move, look for a better one."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"I often play a move I know how to refute."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"You need not play well - just help your opponent to play badly."#,
        r#"Genrikh Chepukaitis"#,
    ),
    (
        r#"When you have finished analyzing all the variations and gone along all the branches of the tree of analysis you must first of all write the move down on your score sheet, before you play it."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"You sit at the board and suddenly your heart leaps. Your hand trembles to pick up the piece and move it. But what Chess teaches you is that you must sit there calmly and think about whether it's really a good idea and whether there are other better ideas."#,
        r#"Stanley Kubrick"#,
    ),
    (
        r#"The hardest part of chess is winning a won game."#,
        r#"Frank Marshall"#,
    ),
    (
        r#"The sign of a great Master is his ability to win a won game quickly and painlessly."#,
        r#"Irving Chernev"#,
    ),
    (
        r#"Under no circumstances should you play fast if you have a winning position. Forget the clock, use all your time and make good moves."#,
        r#"Pal Benko"#,
    ),
    (
        r#"You're never beaten until you admit it."#,
        r#"General George S. Patton, Jr."#,
    ),
    (
        r#"However hopeless the situation appears to be there yet always exists the possibility of putting up a stubborn resistance."#,
        r#"Paul Keres"#,
    ),
    (
        r#"Don't be afraid of losing, be afraid of playing a game and not learning something."#,
        r#"Dan Heisman"#,
    ),
    (
        r#"Nothing is so healthy as a trashing at the proper time, and from few won games have I learned as much as I have from most of my defeats."#,
        r#"Capablanca"#,
    ),
    (
        r#"I prefer to lose a really good game than to win a bad one."#,
        r#"David Levy"#,
    ),
    (
        r#"You may learn much more from a game you lose than from a game you win. You will have to lose hundreds of games before becoming a good player."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Most players ... do not like losing, and consider defeat as something shameful. This is a wrong attitude. Those who wish to perfect themselves must regard their losses as lessons and learn from them what sorts of things to avoid in the future."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Setbacks and losses are both inevitable and essential if you're going to improve and become a good, even great, competitor. The art is in avoiding catastrophic losses in the key battles."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Losing can persuade you to change what doesn't need to be changed, and winning can convince you everything is fine even if you are on the brink of disaster."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Loss generally occurs when a player overrates his advantage or for other reasons seeks to derive from a minute advantage a great return such as a forced win."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"All action takes place, so to speak, in a kind of twilight, which like a fog or moonlight, often tends to make things seem grotesque and larger than they really are."#,
        r#"Clausewitz"#,
    ),
    (
        r#"Rule #29: "Always make your opponent think you know more than you really know.""#,
        r#"General Phil Sheridan"#,
    ),
    (
        r#"You must take your opponent into a deep dark forest where 2+2=5, and the path leading out is only wide enough for one."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"You need not play well - just help your opponent to play badly."#,
        r#"Genrikh Chepukaitis"#,
    ),
    (
        r#"You must not let your opponent know how you feel."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"When your opponent can easily anticipate every move you make, your strategy deteriorates and becomes commoditized."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"You can't overestimate the importance of psychology in chess, and as much as some players try to downplay it, I believe that winning requires a constant and strong psychology not just at the board but in every aspect of your life."#,
        r#"Garry Kasparov"#,
    ),
    (r#"I start out by believing the worst."#, r#"Napoleon"#),
    (
        r#"Few things are as psychologically brutal as chess."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Psychology is the most important factor in chess."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Emotional instability can be one of the factors giving rise to a failure by chess players in important duels. Under the influence of surging emotions (and not necessarily negative ones) we sometimes lose concentration and stop objectively evaluating the events that are taking place on the board."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"...as man under pressure tends to give in to physical and intellectual weakness, only great strength of will can lead to the objective."#,
        r#"Clausewitz"#,
    ),
    (r#"In war, truth is the first casualty."#, r#"Aeschylus"#),
    (
        r#"Drawing general conclusions about your main weaknesses can provide a great stimulus to further growth."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"In chess, as in life, a man is his own most dangerous opponent."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"My most difficult opponent is myself. When I am playing I often involuntarily make a world champion out of a candidate master."#,
        r#"Lev Polugaevsky"#,
    ),
    (
        r#"Mistrust is the most necessary characteristic of the chess player."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Independence of thought is a (most) valuable quality in a chess-player, both at the board and when preparing for a game."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Your practical results will improve when you play what you know, like and have confidence in."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Winning is not a secret that belongs to a very few, winning is something that we can learn by studying ourselves, studying the environment and making ourselves ready for any challenge that is in front of us."#,
        r#"Garry Kasparov"#,
    ),
    (r#"The fear of war is worse than war itself."#, r#"Seneca"#),
    (
        r#"There are two classes of men; those who are content to yield to circumstances and who play whist; those who aim to control circumstances, and who play chess."#,
        r#"Mortimer Collins"#,
    ),
    (
        r#"...man is a frivolous, a specious creature, and like a chess-player cares more for the process of attaining his goal than for the goal itself."#,
        r#"Dostoyevsky"#,
    ),
    (
        r#"In life, as in Chess, one's own Pawns block one's way. A man's very wealth, ease, leisure, children, books, which should help him to win, more often checkmate him."#,
        r#"Charles Buxton"#,
    ),
    (
        r#"The technical phase can be boring because there is little opportunity for creavivity, for art. Boredom leads to complacency and mistakes."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Botvinnik tried to take the mystery out of Chess, always relating it to situations in ordinary life. He used to call Chess a typical inexact problem similar to those which people are always having to solve in everyday life."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Perception is strong and sight weak. In strategy it is important to see distant things as if they were close and to take a distanced view of close things."#,
        r#"Musashi"#,
    ),
    (
        r#"All great events hang by a single thread. The clever man takes advantage of everything, neglects nothing that may give him some added opportunity; the less clever man, by neglecting one thing, sometimes misses everything."#,
        r#"Napoleon"#,
    ),
    (
        r#"To know ten thousand things, know one well."#,
        r#"Miyamoto Musashi"#,
    ),
    (
        r#"As has happened so often in history, victory had bred a complacency and fostered an orthodoxy which led to defeat in the next war."#,
        r#"Sir Basil H. Liddell-Hart (Strategy, 1954; on the French military development between the World Wars)"#,
    ),
    (
        r#"I've seen - both in myself and my competitors - how satisfaction can lead to a lack of vigilance, then to mistakes and missed opportunities."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"It's the unconquerable soul of man, and not the nature of the weapon he uses, that ensures victory."#,
        r#"Napoleon"#,
    ),
    (
        r#"All right, they're on our left, they're on our right, they're in front of us, they're behind us...they can't get away this time."#,
        r#"Lt Gen Lewis B. Puller, USMC"#,
    ),
    (
        r#"A defeatist spirit must inevitably lead to disaster."#,
        r#"Eugene Znosko-Borovski"#,
    ),
    (
        r#"No one ever won a game by resigning."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"If a mistake or an inaccuracy occurs, there is no need to assume 'all is lost' and mope - one must reorient oneself quickly, and find a new plan to fit the new situation."#,
        r#"David Bronstein"#,
    ),
    (
        r#"How come the little things bother you when you are in a bad position? They don't bother you in good positions."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"He who has a slight disadvantage plays more attentively, inventively and more boldly than his antagonist who either takes it easy or aspires after too much. Thus a slight disadvantage is very frequently seen to convert into a good, solid advantage."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Later, ... I began to succeed in decisive games. Perhaps because I realised a very simple truth: not only was I worried, but also my opponent."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#""Oh! this opponent, this collaborator against his will, whose notion of Beauty always differs from yours and whose means (strength, imagination, technique) are often too limited to help you effectively! What torment, to have your thinking and your fantasy tied down by another person!"#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Nowadays grandmasters no longer study their opponent's games so much, but they study his character, his behavior and his temperament in the most thorough fashion."#,
        r#"David Bronstein"#,
    ),
    (
        r#"I am trying to beat the guy sitting across from me and trying to choose the moves that are most unpleasant for him and his style."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"The effect to be sought is the dislocation of the opponent's mind and dispositions -- such an effect is the true gauge of an indirect approach."#,
        r#"Sir Basil H. Liddell-Hart (Strategy, 1954)"#,
    ),
    (
        r#"When strong, avoid them. If of high morale, depress them. Seem humble to fill them with conceit. If at ease, exhaust them. If united, separate them. Attack their weaknesses. Emerge to their surprise."#,
        r#"Sun Tzu"#,
    ),
    (r#"A player surprised is half beaten."#, r#"Chess Proverb"#),
    (
        r#"Ultimately, what separates a winner from a loser at the grandmaster level is the willingness to do the unthinkable. A brilliant strategy is, certainly, a matter of intelligence, but intelligence without audaciousness is not enough. Given the opportunity, I must have the guts to explode the game, to upend my opponent’s thinking and, in so doing, unnerve him. So it is in business: One does not succeed by sticking to convention."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Water shapes its course according to the nature of the ground over which it flows; the soldier works out his victory in relation to the foe whom he is facing."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"One must indeed be ignorant of the methods of genius to suppose that it allows itself to be cramped by forms. Forms are for mediocrity, and it is fortunate that mediocrity can act only according to routine. Ability takes its flight unhindered."#,
        r#"Napoleon"#,
    ),
    (
        r#"It is a mistake, too, to say that the face is the mirror of the soul. The truth is, men are very hard to know, and yet, not to be deceived, we must judge them by their present actions, but for the present only."#,
        r#"Napoleon"#,
    ),
    (
        r#"Some Warriors look fierce, but are mild. Some seem timid, but are vicious. Look beyond appearances; position yourself for the advantage."#,
        r#"Deng Ming-Dao"#,
    ),
    (
        r#"You can discover what your enemy fears most by observing the means he uses to frighten you."#,
        r#"an anonymous politician"#,
    ),
    (
        r#"Your body has to be in top condition. Your Chess deteriorates as your body does. You can't separate body from mind."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Above all else, before playing in competitions a player must have regard to his health, for if he is suffering from ill-health he cannot hope for success. In this connection the best of all tonics is 15 to 20 days in the fresh air, in the country."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Since your mental state can have such dramatic effects on your body, obviously your physical condition can affect your mental well-being. It follows that regular physical conditioning should be part of your overall chess training."#,
        r#"Pal Benko"#,
    ),
    (
        r#"I spend around one hour per day on physical exercise. Exercise is a must for every chess player. As the proverb says, 'A sound mind in a sound body.'"#,
        r#"Humpy Koneru"#,
    ),
    (
        r#"Method rules his training, which blends the physical with the mental. How many chess masters put in, prior to an important match, an allotted time daily to bicycling and shadow-boxing, followed by a cold douche and a brisk rub down?"#,
        r#"Hans Kmoch, on Max Euwe"#,
    ),
    (
        r#"The stomach is an essential part of the Chess master."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"The laws of chess do not permit a free choice: you have to move whether you like it or not."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"In short, the ideal way of playing a game would be rapid development of the pieces of strategic use for attack or defense, taking into account the fact that the two elements are Time and Position. Calm in defense and decisiveness in attack."#,
        r#"Capablanca"#,
    ),
    (
        r#"If you want to know how the Battle of the Bulge was won, ask my G4 (Logistics) Officer..."#,
        r#"Patton"#,
    ),
    (
        r#"He will win who knows how to handle both superior and inferior forces."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"However, if you study from the point of view of the general principles you are taking a more certain path for although a player’s intellect can fail at a given moment, principles well used never fail."#,
        r#"Capablanca"#,
    ),
    (
        r#"Strategy without tactics is the slowest route to victory. Tactics without strategy is the noise before defeat."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"The tactician must know what to do whenever something needs doing; the strategist must know what to do when nothing needs doing."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Strategy requires thought, tactics require observation."#,
        r#"Max Euwe"#,
    ),
    (
        r#"We often hear the terms 'positional' and 'tactical' used as opposites. But this is as wrong as to consider a painting's composition unrelated to its subject. Just as there is no such thing as 'artistic' art, so there is no such thing as 'positional' chess."#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"No matter how much theory progresses, how radically styles change, chess play is inconceivable without tactics."#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"In general I consider that in chess everything rests on tactics. If one thinks of strategy as a block of marble, then tactics are the chisel with which a master operates, in creating works of chess art."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"Tactics flow from a superior position."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Any material change in a position must come about by mate, a capture, or a Pawn promotion."#,
        r#"Purdy"#,
    ),
    (
        r#"Concentrate on material gains. Whatever your opponent gives you take, unless you see a good reason not to."#,
        r#"Bobby Fischer"#,
    ),
    (r#"Every move creates a weakness."#, r#"Siegbert Tarrasch"#),
    (
        r#"So in war, the way is to avoid what is strong, and strike at what is weak."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Fortune, which has a great deal of power in other matters but especially in war, can bring about great changes in a situation through very slight forces."#,
        r#"Julius Caesar"#,
    ),
    (
        r#"The criterion of real strength is a deep penetration into the secrets of a position."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"We are not fit to lead an army on the march unless we are familiar with the face of the country -- its mountains and forests, its pitfalls and precipices, its marshes and swamps."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"The laws of circumstance are abolished by new circumstances."#,
        r#"Napoleon"#,
    ),
    (
        r#"It is the aim of the modern school, not to treat every position according to one general law, but according to the principle inherent in the position."#,
        r#"Richard Reti"#,
    ),
    (
        r#"Bring all your pieces out! Give them scope! Occupy the central squares!"#,
        r#"Tarrasch"#,
    ),
    (
        r#"Chess is a terrible game. If you have no center, your opponent has a freer position. If you do have a center, then you really have something to worry about!"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Once we have chosen the right formation in the centre we have created opportunities for our pieces and laid the foundation of subsequent victory."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"If the defender is forced to give up the center, then every possible attack follows almost of itself."#,
        r#"Tarrasch"#,
    ),
    (
        r#"By reinforcing every part, (the opponent) weakens every part."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Weak points or holes in the opponent’s position must be occupied by pieces not Pawns."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Do you realize Fischer almost never has any bad pieces? He exchanges them, and the bad pieces remain with his opponents."#,
        r#"Yuri Balashov"#,
    ),
    (
        r#"The most important feature of the Chess position is the activity of the pieces. This is absolutely fundamental in all phases of the game: Opening, Middlegame and especially Endgame. The primary constraint on a piece’s activity is the Pawn structure."#,
        r#"Michael Stean"#,
    ),
    (
        r#"Pawns: they are the soul of this game, they alone form the attack and defense."#,
        r#"Philidor"#,
    ),
    (
        r#"Nothing so easily ruins a position as pawn moves."#,
        r#"Tarrasch"#,
    ),
    (
        r#"The older I grow, the more I value pawns."#,
        r#"Paul Keres"#,
    ),
    (
        r#"The task of the positional player is systematically to accumulate slight advantages and try to convert temporary advantages into permanent ones, otherwise the player with the better position runs the risk of losing it."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"A passed Pawn increases in strength as the number of pieces on the board diminishes"#,
        r#"Capablanca"#,
    ),
    (
        r#"The winning of a Pawn among good players of even strength often means the winning of the game."#,
        r#"Capablanca"#,
    ),
    (
        r#"Simplify, simplify, simplify! I say, let your affairs be as two or three, and not a hundred or a thousand; instead of a million count half a dozen, and keep your accounts on your thumb-nail."#,
        r#"Henry David Thoreau"#,
    ),
    (
        r#"Though most people love to look at the games of the great attacking masters, some of the most successful players in history have been the quiet positional players. They slowly grind you down by taking away your space, tying up your pieces, and leaving you with virtually nothing to do !"#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"The highest art of the chessplayer lies in not allowing your opponent to show you what he can do."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"[The] aim is not so much to seek battle as to seek a strategic situation so advantageous that if it does not of itself produce the decision, its continuation by a battle is sure to achieve this. In other words, dislocation is the aim of strategy."#,
        r#"Sir Basil H. Liddell-Hart (Strategy)"#,
    ),
    (
        r#"The battlefield is a scene of constant chaos. The winner will be the one who controls that chaos, both his own and the enemy's."#,
        r#"Napoleon Bonaparte"#,
    ),
    (
        r#"I love all positions. Give me a difficult positional game, I will play it. Give me a bad position, I will defend it. Openings, endgames, complicated positions, dull draws, I love them and I will do my very best. But totally won positions, I cannot stand them."#,
        r#"Hein Donner"#,
    ),
    (
        r#"It is rightly said that the most difficult thing in chess is winning a won position."#,
        r#"Vladimir Kramnik"#,
    ),
    (
        r#"Thus it is that in war the victorious strategist only seeks battle after the victory is won, whereas he who is destined to defeat first fights and afterwards looks for victory."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"It is not a move, even the best move that you must seek, but a realizable plan."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"It is better to follow out a plan consistently even if it isn't the best one than to play without a plan at all. The worst thing is to wander about aimlessly."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"A bad plan is better than none at all."#,
        r#"Frank Marshall"#,
    ),
    (
        r#"To find the right plan is just as hard as looking for its sound justification."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The enemy of a good plan is the dream of a perfect plan."#,
        r#"Clausewitz"#,
    ),
    (
        r#"A plan is made for a few moves only, not for the whole game."#,
        r#"Reuben Fine"#,
    ),
    (
        r#"Capture of the adverse King is the ultimate but not the first object of the game."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"Whoever sees no other aim in the game than that of giving checkmate to one's opponent will never become a good chess player."#,
        r#"Max Euwe"#,
    ),
    (
        r#"When strong, avoid them. If of high morale, depress them. Seem humble to fill them with conceit. If at ease, exhaust them. If united, separate them. Attack their weaknesses. Emerge to their surprise."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"All obvious moves look dubious in analysis after the game."#,
        r#"Korchnoi"#,
    ),
    (
        r#"Perception is strong and sight weak. In strategy it is important to see distant things as if they were close and to take a distanced view of close things."#,
        r#"Musashi"#,
    ),
    (
        r#"The study of typical plans is something that the leading grandmasters devote a great deal of time to. I would say that the most far-seeing of them devote as much time to this as to the study of openings."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Analysis is a glittering opportunity for training: it is just here that capacity for work, perseverance and stamina are cultivated, and these qualities are, in truth, as necessary to a chess player as a marathon runner."#,
        r#"Lev Polugaevsky"#,
    ),
    (
        r#"Knowing the enemy enables you to take the offensive, knowing yourself enables you to stand on the defensive."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Ponder and deliberate before you make a move."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"If we wish to wrest an advantage from the enemy, we must not fix our minds on that alone, but allow for the possibility of the enemy also doing some harm to us, and let this enter as a factor into our calculations."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"A Chess game is a dialogue, a conversation between a player and his opponent. Each move by the opponent may contain threats or be a blunder, but a player cannot defend against threats or take advantage of blunders if he does not first ask himself: What is my opponent planning after each move?"#,
        r#"Bruce A. Moon"#,
    ),
    (
        r#"What is the Threat??"#,
        r#"Anon (a question to always ask of both your own and the opponent's moves...)"#,
    ),
    (
        r#"A Threat is more powerful than its execution."#,
        r#"Tartakover"#,
    ),
    (
        r#"The spot where we intend to fight must not be made known; for then the enemy will have to prepare against a possible attack at several different points."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"One charming characteristic of many flank attacks I could mention is that they do not very often lead to simplification: if the attack is parried, there usually are still opportunities left for initiating action in another sector."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"To ensure attaining an objective, one should have alternate objectives. An attack that converges on one point should threaten, and be able to diverge against another. Only by this flexibility of aim can strategy be attuned to the uncertainty of war."#,
        r#"Sir Basil H. Liddell-Hart (Strategy, 1954)"#,
    ),
    (
        r#"So do many calculations lead to victory, and few calculations to defeat."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"I claim that nothing else is so effective in encouraging the growth of chess strength as such independent analysis, both of the games of the great players and your own."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"It is hardly useful if you trustingly play through variation after variation from a book. It is a great deal more useful and more interesting if you take part actively in the analysis, find something yourself, and try to refute some of the author's conclusions."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"White lost because he failed to remember the right continuation and had to think up the moves himself."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"The most difficult art is not in the choice of men, but in giving to the men chosen the highest service of which they are capable."#,
        r#"Napoleon"#,
    ),
    (
        r#"Dazzling combinations are for the many, shifting wood is for the few."#,
        r#"George Kieninger"#,
    ),
    (
        r#"Human affairs are like a chess game: only those who do not take it seriously can be called good players."#,
        r#"Hung Tzu Ch'eng"#,
    ),
    (
        r#"Everything in war is very simple, but the simplest thing is difficult."#,
        r#"Clausewitz"#,
    ),
    (r#"Speed is fine but accuracy is final."#, r#"Bill Jordan"#),
    (
        r#"Inclined to simplicity, I always play carefully and try to avoid unnecessary risks. I consider my method to be right as any superfluous “daring” runs counter to the essential character of chess, which is not a gamble but a purely intellectual combat conducted in accordance with the exact rules of logic."#,
        r#"Capablanca"#,
    ),
    (
        r#"A win by an unsound combination, however showy, fills me with artistic horror."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"Never do an enemy a small injury."#,
        r#"Niccolo Machiavelli"#,
    ),
    (
        r#"A combination is a blend of ideas"#,
        r#"pins, forks, discovered checks, double attacks – which endow the pieces with magical powers. – I. Chernev"#,
    ),
    (
        r#"It is a profound mistake to imagine that the art of combination depends only on natural talent, and that it cannot be learned. Every player knows that all (or almost all) combinations arise from a recollection of familiar elements."#,
        r#"Richard Reti"#,
    ),
    (
        r#"According to such great attacking players as Bronstein and Tal, most combinations are inspired by the player's memories of earlier games."#,
        r#"Pal Benko"#,
    ),
    (
        r#"In almost any position the boundless possibilities of chess enable a new or at least a little-studied continuation to be found."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"Combinations have always been the most intriguing aspect of Chess. The masters look for them, the public applauds them, the critics praise them. It is because combinations are possible that chess is more than a lifeless mathematical exercise. They are the poetry of the game; they are to chess what melody is to music. They represent the triumph of mind over matter."#,
        r#"Rueben Fine"#,
    ),
    (
        r#"The pleasure to be derived from a chess combination lie in the feeling that a human mind is behind the game, dominating the inanimate pieces ... and giving them breath of life."#,
        r#"Richard Reti"#,
    ),
    (
        r#"By positional play a master tries to prove and exploit true values, whereas by combinations he seeks to refute false values ... A combination produces an unexpected re-assessment of values."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"In battle, there are not more than two methods of attack—the direct and the indirect; yet these two in combination give rise to an endless series of maneuvers."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"This high proportion of history's decisive campaigns, the significance of which is enhanced by the comparative rarity of the direct approach, enforces the conclusion that the indirect is by far the most hopeful and economic form of strategy."#,
        r#"Sir Basil H. Liddell-Hart (Strategy, 1954)"#,
    ),
    (
        r#"It has been stated that a characteristic mark of a combination is surprise; surprise for the defender, not for the assailant, since otherwise the combination will probably be unsound."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"The combination player thinks forward; he starts from the given position, and tries the forceful moves in his mind."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Half the variations which are calculated in a tournament game turn out to be completely superfluous. Unfortunately, no one knows in advance which half."#,
        r#"Jan Timman"#,
    ),
    (
        r#"A thorough understanding of the typical mating continuations makes the most complicated sacrificial combinations leading up to them not only not difficult, but almost a matter of course."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"The most difficult art is not in the choice of men, but in giving to the men chosen the highest service of which they are capable."#,
        r#"Napoleon"#,
    ),
    (
        r#"Impossible is the word found only in a fool's dictionary. Wise people create opportunities for themselves and make everything possible..."#,
        r#"Napoleon"#,
    ),
    (
        r#"So in war, the way is to avoid what is strong, and strike at what is weak."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"If in a battle, I seize a bit of debatable land with a handful of soldiers, without having done anything to prevent an enemy bombardment of the position, would it ever occur to me to speak of a conquest of the terrain in question? Obviously not. Then why should I do so in chess?"#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"Strategically important points should be overprotected. If the pieces are so engaged, they get their regard in the fact that they will then find themselves well posted in every respect."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"Pursue one great decisive aim with force and determination."#,
        r#"Clausewitz"#,
    ),
    (
        r#"The spot where we intend to fight must not be made known; for then the enemy will have to prepare against a possible attack at several different points."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"If he sends reinforcements everywhere, he will everywhere be weak."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Concentration is the secret of strengths in politics, in war, in trade, in short in all management of human affairs."#,
        r#"Ralph Waldo Emerson"#,
    ),
    (
        r#"… in chess"#,
        r#"as in any conflict – success lies in the attack. – Max Euwe"#,
    ),
    (
        r#"The process of making pieces in Chess do something useful (whatever it may be) has received a special name: it is called the attack. The attack is that process by means of which you remove obstructions."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Opportunities multiply as they are seized."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"The most powerful weapon in Chess is to have the next move."#,
        r#"David Bronstein"#,
    ),
    (
        r#"...only the player with the initiative has the right to attack."#,
        r#"William Steinitz"#,
    ),
    (
        r#"Examine moves that smite! A good eye for smites is far more important than a knowledge of strategical principles."#,
        r#"Purdy"#,
    ),
    (
        r#"You have to have the fighting spirit. You have to force moves and take chances."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Thus the expert in battle moves the enemy, and is not moved by him."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"The first principle of attack - Don't let the enemy develop!"#,
        r#"Rueben Fine"#,
    ),
    (r#"Logistics is the Soul of War."#, r#"Napoleon"#),
    (
        r#"In maneuver warfare, we attempt not to destroy the entire enemy force but to render most of it irrelevant."#,
        r#"Lt. Col. Robert R. Leonhard, U.S.A."#,
    ),
    (
        r#"If your opponent cannot do anything active, then don't rush the position; instead you should let him sit there, suffer, and beg you for a draw."#,
        r#"Jeremy Silman"#,
    ),
    (
        r#"When you have an enemy in your power, deprive him of the means of ever injuring you."#,
        r#"Napoleon"#,
    ),
    (
        r#"The important thing in strategy is to suppress the enemy's useful actions but allow his useless actions."#,
        r#"Musashi"#,
    ),
    (
        r#"The spot where we intend to fight must not be made known; for then the enemy will have to prepare against a possible attack at several different points."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"The highest generalship is to compel the enemy to disperse his army, and then to concentrate superior force against each fraction in turn."#,
        r#"Col. Henderson"#,
    ),
    (
        r#"Surprise becomes effective when we suddenly face the enemy at one point with far more troops than he expected. This type of numerical superiority is quite distinct from numerical superiority in general: it is the most powerful medium in the art of war."#,
        r#"Clausewitz"#,
    ),
    (
        r#"So in war, the way is to avoid what is strong, and strike at what is weak."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Without error there can be no brilliancy."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"[Chess] is the finest mental exercise. It develops concentration and logical reasoning; and it is one of the few games in which you cannot rectify a mistake. If you make a mistake, you lose, unless your opponent makes a worse mistake."#,
        r#"Capablanca"#,
    ),
    (
        r#"A game is always won through a mistake."#,
        r#"Tartakower"#,
    ),
    (
        r#"The blunders are all there on the board, waiting to be made."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Gentlemen, when the enemy is committed to a mistake we must not interrupt him too soon."#,
        r#"Horatio Nelson"#,
    ),
    (
        r#"Hence that general is skilful in attack whose opponent does not know what to defend; and he is skillful in defense whose opponent does not know what to attack."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Select the tactic of seeming to come from the East and attacking from the West; avoid the solid, attack the hollow; attack; withdraw; deliver a lightning blow, seek a lightning decision. When guerrillas engage a stronger enemy, they withdraw when he advances; harass him when he stops; strike him when he is weary; pursue him when he withdraws."#,
        r#"Mao Tse-Tung (On Guerrilla Warfare, 1937)"#,
    ),
    (
        r#"A quiet move in the midst of an attack is the master's trademark."#,
        r#"Anon"#,
    ),
    (
        r#"Not all artists may be chess players, but all chess players are artists."#,
        r#"Marcel Duchamp"#,
    ),
    (
        r#"… a 'quiet' move is the epitome of finesse. A soft answer turns away wrath, but its subdued quality makes it no less efficient."#,
        r#"Hans Kmoch"#,
    ),
    (
        r#"What would Chess be without silly mistakes?"#,
        r#"Kurt Richter"#,
    ),
    (
        r#"People who want to improve should take their defeats as lessons, and endeavor to learn what to avoid in the future. You must also have the courage of your convictions. If you think your move is good, make it."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Confidence is very important"#,
        r#"even pretending to be confident. If you make a mistake but do not let your opponent see what you are thinking then he may overlook the mistake. – Viswanathan Anand"#,
    ),
    (
        r#"To avoid mistakes is the beginning, as it is the end, of mastery in chess."#,
        r#"Znosko-Borovsky"#,
    ),
    (
        r#"Chess is infinite, and one has to make only one ill-considered move, and one`s opponent`s wildest dreams will become reality."#,
        r#"David Bronstein"#,
    ),
    (
        r#"One bad move nullifies forty good ones."#,
        r#"I.A. Horowitz"#,
    ),
    (
        r#"A descriptive justification can be given for almost every mistake."#,
        r#"adapted from Nigel Davies"#,
    ),
    (
        r#"Errors have nothing to do with luck; they are caused by time pressure, discomfort or unfamiliarity with a position, distractions, feelings of intimidation, nervous tension, overambition, excessive caution, and dozens of other psychological factors."#,
        r#"Pal Benko"#,
    ),
    (
        r#"In the endgame, the most common errors, besides those resulting from ignorance of theory, are caused by either impatience, complacency, exhaustion, or all of the above."#,
        r#"Pal Benko"#,
    ),
    (
        r#"Some things are really hard to do, almost impossible to do, like playing perfectly in extremely complicated positions. But it really bugs me when I miss things that I really shouldn't have. I am always going to make mistakes. I don't have any illusions that my understanding of chess is perfect or anything like that. It's just that I have to work on relatively simple mistakes. When I can lower the percentage of such mistakes then things are going to be much better."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"The first order of business for a General is to secure himself against defeat."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Winning isn't everything... but losing is nothing."#,
        r#"Mednis, on the importance of fighting for a draw To secure ourselves against defeat lies in our own hands, but the opportunity of defeating the enemy is provided by the enemy himself. – Sun Tzu"#,
    ),
    (
        r#"When you don’t know what to play, wait for an idea to come into your opponent’s mind. You may be sure that idea will be wrong."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"When you defend, try not to worry or become upset. Keep your cool and trust your position - it's all you've got."#,
        r#"Pal Benko"#,
    ),
    (
        r#"Setbacks and losses are both inevitable and essential if you're going to improve and become a good, even great, competitor. The art is in avoiding catastrophic losses in the key battles."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"A defensive war is apt to betray us into too frequent detachment. Those generals who have had but little experience attempt to protect every point, while those who are better acquainted with their profession, having only the capital object in view, guard against a decisive blow, and acquiesce in small misfortunes to avoid greater."#,
        r#"Frederick the Great"#,
    ),
    (
        r#"To avoid losing a piece, many a person has lost the game."#,
        r#"Savielly Tartakover"#,
    ),
    (
        r#"It is dangerous to maintain equality at the cost of placing the pieces passively."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"Every action is seen to fall into one of three main categories, guarding, hitting, or moving. Here, then, are the elements of combat, whether in war of pugilism."#,
        r#"B. H. Liddell-Hart"#,
    ),
    (r#"Do nothing which is of no use."#, r#"Musashi"#),
    (
        r#"...only the player with the initiative has the right to attack."#,
        r#"William Steinitz"#,
    ),
    (
        r#"It's less about physical training, in the end, than it is about the mental preparation: boxing is a chess game. You have to be skilled enough and have trained hard enough to know how many different ways you can counterattack in any situation, at any moment."#,
        r#"Jimmy Smits"#,
    ),
    (
        r#"The best form of defense is attack."#,
        r#"Karl von Clausewitz"#,
    ),
    (
        r#"… the old aphorism holds good, that after the attack has been repulsed, the counterattack is generally decisive."#,
        r#"Reti"#,
    ),
    (
        r#"In modern praxis lost positions are salvaged most often when the play is highly complicated with many sharp dynamic variations to be calculated."#,
        r#"Leonid Shamkovich"#,
    ),
    (
        r#"Playing for complications is an extreme measure that a player should adopt only when he cannot find a clear and logical plan."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"A good sacrifice is one that is not necessarily sound but leaves your opponent dazed and confused."#,
        r#"Rudolph Spielmann"#,
    ),
    (
        r#"Although our intellect always longs for clarity and certainty, our nature often finds uncertainty fascinating."#,
        r#"Clausewitz"#,
    ),
    (
        r#"From the sublime to the ridiculous there is but one step."#,
        r#"Napoleon"#,
    ),
    (
        r#"There are cases in which the greatest daring is the greatest wisdom."#,
        r#"Clausewitz, (On War)"#,
    ),
    (r#"Haste, the great enemy."#, r#"Eugene Znosko-Borowski"#),
    (r#"Time trouble is blunder time."#, r#"Alexander Kotov"#),
    (
        r#"Whatever the Way, the master of strategy does not appear fast….Of course, slowness is bad. Really skillful people never get out of time, and are always deliberate, and never appear busy."#,
        r#"Musashi"#,
    ),
    (
        r#"By playing slowly during the early phases of a game I am able to grasp the basic requirements of each position. Then, despite being in time pressure, I have no difficulty in finding the best continuation. Incidentally, it is an odd fact that more often than not it is my opponent who gets the jitters when I am compelled to make these hurried moves."#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"The worst enemy of the strategist is the clock. Time trouble ... Reduces us all to pure reflex and reaction, tactical play. Emotion and instinct cloud our strategic vision when there is no time for proper evaluation."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"The fact that a player is very short of time is to my mind, as little to be considered as an excuse as, for instance, the statement of the law-breaker that he was drunk at the time he committed the crime."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"The most important lesson I learned ... was that the winner of a gunplay usually was the one who took his time."#,
        r#"Wyatt Earp"#,
    ),
    (
        r#"Go through detailed variations in your own time, think in a general way about the position in the opponent's time and you will soon find that you get into time trouble less often, that your games have more content to them, and that their general standard rises."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"When my opponent's clock is going I discuss general considerations in an internal dialogue with myself. When my own clock is going I analyse concrete variations."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"You will already have noticed how often Capablanca repeated moves, often returning to positions which he had had before. This is not lack of deciciveness or slowness, but the employment of a basic endgame principle which is 'Do not hurry.'"#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Haste is never more dangerous than when you feel that victory is in your grasp."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"The game might be divided into three parts, the opening, the middle-game and the end-game. There is one thing you must strive for, to be equally efficient in the three parts."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Chess players should acquire knowledge of the three phases [opening, middlegame, endgame] of the game equably, and not pay excessive study to any one. In the opening, development must be sought, and the pieces placed in a natural position where they will maintain the maximum of usefulness. In the middle game, the pieces should not be transferred to places from which they cannot easily return to another part of the field. In the end game, time-saving is the essence of the play."#,
        r#"Capablanca"#,
    ),
    (
        r#"Do not permit yourself to fall in love with the end-game play to the exclusion of entire games. It is well to have the whole story of how it happened; the complete play, not the denouement only. Do not embrace the rag-time and vaudeville of chess."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The art of treating the opening stage of the game correctly and without error is basically the art of using time efficiently."#,
        r#"Svetozar Gligoric"#,
    ),
    (
        r#"Where openings are concerned, chess masters are like a flock of sheep; everyone follows the first master’s example. Of course it is true that, as in everything, there are exceptions. It must always be remembered that White can hope only to obtain a positional advantage and not a game that is relatively easy to win."#,
        r#"Capablanca"#,
    ),
    (
        r#"In the beginning of the game ignore the search for combinations, abstain from violent moves, aim for small advantages, accumulate them, and only after these ends search for the combination"#,
        r#"and then with all the power of will and intellect, because then the combination must exist, however deeply hidden. – Emanuel Lasker"#,
    ),
    (
        r#"In order to make progress in chess, it is necessary to pay special attention to all the general principles, spending a little less time on the openings. Play the openings on the basis of your general knowledge of how to mobilize pieces and do not become involved in technicalities about whether the books recommend this or that move; to learn the openings by heart it is necessary to study a great number of books which, moreover, are sometimes wrong."#,
        r#"Capablanca"#,
    ),
    (
        r#"If the point of playing chess is as a battle of the intellect then most people would say that the memorization of other peoples ideas is something that is anathema to the spirit of chess."#,
        r#"Nigel Davies"#,
    ),
    (
        r#"Ordinarily men exercise their memory much more than their judgment."#,
        r#"Napoleon"#,
    ),
    (
        r#"One of the objectives of opening play is to try to surprise your opponent."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"If a chess statistician were to try and satisfy his curiosity over which stage of the game proved decisive in the majority of cases, he would certainly come to the conclusion that it is the middlegame that provides the most decisive stage."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"My forte was the middlegame. I had a good feeling for the critical moments of the play. This undoubtedly compensated for my lack of opening preparation and, possibly, not altogether perfect play in the endgame. In my games things often did not reach the endgame!"#,
        r#"Boris Spassky"#,
    ),
    (
        r#"The middle game, where the struggle is really fought, will take a variable number of moves, and will be named so until the certainty of mate for one of the two players is ninety percent."#,
        r#"Madame Flash, Je gagne aux éches, Marabout-Flash 1963"#,
    ),
    (
        r#"The middlegame I repeat is chess itself, chess with all its possibilities, its attacks, defences, sacrifices, etc."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"If a chess statistician were to try and satisfy his curiosity over which stage of the game proved decisive in the majority of cases, he would certainly come to the conclusion that it is the middlegame that provides the most decisive stage."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"I detest the endgame. A well-played game should be practically decided in the middlegame."#,
        r#"David Janowski"#,
    ),
    (
        r#"A thorough understanding of the typical mating continuations makes the most complicated sacrificial combinations leading up to them not only not difficult, but almost a matter of course."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Books on the openings abound; nor are works on the end game wanting; but those on the middle game can be counted on the fingers of one hand."#,
        r#"Harry Golombek"#,
    ),
    (
        r#"It often happens that a player is so fond of his advantageous position that he is reluctant to transpose to a winning endgame."#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"Even in the heat of a middlegame battle the master still has to bear in mind the outlines of a possible future ending."#,
        r#"David Bronstein"#,
    ),
    (
        r#"After a bad opening, there is hope for the middle game. After a bad middle game, there is hope for the endgame. But once you are in the endgame, the moment of truth has arrived."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"A player can sometimes afford the luxury of an inaccurate move, or even a definite error, in the opening or middlegame without necessarily obtaining a lost position. In the endgame … an error can be decisive, and we are rarely presented with a second chance."#,
        r#"Paul Keres"#,
    ),
    (
        r#"In the endgame, the most common errors, besides those resulting from ignorance of theory, are caused by either impatience, complacency, exhaustion, or all of the above."#,
        r#"Pal Benko"#,
    ),
    (
        r#"Lack of proper endgame technique allows many players to escape from lost positions, even without any spectacular play on their (opponent's) part."#,
        r#"Leonid Shamkovich"#,
    ),
    (
        r#"The endgame is an arena in which miraculous escapes are not uncommon."#,
        r#"Leonid Shamkovich"#,
    ),
    (
        r#"Ninety percent of the book variations have no great value, because either they contain mistakes or they are based on fallacious assumptions; just forget about the openings and spend all that time on the endings."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Capablanca did not apply himself to opening theory (in which he never therefore achieved much), but delved deeply into the study of end-games and other simple positions which respond to technique rather than to imagination."#,
        r#"Max Euwe"#,
    ),
    (
        r#"In my own experience, the main benefits are often realized when the endgames you have studied never make it onto the board. Endgames often arise in variations, and it's important to develop a good 'feel' for which ones are likely to pose practical problems for the opponent. Likewise, the confidence to simplify into an inferior but tenable endgame safe in the knowledge that you know how to handle it is invaluable."#,
        r#"Luke McShane"#,
    ),
    (
        r#"The king, which during the opening and middlegame stage is often a burden because it has to be defended, becomes in the endgame a very important and aggressive piece, and the beginner should realize this, and utilize his king as much as possible."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Endings of one rook and pawns are about the most common sort of endings arising on the chess board. Yet though they do occur so often, few have mastered them thoroughly. They are often of a very difficult nature, and sometimes while apparently very simple they are in reality extremely intricate."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"Even the best grandmasters in the world have had to work hard to acquire the technique of rook endings."#,
        r#"Paul Keres"#,
    ),
    (
        r#"Lack of patience is probably the most common reason for losing a game, or drawing games that should have been won."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"Any experienced player knows how a change in the character of the play influences your psychological mood."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Conform to the enemy's tactics until a favorable opportunity offers; then come forth and engage in a battle that shall prove decisive."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Energy may be likened to the bending of a crossbow; decision, to the releasing of a trigger."#,
        r#"Sun Tzu"#,
    ),
    (r#"Never make a good move too soon."#, r#"James Mason"#),
    (
        r#"There is only one decisive victory: the last."#,
        r#"Karl von Clausewitz"#,
    ),
    (
        r#"The ability to create and to control the tension of battle is perhaps the principal attainment of the great player."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Move not unless you see an advantage; use not your troops unless there is something to be gained; fight not unless the position is critical."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"The more usual reason for adopting a strategy of limited aim is that of awaiting a change in the balance of force ... The essential condition of such a strategy is that the drain on him should be disproportionately greater than on oneself."#,
        r#"Sir Basil H. Liddell-Hart (Strategy, 1954)"#,
    ),
    (
        r#"Sometimes the hardest thing to do in a pressure situation is to allow the tension to persist. The temptation is to make a decision, any decision, even if it is an inferior choice."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"The psychological effects of having to hold a prospectless position for what might seem an infinite amount of time does nothing to aid the defender's concentration."#,
        r#"Michael Stean"#,
    ),
    (
        r#"There is nothing wrong with trying to exploit the natural human tendency to become impatient when forced to play a boring position."#,
        r#"Pal Benko"#,
    ),
    (
        r#"Emotional instability can be one of the factors giving rise to a failure by chess players in important duels. Under the influence of surging emotions (and not necessarily negative ones) we sometimes lose concentration and stop objectively evaluating the events that are taking place on the board."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"I had a slightly inferior endgame that probably should have been drawn, but Kortchnoi kept torturing me with little threats until finally, exhausted and exasperated, I made a losing mistake."#,
        r#"Pal Benko"#,
    ),
    (
        r#"The best indicator of a chess player's form is his ability to sense the climax of the game."#,
        r#"Boris Spassky"#,
    ),
    (
        r#"In the perfect Chess combination as in a first-rate short story, the whole plot and counter-plot should lead up to a striking finale, the interest not being allayed until the very last moment."#,
        r#"Yates and Winter"#,
    ),
    (
        r#"In chess, as in life, opportunity strikes but once."#,
        r#"David Bronstein"#,
    ),
    (
        r#"The true science of martial arts means practicing them in such a way that they will be useful at any time, and to teach them in such a way that they will be useful in all things."#,
        r#"Miyamoto Musashi"#,
    ),
    (
        r#"The chesse, of all games wherein is no bodily exercise, is mooste to be commended; for therein is right subtile engine, whereby the wytte is made more sharpe and remembrance quickened. And it is the more commendable and also more commodiouse if the players haue radde the moralization of the chesse, and whan they playe do think upon hit; whiche bokes be in englisshe. But they be very scarse, by cause fewe men do seeke in plaies for vertue or wisdome."#,
        r#"Sir Thomas Eliot (1531)"#,
    ),
    (
        r#"We are in truth but pieces on this chess board of life, which in the end we leave, only to drop one by one into the grave of nothingness. (c 1120)"#,
        r#"Omar Khayyam"#,
    ),
    (
        r#"When the Chess game is over, the Pawn and the King go back to the same box."#,
        r#"Irish proverb"#,
    ),
    (
        r#"No chess grandmaster is normal; they only differ in the extent of their madness."#,
        r#"Viktor Korchnoi."#,
    ),
    (
        r#"A Chess game is divided into three stages: the first, when you hope you have the advantage, the second when you believe that you have an advantage, and the third … when you know you're going to lose!"#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Chess isn’t a game of speed, it is a game of speech through actions."#,
        r#"Matthew Selman (on the similarities of chess to a negotiation)"#,
    ),
    (
        r#"Let the perfectionist play postal."#,
        r#"GM Yasser Seirawan"#,
    ),
    (
        r#"You can ensure the safety of your defense if you only hold positions that cannot be attacked."#,
        r#"Sun Tzu"#,
    ),
    (
        r#"Modern chess is too much concerned with things like pawn structure. Forget it - checkmate ends the game."#,
        r#"Nigel Short"#,
    ),
    (
        r#"When you don’t know what to play, wait for an idea to come into your opponent’s mind. You may be sure that idea will be wrong."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"The boy doesn't have a clue about chess, and there's no future at all for him in this profession."#,
        r#"Botvinnik, said about a young 12 year old boy named Anatoly Karpov"#,
    ),
    (
        r#"Daring ideas are like chessmen moved forward. They may be beaten, but they may start a winning game."#,
        r#"Johann Wolfgang von Goethe"#,
    ),
    (
        r#"It was like when you make a move in chess and just as you take your finger off the piece, you see the mistake you've made, and there's this panic because you don't know yet the scale of disaster you've left yourself open to."#,
        r#"Kazuo Ishiguro, Never Let Me Go"#,
    ),
    (
        r#"All this twaddle, the existence of God, atheism, determinism, liberation, societies, death, etc., are pieces of a chess game called language, and they are amusing only if one does not preoccupy oneself with 'winning or losing this game of chess."#,
        r#"Marcel Duchamp"#,
    ),
    (
        r#"How dreadful...to be caught up in a game and have no idea of the rules."#,
        r#"Caroline Stevermer, Sorcery & Cecelia: or The Enchanted Chocolate Pot"#,
    ),
    (
        r#"A queenoffers her hand to be kissed,& can form it into a fistwhile smiling the whole damn time."#,
        r#"Elizabeth Acevedo, Clap When You Land"#,
    ),
    (
        r#"In chess, as a purely intellectual game, where randomness is excluded, - for someone to play against himself is absurd ...It is as paradoxical, as attempting to jump over his own shadow."#,
        r#"Stefan Zweig, Chess Story"#,
    ),
    (
        r#"It's an entire world of just 64 squares. I feel safe in it. I can control it; I can dominate it. And it's predictable. So, if I get hurt, I only have myself to blame."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"[Chess] is a foolish expedient for making idle people believe they are doing something very clever, when they are only wasting their time. "#,
        r#"George Bernard Shaw, The Irrational Knot"#,
    ),
    (
        r#"I'm a chess piece. A pawn,' she said. 'I can be sacrificed, but I cannot be captured. To be captured would be the end of the game."#,
        r#"Paolo Bacigalupi, Ship Breaker"#,
    ),
    (
        r#"When the Pawn Hits the Conflicts He Thinks Like a King What He Knows Throws the Blows When He Goes to the Fight and He'll Win the Whole Thing 'Fore He Enters the Ring There's No Body to Batter When Your Mind Is Your Might So When You Go Solo, You Hold Your Own Hand and Remember That Depth Is the Greatest of Heights and If You Know Where You Stand, Then You Know Where to Land and If You Fall It Won't Matter, Cuz You'll Know That You're Right"#,
        r#"Fiona Apple"#,
    ),
    (
        r#"People, she was discovering, were like cockroaches: If you allowed one in, more were sure to follow."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"When you see a good move, look for a better one"#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"But she never thought about the way Terrible looked, at least not that way. He hadn’t been ugly to her for months; he’d gone from just being a face she was familiar with to being a face she loved to look at, a face that made her….happy. Who gave a shit what anyone else saw when they looked at him, when they saw the crooked, many times broken nose, or the scars, or the jutting brow or thick jaw and heavy muttonchops? She knew what she saw, and that was all that mattered. Knew what was behind those hard dark eyes, and wanted it more than anything."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"IF her life had taught her anything, it was that you never really knew what people had going on beneath the surface. People were shit. The only difference between them and animals was people felt the need to hide it."#,
        r#"Stacia Kane, Unholy Ghosts"#,
    ),
    (
        r#"How the hell did people do this, this emotion-and-forgiveness thing? How did they stand these feelings? She could barely handle it and she had lovely, necessary, reason-for-living drugs to smooth over the rough spots. How did people do this shit sober?"#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"An apocryphal story - the word "apocryphal" here means "obviously untrue" - tells of two people, long ago, who were very bored, and that instead of complaining about it they sat up all night and invented the game of chess so that everyone else in the world, on evenings when there is nothing to do, can also be bored by the perplexing and tedious game they invented."#,
        r#"Lemony Snicket, Horseradish"#,
    ),
    (
        r#"Living together/ is one move closer/ to living apart"#,
        r#"Kristi Maxwell"#,
    ),
    (
        r#"I like the moment when I break a man's ego"#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Life is more than just chess.Though king dies, life goes on."#,
        r#"Toba Beta, Master of Stupidity"#,
    ),
    (
        r#"Terrible thought she was brave. She remembered it now, heard his voice in her head as if he stood next to her. "They scared. Not you, though." Terrible thought she was brave, and if he - a man whose name was Terrible, a man whose path people scrambled to get out of - thought so, it must be true. She could do this, she would do this."#,
        r#"Stacia Kane, Unholy Ghosts"#,
    ),
    (
        r#"But is it not already an insult to call chess anything so narrow as a game? Is it not also a science, an art, hovering between these categories like Muhammad's coffin between heaven and earth, a unique yoking of opposites, ancient and yet eternally new, mechanically constituted and yet an activity of the imagination alone, limited to a fixed geometric area but unlimited in its permutations, constantly evolving and yet sterile, a cogitation producing nothing, a mathematics calculating nothing, an art without an artwork, an architecture without substance and yet demonstrably more durable in its essence and actual form than all books and works, the only game that belongs to all peoples and all eras, while no one knows what god put it on earth to deaden boredom, sharpen the mind, and fortify the spirit?"#,
        r#"Stefan Zweig, Chess Story"#,
    ),
    (
        r#"You ain’t know nothing, a man scoffed. How I’m supposed to trust some junkie Churchwitch-The words sliced through her like razor-sharp fangs. Her face flooded with shame, so hot she imagined it steamed in the icy air. At least it wasn’t difficult to identify the speaker. All she had to do was look for the man with Terrible’s fist locked around his neck.Ain’t think I hear you right, Terrible said in a calm, quiet voice. Wanna louden up? The man shook his head His eyes bulged. He looked like a bug, with his hands clenching into tiny useless fists. You sure? You got else to say, you best say it now, instead of later. Now we got us watchers. Later might not be true, dig? The man dug."#,
        r#"Stacia Kane, Unholy Magic"#,
    ),
    (
        r#"Chess isn't always competitive. Chess can also be beautiful."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"Chess lied to herself every day; it was just something she did, like taking her pills or making sure she had a pen in her bag. Little lies, mostly. Insignificant. Of course there were big ones there, too, like telling herself that she was more than just a junkie who got lucky enough to possess a talent not everyone had. That she was alone by choice and that she was not terrified of other people because they couldn’t be trusted, because they carried filth in their minds and pain in their hands and they would smear both all over her given half the chance."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"and a most curious country it was. There were a number of tiny little brooks running straight across it from side to side, and the ground between was divided up into squares by a number of little green hedges, that reached from brook to brook.I declare it's marked out just like a large chessboard!' Alice said at last. 'There ought to be some men moving about somewhere--and so there are!' she added in a tone of delight, and her heart began to beat quick with excitement as she went on. 'It's a great huge game of chess that's being played--all over the world--if this is the world at all, you know. Oh, what fun it is!"#,
        r#"Lewis Carroll, Alice's Adventures in Wonderland / Through the Looking-Glass"#,
    ),
    (
        r#"Tis action moves the world....[in] the game of chess, mind that: ye cannot leave your men to stand unmoving on the board and hope to win. A soldier must first step upon the battlefield if does mean to cross it."#,
        r#"Susanna Kearsley, The Winter Sea"#,
    ),
    (
        r#"If God wanted humans to play chess, he would have made us black or white."#,
        r#"Zenna Henderson"#,
    ),
    (
        r#"So aint you think just causen you in this car now means any damn thing. It aint. He pretending it do, he lying and saying it do, but it aint. Pretend that other dame just he friend, so he say, but aint like it true.Some churchbitch she is too. Leastaways that what Amy telling me. Amy say she met her once and she aint shit."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"Most Debunkers spent their money on actual things, rather than just buying anything they could swallow, smoke or snort. Unlike Chess."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"When had being an addict gotten so fucking hard? So exhausting? It had been so easy for so long; she had a steady supply, she kept to herself, nobody bothered her. Now she was constantly up to her ears in intrigue and complications, being torn in every direction but her own, all thanks to her need for those pills"#,
        r#"Stacia Kane, Unholy Magic"#,
    ),
    (
        r#"He examined the chess problem and set out the pieces. It was a tricky ending, involving a couple of knights.'White to play and mate in two moves.'Winston looked up at the portrait of Big Brother. White always mates, he thought with a sort of cloudy mysticism. Always, without exception, it is so arranged. In no chess problem since the beginning of the world has black ever won. Did it not symbolize the eternal, unvarying triumph of Good over Evil? The huge face gazed back at him, full of calm power. White always mates."#,
        r#"George Orwell, 1984"#,
    ),
    (
        r#"Chess is all about getting the king into check, you see. It's about killing the father. I would say that chess has more to do with the art of murder than it does with the art of war."#,
        r#"Arturo Pérez-Reverte, The Flanders Panel"#,
    ),
    (
        r#"A great chessplayer is not a great man, for he leaves the world as he found it."#,
        r#"William Hazlitt, Table-Talk, Essays on Men and Manners"#,
    ),
    (
        r#"Now mayhap you quit givin Terrible the fuckin slurpy-eyes an give Bump the listening, yay? Thinkin you can? Gots some fuckin chattering wants doin, needs you fuckin head on straight up."#,
        r#"Stacia Kane, Sacrificial Magic"#,
    ),
    (
        r#"Pawns are such fascinating pieces, too...So small, almost insignificant, and yet--they can depose kings. Don't you find that interesting?"#,
        r#"Lavie Tidhar, The Bookman"#,
    ),
    (
        r#"It was her problem, and she'd deal with it. Because dealing with personal problems was so fucking high on her list of skills."#,
        r#"Stacia Kane, Sacrificial Magic"#,
    ),
    (
        r#"I’m not into danger, either. Aw, Chess. You so into it you ain’t climb out with a rope. Why else you do your job, live down here, buy from Bump? It’s just—I mean—I just do, is all. Her cheeks burned. She shouldn’t have let him come in here. She should have just sent him home and let him wash his stupid shirt himself. No shame in it. Some of us needs an edge on things make us feel right, else we ain’t like feeling at all, aye?"#,
        r#"Stacia Kane, Unholy Ghosts"#,
    ),
    (
        r#"We are men who find chess fascinating. Did you expect our lives to be secretly interesting?"#,
        r#"Noah Boyd, Agent X"#,
    ),
    (
        r#"Like most conversations and most chess games, we all start off the same and we all end the same, with a brief moment of difference in between. Fertilization to fertilizer. Ashes to ashes. And we spark across the gap."#,
        r#"Brian Christian, The Most Human Human: What Talking with Computers Teaches Us About What It Means to Be Alive"#,
    ),
    (
        r#"His hands on the sides of her face, on her neck, holding her there. "Chessie...shit, Chessie, I love you so bad." His teeth on her throat, biting hard, his lips soothing the spot. "So fucking much, so...so bad."#,
        r#"Stacia Kane, Sacrificial Magic"#,
    ),
    (
        r#"Love is like a game of chess. You're white. He's black. You wait for him to make a move, while staring into his handsome, melting-you-on-the-inside eyes, then realize what a dummy he is to not tell you straight out to go first. The beginning is the crush stage. You begin to realize how much you want to defeat him, or make him fall in love with you. By the time you get to the heat of the game, you both moved and are hopefully dating. If you haven't forfeit then because you don't want to be cheated on, you make another move- head on shoulder, hand holding, etc. Black makes another move-he gives you his jacket on a freezing night. By the endgame, he either realizes how stupid he was to play with you and forfeits, or he realizes how smart you are and lets you defeat him (and love you). By the time you win, you're married to him. A happily ever after game of chess."#,
        r#"Amrita Ramanathan"#,
    ),
    (
        r#"The game gives us a satisfaction that Life denies us. And for the Chess player, the success which crowns his work, the great dispeller of sorrows, is named 'combination'."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"There had been a few times over the past year when she felt like this, with her mind not only dizzied but nearly terrified by the endlessness of chess."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"Grandmaster games are said to begin with novelty, which is the first move of the game that exits the book. It could be the fifth, it could be the thirty-fifth. We think about a chess game as beginning with move one and ending with checkmate. But this is not the case. The games begins when it gets out of book, and it end when it goes into book..And this is why Game 6 [between Garry Kasparov and Deep Blue] didn't count...Tripping and falling into a well on your way to the field of battle is not the same thing as dying in it...Deep Blue is only itself out of book; prior to that it is nothing. Just the ghosts of the game itself."#,
        r#"Brian Christian, The Most Human Human: What Talking with Computers Teaches Us About What It Means to Be Alive"#,
    ),
    (
        r#"She’d fucked him over hardcore. She’d betrayed him and she’d lied to him, and she knew that as far as he was concerned she’d led him on and used him as well, had consorted with people who wanted to see him dead and given them information to help them make him so. Most of all, she’d hurt him. And if the pain in her chest was anything close to what he’d felt, she was more than willing to admit he deserved to get his own back. Was willing to do more than admit it; was willing to take it, in the hopes he’d eventually decide she’d been punished enough and they could maybe move on."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (r#"The Pin is mightier than the sword"#, r#"Fred Reinfeld"#),
    (
        r#"Beer bottles, whiskey bottles, brown glass, green. They fell to the lawn and I'd feel serene. Adam was king to my stilted queen."#,
        r#"Kate Bernheimer, The Complete Tales of Ketzia Gold"#,
    ),
    (
        r#"Aye. Freaky iffen you ask me. But guessing that what Terrible like, aye. What he deserve sneaking off into the bathroom with some rigmutton cunt, leaving me on my alones in the bar, and other men talking to me and saying I got me a date there and he fucking some whore while everyone outside the bathroom hearing them."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"I believe that, not only in chess, but in life in general, people place too much stock in ratings – they pay attention to which TV shows have the highest ratings, how many friends they have on Facebook, and it’s funny. The best shows often have low ratings and it is impossible to have thousands of real friends."#,
        r#"Boris Gelfand"#,
    ),
    (
        r#"With magic almost anything was possible; all objects had energy, and energy could be manipulated."#,
        r#"Stacia Kane, Unholy Ghosts"#,
    ),
    (
        r#"Her sigh felt dragged from the depths of her soul. Great. Working for Bump again."#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"Whereas a novice makes moves until he gets checkmated (proof), a Grand Master realizes 20 moves in advance that it’s futile to continue playing (conceptualizing)."#,
        r#"Bill Gaede"#,
    ),
    (
        r#"Then again, she had-through a bizarre combination of skill, dumb luck and incredible misfortune-managed to build up a file any Debunker would envy."#,
        r#"Stacia Kane, Sacrificial Magic"#,
    ),
    (
        r#"What is a weak pawn? A pawn that is exposed to attack and also difficult to defend is a weak pawn. There are several varieties: isolated, doubled, too advanced, retarded."#,
        r#"Samuel Reshevsky, Art of Positional Play"#,
    ),
    (
        r#"Like poetry, chess is also kinda poetic."#,
        r#"Joshua the poetic penguin"#,
    ),
    (
        r#"One night stands are like chess, only the loser desires a rematch."#,
        r#"Robert Black"#,
    ),
    (
        r#"I remember, back in college, how many possibilities life seemed to hold. Variations. I knew, of course, that I'd only live one of my fantasy lives, but for a few years there, I had them all, all the branches, all the variations. One day I could dream of being a novelist, one day I would be a journalist covering Washington, the next - oh, I don't know, a politician, a teacher, whatever. My dream lives. Full of dream wealth and dream women. All the things I was going to do, all the places I was going to live. They were mutually exclusive, of course, but since I didn't have any of them, in a sense I had them all. Like when you sit down at a chessboard to begin a game, and you don't know what the opening will be. Maybe it will be a Sicilian, or a French, or a Ruy Lopez. They all coexist, all the variations, until you start making the moves. You always dream of winning, no matter what line you choose, but the variations are still … different." … "Once the game begins, the possibilities narrow and narrow and narrow, the other variations fade, and you're left with what you've got - a position half of your own making, and half chance, as embodied by that stranger across the board. Maybe you've got a good game, or maybe you're in trouble, but in any case there's just that one position to work from. The might-have-beens are gone." (Unsound Variations)"#,
        r#"George R.R. Martin, Dreamsongs, Volume II"#,
    ),
    (
        r#"If one reads attentively, Wittgenstein writes as much in one of the rare pas- sages in which he makes use (in English) of the term to constitute with respect to the rules of chess:What idea do we have of the king of chess, and what is its relation to the rules of chess? . . . Do these rules follow from the idea? No, the rules are not something contained in the idea and got by analyzing it. They constitute it. . . . The rules constitute the freedom of the pieces. (Wittgenstein 5, p. 86)Rules are not separable into something like an idea or a concept of the king (the king is the piece that is moved according to this or that rule): they are immanent to the movements of the king; they express the autoconstitution process of their game. In the autoconstitution of a form of life what is in question is its freedom."#,
        r#"Giorgio Agamben, The Omnibus Homo Sacer"#,
    ),
    (
        r#"I prefer to make my annotations 'hot on the heels', as it were, when the fortunes of battle, the worries, hopes and disappointments are still sufficiently fresh in my mind. Much as I would like to, I cannot say this about these few games which will be given below. In fact, if the annotator should begin to use phrases of the type: 'in reply to...I had worked out the following variation...', the reader will rightly say 'Grandmaster, you are showing off', since the 'oldest' of these games is now more than 25 years old, and even the 'newest' more than 20. Therefore, I would ask you not to regard the following 'stylised' annotations too severely. "#,
        r#"Mikhail Tal, The Life and Games of Mikhail Tal"#,
    ),
    (
        r#"You can’t change the system unless you know it well. You can’t play chess well unless you know all the moves."#,
        r#"Abhaidev, The Influencer: Speed Must Have a Limit"#,
    ),
    (
        r#"What I wanted to tell you about Philidor was that Diderot wrote him a letter. You know Diderot?""The French Revolution?""Yeah. Philidor was doing blindfold exhibitions and burning out his brain, or whatever it was they thought you did in the eighteenth century. Diderot wrote him: 'It is foolish to run the risk of going mad for vanity's sake.' I think of that sometimes when I'm analyzing my ass over a chessboard."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"That is the trick of it. You see, Time works differently in Chess. He pulled out his pocket watch and let it dangle like a pendulum overhis desk. Sometimes he moves forward and sometimes he moves backward,sometimes he goes fast or slow and sometimes he pauses altogether. But as longas I keep moving, as long as I am always moving in the opposite direction fromTime, he can never find me, and I can never meet my fate."#,
        r#"Marissa Meyer, Heartless"#,
    ),
    (
        r#"Don't you get bored sometimes?, she asked.What else is there?, Benny replied."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"Chess certainly isn't all there is, Mrs. Wheatley continued.It's what I know, replied Beth."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"Papi taught me every piecehas its own space.Papi taught me every piecemoves in its own way.Papi taught me every piecehas its own purpose.The squares do not overlap. & neither do the pieces.The only time two pieces stand in the same squareis the second before oneis being taken & replaced."#,
        r#"Elizabeth Acevedo, Clap When You Land"#,
    ),
    (
        r#"She had heard of the genetic code that could shape an eye or hand from passing proteins. Deoxyribonucleic acid. It contained the entire set of instructions for constructing a respiratory system and a digestive one, as well as the grip of an infant's hand. Chess was like that. The geometry of a position could be read and reread and not exhausted of possibility. You saw deeply into the layer of it, but there was another layer beyond that, and another, and another."#,
        r#"Walter Tevis, The Queen's Gambit"#,
    ),
    (
        r#"World-class chess players, in addition to being considered awesomely smart, are generally assumed to have superhuman memories, and with good reason. Champions routinely put on exhibitions in which they play lesser opponents while blindfolded; they hold the entire chessboard in their heads. Some of these exhibitions strike the rest of us as simply beyond belief. The Czech master Richard Reti once played twenty nine blindfolded games simultaneously. (Afterward he left his briefcase at the exhibition site and commented on what a poor memory he had.)"#,
        r#"Geoff Colvin, Talent is Overrated: What Really Separates World-Class Performers from Everybody Else"#,
    ),
    (
        r#"He was a sweet man who loved words, and played chess with ghosts."#,
        r#"Eley Williams, The Liar's Dictionary"#,
    ),
    (
        r#"We must assume, I think, thatthe forward projection of what imagination he had, stopped at the act, on the brink ofall its possible consequences; ghost consequences, comparable to the ghost toes of anamputee or to the fanning out of additional squares which a chess knight (that skips-pace piece), standing on a marginal file, "feels" in phantom extensions beyond theboard, but which have no effect whatever on his real moves, on the real play."#,
        r#"Vladimir Nabokov, Pale Fire"#,
    ),
    (
        r#"We must assume, I think, that the forward projection of what imagination he had, stopped at the act, on the brink of all its possible consequences; ghost consequences, comparable to the ghost toes of an amputee or to the fanning out of additional squares which a chess knight (that skipspace piece), standing on a marginal file, "feels" in phantom extensions beyond the board, but which have no effect whatever on his real moves, on the real play."#,
        r#"Vladimir Nabokov, Pale Fire"#,
    ),
    (
        r#"There is profound meaning in the game of chess. The board itself is life and death, painted as such in black and white. The pieces are those that make a life fundamentally healthy. The pawns are attributes we gather with nourishment and significance. The knight is our ability to be mobile and travel in whatever form it takes. The rook or castle is a place we can call home and protect ourselves from the elements. The bishop is that of our community and our belonging. The king is our mortal body; without it, we can no longer play the game. The queen is the spirit of the body - what drives our imagination, urges, a life force. A captured queen removes energy from the game, and the player may become complacent. A crowning reminder of the game is that the spirit can be possessed again through our attributes."#,
        r#"Lorin Morgan-Richards"#,
    ),
    (
        r#"Într-o dimineață, Igor Șansa se trezi metamorfozat într-un pion."#,
        r#"Ionut Iamandi, Șahiștii. Povestiri"#,
    ),
    (
        r#"Life is an exchange; you'd think a chess player would know that."#,
        r#"Elizabeth Acevedo, Clap When You Land"#,
    ),
    (
        r#"In life, as in chess, learning must be constant - both new things and fresh ways of learning them. The process will invariably involve a certain degree of unlearning, and possessing the readiness to that is utterly important. If your way of doing things isn't working, clinging to your conclusions is only going to hold you back. You have to get to the root of a snag in order to make a breakthrough, because it's possible that what you thought you knew isn't actually the way it is. Unlearning is perhaps the hardest thing to do, but it is a necessity if growth and success are your goals."#,
        r#"Vishwanathan Anand"#,
    ),
    (
        r#"At the beginning of a game, there are no variations. There is only one way to set up a board. There are nine million variations after the first six moves. And after eight moves there are two hundred and eighty-eight billion different positions. And those possibilities keep growing. [...] In chess, as in life, possibility is the basis of everything. Every hope, every dream, every regret, every moment of living. (p.195)"#,
        r#"Matt Haig, The Midnight Library"#,
    ),
    (
        r#"Coaching is more like chess; it’s about out-thinking and outsmarting the other team."#,
        r#"C. Vivian Stringer, Standing Tall: Lessons in Turning Adversity into Victory"#,
    ),
    (
        r#"Life is like a game where pawns can become queens, but not everyone knows how to play. Some people stay pawn their whole lives because they never learned to make the right moves."#,
        r#"Alice Feeney, Rock Paper Scissors"#,
    ),
    (r#"Can we go to chessy now?"#, r#"Greyson Sweeny"#),
    (
        r#"Truth derives its strength not so much from itself as from the brilliant contrast it makes with what is only apparently true. This applies especially to Chess, where it is often found that the profoundest moves do not much startle the imagination."#,
        r#"Emanuel Lasker, Common Sense in Chess"#,
    ),
    (
        r#"...you could never be completely sure of the other person, so never make a move until you were sure of yourself."#,
        r#"Liz Braswell, Part of Your World"#,
    ),
    (
        r#"You’re just a pawn on the chessboard, Leo Valdez. I was referring to the player who set this ridiculous quest in motion, bringing the Greeks and Romans together."#,
        r#"Rick Riordan, The Mark of Athena"#,
    ),
    (
        r#"Chess does not only teach us to analyse the present situation, but it also enables us to think about the possibilities and consequences. This is the art of forward-thinking."#,
        r#"Shivanshu K. Srivastava"#,
    ),
    (
        r#"When you don’t know what to do, wait for your opponent to get an idea — it’s sure to be wrong!"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"On the chessboard lies and hypocrisy do not survive long. The creative combination lays bare the presumption of a lie; the merciless fact, culminating in a checkmate, contradicts the hypocrite."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Fighting was chess, anticipating the move of one's opponent and countering it before one got hit."#,
        r#"Holly Black, The Wicked King"#,
    ),
    (
        r#"The most helpful thing I learnt from chess is to make good decisions on incomplete data in a limited amount of time."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"A man shined to her left. He was called Lorenzo and he drank a hot chocolate with whole milk. He sipped it with fleshy, pink lips and60k.f.gulped it down his large neck that seemed to be a kind of engine. The gulp went down his chest, where his muscles cooled after his calisthenics, and sunk somewhere behind the walls of his tight, tan stomach. He was a chess set of a man. He had burly knights as biceps, thick bishops as legs, healthy pawns as his troop of fingers, and the battlement of rooks as his fortified abs of stone."#,
        r#"Karl Kristian Flores, A Happy Ghost"#,
    ),
    (r#"He sacrificed the ROOOK"#, r#"Gotham Chess"#),
    (
        r#"Chess is all about maintaining coherent strategies. It’s about not giving up when the enemy destroys one plan but to immediately come up with the next. A game isn’t won and lost at the point when the king is finally cornered. The game's sealed when a player gives up having any strategy at all. When his soldiers are all scattered, they have no common cause, and they move one piece at a time, that’s when you’ve lost."#,
        r#"Kazuo Ishiguro, A Pale View of Hills"#,
    ),
    (
        r#"Vimes had never got on with any game much more complex than darts. Chess in particular had always annoyed him. It was the dumb way the pawns went off and slaughtered their fellow pawns while the kings lounged about doing nothing that always got to him; if only the pawns united, maybe talked the rooks round, the whole board could've been a republic in a dozen moves."#,
        r#"Terry Pratchett, Thud!"#,
    ),
    (
        r#"You need to realise something if you are ever to succeed at chess,’ she said, as if Nora had nothing bigger to think about. ‘And the thing you need to realise is this: the game is never over until it is over. It isn’t over if there is a single pawn still on the board. If one side is down to a pawn and a king, and the other side has every player, there is still a game. And even if you were a pawn – maybe we all are – then you should remember that a pawn is the most magical piece of all. It might look small and ordinary but it isn’t. Because a pawn is never just a pawn. A pawn is a queen-in-waiting. All you need to do is find a way to keep moving forward. One square after another. And you can get to the other side and unlock all kinds of power.'Mrs. Elm"#,
        r#"Matt Haig, The Midnight Library"#,
    ),
    (
        r#"The two board games that best approximate the strategies of war are chess and the Asian game of go. In chess, the board is small. In comparison to go, the attack comes relatively quickly, forcing a decisive battle.... Go is much less formal. It is played on a large grid, with 361 intersections — nearly six times as many positions as in chess.... [A game of go] can last up to three hundred moves. The strategy is more subtle and fluid than chess, developing slowly; the more complex the pattern your stones initially create on the board, the harder it is for your opponent to understand your strategy. Fighting to control a particular area is not worth the trouble: You have to think in larger terms, to be prepared to sacrifice an area in order eventually to dominate the board. What you are after is not an entrenched position but mobility. With mobility you can isolate your opponent in small areas and then encircle them... Chess is linear, position oriented, and aggressive; go is nonlinear and fluid. Aggression is indirect until the end of the game, when the winner can surround the opponents' stones at an accelerated pace."#,
        r#"Robert Greene, The 48 Laws of Power"#,
    ),
    (
        r#"Fancy what a game of chess would be if all the chessmen had passions and intellects, more or less small and cunning; if you were not only uncertain about your adversary's men, but a little uncertain also about your own; if your knight could shuffle himself on to a new square by the sly; if your bishop, at your castling, could wheedle your pawns out of their places; and if your pawns, hating you because they are pawns, could make away from their appointed posts that you might get checkmate on a sudden. You might be the longest-headed of deductive reasoners, and yet you might be beaten by your own pawns. You would be especially likely to be beaten, if you depended arrogantly on your mathematical imagination, and regarded your passionate pieces with contempt. Yet this imaginary chess is easy compared with the game a man has to play against his fellow-men with other fellow-men for his instruments."#,
        r#"George Eliot, Felix Holt: The Radical"#,
    ),
    (
        r#"Excelling at chess has long been considered a symbol of more general intelligence. That is an incorrect assumption in my view, as pleasant as it might be."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Но человек существо легкомысленное и неблаговидное и, может быть, подобно шахматному игроку, любит только один процесс достижения цели, а не самую цель."#,
        r#"Fyodor Dostoyevsky, Notes from Underground"#,
    ),
    (
        r#"I am not the piece, I am not of the piece, I am not in the piece. I am the move"#,
        r#"Niranjan Navalgund"#,
    ),
    (
        r#"Chess enjoys a not wholly undeserved reputation for psychic derangement. It is an endeavor associated, when not with frank madness, with oddness and isolation. I remember a psychiatrist friend visiting me at a chess club in downtown Boston once. He walked in, sat down, looked around and said, ‘Jeez, I could run a group here."#,
        r#"Charles Krauthammer, The Point of It All: A Lifetime of Great Loves and Endeavors"#,
    ),
    (
        r#"Chess, like love, is infectious at any age - Salo Flohr"#,
        r#"Irving Chernev, The Most Instructive Games of Chess Ever Played"#,
    ),
    (
        r#"Celebrating the Queen!Chess tells you that the king can't do anything alone8 March International Women's Dayशतरंज बताती है राजा अकेला कुछ नहीं कर सकता है"#,
        r#"Vineet Raj Kapoor"#,
    ),
    (
        r#"I thought you wanted me to teach you how to play. (Chess)Each possible move represents a different game - a different universe in which you make a better move.By the second move there are 72,084 possible games. By the 3rd - 9 million. By the 4th….There are more possible games of chess than there are atoms in the universe. No one could possibly predict them all, even you. Which means that first move can be terrifying. It’s the furthest point from the end of the game. There’s a virtually infinite sea of possibilities between you and the other side but it also means that if you make a mistake, there’s a nearly infinite amount of ways to fix it so you should simply relax and play."#,
        r#"Person of Interest s04e11"#,
    ),
    (
        r#"Remember that in chess, it's only the square you land on that matters."#,
        r#"Bill Robertie, Beginning Chess Play"#,
    ),
    (
        r#"Our uniqueness implies unique responsibility that we as individuals have in the world. It also implies an unavoidable loneliness. Here Ghandi’s words come to mind: Even if you are a minority of one, the truth is still the truth. We are all minorities of one in the sense of our uniqueness and loneliness. But in searching for the truth and the meaning of our lives, we intercept with others who are doing the same and our loneliness at least will not have to be experienced as isolation. And in chess too, we intercept with others in this common interest that is much like life, where everything we do matters, where we have to participate responsibly, and the more responsible our participation is, the more we feel at home. As such it can have a highly affirmative effect on the person, a sense that the individual gets: Yes, I belong to this world, I am part of how things get decided, of how things get achieved. I share this with others."#,
        r#"Roumen Bezergianov, Character Education with Chess"#,
    ),
    (
        r#"The pieces are connected to each other and the King and they are in this dynamic rhythm amongst themselves and with the opponent’s pieces, wherein lies their purpose. Each move is an attempt to change that balance and to establish a new, more favorable balance and that is why in chess (and in life) we are most vulnerable when we are most aggressive—the aggressive move essentially causes us to lose balance."#,
        r#"Roumen Bezergianov, Character Education with Chess"#,
    ),
    (
        r#"Why settle for thinking like a human if you can be a god."#,
        r#"Garry Kasparov, Deep Thinking: Where Machine Intelligence Ends and Human Creativity Begins"#,
    ),
    (
        r#"They say that chess was born in bloodshed."#,
        r#"Paolo Maurensig, La variante di Lüneburg"#,
    ),
    (
        r#"Life is a lot like chess," he said.…"All a matter of choices. Every move you face choices, and every choice leads to different variations. It branches and then branches again, and sometimes the variation you pick isn't as good as it looked, isn't sound at all. But you don't know that until your game is over."(Unsound Variations)"#,
        r#"George R.R. Martin, Dreamsongs, Volume II"#,
    ),
    (r#"Checkers is for tramps."#, r#"Paul Charles Morphy"#),
    (
        r#"The Empire State Building is a lady. She’s like the queen in chess, the most powerful piece. She’s like America."#,
        r#"A.D. Aliwat, In Limbo"#,
    ),
    (
        r#"Conocía desde luego, por propia experiencia, el misterioso poder de atracción del «juego de reyes», de ese juego entre los juegos, el único entre los ideados por el hombre que escapa soberanamente a cualquier tiranía del azar, y otorga los laureles de la victoria exclusivamente al espíritu o, mejor aún, a una forma muy característica de agudeza mental."#,
        r#"Stefan Zweig, Chess Story"#,
    ),
    (r#"People are like chess pieces!"#, r#"Deyth Banger"#),
    (
        r#"I am not the piece, I am not of the piece, I am not in the piece. I am the move!"#,
        r#"Niranjan Navalgund"#,
    ),
    (
        r#"The King in chess is indeed a symbol of unity and wholeness and the other pieces are not separate entities but rather parts of the One Thing, as Campbell put it."#,
        r#"Roumen Bezergianov, Character Education with Chess"#,
    ),
    (
        r#"Chess is not a game, it's a war."#,
        r#"Joshua the poetic penguin"#,
    ),
    (
        r#"I always plan for longterm, life to me is a never ending chess match"#,
        r#"James D. Wilson"#,
    ),
    (
        r#"ख़ुद ही देखो तुमने शह किसकी पाई हैज़ाहिर है तुम्हें कि मात कहाँ से आई हैKHUDD HI DEKHO, TUMNEIN SHAH KISKI PAI HAIZAHIR HAI TUMHEIN KI MAT KAHAN SE AAYI HAI"#,
        r#"Vineet Raj Kapoor"#,
    ),
    (
        r#"There is no moral outcome of a chess match or a poker game as long as skill and stealth rather than cheating have been used."#,
        r#"Francis P. Karam, The Truth Engine: Cross-Examination Outside the Box"#,
    ),
    (
        r#"While it is thrilling to hear of such faith in me, he answered, repositioning a bishop at a threatening angle, I admit any potential witch has the capacity to frighten me. Any witch might be capable of bringing a second Mazriv and drowning the remains of the world in despair."#,
        r#"Uri Gatt Gutman, Winds of Strife"#,
    ),
    (
        r#"Truth is simple yet purposely complex."#,
        r#"Wald Wassermann"#,
    ),
    (
        r#"Remember the white knight."Though it seemed so long ago, he well remembered their conversation about the chess problem. The white knight had made a move, changed his mind, and started over."And do you believe this was a good move?" Mr. Benedict had asked."No, sir," Reynie had answered."Why, then, do you think he made it?"And Reynie had replied, "Perhaps because he doubted himself."#,
        r#"Trenton Lee Stewart, The Mysterious Benedict Society"#,
    ),
    (
        r#"As a rule, the more mistakes there are in a game, the more memorable it remains, because you have suffered and worried over each mistake at the board."#,
        r#"Viktor Korchnoi, My Best Games"#,
    ),
    (
        r#"I used to attack because it was the only thing I knew. Now I attack because I know it works."#,
        r#"Garry Kasparov, How Life Imitates Chess: Making the Right Moves, from the Board to the Boardroom"#,
    ),
    (
        r#"Don't fight over dead Kings and Queens. The Kings never fought for you."#,
        r#"Vineet Raj Kapoor"#,
    ),
    (
        r#"Once the game is over,’  he says,  ‘the king and the pawn go back in the same box.’ In life and death we are equal."#,
        r#"Lisa Renee Jones, Demand"#,
    ),
    (
        r#"Chess problems are the hymn-tunes of mathematics."#,
        r#"G H Hardy, A Mathematician's Apology"#,
    ),
    (
        r#"We are like chess players who are trying to predict the opponent’s future moves, but in this case, we are dealing with life itself. True masters do not play the game on a single chessboard, but on multiple chessboards at the same time. And what’s the difference between grandmasters and masters? Surprises. The moves that cannot be predicted by the opponent. Life can play a simultaneous game with seven billion people at the same time and it can take each and every one of us by surprise. And we still believe we are capable of winning, because we can predict three of four moves ahead. We are insignificant."#,
        r#"Jaka Tomc, 720 Heartbeats"#,
    ),
    (
        r#"Vladimir Vladimirovich Putin is the President of Russia and the United States of America."#,
        r#"A.K. Kuykendall"#,
    ),
    (
        r#"I learned about opening moves and why it's important to control the center early on; the shortest distance between two points is straight down the middle."#,
        r#"Amy Tan, The Joy Luck Club"#,
    ),
    (
        r#"Не знаешь, как ходить – ходи крайней пешкой. Позиция не изменится, но ты вроде что-то сделал."#,
        r#"Paúliuk Bykoúski"#,
    ),
    (
        r#"It's usually the father who teaches the child his first moves in the game. And the dream of any son who plays chess is to beat his father. To kill the king. Besides, it soon becomes evident in chess that the father, or the king, is the weakest piece on the board. He's under continual act, in constant need of protection, of such tactics as castling, and he can only move one square at a time. Paradoxically, the king is also indispensable. The king gives the game its name, since the word 'chess' derives from the Persian word shah meaning king, and is pretty much the same in most languages."#,
        r#"Arturo Pérez-Reverte, The Flanders Panel"#,
    ),
    (
        r#"Playing chess with my father is torture. I have to sit very upright on the edge of my chair and respect the rules of impassivity while I consider my next move. I can feel myself dissolving under his stare. When I move a pawn he asks sarcastically, 'Have you really thought about what you're doing?' I panic and want to move the pawn back. He doesn't allow it: 'You've touched the piece, now you have to follow through. Think before you act. Think."#,
        r#"Maude Julien, The Only Girl in the World"#,
    ),
    (
        r#"The Beauty of the Chess Set is of no import to the Player//Lessons in Aesthetics"#,
        r#"Vineet Raj Kapoor"#,
    ),
    (
        r#"We hardly said anything, we seemed to communicate through the chessmen, there was something very symbolic about my winning. That he wished me to feel. I don’t know what it was. I don’t know whether it was that he wanted me to see my virtue triumphed over his vice or something subtler, that sometimes losing is winning."#,
        r#"John Fowles, The Collector"#,
    ),
    (
        r#"Count Ambrosius was out of practice at chess. The usual game was dice, and he was not risking that against an infant soothsayer. Chess, being a matter of mathematics rather than magic, was less susceptible to the black arts.' --twelve-year-old Merlin, The Crystal Cave, p. 131 of 384"#,
        r#"Mary Stewart, Merlin Trilogy Collection Vol (1-5) 5 Books Bundle By Mary Stewart (The Crystal Cave,The Wicked Day) Gift Wrapped Slipcase Specially For You"#,
    ),
    (
        r#"Life is short, precious, and should not be wasted. Everyone has a chance at it. We’re equals after all. There are no pawns, no kings, and no queens. We’re all humans and we all have the same value."#,
        r#"Cristelle Comby, Blind Chess"#,
    ),
    (
        r#"...why did you stick with me? That’s easy, Alice, he said. His amber eyes melted into hers. It’s the first rule of chess: always protect your queen."#,
        r#"J.M. Sullivan, Alice"#,
    ),
    (
        r#"You have a friend named Bug? Sure. Why wouldn’t I? Chess asked. I have another friend who keeps telling me to call her Alice even though I give her way better names— which is even weirder if you ask me. He gave her a pointed look."#,
        r#"J.M. Sullivan, Alice"#,
    ),
    (
        r#"He knew once he stepped into that kind of environment, again, the options would be limited. He’d no longer have the freedom or control to make any important decisions. He’d be just another pawn to be used on the chessboard by the white shirt bosses, who would likely be making their decisions from a safe distant location and passing them along down the totem pole. It was just how his job worked."#,
        r#"Jason Medina, The Manhattanville Incident: An Undead Novel"#,
    ),
    (
        r#"Chess is a miniature version of life. To be successful, you need to be disciplined, assess resources, consider responsible choices and adjust when circumstances change."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"I put my hand on a bishop, my would be assassin, and thought of my father's heights when he won, how he galloped around. The depths of his despair at losing, I expected, would be equal to the peaks. He'd mope about, his face fallen and miserable, his posture stooped as if his back ached. I took my hand from the piece and leaned back in deliberation."#,
        r#"Rion Amilcar Scott, Insurrections: Stories"#,
    ),
    (
        r#"The paradox of illuminating complexity is that it is inherently difficult to do so without erasing all of the nuance."#,
        r#"David Shenk, The Immortal Game: A History of Chess, or How 32 Carved Pieces on a Board Illuminated Our Understanding of War, Art, Science and the Human Brain"#,
    ),
    (
        r#"By the turn of the twentieth century, every state in the South had laws on the books that disenfranchised blacks and discriminated against them in virtually ever sphere of life, lending sanction to a racial ostracism that extended to schools, churches, housing, jobs, restrooms, hotels, restaurants, hospitals, orphanages, prisons, funeral homes, morgues, and cemeteries. Politicians competed with each other by proposing and passing every more stringent, oppressive, and downright ridiculous legislation (such as laws specifically prohibiting blacks and whites from playing chess together.)"#,
        r#"Michelle Alexander, The New Jim Crow: Mass Incarceration in the Age of Colorblindness"#,
    ),
    (
        r#"Chess competitions have a women’s and men’s division. The ego is a very powerful thing."#,
        r#"Freequill"#,
    ),
    (r#"Life is a deadly chess lie."#, r#"Deyth Banger"#),
    (
        r#"Fradique looked intensely at me. 'You are the proof that God exists,' he said, 'and that he is quite mad'. He leaned towards me and kissed me, and I kissed him. Later we went back to looking at the maps, and played a game of chess. I asked him what he had meant when he spoke about God's madness. Fradique laughed. 'Only a thoroughly insane God could conceive of an angel, and then place her in Hell."#,
        r#"José Eduardo Agualusa, Nação Crioula"#,
    ),
    (
        r#"Tablebases [logs of complete chess games played backwards from the end-state of checkmate] are the clearest case of human chess vs. alien chess. A decade of trying to teach computers how to play endgames was rendered obsolete in an instant thanks to a new tool. This is a pattern we see over and over again in everything related to intelligent machines. It's wonderful if we can teach machines to think like we do, but why settle for thinking like a human if you can be a god?(jm3: Frustratingly for the humans, it was not disclosed whether IBM's Deep Blue stored and consulted endgame tablebases during competition)."#,
        r#"Garry Kasparov, Deep Thinking: Where Machine Intelligence Ends and Human Creativity Begins"#,
    ),
    (
        r#"Do not expect Ordinary Gameplay on the upside of Life."#,
        r#"Vineet Raj Kapoor, UNCHESS: Untie Your Shoes and Walk on the Chessboard of Life"#,
    ),
    (
        r#"The final aim of all of us playing on the board of life is to somehow break out of this board and be free"#,
        r#"Vineet Raj Kapoor, UNCHESS: Untie Your Shoes and Walk on the Chessboard of Life"#,
    ),
    (
        r#"Gameplay is all our life. Either we guard, attack or develop pieces."#,
        r#"Vineet Raj Kapoor, UNCHESS: Untie Your Shoes and Walk on the Chessboard of Life"#,
    ),
    (
        r#"Our parting was like a stalemate….Neither of us won. Yet both of us lost.And worse still … that unshakable feeling that nothing was ever really finished."#,
        r#"Ranata Suzuki"#,
    ),
    (
        r#"The passion for playing chess is one of the most unaccountable in the world. It slaps the theory of natural selection in the face. It is the most absorbing of occupations. The least satisfying of desires. A nameless excrescence upon life. It annihilates a man. You have, let us say, a promising politician, a rising artist that you wish to destroy. Dagger or bomb are archaic and unreliable - but teach him, inoculate him with chess."#,
        r#"H.G. Wells"#,
    ),
    (
        r#"It was inevitable: the scent of bitter almonds always reminded him of the fate of unrequited love. Dr. Juvenal Urbino noticed is as soon as he entered the still darkened house where he has hurried on an urgent call to attend a case that for him had lost all urgency many years before. The Antillean refugee Jeremiah de Saint-Amour, disabled war veteran, photographer of children, and his most sympathetic opponent in chess, had escaped the torments of memory with the aromatic fumes of gold cyanide."#,
        r#"Gabriel García Márquez, Love in the Time of Cholera"#,
    ),
    (
        r#"All that matters on the chessboard is good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"It was going to be our job to annoy someone? I know—it’s a dream come true!"#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"If I had wanted to be a pawn in a game, I would have taken up chess..."#,
        r#"Virginia Alison"#,
    ),
    (
        r#"Rosa!" Sally says. "The police are here to help you, not to hear a lecture on comparative murder rates."#,
        r#"Justine Larbalestier, My Sister Rosa"#,
    ),
    (
        r#"After an hour the score was: Quancita—34 Radiz—51 Sally—froglegs Perla—9 and 21 Me— hoo-hoo-hooo"#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"In chess, without the king, the other pieces would all be "dead", so their existance is supported by the king, but they need to serve the king with their capacity for action in order to have a good game."#,
        r#"Roumen Bezergianov"#,
    ),
    (
        r#"You have good instincts,trust them. Thinking through every step is fine if you're playing chess, but this isn't chess."#,
        r#"Rick Yancey, The 5th Wave"#,
    ),
    (
        r#"Sometimes, all company forsaking,They settle to a game of chessAnd, leaning on a table, guessWhat move the other may be making,And Lensky with a dreamy look,Allows his pawn to take his rook."#,
        r#"Alexander Pushkin, Eugene Onegin"#,
    ),
    (
        r#"Uh, she said maybe your eyes matched the Fog like a synchronous magnetic field? I don’t even know what language that is."#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Better I won't answer on this question, I think it's useless so far the future is that it will go in conflict how I know this?It's a tactic taken from chess!"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Did Cap’n Vidious leave that? He is such a cuddlebunny. Yeah, I said, that’s exactly how I’d describe him."#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"You know what to do? Wander around, I said. Until I spot a self-assembled whangdoodle from the Foggy depths."#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Did you just call me ‘sweetie’? I asked. She shoved my shoulder. No."#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Well, the bad news, Swedish said from the wheel, is that Chess still thinks he’s funny. What’s the good news? Loretta asked, leaning on our little copper-tubed harpoon. That Kodoc dropped a bomb on the city?"#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Now you’re listening to Swedish ? I asked her. He thinks I’m the Compass because every time I see ticktocks, I happen to be there!"#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"If Chess is the switch, Loretta said, how does he turn the Fog off? Bea bit her lower lip. I don’t know—ask Chess. How would I know? I said. You try being a switch."#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Why would they follow me? I asked Swedish. It doesn’t make sense. They’re made of trash and Fog, he said. You think they make sense?"#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"With a silent order, I urged Snout forward—but he veered away, charging toward Hazel instead. No, Snout! I thought. Toward the roof! He ignored me. That was the problem with a machine that obeyed your thoughts. Instead of doing what you said, it did what you wanted. The Predator ! Hazel shouted at me as I heaved toward the irrigation tower. Stop the Predator ! I’m trying! I yelled back. I can’t! Why not? ’Cause this stupid thing brought me to you instead.Why? Then she looked at my face again and said, Aw, that’s sweet. I flushed. Oh, shut up."#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Wait a second, I said. Did we just win?"#,
        r#"Joel N. Ross, The Lost Compass"#,
    ),
    (
        r#"Life is a Game of Chess Between Me and God; Let the Best Player Win"#,
        r#"Ashu Gaur"#,
    ),
    (
        r#"...That is my biography from the first day of my chess life to the present.JOURNALIST. And your plans.PLAYER. To play!"#,
        r#"Mikhail Tal, The Life and Games of Mikhail Tal"#,
    ),
    (
        r#"Here are few important things to know about chess, if somebody you have beat decline another fight... Let's see what's the feeling to lose, let's feel it, let's be angry, let's the anger full him... (So far this is enjoying to watch...)(Little Off the topic, but so far when I finished one quiz about chess it was said that I am a Mad Scientist)...- When you lose just go and see the enemy learn from him lessons about chess, ask him questions and many other useful stuff... then one day after few weeks or who knows when... suggest another battle You VS Him...To win and to think different you need tactics they are 90% of the game the other 10% are to know how something moves...So as my suggestion solve problems on the phone or on the computer on the tablet or where you want... solve them... they will help you in a battle... You will remember one problem and somehow you will be in it and you will know the answer.When you are tired listen to music and try something else..."#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Tactics and Checkmate in 1 move, show me some interesting stuff about chess... so far I can say that I see the chessboard different."#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Chess Tactics are important in the study.... I don't know about that but I know that this son of biatch beats me on chess games and on chess tactics puzzles he sucks!"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Off.. offf leave the game in my hands if I am going to lose I wanna lose without special help, what you are saying is crap... castle, castle, castle in chess is important... fuck off that guy won without even castling."#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Life is a mysterious and witty intermingling of fate and events"#,
        r#"Alexandra Kosteniuk"#,
    ),
    (
        r#"In life, as in chess, it is always better to analyze one's motives and intentions."#,
        r#"Vladimir Nabokov, Pnin"#,
    ),
    (
        r#"Of course, errors are not good for a chess game, but errors are unavoidable and in any case, a game without errors, or as they say 'flawless game' is colorless."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Deep Blue didn't win by being smarter than a human; it won by being millions of times faster than a human. Deep Blue had no intuition. An expert human player looks at a board position and immediately sees what areas of play are most likely to be fruitful or dangerous, whereas a computer has no innate sense of what is important and must explore many more options. Deep Blue also had no sense of the history of the game, and didn't know anything about its opponent. It played chess yet didn't understand chess, in the same way a calculator performs arithmetic bud doesn't understand mathematics."#,
        r#"Jeff Hawkins, On Intelligence"#,
    ),
    (
        r#"The move is there, but you must see it."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"A deep laugh stirred in his chest, and his thumb brushed over the backs of her fingers before he withdrew his hand. She felt the rasp of a callus on his thumb, the sensation not unlike the tingling scrape of a cat’s tongue. Bemused by her own response to him, Annabelle looked down at the chess piece in her hand. That is the queen—the most powerful piece on the board. She can move in any direction, and go as far as she wishes. There was nothing overtly suggestive in his manner of speaking …but when he spoke softly, as he was doing at that moment, there was a husky depth in his voice that made her toes curl inside her slippers. More powerful than the king? she asked. Yes. The king can only move one square at a time. But the king is the most important piece. Why is he more important than the queen if he’s not the most powerful? Because once he is captured, the game is over."#,
        r#"Lisa Kleypas, Secrets of a Summer Night"#,
    ),
    (
        r#"Where mermaids live looks a bit like your pool.' said Bernard. 'Except they build houses out of whale bones and the wreckage of sunken ships. They play chess with seahorses. They wear capes of fish scales and sleep on beds made from seaweed.'As we listened, I thought I heard a slight splashing from the far end of the pool.'At night,' Bernard continued, 'they turn on an electric eel for a night-light, and they light a fire, and the smoke goes up a chimney made from coral.''Wait a minute,' interupted Zoe, clearly immersed in Bernard's description. 'If they live underwater, how could they have a fire?''You should ask them,' said Bernard.Zoe and I open our eyes.Now, look, I know the light was just playing tricks on us. And I know we'd all probably inhaled too much sequin glue. But for the briefest moment, the blue of Zoe's pool gave way to deeper, darker aqua-colored water. The few plants and rocks were replaced with a lagoon and a waterfall where several mermaids lounged half in the water, half in the sun. They splashed and dove, their laughter making the same sound as the water."#,
        r#"Michelle Cuevas, Confessions of an Imaginary Friend"#,
    ),
    (
        r#"I will, therefore, take occasion to assert that the higher powers of the reflective intellect are more decidedly and more usefully tasked by the unostentatious game of draughts than by a the elaborate frivolity of chess."#,
        r#"Edgar Allan Poe, The Murders in the Rue Morgue: The Dupin Tales"#,
    ),
    (
        r#"She plays chess from the passions and I play it from logic and she usually wins. Once, I took her queen and she hit me.Though, he recalled, not sufficiently brutally to require that he tie her wrists together with his belt, force her to kneel and beat her until she toppled over sideways. She raised a strangely joyous face to him; the pallor of her skin and the almost miraculous lustre of her eyes startled and even awed him."#,
        r#"Angela Carter, Love"#,
    ),
    (
        r#"The cherished dream of every chessplayer is to play a match with the World Champion. But here is the paradox: the closer you come to the realization of this goal, the less you think about it."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Divided - No tides of time or distance will wash away your step. It does not fleet as they do, those gladiators and their mighty spears or the beasts that howl into the dark for release. Our story carves deeper, pitilessly, infinitely. A wound that bleeds the ink that stained your palm and the tears of an impossible tomorrow."#,
        r#"RJ Arkhipov"#,
    ),
    (
        r#"Chess is a game with simple rules and pieces, a small sixty-four-space board, but there are more possible chess games than there are atoms in the universe."#,
        r#"Austin Grossman, You"#,
    ),
    (
        r#"The Offing - And if the sky itself, no matter its hue, were to fracture... What then? Would I then know freedom's name?In my wake lies the shore—a past where I had been happy—refusing to yield to the tide. Before me, upon the horizon, is the sun... hesitant... inert... A new day cannot rise if its ancestor does not fall. Am I but a pawn in this game? I cannot command the sun to set, nor will the moon to take its place and wash the shore away. That power belongs to kings.To drown in the offing. Such sovereign beauty. Such exquisite pain."#,
        r#"RJ Arkhipov"#,
    ),
    (
        r#"Zugzwang. It's when you have no good moves. But you still have to move."#,
        r#"Michael Chabon"#,
    ),
    (
        r#"У нас есть шахматы с собой,Шекспир и Пушкин, с нас довольно."#,
        r#"Vladimir Nabokov, Стихотворения"#,
    ),
    (
        r#"The art of chess is in knowing which is the most valuable piece in play, then having the courage to sacrifice it for the win."#,
        r#"Thomm Quackenbush, Flies to Wanton Boys"#,
    ),
    (
        r#"But I find something compelling in the game's choreography, the way one move implies the next. The kings are an apt metaphor for human beings: utterly constrained by the rules of the game, defenseless against bombardment from all sides, able only to temporarily dodge disaster by moving one step in any direction."#,
        r#"Jennifer duBois, A Partial History of Lost Causes"#,
    ),
    (
        r#"What Will Linger/Hollow of Him - They crept so quietly back. Mere hints of words, at first, then whispers in the loud echoing a winter past. In this place, hollow of Him, his poetry resounded. I could almost taste the fragments of the worlds he had discovered. I remember the ache in his words; you could see each syllable smoulder in his gaze."#,
        r#"RJ Arkhipov"#,
    ),
    (
        r#"Táctica es saber qué hacer cuando hay algo para hacer. Estrategia es saber qué hacer cuando no hay nada para hacer."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Today, some chess games can't go in 40 minutes. One game not every time can finish fast, sometimes it's difficult to win and the ways are odd."#,
        r#"Deyth Banger"#,
    ),
    (r#"Chess is an easy game!"#, r#"Deyth Banger"#),
    (
        r#"Magic = Chess, Chess is with strategy, doing moves which people think that they are random, but they aren't random. They are part of the strategy. The same is in the magic. However, the magic, have the ability to distraction!"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"I never have heard somebody talking about games which are educationable or with logic like chess for example. Nobody watch it from the people which watch football and are good at math. Why they don't do it??"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Everyone have a game, everyone is playing in his own game and his own game he is the king the others are the other figures from the chess."#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Le jeu dechec, sige de Franske, n'est pas assez jeu: Det er, Skakspill og andre af samme Betydelse, er ikke Spill, men et Studium. Saadant kand man presentere dem, som intet have at bestille, og som af Ørkesløshed kand frygte for Hiernens Forrustelse, men ikke arbeydsomme Folk, der udi Spill og Selskab søge Recreation."#,
        r#"Ludvig Holberg, Epistler"#,
    ),
    (
        r#"Check your moves well, because it can cost one pawn or losing a lot of just from three moves!"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"You want a fact???...I'm bad at math but good at chess, I beat the best guy on chess... so you make your own conclusions!"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"It's hard to run from checkmate, checkmate is like the dead... but it's possible to block it. Unfortunately dead you can't block it, but checkmate can be blocked, in such way so the player can't make it."#,
        r#"Deyth Banger"#,
    ),
    (
        r#"We move like the chess game, first the white... then the black... the moves are too fast so far it's difficult to see it. But it's logical, isn't it?"#,
        r#"Deyth Banger"#,
    ),
    (
        r#"In chess puzzles and tactics you see moves which in game is difficult to see mainly because of time and more often from the fast moves..."#,
        r#"Deyth Banger"#,
    ),
    (
        r#"Imagine God and Man set down together to play that game of chess that we call life. The one player is a master, the other a bungling amateur, so the outcome of the game cannot be in question. The amateur has free will, he does what he pleases, for it was he who chose to set up his will against that of the master in the first place; he throws the whole board into confusion time and again and by his foolishness delays the orderly ending of it all for countless generations, but every stupid move of his is dealt with by a masterly counterstroke, and slowly but inexorably the game sweeps on to the master's victory. But, mind you, the game could not move on at all without the full complement of pieces; Kings, Queens, Bishops, Knights, Pawns; the master does not lose sight of a single one of them."#,
        r#"Elizabeth Goudge"#,
    ),
    (
        r#"Even a pawn can take down a queen."#,
        r#"Chanda Hahn, Fairest"#,
    ),
    (
        r#"The beautiful wooden board on a stand in my father’s study. The gleaming ivory pieces. The stern king. The haughty queen. The noble knight. The pious bishop. And the game itself, the way each piece contributed its individual power to the whole. It was simple. It was complex. It was savage; it was elegant. It was a dance; it was a war. It was finite and eternal. It was life."#,
        r#"Rick Yancey, The Infinite Sea"#,
    ),
    (
        r#"The mother is the most essential piece on the board, the one you must protect. Only she has the range. Only she can move in multiple directions. Once she's gone, it's a whole different game."#,
        r#"Kelly Corrigan, Glitter and Glue"#,
    ),
    (
        r#"If I am a pawn in someone else's chess game, you better believe I am going to demand an explanation before being shoved at some rook. I'll play my part, damn it, but I want the courtesy of being asked for my consent!"#,
        r#"Thomm Quackenbush, Danse Macabre (Night's Dream, #2)"#,
    ),
    (
        r#"To play for a draw, at any rate with white, is to some degree a crime against chess."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"A relationship is a game of chess."#,
        r#"Sophie Kinsella, Can You Keep a Secret?"#,
    ),
    (
        r#"When the end comes, I will meet it raging."#,
        r#"george r.r. martin, Dreamsongs, Volume II"#,
    ),
    (
        r#"I’m not forcing you to do anything. You need to make your own damn decisions . And I'm not playing this game where we ignore reality and pretend to have a normal conversation for a few hours. You need to face reality and stop turning life into a movie. I'm not a puppet in your show. This is real life and you're always trying to ignore it for some cheap fantasy version where no problems exist. That's not noble of you, okay? You're not strong. You're a weak person like the rest of us. You've just learned to excel at avoiding issues. But there are issues . Life has so many freaking issues and if you can't force your own self to face life and make decisions without someone telling you what the hell to do, you're just going to end up another chess piece moved around by others."#,
        r#"Marilyn Grey, When the City Sleeps"#,
    ),
    (
        r#"You and I should play sometime. I think you would like it,' she said." It's a game of strategy, mostly. The strong pieces are in the back row, while the weak pieces - the pawns - are all in the front, ready to take the brunt of the attack. Because of their limited movement and vulnerability, most people underestimate them and only use them to protect the more powerful pieces. But when I play I protect my pawns.'... 'They may be weak when the game begins, but their potential is remarkable. Most of the time, they'll be taken by the other side and held captive until the end of the game. But if you're careful - if you keep your eyes open and pay attention to what your oppenent is doing, if you protect your pawns and they reach the other side of the board, do you know what happens then?' I shook my head, and she smiled. "Your pawn becomes a queen."... 'Because they kept moving forward and triumphed against impossible odds, they become the most powerful piece in the game."#,
        r#"Aimee Carter, Pawn"#,
    ),
    (
        r#"In chess you might find a good move. Then you might find a better move. But take your time. Find the best move."#,
        r#"Josh Waitzkin"#,
    ),
    (
        r#"Life is like playing the game of Chess against yourself, to win you must beat the blacks of you."#,
        r#"Hassaan Ali"#,
    ),
    (
        r#"Sometimes, as in a game of chess, we must strategically regress so that we might progress toward our ultimate objective."#,
        r#"Crystal Woods, Write like no one is reading 2"#,
    ),
    (
        r#"So if you think that when you are better, it means that you can smash ahead and mate the guy, you are wrong, that is not what better means. What better means is that your position has the potential, if played correctly, to turn out well. So do not think that when you are better and when you are attacking that you can just force mate. That is not what it is about. Often the way to play best, the way to play within the position, is to maintain it."#,
        r#"Josh Waitzkin"#,
    ),
    (
        r#"To many people chess is an extreme sport. It requires a lot of thinking."#,
        r#"Ljupka Cvetanova, The New Land"#,
    ),
    (
        r#"The art of chess is akin to the art of war itself; full of strategy and cunning, yet clever placements."#,
        r#"Jennifer Megan Varnadore"#,
    ),
    (
        r#"I do not know how old I was when I learned to play chess. I could not have been older than eight, because I still have a chessboard on whose side my father inscribed, with a soldering iron, Saša Hemon 1972. I loved the board more than chess—it was one of the first things I owned. Its materiality was enchanting to me: the smell of burnt wood that lingered long after my father had branded it; the rattle of the thickly varnished pieces inside, the smacking sound they made when I put them down, the board’s hollow wooden echo. I can even recall the taste—the queen’s tip was pleasantly suckable; the pawns’ round heads, not unlike nipples, were sweet. The board is still at our place in Sarajevo, and, even if I haven’t played a game on it in decades, it is still my most cherished possession, providing incontrovertible evidence that there once lived a boy who used to be me."#,
        r#"Aleksandar Hemon, The Book of My Lives"#,
    ),
    (
        r#"Life is like playing the game of chess against yourself, to win, you must beat the blacks of you."#,
        r#"Hassaan Ali"#,
    ),
    (
        r#"I much preferred winning to thinking and I didn't like losing at all."#,
        r#"Aleksandar Hemon, The Book of My Lives"#,
    ),
    (
        r#"I have just spent the better part of a week sorting out the miasma of lunatic alibis known as your correspondence in an effort to adjust matters, that our game may be finished simply once and for all."#,
        r#"Woody Allen, Getting Even"#,
    ),
    (
        r#"Everyone wants to be wanted and if all people wait for someone else to invest in them, the world will be stuck in an eternal stalemate: nobody moves and nobody wins."#,
        r#"Laura L."#,
    ),
    (
        r#"And perhaps it was precisely because she knew nothing at all about chess that chess for her was not simply a parlor game or a pleasant pastime, but a mysterious art equal to all the recognized arts. She had never been in close contact with such people — there was no one to compare him with except those inspired eccentrics, musicians and poets whose image one knows as clearly and as vaguely as that of a Roman Emperor, an inquisitor or a comedy miser. Her memory contained a modest dimly lit gallery with a sequence of all the people who had in any way caught her fancy."#,
        r#"Vladimir Nabokov, The Luzhin Defense"#,
    ),
    (
        r#"I wanted us to share the sense that the number of wrong moves far exceeds the number of good moves, to share the frightening instability of the correct decision, to bond in being confounded."#,
        r#"Aleksandar Hemon, The Book of My Lives"#,
    ),
    (
        r#"The chemists who uphold dualism are far from being agreed among themselves; nevertheless, all of them in maintaining their opinion, rely upon the phenomena of chemical reactions. For a long time the uncertainty of this method has been pointed out: it has been shown repeatedly, that the atoms put into movement during a reaction take at that time a new arrangement, and that it is impossible to deduce the old arrangement from the new one. It is as if, in the middle of a game of chess, after the disarrangement of all the pieces, one of the players should wish, from the inspection of the new place occupied by each piece, to determine that which it originally occupied."#,
        r#"Auguste Laurent, Chemical Method, Notation, Classification, & Nomenclaturi"#,
    ),
    (
        r#"She’d been feeling pleased with a couple of moves, had thought she was gaining the upper hand. But she’d been overconfident. She’d failed to notice his stealthy approach until he turned the tables on her with a move she hadn’t expected and now she was fighting for her life."#,
        r#"Emily Arden, Lie to me"#,
    ),
    (
        r#"If you want to build a high performance organization, you’ve got to play chess, not checkers."#,
        r#"Mark Miller, Chess, Not Checkers: Elevate Your Leadership Game"#,
    ),
    (
        r#"Love was full of secrets. Love masked so many evils. Love controlled people, it lied to them, it made them believe things that weren’t true and it hid the truth from them. People said love was blind, but what they meant was that love blinded them. It made them more vulnerable than anything else could.And it felt so fucking good."#,
        r#"Stacia Kane"#,
    ),
    (
        r#"I pull on her tether all the time but it won’t sink in. I have a feeling I’m using too much magic. I can’t hold so many under my control and pull them in deep. Dean is the only one I have fully immersed. I am the puppet master. I am the only player on the board.Pacey doesn’t even know that the game has begun."-Lilith"#,
        r#"Ashley Jeffery, Released Lilith: Part 2"#,
    ),
    (
        r#"To many people chess is an extreme sport. It requires thinking."#,
        r#"Ljupka Cvetanova, The New Land"#,
    ),
    (
        r#"You can win in business by playing checkers until someone sneaks in one night after you’ve closed for the day and flips the board."#,
        r#"Mark Miller, Chess, Not Checkers: Elevate Your Leadership Game"#,
    ),
    (
        r#"Bridget did not budge, and her face was without expression. She sat, quietly defeated like a chess player who lost a career match in less than four moves."#,
        r#"Emmie White, Captive"#,
    ),
    (
        r#"I ain't...Don't know how to say it up right. Never--Fuck, Chess. Thought you was dead once before, you recall? Never felt so bad in my life, not ever. Then on the other day, thought you was gone and just....I can't do it, bein without you."#,
        r#"Stacia Kane, Chasing Magic"#,
    ),
    (
        r#"Chess teaches foresight, by having to plan ahead; vigilance, by having to keep watch over the whole chess board; caution, by having to restrain ourselves from making hasty moves; and finally, we learn from chess the greatest maxim in life - that even when everything seems to be going badly for us we should not lose heart, but always hoping for a change for the better, steadfastly continue searching for the solutions to our problems."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"Chess is everything: art, science, and sport."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"There are two types of sacrifices: correct ones, and mine."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Only Uncle Pascha ignored her. He was contemplating his chessboard. She doubted that he'd move his piece today. It had been his turn for only six months. Once, he had gone three years between moves. He preferred a leisurely game."#,
        r#"Sarah Beth Durst, Drink, Slay, Love"#,
    ),
    (
        r#"Go is to Western chess what philosophy is to double-entry accounting."#,
        r#"Trevanian, Shibumi"#,
    ),
    (
        r#"I’m not much of a chess player, but there is an aspect of the game that I find fascinating. After a while, you can almost see lines of force between the pieces. Areas of danger where it is physically impossible to move pieces into. Clouds of possibility, forbidden zones."#,
        r#"Hannu Rajaniemi, The Fractal Prince"#,
    ),
    (
        r#"I don't simply create probabilities, I guide them."#,
        r#"Lionel Suggs"#,
    ),
    (
        r#"She has gathered that the man in the grey suit whom her father called Alexander also has a student, and there will be some sort of game. "Like chess?" she asks once. "No," her father says. "Not like chess."#,
        r#"Erin Morgenstern, The Night Circus"#,
    ),
    (
        r#"Yeah, I lied and I shouldn’t have and it was lousy of me and I’m sorry. I never meant to hurt you, I never wanted that, and I wish so bad I could take it all back, okay? But we both know which one of us is lying now and it’s not me. So you call me when you want to actually talk to me and not just yell at me or tell me what a shitty person I am. I already…yeah, I already know that, okay?"#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"Thou mutters, Miss Putnam. Speak up.Like she couldn’t hear. She’d hear Chess if Chess ran to the other end of the room, covered her mouth with her hands, and whispered Fuck you, but she couldn’t hear Chess standing four feet away from her."#,
        r#"Stacia Kane, Sacrificial Magic"#,
    ),
    (
        r#"The reason I like the game chess is because each move has countless repercussions, but you're in charge of them. And it's your ability to see into the future and the effects of the decisions you've made that males you either a good or not a good chess player. It's not luck."#,
        r#"Bono, Bono: In Conversation with Michka Assayas"#,
    ),
    (
        r#"I'm playing chess while everyone else is playing checkers, that's why I always lose"#,
        r#"Josh Stern"#,
    ),
    (
        r#"Chess is the most intimate game in the world. It's like making love. By the time we finish our first slow game, I will know all his thoughts."#,
        r#"Eloisa James, Desperate Duchesses"#,
    ),
    (
        r#"Do not permit yourself to fall in love with the end-game play to the exclusion of entire games. It is well to have the whole story of how it happened; the complete play, not the denouement only. Do not embrace the rag-time and vaudeville of chess."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"Some guys live in worlds where pawns stay pawns. I'm one move from king."#,
        r#"Darnell Lamont Walker"#,
    ),
    (
        r#"I want to be an aggressive player;One who always find joy in every move."#,
        r#"Kenneth de Guzman"#,
    ),
    (
        r#"Sure. Focused. Let's totally ignore any possible other avenues and just tunnel-vision our way along. Maybe we'll get lucky and blunder into a Lamaru hangout, right?"#,
        r#"Stacia Kane, City of Ghosts"#,
    ),
    (
        r#"How gorgeous this chess set is.' Each piece was a delicate marble fantasy of medieval warfare. The paint had long ago worn off, except for faint touches of red, in the fury of the king's eyes, on the queen's lower lip, in the bishop's robe."#,
        r#"Eloisa James, Desperate Duchesses"#,
    ),
    (
        r#"Ecco, sai giocare a scacchi. Adesso devi diventare un giocatore. Ci vorrà un po' di più."#,
        r#"Guenassia Jean-Michel, Le Club des incorrigibles optimistes"#,
    ),
    (
        r#"... chess made my life more pleasuring and more rich."#,
        r#"Leonid Zorin"#,
    ),
    (
        r#"A chess player never has a heart attack in a good position."#,
        r#"Larsen"#,
    ),
    (
        r#"A chess tournament disguised as a circus."#,
        r#"John Connally"#,
    ),
    (
        r#"A computer beat me in chess, but it was no match when it came to kickboxing."#,
        r#"Philips"#,
    ),
    (
        r#"A defeatist spirit must inevitably lead to disaster."#,
        r#"Znosko Borovski"#,
    ),
    (
        r#"A draw can be obtained not only by repeating moves, but also by one weak move."#,
        r#"Tartakower"#,
    ),
    (
        r#"A gambit never becomes sheer routine as long as you fear you may lose the king and pawn ending!"#,
        r#"Larsen"#,
    ),
    (
        r#"A gambit never becomes sheer routine as long as you fear you may lose the king and pawn ending."#,
        r#"Larsen"#,
    ),
    (
        r#"A good sacrifice is one that is not necessarily sound but leaves your opponent dazed and confused."#,
        r#"Short"#,
    ),
    (
        r#"A passed pawn increases in strength as the number of pieces on the board diminishes."#,
        r#"Capablanca"#,
    ),
    (
        r#"A strong memory, concentration, imagination and a strong will is required to become a great chess player."#,
        r#"Fischer"#,
    ),
    (
        r#"A strong player requires only a few minutes of thought to get to the heart of the conflict. [...]"#,
        r#"Bronstein"#,
    ),
    (
        r#"A win is a win regardless of how badly destroyed your queenside pawns are!"#,
        r#"Nakamura"#,
    ),
    (
        r#"A woman can beat any man; it's difficult to imagine another kind of sport where woman can beat a man. That's why I like chess."#,
        r#"Kosteniuk"#,
    ),
    (
        r#"All that matters on the chessboard is good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"An impatient person plays differently than a more patient person."#,
        r#"Kramnik"#,
    ),
    (r#"Analyze! Analyze! Analyze!"#, r#"Alekhine"#),
    (
        r#"Any experienced player knows how a change in the character of the play influences your psychological mood."#,
        r#"Kasparov"#,
    ),
    (
        r#"At the board there sits a living person, with all his everyday thoughts and worries, sometimes far removed from chess."#,
        r#"Bronstein"#,
    ),
    (
        r#"Baseball is a kind of collective chess with arms and legs in full play under sunlight."#,
        r#"Jacques Barzun"#,
    ),
    (
        r#"Becoming successful at chess allows you to discover your own personality. That's what I want for the kids I teach."#,
        r#"Robovic"#,
    ),
    (
        r#"Before the endgame the gods have placed the middlegame."#,
        r#"Tarrasch"#,
    ),
    (r#"Blitz chess kills your ideas."#, r#"Fischer"#),
    (
        r#"By all means examine the games of the great chess players, but don't swallow them whole."#,
        r#"Karpov"#,
    ),
    (
        r#"Chess can be described as the movement of pieces eating one another."#,
        r#"Duchamp"#,
    ),
    (
        r#"Chess demands total concentration and a love for the game."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Chess first of all teaches you to be objective."#,
        r#"Alekhine"#,
    ),
    (
        r#"Chess holds its master in its own bonds, shaking the mind and brain so that the inner freedom of the very strongest must suffer."#,
        r#"Einstein"#,
    ),
    (
        r#"Chess is a beautiful mistress to whom we keep coming back, no matter how many times she rejects us."#,
        r#"Larsen"#,
    ),
    (r#"Chess is a beautiful mistress."#, r#"Larsen"#),
    (r#"Chess is a cold bath for the mind."#, r#"Simon"#),
    (
        r#"Chess is a gold mine, as yet hardly explored by the press."#,
        r#"LeontxoGarcía"#,
    ),
    (r#"Chess is a natural cerebral high."#, r#"Browne"#),
    (
        r#"Chess is a sea in which a gnat may drink and an elephant may bathe."#,
        r#"Indian Proverb"#,
    ),
    (r#"Chess is a test of wills."#, r#"Keres"#),
    (r#"Chess is as much a mystery as women."#, r#"Purdy"#),
    (
        r#"Chess is beautiful enough to waste your life for."#,
        r#"Ree"#,
    ),
    (
        r#"Chess is eminently and emphatically the philosopher's game."#,
        r#"Morphy"#,
    ),
    (
        r#"Chess is in its essence a game, in its form an art, and in its execution a science."#,
        r#"Baronvonder Lasa"#,
    ),
    (r#"Chess is life in miniature."#, r#"Kasparov"#),
    (r#"Chess is not for timid souls."#, r#"Steinitz"#),
    (r#"Chess is not only knowledge and logic."#, r#"Alekhine"#),
    (
        r#"Chess is so inspiring that I do not believe a good player is capable of having an evil thought during the game."#,
        r#"Steinitz"#,
    ),
    (
        r#"Chess is the game which reflects most honour on human wit."#,
        r#"Voltaire"#,
    ),
    (r#"Chess is the touchstone of intellect."#, r#"Goethe"#),
    (
        r#"Chess is useful in everyday situations like planning, concentration, combinations. You learn to win also to lose and be creative."#,
        r#"Polgar"#,
    ),
    (
        r#"Chess is war over the board. The object is to crush the opponent's mind."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Chess isn’t a game of speed, it is a game of speech through actions."#,
        r#"Selman"#,
    ),
    (
        r#"Chess mastery essentially consists of analyzing chess positions accurately."#,
        r#"Botvinnik"#,
    ),
    (r#"Chess opens and enriches your mind."#, r#"Robovic"#),
    (
        r#"Chess players look calm and collected, but in their mind they are out to castrate their opponent"#,
        r#"Robert Littell"#,
    ),
    (
        r#"Chess programs are our enemies, they destroy the romance of chess. They take away the beauty of the game. [...]"#,
        r#"Aronian"#,
    ),
    (
        r#"Chess will always be the master of us all."#,
        r#"Alekhine"#,
    ),
    (
        r#"Chess you don't learn, chess you understand."#,
        r#"Korchnoi"#,
    ),
    (r#"Chess, like love, is infectious at any age."#, r#"Flohr"#),
    (
        r#"Combinations with a queen sacrifice are among the most striking and memorable [...]"#,
        r#"Karpov"#,
    ),
    (
        r#"Computers are useless, they can only give you answers."#,
        r#"Picasso"#,
    ),
    (
        r#"Contrary to many young colleagues I do believe that it makes sense to study the classics."#,
        r#"Carlsen"#,
    ),
    (
        r#"Don't be afraid of losing, be afraid of playing a game and not learning something."#,
        r#"Heisman"#,
    ),
    (
        r#"Don't worry about your rating, work on your playing strength and your rating will follow."#,
        r#"Heisman"#,
    ),
    (
        r#"Drawn games are sometimes more scintillating than any conclusive contest."#,
        r#"Tartakower"#,
    ),
    (
        r#"During a chess competition a chess master should be the combination of a beast of prey and a monk."#,
        r#"Alekhine"#,
    ),
    (
        r#"During a chess competition a chessmaster should be a combination of a beast of prey and a monk."#,
        r#"Alekhine"#,
    ),
    (
        r#"Even the best grandmasters in the world have had to work hard to acquire the technique of rook endings."#,
        r#"Keres"#,
    ),
    (
        r#"Every top player has his own style, just as every painter has his own personal signature."#,
        r#"Kramnik"#,
    ),
    (
        r#"Everything is in a state of flux, and this includes the world of chess."#,
        r#"Botvinnik"#,
    ),
    (
        r#"Few things are as psychologically brutal as chess."#,
        r#"Kasparov"#,
    ),
    (
        r#"Fischer is completely natural. He plays no roles. He's like a child - very, very simple."#,
        r#"Rajscanyi"#,
    ),
    (
        r#"For chess, that superb, cold, infinitely satisfying anodyne to life, I feel the ardor of a lover, the humility of a disciple."#,
        r#"Wakefield"#,
    ),
    (
        r#"For me, chess is life and every game is like a new life. Every chess player gets to live many lives in one lifetime."#,
        r#"Gufeld"#,
    ),
    (
        r#"For me, chess is not a profession. It is a way of life, a passion."#,
        r#"Anand"#,
    ),
    (
        r#"For us chess players the language of artist is something natural."#,
        r#"Kramnik"#,
    ),
    (
        r#"Genius. It's a word. What does it really mean? If I win I'm a genius. If I don't, I'm not."#,
        r#"Fischer"#,
    ),
    (
        r#"Good positions don't win games, good moves do."#,
        r#"Abrahams"#,
    ),
    (
        r#"He who fears the isolated queen's pawn should give up chess."#,
        r#"Tarrasch"#,
    ),
    (
        r#"I always urge players to study composed problems and endgames."#,
        r#"Benko"#,
    ),
    (
        r#"I am not some sort of freak. I might be very good at chess but I'm just a normal person."#,
        r#"Carlsen"#,
    ),
    (
        r#"I am still a victim of chess. It has all the beauty of art - and much more."#,
        r#"Duchamp"#,
    ),
    (
        r#"I believe that the best style is a universal one: tactical and positional at the same time."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"I detest the endgame. A well-played game should be practically decided in the middlegame."#,
        r#"Janowski"#,
    ),
    (
        r#"I don't know how to play chess, but to me, life is like a game of chess."#,
        r#"Mr Brainwash"#,
    ),
    (
        r#"I get more upset at losing at other things than chess. I always get upset when I lose at Monopoly."#,
        r#"Carlsen"#,
    ),
    (
        r#"I give 98 percent of my mental energy to chess. Others give only 2 percent."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I have added these principles to the law: get the Knights into action before both bishops are developed."#,
        r#"Lasker"#,
    ),
    (
        r#"I have never in my life played the french defense, which is the dullest of all openings."#,
        r#"Steinitz"#,
    ),
    (
        r#"I learned that fighting on the chess board could also have an impact on the political climate in the country."#,
        r#"Kasparov"#,
    ),
    (
        r#"I lost the match. I blame only myself for this. There were many opportunities to win. But I missed them, no one else."#,
        r#"Karpov"#,
    ),
    (
        r#"I may be an old lion, but if someone puts his finger in my mouth, I'll bite it off!"#,
        r#"Steinitz"#,
    ),
    (
        r#"I play honestly and I play to win. If I lose, I take my medicine."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I prepare myself well. I know what I can do before I go in. I'm always confident."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I think a player constantly improves his understanding of chess with experience."#,
        r#"Kramnik"#,
    ),
    (
        r#"I think it's almost definite that the game is a draw theoretically."#,
        r#"Fischer"#,
    ),
    (
        r#"I will be again one of the best players in the world. To God, everything is possible. Most people think I won't make it [...]"#,
        r#"Mecking"#,
    ),
    (
        r#"I would never suggest to anyone that they drop school for chess. [...] your social skills need to be developed there."#,
        r#"Anand"#,
    ),
    (r#"I'd rather have a pawn than a finger."#, r#"Fine"#),
    (
        r#"I'm Alekhine, World Chess Champion, I don't need passport!"#,
        r#"Alekhine"#,
    ),
    (
        r#"If chess is a vast jungle, computers are the chainsaws in a giant environmentally insensitive logging company."#,
        r#"Short"#,
    ),
    (
        r#"If one piece stands poorly, the whole game stands poorly."#,
        r#"Tarrasch"#,
    ),
    (
        r#"If playing chess were made illegal by law, I would become an outlaw."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"If we talk about pure abilities and skills, I believe there should be no reason why women cannot play as well as men."#,
        r#"Polgar"#,
    ),
    (
        r#"In a game of chess, the fundamental law of development is struggle."#,
        r#"Suetin"#,
    ),
    (
        r#"In chess, as in life, opportunity strikes but once."#,
        r#"Bronstein"#,
    ),
    (
        r#"In chess, as in life, opportunity strikes but once."#,
        r#"Bronstein"#,
    ),
    (
        r#"In chess, just as in life, today's bliss may be tomorrow's poison."#,
        r#"Assiac"#,
    ),
    (
        r#"In life, unlike chess, the game continues after checkmate."#,
        r#"Isaac Asimov"#,
    ),
    (
        r#"In my opinion, the style of a player should not be formed under the influence of any single great master."#,
        r#"Smyslov"#,
    ),
    (
        r#"In the laboratory the gambits all test unfavorably, but the old rule wears well, that all gambits are sound over the board."#,
        r#"Napier"#,
    ),
    (
        r#"It doesn't require much for misfortune to strike in King's Gambit. One incautious move, and Black can be on the edge of the abyss."#,
        r#"Karpov"#,
    ),
    (
        r#"It is easy to play against the young players, for me they are like an open book."#,
        r#"Petrosian"#,
    ),
    (
        r#"It is not a move, even the best move, that you must seek, but a realisable plan."#,
        r#"Borovsky"#,
    ),
    (
        r#"It is not enough to be a good player... you must also play well."#,
        r#"Tarrasch"#,
    ),
    (
        r#"It is very difficult to play a single blitz game! You want to play for a long time. So I tend not to do that anymore."#,
        r#"Anand"#,
    ),
    (
        r#"It often happens that a player is so fond of his advantageous position that he is reluctant to transpose to a winning endgame."#,
        r#"Reshevsky"#,
    ),
    (
        r#"It's easy to get obsessed with chess. That's what happened with Fischer and Paul Morphy. I don't have that same obsession."#,
        r#"Carlsen"#,
    ),
    (
        r#"It's true that in chess as in politics, fund-raising and glad-handing matter."#,
        r#"Kasparov"#,
    ),
    (
        r#"Lack of patience is probably the most common reason for losing a game, or drawing games that should have been won."#,
        r#"Larsen"#,
    ),
    (
        r#"Like mortars in old war films, they [the rooks] are often ready to destroy the opponent's unsupported defences."#,
        r#"Suetin"#,
    ),
    (
        r#"Losing your objectivity almost always means losing the game."#,
        r#"Bronstein"#,
    ),
    (
        r#"Many have become chess masters - no one has become the master of chess."#,
        r#"Tarrasch"#,
    ),
    (
        r#"Memorization of variations could be even worse than playing in a tournament without looking in the books at all."#,
        r#"Botvinnik 100th birthday"#,
    ),
    (
        r#"Methodical thinking is of more use in chess than inspiration."#,
        r#"Purdy"#,
    ),
    (r#"My father said 'When in doubt, castle.'"#, r#"Vonnegut"#),
    (
        r#"My opponent is Short and the match will be short."#,
        r#"Kasparov"#,
    ),
    (
        r#"My opponents make good moves too. Sometimes I don't take these things into consideration."#,
        r#"Fischer"#,
    ),
    (
        r#"No matter how much theory progresses, how radically styles change, chess play is inconceivable without tactics."#,
        r#"Reshevsky"#,
    ),
    (
        r#"No pawn exchanges, no file-opening, no attack."#,
        r#"Nimzowitsch"#,
    ),
    (
        r#"No, I do not want to sacrifice four days for two games. My time is too valuable to do that."#,
        r#"Karpov"#,
    ),
    (
        r#"Not all artists may be chess players, but all chess players are artists."#,
        r#"Duchamp"#,
    ),
    (
        r#"Nowadays, if you are not a GM by age 14, you can forget about it."#,
        r#"Anand"#,
    ),
    (
        r#"On the chessboard lies and hypocrisy do not last long."#,
        r#"Lasker"#,
    ),
    (r#"One bad move nullifies forty good ones."#, r#"Horowitz"#),
    (
        r#"Pawns not only create the sketch for the whole painting, they are also the soil, the foundation, of any position."#,
        r#"Karpov"#,
    ),
    (
        r#"People don't quit playing because they grow old. They grow old because they quit playing."#,
        r#"Hall"#,
    ),
    (
        r#"Physical training is very important for a chess player."#,
        r#"Aronian"#,
    ),
    (
        r#"Play on both sides of the board is my favourite strategy."#,
        r#"Alekhine"#,
    ),
    (
        r#"Psychologically, you have to have confidence in yourself and this confidence should be based on fact."#,
        r#"Fischer"#,
    ),
    (
        r#"Question to Rubinstein: "Who is your opponent tonight?" Answer: "Tonight I am playing against the black pieces."#,
        r#"Rubinstein"#,
    ),
    (
        r#"Sit there for five hours? Certainly not! A player must walk about between moves, it helps his thinking."#,
        r#"Kotov"#,
    ),
    (
        r#"Some part of a mistake is always correct."#,
        r#"Tartakower"#,
    ),
    (
        r#"Some people think that if their opponent plays a beautiful game, it’s OK to lose. I don’t. You have to be merciless."#,
        r#"Carlsen"#,
    ),
    (
        r#"Spassky sits at the board with the same dead expression whether he's mating or being mated."#,
        r#"Fischer"#,
    ),
    (
        r#"Strategy requires thought, tactics require observation."#,
        r#"Euwe"#,
    ),
    (r#"Style? I have no style."#, r#"Karpov"#),
    (
        r#"The Soviet game is chess. Ours is poker. We will have to play a creative mixture of both games."#,
        r#"George Shultz"#,
    ),
    (
        r#"The ability to work hard for days on end without losing focus is a talent."#,
        r#"Kasparov"#,
    ),
    (
        r#"The amount of points that can be gained by correct endgame play is enormous, yet often underestimated by youngsters and amateurs."#,
        r#"Mednis"#,
    ),
    (
        r#"The criterion of real strength is a deep penetration into the secrets of a position."#,
        r#"Petrosian"#,
    ),
    (
        r#"The essence of Capablanca's greatness is his rare talent for avoiding all that can complicate or confuse the conflict."#,
        r#"Euwe"#,
    ),
    (
        r#"The fact that the 7 hours time control allows us to play a great deep game is not of great importance for mass-media."#,
        r#"Shirov"#,
    ),
    (
        r#"The great master places a knight at e5; checkmate follows by itself."#,
        r#"Tartakower"#,
    ),
    (r#"The hallmark of the artist is simplicity."#, r#"Evans"#),
    (r#"The king is a strong piece - use it!"#, r#"Fine"#),
    (
        r#"The laws of chess do not permit a free choice: you have to move whether you like it or not."#,
        r#"Lasker"#,
    ),
    (
        r#"The loss of my childhood was the price for becoming the youngest world champion in history."#,
        r#"Kasparov"#,
    ),
    (
        r#"The most powerful weapon in chess is to have the next move."#,
        r#"Bronstein"#,
    ),
    (
        r#"The only thing chess players have in common is chess."#,
        r#"Lodewijk"#,
    ),
    (
        r#"The power of hanging pawns is based precisely in their mobility, in their ability to create acute situations instantly."#,
        r#"Spassky"#,
    ),
    (
        r#"The process of making pieces in chess do something useful [...] has received a special name: it is called the attack."#,
        r#"Lasker"#,
    ),
    (
        r#"The rise of the soviet school to the summit of world chess is a logical result of socialist cultural development."#,
        r#"Kotov"#,
    ),
    (
        r#"The scheme of a game is played on positional lines; the decision of it, as a rule, is effected by combinations."#,
        r#"Reti"#,
    ),
    (
        r#"The system set up by FIDE [...] Insures that there will always be a russian world champion; The russians arranged it that way."#,
        r#"Fischer"#,
    ),
    (
        r#"The winning of a pawn among good players of even strength often means the winning of the game."#,
        r#"Capablanca"#,
    ),
    (
        r#"There are more adventures on a chessboard than on all the seas of the world."#,
        r#"Pierre Orlan"#,
    ),
    (
        r#"There is no better place for learning to work independently and to extend your horizon than in higher school."#,
        r#"Botvinnik"#,
    ),
    (
        r#"There is no remorse like the remorse of chess."#,
        r#"Wells"#,
    ),
    (
        r#"There just isn't enough televised chess."#,
        r#"Letterman"#,
    ),
    (
        r#"These young guys are playing checkers. I'm out there playing chess."#,
        r#"Kobe Bryant"#,
    ),
    (
        r#"They compare me with Lasker, which is an exaggerated honour. Lasker made mistakes in every game and I only in every second one!"#,
        r#"Tal"#,
    ),
    (
        r#"Time is precious when you do not have enough of it."#,
        r#"Kramnik"#,
    ),
    (
        r#"To avoid losing a piece, many a person has lost the game."#,
        r#"Tartakower"#,
    ),
    (
        r#"To be champion requires more than simply being a strong player; one has to be a strong human being as well."#,
        r#"Karpov"#,
    ),
    (
        r#"Two passed pawns advancing on the enemy pieces have brought me more than a dozen points in tournaments."#,
        r#"Bronstein"#,
    ),
    (
        r#"Two passed pawns on the sixth beat everything, up to a royal flush."#,
        r#"Rogers"#,
    ),
    (
        r#"Under no circumstances should you play fast if you have winning position. Forget the clock, use all your time and make good moves."#,
        r#"Benko"#,
    ),
    (
        r#"Unfortunately, many regard the critic as an enemy, instead of seeing him as a guide to the truth [...]."#,
        r#"Steinitz"#,
    ),
    (
        r#"Up to this point white has been following well-known analysis. But now he makes a fatal error: he begins to use his own head."#,
        r#"Tarrasch"#,
    ),
    (
        r#"Variations can be interesting, if they show the beauty of chess."#,
        r#"Bronstein"#,
    ),
    (
        r#"We cannot resist the fascination of sacrifice, since a passion for sacrifices is part of a chessplayer's nature."#,
        r#"Spielmann"#,
    ),
    (r#"We like to think."#, r#"Garry Kasparov"#),
    (
        r#"We need strong personalities and only one world champion to attract sponsors."#,
        r#"Karpov"#,
    ),
    (
        r#"Weak points or holes in the opponent's position must be occupied by pieces, not pawns."#,
        r#"Tarrasch"#,
    ),
    (
        r#"When I asked Fischer why he had not played a certain move in our game, he replied: 'Well, you laughed when I wrote it down!'"#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"When I looked out of the window during the game it was raining for a while. I love the weather to be cool and it did change my mood."#,
        r#"So"#,
    ),
    (
        r#"When the chess game is over, the pawn and the king go back to the same box."#,
        r#"Irish Proverb"#,
    ),
    (
        r#"When you absolutely don't know what to do anymore, it is time to panic."#,
        r#"Wiel"#,
    ),
    (
        r#"When you sit down to play a game you should think only about the position, but not about the opponent."#,
        r#"Capablanca"#,
    ),
    (
        r#"Whomever sees no other aim in the game than that of giving checkmate to one's opponent will never become a good chess player."#,
        r#"Euwe"#,
    ),
    (
        r#"With opposite coloured bishops the attacking side has in effect an extra piece in the shape of his bishop."#,
        r#"Botvinnik"#,
    ),
    (
        r#"Yes, I have played a blitz game once. It was on a train, in 1929."#,
        r#"Botvinnik"#,
    ),
    (
        r#"You cannot play at chess if you are kind-hearted."#,
        r#"French Proverb"#,
    ),
    (
        r#"You could say that both Fischer and Carlsen had or have the ability to let chess look simple."#,
        r#"Anand"#,
    ),
    (
        r#"You know, comrade Pachman, I don't enjoy being a Minister, I would rather play chess like you, or make a revolution in Venezuela."#,
        r#"Guevara"#,
    ),
    (
        r#"You must not let your opponent know how you feel."#,
        r#"Kotov"#,
    ),
    (
        r#"You sit at the board and suddenly your heart leaps. Your hand trembles to pick up the piece and move it."#,
        r#"Kubrick"#,
    ),
    (
        r#"Your body has to be in top condition. Your chess deteriorates as your body does. You can't separate body from mind."#,
        r#"Fischer"#,
    ),
    (
        r#"Your only task in the opening is to reach a playable middlegame."#,
        r#"Portisch"#,
    ),
    (
        r#"Youth has triumphed."#,
        r#"84-year-old Mieses after defeating 86-year-old Dirk van Foreest"#,
    ),
    (
        r#"Zugzwang is like getting trapped on a safety island in the middle of a highway when a thunderstorm starts."#,
        r#"Bisguier"#,
    ),
    (
        r#"[...] I have come to the personal conclusion that while all artists are not chess players, all chess players are artists."#,
        r#"Duchamp"#,
    ),
    (
        r#"[...] I hope my victory will make for greater interest in chess back home in the [United] States."#,
        r#"Nakamura after Tata Steel Tournament."#,
    ),
    (
        r#"[...] I rather look forward to a computer program winning the world chess championship. Humanity needs a lesson in humility."#,
        r#"Dawkins"#,
    ),
    (
        r#"[...] The ability to keep absorbing new information after many hours of study is a talent."#,
        r#"Kasparov"#,
    ),
    (
        r#"[...] because combinations are possible that chess is more than a lifeless mathematical exercise. They are the poetry of the game."#,
        r#"Fine"#,
    ),
    (
        r#"[...] my earnest desire is never to play for any stake but honor."#,
        r#"Morphy"#,
    ),
    (
        r#"[...] the queen is the most valuable and important piece and the whole outcome can depend upon how successfully she plays her role."#,
        r#"Kotov"#,
    ),
    (
        r#"[...] we learn by chess the habit of not being discouraged by present appearances in the state of our affairs. [...]"#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"[...] when you feel yourself an alien in the world, play chess. This will raise your spirits and be your counselor in war."#,
        r#"Aristotle"#,
    ),
    (
        r#"[...] you cannot make a great painter or a musician unless the gifts of painting or music are innate in the person."#,
        r#"Alekhine"#,
    ),
    (
        r#"Bishops move diagonally. That's why they often turn up where the kings don't expect them to be."#,
        r#"Pratchett"#,
    ),
    (
        r#"Chess constitutes an effective means in order to education and training of the intellect of man."#,
        r#"Che Guevara"#,
    ),
    (r#"Chess is so deep, I simply feel lost."#, r#"Kramnik"#),
    (
        r#"Chess is the only game greater than its players."#,
        r#"Tim Rice"#,
    ),
    (
        r#"Discovered check is the dive-bomber of the chessboard."#,
        r#"Fine"#,
    ),
    (
        r#"Turning chess into poker and hoping for a 'bluff' is not one of my convictions."#,
        r#"Petrosian"#,
    ),
    (
        r#"A chess problem is an exercise in pure mathematics."#,
        r#"Godfrey Hardy"#,
    ),
    (r#"A good player is always lucky."#, r#"Capablanca"#),
    (r#"A player surprised is half beaten."#, r#"Proverb"#),
    (r#"All my games are real."#, r#"Fischer"#),
    (r#"Best by test: 1. e4."#, r#"Fischer"#),
    (
        r#"Chess is no whit inferior to the violin, and we have a large number of professional violinists."#,
        r#"Botvinnik"#,
    ),
    (
        r#"Chess is played with the mind and not with the hands!"#,
        r#"Kahn"#,
    ),
    (
        r#"Chess is ruthless: you've got to be prepared to kill people."#,
        r#"Nigel Short"#,
    ),
    (
        r#"Chess is thirty to forty percent psychology. You don't have this when you play a computer. I can't confuse it."#,
        r#"Polgar"#,
    ),
    (
        r#"Daring ideas are like chess men moved forward. They may be beaten, but they may start a winning game."#,
        r#"Goethe"#,
    ),
    (
        r#"Daring ideas are like chessmen moving forward; they may be beaten, but they may start a winning game."#,
        r#"Goethe"#,
    ),
    (
        r#"Different people feel differently about resigning."#,
        r#"Fischer"#,
    ),
    (r#"I am still a victim of chess."#, r#"Duchamp"#),
    (
        r#"I began to succeed in decisive games. Perhaps because I realized a very simple truth: not only was I worried, but also my opponent."#,
        r#"Tal"#,
    ),
    (
        r#"I have always a slight feeling of pity for the man who has no knowledge of chess."#,
        r#"Tarrasch"#,
    ),
    (
        r#"I like the moment when I break a man's ego."#,
        r#"Fischer"#,
    ),
    (r#"I'd rather have a Pawn than a finger."#, r#"Fine"#),
    (
        r#"If your opponent offers you a draw, try to work out why he thinks he's worse off"#,
        r#"Nigel Short"#,
    ),
    (
        r#"In chess, as it is played by masters, chance is practically eliminated."#,
        r#"Lasker"#,
    ),
    (
        r#"In chess, just as in life, today's bliss may be tomorrow's poison."#,
        r#"Assaic"#,
    ),
    (
        r#"It’s just you and your opponent at the board and you are trying to prove something"#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Of chess it has been said that life is not long enough for it, but that is the fault of life, not chess"#,
        r#"Napier"#,
    ),
    (
        r#"Openings teach you openings. Endgames teach you chess!"#,
        r#"Gerzadowicz"#,
    ),
    (r#"Tactics flow from a superior position."#, r#"Fischer"#),
    (
        r#"The battle for the ultimate truth will never be won. And that's why chess is so fascinating."#,
        r#"Kmoch"#,
    ),
    (
        r#"The beauty of a move lies not in it's appearance but in the thought behind it."#,
        r#"Nimzovich"#,
    ),
    (
        r#"The defensive power of a pinned piece is only imaginary."#,
        r#"Nimzowitsch"#,
    ),
    (r#"The hardest game to win is a won game."#, r#"Lasker"#),
    (
        r#"The laws of chess are as beautiful as those governing the universe - and as deadly."#,
        r#"Neville"#,
    ),
    (
        r#"The laws of chess do not permit a free choice: you have to move whether you like it or not."#,
        r#"Lasker"#,
    ),
    (r#"The pin is mightier than the sword."#, r#"Reinfeld"#),
    (
        r#"Those who say they understand chess, understand nothing."#,
        r#"Hubner"#,
    ),
    (
        r#"We can't resist the fascination of sacrifice, since a passion for sacrifices is part of a chessplayer’s nature."#,
        r#"Spielmann"#,
    ),
    (
        r#"We must make sure that chess will not be like a dead language, very interesting, but for a very small group."#,
        r#"Sytze Faber"#,
    ),
    (
        r#"What would chess be without silly mistakes?"#,
        r#"Richter"#,
    ),
    (
        r#"Winning isn't everything... but losing is nothing."#,
        r#"Mednis"#,
    ),
    (
        r#"You know, comrade Pachman, I don't enjoy being a minister. I would rather play chess like you"#,
        r#"Che Guevara"#,
    ),
    (
        r#"Your body has to be in top condition. Your chess deteriorates as your body does. You can't separate body from mind"#,
        r#"Bobby Fischer"#,
    ),
    (
        r#""Chess beauty" is in the geometry of the game, and the struggle of plans."#,
        r#"Levon Aronian"#,
    ),
    (
        r#"'King of chess'"#,
        r#"… what Emanuel Lasker whispered to his wife before he died ..."#,
    ),
    (
        r#"... My achievements in the field of chess are the result of immense hard work in studying theory ..."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"... les Pions. Il sont l’âme des Echecs ... ... the Pawns: They are the very Life of this Game."#,
        r#"Francois-Andre Danican Philidor"#,
    ),
    (
        r#"A Queen's sacrifice, even when fairly obvious, always rejoices the heart of the chess-lover."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"A bad day of Chess is better than any good day at work."#,
        r#"Anonymous"#,
    ),
    (
        r#"A bad plan is better than none at all."#,
        r#"Frank Marshall"#,
    ),
    (
        r#"A bishop is better than a knight in all but blocked pawn positions."#,
        r#"?"#,
    ),
    (
        r#"A chess player never has a heart attack in a good position."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"A considerable role in the forming of my style was played by an early attraction to study composition."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"A defeatist spirit must inevitably lead to disaster."#,
        r#"Eugene Znosko-Borovski"#,
    ),
    (
        r#"A draw can be obtained not only by repeating moves, but also by one weak move."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"A gambit never becomes sheer routine as long as you fear you may lose the king and pawn ending!"#,
        r#"Bent Larsen"#,
    ),
    (
        r#"A great chess player always has a very good memory."#,
        r#"Leonid Shamkovich"#,
    ),
    (
        r#"A man that will take back a move at Chess will pick a pocket."#,
        r#"Richard Fenton"#,
    ),
    (
        r#"A man that will take back a move at chess will pick a pocket."#,
        r#"Richard Fenton"#,
    ),
    (
        r#"A passed pawn increase in strength as the number of pieces on the board diminishes."#,
        r#"Jose Capablanca"#,
    ),
    (r#"A pawn is a pawn.- Unknown"#, r#"?"#),
    (
        r#"A pawn, when separated from his fellows, will seldom or never make a fortune."#,
        r#"Francois-Andre Danican Philidor"#,
    ),
    (
        r#"A plan is made for a few moves only, not for the whole game."#,
        r#"Rueben Fine"#,
    ),
    (r#"A player surprised is half beaten."#, r#"Proverb"#),
    (
        r#"A queen and a rook will always checkmate a naked king."#,
        r#"?"#,
    ),
    (
        r#"A sacrifice is best refuted by accepting it."#,
        r#"Wilhelm Steinitz"#,
    ),
    (r#"A sustained initiative is worth some material."#, r#"?"#),
    (
        r#"A win by an unsound combination, however showy, fills me with artistic horror."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"A wing attack is best met by a counterattack in the center."#,
        r#"?"#,
    ),
    (
        r#"All I want to do, ever, is just play Chess."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"All conceptions in the game of chess have a geometrical basis."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"All that matters on the chessboard is good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Along with my retirement from chess analytical work seems to have gone too."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Amberley excelled at chess -- a mark, Watson, of a scheming mind."#,
        r#"Sir Arthur Conan Doyle"#,
    ),
    (r#"An extra pawn is worth a little trouble."#, r#"?"#),
    (
        r#"An innovation need not be especially ingenious, but it must be well worked out."#,
        r#"Paul Keres"#,
    ),
    (
        r#"An isolated pawn spreads gloom all over the chessboard."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"And whan I saw my fers (Queen) awaye,/ Allas! I kouthe no lenger playe."#,
        r#"the knight in Chaucer's 'Book of the Duchess' (1369)"#,
    ),
    (
        r#"Any material change in a position must come about by mate, a capture, or a Pawn promotion."#,
        r#"Cecil Purdy"#,
    ),
    (
        r#"Apart from direct mistakes, there is nothing more ruinous than routine play, the aim of which is mechanical development."#,
        r#"Alexei Suetin"#,
    ),
    (
        r#"As a chess player one has to be able to control one’s feelings, one has to be as cold as a machine."#,
        r#"Levon Aronian"#,
    ),
    (
        r#"As a rule, pawn endings have a forced character, and they can be worked out conclusively."#,
        r#"Mark Dvoretsky"#,
    ),
    (r#"Avoid doubled, isolated, and backward pawns."#, r#"?"#),
    (
        r#"Avoid the crowd. Do your own thinking independently. Be the Chess player, not the Chess piece."#,
        r#"Ralph Charell"#,
    ),
    (
        r#"Avoidance of mistakes is the beginning, as it is the end, of mastery in chess."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"Before beginning a wing attack,make sure your center is secure."#,
        r#"?"#,
    ),
    (
        r#"Before the endgame the gods have placed the middlegame."#,
        r#"Siegbert Tarrasch"#,
    ),
    (r#"Best by test (on 1. e4)"#, r#"Bobby Fischer"#),
    (
        r#"Black is now in desperate need of a good idea. Or, to put it standard chess notation, +-."#,
        r#"Mark Dvoretsky"#,
    ),
    (
        r#"Black's d5-square is too weak. (on the Dragon variation)"#,
        r#"Ulf Andersson"#,
    ),
    (
        r#"Blessed be the memory of him who gave the world this immortal game."#,
        r#"A. G. Gardiner"#,
    ),
    (r#"Blitz chess kills your ideas."#, r#"Bobby Fischer"#),
    (r#"Blockade passed pawns with the king."#, r#"?"#),
    (
        r#"Break a bind in order to free your pieces, even if it costs a pawn."#,
        r#"?"#,
    ),
    (
        r#"But you see when I play a game of Bobby, there is no style. Bobby played perfectly. And perfection has no style."#,
        r#"Miguel Najdorf"#,
    ),
    (
        r#"Campo, as everyone in chess recognized, had a 2600 rating as a politician. (on F.I.D.E. President Florencio Campomanes)"#,
        r#"Raymond Keene"#,
    ),
    (
        r#"Can you imagine the relief it gives a mother when her child amuses herself quietly for hours on end?"#,
        r#"Klara Polgar"#,
    ),
    (
        r#"Capture of the adverse King is the ultimate but not the first object of the game."#,
        r#"Wilhelm Steinitz"#,
    ),
    (r#"Castle early and often."#, r#"?"#),
    (r#"Centralize the action of all your pieces."#, r#"?"#),
    (r#"Centralize your pieces to make them powerful."#, r#"?"#),
    (r#"Checkers is for tramps."#, r#"Paul Morphy"#),
    (r#"Chess demands total concentration."#, r#"Bobby Fischer"#),
    (
        r#"Chess first of all teaches you to be objective."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Chess holds its master in its own bonds, shakling the mind and brain so that the inner freedom of the very strongest must suffer."#,
        r#"Albert Einstein"#,
    ),
    (r#"Chess is 99% tactics."#, r#"Rudolph Teichmann"#),
    (
        r#"Chess is a contest between two men which lends itself particularly to the conflicts surrounding aggression."#,
        r#"Rueben Fine"#,
    ),
    (
        r#"Chess is a cure for headaches."#,
        r#"John Maynard Keynes"#,
    ),
    (
        r#"Chess is a fighting game which is purely intellectual and includes chance."#,
        r#"Richard Reti"#,
    ),
    (
        r#"Chess is a game which reflects most honor on human wit."#,
        r#"Voltaire"#,
    ),
    (
        r#"Chess is a good mistress but a bad master."#,
        r#"Gerald Abrahams"#,
    ),
    (
        r#"Chess is a matter of delicate judgement, knowing when to punch and how to duck."#,
        r#"Bobby Fischer"#,
    ),
    (r#"Chess is a meritocracy."#, r#"Lawrence Day"#),
    (
        r#"Chess is a sea in which a gnat may drink and an elephant may bathe."#,
        r#"Hindu proverb"#,
    ),
    (r#"Chess is a sport. A violent sport."#, r#"Marcel Duchamp"#),
    (
        r#"Chess is a sport. The main object in the game of chess remains the achievement of victory."#,
        r#"Max Euwe"#,
    ),
    (
        r#"Chess is as elaborate a waste of human intelligence as you can find outside an advertising agency."#,
        r#"Raymond Chandler"#,
    ),
    (
        r#"Chess is beautiful enough to waste your life for."#,
        r#"Hans Ree"#,
    ),
    (
        r#"Chess is changing. I hope chess is getting more popular, more spectacular."#,
        r#"Alexandra Kosteniuk"#,
    ),
    (r#"Chess is imagination."#, r#"David Bronstein"#),
    (
        r#"Chess is intellectual gymnastics."#,
        r#"Wilhelm Steinitz"#,
    ),
    (r#"Chess is life."#, r#"Bobby Fischer"#),
    (r#"Chess is mental torture."#, r#"Garry Kasparov"#),
    (
        r#"Chess is my life, but my life is not chess."#,
        r#"Anatoly Karpov"#,
    ),
    (r#"Chess is my life."#, r#"Victor Kortchnoi"#),
    (r#"Chess is not for the timid."#, r#"Irving Chernev"#),
    (r#"Chess is not for timid souls."#, r#"Wilhelm Steinitz"#),
    (
        r#"Chess is not only knowledge and logic."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Chess is not relaxing; it's stressful even if you win."#,
        r#"Jennifer Shahade"#,
    ),
    (r#"Chess is one long regret."#, r#"Stephen Leacock"#),
    (
        r#"Chess is only a recreation and not an occupation."#,
        r#"Vladimir Lenin"#,
    ),
    (
        r#"Chess is ruthless: you've got to be prepared to kill people."#,
        r#"Nigel Short"#,
    ),
    (
        r#"Chess is so inspiring that I do not believe a good player is capable of having an evil thought during the game."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"Chess is so inspiring that I do not believe a good player is capable of haviong an evil thought during the game."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"Chess is such a ridiculously silly game sometimes..."#,
        r#"Hikaru Nakamura"#,
    ),
    (r#"Chess is the art of analysis."#, r#"Mikhail Botvinnik"#),
    (
        r#"Chess is the art which expresses the science of logic."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Chess is the gymnasium of the mind."#,
        r#"Adolf Anderssen"#,
    ),
    (r#"Chess is the gymnasium of the mind."#, r#"Blaise Pascal"#),
    (
        r#"Chess is thirty to forty percent psychology. You don't have this when you play a computer. I can't confuse it."#,
        r#"Judit Polgar"#,
    ),
    (
        r#"Chess is thriving. There are ever less round robin tournaments and ever more World Champions."#,
        r#"Robert Huebner"#,
    ),
    (
        r#"Chess is too much to be merely a game but too little to be anything more."#,
        r#"Unknown"#,
    ),
    (
        r#"Chess is war over the board. The object is to crush the opponents mind."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Chess makes man wiser and clear-sighted."#,
        r#"Vladimir Putin"#,
    ),
    (
        r#"Chess masters as well as chess computers deserve less reverence than the public accords them."#,
        r#"Eliot Hearst"#,
    ),
    (
        r#"Chess never has been and never can be aught but a recreation."#,
        r#"Paul Morphy"#,
    ),
    (
        r#"Chess, like love, is infectious at any age."#,
        r#"Salo Flohr"#,
    ),
    (
        r#"Chess, like love, like music, has the power to make man happy."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Combinations with a queen sacrifice are among the most striking and memorable ..."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"Concentrate on material gains. Whatever your opponent gives you take, unless you see a good reason not to."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Contrary to many young colleagues I do believe that it makes sense to study the classics."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"Daring ideas are like Chess men moved forward. They may be beaten, but they may start a winning game."#,
        r#"Johann Wolfgang von Goethe"#,
    ),
    (
        r#"Dazzling combinations are for the many, shifting wood is for the few."#,
        r#"Georg Kieninger"#,
    ),
    (
        r#"Develop a new piece with each move in the opening."#,
        r#"?"#,
    ),
    (r#"Develop knights before bishops."#, r#"?"#),
    (r#"Develop with threats."#, r#"?"#),
    (
        r#"Discovered check is the dive bomber of the chess board."#,
        r#"Rueben Fine"#,
    ),
    (
        r#"Do not move pawns in front of your castled king."#,
        r#"?"#,
    ),
    (
        r#"Do not pin your opponent's f3- or f6-knghit to his queen with your bishop until after he's castled."#,
        r#"?"#,
    ),
    (
        r#"Do not place your pawns on the color of your bishop."#,
        r#"?"#,
    ),
    (r#"Don't attack unless you have the superior game."#, r#"?"#),
    (r#"Don't bring your queen out too early."#, r#"?"#),
    (
        r#"Don't expose your king while the enemy queen is still on the board."#,
        r#"?"#,
    ),
    (
        r#"Don't move the same piece twice if you can help it."#,
        r#"?"#,
    ),
    (
        r#"Don't move the same piece twice in the opening if you can help it."#,
        r#"?"#,
    ),
    (
        r#"Don't sacrifice without a clear and adequate reason."#,
        r#"?"#,
    ),
    (
        r#"Don't worry about your rating, work on your playing strength and your rating will follow."#,
        r#"Dan Heisman"#,
    ),
    (
        r#"Drawing general conclusions about your main weaknesses can provide a great stimulus to further growth."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Drawn games are sometimes more scintillating than any conclusive contest."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"During a chess tournament a master must envisage himself as a cross between an ascetic monk and a beast of prey."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"During a game I cease to think about the result as I become so enthralled by what’s happening on the board."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"Even a poor plan is better than no plan at all."#,
        r#"Mikhail Chigorin"#,
    ),
    (
        r#"Even in the heat of a middlegame battle the master still has to bear in mind the outlines of a possible future ending."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Even the best grandmasters in the world have had to work hard to acquire the technique of rook endings."#,
        r#"Paul Keres"#,
    ),
    (
        r#"Even the laziest king flees wildly in the face of double check."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"Every Chess master was once a beginner."#,
        r#"Irving Chernev"#,
    ),
    (r#"Every move creates a weakness."#, r#"Siegbert Tarrasch"#),
    (
        r#"Every move is an opportunity to interfere with your opponent's plans, or to further your own plans."#,
        r#"?"#,
    ),
    (
        r#"Every move the endgame is of the utmost importance because you are closer to the moment of truth."#,
        r#"Pal Benko"#,
    ),
    (
        r#"Every player should be able to make use of such a subtle strategic weapon as a pawn sacrifice."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"Every time I win a tournament I have to think that there is something wrong with modern chess."#,
        r#"Viktor Korchnoi"#,
    ),
    (
        r#"Everything is in a state of flux, and this includes the world of Chess."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Everything is in a state of flux, and this includes the world of chess."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Few things are as psychologically brutal as chess."#,
        r#"Garry Kasparov"#,
    ),
    (r#"Fewer pawn islands means a healthier position."#, r#"?"#),
    (
        r#"First restrain, next blockade, lastly destroy."#,
        r#"Aron Nimzowitsch"#,
    ),
    (
        r#"First-class players lose to second-class players because second-class players sometimes play a first-class game."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Fischer is Fischer, but a knight is a knight!"#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"For a game it is too serious, for seriousness too much of a game."#,
        r#"Moses Mendelssohn 1729-86"#,
    ),
    (
        r#"For me art and chess are closely related, both are forms in which the self finds beauty and expression."#,
        r#"Vladimir Kramnik"#,
    ),
    (
        r#"GM Naiditsch reckoned that me playing the King's Indian against Anand was something akin to a samurai running at a machine gun with a sword."#,
        r#"Hikaru Nakamura"#,
    ),
    (
        r#"Genius. It's a word. What does it really mean? If I win I'm a genius. If I don't, I'm not."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Good players develop a tactical instinct, a sense of what is possible or likely and what is not worth calculating."#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"Good positions don’t win games, good moves do."#,
        r#"Gerald Abrahams"#,
    ),
    (r#"Haste, the great enemy."#, r#"Eugene Znosko-Borowski"#),
    (
        r#"He who fears an isolated Queen’s Pawn should give up Chess."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"He who fears the isolated queen's pawn should give up chess."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"He who takes the Queen’s Knight’s Pawn will sleep in the streets."#,
        r#"Anonymous"#,
    ),
    (
        r#"Help your pieces so they can help you."#,
        r#"Paul Morphy"#,
    ),
    (
        r#"How come the little things bother you when you are in a bad position? They don't bother you in good positions."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"However hopeless the situation appears to be there yet always exists the possibility of putting up a stubborn resistance."#,
        r#"Paul Keres"#,
    ),
    (
        r#"I always urge players to study composed problems and endgames."#,
        r#"Pal Benko"#,
    ),
    (
        r#"I believe that the best style is a universal one, tactical and positional at the same time ..."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"I cannot reply so many tweets at a time. If you want to talk with me, please reply in turn :-)"#,
        r#"chess_chat"#,
    ),
    (
        r#"I could give any woman in the world a piece and a move; to Gaprindashvili even, a knight."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I definitely miss the rush from wiping out an opponent."#,
        r#"Bruce Pandolfini"#,
    ),
    (
        r#"I don't believe in psychology. I believe in good moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I don't have any solution, but I certainly admire the problem."#,
        r#"Ashleigh Brilliant"#,
    ),
    (
        r#"I failed to make the chess team because of my height."#,
        r#"Woody Allen"#,
    ),
    (
        r#"I feel as if I were a piece in a game of chess, when my opponent says of it: That piece cannot be moved."#,
        r#"Soren Kierkegaard"#,
    ),
    (
        r#"I give 98 percent of my mental energy to Chess. Others give only 2 percent."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I guess a certain amount of temperament is expected of Chess geniuses."#,
        r#"Ron Gross"#,
    ),
    (
        r#"I have always disliked the fierce competitive spirit embodied in that highly intellectual game."#,
        r#"Albert Einstein"#,
    ),
    (
        r#"I have frequently stated that I regard chess as an art form, where creativity prevails over other factors."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"I have never in my life played the French Defence, which is the dullest of all openings."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"I have not given any drawn or lost games, because I thought them inadequate to the purpose of the book."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"I know people who have all the will in the world, but still can't play good chess."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I know that with perfect play, God versus God, Fritz versus Fritz, chess is a draw ..."#,
        r#"Nigel Short"#,
    ),
    (
        r#"I know the knights walke in this game too well, Hee maye skip over mee, and where am I then?"#,
        r#"Thomas Middleton (1624)"#,
    ),
    (
        r#"I learned that fighting on the chess board could also have an impact on the political climate in the country."#,
        r#"Garry Kasparov"#,
    ),
    (r#"I leave this to you."#, r#"Bobby Fischer"#),
    (
        r#"I like 1.e4 very much, but my results are better with 1.d4."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"I like the moment when I break a man's ego."#,
        r#"Bobby Fischer"#,
    ),
    (r#"I like to make them squirm."#, r#"Bobby Fischer"#),
    (
        r#"I like when the game turns into a contest of ideas and not a battle between home analysis."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"I look one move ahead... the best!"#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"I love all positions. Give me a difficult positional game, I will play it. But totally won positions, I cannot stand them"#,
        r#"Jan Hein Donner"#,
    ),
    (
        r#"I often play a move I know how to refute."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"I play way too much blitz chess. It rots the brain just as surely as alcohol."#,
        r#"Nigel Short"#,
    ),
    (
        r#"I see my own style as being a symbiosis of the styles of Alekhine, Tal and Fischer."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"I simply fought the way I always do."#,
        r#"Teimour Radjabov"#,
    ),
    (
        r#"I think a player constantly improves his understanding of chess with experience."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"I think it's almost definite that the game is a draw theoretically."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"I won't play with you anymore. You have insulted my friend! (when an opponent cursed himself for a blunder)"#,
        r#"Miguel Najdorf"#,
    ),
    (r#"I'd rather have a pawn than a finger."#, r#"Rueben Fine"#),
    (
        r#"I'm not in the habit of calculating how many points I've won or lost after each game."#,
        r#"Sergey Karjakin"#,
    ),
    (
        r#"If I win a tournament, I win it by myself. I do the playing. Nobody helps me."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"If a ruler does not understand Chess, how can he rule over a kingdom?"#,
        r#"King Khusros II AD 580-628"#,
    ),
    (
        r#"If chess is a vast jungle, computers are the chainsaws in a giant environmentally insensitive logging company."#,
        r#"Nigel Short"#,
    ),
    (
        r#"If cunning alone were needed to excel, women would be the best chess players."#,
        r#"Adolf Albin"#,
    ),
    (
        r#"If people will be interested in me, they will be interested in chess also."#,
        r#"Alexandra Kosteniuk"#,
    ),
    (
        r#"If the defender is forced to give up the center, then every possible attack follows almost of itself."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"If the media didn't know I played chess, there'd be no angle on me at all."#,
        r#"Lennox Lewis"#,
    ),
    (
        r#"If we talk about pure abilities and skills, I believe there should be no reason why women cannot play as well as men."#,
        r#"Susan Polgar"#,
    ),
    (
        r#"If you are free, Let's talk about chess with me!"#,
        r#"chess_chat"#,
    ),
    (
        r#"If you are one or two pawns ahead, exchange pieces but not pawns."#,
        r#"?"#,
    ),
    (
        r#"If you are one or two pawns behind, exchange pawns but not pieces."#,
        r#"?"#,
    ),
    (
        r#"If you are only one pawn ahead, trade pieces but not pawns."#,
        r#"?"#,
    ),
    (
        r#"If you control more than half of the squares on the board,you have an advantage."#,
        r#"?"#,
    ),
    (
        r#"If you don't win, it's not a great tragedy"#,
        r#"the worst that happens is that you lose a game. -Bobby Fischer"#,
    ),
    (
        r#"If you have an advantage, do not leave all the pawns on one side."#,
        r#"?"#,
    ),
    (
        r#"If you're too busy to play chess... you're too busy."#,
        r#"Unknown"#,
    ),
    (
        r#"If your opponent offers you a draw, try to work out why he thinks he's worse off."#,
        r#"Nigel Short"#,
    ),
    (
        r#"In Chess, as it is played by masters, chance is practically eliminated."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"In Chess, just as in life, today’s bliss may be tomorrow’s poison."#,
        r#"Assaic"#,
    ),
    (
        r#"In blitz, the knight is stronger than the bishop."#,
        r#"Vlastmil Hort"#,
    ),
    (
        r#"In chess, as in life, a man is his own most dangerous opponent."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"In chess, at least, the brave inherit the earth."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"In chess, there is only one mistake: over-estimation of your opponent. All else is either bad luck or weakness."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"In complicated positions, Bobby Fischer hardly had to be afraid of anybody."#,
        r#"Paul Keres"#,
    ),
    (
        r#"In cramped positions, free yourself by exchanging."#,
        r#"?"#,
    ),
    (
        r#"In general chess players do have a tendency towards autism, and for some it's bordering on a mental disorder."#,
        r#"Alexander Grischuk"#,
    ),
    (
        r#"In my opinion, the style of a player should not be formed under the influence of any single great master."#,
        r#"Vasily Smyslov"#,
    ),
    (
        r#"In positions with an unusual disparty in material,the initiative is often the deciding factor."#,
        r#"?"#,
    ),
    (
        r#"In the Soviets' view, chess was not merely an art or a science or even a sport; it was what it had been invented to simulate: war."#,
        r#"Pal Benko"#,
    ),
    (
        r#"In the ending the king is a powerful piece for assisting his own pawns, or stopping the adverse pawns."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"In the middlegame, the king is merely an extra, but in the endgame, he is one of the star actors."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"Intellectual activity is perhaps the greatest pleasure of life; chess is one of the forms of intellectual activity."#,
        r#"Siegbert Tarrasch"#,
    ),
    (r#"Is Bobby Fischer quite sane?"#, r#"Salo Flohr"#),
    (
        r#"It cannot be too greatly emphasized that the most important role in pawn endings is played by the king."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"It has always been recognized that chess is an art, and its best practitioners have been described as artists."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"It is a curse upon man. There is no happiness in chess."#,
        r#"H.G. Wells"#,
    ),
    (
        r#"It is always better to sacrifice your opponents' men."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"It is dangerous to maintain equality at the cost of placing the pieces passively."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"It is easy to play against the young players, for me they are like an open book."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"It is impossible to ignore a highly important factor of the chess struggle -- psychology."#,
        r#"Yuri Averbakh"#,
    ),
    (
        r#"It is no time to be playing Chess when the house is on fire."#,
        r#"Italian Proverb"#,
    ),
    (
        r#"It is not a move, even the best move that you must seek, but a realizable plan."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"It is not enough to be a good player... you must also play well."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"It is peculiar but a fact nevertheless, that the gamblers in chess have enthusiastic followers."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"It is rightly said that the most difficult thing in chess is winning a won position."#,
        r#"Vladimir Kramnik"#,
    ),
    (
        r#"It is said that an ounce of common sense can outweigh a ton of 'variations'."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"It is very difficult to play a single blitz game! You want to play for a long time. So I tend not to do that anymore."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"It's a shame to be the face of chess and to play chess badly."#,
        r#"Alexandra Kosteniuk"#,
    ),
    (
        r#"It's little quirks like this that could make life difficult for a chess machine."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"It's quite difficult for me to imagine my life without chess."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"It's true that in chess as in politics, fund-raising and glad-handing matter."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"It’s always better to sacrifice your opponent’s men."#,
        r#"Savielly Tartakover"#,
    ),
    (r#"I’d rather have a Pawn than a finger."#, r#"Reuben Fine"#),
    (
        r#"I’m changing and becoming more aggressive."#,
        r#"Teimour Radjabov"#,
    ),
    (
        r#"Jacob, how do you actually become a Grand Wizard? (question posed to GM (not GW) Jacob Aagaard)"#,
        r#"?"#,
    ),
    (
        r#"Leave the pawns alone,except for center pawns and passed pawns."#,
        r#"?"#,
    ),
    (
        r#"Let the perfectionist play postal."#,
        r#"Yasser Seirawan"#,
    ),
    (
        r#"Life is a kind of Chess, with struggle, competition, good and ill events."#,
        r#"Benjamin Franklin"#,
    ),
    (
        r#"Life is like a game of chess, changing with each move."#,
        r#"Chinese proverb"#,
    ),
    (r#"Life's too short for chess."#, r#"Henry James Byron"#),
    (
        r#"Losing is fun when you decide to fall asleep and blunder right before time control!"#,
        r#"Hikaru Nakamura"#,
    ),
    (r#"Make as few pawn moves as possible."#, r#"?"#),
    (
        r#"Many men, many styles; what is chess style but the intangible expression of the will to win."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"Memorization of variations could be even worse than playing in a tournament without looking in the books at all."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Methodical thinking is of more use in chess than inspiration."#,
        r#"Cecil Purdy"#,
    ),
    (
        r#"Mistakes are a consequence of tension."#,
        r#"Magnus Carlsen"#,
    ),
    (
        r#"Mistrust is the most necessary characteristic of the chess player."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Modern chess is too concerned with things like pawn structure. Forget it, checkmate ends the game."#,
        r#"Nigel Short"#,
    ),
    (
        r#"Morphy was probably the greatest genius of them all."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"Most strong players are completely self-centered…. They are blind to how other people feel or else simply don't care."#,
        r#"Joel Lautier"#,
    ),
    (
        r#"My favourite victory is when it is not even clear where my opponent made a mistake."#,
        r#"Peter Leko"#,
    ),
    (
        r#"My life has been determined by the move e2-e1N."#,
        r#"Johan Barendregt"#,
    ),
    (
        r#"My opponent is Short and the match will be short."#,
        r#"Kasparov's quip before his 1993 PCA World Championship match with Nigel Short."#,
    ),
    (
        r#"My opponent left a glass of whisky 'en prise' and I took it 'en passant'."#,
        r#"Henry Blackburne"#,
    ),
    (
        r#"My opponents make good moves too. Sometimes I don't take these things into consideration."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"My sister bought me a set at a candy store and taught me the moves."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"My theory of a key-move was always to make it just the reverse of what a player in 999 out of 1000 could look for."#,
        r#"Sam Loyd"#,
    ),
    (
        r#"Never copy yourself, always copy someone else."#,
        r#"Pablo Picasso"#,
    ),
    (r#"Never make a good move too soon."#, r#"James Mason"#),
    (
        r#"No Chess Grandmaster is normal; they only differ in the extent of their madness."#,
        r#"Victor Kortchnoi"#,
    ),
    (
        r#"No fool can play chess, and only fools do."#,
        r#"German proverb"#,
    ),
    (
        r#"No matter how much theory progresses, how radically styles change, chess play is inconceivable without tactics."#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"No one ever won a game by resigning."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"No one ever won a game by resigning."#,
        r#"Saviely Tartakower"#,
    ),
    (
        r#"No one has ever played these endgames with such elegant ease as Capablanca."#,
        r#"Richard Reti"#,
    ),
    (
        r#"No price is too great for the scalp of the enemy King."#,
        r#"Koblentz"#,
    ),
    (
        r#"Not all artists are Chess players, but all Chess players are artists."#,
        r#"Marcel Duchamp"#,
    ),
    (r#"Not all rook endings are drawn!"#, r#"?"#),
    (
        r#"Nothing excites jaded Grandmasters more than a theoretical novelty."#,
        r#"Dominic Lawson"#,
    ),
    (
        r#"Now I have the pawn and the compensation."#,
        r#"Roman Dzindzichashvili (during a blitz game)"#,
    ),
    (
        r#"Nowadays tournaments are for nurseries. Look at those kiddies."#,
        r#"Miguel Najdorf (pointing to Fischer, Spassky, and Larsen in 1966)"#,
    ),
    (
        r#"Nowadays, when you're not a grandmaster at 14, you can forget about it."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"Of Chess it has been said that life is not long enough for it, but that is the fault of life, not chess."#,
        r#"William Ewart Napier"#,
    ),
    (
        r#"Often, in the Ruy Lopez, one must be patient, wait and carry on a lengthy and wearisome struggle."#,
        r#"Boris Spassky"#,
    ),
    (
        r#"On the chessboard lies and hypocrisy do not last long."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"One of the objectives of opening play is to try to surprise your opponent."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Only play into a variation in which your opponent is strong if you have your own personal novelty ready!"#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Only the player with the initiative has the right to attack."#,
        r#"Wilhelm Steinitz"#,
    ),
    (r#"Open with a center pawn."#, r#"?"#),
    (
        r#"Openings teach you openings. Endgames teach you chess!"#,
        r#"Stephan Gerzadowicz"#,
    ),
    (r#"Passed pawns must be pushed."#, r#"?"#),
    (
        r#"Passed pawns should be blocked by the king, the only piece that is not harmed by watching a pawn is the knight."#,
        r#"?"#,
    ),
    (
        r#"Pawn endings are to Chess what putting is to golf."#,
        r#"Cecil Purdy"#,
    ),
    (
        r#"Pawns not only create the sketch for the whole painting, they are also the soil, the foundation, of any position."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"Pay particular attention to the f2-and f7-squares."#,
        r#"?"#,
    ),
    (r#"Perpetual check looms in all queen endings."#, r#"?"#),
    (
        r#"Place the board so that the sun is in your opponent’s eyes."#,
        r#"Ruy Lopez de Segura (c 1530-1580)"#,
    ),
    (
        r#"Place your pawns on the opposite color square as your bishop."#,
        r#"?"#,
    ),
    (
        r#"Play on both sides of the board is my favourite strategy."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Play the move that forces the win in the simplest way. Leave the brilliancies to Alekhine, Keres and Tal."#,
        r#"Irving Chernev"#,
    ),
    (r#"Play to get control of the center."#, r#"?"#),
    (
        r#"Playing without a concurrent critical review of one's skills will simply get you nowhere."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Previously, Oberhansli was practically unknown even in his own country."#,
        r#"Moritz Henneberger"#,
    ),
    (
        r#"Psychology is the most important factor in chess."#,
        r#"Alexander Alekhine"#,
    ),
    (
        r#"Quiet moves often make a stronger impression than a wild combination with heavy sacrifices."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"Quite unique among chess openings, the King's gambit is especially apt for talent, for genius, for heroism."#,
        r#"Tony Santasiere"#,
    ),
    (
        r#"Remember, my darling, the most dangerous thing for the family life is – chess!"#,
        r#"character from 1925 Russian silent film 'Chess Fever'"#,
    ),
    (
        r#"Ridicule can do much, for instance embitter the existence of young talents."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (r#"Robert Fischer is a law unto himself."#, r#"Larry Evans"#),
    (r#"Rooks belong behind passed pawns."#, r#"?"#),
    (
        r#"Rooks require open files and ranks in order to reach their full potential."#,
        r#"?"#,
    ),
    (
        r#"Seize the outpost K5 with your knight, and you can go to sleep. Checkmate will come by itself."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Self-criticism is the best means of making progress."#,
        r#"Sergey Karjakin"#,
    ),
    (
        r#"Simplicity is the highest goal, achievable when you have overcome all difficulties."#,
        r#"Frederic Chopin"#,
    ),
    (
        r#"Sir, the slowness of genius is hard to bear, but the slowness of mediocrity is insufferable."#,
        r#"Henry Thomas Buckle"#,
    ),
    (
        r#"Sit there for five hours? Certainly not! A player must walk about between moves, it helps his thinking."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"Sit your opponent with the sun in his eyes."#,
        r#"Ruy Lopez (1561)"#,
    ),
    (
        r#"Some part of a mistake is always correct."#,
        r#"Probably the same person"#,
    ),
    (
        r#"Some part of a mistake is always correct."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"Some pieces in the King's Indian appear on a 'special price' list: the dark square bishops are at the top of that list."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Somebody usually gets the better deal in every exchange."#,
        r#"?"#,
    ),
    (
        r#"Start thinking about the endgame in the middlegame."#,
        r#"?"#,
    ),
    (
        r#"Tactics flow from a superior position."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"That's what chess is all about. One day you give your opponent a lesson, the next day he gives you one."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"The beauty of a move lies not in its appearance but in the thought behind it."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"The beauty of a move lies not in its’ appearance but in the thought behind it."#,
        r#"Aaron Nimzovich"#,
    ),
    (r#"The best defense is a couterattack."#, r#"?"#),
    (
        r#"The best indicator of a chess player's form is his ability to sense the climax of the game."#,
        r#"Boris Spassky"#,
    ),
    (
        r#"The best way to learn endings, as well as openings, is from the games of the masters."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"The blunders are all there on the board, waiting to be made."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"The combination player thinks forward; he starts from the given position, and tries the forceful moves in his mind."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The criterion of real strength is a deep penetration into the secrets of a position."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"The defensive power of a pinned piece is but imaginary."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"The defensive power of a pinned piece is only imaginary."#,
        r#"Aaron Nimzovich"#,
    ),
    (
        r#"The easiest endings to draw are those with bishops of opposite colors."#,
        r#"?"#,
    ),
    (
        r#"The easiest endings to win are pure pawn endings."#,
        r#"?"#,
    ),
    (
        r#"The essence of Chess is thinking about what Chess is."#,
        r#"David Bronstein"#,
    ),
    (
        r#"The essential disadvantage of the isolated pawn ... lies not in the pawn itself, but in the square in front of the pawn."#,
        r#"Richard Reti"#,
    ),
    (
        r#"The fact that the 7 hours time control allows us to play a great deep game is not of great importance for mass-media."#,
        r#"Alexei Shirov"#,
    ),
    (
        r#"The first great chess players, including the world champion, got by perfectly well without constant coaches."#,
        r#"Anatoly Karpov"#,
    ),
    (
        r#"The first principle of attack: don't let the enemy develop !"#,
        r#"Rueben Fine"#,
    ),
    (
        r#"The future belongs to he who has the bishops."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"The hallmark of the artist is simplicity."#,
        r#"Larry Evans"#,
    ),
    (
        r#"The hardest game to win is a won game."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"The hardest part of chess is winning a won game."#,
        r#"Frank Marshall"#,
    ),
    (
        r#"The highest art of the chessplayer lies in not allowing your opponent to show you what he can do."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"The human element, the human flaw and the human nobility: those are the reasons that chess matches are won or lost."#,
        r#"Viktor Korchnoi"#,
    ),
    (
        r#"The initiative is an advantage. Take it whenever you can, and take it back when you don't have it, if at all possible."#,
        r#"?"#,
    ),
    (
        r#"The isolated pawn casts gloom over the entire chessboard."#,
        r#"Aaron Nimzowitsch"#,
    ),
    (
        r#"The key to ultimate succcess is the determination to progress day by day."#,
        r#"Edmar Mednis"#,
    ),
    (r#"The king is a strong piece, use it !"#, r#"Rueben Fine"#),
    (r#"The king must be a active in the ending."#, r#"?"#),
    (
        r#"The king pawn and the queen pawn are the only ones to be moved in the early part of the game."#,
        r#"Wilhelm Steinitz"#,
    ),
    (
        r#"The laws of chess do not permit a free choice: you have to move whether you like it or not."#,
        r#"Emanuel Lasker"#,
    ),
    (r#"The loser is always at fault."#, r#"Vasily Panov"#),
    (
        r#"The middlegame I repeat is chess itself, chess with all its possibilities, its attacks, defences, sacrifices, etc."#,
        r#"Eugene Znosko-Borovsky"#,
    ),
    (
        r#"The mistakes are there waiting to be made."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"The move...d7-d5 is the antidote for the poison in many gambits."#,
        r#"?"#,
    ),
    (
        r#"The number of 'unneccessary' errors that have been committed on move 41 are legion."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"The older I grow, the more I value Pawns."#,
        r#"Paul Keres"#,
    ),
    (
        r#"The only good Rook is a working Rook!"#,
        r#"Samuel Reshevsky"#,
    ),
    (
        r#"The only thing Chess players have in common is Chess."#,
        r#"Lodewijk Prins"#,
    ),
    (
        r#"The path from a1 to a8 is the same length as the path from a1 to h8."#,
        r#"?"#,
    ),
    (r#"The pin is mightier than the sword."#, r#"Fred Reinfeld"#),
    (
        r#"The placing of the center pawns determines the 'topography' of a game of chess."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"The pupil wants not so much to learn, as to learn how to learn."#,
        r#"Samuel Boden"#,
    ),
    (
        r#"The scheme of a game is played on positional lines; the decision of it, as a rule, is effected by combinations."#,
        r#"Richard Reti"#,
    ),
    (
        r#"The sign of a great Master is his ability to win a won game quickly and painlessly."#,
        r#"Irving Chernev"#,
    ),
    (
        r#"The stomach is an essential part of the chessmaster."#,
        r#"Bent Larsen"#,
    ),
    (
        r#"The winner of the game is the player who makes the next-to-last mistake."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"The young people have read my book. Now I have no chance."#,
        r#"Efim Bogolubow"#,
    ),
    (
        r#"There are two types of sacrifices: correct ones and mine."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"There are two types of sacrifices: correct ones, and mine."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"There can be no finer example of the inspiring powers of competition to shatter the status quo than Hungary's Judit Polgar."#,
        r#"Kasparov"#,
    ),
    (
        r#"There is no better place for learning to work independently and to extend your horizon than in higher school."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"There is no remorse like a remorse of chess. It is a curse upon man. There is no happiness in chess."#,
        r#"H.G. Wells"#,
    ),
    (
        r#"There is nothing that disgusts a man like getting beaten at chess by a woman."#,
        r#"Charles Dudley Warner"#,
    ),
    (
        r#"There just isn't enough televised chess."#,
        r#"David Letterman"#,
    ),
    (
        r#"There just isn’t enough televised Chess."#,
        r#"David Letterman"#,
    ),
    (
        r#"These young guys are playing checkers. I'm out there playing chess."#,
        r#"Kobe Bryant"#,
    ),
    (
        r#"Those who say they understand Chess, understand nothing."#,
        r#"Robert Huebner"#,
    ),
    (
        r#"Throughout my chess career I sought out new challenges, looking for things no one had done before."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"To avoid losing a piece, many a person has lost the game."#,
        r#"Savielly Tartakower"#,
    ),
    (
        r#"To be at their best,bishops require open diagonals and attackable weaknesses."#,
        r#"?"#,
    ),
    (
        r#"To free your game, take off some of your adversary's men, if possible, for nothing."#,
        r#"Captain Bertin"#,
    ),
    (
        r#"To have a knight planted in your game at K6 is worse than a rusty nail in your knee."#,
        r#"Efim Bogolubow"#,
    ),
    (
        r#"To improve at chess you should in the first instance study the endgame."#,
        r#"Jose Capablanca"#,
    ),
    (
        r#"To lose one's objective attitude to a position, nearly always means ruining your game."#,
        r#"David Bronstein"#,
    ),
    (
        r#"To play a match for the World Championship is the cherished dream of every chess player."#,
        r#"David Bronstein"#,
    ),
    (
        r#"To win without pawns, you must be a rook or two minor pieces ahead."#,
        r#"?"#,
    ),
    (
        r#"Try to maintain at least one pawn in the center."#,
        r#"?"#,
    ),
    (
        r#"Try to play after your opponent has eaten or drunk freely."#,
        r#"Luis Ramírez de Lucena (c. 1465 – c.1530)"#,
    ),
    (
        r#"Turning chess into poker and hoping for a "bluff" is not one of my convictions."#,
        r#"Tigran Petrosian"#,
    ),
    (
        r#"Two bishops vs. bishop and knight constitute a tangible advantage."#,
        r#"?"#,
    ),
    (
        r#"Two passed pawns advancing on the enemy pieces have brought me more than a dozen points in tournaments."#,
        r#"David Bronstein"#,
    ),
    (
        r#"Two passed pawns on the sixth beat everything, up to a royal flush."#,
        r#"Ian Rogers"#,
    ),
    (
        r#"Typically, in the last round of open tournaments the level of play is markedly lower, the number of blunders higher."#,
        r#"Pal Benko"#,
    ),
    (
        r#"We cannot resist the fascination of sacrifice, since a passion for sacrifices is part of a Chessplayer’s nature."#,
        r#"Rudolf Spielman"#,
    ),
    (r#"We like to think."#, r#"Garry Kasparov"#),
    (
        r#"Weak points or holes in the opponent's position must be occupied by pieces, not pawns."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Weak points or holes in the opponent’s position must be occupied by pieces not Pawns."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"What a crazy crazy round today..."#,
        r#"Viswanathan Anand"#,
    ),
    (
        r#"What do I do when we're not taping? Sit in a dark room and refine my plans for someday ruling Earth from a blimp. And chess."#,
        r#"Ryan Stiles"#,
    ),
    (
        r#"What we need are lots of girls who aren't as good as us, who'll treat us with the proper respect and reverance."#,
        r#"David Norwood"#,
    ),
    (
        r#"What would Chess be without silly mistakes?"#,
        r#"Kurt Richter"#,
    ),
    (
        r#"When I have White, I win because I am White; when I have Black, I win because I am Bogolyubov."#,
        r#"Efim Bogolyubov"#,
    ),
    (
        r#"When chess masters err, ordinary wood pushers tend to derive a measure of satisfaction, if not actual glee."#,
        r#"I.A. Horowitz"#,
    ),
    (
        r#"When choosing between two pawn captures,it's generally better to capture toward the center."#,
        r#"?"#,
    ),
    (
        r#"When everything on the board is clear it can be so difficult to conceal your thoughts from your opponent."#,
        r#"David Bronstein"#,
    ),
    (
        r#"When the Chess game is over, the Pawn and the king go back to the same box."#,
        r#"Irish proverb"#,
    ),
    (
        r#"When you absolutely don't know what to do anymore, it is time to panic."#,
        r#"John van der Wiel"#,
    ),
    (
        r#"When you absolutely don't know what to do anymore, it is time to panic."#,
        r#"Unknown"#,
    ),
    (
        r#"When you are ahead in material, exchenge as many pieces as possible, espeially queens."#,
        r#"?"#,
    ),
    (
        r#"When you defend, try not to worry or become upset. Keep your cool and trust your position, it's all you've got."#,
        r#"Pal Benko"#,
    ),
    (
        r#"When you see a good move, look for a better one."#,
        r#"Emanuel Lasker"#,
    ),
    (
        r#"When your opponent has one or more pieces exposed, look for a combination."#,
        r#"?"#,
    ),
    (
        r#"While I do my absolute best to commit harakiri, (chess) at least her calming presence from afar puts it all in perspective."#,
        r#"Hikaru Nakamura"#,
    ),
    (
        r#"White has no positional equivalent for the centralized pawn."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"Whoever sees no other aim in the game than that of giving checkmate to one’s opponent will never become a good Chess player."#,
        r#"Max Euwe"#,
    ),
    (
        r#"Winning isn’t everything... but losing is nothing."#,
        r#"Mednis"#,
    ),
    (
        r#"With opposite coloured bishops the attacking side has in effect an extra piece in the shape of his bishop."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"Women, by their nature, are not exceptional chess players: they are not great fighters."#,
        r#"Garry Kasparov"#,
    ),
    (
        r#"Yes, I have played a blitz game once. It was on a train, in 1929."#,
        r#"Mikhail Botvinnik"#,
    ),
    (
        r#"You are for me the Queen on d8 and I am the Pawn on d7!!"#,
        r#"Eduard Gufeld"#,
    ),
    (
        r#"You can only get good at Chess if you love the game."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"You can permit yourself any liberty in the opening except the luxury of a passive position."#,
        r#"Grigory Sanakoev"#,
    ),
    (
        r#"You cannot play at Chess if you are kind-hearted."#,
        r#"French Proverb"#,
    ),
    (
        r#"You have to have the fighting spirit. You have to force moves and take chances."#,
        r#"Bobby Fischer"#,
    ),
    (
        r#"You must attack when you have the superior game, or you will fofeit your advantage."#,
        r#"?"#,
    ),
    (
        r#"You must not let your opponent know how you feel."#,
        r#"Alexander Kotov"#,
    ),
    (
        r#"You must take your opponent into a deep dark forest where 2+2=5, and the path leading out is only wide enough for one."#,
        r#"Mikhail Tal"#,
    ),
    (
        r#"You need not play well, just help your opponent to play badly."#,
        r#"Genrikh Chepukaitis"#,
    ),
    (
        r#"Your only task in the opening is to reach a playable middlegame."#,
        r#"Lajos Portisch"#,
    ),
    (
        r#"Your practical results will improve when you play what you know, like and have confidence in."#,
        r#"Edmar Mednis"#,
    ),
    (
        r#"Youth has triumphed."#,
        r#"84-year-old Jacques Mieses (upon defeating 86-year-old Dirk van Foreest)"#,
    ),
    (
        r#"… a lively imagination can exercise itself most fully and creatively in conjuring up magnificent combinations."#,
        r#"Siegbert Tarrasch"#,
    ),
    (
        r#"… on the right occasion a bold choice of opening can unnerve even the most steely opponent."#,
        r#"Neil McDonald"#,
    ),
    (
        r#"… though combinations are without number, the number of ideas are limited."#,
        r#"Eugene Znosko-Borowski"#,
    ),
    (
        r#"So often, as any player will agree, it is hopes and fears which seem to influence the choice of a move. Notoriously, the weaker player will tend to exaggerate both his advantages and his disadvantages, thinking that he has a win in a good position, and a loss with a bad one. This emotional liability seems less obvious at higher levels..."#,
        r#"Jonathan Rowson - The Seven Deadly Chess Sins"#,
    ),
];
