// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package cron

import (
	"context"
	"database/sql"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/store/repos"
	"github.com/drone/drone/store/shared/db/dbtest"
)

var noContext = context.TODO()

func TestCron(t *testing.T) {
	conn, err := dbtest.Connect()
	if err != nil {
		t.Error(err)
		return
	}
	defer func() {
		dbtest.Reset(conn)
		dbtest.Disconnect(conn)
	}()

	// seeds the database with a dummy repository.
	repo := &core.Repository{UID: "1", Slug: "octocat/hello-world"}
	repos := repos.New(conn)
	if err := repos.Create(noContext, repo); err != nil {
		t.Error(err)
	}

	store := New(conn).(*cronStore)
	t.Run("Create", testCronCreate(store, repos, repo))
}

func testCronCreate(store *cronStore, repos core.RepositoryStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		item := &core.Cron{
			RepoID: repo.ID,
			Name:   "nightly",
			Expr:   "00 00 * * *",
			Next:   1000000000,
		}
		err := store.Create(noContext, item)
		if err != nil {
			t.Error(err)
		}
		if item.ID == 0 {
			t.Errorf("Want cron ID assigned, got %d", item.ID)
		}

		t.Run("Find", testCronFind(store, item))
		t.Run("FindName", testCronFindName(store, repo))
		t.Run("List", testCronList(store, repo))
		t.Run("Read", testCronReady(store, repo))
		t.Run("Update", testCronUpdate(store, repo))
		t.Run("Delete", testCronDelete(store, repo))
		t.Run("Fkey", testCronForeignKey(store, repos, repo))
	}
}

func testCronFind(store *cronStore, cron *core.Cron) func(t *testing.T) {
	return func(t *testing.T) {
		item, err := store.Find(noContext, cron.ID)
		if err != nil {
			t.Error(err)
		} else {
			t.Run("Fields", testCron(item))
		}
	}
}

func testCronFindName(store *cronStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		item, err := store.FindName(noContext, repo.ID, "nightly")
		if err != nil {
			t.Error(err)
		} else {
			t.Run("Fields", testCron(item))
		}
	}
}

func testCronList(store *cronStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		list, err := store.List(noContext, repo.ID)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := len(list), 1; got != want {
			t.Errorf("Want count %d, got %d", want, got)
		} else {
			t.Run("Fields", testCron(list[0]))
		}
	}
}

func testCronReady(store *cronStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		item := &core.Cron{
			RepoID: repo.ID,
			Name:   "daily",
			Expr:   "00 00 * * *",
			Next:   1000000002, // ignored (1 second too late)
		}
		err := store.Create(noContext, item)
		if err != nil {
			t.Error(err)
			return
		}
		list, err := store.Ready(noContext, 1000000001)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := len(list), 1; got != want {
			t.Errorf("Want count %d, got %d", want, got)
		} else {
			t.Run("Fields", testCron(list[0]))
		}
	}
}

func testCronUpdate(store *cronStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		before, err := store.FindName(noContext, repo.ID, "nightly")
		if err != nil {
			t.Error(err)
			return
		}
		err = store.Update(noContext, before)
		if err != nil {
			t.Error(err)
			return
		}
		after, err := store.Find(noContext, before.ID)
		if err != nil {
			t.Error(err)
			return
		}
		if after == nil {
			t.Fail()
		}
	}
}

func testCronDelete(store *cronStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		cron, err := store.FindName(noContext, repo.ID, "nightly")
		if err != nil {
			t.Error(err)
			return
		}
		err = store.Delete(noContext, cron)
		if err != nil {
			t.Error(err)
			return
		}
		_, err = store.Find(noContext, cron.ID)
		if got, want := sql.ErrNoRows, err; got != want {
			t.Errorf("Want sql.ErrNoRows, got %v", got)
			return
		}
	}
}

func testCronForeignKey(store *cronStore, repos core.RepositoryStore, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		item := &core.Cron{
			RepoID: repo.ID,
			Name:   "nightly",
			Expr:   "00 00 * * *",
		}
		store.Create(noContext, item)
		before, _ := store.List(noContext, repo.ID)
		if len(before) == 0 {
			t.Errorf("Want non-empty cron list")
			return
		}

		err := repos.Delete(noContext, repo)
		if err != nil {
			t.Error(err)
			return
		}
		after, _ := store.List(noContext, repo.ID)
		if len(after) != 0 {
			t.Errorf("Want empty cron list")
		}
	}
}

func testCron(item *core.Cron) func(t *testing.T) {
	return func(t *testing.T) {
		if got, want := item.Name, "nightly"; got != want {
			t.Errorf("Want cron name %q, got %q", want, got)
		}
		if got, want := item.Expr, "00 00 * * *"; got != want {
			t.Errorf("Want cron name %q, got %q", want, got)
		}
	}
}
