// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package secret

import (
	"context"
	"crypto/aes"
	"crypto/cipher"
	"encoding/base64"
	"errors"

	"github.com/drone/drone-yaml/yaml"
	"github.com/drone/drone/core"
	"github.com/drone/drone/logger"
)

// Encrypted returns a new encrypted Secret controller.
func Encrypted() core.SecretService {
	return new(encrypted)
}

type encrypted struct {
}

func (c *encrypted) Find(ctx context.Context, in *core.SecretArgs) (*core.Secret, error) {
	logger := logger.FromContext(ctx).
		WithField("name", in.Name).
		WithField("kind", "secret")

	// lookup the named secret in the manifest. If the
	// secret does not exist, return a nil variable,
	// allowing the next secret controller in the chain
	// to be invoked.
	data, ok := getEncrypted(in.Conf, in.Name)
	if !ok {
		logger.Trace("secret: encrypted: no matching secret")
		return nil, nil
	}

	// if the build event is a pull request and the source
	// repository is a fork, the secret is not exposed to
	// the pipeline, for security reasons.
	if in.Repo.Private == false &&
		in.Build.Event == core.EventPullRequest &&
		in.Build.Fork != "" {
		logger.Trace("secret: encrypted: restricted from forks")
		return nil, nil
	}

	decoded, err := base64.StdEncoding.DecodeString(string(data))
	if err != nil {
		logger.WithError(err).Trace("secret: encrypted: cannot decode")
		return nil, err
	}

	decrypted, err := decrypt(decoded, []byte(in.Repo.Secret))
	if err != nil {
		logger.WithError(err).Trace("secret: encrypted: cannot decrypt")
		return nil, err
	}

	logger.Trace("secret: encrypted: found matching secret")

	return &core.Secret{
		Name: in.Name,
		Data: string(decrypted),
	}, nil
}

func getEncrypted(manifest *yaml.Manifest, match string) (data string, ok bool) {
	for _, resource := range manifest.Resources {
		secret, ok := resource.(*yaml.Secret)
		if !ok {
			continue
		}
		if secret.Name != match {
			continue
		}
		if secret.Data == "" {
			continue
		}
		return secret.Data, true
	}
	return
}

func decrypt(ciphertext []byte, key []byte) (plaintext []byte, err error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	if len(ciphertext) < gcm.NonceSize() {
		return nil, errors.New("malformed ciphertext")
	}

	return gcm.Open(nil,
		ciphertext[:gcm.NonceSize()],
		ciphertext[gcm.NonceSize():],
		nil,
	)
}
