// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package converter

import (
	"errors"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/golang/mock/gomock"
)

func TestMemoize(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	conf := &core.Config{Data: "{kind: pipeline, type: docker, steps: []}"}
	args := &core.ConvertArgs{
		Build:  &core.Build{After: "3950521325d4744760a96c18e3d0c67d86495af3"},
		Repo:   &core.Repository{ID: 42},
		Config: conf,
	}

	base := mock.NewMockConvertService(controller)
	base.EXPECT().Convert(gomock.Any(), gomock.Any()).Return(args.Config, nil)

	service := Memoize(base, 10).(*memoize)
	_, err := service.Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if got, want := service.cache.Len(), 1; got != want {
		t.Errorf("Expect %d items in cache, got %d", want, got)
	}

	args.Config = nil // set to nil to prove we get the cached value
	res, err := service.Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if res != conf {
		t.Errorf("Expect result from cache")
	}

	if got, want := service.cache.Len(), 1; got != want {
		t.Errorf("Expect %d items in cache, got %d", want, got)
	}
}

func TestMemoize_Tag(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		Build:  &core.Build{Ref: "refs/tags/v1.0.0"},
		Repo:   &core.Repository{ID: 42},
		Config: &core.Config{Data: "{kind: pipeline, type: docker, steps: []}"},
	}

	base := mock.NewMockConvertService(controller)
	base.EXPECT().Convert(gomock.Any(), gomock.Any()).Return(args.Config, nil)

	service := Memoize(base, 10).(*memoize)
	res, err := service.Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if res != args.Config {
		t.Errorf("Expect result from cache")
	}
}

func TestMemoize_Empty(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		Build:  &core.Build{After: "3950521325d4744760a96c18e3d0c67d86495af3"},
		Repo:   &core.Repository{ID: 42},
		Config: &core.Config{Data: ""}, // empty
	}

	base := mock.NewMockConvertService(controller)
	base.EXPECT().Convert(gomock.Any(), gomock.Any()).Return(args.Config, nil)

	service := Memoize(base, 10).(*memoize)
	res, err := service.Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if res != nil {
		t.Errorf("Expect nil response")
	}
	if got, want := service.cache.Len(), 0; got != want {
		t.Errorf("Expect %d items in cache, got %d", want, got)
	}
}

func TestMemoize_Nil(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		Build:  &core.Build{After: "3950521325d4744760a96c18e3d0c67d86495af3"},
		Repo:   &core.Repository{ID: 42},
		Config: nil,
	}

	base := mock.NewMockConvertService(controller)
	base.EXPECT().Convert(gomock.Any(), gomock.Any()).Return(args.Config, nil)

	service := Memoize(base, 10).(*memoize)
	res, err := service.Convert(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if res != nil {
		t.Errorf("Expect nil response")
	}
	if got, want := service.cache.Len(), 0; got != want {
		t.Errorf("Expect %d items in cache, got %d", want, got)
	}
}

func TestMemoize_Error(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConvertArgs{
		Build: &core.Build{After: "3950521325d4744760a96c18e3d0c67d86495af3"},
		Repo:  &core.Repository{ID: 42},
	}

	want := errors.New("not found")
	base := mock.NewMockConvertService(controller)
	base.EXPECT().Convert(gomock.Any(), gomock.Any()).Return(nil, want)

	service := Memoize(base, 10).(*memoize)
	_, err := service.Convert(noContext, args)
	if err == nil {
		t.Errorf("Expect error from base returned to caller")
		return
	}
	if got, want := service.cache.Len(), 0; got != want {
		t.Errorf("Expect %d items in cache, got %d", want, got)
	}
}
