// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package user

import (
	"bytes"
	"encoding/json"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/mock"
	"github.com/drone/drone/core"

	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
)

func TestUpdate(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	userInput := &core.User{
		Login: "octocat",
		Email: "octocat@github.com",
	}
	user := &core.User{
		Login: "octocat",
		Email: "",
	}

	users := mock.NewMockUserStore(controller)
	users.EXPECT().Update(gomock.Any(), user)

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(userInput)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/api/user", in)
	r = r.WithContext(
		request.WithUser(r.Context(), user),
	)

	HandleUpdate(users)(w, r)
	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	if got, want := user.Email, "octocat@github.com"; got != want {
		t.Errorf("Want user email %v, got %v", want, got)
	}

	got, want := new(core.User), user
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

// the purpose of this unit test is to verify that an invalid
// (in this case missing) request body will result in a bad
// request error returned to the client.
func TestUpdate_BadRequest(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	mockUser := &core.User{
		ID:    1,
		Login: "octocat",
	}

	in := new(bytes.Buffer)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/api/user", in)
	r = r.WithContext(
		request.WithUser(r.Context(), mockUser),
	)

	HandleUpdate(nil)(w, r)
	if got, want := w.Code, 400; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), &errors.Error{Message: "EOF"}
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

// the purpose of this unit test is to verify that an error
// updating the database will result in an internal server
// error returned to the client.
func TestUpdate_ServerError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	userInput := &core.User{
		Login: "octocat",
		Email: "octocat@github.com",
	}
	user := &core.User{
		Login: "octocat",
		Email: "",
	}

	users := mock.NewMockUserStore(controller)
	users.EXPECT().Update(gomock.Any(), user).Return(errors.ErrNotFound)

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(userInput)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/api/user", in)
	r = r.WithContext(
		request.WithUser(r.Context(), user),
	)

	HandleUpdate(users)(w, r)
	if got, want := w.Code, 500; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}
