/* gkrellm-bluez - A BlueZ monitor plug-in for GKrellM2
 *
 * Copyright (C) 2006 Ludovic Cintrat <lcintrat@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "gkrellm-bluez.h"

#include "gkrellm-bluez-chart.h"

#define DECAL_NET_LEDS      "decal_net_leds"

#define GRID_RESOLUTION_MIN 0
#define GRID_RESOLUTION_MAX 100

#define GRID_STEP0          5
#define GRID_STEP1          10
#define GRID_DIGITS         0
#define GRID_WIDTH          50

/*  prototypes  */

static gboolean chart_expose_event_callback       (GtkWidget          *widget,
        GdkEventExpose     *event,
        gpointer            user_data);
static gboolean chart_button_press_event_callback (GtkWidget          *widget,
        GdkEventButton     *event,
        gpointer            user_data);
static gboolean panel_expose_event_callback       (GtkWidget          *widget,
        GdkEventExpose     *event,
        gpointer            user_data);
static gboolean panel_button_press_event_callback (GtkWidget          *widget,
        GdkEventButton     *event,
        gpointer            user_data);

static void     reorder_charts                    (GtkWidget* gkrellm_bluez_vbox);
static void     draw_chart                        (gpointer            user_data);
static gchar *  create_chart_text                 (GkrellmBlueZMonitor *bluezmon);
static gchar *  strreplace                        (const gchar        *string,
						   const gchar        *delimiter,
						   const gchar        *replacement);
static gint map_x(gint x, gint width);
static void draw_led(GkrellmBlueZMonitor *bluezmon, int rxtx, int led_index);

/*  data  */

static GdkPixmap       *decal_net_led_pixmap;
static GdkBitmap       *decal_net_led_mask;
static GkrellmPiximage *decal_net_led_piximage;


/*  public functions  */

void
gkrellm_bluez_chart_create (GkrellmBlueZMonitor *bluezmon, 
			    GtkWidget* gkrellm_bluez_vbox,
			    gboolean first_create)
{
    GkrellmStyle    *style;
    GkrellmPiximage *krell_image;

    if (first_create)
    {
      g_assert (bluezmon != NULL);
      g_assert (bluezmon->chart == NULL);

      bluezmon->chart = gkrellm_chart_new0 ();
      bluezmon->chart->panel = gkrellm_panel_new0 ();

      reorder_charts (gkrellm_bluez_vbox);
    }


    style = gkrellm_panel_style (gkrellm_bluez_style_id);

    gkrellm_chart_create (gkrellm_bluez_vbox,
                          gkrellm_bluez,
                          bluezmon->chart,
                          &bluezmon->config);

    /****** RX/TX chart data ******/

#define create_data(bluezmon, data, data_name) \
   bluezmon->data = gkrellm_add_default_chartdata (bluezmon->chart,    \
						   _(data_name));      \
                                                                       \
   gkrellm_monotonic_chartdata (bluezmon->data, TRUE);                 \
                                                                       \
   gkrellm_set_chartdata_draw_style_default (bluezmon->data,           \
					     CHARTDATA_IMPULSE);      

    create_data(bluezmon, tx_data, "tx bytes");
    create_data(bluezmon, rx_data, "rx bytes");

      /****** Chart config ******/

      gkrellm_chartconfig_grid_resolution_adjustment (bluezmon->config,
						      FALSE, 1.0,
						      GRID_RESOLUTION_MIN,
						      GRID_RESOLUTION_MAX,
						      GRID_STEP0,
						      GRID_STEP1,
						      GRID_DIGITS,
						      GRID_WIDTH);
      
      gkrellm_chartconfig_grid_resolution_label (bluezmon->config,
						 _("rx/tx bytes per sec"));

    /****** Chart ******/

    gkrellm_set_draw_chart_function (bluezmon->chart, draw_chart, bluezmon);

    gkrellm_alloc_chartdata (bluezmon->chart);

    /****** LEDs ******/
  
    gint rx_led_x, tx_led_x, rx_led_y, tx_led_y; 
  
    gkrellm_net_led_positions(&rx_led_x, &rx_led_y, &tx_led_x, &tx_led_y);
  
    bluezmon->rxled = gkrellm_create_decal_pixmap(bluezmon->chart->panel, 
						  decal_net_led_pixmap,
						  decal_net_led_mask,
						  N_LEDS, style, 0, rx_led_y);
  
    bluezmon->rxled->x = map_x(rx_led_x, bluezmon->rxled->w);
  
    bluezmon->txled = gkrellm_create_decal_pixmap(bluezmon->chart->panel,
						  decal_net_led_pixmap,
						  decal_net_led_mask, 
						  N_LEDS, style, 0, tx_led_y);
  
    bluezmon->txled->x = map_x(tx_led_x, bluezmon->txled->w);

    /****** Krell ******/

    krell_image     = gkrellm_krell_panel_piximage (gkrellm_bluez_style_id);
    bluezmon->krell = gkrellm_create_krell (bluezmon->chart->panel, krell_image, style);

    gkrellm_monotonic_krell_values (bluezmon->krell, TRUE);
    

    /****** Panel ******/

    gkrellm_panel_label_on_top_of_decals (bluezmon->chart->panel, TRUE);

    gkrellm_setup_launcher (bluezmon->chart->panel,
                            &bluezmon->launcher,
                            CHART_PANEL_TYPE, 0);

    gkrellm_panel_configure (bluezmon->chart->panel,
                             bluezmon->interface, style);

    gkrellm_panel_create (bluezmon->chart->box,
                          gkrellm_bluez,
                          bluezmon->chart->panel);


    if (first_create)
    {
      g_signal_connect (bluezmon->chart->drawing_area,
			"expose-event",
			G_CALLBACK (chart_expose_event_callback),
			bluezmon);
      
      g_signal_connect (bluezmon->chart->drawing_area,
			"button-press-event",
			G_CALLBACK (chart_button_press_event_callback),
			bluezmon);
    
      g_signal_connect (bluezmon->chart->panel->drawing_area,
			"expose-event",
			G_CALLBACK (panel_expose_event_callback),
			bluezmon);
      
      g_signal_connect (bluezmon->chart->panel->drawing_area,
			"button-press-event",
			G_CALLBACK (panel_button_press_event_callback),
			bluezmon);
    }
}


void
gkrellm_bluez_chart_refresh (GkrellmBlueZMonitor *bluezmon)
{
  gkrellm_store_chartdata (bluezmon->chart, 0, 
			   bluezmon->tx_bytes, 
			   bluezmon->rx_bytes);
  
  gkrellm_refresh_chart(bluezmon->chart);
}


void 
gkrellm_bluez_panel_refresh (GkrellmBlueZMonitor *bluezmon)
{
  /*** LEDs ***/

  if (bluezmon->rx_bytes > bluezmon->rx_bytes_old)
    draw_led(bluezmon, RX_LED, RX_ON);
  else
    draw_led(bluezmon, RX_LED, RX_OFF);
  
  if (bluezmon->tx_bytes > bluezmon->tx_bytes_old)
    draw_led(bluezmon, TX_LED, TX_ON);
  else
    draw_led(bluezmon, TX_LED, TX_OFF);

  /*** Krell ***/

  gkrellm_update_krell(bluezmon->chart->panel, bluezmon->krell,
		       bluezmon->tx_bytes + bluezmon->rx_bytes);

  gkrellm_draw_panel_layers(bluezmon->chart->panel);
}


void
gkrellm_bluez_load_extra_piximages(void)
{
    gint			w, h;

    /* Check for theme_dir/net/decal_net_leds.png.
     */
    gkrellm_load_piximage(DECAL_NET_LEDS, NULL,
			  &decal_net_led_piximage, NET_STYLE_NAME);
    
    w = gdk_pixbuf_get_width(decal_net_led_piximage->pixbuf);
    w *= gkrellm_get_theme_scale();
    
    h = gdk_pixbuf_get_height(decal_net_led_piximage->pixbuf) / N_LEDS;
    h *= gkrellm_get_theme_scale();
    
    gkrellm_scale_piximage_to_pixmap(decal_net_led_piximage,
				     &decal_net_led_pixmap, 
				     &decal_net_led_mask, 
				     w, h * N_LEDS);
}


/*  private functions  */

static void
draw_led(GkrellmBlueZMonitor *bluezmon, int rxtx, int led_index)
{
    GkrellmPanel	*p;
    GkrellmDecal	*led;

    if (!bluezmon->chart)
      return;

    p = bluezmon->chart->panel;

    if (rxtx == RX_LED)
      led = bluezmon->rxled;
    else
      led = bluezmon->txled;
    
    gkrellm_draw_decal_pixmap(p, led, led_index);
}

static gint
map_x(gint x, gint width)
{
    gint	xnew, chart_width;

    xnew = x;
    chart_width = gkrellm_chart_width();
    if (x < 0)
      xnew += chart_width - width;
    return xnew;
}


static gboolean
chart_expose_event_callback (GtkWidget      *widget,
                             GdkEventExpose *event,
                             gpointer        user_data)
{
    GkrellmBlueZMonitor *bluezmon;

    g_assert (widget != NULL);
    g_assert (event != NULL);
    g_assert (user_data != NULL);

    bluezmon = user_data;

    gdk_draw_drawable (widget->window,
                       widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                       bluezmon->chart->pixmap, event->area.x, event->area.y,
                       event->area.x, event->area.y,
                       event->area.width, event->area.height);

    return FALSE;
}

static gboolean
chart_button_press_event_callback (GtkWidget      *widget,
                                   GdkEventButton *event,
                                   gpointer        user_data)
{
    GkrellmBlueZMonitor *bluezmon;

    g_assert (widget != NULL);
    g_assert (event != NULL);
    g_assert (user_data != NULL);

    bluezmon = user_data;

    switch (event->button)
    {
    case 1:
        if (event->type == GDK_BUTTON_PRESS)
        {
            bluezmon->hide_text = ! bluezmon->hide_text;

            gkrellm_config_modified ();

            gkrellm_refresh_chart (bluezmon->chart);
        }
        else if (event->type == GDK_2BUTTON_PRESS)
        {
            gkrellm_chartconfig_window_create (bluezmon->chart);
        }
        break;

    case 3:
        gkrellm_chartconfig_window_create (bluezmon->chart);
        break;
    }

    return FALSE;
}

static gboolean
panel_expose_event_callback (GtkWidget      *widget,
                             GdkEventExpose *event,
                             gpointer        user_data)
{
    GkrellmBlueZMonitor *bluezmon;

    g_assert (widget != NULL);
    g_assert (event != NULL);
    g_assert (user_data != NULL);

    bluezmon = user_data;

    gdk_draw_drawable (widget->window,
                       widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                       bluezmon->chart->panel->pixmap, event->area.x, event->area.y,
                       event->area.x, event->area.y,
                       event->area.width, event->area.height);

    return FALSE;
}

static gboolean
panel_button_press_event_callback (GtkWidget      *widget,
                                   GdkEventButton *event,
                                   gpointer        user_data)
{
    g_assert (widget != NULL);
    g_assert (event != NULL);
    g_assert (user_data != NULL);

    if (event->button == 3)
        gkrellm_open_config_window (gkrellm_bluez);

    return FALSE;
}

static void
reorder_charts (GtkWidget* gkrellm_bluez_vbox)
{
    GkrellmBlueZMonitor *bluezmon;
    GList              *list;
    gint                i = 0;

    for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
        bluezmon = list->data;

        if (bluezmon->chart)
        {
            gtk_box_reorder_child (GTK_BOX (gkrellm_bluez_vbox),
                                   GTK_WIDGET (bluezmon->chart->box),
                                   i);

            i++;
        }
    }
}

static void
draw_chart (gpointer user_data)
{
    GkrellmBlueZMonitor *bluezmon;
    gchar              *tmp;

    g_assert (user_data != NULL);

    bluezmon = user_data;

    gkrellm_draw_chartdata (bluezmon->chart);

    if (! bluezmon->hide_text && gkrellm_bluez_format_string)
    {
        tmp = create_chart_text (bluezmon);

        gkrellm_draw_chart_text (bluezmon->chart, gkrellm_bluez_style_id, tmp);

        g_free (tmp);
    }

    gkrellm_draw_chart_to_screen (bluezmon->chart);
}

static void
convert_bytes (gchar** value, gint32 bytes)
{
    if (bytes == 0)
        *value = g_strdup ("0");
    else if (bytes >= GIGA)
        *value = g_strdup_printf ("%.1fG", ((float)bytes / (float)GIGA));
    else if (bytes >= MEGA)
        *value = g_strdup_printf ("%.1fM", ((float)bytes / (float)MEGA));
    else if (bytes >= KILO)
        *value = g_strdup_printf ("%.1fK", ((float)bytes / (float)KILO));
    else
        *value = g_strdup_printf ("%d", bytes);
}

static gchar *
create_chart_text (GkrellmBlueZMonitor *bluezmon)
{
    gchar   *tmp;
    gchar   *ret;
    gchar   *value;

    g_assert (bluezmon != NULL);

    ret = g_strdup (gkrellm_bluez_format_string);

    if (strstr (ret, "$M"))
    {
        convert_bytes( &value, gkrellm_get_chart_scalemax (bluezmon->chart) );

        tmp = strreplace (ret, "$M", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$T"))
    {
        convert_bytes( &value, gkrellm_get_current_chartdata(bluezmon->rx_data)
                       +gkrellm_get_current_chartdata(bluezmon->tx_data) );

        tmp = strreplace (ret, "$T", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$r"))
    {
        convert_bytes( &value, gkrellm_get_current_chartdata(bluezmon->rx_data) );

        tmp = strreplace (ret, "$r", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$t"))
    {
        convert_bytes( &value, gkrellm_get_current_chartdata(bluezmon->tx_data) );

        tmp = strreplace (ret, "$t", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$O"))
    {
        convert_bytes( &value, bluezmon->tx_bytes + bluezmon->rx_bytes );

        tmp = strreplace (ret, "$O", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$i"))
    {
        convert_bytes( &value, bluezmon->rx_bytes );

        tmp = strreplace (ret, "$i", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$o"))
    {
        convert_bytes( &value, bluezmon->tx_bytes );

        tmp = strreplace (ret, "$o", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$c"))
    {
        value = g_strdup_printf( "%d", bluezmon->connections );

        tmp = strreplace (ret, "$c", value);
        g_free (value);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    if (strstr (ret, "$l"))
    {
        tmp = strreplace (ret, "$l", bluezmon->interface);

        g_free (ret);
        ret = g_strdup (tmp);
        g_free (tmp);
    }

    return ret;
}

static gchar *
strreplace (const gchar *string,
            const gchar *delimiter,
            const gchar *replacement)
{
    gchar  *ret;
    gchar **tmp;

    g_return_val_if_fail (string != NULL, NULL);
    g_return_val_if_fail (delimiter != NULL, NULL);
    g_return_val_if_fail (replacement != NULL, NULL);

    tmp = g_strsplit (string, delimiter, 0);
    ret = g_strjoinv (replacement, tmp);
    g_strfreev (tmp);

    return ret;
}
