
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist_xml.c 2608 2007-07-25 10:33:53Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "codeset.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "playlist_xml.h"
#include "utils.h"
#include "xmlparser.h"

#define OXINE_PL_XML "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
#define OXINE_PL_DTD "<!DOCTYPE oxine_playlist SYSTEM \"http://oxine.sf.net/DTD/playlist.dtd\">\n\n"


bool
playlist_xml_save (playlist_t * playlist, const char *mrl)
{
    FILE *f = fopen (mrl, "w");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return false;
    }

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init (codeset, "utf8");

    fprintf (f, OXINE_PL_XML);
    fprintf (f, OXINE_PL_DTD);
    char *playmode = NULL;
    switch (playlist->playmode) {
    case PLAYBACK_MODE_NORMAL:
        playmode = "normal";
        break;
    case PLAYBACK_MODE_REPEAT:
        playmode = "repeat";
        break;
    case PLAYBACK_MODE_RANDOM:
        playmode = "random";
        break;
    default:
        break;
    }

    fprintf (f, "<oxine_playlist current=\"%d\" playmode=\"%s\">\n",
             playlist_get_current_pos (playlist), playmode);
    playitem_t *playitem = playlist_first (playlist);
    while (playitem) {
        fprintf (f, "  <playitem played=\"%s\" start_time=\"%d\">\n",
                 playitem->played ? "true" : "false", playitem->start_time);

        char *esc_str0 = xml_escape (playitem->title,
                                     xr, XML_ESCAPE_NO_QUOTE);
        char *esc_str1 = xml_escape (playitem->mrl,
                                     xr, XML_ESCAPE_DOUBLE_QUOTE);
        char *esc_str2 = xml_escape (playitem->subtitle_mrl,
                                     xr, XML_ESCAPE_DOUBLE_QUOTE);
        char *esc_str3 = xml_escape (playitem->thumbnail_mrl,
                                     xr, XML_ESCAPE_DOUBLE_QUOTE);
        {
            fprintf (f, "    <title>%s</title>\n", esc_str0);
        }
        {
            fprintf (f, "    <mrl           href=\"%s\" />\n", esc_str1);
        }
        if (esc_str2) {
            fprintf (f, "    <subtitle_mrl  href=\"%s\" />\n", esc_str2);
        }
        if (esc_str3) {
            fprintf (f, "    <thumbnail_mrl href=\"%s\" />\n", esc_str3);
        }
        ho_free (esc_str0);
        ho_free (esc_str1);
        ho_free (esc_str2);
        ho_free (esc_str3);

        fprintf (f, "  </playitem>\n");
        playitem = playlist_next (playlist, playitem);
    }
    fprintf (f, "</oxine_playlist>\n");
    fclose (f);

    recode_done (xr);
    ho_free (codeset);

    return true;
}


static void
read_playitem (const char *xml_mrl, playlist_t * playlist, xml_node_t * node)
{
    bool played = false;
    int start_time = 0;
    char *mrl = NULL;
    char *title = NULL;
    char *subtitle_mrl = NULL;
    char *thumbnail_mrl = NULL;

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("utf8", codeset);

    const char *str_played = xml_parser_get_property (node, "played");
    if (str_played && (strcasecmp (str_played, "true") == 0)) {
        played = true;
    }

    const char *str_start_time = xml_parser_get_property (node, "start_time");
    if (str_start_time) {
        start_time = atoi (str_start_time);
    }

    node = node->child;
    while (node) {
        if (strcasecmp (node->name, "mrl") == 0) {
            const char *href = xml_parser_get_property (node, "href");
            mrl = recode (xr, href);
        }

        else if (strcasecmp (node->name, "title") == 0) {
            title = recode (xr, node->data);
        }

        else if (strcasecmp (node->name, "subtitle_mrl") == 0) {
            const char *href = xml_parser_get_property (node, "href");
            subtitle_mrl = recode (xr, href);
        }

        else if (strcasecmp (node->name, "thumbnail_mrl") == 0) {
            const char *href = xml_parser_get_property (node, "href");
            thumbnail_mrl = recode (xr, href);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }

    if (mrl) {
        playitem_t *playitem = playlist_add (playlist, title, mrl);
        playitem->played = played;
        playitem->start_time = start_time;
        if (subtitle_mrl) {
            ho_free (playitem->subtitle_mrl);
            playitem->subtitle_mrl = ho_strdup (subtitle_mrl);
        }
        if (thumbnail_mrl) {
            ho_free (playitem->thumbnail_mrl);
            playitem->thumbnail_mrl = ho_strdup (thumbnail_mrl);
        }
    }
    else {
        error (_("Found an entry that did not contain a MRL!"));
    }

    ho_free (mrl);
    ho_free (title);
    ho_free (subtitle_mrl);
    ho_free (thumbnail_mrl);

    recode_done (xr);
    ho_free (codeset);
}


static void
read_nodes (const char *xml_mrl, playlist_t * playlist, xml_node_t * node)
{
    while (node) {
        if (!strcasecmp (node->name, "playitem")) {
            read_playitem (xml_mrl, playlist, node);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }
        node = node->next;
    }
}


bool
playlist_xml_load (playlist_t * playlist, const char *mrl)
{
    bool success = false;
    int size;
    xml_node_t *node;

    char *buffer = read_entire_file (mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    xml_parser_init (buffer, size, XML_PARSER_CASE_INSENSITIVE);
    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), mrl);
        goto parser_free;
    }

    if (strcasecmp (node->name, "oxine_playlist")) {
        error (_("Root node of '%s' must be '%s'!"), mrl, "oxine_playlist");
        goto parser_free;
    }

    playlist_change_cb_t change_cb = playlist->change_cb;
    playlist->change_cb = NULL;

    read_nodes (mrl, playlist, node->child);

    const char *str_current = xml_parser_get_property (node, "current");
    playlist_set_current_pos (playlist, atoi (str_current));

    const char *str_mode = xml_parser_get_property (node, "playmode");
    if (str_mode) {
        if (strcasecmp (str_mode, "normal") == 0) {
            playlist_set_playmode (playlist, PLAYBACK_MODE_NORMAL);
        }
        else if (strcasecmp (str_mode, "repeat") == 0) {
            playlist_set_playmode (playlist, PLAYBACK_MODE_REPEAT);
        }
        else if (strcasecmp (str_mode, "random") == 0) {
            playlist_set_playmode (playlist, PLAYBACK_MODE_RANDOM);
        }
    }

    playlist->change_cb = change_cb;

    success = true;
parser_free:
    xml_parser_free_tree (node);
buffer_free:
    ho_free (buffer);

    return success;
}
