
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_event.c 2659 2007-08-22 15:07:45Z mschwerin $
 *
 */

#include "config.h"

#include <assert.h>
#include <pthread.h>

#include "heap.h"
#include "logger.h"
#include "mutex.h"
#include "scheduler.h"
#include "utils.h"

#include "odk.h"
#include "odk_http.h"
#include "odk_joystick.h"
#include "odk_lirc.h"
#include "odk_private.h"

/**
 * This mutex is used to make sure the frontend only gets one event at the
 * time.
 */
pthread_mutex_t frontend_mutex;
pthread_mutexattr_t frontend_mutex_attr;


/// Frees one event handler object.
static void
event_handlers_destroy (void *event_handler_p)
{
    odk_event_handler_t *eh = (odk_event_handler_t *) event_handler_p;
    ho_free (eh);
}


/// Returns TRUE if the event handlers must be swapped to be in order.
static bool
swap_handler (void *h1_p, void *h2_p)
{
    odk_event_handler_t *h1 = (odk_event_handler_t *) h1_p;
    odk_event_handler_t *h2 = (odk_event_handler_t *) h2_p;

    return (h1->priority < h2->priority);
}


void
odk_add_event_handler (odk_t * odk, event_handler_t eh, void *data,
                       odk_event_handler_priority_t priority)
{
    assert (eh);

    mutex_lock (&frontend_mutex);

    odk_event_handler_t *cur = l_list_first (odk->event_handlers);
    while (cur) {
        if (cur->event_handler == eh) {
            goto out_unlock;
        }
        cur = l_list_next (odk->event_handlers, cur);
    }

    odk_event_handler_t *handler = ho_new (odk_event_handler_t);
    handler->event_handler = eh;
    handler->event_handler_data = data;
    handler->priority = priority;

    l_list_append (odk->event_handlers, handler);
    l_list_sort (odk->event_handlers, swap_handler);

  out_unlock:
    mutex_unlock (&frontend_mutex);
}


void
odk_del_event_handler (odk_t * odk, event_handler_t eh)
{
    assert (eh);

    mutex_lock (&frontend_mutex);

    odk_event_handler_t *cur = l_list_first (odk->event_handlers);
    while (cur) {
        if (cur->event_handler == eh) {
            l_list_remove (odk->event_handlers, cur);
            ho_free (cur);
            goto out_unlock;
        }
        cur = l_list_next (odk->event_handlers, cur);
    }

  out_unlock:
    mutex_unlock (&frontend_mutex);
}


bool
odk_get_forward_events_to_xine (odk_t * odk)
{
    return odk->forward_events_to_xine;
}


void
odk_set_forward_events_to_xine (odk_t * odk, bool forward)
{
    odk->forward_events_to_xine = forward;
}


void
odk_xine_event_send (odk_t * odk, bool force, int type)
{
    if (!odk->forward_events_to_xine && !force)
        return;

    xine_event_t ev;
    ev.type = type;
    ev.data = NULL;
    ev.data_length = 0;
    xine_event_send (odk->main_stream, &ev);
}


void
odk_oxine_event_send (odk_t * odk, oxine_event_t * event)
{
    mutex_lock (&frontend_mutex);

    odk_event_handler_t *handler = l_list_first (odk->event_handlers);
    while (handler) {
        handler->event_handler (handler->event_handler_data, event);
        handler = l_list_next (odk->event_handlers, handler);
    }

    mutex_unlock (&frontend_mutex);
}


/// Scales the mouse position to the current video size.
static void
scale_mouse_position (odk_t * odk, int *x, int *y)
{
    /* Scale position. */
    x11_rectangle_t rect;
    rect.x = *x;
    rect.y = *y;
    rect.w = 0;
    rect.h = 0;

    if (xine_port_send_gui_data (odk->win->video_port,
                                 XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                                 (void *) &rect) == -1) {
        return;
    }

    *x = rect.x;
    *y = rect.y;
}


/**
 * Handles events coming from the current video output window and the input
 * plugins (e.g. LIRC, joystick).
 */
static void
odk_event_handler (void *odk_p, oxine_event_t * event)
{
    odk_t *odk = (odk_t *) odk_p;

    switch (event->type) {
    case OXINE_EVENT_OUTPUT_FORMAT_CHANGED:
        odk_osd_adapt_size (odk, NULL);
        break;
    case OXINE_EVENT_BUTTON:
    case OXINE_EVENT_MOTION:
        /* Send an event to the xine engine. */
        if (odk->forward_events_to_xine) {
            int vx = event->data.mouse.pos.x;
            int vy = event->data.mouse.pos.y;
            scale_mouse_position (odk, &vx, &vy);

            xine_input_data_t inp;
            inp.x = vx;
            inp.y = vy;
            inp.button = event->source.button;

            xine_event_t xine_event;
            xine_event.type = event->type;
            xine_event.data = &inp;
            xine_event.data_length = sizeof (inp);

            xine_event_send (odk->main_stream, &xine_event);
        }

        /* Scale x and y to video size. */
        if (!odk_osd_use_unscaled_osd (odk)) {
            int vx = event->data.mouse.pos.x;
            int vy = event->data.mouse.pos.y;
            scale_mouse_position (odk, &vx, &vy);

            event->data.mouse.pos.x = vx;
            event->data.mouse.pos.y = vy;
        }

        /* Scale x any y to OSD size. */
        {
            event->data.mouse.pos.x =
                (int) ((double) (event->data.mouse.pos.x - odk->osd.x_off) /
                       odk->osd.hscale);
            event->data.mouse.pos.y =
                (int) ((double) (event->data.mouse.pos.y - odk->osd.y_off) /
                       odk->osd.vscale);
        }

        break;
    case OXINE_EVENT_KEY:
        /* If we're currently playing television we remap some keys. */
        if (odk_current_is_television (odk)) {
            switch (event->source.key) {
            case OXINE_KEY_MENU2:
                event->source.key = OXINE_KEY_CHANNELS;
                break;
            case OXINE_KEY_MENU3:
                event->source.key = OXINE_KEY_SCHEDULE;
                break;
            case OXINE_KEY_MENU4:
                event->source.key = OXINE_KEY_RECORDINGS;
                break;
            default:
                break;
            }
        }

        /* If we're currently playing television or images we remap some
         * more keys. */
        if (odk_current_is_television (odk) || odk_current_is_image (odk)) {
            switch (event->source.key) {
            case OXINE_KEY_MENU5:
                event->source.key = OXINE_KEY_RED;
                break;
            case OXINE_KEY_MENU6:
                event->source.key = OXINE_KEY_GREEN;
                break;
            case OXINE_KEY_MENU7:
                event->source.key = OXINE_KEY_YELLOW;
                break;
            case OXINE_KEY_MENU8:
                event->source.key = OXINE_KEY_BLUE;
                break;
            default:
                break;
            }
        }

        /* Send an event to the xine engine. */
        if (odk->forward_events_to_xine
            && (event->source.key < OXINE_KEY_NULL)) {
            xine_event_t xine_event;

            xine_event.stream = odk->main_stream;
            xine_event.type = event->source.key;
            xine_event.data = NULL;
            xine_event.data_length = 0;

            xine_event_send (odk->main_stream, &xine_event);
        }
        break;
    default:
        break;
    }

    /* Send the event to the frontend. */
    odk_oxine_event_send (odk, event);
}


/// Handles events coming from the xine engine.
static void
xine_event_handler (void *p, const xine_event_t * xine_event)
{
    odk_t *odk = (odk_t *) p;

#ifdef DEBUG
    static bool info_printed = false;
    if (!info_printed) {
        debug ("   xine-lib thread: 0x%X", (int) pthread_self ());
        info_printed = true;
    }
#endif

    switch (xine_event->type) {
    case XINE_EVENT_FRAME_FORMAT_CHANGE:
        {
            xine_format_change_data_t *data =
                (xine_format_change_data_t *) xine_event->data;
            odk_osd_adapt_size (odk, data);
        }
        break;
    case XINE_EVENT_UI_PLAYBACK_FINISHED:
        if (odk->current_mode == ODK_MODE_NULL)
            return;
        if (odk->current_mode == ODK_MODE_LOGO)
            return;
        if (xine_event->stream == odk->background_stream)
            return;
        if (odk_current_is_image (odk))
            return;

        /* Restart the animation stream */
        if (xine_event->stream == odk->animation_stream) {
            xine_play (odk->animation_stream, 0, 0);
            return;
        }

        /* The main stream has ended. */
        if (xine_event->stream == odk->main_stream) {
            /* If we have alternative streams waiting we play one of them. */
            if (playlist_length (odk->current_alternatives) > 0) {
                bool success = false;
                playitem_t *item = NULL;
                playlist_t *list = odk->current_alternatives;
                playlist_t *copy = playlist_new (NULL, NULL);

                oxine_event_t ev;
                ev.type = OXINE_EVENT_WAITING_FOR_ALTERNATIVE;
                odk_oxine_event_send (odk, &ev);

                /* We have to lock the frontend mutex here, because we don't
                 * want another thread to enter odk_play_stream while we're
                 * in that function. This could happen, if the user stops
                 * the current stream. */
                mutex_lock (&frontend_mutex);

                /* We make a copy of the alternatives, because the
                 * alternatives of the current title are deleted by
                 * odk_play_stream. */
                playlist_lock (odk->current_alternatives);
                item = playlist_first (list);
                while (item) {
                    playlist_add (copy, item->title, item->mrl); 
                    item = playlist_next (list, item);
                }
                playlist_unlock (odk->current_alternatives);

                /* Now we try to play one of the alternatives from
                 * the list we just copied. */
                item = playlist_first (copy);
                while (!success && item) {
                    success = odk_play_stream (odk, item->title, item->mrl,
                                               NULL, 0, odk->current_mode);
                    item = playlist_next (copy, item);
                }
                playlist_free (copy);

                mutex_unlock (&frontend_mutex);

                /* If we were not able to play one of the alternative
                 * streams, we send a stop event. */
                if (!success) {
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_PLAYBACK_ERROR;
                    odk_oxine_event_send (odk, &ev);
                }
            }

            /* Else we inform the frontend about the ending of the stream. */
            else {
                odk->current_mode = ODK_MODE_NULL;
                oxine_event_t ev;
                ev.type = xine_event->type;
                odk_oxine_event_send (odk, &ev);
            }
        }
        break;
    case XINE_EVENT_UI_CHANNELS_CHANGED:
        /* If current mode is still ODK_MODE_NULL we ignore this event, as
         * we're going to send a OXINE_EVENT_PLAYBACK_STARTED event after
         * setting the new mode in odk_mutexed_play anyway. */
        if (odk->current_mode == ODK_MODE_NULL)
            return;

        /* We use this event to find out if the meta info has changed
         * without a new track starting (e.g. web-radio). */
        if (xine_event->stream == odk->main_stream) {
            char *current_title = odk_get_meta_info (odk, META_INFO_TITLE);

            if (!current_title && odk->current_title) {
                oxine_event_t ev;
                ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                odk_oxine_event_send (odk, &ev);
            }

            else if (current_title && odk->current_title
                     && (strcmp (current_title, odk->current_title) != 0)) {
                ho_free (odk->current_title);
                odk->current_title = current_title;
                oxine_event_t ev;
                ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                odk_oxine_event_send (odk, &ev);
            }

            else if (current_title) {
                ho_free (current_title);
            }
        }
        break;
    case XINE_EVENT_PROGRESS:
        if (xine_event->stream == odk->main_stream) {
            xine_progress_data_t *pevent =
                (xine_progress_data_t *) xine_event->data;

            /* Actually we should send this event to the frontend. But as
             * these events are generated by xine-lib while still in
             * xine_play and because the frontend mutex is locked by the
             * event handler thread when we're in xine_play this would lead
             * to a deadlock. */
#if 0
            oxine_event_t ev;
            ev.type = OXINE_EVENT_PROGRESS;
            ev.data.progress.percent = pevent->percent;
            ev.data.progress.description = pevent->description;
            odk_oxine_event_send (odk, &ev);
#else
            debug ("%s %3d %%", pevent->description, pevent->percent);
#endif
        }
        break;
    case XINE_EVENT_MRL_REFERENCE:
        /* If we receive an alternative stream, we add it to a list of
         * alternatives. We have to lock the frontend mutex before inserting
         * into the alternatives list, because we might deadlock otherwise. */
        if (xine_event->stream == odk->main_stream) {
            xine_mrl_reference_data_t *ref =
                (xine_mrl_reference_data_t *) xine_event->data;

            debug ("Got alternative MRL [%d]: '%s'",
                   ref->alternative, ref->mrl);

            mutex_lock (&frontend_mutex);

            playlist_insert (odk->current_alternatives, ref->alternative,
                             odk->current_title, ref->mrl);

            mutex_unlock (&frontend_mutex);
        }
        break;
    case XINE_EVENT_UI_MESSAGE:
#ifdef DEBUG
        if (xine_event->stream == odk->main_stream) {
            xine_ui_message_data_t *msg =
                (xine_ui_message_data_t *) xine_event->data;

            debug ("Got message from xine-lib: '%s'", msg->messages);
        }
#endif
        break;
    default:
        break;
    }
}


void
odk_listen_to (odk_t * odk, const char *id)
{
#ifdef HAVE_LIRC
    if (!odk->lirc && (strcmp (id, "lirc") == 0)) {
        odk->lirc = start_lirc (odk, odk_event_handler);
    }
#endif /* HAVE_LIRC */
#ifdef HAVE_JOYSTICK
    if (!odk->joystick && (strcmp (id, "joystick") == 0)) {
        odk->joystick = start_joystick (odk, odk_event_handler);
    }
#endif /* HAVE_JOYSTICK */
#ifdef HAVE_HTTP
    if (!odk->http && (strcmp (id, "http") == 0)) {
        odk->http = start_http (odk, odk_event_handler);
    }
#endif /* HAVE_HTTP */
}


void
odk_event_init (odk_t * odk)
{
#ifdef DEBUG
    assert (odk);
    assert (odk->win);
    assert (odk->main_stream);
    assert (odk->background_stream);
    assert (odk->animation_stream);
#endif

    pthread_mutexattr_init (&frontend_mutex_attr);
    pthread_mutexattr_settype (&frontend_mutex_attr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&frontend_mutex, &frontend_mutex_attr);

    odk->event_handlers = l_list_new ();
    odk->forward_events_to_xine = false;
    odk->win->set_event_handler (odk->win, odk_event_handler, odk);

    xine_event_queue_t *q0 = xine_event_new_queue (odk->main_stream);
    xine_event_queue_t *q1 = xine_event_new_queue (odk->background_stream);
    xine_event_queue_t *q2 = xine_event_new_queue (odk->animation_stream);

    xine_event_create_listener_thread (q0, xine_event_handler, odk);
    xine_event_create_listener_thread (q1, xine_event_handler, odk);
    xine_event_create_listener_thread (q2, xine_event_handler, odk);
}


void
odk_event_free (odk_t * odk)
{
#ifdef HAVE_LIRC
    if (odk->lirc) {
        stop_lirc (odk->lirc);
        odk->lirc = NULL;
    }
#endif
#ifdef HAVE_JOYSTICK
    if (odk->joystick) {
        stop_joystick (odk->joystick);
        odk->joystick = NULL;
    }
#endif
#ifdef HAVE_HTTP
    if (odk->http) {
        stop_http (odk->http);
        odk->http = NULL;
    }
#endif /* HAVE_HTTP */
    if (odk->event_handlers) {
        l_list_free (odk->event_handlers, event_handlers_destroy);
        odk->event_handlers = NULL;
    }

    pthread_mutex_destroy (&frontend_mutex);
    pthread_mutexattr_destroy (&frontend_mutex_attr);
}
