
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: logger.c 2403 2007-07-02 15:26:08Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <execinfo.h>
#include <errno.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "utils_files.h"

#define DEFAULT_LOGFILE "/tmp/oxine.log"
static char *logfile = NULL;


static void
print_line_head (FILE * f, logger_level_t level, const char *file, int line,
                 struct tm *tm)
{
    const char *level_str[] = {
        "FATAL", "ERROR", "WARN", "INFO", "DEBUG", "TODO"
    };

#ifdef DEBUG
    fprintf (f, "[%d-%02d-%02d %02d:%02d:%02d] [%X] [%15.15s:%4d] %5s: ",
             tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour,
             tm->tm_min, tm->tm_sec, (int) pthread_self (), file, line,
             level_str[level]);
#else
    fprintf (f, "[%d-%02d-%02d %02d:%02d:%02d] [%15.15s:%4d] %5s: ",
             tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour,
             tm->tm_min, tm->tm_sec, file, line, level_str[level]);
#endif
}


static void
logger_log_rotate (const char *filename)
{
    const time_t now = time (NULL);
    struct tm *tm_now = localtime (&now);

    int i = 5;
    while (i >= 0) {
        char *fo = ho_strdup_printf ("%s.%d", filename, i);
        char *fn = ho_strdup_printf ("%s.%d", filename, i + 1);

        if (file_exists (fo) && (rename (fo, fn) != 0)) {
            print_line_head (stderr, LEVEL_ERROR, __FILE__, __LINE__, tm_now);
            fprintf (stderr, _("Could not move '%s' to '%s': %s!"),
                     fo, fn, strerror (errno));
            fprintf (stderr, "\n");
        }

        ho_free (fo);
        ho_free (fn);

        i -= 1;
    }

    {
        char *fn = ho_strdup_printf ("%s.0", filename);
        if (file_exists (filename) && (rename (filename, fn) != 0)) {
            print_line_head (stderr, LEVEL_ERROR, __FILE__, __LINE__, tm_now);
            fprintf (stderr, _("Could not move '%s' to '%s': %s!"),
                     filename, fn, strerror (errno));
            fprintf (stderr, "\n");
        }
        ho_free (fn);
    }
}


void
logger_init (const char *filename)
{
    const time_t now = time (NULL);
    struct tm *tm_now = localtime (&now);

    /* We try to open the provided filename. */
    if (filename) {
        logfile = ho_strdup (filename);
        FILE *f = fopen (logfile, "a");
        if (!f) {
            print_line_head (stderr, LEVEL_ERROR, __FILE__, __LINE__, tm_now);
            fprintf (stderr, _("Could not open '%s': %s!"), logfile,
                     strerror (errno));
            fprintf (stderr, "\n");
            ho_free (logfile);
        }
    }

    /* If no filename was provided or we could not open the provided filename,
     * we fall back to the default logfile. */
    if (!logfile) {
        if (!filename || (strcmp (filename, DEFAULT_LOGFILE) != 0)) {
            logger_init (DEFAULT_LOGFILE);
        }
        return;
    }

    /* Rotate the log files. */
    logger_log_rotate (logfile);
}


void
logger_free (void)
{
    ho_free (logfile);
    logfile = NULL;
}


void
_log (logger_level_t level,
      const char *file, int line, const char *format, ...)
{
    const time_t now = time (NULL);
    struct tm *tm_now = localtime (&now);

    FILE *fcon = stdout;
    if (level < LEVEL_INFO) {
        fcon = stderr;
    }

    FILE *flog = NULL;
    if (logfile) {
        flog = fopen (logfile, "a");
        /* We tried to catch any problems with the logfile at the top. If we
         * get problems here we fail quietly and give up. */
        if (!flog) {
            ho_free (logfile);
        }
    }

    if (fcon) {
        va_list args;
        print_line_head (fcon, level, file, line, tm_now);
        va_start (args, format);
        vfprintf (fcon, format, args);
        va_end (args);
        fprintf (fcon, "\n");
    }

    if (flog) {
        va_list args;
        print_line_head (flog, level, file, line, tm_now);
        va_start (args, format);
        vfprintf (flog, format, args);
        va_end (args);
        fprintf (flog, "\n");
    }

    if (level == LEVEL_FATAL) {
        void *array[10];
        size_t i;

        size_t size = backtrace (array, 10);
        char **strings = backtrace_symbols (array, size);

        if (!strings) {
            return;
        }

        print_line_head (fcon, level, file, line, tm_now);
        fprintf (fcon, "A fatal error occurred! Printing backtrace:\n");
        for (i = 0; i < size; i++) {
            print_line_head (fcon, level, file, line, tm_now);
            fprintf (fcon, "     %s\n", strings[i]);
        }

        free (strings);
    }

    if (flog) {
        fclose (flog);
    }
}
