
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: event.h 2465 2007-07-07 15:37:46Z mschwerin $
 *
 */

#ifndef HAVE_EVENT_H
#define HAVE_EVENT_H

#include <xine.h>

/// These are the possible oxine event types.
typedef enum {
    /* ***********************************************
     * xine-lib events going to the frontend
     * *********************************************** */
    /// Playback has finished.
    OXINE_EVENT_PLAYBACK_FINISHED = XINE_EVENT_UI_PLAYBACK_FINISHED,
    OXINE_EVENT_CHANNELS_CHANGED = XINE_EVENT_UI_CHANNELS_CHANGED,
    OXINE_EVENT_SET_TITLE = XINE_EVENT_UI_SET_TITLE,
    /// xine-lib wants to send us a message.
    OXINE_EVENT_MESSAGE = XINE_EVENT_UI_MESSAGE,
    /// The format of the video output has changed.
    OXINE_EVENT_FRAME_FORMAT_CHANGED = XINE_EVENT_FRAME_FORMAT_CHANGE,
    /* ***********************************************
     * xine-lib events coming from the frontend 
     * *********************************************** */
    /// A mouse button was pressed.
    OXINE_EVENT_BUTTON = XINE_EVENT_INPUT_MOUSE_BUTTON,
    /// The mouse was moved.
    OXINE_EVENT_MOTION = XINE_EVENT_INPUT_MOUSE_MOVE,
    /* ***********************************************
     * oxine events going to the frontend
     * *********************************************** */
    /// Playback was started.
    OXINE_EVENT_PLAYBACK_STARTED = 1000,
    /// Playback was stopped by the user.
    OXINE_EVENT_PLAYBACK_STOPPED,
    /// There bas a playback error.
    OXINE_EVENT_PLAYBACK_ERROR,
    /// We're waiting for an alternative stream.
    OXINE_EVENT_WAITING_FOR_ALTERNATIVE,
    /// A keypress (keyboard, LIRC) event.
    OXINE_EVENT_KEY,
    /// The size of the output window has changed.
    OXINE_EVENT_OUTPUT_FORMAT_CHANGED,
    /// The format of the OSD has changed.
    OXINE_EVENT_OSD_FORMAT_CHANGED,
    /// A configuration entry has changed.
    OXINE_EVENT_CONFIG_CHANGED,
    /// Progress has been made.
    OXINE_EVENT_PROGRESS,
    /// The GUI should be repainted.
    OXINE_EVENT_GUI_REPAINT,
    /// The GUI is to be hidden.
    OXINE_EVENT_GUI_HIDE,

    /**
     * This event should be sent once in a while (actually at least twice
     * every second) to give our widget toolkit a chance of updating any
     * widget that might need a repaint.
     */
    OXINE_EVENT_OTK_UPDATE,
    /// This is used in the filelist menu to update the thumbnails.
    OXINE_EVENT_THUMBNAIL_UPDATE,
} oxine_event_id_t;


/// These are the possible key events.
typedef enum {
    /* mapped xine keys */
    OXINE_KEY_MENU1 = XINE_EVENT_INPUT_MENU1,
    OXINE_KEY_MENU2 = XINE_EVENT_INPUT_MENU2,
    OXINE_KEY_MENU3 = XINE_EVENT_INPUT_MENU3,
    OXINE_KEY_MENU4 = XINE_EVENT_INPUT_MENU4,
    OXINE_KEY_MENU5 = XINE_EVENT_INPUT_MENU5,
    OXINE_KEY_MENU6 = XINE_EVENT_INPUT_MENU6,
    OXINE_KEY_MENU7 = XINE_EVENT_INPUT_MENU7,
    OXINE_KEY_UP = XINE_EVENT_INPUT_UP,
    OXINE_KEY_DOWN = XINE_EVENT_INPUT_DOWN,
    OXINE_KEY_LEFT = XINE_EVENT_INPUT_LEFT,
    OXINE_KEY_RIGHT = XINE_EVENT_INPUT_RIGHT,
    OXINE_KEY_ACTIVATE = XINE_EVENT_INPUT_SELECT,
    OXINE_KEY_NEXT = XINE_EVENT_INPUT_NEXT,
    OXINE_KEY_PREV = XINE_EVENT_INPUT_PREVIOUS,
    OXINE_KEY_0 = XINE_EVENT_INPUT_NUMBER_0,
    OXINE_KEY_1 = XINE_EVENT_INPUT_NUMBER_1,
    OXINE_KEY_2 = XINE_EVENT_INPUT_NUMBER_2,
    OXINE_KEY_3 = XINE_EVENT_INPUT_NUMBER_3,
    OXINE_KEY_4 = XINE_EVENT_INPUT_NUMBER_4,
    OXINE_KEY_5 = XINE_EVENT_INPUT_NUMBER_5,
    OXINE_KEY_6 = XINE_EVENT_INPUT_NUMBER_6,
    OXINE_KEY_7 = XINE_EVENT_INPUT_NUMBER_7,
    OXINE_KEY_8 = XINE_EVENT_INPUT_NUMBER_8,
    OXINE_KEY_9 = XINE_EVENT_INPUT_NUMBER_9,
    /* mapped VDR keys */
    OXINE_KEY_RECORD = XINE_EVENT_VDR_RECORD,
    OXINE_KEY_RED = XINE_EVENT_VDR_RED,
    OXINE_KEY_GREEN = XINE_EVENT_VDR_GREEN,
    OXINE_KEY_YELLOW = XINE_EVENT_VDR_YELLOW,
    OXINE_KEY_BLUE = XINE_EVENT_VDR_BLUE,
    OXINE_KEY_CHANNELS = XINE_EVENT_VDR_CHANNELS,
    OXINE_KEY_SCHEDULE = XINE_EVENT_VDR_SCHEDULE,
    OXINE_KEY_RECORDINGS = XINE_EVENT_VDR_RECORDINGS,
    /* boundary */
    OXINE_KEY_NULL = 1000,
    /* stream */
    OXINE_KEY_PPLAY,
    OXINE_KEY_PLAY,
    OXINE_KEY_PAUSE,
    OXINE_KEY_STOP,
    /// Go to an absolute position in the stream.
    OXINE_KEY_SEEK,
    /// Skip in the stream relative to the current position.
    OXINE_KEY_SKIP,
    OXINE_KEY_FASTFWD,
    OXINE_KEY_FASTRWD,
    OXINE_KEY_SPEED,
    /* audio settings */
    OXINE_KEY_VOLUME,
    OXINE_KEY_VOLUME_MUTE,
    OXINE_KEY_AUDIO_CHANNEL,
    OXINE_KEY_AUDIO_OFFSET,
    /* video settings */
    OXINE_KEY_SATURATION,
    OXINE_KEY_BRIGHTNESS,
    OXINE_KEY_CONTRAST,
    OXINE_KEY_HUE,
    OXINE_KEY_ZOOM,
    OXINE_KEY_ZOOM_X,
    OXINE_KEY_ZOOM_Y,
    OXINE_KEY_ZOOM_RESET,
    OXINE_KEY_ASPECT_RATIO,
    OXINE_KEY_DEINTERLACE,
    OXINE_KEY_SPU_CHANNEL,
    OXINE_KEY_SPU_OFFSET,
    /* OSD navigation */
    OXINE_KEY_SHOW_OSD,
    OXINE_KEY_FIRST,
    OXINE_KEY_LAST,
    OXINE_KEY_PAGE_UP,
    OXINE_KEY_PAGE_DOWN,
    OXINE_KEY_BACK,
    OXINE_KEY_HOME,
    OXINE_KEY_SELECT,
    OXINE_KEY_PREV_WIDGET,
    OXINE_KEY_NEXT_WIDGET,
    /* menu access */
    OXINE_KEY_MENU_HELP,
    OXINE_KEY_MENU_MAIN,
    OXINE_KEY_MENU_MUSIC,
    OXINE_KEY_MENU_VIDEO,
    OXINE_KEY_MENU_IMAGE,
    OXINE_KEY_MENU_PLAYBACK,
    OXINE_KEY_MENU_PLAYLIST,
    OXINE_KEY_MENU_SETTINGS,
    OXINE_KEY_MENU_WEATHER,
    OXINE_KEY_TELEVISION,
    /* misc */
    OXINE_KEY_MENU8,
    OXINE_KEY_SAVE,
    OXINE_KEY_INSERT,
    OXINE_KEY_REMOVE,
    OXINE_KEY_CUT,
    OXINE_KEY_PASTE,
    OXINE_KEY_COPY,
    OXINE_KEY_EJECT,
    OXINE_KEY_PLAYMODE,
    OXINE_KEY_FULLSCREEN,
    OXINE_KEY_QUIT,
    OXINE_KEY_SHUTDOWN,
    OXINE_KEY_ROTATE_LEFT,
    OXINE_KEY_ROTATE_RIGHT,
    /// Do not map the input key to an event.
    OXINE_KEY_INPUT
} oxine_key_id_t;


typedef enum {
    /// No button was pressed.
    OXINE_MOUSE_BUTTON_NULL,
    /// Mouse button 1 was pressed.
    OXINE_MOUSE_BUTTON_1,
    OXINE_MOUSE_BUTTON_LEFT = OXINE_MOUSE_BUTTON_1,
    /// Mouse button 2 was pressed.
    OXINE_MOUSE_BUTTON_2,
    OXINE_MOUSE_BUTTON_MIDDLE = OXINE_MOUSE_BUTTON_2,
    /// Mouse button 3 was pressed.
    OXINE_MOUSE_BUTTON_3,
    OXINE_MOUSE_BUTTON_RIGHT = OXINE_MOUSE_BUTTON_3,
    /// Mouse button 4 was pressed.
    OXINE_MOUSE_BUTTON_4,
    OXINE_MOUSE_SCROLLWHEEL_UP = OXINE_MOUSE_BUTTON_4,
    /// Mouse button 5 was pressed.
    OXINE_MOUSE_BUTTON_5,
    OXINE_MOUSE_SCROLLWHEEL_DOWN = OXINE_MOUSE_BUTTON_5,
    /// Mouse button 6 was pressed.
    OXINE_MOUSE_BUTTON_6,
    /// Mouse button 7 was pressed.
    OXINE_MOUSE_BUTTON_7,
} oxine_button_id_t;


/// This represents the source of an event.
typedef union {
    /// The key that was pressed.
    oxine_key_id_t key;
    /// The mouse button that was pressed.
    oxine_button_id_t button;
} oxine_event_source_t;


/// This represents the data of an event.
typedef union {
    /// Information about a mouse event.
    struct {
        /// Information about the position of the mouse pointer.
        struct {
            int x;
            int y;
        } pos;
        /// Information about the mouse pointer.
        struct {
            int num;
            int modifier;
        } button;
    } mouse;

    /// Information about a keyboard event.
    struct {
        int key;
        int modifier;
    } keyboard;

    struct {
        xine_cfg_entry_t *entry;
    } config;

    struct {
        int percent;
        const char *description;
    } progress;

    void *user_data;
} oxine_event_data_t;


/// This represents an oxine event.
typedef struct {
    /// The event type.
    oxine_event_id_t type;
    /// The source of this event.
    oxine_event_source_t source;
    /// The data of this event.
    oxine_event_data_t data;

    /**
     * How much are we to change the destination parameter. When, for
     * example, changing the volume this specifies the amount of change to
     * apply.
     */
    int how;
} oxine_event_t;

typedef void (*event_handler_t) (void *user_data, oxine_event_t * event);

void oxine_event_handler (void *oxine_p, oxine_event_t * ev);

#endif /* HAVE_EVENT_H */
