// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Polyform License
// that can be found in the LICENSE file.

package auths

import (
	"bytes"
	"encoding/base64"
	"os"
	"testing"

	"github.com/drone/drone-go/drone"
	"github.com/google/go-cmp/cmp"
)

func TestParse(t *testing.T) {
	got, err := ParseString(sample)
	if err != nil {
		t.Error(err)
		return
	}
	want := []*drone.Registry{
		{
			Address:  "index.docker.io",
			Username: "octocat",
			Password: "correct-horse-battery-staple",
		},
	}
	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
	}
}

func TestParseBytes(t *testing.T) {
	got, err := ParseBytes([]byte(sample))
	if err != nil {
		t.Error(err)
		return
	}
	want := []*drone.Registry{
		{
			Address:  "index.docker.io",
			Username: "octocat",
			Password: "correct-horse-battery-staple",
		},
	}
	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
	}
}

func TestParseErr(t *testing.T) {
	_, err := ParseString("")
	if err == nil {
		t.Errorf("Expect unmarshal error")
	}
}

func TestParseFile(t *testing.T) {
	got, err := ParseFile("./testdata/config.json")
	if err != nil {
		t.Error(err)
		return
	}
	want := []*drone.Registry{
		{
			Address:  "index.docker.io",
			Username: "octocat",
			Password: "correct-horse-battery-staple",
		},
	}
	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
	}
}

func TestParseUsernamePassword(t *testing.T) {
	got, err := ParseString(sample2)
	if err != nil {
		t.Error(err)
		return
	}
	want := []*drone.Registry{
		{
			Address:  "index.docker.io",
			Username: "octocat",
			Password: "correct-horse-battery-staple",
		},
	}
	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
	}
}

func TestParseFileErr(t *testing.T) {
	_, err := ParseFile("./testdata/x.json")
	if _, ok := err.(*os.PathError); !ok {
		t.Errorf("Expect error when file does not exist")
	}
}

func TestEncodeDecode(t *testing.T) {
	username := "octocat"
	password := "correct-horse-battery-staple"

	encoded := encode(username, password)
	decodedUsername, decodedPassword := decode(encoded)
	if got, want := decodedUsername, username; got != want {
		t.Errorf("Want decoded username %s, got %s", want, got)
	}
	if got, want := decodedPassword, password; got != want {
		t.Errorf("Want decoded password %s, got %s", want, got)
	}
}

func TestDecodeInvalid(t *testing.T) {
	username, password := decode("b2N0b2NhdDp==")
	if username != "" || password != "" {
		t.Errorf("Expect decoding error")
	}
}

func TestHeader(t *testing.T) {
	username := "octocat"
	password := "correct-horse-battery-staple"
	result := Header(username, password)
	got, err := base64.URLEncoding.DecodeString(result)
	if err != nil {
		t.Error(err)
		return
	}
	want := []byte(`{"username":"octocat","password":"correct-horse-battery-staple"}`)
	if bytes.Equal(got, want) == false {
		t.Errorf("Could not decode credential header")
	}
}

var sample = `{
	"auths": {
		"https://index.docker.io/v1/": {
			"auth": "b2N0b2NhdDpjb3JyZWN0LWhvcnNlLWJhdHRlcnktc3RhcGxl"
		}
	}
}`

var sample2 = `{
	"auths": {
		"https://index.docker.io/v1/": {
			"username": "octocat",
			"password": "correct-horse-battery-staple"
		}
	}
}`
