/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Error_Dialog()
 *
 *  Opens an error dialog box
 */
  void
Error_Dialog( void )
{
  GtkBuilder *builder;
  Cleanup();
  if( error_dialog == NULL )
  {
    error_dialog = create_error_dialog( &builder );
    gtk_widget_show( error_dialog );
    g_object_unref( builder );
  }
} /*  Error_Dialog() */

/*------------------------------------------------------------------------*/

/* File_Chooser()
 *
 * Opens a file chooser dialog
 */
  void
File_Chooser( void )
{
  GtkBuilder *builder;
  if( filechooser == NULL )
  {
    filechooser = create_filechooser( &builder );
    gtk_widget_show( filechooser );
    g_object_unref( builder );
  }
} /*  File_Chooser() */

/*------------------------------------------------------------------------*/

/* Cancel_Timer()
 *
 * Handles cancellation of start-stop timer
 */
  gboolean
Cancel_Timer( gpointer data )
{
  Show_Message(
      _("Cancelling Start - Stop timer"), "black" );
  ClearFlag( START_STOP_TIMER );

  /* Take action according to flags */
  if( isFlagSet(ACTION_RECORD_APT) ||
      isFlagSet(ACTION_PROCESS_DSP) )
  {
    /* Set the buffer synchronize function as idle callback.
     * This will make Dsp_Process_Image() or Record_APT_Signal()
     * as the idle callback if it succeeds in syncing the buffer */
    SetFlag( ACTION_SYNC_BUFFER );
    g_idle_add( Synchronize_Buffer, NULL );

  } /* if( isFlagSet(ACTION_RECORD_APT) || */

  return( FALSE );
} /* Cancel_Timer() */

/*------------------------------------------------------------------------*/

/*  Sensitize_Menu_Item()
 *
 *  Handles dynamic sensitivity changes to pop-up menu items
 */
  void
Sensitize_Menu_Item( gchar *item_name, gboolean flag )
{
  GtkWidget *menu_item;
  menu_item = Builder_Get_Object( popup_menu_builder, item_name );
  gtk_widget_set_sensitive( menu_item, flag );
} /*  Sensitize_Menu_Item() */

/*------------------------------------------------------------------------*/

/*  Popup_Menu()
 *
 *  Opens pop-up menu
 */
  void
Popup_Menu( void )
{
  static int first_call = TRUE;

  /* Initialize on first call */
  if( first_call )
  {
    popup_menu = create_popup_menu( &popup_menu_builder );
    first_call = FALSE;
  }

  /* If any action is running */
  if( isFlagSet(ACTION_FLAGS_ALL) )
  {
    Sensitize_Menu_Item( "select_satellite",     FALSE );
    Sensitize_Menu_Item( "operation_timer",    FALSE );
    Sensitize_Menu_Item( "start_stop_timer",   FALSE );

    if( isFlagSet(START_STOP_TIMER) )
      Sensitize_Menu_Item( "cancel_timer",     TRUE  );
    else
      Sensitize_Menu_Item( "cancel_timer",     FALSE );

    Sensitize_Menu_Item( "decode_from_dsp",    FALSE );
    Sensitize_Menu_Item( "record_to_file",     FALSE );
    Sensitize_Menu_Item( "decode_from_file",   FALSE );
    Sensitize_Menu_Item( "stop", TRUE );
  }
  else /* If no action is running */
  {
    if( isFlagClear(START_STOP_TIMER) )
    {
      Sensitize_Menu_Item( "operation_timer",    TRUE  );
      Sensitize_Menu_Item( "start_stop_timer",   TRUE  );
      Sensitize_Menu_Item( "cancel_timer",       FALSE );
      Sensitize_Menu_Item( "decode_from_file",   TRUE  );
    }
    else
    {
      Sensitize_Menu_Item( "operation_timer",    FALSE );
      Sensitize_Menu_Item( "start_stop_timer",   FALSE );
      Sensitize_Menu_Item( "cancel_timer",       TRUE  );
      Sensitize_Menu_Item( "decode_from_file",   FALSE );
    }

    Sensitize_Menu_Item( "select_satellite",  TRUE  );
    Sensitize_Menu_Item( "decode_from_dsp", TRUE  );
    Sensitize_Menu_Item( "record_to_file",  TRUE  );
    Sensitize_Menu_Item( "stop",            FALSE );

  } /* if( isFlagSet(ACTION_FLAGS_ALL) ) */

  gtk_menu_popup_at_pointer( GTK_MENU(popup_menu), NULL );
} /*  Popup_Menu() */

/*------------------------------------------------------------------------*/

/* Receive_Togglebutton_Toggled()
 *
 * Handles the on_receive_togglebutton_toggled CB
 */
  void
Receive_Togglebutton_Toggled( GtkToggleButton *togglebutton )
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    /* Abort if action pending */
    if( isFlagSet(ACTION_PENDING) ) return;

    /* If not in timeout sleep */
    if( isFlagClear(START_STOP_TIMER) )
    {
      SetFlag( ACTION_SYNC_BUFFER );
      SetFlag( ACTION_PROCESS_DSP );

      /* Set the buffer sync function as idle callback.  */
      /* This will make Dsp_Process_Image() as the idle  */
      /* callback if it succeeds in syncing the buffer.  */
      if( isFlagSet(USE_RTLSDR_RX) )
        Show_Message(
            _("Directly Decoding from RTLSDR DSP"), "black" );
      else
        Show_Message(
            _("Directly Decoding from Sound DSP"), "black" );

      g_idle_add( Synchronize_Buffer, NULL );

    }  /* if( isFlagClear(START_STOP_TIMER) ) */
    else
    {
      /* If in timer period, only set required action */
      if( !isFlagSet(ACTION_FLAGS_ALL) )
      {
        SetFlag( ACTION_PROCESS_DSP );
        Show_Message( _("Suspended: Decode from dsp "), "black" );
      }
    }
  } /* if( gtk_toggle_button_get_active(togglebutton) ) */
  else
    ClearFlag( ACTION_FLAGS_ALL );

} /* Receive_Togglebutton_Toggled() */

/*------------------------------------------------------------------------*/

/* Filechooser_Response()
 *
 * Handles the on_filechooser_response CB
 */
  void
Filechooser_Response( gint response_id )
{
  char file_name[MAX_FILE_NAME];
  int len;

  /* Cancel action on close or cancel */
  if( (response_id == GTK_RESPONSE_DELETE_EVENT) ||
      (response_id == GTK_RESPONSE_CANCEL) )
  {
    ClearFlag( ACTION_FLAGS_ALL );
    gtk_widget_destroy( filechooser );
    return;
  }

  /* Check length of file name */
  len = (int)strlen(gtk_file_chooser_get_filename
      (GTK_FILE_CHOOSER(filechooser)));
  if( len > MAX_FILE_NAME )
  {
    ClearFlag( ACTION_FLAGS_ALL );
    Show_Message( _("File name to long"), "red" );
    Error_Dialog();
    return;
  }

  /* Check for blank or globbed file name */
  Strlcpy( file_name, gtk_file_chooser_get_filename
      (GTK_FILE_CHOOSER(filechooser)), sizeof(file_name) );
  if( (strcmp(basename(file_name), "record") == 0) ||
      (strcmp(basename(file_name), "images") == 0) )
  {
    ClearFlag( ACTION_FLAGS_ALL );
    Show_Message( _("Blank or invalid file name"), "red" );
    Error_Dialog();
    return;
  }

  /* Pass on file name according to action */
  if( isFlagSet(ACTION_PROCESS_DSP) ) /* Decode from dsp */
    Strlcpy( image_file, file_name, sizeof( image_file ) );
  else /* Record or decode samples file */
    Strlcpy( samples_file, file_name, sizeof( samples_file ) );

  /* Take action according to flags */
  if( isFlagClear(START_STOP_TIMER) )
  {
    if( isFlagClear(ACTION_PROCESS_FILE) )
    {
      /* Set the buffer sync function as idle callback. */
      /* This will make the selected action as the idle */
      /* callback if it succeeds in syncing the buffer. */
      if( isFlagSet(ACTION_PROCESS_DSP) )
        Show_Message( _("Directly decoding from dsp "), "black" );
      else if( isFlagSet(ACTION_RECORD_APT) )
        Show_Message( _("Recording apt signal to file "), "black" );

      SetFlag( ACTION_SYNC_BUFFER );
      g_idle_add( Synchronize_Buffer, NULL );

    } /* isFlagClear(ACTION_PROCESS_FILE) */
    else g_idle_add( File_Process_Image, NULL );
  }  /* if( isFlagClear(START_STOP_TIMER) ) */
  else /* If in timeout sleep, only show next action */
  {
    if( isFlagSet(ACTION_PROCESS_DSP) )
      Show_Message( _("Suspended: Decode from dsp "), "black" );
    else if( isFlagSet(ACTION_RECORD_APT) )
      Show_Message( _("Suspended: Record to file "), "black" );

  }  /* if( isFlagClear(START_STOP_TIMER) ) */

  /* Close menu dialog */
  gtk_widget_destroy( filechooser );

} /* Filechooser_Response() */

/*------------------------------------------------------------------------*/

/* Start_Stop_OK_Clicked()
 *
 * Handles the on_start_stop_ok_clicked CB
 *
 */
  void
Start_Stop_OK_Clicked( GtkButton *button )
{
  int
    start_hrs,  /* Start time hours   */
    start_min,  /* Start time minutes */
    stop_hrs,   /* Stop time hours    */
    stop_min,   /* Stop time minutes  */
    sleep_sec,  /* Sleep time in sec  */
    stop_sec,   /* Stop time in sec since 00:00 hrs  */
    start_sec,  /* Start time in sec since 00:00 hrs */
    time_sec;   /* Time now in sec since 00:00 hrs   */

  /* Message string buffer */
  char mesg[MESG_SIZE];

  /* Used to read real time */
  struct tm time_now;
  time_t t;


  /* Extract hours and minutes of start time */
  start_hrs = atoi( gtk_entry_get_text
      (GTK_ENTRY(Builder_Get_Object(startstop_builder, "start_hrs"))) );
  start_min = atoi( gtk_entry_get_text
      ( GTK_ENTRY(Builder_Get_Object(startstop_builder, "start_min"))) );

  /* Time now */
  t = time( &t );
  time_now = *gmtime( &t );
  time_sec = time_now.tm_hour*3600 +
    time_now.tm_min*60 +
    time_now.tm_sec;

  start_sec = start_hrs*3600 + start_min*60;
  sleep_sec = start_sec - time_sec;
  if( sleep_sec < 0 )
    sleep_sec += 86400; /* Next day */

  /* Extract hours and minutes of stop time */
  stop_hrs = atoi( gtk_entry_get_text
      ( GTK_ENTRY(Builder_Get_Object(startstop_builder, "stop_hrs"))) );
  stop_min = atoi( gtk_entry_get_text
      ( GTK_ENTRY(Builder_Get_Object(startstop_builder, "stop_min"))) );

  stop_sec  = stop_hrs*3600 + stop_min*60;
  stop_sec -= time_sec;
  if( stop_sec < 0 )
    stop_sec += 86400; /* Next day */

  /* Difference between start-stop times in sec */
  duration = stop_sec - sleep_sec;

  /* Data sanity check */
  if( duration <= 0 )
  {
    Show_Message( _("Stop time ahead of start time"), "red" );
    Error_Dialog();
    return;
  }

  if( duration > 720 )
  {
    snprintf( mesg, MESG_SIZE,
        _("Duration (%d sec) seems excessive"), duration );
    Show_Message( mesg, "red" );
    Error_Dialog();
    return;
  }

  gtk_widget_destroy( startstop_timer );

  /* Notify sleeping */
  snprintf( mesg, MESG_SIZE,
      _("Suspended till %02d:%02d\n"\
        "Operation timer set to %d sec"),
      start_hrs, start_min, duration );
  Show_Message( mesg, "black" );

  /* Set sleep flag and wakeup timeout */
  SetFlag( START_STOP_TIMER );
  cancel_timer_tag = g_timeout_add
    ( (guint)(sleep_sec * 1000), Cancel_Timer, NULL );

  Initialize_Top_Window();

} /* Start_Stop_OK_Clicked() */

/*------------------------------------------------------------------------*/

/* Enter_Center_Freq()
 *
 * Enters the center frequency to relevant
 * entry widget and displays in messages
 */
  void
Enter_Center_Freq( uint32_t freq )
{
  char text[12];

  /* Get center freq value in Hz, entry value is in kHz */
  GtkEntry *entry = GTK_ENTRY(
      Builder_Get_Object(main_window_builder, "freq_entry") );
  snprintf( text, sizeof(text), "%8.1f", (double)freq / 1000.0 );
  gtk_entry_set_text( entry, text );

} /* Enter_Center_Freq() */

/*------------------------------------------------------------------------*/

/* Timeout_Setup()
 *
 * Handles on_timeout_okbutton_clicked CB
 */
  void
Timeout_Setup( GtkButton *button )
{
  GtkWidget *spinbutton;
  char mesg[MESG_SIZE];

  /* Set processing duration in sec */
  spinbutton = Builder_Get_Object( timer_dialog_builder, "spinbutton" );
  duration = 60 * gtk_spin_button_get_value_as_int(
      GTK_SPIN_BUTTON(spinbutton) );

  /* Warn if duration is too long */
  if( duration > 720 )
  {
    snprintf( mesg, MESG_SIZE,
        _("Duration (%d sec) seems excessive"), duration );
    Show_Message( mesg, "red" );
    Error_Dialog();
  }

  gtk_widget_destroy( timer_dialog );
  Initialize_Top_Window();

} /* Timeout_Setup() */

/*------------------------------------------------------------------------*/

/* Hours_Entry()
 *
 * Handles on_hrs_entry_changed CB
 */
  void
Hours_Entry( GtkEditable *editable )
{
  gchar buff[3];
  int idx, len;

  /* Get user entry */
  Strlcpy( buff,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(buff) );
  len = (int)strlen( buff );

  /* Reject non-numeric entries */
  for( idx = 0; idx < len; idx++ )
    if( (buff[idx] < '0') || (buff[idx] > '9') )
    {
      Show_Message( _("Non-numeric entry"), "red" );
      Error_Dialog();
      return;
    }

  /* Reject invalid entries */
  idx = atoi( gtk_entry_get_text(GTK_ENTRY(editable)) );
  if( (idx < 0) || (idx > 23) )
  {
    Show_Message( _("Value out of range"), "red" );
    Error_Dialog();
    return;
  }

} /* Hours_Entry() */

/*------------------------------------------------------------------------*/

/* Minutes_Entry()
 *
 * Handles the on_min_entry_changed CB
 */
void
Minutes_Entry( GtkEditable *editable )
{
  gchar buff[3];
  int idx, len;

  /* Get user entry */
  Strlcpy( buff, gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(buff) );
  len = (int)strlen( buff );

  /* Reject non-numeric entries */
  for( idx = 0; idx < len; idx++ )
    if( (buff[idx] < '0') || (buff[idx] > '9') )
    {
      Show_Message( _("Non-numeric entry"), "red" );
      Error_Dialog();
      return;
    }

  /* Reject invalid entries */
  idx = atoi( gtk_entry_get_text(GTK_ENTRY(editable)) );
  if( (idx < 0) || (idx > 59) )
  {
    Show_Message( _("Value out of range"), "red" );
    Error_Dialog();
    return;
  }

} /* Minutes_Entry() */

/*------------------------------------------------------------------------*/

/* Fft_Drawingarea_Size_Alloc()
 *
 * Handles the on_ifft_drawingarea_size_allocate CB
 */
void
Fft_Drawingarea_Size_Alloc( GtkAllocation *allocation )
{
  /* Destroy existing pixbuff */
  if( wfall_pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(wfall_pixbuf) );
    wfall_pixbuf = NULL;
  }

  /* Create waterfall pixbuf */
  wfall_pixbuf = gdk_pixbuf_new(
      GDK_COLORSPACE_RGB, FALSE, 8,
      allocation->width, allocation->height );
  if( wfall_pixbuf == NULL )
  {
    Show_Message( _("Failed to create pixbuf for waterfall"), "red" );
    return;
  }

  /* Get waterfall pixbuf details */
  wfall_pixels = gdk_pixbuf_get_pixels( wfall_pixbuf );
  wfall_width  = gdk_pixbuf_get_width ( wfall_pixbuf );
  wfall_height = gdk_pixbuf_get_height( wfall_pixbuf );
  wfall_rowstride  = gdk_pixbuf_get_rowstride( wfall_pixbuf );
  wfall_n_channels = gdk_pixbuf_get_n_channels( wfall_pixbuf );
  gdk_pixbuf_fill( wfall_pixbuf, 0 );

  /* Initialize ifft. Waterfall with is an odd number
   * to provide a center line. IFFT requires a width
   * that is a power of 2 */
  Initialize_IFFT( (int16_t)wfall_width + 1 );

} /* Fft_Drawingarea_Size_Alloc() */

/*------------------------------------------------------------------------*/

/* BW_Entry_Activate()
 *
 * Handles the activate callback for bandwidth entry
 */
  void
BW_Entry_Activate( GtkEntry *entry )
{
  /* Get bandwidth value in Hz and set */
  double bw = 1000.0 * atof( gtk_entry_get_text(entry) );
  rc_data.rtlsdr_lpf_bw = (int)bw;
  filter_data_i.cutoff    = (double)(rc_data.rtlsdr_lpf_bw / 2);
  filter_data_i.cutoff   /= (double)RTL_DSP_RATE;
  filter_data_q.cutoff    = filter_data_i.cutoff;
  Init_Chebyshev_Filter( &filter_data_i );
  Init_Chebyshev_Filter( &filter_data_q );

  /* Show Bandwidth in messages */
  char text[48];
  bw /= 1000.0;
  snprintf( text, sizeof(text),
      _("Setting Low Pass Filter cutoff to %0.1fkHz"), bw );
  Show_Message( text, "black" );

} /* BW_Entry_Activate() */

/*------------------------------------------------------------------------*/

