// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief gamma gamma -> rho+ rho-
  class CELLO_1989_I267081 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CELLO_1989_I267081);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      // book histos
      book(_mult[0], 1, 1, 1);
      book(_mult[1], 2, 1, 1);
      book(_nRho, "TMP/nRho");
      for (const string& en : _mult[0].binning().edges<0>()) {
        const double eval = stod(en)*GeV;
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()] -= 1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // find the final-state particles
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      // find any rho mesons
      Particles rho=apply<UnstableParticles>(event, "UFS").particles(Cuts::abspid==213);
      for (size_t ix=0; ix<rho.size(); ++ix) {
       	if (rho[ix].children().empty()) continue;
       	map<long,int> nRes=nCount;
       	int ncount = ntotal;
       	findChildren(rho[ix],nRes,ncount);
        bool matched = false;
        for (size_t iy=ix+1; iy<rho.size(); ++iy) {
          if (rho[iy].children().empty()) continue;
          if (rho[ix].pid()!=-rho[iy].pid()) continue;
          map<long,int> nRes2=nRes;
          int ncount2 = ncount;
          findChildren(rho[iy],nRes2,ncount2);
          if (ncount2 !=0) continue;
          matched=true;
          for (const auto& val : nRes2) {
            if (val.second!=0) {
              matched = false; break;
            }
          }
          if (matched)  break;
        }
        if (matched) {
          _nRho->fill(); break;
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_nRho, crossSection()/nanobarn/sumOfWeights());
      _mult[0]->binAt(_sqs).set(_nRho->val(), _nRho->err(), "stats");
      _mult[1]->binAt(_sqs).set(_nRho->val(), _nRho->err(), "stats");
    }

    /// @}


    /// @name Histograms
    /// @{
    string _sqs = "";
    CounterPtr _nRho;
    BinnedEstimatePtr<string> _mult[2];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(CELLO_1989_I267081);
}
