// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"



namespace Rivet {


  class CDF_2012_NOTE10874 : public Analysis {
  public:

    RIVET_DEFAULT_ANALYSIS_CTOR(CDF_2012_NOTE10874);


  public:

    void init() {
      const ChargedFinalState cfs(Cuts::abseta < 1.0 && Cuts::pT >= 0.5*GeV);
      declare(cfs, "CFS");

      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_p[en+"nch_transverse"], 1, 1, ih);
        book(_p[en+"ptSumDen"], 2, 1, ih);
        book(_p[en+"avePt"], 3, 1, ih);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    // Little helper function to identify Delta(phi) regions
    inline int region_index(double dphi) {
      assert(inRange(dphi, 0.0, PI, CLOSED, CLOSED));
      if (dphi < PI/3.0) return 0;
      if (dphi < 2*PI/3.0) return 1;
      return 2;
    }


    void analyze(const Event& event) {
      const ChargedFinalState& cfs = apply<ChargedFinalState>(event, "CFS");
      if (cfs.size() < 1) vetoEvent;

      Particles particles = cfs.particlesByPt();
      Particle p_lead = particles[0];
      const double philead = p_lead.phi();
      const double pTlead  = p_lead.pT();

      int    tNch = 0;
      double ptSum = 0.0;
      for (const Particle& p : particles) {
        const double pT = p.pT();
        const double dPhi = deltaPhi(philead, p.phi());
        const int ir = region_index(dPhi);
        if (ir==1) {
          ++tNch;
          ptSum += pT;
        }
      }

      const double dEtadPhi = 4.0*PI/3.0;

      _p[_sqs+"nch_transverse"]->fill(pTlead/GeV, tNch/dEtadPhi);
      _p[_sqs+"ptSumDen"]->fill(pTlead/GeV, ptSum/dEtadPhi);

      if (tNch > 0)  _p[_sqs+"avePt"]->fill(pTlead/GeV, ptSum/tNch);
    }


  private:

    map<string,Profile1DPtr> _p;

    string _sqs = "";

  };


  RIVET_DECLARE_PLUGIN(CDF_2012_NOTE10874);

}
