//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers_misc.cc
#include <iostream.h>
#include <math.h>
#include <limits.h>	// for ULONG_MAX

#ifndef _TIERS_HH
#include "tiers.hh"
#endif

extern "C" 
{
  long random(void);
}



////////////////////////////////////////////////////////////////////////
// Model::Euclid
//
// Calculate the Euclidean distance between two points
////////////////////////////////////////////////////////////////////////
const unsigned long int
Model::Euclid(Node &A, Node &B)
{
  // Need to use floats to deal with large numbers when squared
  // No: max number is GRID*GRID, so if GRID < sqrt ULONG_MAX, 
  // i.e. < approx 6K, we can do without floats
  signed long int deltaX = A.x - B.x;
  signed long int deltaY = A.y - B.y;
  double sqrtsum = floor(sqrt((double)(deltaX*deltaX) + 
							  (double)(deltaY*deltaY)));
  return (unsigned long int)(sqrtsum);
}



////////////////////////////////////////////////////////////////////////
// Model::place
//
// Place a point in a grid. Can optionally test for proximity to other nodes
////////////////////////////////////////////////////////////////////////
bool
Model::place(unsigned long int aNode, Node *nodes, unsigned long int NumNodes)
{
  bool ret = true;	
  bool test = false;
  NodeType type = nodes[0].type;	// set before place is called

  switch (type)
	{
	case WAN: 
	  {
		test = PROXIMITY_TEST_WAN;
		break;
	  }
	case MAN: 
	  {
		test = PROXIMITY_TEST_MAN;
		break;
	  }
	case LAN: 
	  {
		test = PROXIMITY_TEST_LAN;
		break;
	  }
	default:
	  {
		cerr << "Tiers::place Unrecognized network type" << endl;
		return false;
	  }
	  
	}

  // Create the coordinates for the node
  nodes[aNode].x = (random() % GRID);
  nodes[aNode].y = (random() % GRID);

  // Repeat the proximity test until true
  while(test && !proximity_test(aNode, nodes, NumNodes))
	{
	  nodes[aNode].x = (random() % GRID);
	  nodes[aNode].y = (random() % GRID);
	}

  return ret;
}


////////////////////////////////////////////////////////////////////////
// Model::proximity_test
//
// Return false if (X,Y) is too close to any other point in nodes
// The 0.5 scale factor is mentioned in appendix A of the paper
// Values for this parameter of up to 1 produces graphs. Above this,
// appropriate values are never chosen
////////////////////////////////////////////////////////////////////////
bool 
Model::proximity_test(unsigned long int aNode, 
					  Node *nodes, 
					  unsigned long int NumNodes)
{
  bool ret = true;
  double f_min_dist = floor(0.5 * (double)GRID/sqrt((double)NumNodes));
  unsigned long int min_dist = (unsigned long int)f_min_dist;

  // Check all the earlier nodes' distances
  for (unsigned long int node = 0; node < aNode; node++)
	{
	  if (Euclid(nodes[node], nodes[aNode]) < min_dist)
		{
		  return false;
		}
	}

  return ret;
}



////////////////////////////////////////////////////////////////////////
// NodesAndEdges::dump_edges
//
// Compact printout of an edge array of a WAN or MAN
// Array = 0 is connectivity matrix, Array = 1 is MST
// Used for debug of code 
// TODO is itself faulty wrt delay values printed. gdb and array indexing bug?
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
dump_edges(const unsigned int ArrayType)
{
  Edge 		*Array = 0;
  unsigned long int UpperLimit = 0;

  if (ArrayType)
	{
	  Array = newedges;
	  UpperLimit = m_MaxNumEdges;
	}
  else
	{
	  UpperLimit = m_NumNodes;
	  Array = edges;
	}

  cout << "[From To Cost BW State]" << endl;
  for (unsigned long int i = 0; i < m_NumNodes; i++)
	{
	  for (unsigned long int j = 0; j < UpperLimit; j++)
		{
		  unsigned long int index = i*UpperLimit+j;
		  cout << "[" 
			   << Array[index].start << " "
			   << Array[index].end << " ";

		  if (ULONG_MAX == Array[index].delay) 
			{
			  cout << "INF ";
			}
		  else
			{
			  cout << (unsigned long int)(Array[i].delay) << " ";
			}

		  cout << Array[index].bw << " "
			   << Array[index].state
			   << "], ";
		  
		}
	  cout << endl;
	}

}
  

////////////////////////////////////////////////////////////////////////
// NodesAndEdges::dump_lanedges
//
// Compact printout of an edge array of a LAN
//
// Used for debug of code
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
dump_lanedges()
{
  cout << "[From To Cost BW State]" << endl;
  for (unsigned long int i = 0; i < m_MaxNumEdges; i++)
	{
	  cout << "[" 
		   << edges[i].start << " "
		   << edges[i].end << " ";

	  if (ULONG_MAX == edges[i].delay) 
		{
		  cout << "INF ";
		}
	  else
		{
		  cout << (unsigned long int)(edges[i].delay) << " ";
		}

	  cout << edges[i].bw << " "
		   << edges[i].state 
		   << "], " << endl;
	}

}
  
//
// GLOBAL FUNCTIONS
//


////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
unsigned long int 
max(unsigned long int A, unsigned long int B)
{
  if (A > B)
	{
	  return A;
	}
  else
	{
	  return B;
	}
}

// end of file

