//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//                  Network Security Analysis Tool                          //
//                 AuditSet.cpp - auditing functions                        //
//                                                                          //
//   Copyright (C) 1999-2002 by Mixter and 2xs ltd. <mixter@2xs.co.il>      //
//                                                                          //
// This program is free software; you can redistribute it and/or modify     //
// it under the terms of the GNU General Public License as published by     //
// the Free Software Foundation; either version 2 of the License, or        //
// (at your option) any later version.                                      //
//                                                                          //
// This program is distributed in the hope that it will be useful,          //
// but WITHOUT ANY WARRANTY; without even the implied warranty of           //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            //
// GNU General Public License for more details.                             //
//                                                                          //
// You should have received a copy of the GNU General Public License        //
// along with this program; if not, write to the Free Software              //
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#pragma implementation "auditset"
#include "AuditSet.h"
#include "mod/shellcode.h"

extern ProgressIndicator pi;
extern Logging Logger;

#ifdef EBUG
extern ofstream dbug;
#endif

// Feel free to add ports to this list
struct portlist ports[] =
{
  {1, "tcpmux", 1},
  {7, "echo", 1},
  {9, "discard", 0},
  {11, "systat", 1},
  {13, "daytime", 0},
  {15, "netstat", 1},
  {17, "qotd", 0},
  {19, "chargen", 1},
  {20, "ftp-data", 1},
  {21, "ftp", 0},
  {22, "ssh", 0},
  {23, "telnet", 0},
  {25, "smtpd", 0},
  {37, "time", 0},
  {43, "nicname", 0},
  {48, "auditd", 1},
  {49, "tacacs", 1},
  {53, "bind", 0},
  {70, "gopher", 0},
  {79, "finger", 1},
  {80, "http", 0},
  {88, "kerberos", 0},
  {98, "linuxconf", 1},
  {109, "pop2", 1},
  {110, "pop3", 0},
  {111, "portmap", 1},
  {113, "auth", 0},
  {115, "sftp", 0},
  {119, "nntp", 0},
  {123, "ntp", 0},
  {139, "netbios/smbd", 1},
  {143, "imap", 0},
  {179, "bgp", 1},
  {213, "ipx", 0},
  {261, "FW-1 AuthAgent", 1},
  {389, "ldap", 0},
  {427, "AppleShare/IP", 1},
  {443, "https", 0},
  {445, "smbd", 1},
  {512, "rexecd", 1},
  {513, "rlogind", 1},
  {514, "rshd", 1},
  {515, "lpd", 1},
  {543, "klogin", 1},
  {540, "uucp", 0},
  {544, "kshell", 1},
  {548, "AppleShare/IP", 1},
  {600, "600 (common backdoor port)", 2},
  {666, "666 (common backdoor port)", 2},
  {762, "quotad", 1},
  {774, "rpasswd", 0},
  {873, "rsync", 0},
  {992, "telnets", 0},
  {993, "imaps", 0},
  {994, "ircs", 0},
  {995, "pop3s", 0},
  {1080, "SOCKS5", 0},
  {1114, "mSQL", 1},
  {1243, "1243 (subseven backdoor port)", 2},
  {1337, "1337 (common backdoor port)", 2},
  {1433, "MS-SQL", 1},
  {1524, "1524 (common backdoor port or ingreslock)", 2},
  {1525, "Oracle SQL", 1},
  {1999, "Cisco backdoor (1999)", 1},
  {2000, "OpenWin", 1},
  {2049, "nfsd", 1},
  {2222, "2222 (common backdoor port)", 2},
  {2223, "mdbms", 1},
  {2224, "mdbms (fastport)", 1},
  {2301, "cim", 1},
  {2606, "netmon", 1},
  {2766, "nlpsd", 1},
  {2998, "realsecure", 1},
  {3007, "lotusmtap", 1},
  {3128, "squid", 0},
  {3279, "admind", 1},
  {3306, "mysql", 1},
  {3333, "3333 (eggdrop default port)", 0},
  {3346, "transparent proxy", 1},
  {5556, "HP-UX RemoteWatch", 1},
  {6000, "XWindows", 1},
  {6667, "ircd", 0},
  {7161, "Cisco DoS (7161)", 1},
  {7070, "RealServer DoS (7070)", 1},
  {8080, "WebProxy (8080)", 0},
  {10401, "kdebug", 1},
  {12345, "12345 (common backdoor port)", 2},
  {22222, "22222 (common backdoor port)", 2},
  {26000, "quake", 1},
  {27655, "trinoo master (backdoor)", 1},
  {31335, "trinoo4 master (backdoor)", 1},
  {31337, "31337 (common backdoor port)", 2},
  {65535, "65535 (common backdoor port)", 2},
  {0, "", 0}
};

AuditSet::AuditSet(void)
{
  firstopen = 0;
  sl = sizeof(sin);
  memset(&sin, 0, sl);
  sin.sin_family = AF_INET;
  sin.sin_port = htons(0);
  signal(SIGALRM, hook);
  signal(SIGPIPE, hook);
}

AuditSet::~AuditSet(void)
{
  firstopen = 0;
  signal(SIGALRM, SIG_DFL);
  signal(SIGPIPE, SIG_DFL);
}

inline char *AuditSet::Hostname(void)
{
  return (ntoa(target));
}

int AuditSet::icmptest(void)
{
  int s = isock(), retval = 0;
  fd_set st;
  struct timeval tv = {timeout * 2, 0};
  char packet[512];
  struct lmicmp *ich = (struct lmicmp *) packet;
  struct lmicmp_netmask {
	  u16 id;
	  u16 seq;
	  u8 mask[4];
  } *inm = (struct lmicmp_netmask *)(packet+sizeof(struct lmip)+sizeof(struct lmicmp));
  struct lmicmp_time {
	  u16 id;
	  u16 seq;
	  u32 origts;
	  u32 recvts;
	  u32 transts;
  } *its = (struct lmicmp_time *)(packet+sizeof(struct lmip)+sizeof(struct lmicmp));
  struct sockaddr_in sa;
  long ref;

  if (pi.vhostip)
    {
     memset(&sin, 0, sizeof(sin));
     sin.sin_family = AF_INET;
     sin.sin_port = htons(0);
     sin.sin_addr.s_addr = pi.vhostip;
     bind(s, (struct sockaddr *) &sin, sizeof(sin));
    }

  if (geteuid())
    return (1);

  sa.sin_family = AF_INET;
  sa.sin_port = 0;
  sa.sin_addr.s_addr = target;

  /* ECHO */
  memset(packet, 0, 512);
  ich->type = 8;
  ich->id = getrandom(0, 65535);
  ich->seq = getrandom(0, 65535);
  ich->sum = 0;
  ich->sum = sum((unsigned short *) ich, (64 >> 1));
  sendto(s, packet, 64, 0, (struct sockaddr *) &sa, sizeof(sa));

  if(pi.ScanIcmp >= 2)
  {
   /* TIME */
   memset(packet, 0, 512);
   ich->seq = getrandom(0, 65535);
   ich->type = 13;
   ich->code = 0;
   gettimeofday((struct timeval *) (ich + 8), NULL);
   ich->sum = 0;
   ich->sum = sum((unsigned short *) ich, (64 >> 1));
   sendto(s, packet, 21, 0, (struct sockaddr *) &sa, sizeof(sa));
  
   /* NETMASK */
   sa.sin_family = AF_INET;
   sa.sin_port = 0;
   sa.sin_addr.s_addr = target;
   memset(packet, 0, 512);
   ich->seq = getrandom(0, 65535);
   ich->sum = 0;
   ich->type = 17;
   ich->code = 0;
   *((char *) (ich + 8)) = 255;
   ich->sum = sum((unsigned short *) ich, (64 >> 1));
   sendto(s, packet, 13, 0, (struct sockaddr *) &sa, sizeof(sa));
  }

  ref = time(NULL);
  ich = (struct lmicmp *) (packet+M_IP);
  
  for(int i=0;i<6;i++)
  {
   FD_ZERO(&st);
   FD_SET(s, &st);

   tv.tv_sec = timeout/2;
   tv.tv_usec = 0;
   
   if (!select(s + 1, &st, NULL, NULL, &tv))
    {
     if (pi.VerboseLogging && i > 4)
       {
        /* logger logfile stdoutpattern hostname fmtstring */
        Logger.msg(L_ICMP, I_ICMP, Hostname(), I_PINGOUT);
       }
     SCLOSE(s);
     return retval;
    }
   else
   {
    unsigned int slen = sizeof(sa);

    retval = 1;
    recvfrom(s, &packet, 500, 0, (struct sockaddr *) &sa, (socklen_t *)&slen);
    
    switch(ich->type)
    {
	    case 0:
		    Logger.msg(L_ICMP, I_ICMP, Hostname(), "- latency %lu secs", (time(NULL) - ref));
		    break;
	    case 14:
		    Logger.msg(L_ICMP, I_ICMP, Hostname(), "- timestamp: orig:%lu/recv:%lu/trans:%lu", its->origts, its->recvts, its->transts);
		    break;
	    case 18:
		    Logger.msg(L_ICMP, I_ICMP, Hostname(), "- netmask reply: %d.%d.%d.%d", inm->mask[0], inm->mask[1], inm->mask[2], inm->mask[3]);
		    break;
    }
   }
 } // for

  SCLOSE(s);
  return (1);
}

void AuditSet::plog(int s)
{
  if (getpeername(s, (struct sockaddr *) &sin, (socklen_t *)&sl) == -1)
    {
#ifdef EBUG
      dbug << "getpeername failed: " << strerror(errno) << ENTER;
#endif
      SCLOSE(s);
      return;
    }
  char *desc = NULL;
  int vuln = 0;

  for (int n = 0; (ports[n].port != 65535); n++)
    if (ports[n].port == ntohs(sin.sin_port))
      {
	desc = ports[n].info;
	vuln = ports[n].vuln;
      }

  if (desc == NULL)
    {
      Logger.msg(L_PORTS, I_PORTL, Hostname(), "- %u %s", ntohs(sin.sin_port), (char *) (vuln ? M_DANGEROUS_SVC : ""));
    }
  else
    {
      Logger.msg(L_PORTS, I_PORTL, Hostname(), "- %s %s", desc, (char *) (vuln ? M_DANGEROUS_SVC : ""));
    }

  if ((vuln == 2) && (pi.ScanBackdoor))
    backdoor(s, sin.sin_port);
  else
    SCLOSE(s);

  return;
}

void AuditSet::nstat(int s)
{
  Logger.msg(L_PORTS, I_PORTL, Hostname(), "- %s", I_NETSTAT);

  if (pi.ScanNetstat < 2)
    {
      SCLOSE(s);
      return;
    }

  // Logger(L_NETSTAT, I_NSTAT, Hostname(), "- %s", I_NETSTAT);

  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
   return;

  Logger.msg(L_NETSTAT, I_NSTAT, Hostname(), "- %s", b);

  SCLOSE(s);
}

void AuditSet::ftp(int s)
{
      char b[BUFSIZE];
      int l = 1;

      memset(b, 0, BUFSIZE);
      while (strstr(b, ENTER) == NULL && l > 0)
	if ((l = nread(timeout, s, b + strlen(b), BUFSIZE - strlen(b) - 1)) <= 0)
         {
          Logger.msg(L_FTP, I_FTP, Hostname(), "- ftp (banner timeout)");
          SCLOSE(s);
          return;
         }

      Logger.msg(L_FTP, I_FTP, Hostname(), "- %s", b);

      if (pi.ScanFtp > 1)
	ftptest(s, Hostname());

  SCLOSE(s);
}

void AuditSet::ssh(int s)
{
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      Logger.msg(L_SSH, I_SSH, Hostname(), "- ssh (banner timeout)");
      SCLOSE(s);
      return;
    }

  strchop(b);

  Logger.msg(L_SSH, I_SSH, Hostname(), "- %s", b);

  SCLOSE(s);
}

void AuditSet::telnet(int s)
{
  char b[BUFSIZE * 2];

  if (pi.ScanTelnet < 2)
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- telnet");
      SCLOSE(s);
      return;
    }

  int len = 0;

  while (len != 1337)
    {
      memset(b, 0, sizeof(b));
      if ((len = nread(timeout, s, b, 1)) <= 0)
	break;
      if (*b == '\xff')
	{
	  nread(timeout/2, s, (b + 1), 2);
	  if (*(b + 1) == '\xfd' && !*(b + 2));
	  else if (*(b + 1) == '\xfd')
	    {
	      *(b + 1) = '\xfc';
	      write(s, b, 3);
	    }
	}
      else
	{
	  if (*b != 0)
	    {
	      memset(b, 0, sizeof(b));
	      nread(timeout/2, s, b, sizeof(b));
	      len = 1337;
	    }
	}
    }
  
    if (pi.ScanTelnet >= 3)
    {
     char sndbuf[10240], rcvbuf[10240];

     for(int i=0;i<5120;i++)
       strcat(sndbuf, "\xff\xf6");

       write(s, sndbuf, strlen(sndbuf));

       if(nread(timeout, s, rcvbuf, 10239) <= 0)
	{
		Logger.msg(L_TELNET, I_TELNET, Hostname(), I_TELNETAYT);
	}
   }

  SCLOSE(s);
  my_strnsubst(b, sizeof(b));
  strlower(b);

  if (strlen(b) < 2)
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- ALL:ALL in hosts.deny?");
      return;
    }
  if (strstr(b, "irix"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- IRIX");
      return;
    }
  if (strstr(b, "system v"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- System V banner");
      return;
    }
  if (strstr(b, "solaris"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Solaris");
      return;
    }
  if (strstr(b, "sunos"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- SunOS");
      return;
    }
  if (strstr(b, "wingate"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Open WinGate");
      return;
    }
  if (strstr(b, "red hat"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- RedHat Linux");
      return;
    }
  if (strstr(b, "suse"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- SuSE Linux");
      return;
    }
  if (strstr(b, "slackware"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Slackware Linux");
      return;
    }
  if (strstr(b, "debian"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Debian Linux");
      return;
    }
  if (strstr(b, "linux"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Linux");
      return;
    }
  if (strstr(b, "freebsd"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- FreeBSD");
      return;
    }
  if (strstr(b, "netbsd"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- FreeBSD");
      return;
    }
  if (strstr(b, "openbsd"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- OpenBSD");
      return;
    }
  if (strstr(b, "bsdi"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- BSDi");
      return;
    }
  if (strstr(b, "bsd"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- BSD");
      return;
    }
  if (strstr(b, "cisco"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Cisco IOS");
      return;
    }
  if (strstr(b, "sco"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- SCO UNIX");
      return;
    }
  if (strstr(b, "vms"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- OpenVMS/VAX");
      return;
    }
  if (strstr(b, "hp"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- HP-UX");
      return;
    }
  if (strstr(b, "next"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- NeXT");
      return;
    }
  if (strstr(b, "dg"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- DG-UNIX");
      return;
    }
  if (strstr(b, "user access"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Radius or router");
      return;
    }
  if (strstr(b, "ascend"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Ascend router");
      return;
    }
  if (strstr(b, "password"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Router password prompt");
      return;
    }
  if (strstr(b, "telnet"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Unknown proxy (\"telnet\" prompt)");
      return;
    }
  if (strstr(b, "gate"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Unknown proxy (\"gate\" prompt)");
      return;
    }
  if (strstr(b, "login"))
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "- Unknown (\"login\" prompt)");
      return;
    }
  if (pi.VerboseLogging)
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), I_UNKNOWN);
    }
  if (pi.ScanTelnet == 4)
    {
      Logger.msg(L_TELNET, I_TELNET, Hostname(), "<< %s >>", b);
    }
  return;
}

void AuditSet::smtp(int s)
{
  int siz = BUFSIZE * 5;
  char b[siz + 1];

  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
    {
      Logger.msg(L_SMTP, I_SMTP, Hostname(), " - smtp (banner timeout)");
      SCLOSE(s);
      return;
    }

  Logger.msg(L_SMTP, I_SMTP, Hostname(), " %s", b);

  if (pi.ScanSmtp < 2)
    {
      sleep(1);
      SCLOSE(s);
      return;
    }

  if (strstr(b, "8.9") != NULL)
    smcheck = 1;
  else
    smcheck = 0;

  write(s, SMTP_HELO, strlen(SMTP_HELO));
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b)) <= 0)
    {
      SCLOSE(s);
      return;
    }

  memset(b, 0, siz);
  write(s, SMTP_EXPN, strlen(SMTP_EXPN));
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b)) <= 0)
    {
      SCLOSE(s);
      return;
    }

  if (strstr(b, "250"))
    {
      Logger.msg(L_SMTP, I_SMTP, Hostname(), I_EXPN);
    }
  else
    {
      write(s, SMTP_VRFY, strlen(SMTP_VRFY));
      memset(b, 0, siz);
      while (strstr(b, ENTER) == NULL)
     if(nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }

      if (strstr(b, "250"))
        Logger.msg(L_SMTP, I_SMTP, Hostname(), I_VRFY);
    }

   if (pi.ScanSmtp < 3)
   {
    write(s, SMTP_QUIT, strlen(SMTP_QUIT));
    sleep(1);
    SCLOSE(s);
    return;
   }
  
  write(s, SMTP_FROM, strlen(SMTP_FROM));
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout,s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }

  write(s, SMTP_TO, strlen(SMTP_TO));
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout,s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }

  write(s, SMTP_DATA, strlen(SMTP_DATA));
  strunf(b);
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }

  write(s, SMTP_END, strlen(SMTP_END));
  strunf(b);
  while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }

  if (strstr(b, "250"))
    {
      Logger.msg(L_SMTP, I_SMTP, Hostname(), I_FAKEMAIL);
    }
  if (strstr(b, "354"))
    {
      Logger.msg(L_SMTP, I_SMTP, Hostname(), I_SPAM);
    }
  if ((pi.ScanSmtp == 4) && (smcheck))
    {
      memset(b, 0, siz);
      write(s, SMTP_FROM, strlen(SMTP_FROM));
      memset(b, 0, siz);
      while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }

      write(s, SMTP_TO89, strlen(SMTP_TO89));
      memset(b, 0, siz);
      while (strstr(b, ENTER) == NULL)
   if (nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
    {
      SCLOSE(s);
      return;
    }
      // Yes, I know this should be outdated by now, but who knows ;/
      if (strstr(b, "250"))
          Logger.msg(L_SMTP, I_SMTP, Hostname(), I_RPMMAIL);
    }
  write(s, SMTP_QUIT, strlen(SMTP_QUIT));
  strunf(b);
  while (strstr(b, ENTER) == NULL)

  if (nread(timeout, s, b + strlen(b), siz - strlen(b))<=0)
   {
     SCLOSE(s);
     return;
   }

  sleep(1);			/* avoid ugly sendmail syslog errors... */
  SCLOSE(s);
}

void AuditSet::nbind(int s)
{
  SCLOSE(s);
  probe_bind(sin, timeout);
}

void AuditSet::finger(int s)
{
  if (pi.ScanFinger < 2)
    {
      SCLOSE(s);
      Logger.msg(L_PORTS, I_PORTL, Hostname(), I_FINGER);
      return;
    }

  if (write(s, FINGER_HELLCODE, sizeof(FINGER_HELLCODE)) == -1)
    {
#ifdef EBUG
      dbug << "write failed: " << strerror(errno) << ENTER;
#endif
      SCLOSE(s);
      return;
    }
  Logger.msg(L_PORTS, I_PORTL, Hostname(), I_FINGER);

  char b[BUFSIZE * 2];

  memset(b, 0, BUFSIZE * 2);
  int i = nread(timeout, s, b, (BUFSIZE * 2) - 1);

  if (strlen(my_strnsubst(b, i + 1)) > 2)
    {
      Logger.msg(L_FINGER, I_FING, Hostname(), " - %s", b);
    }
  SCLOSE(s);
}

void AuditSet::http(int s)
{
  webscan(s);
  SCLOSE(s);
}

void AuditSet::pop2(int s)
{
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      Logger.msg(L_POP2, I_POP2, Hostname(), " - POP2 (banner timeout)");
      SCLOSE(s);
      return;
    }

  Logger.msg(L_POP2, I_POP2, Hostname(), " - %s", b);
  SCLOSE(s);
}

void AuditSet::pop3(int s)
{
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      Logger.msg(L_POP3, I_POP3, Hostname(), " - POP3 (banner timeout)");
      SCLOSE(s);
      return;
    }
  strchop(b);

  Logger.msg(L_POP3, I_POP3, Hostname(), " - %s", b);
  SCLOSE(s);
}

void AuditSet::pmap(int s)
{
  SCLOSE(s);
  Logger.msg(L_PORTS, I_PORTL, Hostname(), I_PMAP);
  if (pi.ScanRpc > 1)
      rpccheck(Hostname());
}

void AuditSet::nntp(int s)
{
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      Logger.msg(L_NNTP, I_NNTP, Hostname(), " - NEWS (banner timeout)");
      SCLOSE(s);
      return;
    }

  Logger.msg(L_NNTP, I_NNTP, Hostname(), " - %s", b);
  SCLOSE(s);
}

void AuditSet::smb(int s)
{
  SCLOSE(s);
  int pid = safe_fork();

  if (pid == 0)
    {
      if (pi.Foreground)
	execlp(I_SMBEXEC, I_SMBEXEC, Hostname(), "1", NULL);
      else
	execlp(I_SMBEXEC, I_SMBEXEC, Hostname(), NULL);
      raise(SIGKILL);
    }
  else
    {
      sleep(timeout * 3);
      kill(pid, SIGKILL);
    }
}

void AuditSet::imap(int s)
{
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      Logger.msg(L_IMAP, I_IMAP, Hostname(), " - IMAP (banner timeout)");
      SCLOSE(s);
      return;
    }

  Logger.msg(L_IMAP, I_IMAP, Hostname(), " - %s", b);
  SCLOSE(s);
}

void AuditSet::backdoor(int s, int pt)
{
  if (send(s, ROOTINFO, sizeof(ROOTINFO), 0) == -1)
    {
#ifdef EBUG
      dbug << "send failed: " << strerror(errno) << ENTER;
#endif
      SCLOSE(s);
      return;
    }
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      SCLOSE(s);
      return;
    }

  if (strlen(b) > 3)
    {
      Logger.msg(L_BACKDOOR, I_PORT, Hostname(), " - %u: << %s >>", ntohs(pt), b);
    }
  SCLOSE(s);
}

void AuditSet::nlps(int s)
{
  Logger.msg(L_PORTS, I_PORT, Hostname(), I_NLPS);

  if (pi.ScanNlps < 2)
    {
      SCLOSE(s);
      return;
    }

  write(s, nlps_x86, sizeof(nlps_x86));
  write(s, ROOTINFO, sizeof(ROOTINFO));
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  if (nread(timeout, s, b, BUFSIZE - 1)<=0)
    {
      SCLOSE(s);
      return;
    }

  if (strlen(b) > 2)
    {
      Logger.msg(L_NLPS, I_NLPL, Hostname(), "- %s", b);
    }
  SCLOSE(s);
}

void AuditSet::ircd(int s)
{
  Logger.msg(L_PORTS, I_PORTL, Hostname(), I_IRCD);
  if (pi.ScanIrcd < 2)
    {
      SCLOSE(s);
      return;
    }
  char b[BUFSIZE * 3];

  write(s, IRC_QUERY, sizeof(IRC_QUERY));
  while (!strstr(b, " 351 "))
    {
      memset(b, 0, (BUFSIZE * 3));
      if (nread(timeout, s, b, (BUFSIZE * 3) - 1)<=0)
       {
        SCLOSE(s);
	return;
       }
    }
  Logger.msg(L_IRCD, I_PORTL, Hostname(), " - %s", b);
  SCLOSE(s);
}

void AuditSet::xwin(int s)
{
  Logger.msg(L_PORTS, I_PORTL, Hostname(), I_XWIN);
  if (pi.ScanXWin < 2)
    {
      SCLOSE(s);
      return;
    }

#ifdef HAVE_X11_XLIB_H
  char disp[BUFSIZE];

  snprintf(disp, BUFSIZE - 1, "%s:0", Hostname());
  /* If you get an error here, use struct Display instead of just
     Display. All newer X11 libraries should have it typedef'd. */
  Display *d;

  if ((d = XOpenDisplay(disp)) != NULL)
    {
      Logger.msg(L_XWIN, I_XWINL, Hostname(), I_XWIN2);
      XCloseDisplay(d);
    }

  SCLOSE(s);
#endif
}

void AuditSet::nbname(void)
{
/* raw nbname session from nbname.c (c) by mynock */
  const char *nbquery =
  "\x03\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x20\x43\x4b\x41\x41"
  "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
  "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x00\x00\x21\x00\x00";
  int sockfd, i, n;
  struct sockaddr_in servaddr, myaddr;
  char buf[1024];
  fd_set readfs;
  struct timeval tv;
  bzero(&servaddr, sizeof(struct sockaddr_in));

  servaddr.sin_family = AF_INET;
  servaddr.sin_port = htons(137);
  servaddr.sin_addr.s_addr = target;
  sockfd = socket(AF_INET, SOCK_DGRAM, 0);
  memset(&myaddr, 0, sizeof(myaddr));
  myaddr.sin_family = AF_INET;
  myaddr.sin_port = htons(137);
  myaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  bind(sockfd, (struct sockaddr *) &myaddr, sizeof(myaddr));
  while (1)
    {
      sendto(sockfd, nbquery, 50, 0, (struct sockaddr *) &servaddr, sizeof(struct sockaddr_in));

      FD_ZERO(&readfs);
      FD_SET(sockfd, &readfs);
      tv.tv_sec = 5;
      tv.tv_usec = 0;
      n = select(sockfd + 1, &readfs, NULL, NULL, &tv);
      switch (n)
	{
	case 1:
	  if ((i = recvfrom(sockfd, buf, 800, 0, NULL, NULL)) < 0)
	    {
	      SCLOSE(sockfd);
	      return;
	    }
	  break;
	case 0:
	  SCLOSE(sockfd);
	  return;
	  break;
	default:
	  SCLOSE(sockfd);
	  return;
	  break;
	}
      if (n == 1)
	break;
    }

  if (strlen(buf + 58) < 2)
    {
      SCLOSE(sockfd);
      return;
    }

  char bufx[128];
  int xx = 0;
  memset(bufx, 0, 128);

  if (strlen(buf + 57) >= 1)
    {
      for (n = 57; (n < i) && (buf[n] != ' '); n++)
	{
          bufx[xx++] = buf[n];
	}

      Logger.msg(L_NETBIOS, I_NBIOS, Hostname(), "- %s", bufx);
    }

  SCLOSE(sockfd);
  return;
}

void AuditSet::bo()
{
  int s = usock();

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(0);
  if (pi.vhostip)
    sin.sin_addr.s_addr = pi.vhostip;

  bind(s, (struct sockaddr *) &sin, sizeof(sin));

  if (!sendpacket(target, 31337, s))
    getinput(s);
#ifdef EBUG
  else
    dbug << "sendping failed: " << Hostname() << ENTER;
#endif

  SCLOSE(s);
}

#ifdef WITH_XPROBE
void AuditSet::os(void)
{
    do_logic(target);
}
#endif

void AuditSet::snmp(void)
{
  snmpwalk(target);
}
