/*
  Copyright (C) 2009 to 2013, 2015, 2016 and 2020 Chris Vine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <unistd.h>

#include <fstream>
#include <algorithm>
#include <ios>
#include <istream>

#include <glib-object.h>
#include <cairo.h>
#include <pango/pango-layout.h>

#include "mainwindow.h"
#include "mount_entries.h"
#include "preferences.h"
#include "dialogs.h"
#if !(GTK_CHECK_VERSION(2,99,0))
#include "utils/cairo_handle.h"
#endif

#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/gobj_handle.h>
#include <c++-gtk-utils/mem_fun.h>
#include <c++-gtk-utils/timeout.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#define MAINWIN_DRAWING_AREA_WIDTH 8
#define MAINWIN_DRAWING_AREA_HEIGHT 16


void MainWindowCB::mainwin_tool_button_clicked(GtkToolButton* button, void* data) {
  MainWindow* instance = static_cast<MainWindow*>(data);
  
  if (button == instance->prefs_button) instance->prefs_impl();
  else if (button == instance->mount_entries_button) instance->mount_entries_impl();
  else if (button == instance->about_button) instance->about_impl();
  else if (button == instance->quit_button) instance->close();
  else write_error("Callback error in MainWindowCB::mainwin_tool_button_clicked()\n", false);
}


void MainWindowCB::mainwin_mount_button_clicked(GtkWidget*, void* data) {
  static_cast<Callback::Callback*>(data)->dispatch();
}

void MainWindowCB::mainwin_noentries_button_clicked(GtkWidget*, void* data) {
  static_cast<MainWindow*>(data)->mount_entries_impl();
}

gboolean MainWindowCB::mainwin_visibility_notify_event(GtkWidget*, GdkEventVisibility* event,
                                                       void* data) {
  MainWindow* instance = static_cast<MainWindow*>(data);

  if(event->state == GDK_VISIBILITY_FULLY_OBSCURED) instance->obscured = true;
  else instance->obscured = false;

  return false;  // carry on processing the visibility notify event
}

gboolean MainWindowCB::mainwin_window_state_event(GtkWidget*, GdkEventWindowState* event,
                                                  void* data) {
  MainWindow* instance = static_cast<MainWindow*>(data);

  if(event->changed_mask & GDK_WINDOW_STATE_ICONIFIED) {
    if(event->new_window_state & GDK_WINDOW_STATE_ICONIFIED) instance->minimised = true;
    else instance->minimised = false;
  }

  return false;  // carry on processing the window state event
}

#if GTK_CHECK_VERSION(2,99,0)

void MainWindowCB::mainwin_button_style_updated(GtkWidget*, void* data) {
  static_cast<MainWindow*>(data)->button_style_updated_impl();
}

gboolean MainWindowCB::mainwin_drawing_area_draw(GtkWidget* drawing_area,
						 cairo_t* cairo,
						 void*) {
  return MainWindow::draw_mount_indicator(drawing_area, cairo);
}

#else

void MainWindowCB::mainwin_button_style_set(GtkWidget*, GtkStyle*  prev_style, void* data) {
  static_cast<MainWindow*>(data)->button_style_set_impl(prev_style);
}

gboolean MainWindowCB::mainwin_drawing_area_expose_event(GtkWidget* drawing_area,
							 GdkEventExpose* event,
							 void*) {
  return MainWindow::draw_mount_indicator(drawing_area, event);
}

#endif /* GTK_CHECK_VERSION(2,99,0) */

void MainWindowCB::mainwin_delete_data(void* data, GClosure*) {
  delete static_cast<Callback::Callback*>(data);
}

MainWindow::MainWindow(bool start_hidden): WinBase(0, prog_config.window_icon),
					   button_vbox(0), button_extents(0, 0),
					   obscured(false), minimised(false),
					   in_dialog(false), button_style_connection(0) {

  gtk_window_set_title(get_win(), "Mount-gtk");

  gui_vbox = gtk_vbox_new(false, 2);
  gtk_container_add(GTK_CONTAINER(get_win()), gui_vbox);

  // set up the tool bar
  GtkToolbar* toolbar = GTK_TOOLBAR(gtk_toolbar_new());
  gtk_box_pack_start(GTK_BOX(gui_vbox), GTK_WIDGET(toolbar),
		     false, false, 0);
  gtk_toolbar_set_show_arrow(toolbar, false);
#if GTK_CHECK_VERSION(2,16,0)
  gtk_orientable_set_orientation(GTK_ORIENTABLE(toolbar), GTK_ORIENTATION_HORIZONTAL);
#else
  gtk_toolbar_set_orientation(toolbar, GTK_ORIENTATION_HORIZONTAL);
#endif

  // register our own button icon size for stock icons to match size of externally
  // defined icons used in this program
  gtk_icon_size_register("MOUNT_GTK_BUTTON_SIZE", 22, 22);

  prefs_button = GTK_TOOL_BUTTON(gtk_tool_button_new_from_stock(GTK_STOCK_PREFERENCES));
  gtk_toolbar_insert(toolbar, GTK_TOOL_ITEM(prefs_button), -1);
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(prefs_button),
				 gettext("Preferences"));
  g_signal_connect(G_OBJECT(prefs_button), "clicked",
		   G_CALLBACK(MainWindowCB::mainwin_tool_button_clicked), this);

  mount_entries_button = GTK_TOOL_BUTTON(gtk_tool_button_new_from_stock(GTK_STOCK_EDIT));
  gtk_toolbar_insert(toolbar, GTK_TOOL_ITEM(mount_entries_button), -1);
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(mount_entries_button),
				 gettext("Add, amend or delete table of devices to mount"));
  g_signal_connect(G_OBJECT(mount_entries_button), "clicked",
		   G_CALLBACK(MainWindowCB::mainwin_tool_button_clicked), this);
  gtk_tool_button_set_label(mount_entries_button, gettext("Edit devices"));

  about_button = GTK_TOOL_BUTTON(gtk_tool_button_new_from_stock(GTK_STOCK_ABOUT));
  gtk_toolbar_insert(toolbar, GTK_TOOL_ITEM(about_button), -1);
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(about_button),
				 gettext("About mount-gtk"));
  g_signal_connect(G_OBJECT(about_button), "clicked",
		   G_CALLBACK(MainWindowCB::mainwin_tool_button_clicked), this);

  quit_button = GTK_TOOL_BUTTON(gtk_tool_button_new_from_stock(GTK_STOCK_QUIT));
  gtk_toolbar_insert(toolbar, GTK_TOOL_ITEM(quit_button), -1);
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(quit_button),
				 gettext("Quit"));
  g_signal_connect(G_OBJECT(quit_button), "clicked",
		   G_CALLBACK(MainWindowCB::mainwin_tool_button_clicked), this);

  populate_mount_points();

  g_signal_connect(G_OBJECT(get_win()), "visibility_notify_event",
                   G_CALLBACK(MainWindowCB::mainwin_visibility_notify_event), this);
  g_signal_connect(G_OBJECT(get_win()), "window_state_event",
                   G_CALLBACK(MainWindowCB::mainwin_window_state_event), this);

  // don't have entry widgets selecting on focus
  g_type_class_unref(g_type_class_ref(GTK_TYPE_ENTRY));
  GtkSettings* settings = gtk_settings_get_default();
  if (settings) {
    g_object_set(settings,
		 "gtk-entry-select-on-focus", false,
		 static_cast<void*>(0));
  }
  else write_error("Can't obtain default GtkSettings object\n");

  GtkWidget* mainwin = GTK_WIDGET(get_win());
  gtk_widget_add_events(mainwin, GDK_VISIBILITY_NOTIFY_MASK);
  gtk_window_set_resizable(get_win(), false);

  // button_vbox is shown in MainWindow::populate_mount_points()
  gtk_widget_show_all(GTK_WIDGET(toolbar));
  gtk_widget_show(gui_vbox);
  if (start_hidden && prog_config.tray_icon) {
    gtk_widget_realize(mainwin);
    gtk_widget_hide(mainwin);
    start_timeout_seconds(5, Callback::make(*this, &MainWindow::start_hidden_check_cb));
  }
  else gtk_widget_show(mainwin);

  if (prog_config.tray_icon) make_tray_icon();

  mounter.monitor_cb.connect(Callback::make(*this, &MainWindow::refresh_display));
}

void MainWindow::on_delete_event() {
  if (prog_config.tray_icon) gtk_widget_hide(GTK_WIDGET(get_win()));
  else close();
}

void MainWindow::prefs_impl() {

  if (in_dialog) return;

  in_dialog = true;

  bool old_tray_icon = prog_config.tray_icon;
  bool old_tooltip = prog_config.tooltip;
  Preferences preferences{get_win()};
  preferences.exec();

  if (old_tray_icon != prog_config.tray_icon) {
    if (prog_config.tray_icon) make_tray_icon();
    else tray_icon_a.reset();
  }
  if (old_tooltip != prog_config.tooltip)
    reset_mount_tooltips();

  in_dialog = false;
}

// this is called if the user elects to change preferences about
// displaying mount point tooltips
void MainWindow::reset_mount_tooltips() const {

  for (const auto& elt: mount_list) reset_mount_tooltips_item(elt);
}

// called by reset_mount_tooltips() and display_mount_points()
void MainWindow::reset_mount_tooltips_item(const SharedPtr<MountItem>& item) const {
  if (prog_config.tooltip && mounter.is_mounted(item->device)) {
    gchar* str = g_strdup_printf(gettext("Mounted at: %s"),
				 mounter.get_mount_point(item->device).c_str());
    gtk_widget_set_tooltip_text(item->button, str);
    g_free(str);
  }
  else gtk_widget_set_has_tooltip(item->button, false);
}

// this is called if any mount points change
void MainWindow::refresh_display() const {

  for (const auto& elt: mount_list) refresh_display_item(elt);
}

void MainWindow::refresh_display_item(const SharedPtr<MountItem>& item) const {

  GtkWidget* box = gtk_bin_get_child(GTK_BIN(item->button));
  if (!box) {
    write_error("Can't find child widget of button\n", false);
    return;
  }

  GtkWidget* drawing_area = 0;
  GtkWidget* label = 0;
  GList* list = gtk_container_get_children(GTK_CONTAINER(box));
  if (list) {
    GList* first = g_list_first(list);
    GList* last = g_list_last(list);
    if (first) drawing_area = GTK_WIDGET(first->data);
    if (last) label = GTK_WIDGET(last->data);
    g_list_free(list);
  }

  if (!drawing_area || !label || drawing_area == label) {
    write_error("Can't find child widgets of button hbox\n", false);
    return;
  }

  // test for change of status
  if (mounter.is_mounted(item->device)
      && !GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(drawing_area),
					     "is_mounted"))) {
    gtk_widget_set_sensitive(item->button, true);
    gtk_label_set_text(GTK_LABEL(label), gettext("Unmount"));
    g_object_set_data(G_OBJECT(drawing_area), "is_mounted",
		      GUINT_TO_POINTER(true));
    gtk_widget_queue_draw(drawing_area);
    if (prog_config.tooltip) {
      gchar* str = g_strdup_printf(gettext("Mounted at: %s"),
				   mounter.get_mount_point(item->device).c_str());
      gtk_widget_set_tooltip_text(item->button, str);
      g_free(str);
    }
  }
  else if (!mounter.is_mounted(item->device)
	   && GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(drawing_area),
						 "is_mounted"))){
    gtk_widget_set_sensitive(item->button, true);
    gtk_label_set_text(GTK_LABEL(label), gettext("Mount"));
    g_object_set_data(G_OBJECT(drawing_area), "is_mounted",
		      GUINT_TO_POINTER(false));
    gtk_widget_queue_draw(drawing_area);
    if (prog_config.tooltip)
      gtk_widget_set_has_tooltip(item->button, false);
  }
}

void MainWindow::set_button_size(const SharedPtr<MountItem>& item) const {

  gtk_widget_set_size_request(item->button, button_extents.first, button_extents.second);
}

void MainWindow::operation_fail(GtkWidget* widget) {

  // test against button_set whether the widget in question (one of the
  // mount/unmount buttons) still exists.  (It is theoretically possible that
  // a new button may have been allocated to the same address as a defunct one and
  // the user is doing a mount/unmount operation on the new button as well as the
  // old one, but in the very unlikely event of that happening it doesn't really
  // matter - the worst that could happen is that the new button is made sensitive
  // earlier than it otherwise would (ie when a mount/unmount operation on it is still
  // in course), so leading to a duplicated attempt by the user to mount or unmount
  // on the same device
  if (button_set.find(widget) != button_set.end()) gtk_widget_set_sensitive(widget, true);
}

void MainWindow::mount_entries_impl() {

  if (in_dialog) return;
  in_dialog = true;

  MountEntries mount_entries{get_win()};
  if (mount_entries.exec()) populate_mount_points();

  in_dialog = false;
}

void MainWindow::about_impl() {

  // stop the non-modal About dialog being shown via the tray
  // icon if a blocking dialog is being shown, but don't cause
  // the presence of the About dialog to prevent a subsequent
  // blocking dialog from being shown
  if (in_dialog) return;

  new AboutDialog;
  // there is no memory leak - AboutDialog is self-owning
  // and will delete itself when it is closed
}

void MainWindow::button_cb(int cb_index) {

  if (cb_index >= static_cast<int>(mount_list.size())) {
    beep();
    write_error("Yikes! Index error in MainWindow::button_cb()\n", false);
  }
  else {
    gtk_widget_set_sensitive(mount_list[cb_index]->button, false);

    if (mounter.is_mounted(mount_list[cb_index]->device)) {
      mounter.unmount(mount_list[cb_index]->device,
                      Callback::make(*this, &MainWindow::operation_fail,
                                     mount_list[cb_index]->button));
    }
    else {
      mounter.mount(mount_list[cb_index]->device,
                    Callback::make(*this, &MainWindow::operation_fail,
                                   mount_list[cb_index]->button));
    }
  }
}

std::pair<int, int> MainWindow::get_max_button_text_extents() {

  int max_width = 0;
  int max_height = 0;

  if (!mount_list.empty()) {

    std::vector<std::string> text_vec;
    text_vec.emplace_back(gettext("Mount"));
    text_vec.emplace_back(gettext("Unmount"));

    int width = 0;
    int height = 0;

    for (const auto& elt: text_vec) {
      // any button will do to get a pango layout (we use the first button in mount_list)
      GobjHandle<PangoLayout> pango_layout(gtk_widget_create_pango_layout(mount_list.front()->button,
									  elt.c_str()));
      pango_layout_get_pixel_size(pango_layout, &width, &height);

      if (width > max_width) {
	max_width = width;
      }
      if (height > max_height) {
	max_height = height;
      }
    }
  }
  return std::pair<int, int>(max_width, max_height);
}

std::pair<int, int> MainWindow::get_max_button_padding_extents() {
#if (GTK_CHECK_VERSION(2,99,0))
  // there is no need to check whether mount_list is empty here,
  // because this method is only called by
  // button_style_updated_impl(), and only when mount_list is not
  // empty
  GtkStyleContext* context = gtk_widget_get_style_context(mount_list.front()->button);
  GtkBorder border;
  GtkStateFlags state = gtk_style_context_get_state(context);
  gtk_style_context_get_border(context, state, &border);
  int width = border.left + border.right;
  int height = border.top + border.bottom;
  gtk_style_context_get_padding(context, state, &border);
  width += border.left + border.right;
  height += border.top + border.bottom;

  return std::pair<int, int>(std::max(width + 6, 18), std::max(height + 2, 12));
#else
  return std::pair<int, int>(18, 12);
#endif
}

void MainWindow::button_style_updated_impl() {

  button_extents = get_max_button_text_extents();

  if (button_extents.first) {

    // add padding
    std::pair<int, int> padding = get_max_button_padding_extents();
    button_extents.first += padding.first;   // width
    button_extents.second += padding.second; // height

    // add provision for the drawing area
    button_extents.first += MAINWIN_DRAWING_AREA_WIDTH;

    // have a minimum height for the drawing area whatever font is chosen
    const int min_height = MAINWIN_DRAWING_AREA_HEIGHT + 12;
    if (button_extents.second < min_height) button_extents.second = min_height;

    for (const auto& elt: mount_list) set_button_size(elt);
  }
}

#if !(GTK_CHECK_VERSION(2,99,0))
void MainWindow::button_style_set_impl(GtkStyle* prev_style) {

  GtkStyle* new_style = gtk_widget_get_style(GTK_WIDGET(get_win()));
  // we only need to proceed further if the font has changed
  if (!prev_style || !pango_font_description_equal(prev_style->font_desc,
						   new_style->font_desc)) {

    button_style_updated_impl();
  }
}
#endif // !(GTK_CHECK_VERSION(2,99,0))

#if GTK_CHECK_VERSION(2,99,0)
bool MainWindow::draw_mount_indicator(GtkWidget* drawing_area,
				      cairo_t* cairo) {

  if (gtk_widget_get_realized(drawing_area)) { // if we started hidden in the system tray
                                               // the drawing area may not yet be realised
    cairo_rectangle(cairo, 
		    1, 1,
		    gtk_widget_get_allocated_width(drawing_area) - 2,
		    gtk_widget_get_allocated_height(drawing_area) - 2);
    if (GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(drawing_area),
					   "is_mounted")))
      cairo_set_source_rgb(cairo, 0.7, 0.0, 0.0);

    else
      cairo_set_source_rgb(cairo, 0.0, 0.7, 0.0);
    cairo_fill_preserve(cairo);
    cairo_set_source_rgb(cairo, 1.0, 1.0, 1.0);
    cairo_set_line_width(cairo, 2);
    cairo_stroke(cairo);
  }
  // processing stops here
  return true;
}
#else
bool MainWindow::draw_mount_indicator(GtkWidget* drawing_area,
				      GdkEventExpose* event) {

  if (GTK_WIDGET_REALIZED(drawing_area)) { // if we started hidden in the system tray
                                           // the drawing area may not yet be realised

    GdkDrawable* drawable = GDK_DRAWABLE(drawing_area->window);

    // provide the cairo context incorporating the destination surface
    CairoContextScopedHandle cairo(gdk_cairo_create(drawable));

    // clip any redrawing to the area of the expose event
    cairo_rectangle(cairo,
		    event->area.x, event->area.y,
		    event->area.width, event->area.height);
    cairo_clip(cairo);

    // now draw the mount indicator
    cairo_rectangle(cairo, 
		    1, 1,
		    (drawing_area->allocation.width) - 2,
		    (drawing_area->allocation.height) - 2);
    if (GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(drawing_area),
					   "is_mounted")))
      cairo_set_source_rgb(cairo, 0.7, 0.0, 0.0);

    else
      cairo_set_source_rgb(cairo, 0.0, 0.7, 0.0);
    cairo_fill_preserve(cairo);
    cairo_set_source_rgb(cairo, 1.0, 1.0, 1.0);
    cairo_set_line_width(cairo, 2);
    cairo_stroke(cairo);
  }
  // processing stops here
  return true;
}
#endif

void MainWindow::populate_mount_points() {

  bool rebuild = false;
  if (button_vbox) {  // not program start-up - rebuild interface
    if (!mount_list.empty()) {
      g_signal_handler_disconnect(mount_list.front()->button,
				  button_style_connection);
    }
    gtk_widget_destroy(button_vbox);
    mount_list.clear();
    button_set.clear();
    rebuild = true;
  }

  gchar** device_list = 0;
  gchar** label_list = 0;

  if ((device_list = g_key_file_get_string_list(prog_config.key_file,
						"Mounts",
						"Devices",
						0, 0))
      && (label_list = g_key_file_get_string_list(prog_config.key_file,
						  "Mounts",
						  "Labels",
						  0, 0))
      && *device_list) {
    
    gchar** device_tmp;
    gchar** label_tmp;
    for (device_tmp = device_list, label_tmp = label_list;
	 device_tmp && *device_tmp && label_tmp && *label_tmp;
	 ++device_tmp, ++label_tmp) {

      SharedPtr<MountItem> mount_item_s{new MountItem};
      mount_item_s->device = g_strstrip(*device_tmp);
      mount_item_s->label = g_strstrip(*label_tmp);

      mount_list.push_back(mount_item_s);

    }
  }
  // nothing in this function can throw - this is fine
  // g_strfreev() does nothing if the argument is NULL
  g_strfreev(device_list);
  g_strfreev(label_list);

  if(!mount_list.empty()) {
    button_vbox = gtk_vbox_new(true, 2);
    gtk_box_pack_start(GTK_BOX(gui_vbox), button_vbox,
		       false, false, 0);
    display_mount_points();

    if (rebuild) { // this is a rebuild - we have previously called set_button_size() via
                   // button_style_updated_impl() when the window was first realised - we
                   // now need to do it for the new set of buttons

      if (button_extents.first) { // we already had buttons before for the current style,
	                          // so reuse the button extents
	for (const auto& elt: mount_list) set_button_size(elt);
      }
      else {       // no button extents for the current style, start from scratch.
                   // We cannot rely on just the style changed signal connection
	           // below causing button_style_updated_impl() to have been called on
	           // program start up, because we may have started without any mount
	           // points
	button_style_updated_impl();
      }
    }
    // any button will do to respond to a button style change (we use the first
    // button in mount_list).  The call to DisplayMountpointFunc() above will
    // have constructed the buttons.
#if GTK_CHECK_VERSION(2,99,0)
    button_style_connection = g_signal_connect(G_OBJECT(mount_list.front()->button),
					       "style_updated",
					       G_CALLBACK(MainWindowCB::mainwin_button_style_updated),
					       this);
#else
    button_style_connection = g_signal_connect(G_OBJECT(mount_list.front()->button),
					       "style_set",
					       G_CALLBACK(MainWindowCB::mainwin_button_style_set),
					       this);
#endif
  }
  else {
    button_vbox = get_noentries_widget();
    gtk_box_pack_start(GTK_BOX(gui_vbox), button_vbox,
		       false, false, 0);
  }
  gtk_widget_show_all(button_vbox);
}


void MainWindow::display_mount_points() {
  int button_count = 0;
  for (const auto& elt: mount_list) {
    GtkWidget* frame = gtk_frame_new(0);
    gtk_box_pack_start(GTK_BOX(button_vbox), frame,
		       false, false, 0);

    GtkWidget* hbox = gtk_hbox_new(false, 2);
    gtk_container_add(GTK_CONTAINER(frame), hbox);

    GtkWidget* label = gtk_label_new(elt->label.c_str());
    gtk_box_pack_start(GTK_BOX(hbox), label,
		       true, true, 4);
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
    
    GtkWidget* button = gtk_button_new();
    gtk_box_pack_start(GTK_BOX(hbox), button,
		       false, false, 0);
    g_signal_connect_data(G_OBJECT(button), "clicked",
			  G_CALLBACK(MainWindowCB::mainwin_mount_button_clicked),
			  Callback::make(*this, &MainWindow::button_cb, button_count++),
			  MainWindowCB::mainwin_delete_data, GConnectFlags(0));
    elt->button = button;
    button_set.insert(button);
    reset_mount_tooltips_item(elt);

    // hbox for insertion in the button
    hbox = gtk_hbox_new(false, 2);
    gtk_container_add(GTK_CONTAINER(button), hbox);

    GtkWidget* drawing_area = gtk_drawing_area_new();
    gtk_box_pack_start(GTK_BOX(hbox), drawing_area,
		       false, false, 0);
    g_object_set_data(G_OBJECT(drawing_area), "is_mounted",
		      GUINT_TO_POINTER(mounter.is_mounted(elt->device)));
#if GTK_CHECK_VERSION(2,99,0)
    g_signal_connect(G_OBJECT(drawing_area), "draw",
		     G_CALLBACK(MainWindowCB::mainwin_drawing_area_draw),
		     0);
#else
    g_signal_connect(G_OBJECT(drawing_area), "expose_event",
		     G_CALLBACK(MainWindowCB::mainwin_drawing_area_expose_event),
		     0);
#endif
    gtk_widget_set_size_request(drawing_area,
				MAINWIN_DRAWING_AREA_WIDTH,
				MAINWIN_DRAWING_AREA_HEIGHT);
    if(mounter.is_mounted(elt->device)) {
      label = gtk_label_new(gettext("Unmount"));
    }
    else {
      label = gtk_label_new(gettext("Mount"));
    }
    gtk_box_pack_start(GTK_BOX(hbox), label,
		       true, true, 0);
  }
}
GtkWidget* MainWindow::get_noentries_widget() {

  GtkWidget* button = gtk_button_new();
  g_signal_connect(G_OBJECT(button), "clicked",
		   G_CALLBACK(MainWindowCB::mainwin_noentries_button_clicked), this);

  GtkWidget* hbox = gtk_hbox_new(false, 2);
  gtk_container_add(GTK_CONTAINER(button), hbox);

  GtkWidget* image = gtk_image_new_from_stock(GTK_STOCK_EDIT, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(hbox), image,
		     false, false, 0);

  GtkWidget* label = gtk_label_new(gettext("Add device to mount"));
  gtk_box_pack_start(GTK_BOX(hbox), label,
		     true, true, 4);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

  return button;
}

void MainWindow::tray_icon_action(TrayIcon::MenuItem item) {

  switch(item) {
  case TrayIcon::prefs:
    prefs_impl();
    break;
  case TrayIcon::devices:
    mount_entries_impl();
    break;
  case TrayIcon::about:
    about_impl();
    break;
  case TrayIcon::quit:
    close();
    break;
  default:
    write_error("Invalid action passed to MainWindow::tray_icon_action()\n");
    break;
  }
}

void MainWindow::tray_icon_activate() {

  GtkWidget* main_widget = GTK_WIDGET(get_win());
  // test GTK_WIDGET_VISIBLE() as well as the 'obscured' variable - if we call
  // hide() on the window it does not count as a visibility notify event!
#if GTK_CHECK_VERSION(2,20,0)
  if (gtk_widget_get_visible(main_widget)
      && !obscured
      && !minimised
      && gtk_widget_get_sensitive(main_widget)) {
    gtk_widget_hide(main_widget);
  }
#else
  if (GTK_WIDGET_VISIBLE(main_widget)
      && !obscured
      && !minimised
      && GTK_WIDGET_SENSITIVE(main_widget)) {
    gtk_widget_hide(main_widget);
  }
#endif
  else gtk_window_present(get_win());
}

void MainWindow::make_tray_icon() {

  tray_icon_a.reset(new TrayIcon);
  tray_icon_a->set_tooltip("Mount-gtk");
  tray_icon_a->activate_cb.connect(Callback::make(*this, &MainWindow::tray_icon_activate));
  tray_icon_a->popup_cb.connect(Callback::make(*this, &MainWindow::tray_icon_action));
}

void MainWindow::start_hidden_check_cb(bool& keep_timeout) {
  if (!tray_icon_a.get() || !tray_icon_a->is_embedded()) {
    // the user has goofed - he has set the program to start hidden in the system tray
    // but has no system tray running!
    gtk_window_present(get_win());
    write_error(gettext("The program has been started with the -s option but there is no system tray!\n"),
		true);
  }
  keep_timeout = false; // this only fires once
}

void MainWindow::present() {
  gtk_window_present(get_win());
}
