/*  cairo_src.c
 *
 *  Initial Implementation: 2011/06
 *
 *  The purpose of this library is to provide easy access from
 *  a Fortran 77/90 program to the Cairo library.
 *
 *  Cairo is a relatively new Unix/Linux graphics library that
 *  supports a number of common devices.  It has a structure that
 *  appears to modeled on Postscript.
 *
 *  Although I wrote this wrapper with a specific application in
 *  mind, I believe it may well be useful for other Fortran
 *  codes.  This code may be used and modified by anyone without
 *  restriction.  
 *
 *  A dummy version of this routine is maintained for systems that
 *  do not support Cairo.
 *
 *  Note that calling C from Fortran is not standard.  I have
 *  provided the following compiler defintions to enhance portability.
 *
 *  1) The default is to assume that the Fortran compiler appends an
 *     underscore to the routine name.  Use -DNOUNDERSCORE if your
 *     compiler does not append the underscore.
 *  2) The default is to assume that the Fortran compiler converts
 *     routine names to lower case.  Use -DUPPERCASE if your
 *     Fortran compiler does not do this (e.g., the Cray).
 *  3) Many Unix compilers support a "-r8", or something similar,
 *     to make single precision 64-bit.  Use -DDOUBLE if you
 *     compile your Fortran with this option.
 *  4) Character strings are the most troublesome issue for
 *     portability.  Passing character strings from Fortran to C
 *     is very compiler dependent.  I have addressed this issue
 *     by passing character strings as arrays of ASCII Decimal
 *     Equivalents (ADE's).  That is, the Fortran converts a
 *     character string to an array of the integer values where
 *     the integer is the ASCII collating sequence (i.e., A = 65,
 *     B = 66, etc.).  The end of the string is denoted by setting
 *     the value to 0.  This is easily accomplished on the Fortran
 *     side by using the ICHAR function.  The C code here then
 *     calls an internal routine to covnert the integer array to
 *     a C string.   Although a bit convoluted, this avoids a lot
 *     of messy portability issues.
 *
 *
 *  The following routines are included:
 *
 *  cainit      - initialize Cairo
 *  caend       - close Cairo
 *  carend      - render the current plot
 *  caeras      - start a new graph (close currently open one as well)
 *  cadraw      - draw a polyline
 *  camove      - move to a point
 *  caseco      - set foreground color (based on color-map table)
 *  casec2      - set foreground color (based on RGB triplet)
 *  casepa      - set line pattern
 *  capoin      - draw a point (i.e., a pixel)
 *  cacirc      - draw a circle
 *  cargfl      - solid fill of a region
 *  catxth      - draw a horizontal character string
 *  catxtv      - draw a vertical character string
 *  carelo      - read mouse position
 *  i_to_s_5    - utility routine to convert array of ADE's to string
 *                array
 *
 */

/*  Site dependent definitions (see comments above) */
/*  Default is an underscore and lower case.  The compiler specified
 *  definitions -DNOUNDERSCORE and -DUPPERCASE can be specified to
 *  override these defaults. */

#ifdef NOUNDERSCORE
#define APPEND_UNDERSCORE 0
#else
#define APPEND_UNDERSCORE 1
#endif
#ifdef UPPERCASE
#define SUBROUTINE_CASE 0
#else
#define SUBROUTINE_CASE 1
#endif
#ifdef DOUBLE
#define PRECISION 1
#else
#define PRECISION 0
#endif
#ifdef INTEGER8
#define INTEGER_PRECISION 1
#else
#define INTEGER_PRECISION 0
#endif

/*  include files */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <strings.h>
#include <cairo.h>
#include <cairo-ps.h>
#include <cairo-pdf.h>
#include <cairo-quartz.h>
#include <cairo-xlib.h>
#include <svg.h>
#include <svg-cairo.h>

/* global definitions */

cairo_surface_t *surface;
cairo_t *cr;
FILE *file;

#define FONT_HEIGHT_CURRENT  0
#define FONT_GAP_CURRENT     0

int decodeEvent(char *event, int *x, int *y);

/* flags for current attribute settings */
static int    OPEN_FLAG = 0;          /* 0 - closed, 1 - open */

#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void  caend_(), cadraw_(), capoin_(), cacirc_(), cargfl_();
void  cainit_(), caeras_(), catxth_(), catxtv_();
void  caseco_(), casec2(), casepa_(), carend_(), caqrelo_();
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void  CAEND_(), CAINIT_(), CADRAW_(), CAPOIN_(), CACIRC_(), CARGFL_();
void  CAINIT_(), CAERAS_(), CATXTH_(), CATXTV_();
void  CASECO_(), CASEC2(), CASEPA_(), CAREND_(), CARELO_(),;
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void  caend(), cadraw(), capoin(), cacirc(), cargfl();
void  cainit(), caeras(), catxth(), catxtv(),catatt();
void  caseco(), casec2(), casepa(), carend(), carelo();
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void  CAEND(),  CADRAW(), CAPOIN(), CACIRC(), CARGFL();
void  CAINIT(), CAERAS(), CATXTH(), CATXTV();
void  CASECO(), CASEC2(), CASEPA(), CARELO();
#endif
void  i_to_s_5();

/* CAINIT  - routine to initialize Cairo.
 *
 * Dataplot currently supports the following Cairo devices:
 *
 *    1 - PDF
 *    2 - Postscript
 *    3 - X11
 *    4 - SVG
 *    5 - Quartz
 *    6 - WIN32
 *    7 - PNG
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void cainit_(idev,nplot,anumhp,anumvp,ired,igreen,iblue,maxclr)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CAINIT_(idev,nplot,anumhp,anumvp,ired,igreen,iblue,maxclr)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void cainit(idev,nplot,anumhp,anumvp,ired,igreen,iblue,maxclr)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CAINIT(idev,nplot,anumhp,anumvp,ired,igreen,iblue,maxclr)
#endif
int  ired[];
int  igreen[];
int  iblue[];
#if INTEGER_PRECISION == 0
int  *maxclr;
int  *nplot, *anumhp, *anumvp, *idev;
#else
int  maxclr[2];
int  nplot[2], anumhp[2], anumvp[2], idev[2], iback[2];
#endif

{
int    nplot_temp, anumhp_temp, anumvp_temp, idev_temp;
int    maxclr_temp;
int    ival1, ival2, ival3;
int    i;
float  val1, val2, val3;
#if INTEGER_PRECISION == 0
   maxclr_temp = *maxclr;
   nplot_temp  = *nplot;
   anumhp_temp = *anumhp;
   anumvp_temp = *anumvp;
   idev_temp = *idev;
#else
   maxclr_temp = *maxclr[0];
   nplot_temp  = nplot[0];
   anumhp_temp = anumhp[0];
   anumvp_temp = anumvp[0];
   idev_temp = idev[0];
#endif
    
    if (OPEN_FLAG == 0) {           /*  Device currently closed */
       OPEN_FLAG = 1;
       if (idev_temp == 1) {        /*  PDF device */
          surface = cairo_pdf_surface_create ("plot.pdf", WIDTH, HEIGHT);
          cr = cairo_create (surface);
       elseif (idev_temp == 2) {    /*  Postscript device */
          surface = cairo_ps_surface_create ("plot.ps", WIDTH, HEIGHT);
          cr = cairo_create (surface);
       elseif (idev_temp == 3) {    /*  X11 device */
       elseif (idev_temp == 4) {    /*  SVG device */
       elseif (idev_temp == 5) {    /*  Quartz device */
       elseif (idev_temp == 6) {    /*  WIN32 device */
          surface = cairo_win32_surface_create (hdc);
          cr = cairo_create (surface);
       elseif (idev_temp == 7) {    /*  PNG device */
          surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
                    IMAGE_WIDTH, IMAGE_HEIGHT);
          cr = cairo_create (surface);
          cairo_save (cr);
       }
    }
}

/* CAERAS  - routine to clear the screen.
 *
 *           1) Check if a plot is currently open.  If yes, write
 *              it to a file and destroy the current image.
 *           2) Create a new image with the specified size specified
 *              in pixels.  Note that orientation (landscape, portrait,
 *              is implicit in the pixel dimensions).  Note that
 *              this routine does not modify the values.
 *           3) Set all colors to be undefined and then set
 *              background and foreground colors.
 *
 *  xpixels   - width (in pixels) for graphics window
 *  ypixels   - height (in pixels) for graphics window
 *  back_col  - background color
 *  file name - file name (in integer ascii decimal equivalents)
 *
 */

#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void caeras_(nplot, anumhp, anumvp, iback)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CAERAS_(nplot, anumhp, anumvp, iback)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void caeras(nplot, anumhp, anumvp, iback)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CAERAS(nplot, anumhp, anumvp, iback)
#endif
#if INTEGER_PRECISION == 0
int  *nplot, *anumhp, *anumvp, *iback;
#else
int  nplot[2], anumhp[2], anumvp[2], iback[2];
#endif
{

   int   nplot_temp, anumhp_temp, anumvp_temp, iback_temp;

#if INTEGER_PRECISION == 0
   nplot_temp  = *nplot;
   anumhp_temp = *anumhp;
   anumvp_temp = *anumvp;
   iback_temp  = *iback;
#else
   nplot_temp  = nplot[0];
   anumhp_temp = anumhp[0];
   anumvp_temp = anumvp[0];
   iback_temp  = iback[0];
#endif

   /* Use "erase rectangle" method to clear screen  */
   if (OPEN_FLAG > 0) {

   }
}

/* CAEND   - routine to end Cairo display and close the display
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void caend_()
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CAEND_()
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void caend()
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CAEND()
#endif

{
   if (OPEN_FLAG > 0) {
      OPEN_FLAG = 0;
   }
}

/* CAREND   - routine to render current plot
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void carend_()
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CAREND_()
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void carend()
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CAREND()
#endif

{
   if (OPEN_FLAG > 0) {
   }
}

/* CADRAW  - draw a polyline.  The line attributes are set in
 *           other routines.
 *
 * xpts   - contains the x coordinates
 * ypts   - contains the y coordinates
 * npts   - the number of points to plot
 * icap   - cap style: 1 = Butt
 *                     2 = Round
 *                     3 = Square
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void cadraw_(xpts, ypts, npts, icap)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CADRAW_(xpts, ypts, npts, icap)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void cadraw(xpts, ypts, npts)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CADRAW(xpts, ypts, npts)
#endif
float   xpts[], ypts[];
#if INTEGER_PRECISION == 0
int   *npts;
int   *icap;
#else
int   npts[2];
int   icap[2];
#endif
{
   int     npts_temp;
   int     icap_temp;

#if INTEGER_PRECISION == 0
   npts_temp = *npts;
   icap_temp = *icap;
#else
   npts_temp = *npts[0];
   icap_temp = *icap[0];
#endif

   if (OPEN_FLAG > 0) {
      cairo_move_to(cr,xpts[0],ypts[0]);
      for (i=1; i = npts_temp; i++) {
         cairo_line_to(cr,xpts[i],ypts[i]);
      }
      cairo_stroke(cr);
   }

}

/* CAMOVE  - move to a point
 *
 * ax1    - contains the x coordinate
 * ay1    - contains the y coordinate
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void camove_(ax1, ay1)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CAMOVE_(ax1,a1)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void camove(ax1,ay1)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CAMOVE(ax1,ay1)
#endif
#if PRECISION == 0
float   *ax1, *ay1;
#else
float ax1[2], ay1[2];
#endif
{
   float   ax1_temp;
   float   ay1_temp;

#if PRECISION == 0
   ax1_temp = *ax1;
   ay1_temp = *ay1;
#else
   ax1_temp = ax1[0];
   ay1_temp = ay1[0];
#endif

   if (OPEN_FLAG > 0) {
      cairo_move_to(cr,ax1_temp,ay1_temp);
   }

}

/* CASECO  - set the color
 *
 * jcol   - index for desired color
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void caseco_(jcol)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CASECO_(jcol)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void caseco(jcol)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CASECO(jcol)
#endif
#if INTEGER_PRECISION == 0
int   *jcol;
#else
int   jcol[2];
#endif
{
   int    jcol_temp;
   float  avalue;

#if INTEGER_PRECISION == 0
   jcol_temp = *jcol;
#else
   jcol_temp = jcol[0];
#endif

   if (OPEN_FLAG > 0) {
      if (jcol_temp > 0) {
      } else {
      }
   }

}

/* CASEC2  - set the color using RGB specification
 *
 * jred   - index for red component
 * jgreen - index for green component
 * jblue  - index for blue component
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void casec2_(jred,jgreen,jblue)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CASEC2_(jred,jgreen,jblue)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void casec2(jred,jgreen,jblue)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CASEC2(jred,jgreen,jblue)
#endif
#if PRECISION == 0
float   *jred;
float   *jgreen;
float   *jblue;
#else
float   jred[2];
float   jgreen[2];
float   jblue[2];
#endif
{
   float    avalue1;
   float    avalue2;
   float    avalue3;

#if PRECISION == 0
   avalue1 = *jred;
   avalue2 = *jgreen;
   avalue3 = *jblue;
#else
   avalue1 = jred[0];
   avalue2 = jgreen[0];
   avalue3 = jblue[0];
#endif

   if (avalue1 < 0) {
      avalue1 = 0.0;
   }
   if (avalue1 > 1) {
      avalue1=1.0;
   }
   if (avalue2 < 0) {
      avalue2 = 0.0;
   }
   if (avalue2 > 1) {
      avalue2=1.0;
   }
   if (avalue3 < 0) {
      avalue3 = 0.0;
   }
   if (avalue3 > 1) {
      avalue3=1.0;
   }

   if (OPEN_FLAG > 0) {
   }

}

/* CASEPA  - set line attribute (color set in AQSECO):
 *
 * jpatt   - the line pattern
 * pthick  - the line thickness (in points)
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void casepa_(xpatt,npatt,pthick,iopt)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CASEPA_(xpatt,npatt,pthick,iopt)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void casepa(xpatt,npatt,pthick,iopt)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CASEPA(xpatt,npatt,pthick,iopt)
#endif
float  xpatt[];
#if PRECISION == 0
float *pthick;
#else
float pthick[2];
#endif
#if INTEGER_PRECISION == 0
int   *npatt;
int   *iopt;
#else
int   npatt[2];
int   iopt[2];
#endif
{
   int     npatt_temp;
   float   pthick_temp;
   int     iopt_temp;

#if PRECISION == 0
   pthick_temp = *pthick;
#else
   pthick_temp = pthick[0];
#endif
#if INTEGER_PRECISION == 0
   npatt_temp  = *npatt;
   iopt_temp   = *iopt;
#else
   npatt_temp  = npatt[0];
   iopt_temp   = iopt[0];
#endif

   if (OPEN_FLAG > 0) {
      if (iopt_temp == 1) {
         if (npatt_temp <= 1) {
         } else {
         }
      }
      if (iopt_temp == 2) {
      }
   }

}

/* CASESI  - set character size
 *
 * pheigh   - the desired point size
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void casesi_(pheigh)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CASESI_(pheigh)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void casesi(pheigh)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CASESI(pheigh)
#endif
#if PRECISION == 0
float *pheigh;
#else
float pheigh[2];
#endif
{
   float   pheigh_temp;

#if PRECISION == 0
   pheigh_temp = *pheigh;
#else
   pheigh_temp = pheigh[0];
#endif

   if (OPEN_FLAG > 0) {
   }

}

/* CAPOIN - draw a point.
 *
 * ix     - contains the x coordinate
 * iy     - contains the y coordinate
 * jcol   - color to use in drawing the point
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void capoin_(ix, iy, ired, igreen, iblue)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CAPOIN_(ix, iy, ired, igreen, iblue)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void capoin(ix, iy, ired, igreen, iblue)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CAPOIN(ix, iy, ired, igreen, iblue)
#endif
#if INTEGER_PRECISION == 0
int   *ix, *iy;
int   *ired, *igreen, *iblue;
#else
int   ix[2], iy[2];
int   ired[2], igreen[2], iblue[2];
#endif
{

   unsigned char rgbImage[3] = {255, 255, 255};
   int    ixtemp;
   int    iytemp;
   int    ired_temp;
   int    igreen_temp;
   int    iblue_temp;

#if INTEGER_PRECISION == 0
   ixtemp = *ix;
   iytemp = *iy;
   ired_temp   = *ired;
   igreen_temp = *igreen;
   iblue_temp  = *iblue;
#else
   ixtemp = ix[0];
   iytemp = iy[0];
   ired_temp = ired[0];
   igreen_temp = igreen[0];
   iblue_temp = iblue[0];
#endif

   rgbImage[0] = ired_temp;
   rgbImage[1] = igreen_temp;
   rgbImage[2] = iblue_temp;


}

/* CARECT - fill a rectangle.
 *
 * ix1    - start x position
 * iy1    - start y position
 * ix2    - length of x
 * iy2    - length of y
 *          otherwise, a convex polygon)
 *
 */
#define MAX_REG_POINTS  1000
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void carect_(ix1, iy1, ix2, iy2)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CARECT_(ix1, iy1, ix2, iy2)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void carect(ix1, iy1, ix2, iy2)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CARECT(ix1, iy1, ix2, iy2)
#endif
#if INTEGER_PRECISION == 0
int   *ix1;
int   *iy1;
int   *ix2;
int   *iy2;
#else
int   ix1[2];
int   iy1[2];
int   iy1[2];
int   iy2[2];
#endif
{
   int     ix1_temp;
   int     iy1_temp;
   int     ix2_temp;
   int     iy2_temp;

#if INTEGER_PRECISION == 0
   ix1_temp = *ix1;
   iy1_temp = *iy1;
   ix2_temp = *ix2;
   iy2_temp = *iy2;
#else
   ix1_temp = ix1[0];
   iy1_temp = iy1[0];
   ix2_temp = ix2[0];
   iy2_temp = iy2[0];
#endif

   if (OPEN_FLAG > 0) {
   }

}

/* CARGFL - fill a region.  Rectangular regions will be filled differently
 *          non-rectangular regions.  Dataplot only handles convex polygons,
 *          so set this (for faster performance).  This routine only does
 *          solid fills.  Hatch patterns must be drawn
 *          by the calling program (i.e., send the individual lines to
 *          the CADRAW routine).
 *
 * xpts   - contains the x coordinates
 * ypts   - contains the y coordinates
 * npts   - the number of points in the polygon (if 2, assume a rectangle,
 *          otherwise, a convex polygon)
 *
 */
#define MAX_REG_POINTS  1000
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void cargfl_(xpts, ypts, npts)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CARGFL_(xpts, ypts, npts)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void cargfl(xpts, ypts, npts)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CARGFL(xpts, ypts, npts)
#endif
float   xpts[], ypts[];
#if INTEGER_PRECISION == 0
int   *npts;
#else
int   npts[2];
#endif
{
   int     npts_temp;

#if INTEGER_PRECISION == 0
   npts_temp = *npts;
#else
   npts_temp = npts[0];
#endif

   if (OPEN_FLAG > 0) {
   }

}

/* CATXTH - draw a horizontal text string.
 *
 * string - text string to draw
 * ixpos  - x position
 * iypos  - y position
 * ijusth - justification (horizontal)
 *          0 - left justified
 *          1 - center justified
 *          2 - right justified
 * ijustv - justiciation (vertical)
 *          0 - center justified
 *          1 - bottom justified
 *          2 - top justified
 * font   - font name
 * error  - error flag
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void catxth_(string, ixpos, iypos, ijusth, ijustv, font, error)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CATXTH_(string, ixpos, iypos, ijusth, ijustv, font, error)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void catxth(string, ixpos, iypos, ijusth, ijustv, font, error)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CATXTH(string, ixpos, iypos, ijusth, ijustv, font, error)
#endif
int    string[];
int    font[];
#if INTEGER_PRECISION == 0
int    *ixpos;
int    *iypos;
int    *ijusth;
int    *ijustv;
int    *error;
#else
int    ixpos[2];
int    iypos[2];
int    ijusth[2];
int    ijustv[2];
int    error[2];
#endif
{

   int    len;                     /* number of characters in string */
   int    len2;                    /* number of characters in string */
   int    string_width;            /* width of string in pixels */
   char   string2[130];            /* converted string */
   char   font2[130];              /* font name */
   int    i;
   int    ixpos_temp;
   int    iypos_temp;
   int    ijusth_temp;
   int    ijustv_temp;
   int    error_temp;
   float  xpos;
   float  ypos;

#if INTEGER_PRECISION == 0
   ixpos_temp = *ixpos;
   iypos_temp = *iypos;
   ijusth_temp = *ijusth;
   ijustv_temp = *ijustv;
#else
   ixpos_temp = ixpos[0];
   iypos_temp = iypos[0];
   ijusth_temp = ijusth[0];
   ijustv_temp = ijustv[0];
#endif

#if INTEGER_PRECISION == 0
   i_to_s_5(string, string2, 130, &len);
   i_to_s_5(font, font2, 130, &len2);
#else
   i_to_s_5(string, string2, 260, &len);
   i_to_s_5(font, font2, 260, &len2);
#endif

   if (len2 > 1) {
   }

   xpos = (float)ixpos_temp;
   ypos = (float)iypos_temp;
   if (ijusth_temp == 0 && ijustv_temp == 0) {   /* Left-bottom */
   } else if (ijusth_temp == 0 && ijustv_temp == 1) {   /* Left-center */
   } else if (ijusth_temp == 0 && ijustv_temp == 2) {   /* Left-Top */
   } else if (ijusth_temp == 1 && ijustv_temp == 0) {   /* Center-bottom */
   } else if (ijusth_temp == 1 && ijustv_temp == 1) {   /* Center-Center */
   } else if (ijusth_temp == 1 && ijustv_temp == 2) {   /* Center-Top */
   } else if (ijusth_temp == 2 && ijustv_temp == 0) {   /* Right-bottom */
   } else if (ijusth_temp == 2 && ijustv_temp == 1) {   /* Right-Center */
   } else if (ijusth_temp == 2 && ijustv_temp == 2) {   /* Right-Top */
   } else {
   }

}

/* CATXTV - draw a vertical text string.
 *
 * string - text string to draw
 * ixpos  - x position
 * iypos  - y position
 * ijusth - justification (horizontal)
 *          0 - left justified
 *          1 - center justified
 *          2 - right justified
 * ijustv - justiciation (vertical)
 *          0 - center justified
 *          1 - bottom justified
 *          2 - top justified
 * font   - font name
 * error  - error flag
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void catxtv_(string, ixpos, iypos, ijusth, ijustv, font, error)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CATXTV_(string, ixpos, iypos, ijusth, ijustv, font, error)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void catxtv(string, ixpos, iypos, ijusth, ijustv, font, error)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CATXTV(string, ixpos, iypos, ijusth, ijustv, font, error)
#endif
int    string[];
int    font[];
#if INTEGER_PRECISION == 0
int    *ixpos;
int    *iypos;
int    *ijusth;
int    *ijustv;
int    *error;
#else
int    ixpos[2];
int    iypos[2];
int    ijusth[2];
int    ijustv[2];
int    error[2];
#endif
{

   int    len;                     /* number of characters in string */
   int    string_width;            /* width of string in pixels */
   char   string2[130];            /* converted string */
   char   font2[130];              /* font name */
   int    i;
   int    ixpos_temp;
   int    iypos_temp;
   int    ijusth_temp;
   int    ijustv_temp;
   int    error_temp;
   float  xpos;
   float  ypos;

#if INTEGER_PRECISION == 0
   ixpos_temp = *ixpos;
   iypos_temp = *iypos;
   ijusth_temp = *ijusth;
   ijustv_temp = *ijustv;
#else
   ixpos_temp = ixpos[0];
   iypos_temp = iypos[0];
   ijusth_temp = ijusth[0];
   ijustv_temp = ijustv[0];
#endif

#if INTEGER_PRECISION == 0
   i_to_s_5(string, string2, 130, &len);
   i_to_s_5(font, font2, 130, &len);
#else
   i_to_s_5(string, string2, 260, &len);
   i_to_s_5(font, font2, 260, &len);
#endif

   xpos = (float)ixpos_temp;
   ypos = (float)iypos_temp;
   if (ijusth_temp == 0 && ijustv_temp == 0) {   /* Left-bottom */
   } else if (ijusth_temp == 0 && ijustv_temp == 1) {   /* Left-center */
   } else if (ijusth_temp == 0 && ijustv_temp == 2) {   /* Left-Top */
   } else if (ijusth_temp == 1 && ijustv_temp == 0) {   /* Center-bottom */
   } else if (ijusth_temp == 1 && ijustv_temp == 1) {   /* Center-Center */
   } else if (ijusth_temp == 1 && ijustv_temp == 2) {   /* Center-Top */
   } else if (ijusth_temp == 2 && ijustv_temp == 0) {   /* Right-bottom */
   } else if (ijusth_temp == 2 && ijustv_temp == 1) {   /* Right-Center */
   } else if (ijusth_temp == 2 && ijustv_temp == 2) {   /* Right-Top */
   } else {
   }

}

/* CARDLO  - routine to read a position from the graphics window (used
 *           by the Dataplot cross-hair command).  The mouse position at
 *           the next mouse click will be determined.
 *
 *           This routine will wait until a mouse button is pressed.
 *
 * ixret   - x coordinate of mouse position when button pressed
 * iyret   - y coordinate of mouse position when button pressed
 * error   - error flag
 *
 */
#if APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 1
void cardlo_(ixret, iyret, error)
#elif APPEND_UNDERSCORE == 1 && SUBROUTINE_CASE == 0
void CARDLO_(ixret, iyret, error)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 1
void cardlo(ixret, iyret, error)
#elif APPEND_UNDERSCORE == 0 && SUBROUTINE_CASE == 0
void CARDLO(ixret, iyret, error)
#endif
#if INTEGER_PRECISION == 0
int  *ixret, *iyret, *error;
#else
int  ixret[2], iyret[2], error[2];
#endif
{
     int running = 0;
     int      x, y;


#if INTEGER_PRECISION == 0
       *ixret = -1;
       *iyret = -1;
       *error = 0;
#else
       ixret[0] = -1;
       iyret[0] = -1;
       error[0] = 0;
#endif


}

/* i_to_s_5  - utitlity routine to convert an integer array containing
 *           Ascii Decimal Equivalents to a character string array.  The
 *           Fortran routines pass character type data as an array of
 *           ADE's, which this routine then converts to C's character
 *           type.  Note that the input array is assumed to be correct
 *           (i.e., a value between 0 and 127) and no error checking is
 *           done on it.
 *
 * string1 - input array containing ADE's.
 * string2 - output array in C character format.
 * maxlen  - maximum length for string2
 * ilen    - length of character string
 *
 */
void i_to_s_5(string1, string2, maxlen, ilen)
int   string1[], maxlen, *ilen;
char  string2[];

{
     int  i;
     int  itemp;
     i = 0;
#if INTEGER_PRECISION == 0
     while (string1[i] != 0 && i < (maxlen - 1) ) {
         itemp = string1[i];
         string2[i] = string1[i];
         i++;
     }
     *ilen = i;
     string2[i]='\0';
}
#else
     while (string1[2*i] != 0 && i < (maxlen - 1) ) {
         itemp = string1[2*i];
         string2[i] = string1[2*i];
         i++;
     }
     *ilen = i;
     string2[i]='\0';
}
#endif

int decodeEvent(char *event, int *x, int *y)
{
   /* int x, y; */
   char **ap, *argv[10];
   
   /* Split arguments separated by a ':'  */
   for (ap = argv; (*ap = strsep(&event, ":")) != NULL;)
      if (**ap != '\0')
         if (++ap >= &argv[10])
            break;

   if (strcmp(argv[0], "42") >= 0)        /* Check for server error */
      return -1; 

   if (argv[0][0] != '1')                 /* Only check for mouse clicks */
      return 0;
   
   /* sscanf(argv[1], "{%d ,%d}", &x, &y);  */  /* Decode Position */
   return 1;
}

