
package Term::Size::Perl;

require Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(chars pixels);

our $VERSION = 0.01;

=head1 NAME

Term::Size::Perl - Perl extension for retrieving terminal size (Perl version)

=head1 SYNOPSIS

    use Term::Size::Perl;

    ($columns, $rows) = Term::Size::Perl::chars *STDOUT{IO};
    ($x, $y) = Term::Size::Perl::pixels;

=head1 DESCRIPTION

Yet another implementation of C<Term::Size>. Now
in pure Perl, with the exception of a C probe run
on build time.

=head1 SEE ALSO

It all began with L<Term::Size> by Tim Goodwin. You may want to
have a look at:

    Term::Size
    Term::Size::Unix
    Term::Size::Win32
    Term::Size::ReadKey (soon)

It would be helpful if you send me the F<Params.pm> generated by
the probe at build time.
Please reports bugs via CPAN RT, 
http://rt.cpan.org/NoAuth/Bugs.html?Dist=Term-Size-Perl

=head1 AUTHOR

A. R. Ferreira, E<lt>ferreira@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by A. R. Ferreira

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

require Term::Size::Perl::Params;
my %params = Term::Size::Perl::Params::params();

# ( row, col, x, y )
sub _winsize {
    my $h = shift || *STDIN;
    my $sz = "\0" x $params{winsize}{sizeof};
    ioctl($h, $params{TIOCGWINSZ}{value}, $sz)
        or return undef;
    return unpack $params{winsize}{mask}, $sz;
}

sub chars {
    my @sz = _winsize(shift);
    return @sz[1, 0] if wantarray;
    return $sz[1];
}

sub pixels {
    my @sz = _winsize(shift);
    return @sz[2, 3] if wantarray;
    return $sz[2];
}

1;
