#
# This file is part of MooseX-TrackDirty-Attributes
#
# This software is Copyright (c) 2011 by Chris Weyl.
#
# This is free software, licensed under:
#
#   The GNU Lesser General Public License, Version 2.1, February 1999
#
package MooseX::TrackDirty::Attributes;
BEGIN {
  $MooseX::TrackDirty::Attributes::AUTHORITY = 'cpan:RSRCHBOY';
}
# git description: 2.002-10-gb528db8
$MooseX::TrackDirty::Attributes::VERSION = '2.003';

# ABSTRACT: Track dirtied attributes

use warnings;
use strict;

use Moose 2.0 ();
use namespace::autoclean;
use Moose::Exporter;

Moose::Exporter->setup_import_methods(
    trait_aliases => [
        [ 'MooseX::TrackDirty::Attributes::Trait::Attribute' => 'TrackDirty' ],
    ],
);

!!42;

__END__

=pod

=encoding UTF-8

=for :stopwords Chris Weyl Ceccarelli Gianni attribute's HashRef's

=for :stopwords Wishlist flattr flattr'ed gittip gittip'ed

=head1 NAME

MooseX::TrackDirty::Attributes - Track dirtied attributes

=head1 VERSION

This document describes version 2.003 of MooseX::TrackDirty::Attributes - released December 23, 2014 as part of MooseX-TrackDirty-Attributes.

=head1 SYNOPSIS

    package Foo;
    use Moose;
    use MooseX::TrackDirty::Attributes;

    # tracking accessor is not automagically applied
    has foo => (is => 'rw');

    # one_is_dirty() is generated by default
    has one => (traits => [ TrackDirty ], is => 'rw');

    # dirtyness "accessor" is generated as two_isnt_clean()
    has two      => (
        traits   => [ TrackDirty ],
        is       => 'rw',
        is_dirty => 'two_isnt_clean',
    );

    # three_is_dirty() and original_value_of_three() are generated
    has three => (
        traits         => [ TrackDirty ],
        is             => 'rw',
        original_value => 'original_value_of_three',
    );

    # ...meanwhile, at the bat-cave

    package main;
    my $foo = Foo->new();

    $foo->one_is_dirty; # false

=head1 DESCRIPTION

MooseX::TrackDirty::Attributes does the necessary metaclass fiddling to track
if attributes are dirty; that is, if they're set to some value and then set
again, to another value.  (The setting can be done by the constructor,
builder, default, accessor, etc.)

An attribute can be returned to a clean state by invoking its clearer.

=head1 CAVEAT

Note that with one (significant) exceptions we can only track
dirtiness at the very first level.  That is, if you have an attribute that is
a HashRef, we can tell that the _attribute_ is dirty if and only if the
actual ref changes, but not if the HashRef's keys/values change. e.g.

    $self->hashref({ new => 'hash' })

would render the 'hashref' attribute dirty, but

    $self->hashref->{foo} = 'bar'

would not.

In plainer language: we can only tell if an attribute's value is dirty if our
accessors are used to modify its values.

The exception to this is...

=head1 IMPLICATIONS FOR NATIVE TRAITS

We now track when a native trait accessor is used to change the contents of
the attribute; this is considered to make the attribute value dirty.

This is still new and experimental, so feedback is quite welcome :)

=head1 ATTRIBUTE OPTIONS

To track a given attribute, the trait must be applied.  This package exports a
"TrackDirty" function that returns the full (ridiculously long) package name
of the trait.

Once applied, we have two additional options that can be passed to the
attribute constructor (usually via 'has'):

=over 4

=item is_dirty => method_name

is_dirty controls what the name of the "is this attribute's value dirty?"
accessor is (returning true on dirty; false otherwise):

By default, the accessor is installed as "{attribute_name}_is_dirty";

If a legal method name is passed, the accessor is installed under that name;

Otherwise we blow up.

=item original_value => method_name

original_value controls what the name for the original value accessor is
installed (returns the original value if dirty, undef otherwise):

By default, we do not install an original_value accessor;

If a legal method name is passed, the accessor is installed under that name;

Otherwise we blow up.

=item cleaner => method_name

cleaner controls what the name for the cleaner accessor is
installed (marks a dirty attribute as clean without clearing):

By default, we do not install an original_value accessor;

If a legal method name is passed, the accessor is installed under that name;

Otherwise we blow up.

=back

=head1 SOURCE

The development version is on github at L<http://https://github.com/RsrchBoy/moosex-trackdirty-attributes>
and may be cloned from L<git://https://github.com/RsrchBoy/moosex-trackdirty-attributes.git>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
https://github.com/RsrchBoy/moosex-trackdirty-attributes/issues

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

Chris Weyl <cweyl@alumni.drew.edu>

=head2 I'm a material boy in a material world

=begin html

<a href="https://www.gittip.com/RsrchBoy/"><img src="https://raw.githubusercontent.com/gittip/www.gittip.com/master/www/assets/%25version/logo.png" /></a>
<a href="http://bit.ly/rsrchboys-wishlist"><img src="http://wps.io/wp-content/uploads/2014/05/amazon_wishlist.resized.png" /></a>
<a href="https://flattr.com/submit/auto?user_id=RsrchBoy&url=https%3A%2F%2Fgithub.com%2FRsrchBoy%2Fmoosex-trackdirty-attributes&title=RsrchBoy's%20CPAN%20MooseX-TrackDirty-Attributes&tags=%22RsrchBoy's%20MooseX-TrackDirty-Attributes%20in%20the%20CPAN%22"><img src="http://api.flattr.com/button/flattr-badge-large.png" /></a>

=end html

Please note B<I do not expect to be gittip'ed or flattr'ed for this work>,
rather B<it is simply a very pleasant surprise>. I largely create and release
works like this because I need them or I find it enjoyable; however, don't let
that stop you if you feel like it ;)

L<Flattr this|https://flattr.com/submit/auto?user_id=RsrchBoy&url=https%3A%2F%2Fgithub.com%2FRsrchBoy%2Fmoosex-trackdirty-attributes&title=RsrchBoy's%20CPAN%20MooseX-TrackDirty-Attributes&tags=%22RsrchBoy's%20MooseX-TrackDirty-Attributes%20in%20the%20CPAN%22>,
L<gittip me|https://www.gittip.com/RsrchBoy/>, or indulge my
L<Amazon Wishlist|http://bit.ly/rsrchboys-wishlist>...  If you so desire.

=head1 CONTRIBUTOR

=for stopwords Gianni Ceccarelli

Gianni Ceccarelli <gianni.ceccarelli@net-a-porter.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2011 by Chris Weyl.

This is free software, licensed under:

  The GNU Lesser General Public License, Version 2.1, February 1999

=cut
