#! /usr/bin/env false

use v6.d;

use Config;
use IRC::Client;

use IRC::Client::Plugin::Reminders::Grammar;
use IRC::Client::Plugin::Reminders::Actions;

constant ReminderGrammar = IRC::Client::Plugin::Reminders::Grammar;
constant ReminderActions = IRC::Client::Plugin::Reminders::Actions;

unit class IRC::Client::Plugin::Reminders is IRC::Client::Plugin;

#| The configuration object for the bot.
has Config $.config;

#| A hash of prepared statements held by this plugin.
has %!stmt;

#| A connection to the database held by this plugin.
has $!db;

#| The interval in minutes in which the bot is supposed to run.
has $!interval;

method reload-config (
	Config $config,
) {
	$!config = $config;

	self.TWEAK;
}

multi method irc-to-me (
	$event where *.text.starts-with('remind me'),
) {
	# TODO: Allow people to set reminders for others?

	my $result = ReminderGrammar.parse(
		$event.text,
		:actions(ReminderActions),
	);

	return "I didn't quite catch that" unless $result;

	my %reminder = $result.made;

	%!stmt<create>.execute(
		%reminder<subject>,
		$event.usermask,
		$event.?channel // Any,
		%reminder<delay>.at,
	);

	"Reminding you to {%reminder<subject>} on {%reminder<delay>.at} (UTC)"
}

method irc-connected (
	$event,
) {
	.debug("Interval set to $!interval minute(s)") with $*LOG;

	# TODO: Check tables for reminders that were missed while the bot was
	# offline.

	# Check for reminders in a seperate thread, so we don't hang the bot.
	# TODO: Dynamic variable isn't set inside the start{}?
	start {
		react {
			whenever Supply.interval($!interval × 60) {
				CATCH {
					default {
						.crit('Error in reminder thread!') with $*LOG;
					}
				}

				.notice('Checking for reminders at ' ~ DateTime.now.utc) with $*LOG;

				my @results = %!stmt<read>.execute.hashes;

				for @results -> %result {
					# Send the reminder.
					my $text = "Reminder to {%result<subject>}";

					if (%result<created_in>) {
						$text = %result<created_by>.split('!', 2).first ~ ': ' ~ $text;
					}

					$event.irc.send(
						:$text,
						:where(%result<created_in> // %result<created_by>),
					);

					# And clean up the reminder from the
					# database.
					%!stmt<delete>.execute(%result<id>);
				}
			}
		}
	}
}

submethod TWEAK
{
	$!interval = $!config.get('irc.plugins.reminders.interval', 1).?Int;

	if (!$!db) {
		if (!$*DB) {
			.emergency('No database connection in $*DB') with $*LOG;
			return;
		}
		.debug('Preparing database') with $*LOG;

		$!db = $*DB.db;

		%!stmt<create> = $!db.prepare(q:to/SQL/);
			INSERT INTO reminders (
				subject,
				created_by,
				created_in,
				trigger_at
			) VALUES ($1, $2, $3, $4)
			SQL

		%!stmt<read> = $!db.prepare(q:to/SQL/);
			SELECT *
			FROM reminders
			WHERE trigger_at <= (NOW() AT TIME ZONE 'utc')
			ORDER BY trigger_at ASC
			SQL
		%!stmt<delete> = $!db.prepare(q:to/SQL/);
			DELETE FROM reminders
			WHERE id = $1
			SQL
	}
}

=begin pod

=NAME    IRC::Client::Plugin::Reminders
=VERSION 0.1.0
=AUTHOR  Patrick Spek <p.spek@tyil.nl>

=begin LICENSE
Copyright © 2020

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU Affero General Public License as published by the Free
Software Foundation, either version 3 of the License, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
details.

You should have received a copy of the GNU Affero General Public License along
with this program.  If not, see http://www.gnu.org/licenses/.
=end LICENSE

=end pod

# vim: ft=raku noet sw=8 ts=8
